/*
  Copyright (c) 2009 Dave Gamble

  Permission is hereby granted, free of charge, to any person obtaining a copy
  of this software and associated documentation files (the "Software"), to deal
  in the Software without restriction, including without limitation the rights
  to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
  copies of the Software, and to permit persons to whom the Software is
  furnished to do so, subject to the following conditions:

  The above copyright notice and this permission notice shall be included in
  all copies or substantial portions of the Software.

  THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
  IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
  FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
  AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
  LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
  OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
  THE SOFTWARE.
*/

#ifndef c_JSON__h
#define c_JSON__h

#ifdef __cplusplus
extern "C"
{
#endif

/* c_JSON Types: */
#define c_JSON_False 0
#define c_JSON_True 1
#define c_JSON_NULL 2
#define c_JSON_Number 3
#define c_JSON_String 4
#define c_JSON_Array 5
#define c_JSON_Object 6

#define c_JSON_IsReference 256

/* The c_JSON structure: */
typedef struct c_JSON {
	struct c_JSON *next,*prev;	/* next/prev allow you to walk array/object chains. Alternatively, use GetArraySize/GetArrayItem/GetObjectItem */
	struct c_JSON *child;		/* An array or object item will have a child pointer pointing to a chain of the items in the array/object. */

	int type;					/* The type of the item, as above. */

	char *valuestring;			/* The item's string, if type==c_JSON_String */
	int valueint;				/* The item's number, if type==c_JSON_Number */
	double valuedouble;			/* The item's number, if type==c_JSON_Number */

	char *string;				/* The item's name string, if this item is the child of, or is in the list of subitems of an object. */
} c_JSON;

typedef struct c_JSON_Hooks {
      void *(*malloc_fn)(size_t sz);
      void (*free_fn)(void *ptr);
} c_JSON_Hooks;

/* Supply malloc, realloc and free functions to c_JSON */
extern void c_JSON_InitHooks(c_JSON_Hooks* hooks);


/* Supply a block of JSON, and this returns a c_JSON object you can interrogate. Call c_JSON_Delete when finished. */
extern c_JSON *c_JSON_Parse(const char *value);
/* Render a c_JSON entity to text for transfer/storage. Free the char* when finished. */
extern char  *c_JSON_Print(c_JSON *item);
/* Render a c_JSON entity to text for transfer/storage without any formatting. Free the char* when finished. */
extern char  *c_JSON_PrintUnformatted(c_JSON *item);
/* Delete a c_JSON entity and all subentities. */
extern void   c_JSON_Delete(c_JSON *c);

/* Returns the number of items in an array (or object). */
extern int	  c_JSON_GetArraySize(c_JSON *array);
/* Retrieve item number "item" from array "array". Returns NULL if unsuccessful. */
extern c_JSON *c_JSON_GetArrayItem(c_JSON *array,int item);
/* Get item "string" from object. Case insensitive. */
extern c_JSON *c_JSON_GetObjectItem(c_JSON *object,const char *string);

/* For analysing failed parses. This returns a pointer to the parse error. You'll probably need to look a few chars back to make sense of it. Defined when c_JSON_Parse() returns 0. 0 when c_JSON_Parse() succeeds. */
extern const char *c_JSON_GetErrorPtr(void);

/* These calls create a c_JSON item of the appropriate type. */
extern c_JSON *c_JSON_CreateNull(void);
extern c_JSON *c_JSON_CreateTrue(void);
extern c_JSON *c_JSON_CreateFalse(void);
extern c_JSON *c_JSON_CreateBool(int b);
extern c_JSON *c_JSON_CreateNumber(double num);
extern c_JSON *c_JSON_CreateString(const char *string);
extern c_JSON *c_JSON_CreateArray(void);
extern c_JSON *c_JSON_CreateObject(void);

/* These utilities create an Array of count items. */
extern c_JSON *c_JSON_CreateIntArray(int *numbers,int count);
extern c_JSON *c_JSON_CreateFloatArray(float *numbers,int count);
extern c_JSON *c_JSON_CreateDoubleArray(double *numbers,int count);
extern c_JSON *c_JSON_CreateStringArray(const char **strings,int count);

/* Append item to the specified array/object. */
extern void c_JSON_AddItemToArray(c_JSON *array, c_JSON *item);
extern void	c_JSON_AddItemToObject(c_JSON *object,const char *string,c_JSON *item);
/* Append reference to item to the specified array/object. Use this when you want to add an existing c_JSON to a new c_JSON, but don't want to corrupt your existing c_JSON. */
extern void c_JSON_AddItemReferenceToArray(c_JSON *array, c_JSON *item);
extern void	c_JSON_AddItemReferenceToObject(c_JSON *object,const char *string,c_JSON *item);

/* Remove/Detatch items from Arrays/Objects. */
extern c_JSON *c_JSON_DetachItemFromArray(c_JSON *array,int which);
extern void   c_JSON_DeleteItemFromArray(c_JSON *array,int which);
extern c_JSON *c_JSON_DetachItemFromObject(c_JSON *object,const char *string);
extern void   c_JSON_DeleteItemFromObject(c_JSON *object,const char *string);

/* Update array items. */
extern void c_JSON_ReplaceItemInArray(c_JSON *array,int which,c_JSON *newitem);
extern void c_JSON_ReplaceItemInObject(c_JSON *object,const char *string,c_JSON *newitem);

/* Duplicate a c_JSON item */
extern c_JSON *c_JSON_Duplicate(c_JSON *item,int recurse);
/* Duplicate will create a new, identical c_JSON item to the one you pass, in new memory that will
need to be released. With recurse!=0, it will duplicate any children connected to the item.
The item->next and ->prev pointers are always zero on return from Duplicate. */

/* ParseWithOpts allows you to require (and check) that the JSON is null terminated, and to retrieve the pointer to the final byte parsed. */
extern c_JSON *c_JSON_ParseWithOpts(const char *value,const char **return_parse_end,int require_null_terminated);

/* Macros for creating things quickly. */
#define c_JSON_AddNullToObject(object,name)		c_JSON_AddItemToObject(object, name, c_JSON_CreateNull())
#define c_JSON_AddTrueToObject(object,name)		c_JSON_AddItemToObject(object, name, c_JSON_CreateTrue())
#define c_JSON_AddFalseToObject(object,name)		c_JSON_AddItemToObject(object, name, c_JSON_CreateFalse())
#define c_JSON_AddBoolToObject(object,name,b)	c_JSON_AddItemToObject(object, name, c_JSON_CreateBool(b))
#define c_JSON_AddNumberToObject(object,name,n)	c_JSON_AddItemToObject(object, name, c_JSON_CreateNumber(n))
#define c_JSON_AddStringToObject(object,name,s)	c_JSON_AddItemToObject(object, name, c_JSON_CreateString(s))

/* When assigning an integer value, it needs to be propagated to valuedouble too. */
#define c_JSON_SetIntValue(object,val)			((object)?(object)->valueint=(object)->valuedouble=(val):(val))

#ifdef __cplusplus
}
#endif

#endif
