import { PDFDocumentElement, PDFImageElement, PDFTextElement, PDFTextNode, PDFViewElement } from './elements.js'

function noop(fn) {
	throw Error(`no-op: ${fn}`)
}

export function createNodeOps(nodeMap) {
	return {
		patchProp: (el, key, preaVal, nextVal) => {
			if (el instanceof PDFImageElement && nextVal) {
				if (key === 'src') {
					el._src = nextVal
				}

				if (key === 'width') {
					el.width = nextVal
				}

				if (key === 'height') {
					el.height = nextVal
				}
			}

			if (nextVal && key === 'styles') {
				for (const [attr, value] of Object.entries(nextVal)) {
					el.styles[attr] = value
				}
			}
		},

		insert: (child, parent, anchor) => {
			if (parent instanceof PDFDocumentElement) {
				nodeMap[parent.id] = parent
			}

			if (!(child.id in nodeMap)) {
				nodeMap[child.id] = child
			}

			parent.children.push(child.id)
			child._parent = parent.id
		},

		createElement: tag => {
			if (tag === 'image') {
				return new PDFImageElement(tag)
			}

			if (tag === 'Text') {
				return new PDFTextElement(tag)
			}

			if (tag === 'View') {
				return new PDFViewElement(tag)
			}

			throw Error(`Unknown tag ${tag}`)
		},

		createText: text => {
			return new PDFTextNode(text)
		},

		parentNode: node => {
			return null
		},

		createComment: () => noop('createComment'),
		setText: () => noop('setText'),
		setElementText: () => noop('setElementText'),
		nextSibling: () => noop('nextSibling'),
		querySelector: () => noop('querySelector'),
		setScopeId: () => noop('setScopeId'),
		cloneNode: () => noop('cloneNode'),
		insertStaticContent: () => noop('insertStaticContent'),
		forcePatchProp: () => noop('forcePatchProp'),
		remove: () => noop('remove')
	}
}
