import { createRenderer } from '@vue/runtime-core'
import { jsPDF } from 'jspdf'
import PDFDocument from 'pdfkit'
import fs from 'fs'

import { getStyleValue, styleRules, defaults } from './styling.js'
import { PDFElement, PDFTextNode, PDFImageElement } from './elements.js'
import { createNodeOps } from './nodeOps.js'

// Modelled after https://github.com/vuejs/vue-next/blob/master/packages/runtime-dom/src/index.ts#L61
export const createApp = (...args) => {
	const nodeMap = {}

	// createNodeOps mutations nodeMap
	// TODO: is there a way to avoid this?
	const nodeOps = createNodeOps(nodeMap)
	const renderer = createRenderer(nodeOps)

	const app = renderer.createApp(...args)
	// const pdf = new document()
	const doc = {
		filename: '',
		filename2: '',
		page: { width: 0, height: 0, x: 0, y: 0, line_height: 1.3 },
		content: []
	}
	const pdfDoc = new jsPDF({ orientation: 'portrait', lineHeight: 1.3, unit: 'pt', format: 'a4', hotfixes: ['px_scaling'] })
	const pdfDoc2 = new PDFDocument()
	const page = doc.page
	page.width = pdfDoc.internal.pageSize.width
	page.height = pdfDoc.internal.pageSize.height
	const pdf = {
		pdfDoc2,
		pdfDoc,
		doc,
		moveDown(px) {
			doc.page.x += px
			doc.content.push({ move_down: px })
			pdfDoc.moveTo(page.x, page.y)
			pdfDoc2.moveDown(px)
		},
		text(text, opt) {
			const dim = pdfDoc.getTextDimensions(text)
			page.y += dim.h * page.line_height
			let offsetX = 0
			if (opt.align === 'center') {
				offsetX = page.width / 2 - dim.w / 2
			} else if (opt.align === 'right') {
				offsetX = page.width - dim.w / 2
			}
			pdfDoc.text(text, page.x + offsetX, page.y, opt)
			doc.content.push(Object.assign({ text: text, y: page.y }, opt))
			pdfDoc2.text(text, opt)
		},
		image(src, width, height, opt) {
			doc.content.push(Object.assign({ image: src, width, height }, opt))
			// Center it nicely. align: center option in PDFKit appears not to work at all.
			if (opt.align === 'center' && width) {
				pdfDoc.addImage(src, 'JPEG', page.x, page.y, width, height || width)
				pdfDoc2.image(src, doc.page.width / 2 - width / 2, undefined, { width: width })
			} else {
				// Just do whatever? TODO: Figure out reasonable defaults.
				pdfDoc.addImage(src, 'JPEG', page.x, page.y, width, height || width)
				pdfDoc2.image(src)
			}
		},
		fill(fill) {
			doc.content.push({ fill })
			pdfDoc.setFillColor(fill)
			pdfDoc.setTextColor(fill)
			pdfDoc2.fill(fill)
		},
		fontSize(font_size) {
			doc.content.push({ font_size })
			pdfDoc.setFontSize(font_size)
			pdfDoc2.fontSize(font_size)
		}
	}

	const { mount } = app

	// Vue Core types this as `any` too
	// https://github.com/vuejs/vue-next/blob/3626ff07fe5107080c52e85018070562c84b796e/packages/runtime-dom/src/index.ts#L61
	app.mount = doc => {
		const proxy = mount(doc)
		renderToPDF(doc, pdf, nodeMap)
		return proxy
	}
	return app
}

function renderToPDF(doc, pdf, nodeMap) {
	// const stream =  pdf.pipe(fs.createWriteStream(doc.filename))

	pdf.doc.filename = doc.filename
	pdf.doc.filename2 = doc.filename.replace('.pdf', '2.pdf')
	const stream = pdf.pdfDoc2.pipe(fs.createWriteStream(pdf.doc.filename2))

	// This is the PDFKit integration. It's pretty messy
	// due to the imperative nature of PDFKit.
	const draw = node => {
		// marginTop must be applied first.
		if (node.styles.marginTop) {
			pdf.moveDown(node.styles.marginTop)
		}

		if (node instanceof PDFElement) {
			for (const rule of styleRules) {
				applyStyle(rule, node)
			}
		}

		if (node instanceof PDFTextNode) {
			const align = getStyleValue('align', node, nodeMap)
			pdf.text(node.value.trim(), {
				align
				// indent: 0
			})
		}

		if (node instanceof PDFImageElement) {
			const align = getStyleValue('align', node, nodeMap)

			// Center it nicely. align: center option in PDFKit appears not to work at all.
			if (align === 'center' && node.width) {
				pdf.image(node.src, pdf.doc.page.width / 2 - node.width / 2, undefined, { width: node.width })
			} else {
				// Just do whatever? TODO: Figure out reasonable defaults.
				pdf.image(node.src)
			}
		}

		// marginBottom must be applied last.
		if (node.styles.marginBottom) {
			pdf.moveDown(node.styles.marginBottom)
		}
	}

	const traverse = node => {
		if (node instanceof PDFElement) {
			for (const child of node.children) {
				draw(nodeMap[child])
				traverse(nodeMap[child])
			}
		}
	}

	const applyStyle = (rule, node) => {
		const value = getStyleValue(rule, node, nodeMap)
		if (value === undefined) {
			throw Error(`Not default style found for ${rule}`)
		}
		if (rule === 'color') {
			const color = getStyleValue(rule, node, nodeMap)
			pdf.fill(color)
		}
		if (rule === 'fontSize') {
			const fontSize = getStyleValue(rule, node, nodeMap)
			pdf.fontSize(fontSize || defaults.fontSize)
		}
	}

	const rootNode = nodeMap['root']
	traverse(rootNode)

	delete nodeMap['root']['_vnode']
	delete nodeMap['root']['__vue_app__']

	console.log(nodeMap)
	console.log('\nCreated structure', pdf.doc)
	pdf.pdfDoc2.end()
	stream.on('finish', () => {
		console.log(`\nWrote to ${pdf.doc.filename2}.`)
	})
	pdf.pdfDoc.save(pdf.doc.filename)
	console.log(`\nWrote to ${pdf.doc.filename}.`)
}
