// emf and wmf to png/jpeg
// https://keestalkstech.com/2016/06/rasterizing-emf-files-png-net-csharp/
// copied from: https://github.com/KeesCBakker/KeesTalksTech-Utility-Pack/tree/master

using System;
using System.Collections.Generic;
using System.Linq;
using System.Text;

using System.Threading.Tasks;
using System.Drawing;
using System.Drawing.Imaging;
using System.IO;

using System.Linq.Expressions;
    public class NameOf
    {
        public static String nameof<T>(Expression<Func<T>> name)
        {
            MemberExpression expressionBody = (MemberExpression)name.Body;
            return expressionBody.Member.Name;
        }
    }

    /// <summary>
    /// Calculates a bounding box.
    /// </summary>
    public class BoundingBox
    {
        /// <summary>
        /// Gets the width.
        /// </summary>
        /// <value>
        /// The width.
        /// </value>
        public float Width { get; private set; }

        /// <summary>
        /// Gets the height.
        /// </summary>
        /// <value>
        /// The height.
        /// </value>
        public float Height { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="BoundingBox"/> class.
        /// </summary>
        /// <param name="width">The width.</param>
        /// <param name="height">The height.</param>
        public BoundingBox(float width, float height)
        {
            this.Width = width;
            this.Height = height;
        }

        /// <summary>
        /// Scales the boundingbox using the specified scale.
        /// </summary>
        /// <param name="scale">The scale.</param>
        /// <returns>A new bounding box.</returns>
        public BoundingBox Scale(float scale)
        {
            return new BoundingBox(Width * scale, Height * scale);
        }

        /// <summary>
        /// Calculates the new bounding box based on the max width and max height.
        /// </summary>
        /// <param name="maxWidth">The maximum width.</param>
        /// <param name="maxHeight">The maximum height.</param>
        /// <returns>A new bounding box.</returns>
        public BoundingBox Calculate(float maxWidth, float maxHeight)
        {
            var resolution = Width / Height;
            var maxResolution = maxWidth / maxHeight;

            if (resolution == maxResolution)
            {
                return new BoundingBox(maxWidth, maxHeight);
            }

            if (maxResolution == 1)
            {
                var diff = maxWidth / Math.Max(Width, Height);
                return new BoundingBox(
                    Width * diff,
                    Height * diff
                );
            }

            if (resolution == 1)
            {
                var min = Math.Min(maxWidth, maxHeight);
                var diff = min / Width;

                return new BoundingBox(
                    Width * diff,
                    Height * diff
                );
            }

            var h = (maxWidth / Width) * Height;
            if (h <= maxHeight)
            {
                return new BoundingBox(maxWidth, h);
            }

            var w = (maxHeight / Height) * Width;
            return new BoundingBox(w, maxHeight);
        }
    }


    /// <summary>
    /// Meta information for a meta file (EMF, WMF).
    /// </summary>
    public class MetafileMeta
    {
        /// <summary>
        /// Gets the width.
        /// </summary>
        /// <value>
        /// The width.
        /// </value>
        public int Width { get; private set; }

        /// <summary>
        /// Gets the height.
        /// </summary>
        /// <value>
        /// The height.
        /// </value>
        public int Height { get; private set; }

        /// <summary>
        /// Gets the horizontal DPI.
        /// </summary>
        /// <value>
        /// The DPI.
        /// </value>
        public float DpiX { get; private set; }

        /// <summary>
        /// Gets the vertical DPI.
        /// </summary>
        /// <value>
        /// The DPI.
        /// </value>
        public float DpiY { get; private set; }

        /// <summary>
        /// Initializes a new instance of the <see cref="MetafileMeta"/> class.
        /// </summary>
        /// <param name="file">The file.</param>
        public MetafileMeta(Metafile file)
        {
            if (file == null)
            {
                throw new ArgumentNullException(NameOf.nameof(() => file));
            }

            Width = file.Width;
            Height = file.Height;

            var header = file.GetMetafileHeader();

            DpiX = header.DpiX;
            DpiY = header.DpiY;
        }

        /// <summary>
        /// Gets the bounding box with dpi correction.
        /// </summary>
        /// <param name="scale">The scale.</param>
        /// <returns>The bounding box.</returns>
        public BoundingBox GetBoundingBoxWithDpiCorrection(uint scale = 1)
        {
            var width = (Width * scale / DpiX * 100);
            var height = (Height * scale / DpiY * 100);

            return new BoundingBox(width, height);
        }
    }
    
    /// <summary>
    /// Utility methods for working with meta files (WMF, EMF).
    /// </summary>
    public static class MetafileUtility
    {
        private static readonly ImageFormat[] _transparentFormats = { ImageFormat.Gif, ImageFormat.Png, ImageFormat.Wmf, ImageFormat.Emf };

        /// <summary>
        /// Gets the metafile meta data.
        /// </summary>
        /// <param name="file">The file.</param>
        /// <returns>The meta data.</returns>
        public static MetafileMeta GetMetafileMetaData(string file)
        {
            if (String.IsNullOrEmpty(file))
            {
                throw new ArgumentNullException(file);
            }

            using (var stream = File.OpenRead(file))
            {
                return GetMetafileMetaData(stream);
            }
        }

        /// <summary>
        /// Gets the metafile meta data.
        /// </summary>
        /// <param name="stream">The stream.</param>
        /// <returns>The meta data.</returns>
        public static MetafileMeta GetMetafileMetaData(Stream stream)
        {
            if (stream == null)
            {
                throw new ArgumentNullException(NameOf.nameof(() => stream));
            }

            var p = stream.Position;
            stream.Position = 0;

            try
            {
                using (var img = new Metafile(stream))
                {
                    return new MetafileMeta(img);
                }
            }
            finally
            {
                stream.Position = p;
            }
        }

        /// <summary>
        /// Saves the meta file.
        /// </summary>
        /// <param name="source">The source.</param>
        /// <param name="destination">The destination.</param>
        /// <param name="box">The box. If no box is specified a 4x scale of the original will be returned.</param>
        /// <param name="backgroundColor">Color of the background.</param>
        /// <param name="format">The format. Default is PNG.</param>
        /// <param name="parameters">The parameters.</param>
        public static void SaveMetaFile(
            Stream source,
            Stream destination,
            BoundingBox box = null,
            Color? backgroundColor = null,
            ImageFormat format = null,
            EncoderParameters parameters = null)
        {
            if (source == null)
            {
                throw new ArgumentNullException(NameOf.nameof(() => source));
            }
            if (destination == null)
            {
                throw new ArgumentNullException(NameOf.nameof(() => destination));
            }

            using (var img = new Metafile(source))
            {
                var f = format ?? ImageFormat.Png;

                //Determine default background color. 
                //Not all formats support transparency. 
                if (backgroundColor == null)
                {
                    backgroundColor = GetDefaultBackgroundColor(f);
                }

                //header contains DPI information
                var header = img.GetMetafileHeader();

                if (box == null)
                {
                    box = new MetafileMeta(img).GetBoundingBoxWithDpiCorrection(4);
                }

                var width = (int)Math.Round(box.Width, 0, MidpointRounding.ToEven);
                var height = (int)Math.Round(box.Height, 0, MidpointRounding.ToEven);

                using (var bitmap = new Bitmap(width, height))
                {
                    using (var g = System.Drawing.Graphics.FromImage(bitmap))
                    {
                        //fills the background
                        g.Clear(backgroundColor.Value);

                        //reuse the width and height to draw the image
                        //in 100% of the square of the bitmap
                        g.DrawImage(img, 0, 0, bitmap.Width, bitmap.Height);
                    }

                    //get codec based on GUID
                    ImageCodecInfo codec = GetCodec(f);

                    bitmap.Save(destination, codec, parameters);
                }
            }
        }

        /// <summary>
        /// Saves the meta file.
        /// </summary>
        /// <param name="sourceFilePath">The source file path.</param>
        /// <param name="destinationFilePath">The destination file path.</param>
        /// <param name="box">The box. If no box is specified a 4x scale of the original will be returned.</param>
        /// <param name="backgroundColor">Color of the background.</param>
        /// <param name="format">The format. Default is PNG.</param>
        /// <param name="parameters">The parameters.</param>
        public static void SaveMetaFile(
            string sourceFilePath,
            string destinationFilePath,
            BoundingBox box = null,
            Color? backgroundColor = null,
            ImageFormat format = null,
            EncoderParameters parameters = null)
        {
            if (String.IsNullOrEmpty(sourceFilePath))
            {
                throw new ArgumentNullException(NameOf.nameof(() => destinationFilePath));
            }
            if (String.IsNullOrEmpty(sourceFilePath))
            {
                throw new ArgumentNullException(NameOf.nameof(() => destinationFilePath));
            }

            using (var destination = File.OpenWrite(destinationFilePath))
            {
                using (var stream = File.OpenRead(sourceFilePath))
                {
                    SaveMetaFile(stream, destination, box, backgroundColor, format);
                }
            }
        }

        /// <summary>
        /// Saves the meta file using two stages. The meta file will be converted to 4x 
        /// its size PNG before it is converted to the target format.
        /// </summary>
        /// <param name="sourceFilePath">The source file path.</param>
        /// <param name="destinationFilePath">The destination file path.</param>
        /// <param name="box">The box.</param>
        /// <param name="backgroundColor">Color of the background.</param>
        /// <param name="format">The format.</param>
        /// <param name="parameters">The parameters.</param>
        public static void SaveMetaFileUsingTwoStages(
            string sourceFilePath,
            string destinationFilePath,
            BoundingBox box = null,
            Color? backgroundColor = null,
            ImageFormat format = null,
            EncoderParameters parameters = null)
        {
            if (String.IsNullOrEmpty(sourceFilePath))
            {
                throw new ArgumentNullException(NameOf.nameof(() => destinationFilePath));
            }
            if (String.IsNullOrEmpty(sourceFilePath))
            {
                throw new ArgumentNullException(NameOf.nameof(() => destinationFilePath));
            }

            using (var source = File.OpenRead(sourceFilePath))
            {
                using (var destination = File.OpenWrite(destinationFilePath))
                {
                    SaveMetaFileUsingTwoStages(source, destination, box, backgroundColor, format, parameters);
                }
            }
        }


        /// <summary>
        /// Saves the meta file using two stages. The meta file will be converted to 4x
        /// its size PNG before it is converted to the target format.
        /// </summary>
        /// <param name="source">The source.</param>
        /// <param name="destination">The destination.</param>
        /// <param name="box">The box.</param>
        /// <param name="backgroundColor">Color of the background.</param>
        /// <param name="format">The format.</param>
        /// <param name="parameters">The parameters.</param>
        public static void SaveMetaFileUsingTwoStages(
            Stream source,
            Stream destination,
            BoundingBox box = null,
            Color? backgroundColor = null,
            ImageFormat format = null,
            EncoderParameters parameters = null)
        {
            if (source == null)
            {
                throw new ArgumentNullException(NameOf.nameof(() => source));
            }
            if (destination == null)
            {
                throw new ArgumentNullException(NameOf.nameof(() => destination));
            }

            using (var png = new MemoryStream())
            {
                BoundingBox pngBox;
                if (box == null)
                {
                    box = GetMetafileMetaData(source).GetBoundingBoxWithDpiCorrection(4);
                    pngBox = box;
                }
                else
                {
                    pngBox = box.Scale(4);
                }

                //safe PNG - default settings
                SaveMetaFile(source, png);

                png.Position = 0;

                using (var pngImage = Image.FromStream(png))
                {
                    using (var targetImage = new Bitmap((int)box.Width, (int)box.Height))
                    {
                        var f = format ?? ImageFormat.Png;

                        using (var g = System.Drawing.Graphics.FromImage(targetImage))
                        {
                            if (backgroundColor == null)
                            {
                                backgroundColor = GetDefaultBackgroundColor(f);
                            }

                            g.Clear(backgroundColor.Value);
                            g.DrawImage(pngImage, 0, 0, targetImage.Width, targetImage.Height);
                        }

                        var codec = GetCodec(f);
                        targetImage.Save(destination, codec, parameters);
                    }
                }
            }
        }

        /// <summary>
        /// Gets the codec.
        /// </summary>
        /// <param name="format">The format.</param>
        /// <returns>The codec or <c>null</c>.</returns>
        private static ImageCodecInfo GetCodec(ImageFormat format)
        {
            return ImageCodecInfo.GetImageEncoders().FirstOrDefault(c => c.FormatID == format.Guid);
        }

        /// <summary>
        /// Gets the default color of the background.
        /// </summary>
        /// <param name="format">The format.</param>
        /// <returns>The color.</returns>
        private static Color GetDefaultBackgroundColor(ImageFormat format)
        {
            var isTransparentFormat = _transparentFormats.Contains(format);
            return isTransparentFormat ? Color.Transparent : Color.White;
        }
    }


public class Program
{
    public static void Main(string[] args)
    {
        var source = "260.emf";
        System.Console.WriteLine("20015.png");
        BoundingBox pngBox = MetafileUtility.GetMetafileMetaData(source).GetBoundingBoxWithDpiCorrection(1); // new BoundingBox(800, 800);
        pngBox = pngBox.Scale((float) 0.6);
        MetafileUtility.SaveMetaFile(source, "20015.png", pngBox, Color.White);
        System.Console.WriteLine("20015-4x.png");
        MetafileUtility.SaveMetaFileUsingTwoStages(source, "20015-4x.png", pngBox, Color.White);
        System.Console.WriteLine("Hello, World!");
    }
}