--- lib/ansicolors.lua
-- see: https://stackoverflow.com/questions/4842424/list-of-ansi-color-escape-sequences
-- ansicolors.lua v1.0.2 (2012-08)
-- Copyright (c) 2009 Rob Hoelz <rob@hoelzro.net>
-- Copyright (c) 2011 Enrique García Cota <enrique.garcia.cota@gmail.com>
--
-- Permission is hereby granted, free of charge, to any person obtaining a copy
-- of this software and associated documentation files (the "Software"), to deal
-- in the Software without restriction, including without limitation the rights
-- to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
-- copies of the Software, and to permit persons to whom the Software is
-- furnished to do so, subject to the following conditions:
--
-- The above copyright notice and this permission notice shall be included in
-- all copies or substantial portions of the Software.
--
-- THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
-- IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
-- FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
-- AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
-- LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
-- OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
-- THE SOFTWARE.
local supported = true

local keys = {
	-- reset
	reset = 0,

	-- misc
	bright = 1,
	dim = 2,
	underline = 4,
	blink = 5,
	reverse = 7,
	hidden = 8,

	-- foreground colors
	black = 30,
	red = 31,
	green = 32,
	yellow = 33,
	blue = 34,
	magenta = 35,
	cyan = 36,
	white = 37,

	-- background colors
	blackbg = 40,
	redbg = 41,
	greenbg = 42,
	yellowbg = 43,
	bluebg = 44,
	magentabg = 45,
	cyanbg = 46,
	whitebg = 47
}

local escapeString = string.char(27) .. '[%dm'
local function escapeNumber(number)
	return escapeString:format(number)
end

local function escapeKeys(str)
	if not supported then
		return ""
	end
	local buffer = {}
	local number
	for word in str:gmatch("%w+") do
		number = keys[word]
		-- if not number then
		--   print("Unknown color key: " .. tostring(word))
		--   printed string contains '%{', it causes color to fail
		-- else
		if number then
			buffer[#buffer + 1] = escapeNumber(number)
		end
	end
	return table.concat(buffer)
end

local function replaceEscapeKeys(_, str)
	return escapeKeys(str)
end

local function replaceCodes(str)
	str = str:gsub("(%%{(.-)})", replaceEscapeKeys)
	return str
end

-- public

local ffi
local function ansicolors(str)
	str = tostring(str or '')
	if not ffi then
		ffi = require "ffi" -- NOT mffi here because mffi uses ansicolors
		local ok, util = pcall(require, "util")
		if ok then
			if util.isCloud() then
				supported = false
			elseif util.from4d() then
				supported = false
			end
		end
	end
	if not supported then
		return string.gsub(str, "(%%{(.-)})", "") -- aws does not show colors, delete color commands
	end
	if ffi.os == "Linux" then
		str = str:gsub("bright", "") -- bright is too dim in Linux
	end
	if str:find('{%', 3, true) then
		str = string.gsub(str, "{%%", "__{__") -- query: like ‰{‰
		str = replaceCodes('%{reset}' .. str .. '%{reset}')
		return string.gsub(str, '__{__', '{%')
	end
	return replaceCodes('%{reset}' .. str .. '%{reset}')
end

local function setSupported(supported_)
	supported = supported_
end

return setmetatable({noReset = replaceCodes, setSupported = setSupported}, {
	__call = function(_, str)
		return ansicolors(str)
	end
})
