--- lib/auth.lua
--
-- @module auth
local util = require "util"
local json = require "json"
local l = require"lang".l
local dt = require "dt"
local fn = require "fn"
local peg = require "peg"
local coro = require "coro"
local useCoro = coro.useCoro(false)
local currentThread
if useCoro then
	currentThread = coro.currentThreadFunction()
end
local from4d = util.from4d()
local currentAuthTbl = {}
local currentUserTbl = {}
local loc = {defaultLanguage = "en"} -- "fi"
local authId = 0

local dsave, qry, dqry, dload, dconn, dschema, dprf, dsql, xxhash
local authTbl -- temporary variable

local function loadLibs()
	if not dschema then
		dschema = require "dschema"
		dconn = require "dconn"
		dload = require "dload"
		dqry = require "dqry"
		dsave = require "dsave"
		dprf = require "dprf"
		qry = require "qry"
		dsql = require "dsql"
		xxhash = require "xxhash"
		-- dsave.notifyChange("auth.clearUserTbl", "per.login_id", clearCache)
	end
end

local thread
local function currentAuthTable(thread_)
	if useCoro then
		thread = currentThread(thread_)
		if currentAuthTbl[thread] == nil then
			authId = authId + 1
			currentAuthTbl[thread] = {auth_id = authId}
		end
		return currentAuthTbl[thread]
	end
	return currentAuthTbl
end

local function setCurrentAuthTable(auth, thread_)
	if useCoro then
		thread = currentThread(thread_)
		-- util.print("setCurrentAuthTable %s, %s", auth, thread)
		if currentAuthTbl[thread] then
			util.recToRec(currentAuthTbl[thread], auth, {replace = true})
			return currentAuthTbl[thread]
		end
		local newAuth = util.clone(auth)
		authId = authId + 1
		newAuth.auth_id = authId
		currentAuthTbl[thread] = newAuth
		return auth
	end
	if currentAuthTbl ~= auth or currentAuthTbl.organization_id ~= auth.organization_id then
		util.recToRec(currentAuthTbl, auth, {replace = true}) -- util.clone(auth)
	end
	return currentAuthTbl
end

local function treadCreated(oldThread, newThread)
	if useCoro then
		local oldAuth = currentAuthTable(oldThread)
		setCurrentAuthTable(oldAuth, newThread)
	end
end

local function currentUserId()
	authTbl = currentAuthTable()
	return authTbl.login_id or ""
end

local function currentOrganizationNumber()
	authTbl = currentAuthTable()
	return authTbl.organization_number or 0
end

local function currentOrganizationId()
	authTbl = currentAuthTable()
	return authTbl.organization_id or ""
end

local function setCurrentOrganization(orgId, callFromAuth)
	authTbl = currentAuthTable()
	if callFromAuth == "set-id" then
		authTbl.organization_id = orgId
		return
	end
	if orgId == nil then
		-- dsql.clearQuery() -- this would cause a new connection, it's not needed
		authTbl.organization_id = ""
		-- authTbl.schema = ""
		return
	end
	loadLibs()
	local currentConn = dconn.currentConnection() -- creates a new local connection if needed {organization_id = authTbl.organization_id}
	local prevOrg = currentOrganizationId()
	if orgId and authTbl.alias_id == orgId then
		return prevOrg, authTbl
	end
	local aliasId = dconn.aliasId(orgId)
	local orgIdFull = aliasId
	if orgIdFull and from4d and (orgIdFull == "4d" or peg.found(orgIdFull, "-4d")) then -- demo-4d-0 or demo-4d2-0
		-- util.print("use plg4d, orgId: %s, orgIdFull: %s", orgId, orgIdFull)
		if util.isLinux() then
			orgIdFull = "demo-fi_demo4d-0"
		else
			orgIdFull = "plg4d-plg4d-0"
		end
	end
	if authTbl.organization_id == orgIdFull and authTbl.database then -- orgIdFull ~= "" and
		if currentConn.organization_id ~= orgIdFull then
			util.printWarning("connection organization id '%s' is not equal to authorization id '%s'", currentConn.organization_id, orgIdFull)
		end
		return prevOrg, authTbl
	end
	if currentConn == nil or currentConn.organization_id ~= orgIdFull then
		dsql.clearQuery()
	else
		dsql.clearQuery(currentConn, false)
	end
	authTbl.organization_id = orgIdFull
	setCurrentAuthTable(authTbl)
	if not callFromAuth then -- and (currentConn == nil or currentConn.organization_id ~= authTbl.organization_id) then
		if orgId ~= "" then
			dconn.setCurrentOrganization(authTbl.organization_id, prevOrg, authTbl)
		end
	end
	local authTbl2 = currentAuthTable()
	if authTbl ~= authTbl2 then
		if not util.tableEqual(authTbl, authTbl2) then
			util.printRed("authTbl is not same as authTbl2, %s, %s", authTbl, authTbl2)
		end
	end
	return prevOrg, authTbl
end

local savePrf = {save = {table = "session", record_type = ""}}
local function saveAuthRecord(rec)
	local sel = {rec} -- must be array of records
	loadLibs()
	local prevOrgId -- todo: fix that auth save goes to same database as auth query
	if rec.organization_id then
		prevOrgId = dconn.setCurrentOrganization(rec.organization_id)
	else
		util.printError("session save record does not contain organization_id: '%s'", json.toJson(rec))
	end
	local _, err = dsave.saveToDatabase(sel, savePrf)
	if prevOrgId and prevOrgId ~= "" and prevOrgId ~= rec.organization_id then
		authTbl = currentAuthTable()
		dconn.setCurrentOrganization(prevOrgId)
	end
	if err then
		util.printError(err)
		return err
	end
	return nil
end

local function querySession(auth, fldArr)
	authTbl = setCurrentAuthTable(auth)
	loadLibs()
	dqry.query("", "ses.session_id", "=", auth.auth_token, "") -- needs record_type as 5. param
	local sel, info = dload.selectionToRecordArray(fldArr)
	return sel, info
end

local function queryUser(loginId, fldArr, option)
	loadLibs()
	dqry.query("", "per.login_id", "=", loginId, "user") -- needs record_type as 5. param
	-- dqry.query("or", "per.person_id", "=", loginId, "user")
	if option ~= "only-login-id" then
		dqry.query("or", "per.email", "=", loginId, "user")
	end
	local sel, info = dload.selectionToRecordArray(fldArr)
	return sel, info
end

local function defaultOrganizationId(user)
	local sel = queryUser(user, {"per.json_data"}, "only-login-id")
	if sel and #sel == 1 then
		local data = json.fromJson(sel[1].json_data)
		return data.default_organization_id
	end
	return nil
end

local function userGroup(user)
	user = user or currentUserId()
	loadLibs()
	local ret = qry.queryJson("person/user_group.json", {login_id = user})
	if ret.data and ret.data[1] then
		local data = json.fromJson(ret.data[1].json_data)
		return data and data.group or nil
	end
	return nil -- FIX: get user groups
end

local authPrf
local function preference()
	if authPrf then
		return authPrf
	end
	loadLibs()
	-- lang = lang or defaultLanguage
	local hash = {}
	authPrf = dprf.prf("auth/preference.json", "no-cache", hash)
	-- nocache because pref is big and used only 1 or 2 times (here + auth_worker)
	if authPrf == nil then
		return nil
	end
	authPrf.not_logged_in = dprf.queryPreference("auth/preference_not_logged_in.json", "no-cache", hash)
	authPrf.logged_in = dprf.queryPreference("auth/preference_logged_in.json", "no-cache", hash)
	authPrf.password_hash = xxhash.hash64string(table.concat(hash))
	return authPrf
end

--[[
local function clearCache(fld) -- dsave.notifyChange() callback
	if fld == "per.login_id" then
		currentUserTbl = {}
	end
end
]]

local function defaultLanguage()
	local prf = dprf.prf("auth/preference.json") -- re-get from database if needed - maybe connection has changed
	local lang = prf and prf.default_language
	if lang == "" then
		lang = nil
	end
	return lang or loc.defaultLanguage
end

local function currentLanguage(user)
	authTbl = currentAuthTable()
	local lang = authTbl.language or defaultLanguage()
	if user then
		return lang
	end
	return lang:lower()
end

local function setCurrentUserId(userId)
	authTbl = currentAuthTable()
	authTbl.login_id = userId
end

local function currentUser()
	authTbl = currentAuthTable()
	local user = ""
	if authTbl ~= nil and authTbl.login_id ~= nil and authTbl.login_id ~= "" then
		if currentUserTbl[authTbl.login_id] == nil then
			local sel = queryUser(authTbl.login_id, {"per.person_id"}, "only-login-id")
			authTbl = currentAuthTable() -- we must set this again after coroutine yield
			if sel and sel[1] then
				currentUserTbl[authTbl.login_id] = sel[1].person_id
			else
				currentUserTbl[authTbl.login_id] = authTbl.login_id -- do not set default
			end
		end
		user = currentUserTbl[authTbl.login_id] or authTbl.login_id
	end
	return user
end

local function userInGroup(user, group)
	local userGroupArr = userGroup(user)
	return fn.index(group, userGroupArr) ~= nil
end

local function currentAuthValid()
	authTbl = currentAuthTable()
	return authTbl and authTbl.organization_id and authTbl.organization_id ~= ""
end

local function currentDatabase()
	authTbl = currentAuthTable()
	return authTbl.database -- is ok to be nil
end

local prevAuthError
local function authError(msg, url, color)
	authTbl = currentAuthTable()
	msg = msg or prevAuthError or l("Authorization, unknown error")
	prevAuthError = msg
	if color == nil then
		color = "bright magenta"
		msg = util.printColor(color, "authorization error: %s", msg)
	else
		msg = util.printRed("authorization error: %s", msg)
	end
	if authTbl then
		authTbl.valid_until = 0
	end
	if url then
		return false, {url = url, info = {error = msg}, auth = authTbl}
	end
	return false, {info = {error = msg}, auth = authTbl}
end

local sessionField = {"ses.session_id", "ses.login_id", "ses.create_time", "ses.valid_until", "ses.organization_id", "ses.record_id", "ses.json_data"}
local function sessionRec(param) -- TODO add option and requests url
	local sel, info = querySession(param.auth, sessionField) -- no query before this -> will return all records
	if type(info) == "string" then
		return authError(info)
	elseif type(info) ~= "table" then
		return authError(l("session query failed, organization '%s'", param and param.auth and tostring(param.auth.organization_id) or ""))
	elseif info.err then
		return authError(info.err)
	elseif not sel then
		-- sel, info = querySession(param.auth, sessionField) -- for debug
		return authError(l("session does not exist, organization '%s', session '%s'", tostring(param.auth.organization_id), tostring(param.auth.auth_token)), "login")
	elseif #sel > 1 then
		return authError(l("more than 1 session found"))
	elseif #sel < 1 then
		return authError(l("session was not found, organization '%s', session '%s'", tostring(param.auth.organization_id), tostring(param.auth.auth_token)), "login")
	end
	return sel[1]
end

local function checkParam(param, option)
	if not param.auth then
		return authError("authorization record is missing", "login")
	elseif type(param.auth.organization_id) ~= "string" or param.auth.organization_id == "" then
		if option ~= "logout" then
			return authError("please login first", "login") -- authorization error, organization id is missing
		end
	elseif type(param.auth.auth_token) ~= "string" or param.auth.auth_token == "" then
		-- setCurrentOrganization("pg") --4d test
		-- param.auth = currentAuthTable()
		if option ~= "logout" then
			return authError("authorization token is missing", "login")
		end
	end
	local rec = sessionRec(param)
	if type(rec) ~= "table" then
		return authError(rec)
	end
	if type(rec.json_data) == "string" then
		rec.json_data = json.fromJson(rec.json_data)
		rec.language = rec.json_data.language
	end
	setCurrentUserId(rec.login_id)
	return nil, rec
end

local function logout(param)
	authTbl = setCurrentAuthTable(param.auth)
	authTbl.organization_id = ""
	--[[{	login_id = rec.login_id,
		auth_token = rec.session_id,
		language = defaultLanguage(),
		organization_id = rec.organization_id
	}]]
	local allow, rec = checkParam(param, "logout")
	if allow == false then
		return allow, rec
	end
	rec.valid_until = dt.currentDateTime() -- prf.logout.use_utc_time
	if rec.json_data == "{}" then
		rec.json_data = nil
	end
	local err = saveAuthRecord(rec) -- update session record
	if err then
		return authError(err)
	end
	return nil, {no_auth_return = true}
end

local function authenticate(url, param) -- TODO add option and requests url
	loadLibs()
	if url == "/rest/nc/ping" then
		return true
	elseif url == "/rest/nc/echo" or url == "/rest/nc/echo2" then
		return true
	elseif url == "/rest/nc/dynamic-form/import" then
		return true
	elseif url == "/api/send" then
		return true
	end
	local orgId
	if param.auth and param.auth.organization_id then
		authTbl = setCurrentAuthTable(param.auth)
		orgId = authTbl.organization_id -- authTbl.organization_id may change during session search and save, we must use local orgId
		setCurrentOrganization(orgId) -- we must set default connection here, session and person search connection will be set based on this auth connection's local connection (local postgre db)
	end
	authTbl = currentAuthTable()
	if url == "/rest/nc/auth/init" then
		return true
	elseif url == "/rest/nc/login" then
		-- dprf.clearCache()
		return true
	elseif url == "/rest/nc/register" then
		return true
	elseif url == "/rest/nc/logout" then -- auth is valid for logout
		return true
	elseif from4d then -- 4d will answer 'no auth token', allow it always
		return true
		-- elseif url == "/rest/nc/soap" then
		-- return true
	elseif url == "/rest/nc/print/pallet-label-old" then
		return true -- TODO: fix auth
	end
	local allow, rec = checkParam(param) -- , url == "/rest/nc/logout" and "logout"
	if allow == false then
		return allow, rec
	end
	local aliasId = dconn.aliasId(rec.organization_id)
	if aliasId then
		rec.organization_id = aliasId
	end
	if not from4d then
		-- util.print("- rest call: organization '%s', url '%s', user '%s', session '%s', valid until '%s', session valid until '%s'", orgId, url, tostring(rec.login_id), tostring(rec.session_id), tostring(param.auth.valid_until), tostring(rec.valid_until))
		if param.auth.valid_until ~= rec.valid_until then
			util.print("- rest call: organization '%s', url '%s', '%s', user '%s', valid until '%s', session valid until '%s'", orgId, url, tostring(param.name or ""), tostring(rec.login_id), tostring(param.auth.valid_until), tostring(rec.valid_until))
		else
			util.print("- rest call: organization '%s', url '%s', '%s', user '%s', valid until '%s'", orgId, url, tostring(param.name or ""), tostring(rec.login_id), tostring(param.auth.valid_until))
		end
	end
	-- local validUntil = dt.dateTimeWithTimeZoneParse(rec.valid_until)
	local validUntil = dt.dateTimeParse(rec.valid_until)
	if url == "/rest/nc/auth/extend" then
		validUntil = validUntil + 60 -- add 60 seconds for validUntil > now to work
	end
	local ret
	local prf = preference()
	local now = prf.logout.use_utc_time and dt.currentDateTimeUtc() or dt.currentDateTime() -- prf.logout.use_utc_time -- now = dt.currentDateTimeUtc()
	if validUntil > now then
		if loc.timeoutHour == nil then
			local pref = preference().logout
			loc.timeoutHour, loc.timeoutMinute, loc.timeoutSecond = pref.timeout.hour, pref.timeout.minute, pref.timeout.second
			util.printOk("- auth timeout hours: %.0f, minutes: %.0f, seconds: %.0f", loc.timeoutHour, loc.timeoutMinute, loc.timeoutSecond)
		end
		validUntil = dt.hmsAdd(now, loc.timeoutHour, loc.timeoutMinute, loc.timeoutSecond)
		validUntil = dt.toString(validUntil)
		local session = {
			organization_id = rec.organization_id,
			session_id = rec.session_id,
			json_data = {language = rec.language},
			valid_until = validUntil,
			record_id = rec.record_id -- needed for update
		}
		ret = {login_id = rec.login_id, valid_until = validUntil, auth_token = rec.session_id, language = rec.language or defaultLanguage(), organization_id = rec.organization_id}
		authTbl = setCurrentAuthTable(ret)
		setCurrentOrganization(rec.organization_id)
		local err = saveAuthRecord(session) -- update session record
		if err then
			if url == "/rest/nc/logout" then -- auth is valid for logout
				return true
			end
			return authError(err)
		end
	elseif url ~= "/rest/nc/logout" then
		-- delete session
		-- dqry.query("", "ses.session_id", "=", param.auth.auth_token, "")
		-- local err = dsave.deleteSelection("ses")
		return authError("session has been expired, please login first", "login", "bright magenta") -- TODO return a code to client to purge localStorage
	end
	--[[ if orgId ~= dconn.organizationId() then
		-- session query and save may have changed current orgId, restore it
		util.printOk("  - call to '%s', login '%s'", orgId, ret.login_id)
		setCurrentOrganization(orgId)
	end ]]
	return true, ret
end

return {
	init = loadLibs,
	querySession = querySession,
	queryUser = queryUser,
	treadCreated = treadCreated,
	setCurrentAuthTable = setCurrentAuthTable,
	currentAuthTable = currentAuthTable,
	-- restoreAuthTable = restoreAuthTable,
	preference = preference,
	-- userPreference = userPreference,
	defaultLanguage = defaultLanguage,
	currentLanguage = currentLanguage,
	setCurrentOrganization = setCurrentOrganization,
	setCurrentUserId = setCurrentUserId,
	currentOrganizationNumber = currentOrganizationNumber,
	currentOrganizationId = currentOrganizationId,
	defaultOrganizationId = defaultOrganizationId,
	currentUser = currentUser,
	currentUserId = currentUserId,
	currentAuthValid = currentAuthValid,
	currentDatabase = currentDatabase,
	userGroup = userGroup,
	userInGroup = userInGroup,
	logout = logout,
	authenticate = authenticate,
	saveAuthRecord = saveAuthRecord,
	sessionRec = sessionRec
}

--[[
local function userPreference(user, userId, groupArr)
	local routePrfName
	loadLibs()
	local prf = dprf.prf("auth/preference_user.json")
	if userId and prf and prf[userId] then
		routePrfName = prf[userId]
	elseif groupArr and groupArr[1] then
		prf = dprf.prf("auth/preference_group.json")
		if prf then
			for _, group in ipairs(groupArr) do
				if prf[group] then
					routePrfName = prf[group]
					break
				end
			end
		end
	end
	if routePrfName then
		prf = dprf.prf("auth/" .. routePrfName .. ".json")
		if prf then
			if user == nil then
				user = {}
			end
			if prf.route and prf.route ~= "" then
				user.route = dprf.prf("auth/" .. prf.route .. ".json")
			end
			if prf.shortcut and prf.shortcut ~= "" then
				user.shortcut = dprf.prf("auth/" .. prf.shortcut .. ".json")
			end
			if prf.menu and prf.menu ~= "" then
				user.menu = dprf.prf("auth/" .. prf.menu .. ".json")
			end
			if prf.output_action and prf.output_action ~= "" then
				user.output_action = dprf.prf("auth/" .. prf.output_action .. ".json")
			end
			if prf.navigation and prf.navigation ~= "" then
				user.navigation = dprf.prf("auth/" .. prf.navigation .. ".json")
			end
		end
	end
	return user
end
 ]]
