--- lib/barcode/quirc.lua
-- https://github.com/dlbeer/quirc
-- todo: test grayscale and flip: https://github.com/defold/extension-qrcode/blob/master/qrcode/src/qrcode.cpp

local quirc = {}

local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local l = require "lang".l

ffi.cdef[[
static const int QUIRC_MAX_REGIONS = 254;
static const int QUIRC_PERSPECTIVE_PARAMS	= 8;
static const int QUIRC_MAX_CAPSTONES= 32;
static const int QUIRC_MAX_GRIDS	 = 8;

typedef uint8_t quirc_pixel_t;
/*
#if QUIRC_MAX_REGIONS < UINT8_MAX // #define UINT8_MAX   0xff
 typedef uint8_t quirc_pixel_t;
#elif QUIRC_MAX_REGIONS < UINT16_MAX
 typedef uint16_t quirc_pixel_t;
#else
 #error "QUIRC_MAX_REGIONS > 65534 is not supported"
#endif
*/

/* This structure describes a location in the input image buffer. */
struct quirc_point {
	int	x;
	int	y;
};

struct quirc_region {
	struct quirc_point	seed;
	int			count;
	int			capstone;
};

struct quirc_capstone {
	int			ring;
	int			stone;

	struct quirc_point	corners[4];
	struct quirc_point	center;
	double			c[QUIRC_PERSPECTIVE_PARAMS];

	int			qr_grid;
};

struct quirc_grid {
	/* Capstone indices */
	int			caps[3];

	/* Alignment pattern region and corner */
	int			align_region;
	struct quirc_point	align;

	/* Timing pattern endpoints */
	struct quirc_point	tpep[3];
	int			hscan;
	int			vscan;

	/* Grid size and perspective transform */
	int			grid_size;
	double			c[QUIRC_PERSPECTIVE_PARAMS];
};

struct quirc {
	uint8_t			*image;
	quirc_pixel_t		*pixels;
	int			w;
	int			h;

	int			num_regions;
	struct quirc_region	regions[QUIRC_MAX_REGIONS];

	int			num_capstones;
	struct quirc_capstone	capstones[QUIRC_MAX_CAPSTONES];

	int			num_grids;
	struct quirc_grid	grids[QUIRC_MAX_GRIDS];
};


/* Obtain the library version string. */
const char *quirc_version(void);

/* Construct a new QR-code recognizer. This function will return NULL
 * if sufficient memory could not be allocated.
 */
struct quirc *quirc_new(void);

/* Destroy a QR-code recognizer. */
void quirc_destroy(struct quirc *q);

/* Resize the QR-code recognizer. The size of an image must be
 * specified before codes can be analyzed.
 *
 * This function returns 0 on success, or -1 if sufficient memory could
 * not be allocated.
 */
int quirc_resize(struct quirc *q, int w, int h);

/* These functions are used to process images for QR-code recognition.
 * quirc_begin() must first be called to obtain access to a buffer into
 * which the input image should be placed. Optionally, the current
 * width and height may be returned.
 *
 * After filling the buffer, quirc_end() should be called to process
 * the image for QR-code recognition. The locations and content of each
 * code may be obtained using accessor functions described below.
 */
uint8_t *quirc_begin(struct quirc *q, int *w, int *h);
void quirc_end(struct quirc *q);


/* This enum describes the various decoder errors which may occur. */
typedef enum {
	QUIRC_SUCCESS = 0,
	QUIRC_ERROR_INVALID_GRID_SIZE,
	QUIRC_ERROR_INVALID_VERSION,
	QUIRC_ERROR_FORMAT_ECC,
	QUIRC_ERROR_DATA_ECC,
	QUIRC_ERROR_UNKNOWN_DATA_TYPE,
	QUIRC_ERROR_DATA_OVERFLOW,
	QUIRC_ERROR_DATA_UNDERFLOW
} quirc_decode_error_t;

/* Return a string error message for an error code. */
const char *quirc_strerror(quirc_decode_error_t err);

/* Limits on the maximum size of QR-codes and their content. */
static const int QUIRC_MAX_BITMAP	= 3917;
static const int QUIRC_MAX_PAYLOAD	= 8896;

/* QR-code ECC types. */
static const int QUIRC_ECC_LEVEL_M     = 0;
static const int QUIRC_ECC_LEVEL_L     = 1;
static const int QUIRC_ECC_LEVEL_H     = 2;
static const int QUIRC_ECC_LEVEL_Q     = 3;

/* QR-code data types. */
static const int QUIRC_DATA_TYPE_NUMERIC       = 1;
static const int QUIRC_DATA_TYPE_ALPHA         = 2;
static const int QUIRC_DATA_TYPE_BYTE          = 4;
static const int QUIRC_DATA_TYPE_KANJI         = 8;

/* Common character encodings */
static const int QUIRC_ECI_ISO_8859_1		= 1;
static const int QUIRC_ECI_IBM437					= 2;
static const int QUIRC_ECI_ISO_8859_2		= 4;
static const int QUIRC_ECI_ISO_8859_3		= 5;
static const int QUIRC_ECI_ISO_8859_4		= 6;
static const int QUIRC_ECI_ISO_8859_5		= 7;
static const int QUIRC_ECI_ISO_8859_6		= 8;
static const int QUIRC_ECI_ISO_8859_7		= 9;
static const int QUIRC_ECI_ISO_8859_8		= 10;
static const int QUIRC_ECI_ISO_8859_9		= 11;
static const int QUIRC_ECI_WINDOWS_874		= 13;
static const int QUIRC_ECI_ISO_8859_13		= 15;
static const int QUIRC_ECI_ISO_8859_15		= 17;
static const int QUIRC_ECI_SHIFT_JIS		= 20;
static const int QUIRC_ECI_UTF_8			= 26;

/* This structure is used to return information about detected QR codes
 * in the input image.
 */
struct quirc_code {
	/* The four corners of the QR-code, from top left, clockwise */
	struct quirc_point	corners[4];

	/* The number of cells across in the QR-code. The cell bitmap
	 * is a bitmask giving the actual values of cells. If the cell
	 * at (x, y) is black, then the following bit is set:
	 *
	 *     cell_bitmap[i >> 3] & (1 << (i & 7))
	 *
	 * where i = (y * size) + x.
	 */
	int			size;
	uint8_t			cell_bitmap[QUIRC_MAX_BITMAP];
};

/* This structure holds the decoded QR-code data */
struct quirc_data {
	/* Various parameters of the QR-code. These can mostly be
	 * ignored if you only care about the data.
	 */
	int			version;
	int			ecc_level;
	int			mask;

	/* This field is the highest-valued data type found in the QR
	 * code.
	 */
	int			data_type;

	/* Data payload. For the Kanji datatype, payload is encoded as
	 * Shift-JIS. For all other datatypes, payload is ASCII text.
	 */
	uint8_t			payload[QUIRC_MAX_PAYLOAD];
	int			payload_len;

	/* ECI assignment number */
	uint32_t		eci;
};

/* Return the number of QR-codes identified in the last processed
 * image.
 */
int quirc_count(const struct quirc *q);

/* Extract the QR-code specified by the given index. */
void quirc_extract(const struct quirc *q, int index,
		   struct quirc_code *code);

/* Decode a QR-code, returning the payload data. */
quirc_decode_error_t quirc_decode(const struct quirc_code *code,
				  struct quirc_data *data);

]]

ffi.cdef[[

/* Dump decoded information on stdout. */
void dump_data(const struct quirc_data *data);

/* Dump a grid cell map on stdout. */
void dump_cells(const struct quirc_code *code);

/* Read a JPEG image into the decoder.
 *
 * Note that you must call quirc_end() if the function returns
 * successfully (0).
 */
int load_jpeg(struct quirc *q, const char *filename);

/* Check if a file is a PNG image.
 *
 * returns 1 if the given file is a PNG and 0 otherwise.
 */
int check_if_png(const char *filename);

/* Read a PNG image into the decoder.
 *
 * Note that you must call quirc_end() if the function returns
 * successfully (0).
 */
int load_png(struct quirc *q, const char *filename);
]]

local lib = util.loadDll("graphics/quirc")

local dataTypeString = {
	[C.QUIRC_DATA_TYPE_NUMERIC] = "NUMERIC",
	[C.QUIRC_DATA_TYPE_ALPHA] = "ALPHA",
	[C.QUIRC_DATA_TYPE_BYTE] = "BYTE",
	[C.QUIRC_DATA_TYPE_KANJI] = "KANJI"
}

local function dumpData(data, ret)
	ret[#ret + 1] =	{
		format = "QR_CODE", -- can't handle anything else
		version = data.version,
		ecc_level = data.ecc_level,
		mask = data.mask,
		data_type = data.data_type,
		data_type_string = dataTypeString[data.data_type] or  "unknown",
		length = data.payload_len,
		barcode = ffi.string(data.payload, data.payload_len),
		eci = data.eci
	}
end

local function dumpInfo(q)
	local count = lib.quirc_count(q)
	local ret = {}
	for i = 0, count - 1 do
		local code = ffi.newNoAnchor("struct quirc_code[1]")
		local data = ffi.newNoAnchor("struct quirc_data[1]")
		lib.quirc_extract(q, i, code)
		local err = lib.quirc_decode(code, data)
		-- lib.dump_cells(code) -- draws barcode as ascii graph
		if err ~= 0 then
			ret.error = l("quirc decoding FAILED: %s", ffi.string(lib.quirc_strerror(err)))
			util.printWarning(ret.error)
		else
			dumpData(data[0], ret) -- lib.dump_data(data)
		end
	end
	return ret
end

function quirc.decodeFile(path)
	local q = lib.quirc_new()
	if ffi.isNull(q) then
		local err = l("can't initialize quirc barcode library")
		util.printError(err)
		return {error = err}
	end
	local status
	if lib.check_if_png(path) ~= 0 then
		status = lib.load_png(q, path)
	else
		status = lib.load_jpeg(q, path)
	end
	if status < 0 then
		lib.quirc_destroy(q)
		local err = l("can't load png or jpeg file '%s'", path)
		util.printWarning(err)
		return {error = err}
	end
	lib.quirc_end(q)
	--[[
	local count = lib.quirc_count(q)
	if count < 1 then
		lib.quirc_destroy(q)
		q = lib.quirc_new()
		if lib.check_if_png(path) ~= 0 then
			status = lib.load_png(q, path)
		else
			status = lib.load_jpeg(q, path)
		end
		lib.quirc_resize(q, 873, 1200)
		lib.quirc_end(q)
		count = lib.quirc_count(q)
	end
	]]
	local retJson = dumpInfo(q)
	lib.quirc_destroy(q)
	return retJson
end

return quirc
