--- lib/barcode/zbar.lua
-- test zbar and field name changes
-- @module barcode

local zbar = {}

local ffi = require "mffi"
local C = ffi.C
local bit = require "bit"
local fs = require "fs"
local l = require "lang".l
local vips = require "vips"
local zbarl = require "barcode/zbar_h".zbar
-- local util = require "util"

--[[
autoreconf -vfi && ./configure --disable-rpath --disable-video --disable-doc --without-qt --without-gtk --without-python --without-imagemagick --with-graphicsmagick --enable-shared --without-x --prefix=/Users/pasi/installed/C/barcode/zbar/build_osx && make clean && make && make install && cp /Users/pasi/installed/C/barcode/zbar/build_osx/lib/libzbar.0.dylib ~/nc/bin/bin_osx_x64/graphics/libzbar.dylib

please verify that the detected configuration matches your expectations:
------------------------------------------------------------------------
X                      --with-x=disabled
pthreads               --enable-pthread=yes
doc                    --enable-doc=yes
v4l                    --enable-video=no
jpeg                   --with-jpeg=yes
Python                 --with-python=no    python
GTK                    --with-gtk=no          Gtk
GObject introspection  --with-gir=no
Qt                     --with-qt=no            Qt
Java                   --with-java=no
Dbus                   --with-dbus=no
GraphicsMagick         --with-graphicsmagick=yes
Enabled codes:         ean databar code128 code93 code39 codabar i25 qrcode sqcode
Disabled codes:        pdf417

        => zbarcam video scanner will *NOT* be built
        => libv4l will *NOT* be used
        => ImageMagick is preferred, as GraphicsMagick doesn't support https
        => GTK support will *NOT* be built
        => the Qt widget will *NOT* be built
        => the Java interface will *NOT* be built
        => the Java unit test will *NOT* be enabled
]]

local function fourcc(a, b, c, d)
	local aa = string.byte(a)
	local bb = string.byte(b)
	local cc = string.byte(c)
	local dd = string.byte(d)
	local val = bit.bor(aa, bit.lshift(bb, 8), bit.lshift(cc, 16), bit.lshift(dd, 24))
	return ffi.cast("unsigned long", val)
  --[[define zbar_fourcc(a, b, c, d)
				((unsigned long)(a) |
        ((unsigned long)(b) << 8) |
        ((unsigned long)(c) << 16) |
        ((unsigned long)(d) << 24))
	]]
end

local function getData(picturePath)
	local path = fs.filePathFix(picturePath)
	if fs.fileExists(picturePath) == false then
		return -- l("file '%s' does not exist")
	end
	local image = vips.Image.new_from_file(path)
	local interpretation = image:interpretation()
	if interpretation ~= "b-w" then
		image = image:colourspace("b-w") -- srgb -> b-w
	end
	image = image:flatten()
	local raw = image:write_to_memory()
	return image:width(), image:height(), raw
end

function zbar.scanImage(picturePath)
	local scanner, image
	local function cleanup()
		if image then
			zbarl.zbar_image_free_data(image)
			zbarl.zbar_image_destroy(image)
		end
		if scanner then
			zbarl.zbar_image_scanner_destroy(scanner)
		end
	end
  --  obtain image data
  local width, height, raw = getData(picturePath)
	if raw == nil then
		cleanup()
		return {error = l("picture '%s' was not found", tostring(picturePath))}
	end
  --  create a reader
	scanner = zbarl.zbar_image_scanner_create()
  --  configure the reader
  --[[local err = zbarl.zbar_image_scanner_set_config(scanner, C.ZBAR_CODE128, C.ZBAR_CFG_ENABLE, 1)
	err = err == 0 and zbarl.zbar_image_scanner_set_config(scanner, C.ZBAR_CODE39, C.ZBAR_CFG_ENABLE, 1) or err
	err = err == 0 and zbarl.zbar_image_scanner_set_config(scanner, C.ZBAR_EAN13, C.ZBAR_CFG_ENABLE, 1) or err
  err = err == 0 and zbarl.zbar_image_scanner_set_config(scanner, C.ZBAR_QRCODE, C.ZBAR_CFG_ENABLE, 1) or err -- only qrcode: C.ZBAR_NONE to 0, C.ZBAR_QRCODE to 1
  -- local err = zbarl.zbar_image_scanner_set_config(scanner, C.ZBAR_NONE, C.ZBAR_CFG_ENABLE, 1)
	if err ~= 0 then
		cleanup()
		return {error = l("zbar_image_scanner_set_config error: %d", err)}
	end
	]]
  --  wrap image data
  image = zbarl.zbar_image_create()
	local formatStr = fourcc('Y','8','0','0') -- ffi.cast("unsigned long*", "Y800") --
  zbarl.zbar_image_set_format(image, formatStr)
  zbarl.zbar_image_set_size(image, width, height)
  zbarl.zbar_image_set_data(image, raw, width * height, nil) -- zbarl.zbar_image_free_data)
  --  scan the image for barcodes
  local symbolCount = zbarl.zbar_scan_image(scanner, image)
	if symbolCount < 0 then
		cleanup()
		return {error = l("zbar_scan_image error: %d", symbolCount)}
	elseif symbolCount == 0 then
		cleanup()
		return {warning = l("zbar did not find any barcodes from file '%s'", tostring(picturePath))}
	end
  --  extract results
	local ret = {}
	local symbol = zbarl.zbar_image_first_symbol(image)
	while ffi.isNotNull(symbol) do
		local typ = zbarl.zbar_symbol_get_type(symbol)
		local data = zbarl.zbar_symbol_get_data(symbol)
		local name = zbarl.zbar_get_symbol_name(typ)
		local rec = {
			barcode_type = ffi.string(name),
			barcode = ffi.string(data)
		}
		ret[#ret + 1] = rec
		-- util.printInfo("decoded %s symbol '%s'", rec.name, rec.data)
		symbol = zbarl.zbar_symbol_next(symbol)
	end
  cleanup()
	return ret
end

function zbar.setVerbosity(verbosity)
	zbarl.zbar_set_verbosity(verbosity)
end

function zbar.version()
	local major = ffi.newNoAnchor("unsigned[1]")
	local minor = ffi.newNoAnchor("unsigned[1]")
	local ret = zbarl.zbar_version(major, minor)
	if ret ~= 0 then
		return nil
	end
	return major[0].."."..minor[0]
end

return zbar