--- lib/barcode/zxing.lua
-- https://github.com/nu-book/zxing-cpp

local zxing = {}

local ffi = require "mffi"
local util = require "util"
local json = require "json"
local l = require "lang".l

if util.isWin() then
	if ffi.arch == "x86" then
		util.loadDll("graphics/libwinpthread-1.dll")
		util.loadDll("graphics/libgcc_s_sjlj-1.dll")
	end
	util.loadDll("graphics/libstdc++-6.dll")
	ffi.load("GdiPlus") -- may not be needed
	util.loadDll("graphics/libZXingGdiPlus.dll")
end
local zxingLib = util.loadDll("graphics/ZXing")

ffi.cdef [[
int decodeBuffer(char *result, uint32_t resultBufferSize, bool fastMode, bool tryRotate, const char *format, unsigned char *buffer, uint32_t bufferLen, uint32_t width, uint32_t height);
int decodePngFile(char *result, uint32_t resultBufferSize, bool fastMode, bool tryRotate, const char *format, const char *path);
]]

local retBufferSize = 8192 -- QRCode max is 7089 characters, data matrix max is 3116 characters
local retBuffer

local function initBuffer()
	if retBuffer == nil then
		retBuffer = ffi.createBuffer(retBufferSize)
		-- buffer MUST be big enough to hold the result, or we will get error -1
	end
end

local function returnJson(ret, path, barcodeFormat)
	local retTbl
	if ret > 0 then
		local str = ffi.string(retBuffer, ret) -- ret is return string length
		str = json.cleanJsonText(str)
		retTbl = {json.fromJson(str)}
	elseif ret == 0 then
		retTbl = {warning = "nothing recognized"}
	elseif ret == -1 then
		retTbl = {["error"] = "failed to read image png file: "..path}
	elseif ret == -2 then
		retTbl = {["error"] = "format '"..barcodeFormat.."' is not valid"}
	else
		retTbl = json.fromJson(l('{"error": "result buffer size is too small, result size is %d, result buffer size is %d"}', math.abs(ret), retBufferSize))
	end
	return retTbl
end

function zxing.decodeBuffer(fastMode, tryRotate, barcodeFormat, buffer, bufferLen, width, height, path)
	initBuffer()
	return returnJson(zxingLib.decodeBuffer(retBuffer, retBufferSize, fastMode, tryRotate, barcodeFormat, buffer, bufferLen, width, height), path, barcodeFormat)
end

function zxing.decodePngFile(fastMode, tryRotate, barcodeFormat, path)
	initBuffer()
	return returnJson(zxingLib.decodePngFile(retBuffer, retBufferSize, fastMode, tryRotate, barcodeFormat, path), path, barcodeFormat)
end

return zxing

--[[ available formats:
AZTEC
CODABAR
CODE_39
CODE_93
CODE_128
DATA_MATRIX
EAN_8
EAN_13
ITF
MAXICODE
PDF_417
QR_CODE
RSS_14
RSS_EXPANDED
UPC_A
UPC_E
UPC_EAN_EXTENSION
]]
