--- zlib.lua
-- zlib binding
-- luapower: https://github.com/luapower/zlib
local ffi = require "mffi"
local util = require "util"
local l = require"lang".l
require "compress/zlib_h"
-- local bit = require "bit"

local uncompressBufferSize --  = util.prf("system/option.json").option.uncompress_buffer_size or 1 * 1000 * 1000 -- == default 1 Mb
local uncompressBuffer -- = ffi.newAnchor('uint8_t[?]', uncompressBufferSize)

local zlib
if ffi.os == "Windows" then
	-- zlib = util.loadDll("zlib.dll")
	zlib = util.loadDll("graphics/zlib1.dll")
elseif ffi.os == "Linux" then
	zlib = util.loadDll("zlib.so")
else
	zlib = util.loadDll("zlib.dylib")
end
local C = zlib

-- see: system/socket.lua
-- local recvbuflen = 128*1024 -- 128kb
-- local sendbuflen = 4*1024*1024 -- x Mb
local defaultBufSize = 128 * 1024 -- 128kb
local defaultBuf = ffi.newAnchor('uint8_t[?]', defaultBufSize)

local function version()
	return ffi.string(C.zlibVersion())
end

local function checkz(ret)
	if ret == 0 then
		return
	end
	return ffi.string(C.zError(ret))
	-- util.printError(ffi.string(C.zError(ret)))
end

local function checkminus1(ret)
	if ret ~= -1 then
		return ret
	end
	util.printError("return value is not -1")
end

local function ptr(o)
	return o ~= nil and o or nil
end

local function flateFunction(api)
	return function(...)
		local ret = api(...)
		if ret == 0 then
			return true
		end
		if ret == C.Z_STREAM_END then
			return false
		end
		checkz(ret)
	end
end

local deflate = flateFunction(C.deflate)
local inflate = flateFunction(C.inflate)

-- FUN TIME: windowBits is range 8..15 (default = 15) but can also be -8..15 for raw deflate with no zlib header
-- or trailer and can also be greater than 15 which reads/writes a gzip header and trailer instead of a zlib wrapper.
-- so I added a format parameter which can be 'deflate', 'zlib', 'gzip' (default = 'zlib') to cover all the cases
-- so that windowBits can express only the window bits in the initial 8..15 range.
-- additionally for inflate, windowBits can be 0 which means use the value in the zlib header of the compressed stream.

local function format_windowBits(format, windowBits)
	if format == 'gzip' then
		windowBits = windowBits + 16
	end
	if format == 'deflate' then
		windowBits = -windowBits
	end
	return windowBits
end

local strm = ffi.newAnchor("z_stream")
local function init_deflate(format, level, method, windowBits, memLevel, strategy)
	level = level or C.Z_DEFAULT_COMPRESSION
	method = method or C.Z_DEFLATED
	windowBits = format_windowBits(format, windowBits or C.Z_MAX_WBITS)
	memLevel = memLevel or 8
	strategy = strategy or C.Z_DEFAULT_STRATEGY

	-- local strm = ffi.newNoAnchor "z_stream"
	checkz(C.deflateInit2_(strm, level, method, windowBits, memLevel, strategy, version(), ffi.sizeof(strm)))
	ffi.gc(strm, C.deflateEnd)
	return strm, deflate
end

local function init_inflate(format, windowBits)
	windowBits = format_windowBits(format, windowBits or C.Z_MAX_WBITS)

	-- local strm = ffi.newNoAnchor "z_stream"
	checkz(C.inflateInit2_(strm, windowBits, version(), ffi.sizeof(strm)))
	ffi.gc(strm, C.inflateEnd)
	return strm, inflate
end

local function inflate_deflate(init)
	return function(read, write, bufsize, ...)
		bufsize = bufsize or 16384
		local stream, flate = init(...)
		local buf
		if bufsize <= defaultBufSize then
			buf = defaultBuf
		else
			buf = ffi.newNoAnchor('uint8_t[?]', bufsize) -- only big messages create new temp bugger
		end
		stream.next_out, stream.avail_out = buf, bufsize
		stream.next_in, stream.avail_in = nil, 0

		local function flush()
			local sz = bufsize - stream.avail_out
			if sz == 0 then
				return
			end
			write(buf, sz)
			stream.next_out, stream.avail_out = buf, bufsize
		end

		local data, size -- data must be anchored as an upvalue!
		while true do
			if stream.avail_in == 0 then -- input buffer empty: refill
				data, size = read()
				if not data then -- eof: finish up
					-- local ret
					repeat
						flush()
					until not flate(stream, C.Z_FINISH)
					flush()
					return
				end
				stream.next_in, stream.avail_in = data, size or #data
			end
			flush()
			if not flate(stream, C.Z_NO_FLUSH) then
				flush()
				return
			end
		end
	end
end

local inflateFunc = inflate_deflate(init_inflate) -- inflate(read, write[, bufsize][, format][, windowBits])
local deflateFunc = inflate_deflate(init_deflate) -- deflate(read, write[, bufsize][, format][, level][, windowBits][, memLevel][, strategy])

-- utility functions

local function compress_tobuffer(data, size, level, buf, sz)
	level = level or -1
	sz = ffi.newNoAnchor('unsigned long[1]', sz)
	checkz(C.compress2(buf, sz, data, size, level))
	return sz[0]
end

local function compressX(data, size, level)
	size = size or #data
	local sz = C.compressBound(size)
	local buf = ffi.newNoAnchor('uint8_t[?]', sz)
	sz = compress_tobuffer(data, size, level, buf, sz)
	return ffi.string(buf, sz)
end

local function uncompress_tobuffer(data, size, buf, sz)
	sz = ffi.newNoAnchor('unsigned long[1]', sz)
	local err = checkz(C.uncompress(buf, sz, data, size))
	if err then
		return nil, l("gzip uncompress_tobuffer error '%s'", tostring(err))
	end
	return sz[0]
end

local function uncompress(data, size, sz)
	size = size or #data
	if size < 5 then
		return nil, l("gzip packet size is too small, less than 5 bytes")
	end
	--[[
	-- this packet size calculation dows not work
	if not sz then
		-- http://www.abeel.be/content/determine-uncompressed-size-gzip-file
		-- data, bodyLen = zlib.gzip(("ABC"):rep(11310017))
		local b4 = string.byte(data:sub(-4, -4))
		local b3 = string.byte(data:sub(-3, -3))
		local b2 = string.byte(data:sub(-2, -2))
		local b1 = string.byte(data:sub(-1, -1))
		-- if not (b1 and b2 and b3 and b4) then
		--	return nil, l("gzip packet size resolution failed, last 4 bytes are not numbers")
		-- end
		sz = bit.bor(bit.lshift(b1, 24), bit.lshift(b2, 16)) + bit.lshift(b3, 8) + b4
		-- same as: sz = bit.lshift(b1, 24) + bit.lshift(b2, 16) + bit.lshift(b3, 8) + b4
	end
	-- if sz < 0 or sz > uncompressBufferSize then --(4 * 1024^3) then -- 10^9 == GB, 1024^3 = GBi
		-- return nil, l("gzip packet size '%s' is invalid", util.formatInt(sz))
	-- end
	print(l("gzip calculated packet size is '%s'", util.formatInt(sz)))
	--]]
	if uncompressBufferSize == nil then
		uncompressBufferSize = util.prf("system/option.json").option.uncompress_buffer_size or 1 * 1000 * 1000 -- == default 1 Mb
	end
	if uncompressBuffer == nil then
		uncompressBuffer = ffi.newAnchor('uint8_t[?]', uncompressBufferSize)
	end
	local err
	sz, err = uncompress_tobuffer(data, size, uncompressBuffer, uncompressBufferSize)
	if err then
		return nil, l("gzip uncompress error '%s'", tostring(err))
	end
	sz = tonumber(sz)
	print(l("gzip uncompressed packet size is %s bytes", util.formatInt(sz)))
	return ffi.string(uncompressBuffer, sz)
end

-- gzip file access functions

local function gzclose(gzfile)
	checkz(C.gzclose(gzfile))
	ffi.gc(gzfile, nil)
end

local function gzopen(filename, mode, bufsize)
	local gzfile = ptr(C.gzopen(filename, mode or 'r'))
	if not gzfile then
		return nil, l("gzip open error '%d'", ffi.errno())
	end
	ffi.gc(gzfile, gzclose)
	if bufsize then
		C.gzbuffer(gzfile, bufsize)
	end
	return gzfile
end

local flush_enum = {none = C.Z_NO_FLUSH, partial = C.Z_PARTIAL_FLUSH, sync = C.Z_SYNC_FLUSH, full = C.Z_FULL_FLUSH, finish = C.Z_FINISH, block = C.Z_BLOCK, trees = C.Z_TREES}

local function gzflush(gzfile, flush)
	checkz(C.gzflush(gzfile, flush_enum[flush]))
end

local function gzread_tobuffer(gzfile, buf, sz)
	return checkminus1(C.gzread(gzfile, buf, sz))
end

local function gzread(gzfile, sz)
	local buf = ffi.newNoAnchor('uint8_t[?]', sz)
	return ffi.string(buf, gzread_tobuffer(gzfile, buf, sz))
end

local function gzwrite(gzfile, data, sz)
	sz = C.gzwrite(gzfile, data, sz or #data)
	if sz == 0 then
		return nil, l("gzip packet size is 0")
	end
	return sz
end

local function gzeof(gzfile)
	return C.gzeof(gzfile) == 1
end

local function gzseek(gzfile, ...)
	local narg = select('#', ...)
	local whence, offset
	if narg == 0 then
		whence, offset = 'cur', 0
	elseif narg == 1 then
		if type(...) == 'string' then
			whence, offset = ..., 0
		else
			whence, offset = 'cur', ...
		end
	else
		whence, offset = ...
	end
	whence = whence == 'set' and 0 or whence == 'cur' and 1
	if not whence then
		util.printError("setting whence failed")
	end
	return checkminus1(C.gzseek(gzfile, offset, whence))
end

local function gzoffset(gzfile)
	return checkminus1(C.gzoffset(gzfile))
end

ffi.metatype('gzFile_s', {__index = {close = gzclose, read = gzread, write = gzwrite, flush = gzflush, eof = gzeof, seek = gzseek, offset = gzoffset}})

-- checksum functions

local function adler32(data, sz, adler)
	adler = adler or C.adler32(0, nil, 0)
	return tonumber(C.adler32(adler, data, sz or #data))
end

local function crc32(data, sz, crc)
	crc = crc or C.crc32(0, nil, 0)
	return tonumber(C.crc32(crc, data, sz or #data))
end

local function gzip(src, level)
	local function reader(s)
		local done
		return function()
			if done then
				return
			end
			done = true
			return s
		end
	end

	local function writer()
		local t = {}
		local i = 0
		return function(data, sz)
			if not data then
				return table.concat(t)
			end
			i = i + 1
			t[i] = ffi.string(data, sz)
		end
	end
	local write = writer()
	deflateFunc(reader(src), write, nil, "gzip", level)
	local dst = write()
	-- io.write("    gzip deflate: "..#src.." / "..#dst..", ")
	return dst, #dst
end

if not ... then
	require 'zlib_test'
end

return {compress = gzip, C = C, version = version, inflate = inflateFunc, deflate = deflateFunc, uncompress_tobuffer = uncompress_tobuffer, uncompress = uncompress, compress_tobuffer = compress_tobuffer, compressX = compressX, open = gzopen, adler32 = adler32, crc32 = crc32}
