--- lib/convert/base85.lua
-- copied from: https://github.com/philanc/plc/blob/master/plc/base85.lua
-- uses this cahrset: https://tools.ietf.org/html/rfc1924
--
-- Copyright (c) 2018  Phil Leblanc  -- see LICENSE file
------------------------------------------------------------------------
--[[

=== z85 - the ZeroMQ variant of Ascii85 encoding

Ascii85 encodes binary strings by using five displayable ascii characters
to represent 4 bytes of data.

The Z85 encoding alphabet is designed to facilitate embedding encoded
strings in source code or scripts (eg. double-quote, single-quote,
backslash are not used).

Specification: https://rfc.zeromq.org/spec:32/Z85/

The Z85 specification makes no provision for padding. The application
must ensure that the length of the string to encode is a multiple of 4.

TODO(?):  add an option to use the RFC 1924 [1] alphabet
which is maybe better because "This character set excludes the
characters "',./:[\] , making it suitable for use in JSON strings" [2]

[1] https://tools.ietf.org/html/rfc1924
[2] https://en.wikipedia.org/wiki/Ascii85


Note: the original Ascii85/btoa will not be implemented. The alphabet is
less convenient that the two variants above, and the special characters
for groups of 4 NULs or 4 spaces are not useful for compressed
and/or encrypted data.
]] --
-- local spack, sunpack = string.pack, string.unpack
local byte, char = string.byte, string.char
local insert, concat = table.insert, table.concat
local floor, format = math.floor, string.format
local util

-- Z85 alphabet - see https://rfc.zeromq.org/spec:32/Z85/
-- local chars = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-:+=^!/*?&<>()[]{}@%$#" -- uses Z85
-- local chars = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ!#$%&()*+-;<=>?@^_`{|}~" -- uses rfc1924
-- ".:/[]" -- Z85 special
-- ";@_`|~" -- rfc1924 special
local chars = "0123456789abcdefghijklmnopqrstuvwxyzABCDEFGHIJKLMNOPQRSTUVWXYZ.-:+=^!/*?&_|()[]{};%$#" -- uses Z85, but without < and > (needs escaping in html) and @, replaced with _|;

-- [[
local function sortChar()
	local charArr = {} -- maps base85 digit ascii representation to their value
	for i = 1, #chars do
		charArr[i] = chars:sub(i, i)
	end
	table.sort(charArr)
	print(table.concat(charArr))
end
sortChar()
-- ]]
-- use rfc1924 but sorted and < and > (needs escaping in html) are replaced `@ with .,
local chars = "!#$%&()*+-./0123456789:;=?ABCDEFGHIJKLMNOPQRSTUVWXYZ[]^_abcdefghijklmnopqrstuvwxyz{|}" -- uses Z85, but sorted

local inv = {} -- maps base85 digit ascii representation to their value
local charArr = {} -- maps base85 digit ascii representation to their value
for i = 1, #chars do
	charArr[i] = chars:sub(i, i)
	inv[byte(chars, i)] = i - 1
end

local function idiv(a, b)
	-- return (a - fmod(a, b)) / b -- (a - a % b) / b, not safe with negative numbers
	return floor(a / b)
end

local function hexToBase85(str)
	local n, r1, r2, r3, r4, r5
	-- #str must be multiple of 8 characters
	if #str % 8 ~= 0 then
		util = util or require "util"
		util.printError("hexToBase85 string length %d must be multiple of 8 characters", #str)
		return nil, "invalid length"
	end
	local et = {} -- used to collect hexToBase85d blocks of 8 characters
	for i = 1, #str, 8 do
		n = tonumber(str:sub(i, i + 7), 16) -- sunpack(">I4", str, i)
		-- ~ 		print(i, n)
		r5 = n % 85
		n = idiv(n, 85)
		r4 = n % 85
		n = idiv(n, 85)
		r3 = n % 85
		n = idiv(n, 85)
		r2 = n % 85
		r1 = idiv(n, 85) -- idiv(n, 85) is always <= 85
		--[[ n = idiv(n, 85)
		r1 = n % 85
		n = idiv(n, 85) ]]
		local eb = char(chars:byte(r1 + 1), chars:byte(r2 + 1), chars:byte(r3 + 1), chars:byte(r4 + 1), chars:byte(r5 + 1))
		insert(et, eb)
	end
	return concat(et)
end

local function base85ToHex(enc)
	local st = {} -- used to collect base85ToHexd blocks of 8 characters
	local n, r
	if #enc % 5 ~= 0 then
		-- hexToBase85d length must be multiple of 5 characters
		util = util or require "util"
		util.printError("base85ToHex string length %d must be multiple of 5 characters", #enc)
		return nil, "invalid length"
	end
	for i = 1, #enc, 5 do
		n = 0
		for j = 0, 4 do
			r = inv[enc:byte(i + j)]
			if r == nil then
				return nil, "invalid char"
			end
			n = n * 85 + r
		end
		local sb = format("%04x", n) -- spack(">I4", n)
		if #sb < 8 then
			sb = ("0000" .. sb):sub(-8) -- add prefix zeros, minimal sb is "0000"
		end
		insert(st, sb)
	end
	return concat(st)
end

return {hexToBase85 = hexToBase85, base85ToHex = base85ToHex}
