--- lib/convert/convert.lua
-- Convert data to another form using json convert description files.
-- @module convert
local convert = {}

local dprf = require "dprf"
local util = require "util"
local json = require "json"
local peg = require "peg"
local pegFound, pegReplace = peg.found, peg.replace
local recdata = require "recdata"
local recData, recDataSet = recdata.get, recdata.set
local fn = require "fn"
local dt = require "dt"
local fs = require "fs"
local l = require"lang".l
local dconn = require "dconn"
local dschema = require "dschema"
local dconv = require "dconv"
local utf = require "utf"

local fieldType, tablePrefix, externalRec, isField = dschema.fieldType, dschema.tablePrefix, dschema.externalRec, dschema.isField
local nameSql = dschema.nameSql
local xml_to_json, convertJsonToXml, json_to_json -- , dconv
local warningChar = "⚠" -- Unicode Character 'WARNING SIGN', http://www.fileformat.info/info/unicode/char/26a0/index.htm
local debug = util.debugger

function convert.setWarningChar(sign)
	local prevWarningChar = warningChar
	warningChar = sign
	return prevWarningChar
end

function convert.warningChar()
	return warningChar
end

function convert.jsonToCss(tbl, name, cr, tab)
	tab = tab or "\t"
	cr = cr or "\n"
	local space = cr == "" and "" or " "
	local txtArr = {"/* " .. (name or "css") .. " */"}
	for class, item in pairs(tbl) do
		txtArr[#txtArr + 1] = "." .. class .. space .. "{"
		for key, val in pairs(item) do
			if key == "font-family" then
				if val == "Helvetica" or val == "Arial" then
					val = "Helvetica," .. space .. "Arial"
				end
				txtArr[#txtArr + 1] = tab .. key .. ":" .. space .. val .. "," .. space .. "sans-serif;"
			else
				txtArr[#txtArr + 1] = tab .. key .. ":" .. space .. val .. ";"
			end
		end
		txtArr[#txtArr + 1] = "}"
	end
	return table.concat(txtArr, cr or "\n")
end

local function addError(err, err2)
	if err2 and err then
		err = err .. "\n" .. err2
	elseif err2 then
		err = err2
	end
	return err
end

local function convertPreference(rec)
	if rec.convert_name and rec.convert_name ~= "" then
		local prfName = rec.convert_name:lower()
		--[[ if peg.find(prfName, "convert/") <= 0 then -- ~= 1 then
			prfName = "convert/" .. prfName
		end ]]
		--[[
		prfName = peg.replace(prfName, "convert_", "")
		prfName = peg.replace(prfName, ".json", "")
		prfName = peg.replace(prfName, " ", "_")
		prfName = "convert_"..prfName..".json"
		]]
		local prf, err = dprf.preferenceFromJson(prfName, "no-cache no-error")
		rec.convert_name = prfName
		if err then
			return prf, err
		end
		-- if util.tableIsEmpty(prf) then
		--	return prf,l("convert preference '%s' was not found", prfName)
		-- end
		return prf
	end
	return nil, l("convert preference name was empty")
end

local function recToDottedFieldNameArray(rec)
	local nameArr = {}
	for fldName, val in pairs(rec) do
		if type(val) == "table" then
			local arr = recToDottedFieldNameArray(val) -- recursive call
			for _, name in ipairs(arr) do
				nameArr[#nameArr + 1] = fldName .. "." .. name
			end
		else
			nameArr[#nameArr + 1] = fldName
		end
	end
	return nameArr
end

local jsonNull, dateFormat, timeFormat, dateTimeFormat -- cache for 1 call only, reset for next call because connection may be different
local function formatFieldData(data, fldType)
	if fldType == "double" or fldType == "integer" or fldType == "bigint" or fldType == "number" then -- rest uses number
		data = tonumber(data) -- TODO: cleaner conversion, clean letters like '1e5' (for exponent) first?, fix also dsql.lua changeValueType()
	elseif fldType == "varchar" or fldType == "text" or fldType == "string" then
		if type(data) == "table" then
			data = json.toJsonRaw(data)
		elseif data == nil then
			return "" -- or "null" ?
		else
			if jsonNull == nil then
				jsonNull = json.jsonNull()
			end
			if data == jsonNull then -- or jsonNull(data)?
				return "" -- or "null" ?
			end
		end
		if fldType == "varchar" then
			data = peg.replace(peg.parseBefore(tostring(data), "\n"), "\t", " ") -- take data before \n, change \t to space
		end
	elseif fldType == "boolean" then
		if data ~= false and data ~= true then
			if data == "" or data == 0 or data == nil then
				data = false
			else
				data = true
			end
		end
	elseif fldType == "date" then
		if type(data) == "number" then
			if not dateFormat then
				dateFormat = dconn.dateFormat()
			end
			data = dt.formatNum(data, dateFormat)
			-- todo: check string format?
		end
	elseif fldType == "timestamp" or fldType == "timestamp with time zone" then
		if not dateTimeFormat then
			dateTimeFormat = dconn.dateTimeFormat()
		end
		if type(data) == "string" then
			data = dt.dateParse(data)
		end
		data = dt.formatNum(data, dateTimeFormat)
	elseif fldType == "time" then
		if not timeFormat then
			timeFormat = dconn.timeFormat()
			if timeFormat == nil then
				timeFormat = false
			end
		end
		if timeFormat and type(data) == "number" then
			data = dt.formatNum(data, timeFormat)
		end
	elseif fldType == "uuid" then
		return data
	elseif fldType == "json" then
		if type(data) == "table" then
			return data
		end
		return nil
	else
		--[[ if type(data) == "table" then
			data = json.toJsonRaw(data) -- todo: check this
		end ]]
		data = tostring(data)
		util.printError("unknown field type '%s', data '%s'", fldType, data)
	end
	return data
end
convert.formatFieldData = formatFieldData

local prevErrorLocalField
local function convertFieldData(arr, query, localFieldArr, convertTo, convertRec, tableName, schema, recordType, restTag)
	-- tableName is needed if convert param is given, if convertTo == "external" then tableNam are mandatory
	local err
	if type(arr) ~= "table" then
		err = util.printError("convert field data array type '%s' is not a table", type(arr))
		return {}, err
	end
	if type(localFieldArr) ~= "table" then
		err = util.printError("convert field data local field array type '%s' is not a table", type(localFieldArr))
		return {}, err
	end
	local result = {}
	local externalFieldRecArr = {}
	local externalFieldArr = {}
	local localFieldTypeArr = {}
	for fldNum, fld in ipairs(localFieldArr) do
		if convertTo == "local" then
			externalFieldArr[fldNum] = nameSql(fld, schema, recordType)
			localFieldTypeArr[fldNum] = fieldType(fld)
		elseif convertTo == "external" then
			externalFieldRecArr[fldNum] = externalRec(fld, schema, recordType)
			externalFieldArr[fldNum] = externalFieldRecArr[fldNum] and externalFieldRecArr[fldNum].field_name
		else
			externalFieldArr[fldNum] = fld
		end
		-- local tag = convertRec and convertRec[fld] and convertRec[fld].tag
	end

	local mainTablePrefix = tablePrefix(tableName) .. "."
	local externalField, externalFieldRec, value, localFieldNoPrefix
	local allDataTag = query and query.parameter and query.parameter.all_data_tag -- usually json_data
	local defaultValueRecFld = query.default_value and query.default_value[tableName]
	local defaultValueRec
	for row, rec in ipairs(arr) do
		--[[ if type(rec) ~= "table" then
			util.printError("convertFieldData: record is not a table")
			return {}, err
		end ]]
		result[row] = {}
		if defaultValueRecFld and convertTo == "external" then
			defaultValueRec = {}
			local err2 = dconv.setDefaultValueRecord(rec, defaultValueRec, query.default_value[tableName], true) -- sets .static, .tag and .code
			err = addError(err, err2)
			if util.tableIsEmpty(defaultValueRec) then
				defaultValueRec = nil
			end
		end
		for fldNum, localField in ipairs(localFieldArr) do
			value = nil
			if peg.startsWith(localField, mainTablePrefix) then
				localFieldNoPrefix = peg.parseAfter(localField, mainTablePrefix)
			else
				localFieldNoPrefix = localField
			end
			externalField = externalFieldArr[fldNum]
			if externalField == nil then
				if prevErrorLocalField ~= localField then
					util.printWarning("local field '%s' can't be converted to external field", localField)
					prevErrorLocalField = localField
				end
				-- do not convert if external name does not exist, but parameter localFieldArr should contain only exportable fields of external table
			else
				if convertRec == nil and convertTo == "external" then
					externalFieldRec = externalFieldRecArr[fldNum]
					if externalFieldRec.default_value ~= nil then
						value = externalFieldRec.default_value -- default_value overrides local rec value but does not override convertRec or defaultValueRec
					end
				end
				if convertRec then
					-- TODO: when this is used?, why dconv.setDefaultValueRecord() in a loop?
					local defaultValueRec2
					defaultValueRec2, err = recData(convertRec, localField, false) -- false 3. param is 'do not show error'
					local defaultValue = defaultValueRec2 and defaultValueRec2.default_value
					if defaultValue then
						local defaultRec = {}
						dconv.setDefaultValueRecord(rec, defaultRec, defaultValue) -- sets .static, .copy_value and .code
						if not util.tableIsEmpty(defaultRec) then
							value, err = recData(defaultRec, localField)
							if value == nil then
								local fldWithoutPrefix = peg.parseAfter(localField, ".")
								value, err = recData(defaultRec, fldWithoutPrefix)
								if value ~= nil then
									util.printWarning("default value record should have prefixed local field name, tag '%s', field '%s', json:\n%s", fldWithoutPrefix, localField, json.toJson(defaultValueRec2))
								end
							end
						end
					end
				end
				if defaultValueRec then
					local defaultValue
					defaultValue, err = recData(defaultValueRec, localField)
					if defaultValue == nil then
						local fldWithoutPrefix = peg.parseAfter(localField, ".")
						defaultValue, err = recData(defaultValueRec, fldWithoutPrefix)
						if defaultValue ~= nil then
							util.printWarning("query default value record should have prefixed local field name, tag '%s', field '%s', json:\n%s", fldWithoutPrefix, localField, json.toJson(defaultValueRec))
						end
					end
					if defaultValue ~= nil then
						value = defaultValue
					end
				end
				if externalField == restTag then -- restTag is usually json_data
					value = rec
				end
				if value == nil then
					if convertTo == "external" then
						value, err = recData(rec, localField)
						if value == nil then
							value, err = recData(rec, localFieldNoPrefix)
							if value == nil and localField:sub(-1) == "2" and peg.found(localField, "json_data") then
								value, err = recData(rec, localField:sub(1, -2))
								if value == nil then
									value, err = recData(rec, localFieldNoPrefix:sub(1, -2))
								end
							end
						end
					else
						if type(rec) ~= "table" then
							err = util.printWarning("convert array row %d value '%s' type '%s' is not a table", row, tostring(rec), type(rec))
						else
							value, err = recData(rec, externalField)
						end
					end
				end
				if err then
					break
				end
				if type(value) == "number" and value < 0 and convertRec == nil and convertTo == "external" then
					externalFieldRec = externalFieldRecArr[fldNum]
					if externalFieldRec.positive_value then
						value = -value
					end
				end
				if convertTo == "external" then
					if value ~= nil then
						value = formatFieldData(value, externalFieldRecArr[fldNum].field_type)
						err = recDataSet(result[row], externalField, value)
						if allDataTag then
							err = recDataSet(result[row], allDataTag .. "." .. externalField, value)
						end
					end
				elseif value ~= nil then
					local tag = localFieldNoPrefix
					if convertRec and convertRec[localField] and convertRec[localField].tag then
						tag = convertRec[localField].tag
						if peg.startsWith(tag, mainTablePrefix) then
							tag = peg.parseAfterStart(tag, mainTablePrefix)
						end
					end
					if type(value) == "table" then
						if next(value) == nil then
							value = nil
						else
							local extRec = externalRec(localField, schema, recordType)
							if extRec == nil then
								if not peg.found(localField, "json_data") then
									err = util.printRed("data convert external record was not found for local field '%s'", localField)
								end
							elseif extRec and extRec.json_tag then
								local value2 = recData(value, extRec.json_tag)
								if value2 == nil then
									err = util.printRed("tag '%s' was not found from local field '%s', external field '%s'", extRec.json_tag, localField, extRec.field)
								else
									value = formatFieldData(value2, localFieldTypeArr[fldNum])
								end
							end
						end
					end
					if err then
						break
					end
					if externalField ~= restTag then
						value = formatFieldData(value, localFieldTypeArr[fldNum])
					end
					err = recDataSet(result[row], tag, value)
					if allDataTag then
						err = recDataSet(result[row], allDataTag .. "." .. tag, value)
					end
				else
					local localValue
					localValue, err = recData(result[row], localFieldNoPrefix)
					if localValue == nil and query.parameter and query.parameter.warning_char then -- do not override table data
						err = recDataSet(result[row], localFieldNoPrefix, warningChar) -- , "use-tag") -- "⚠" -- Unicode Character 'WARNING SIGN', http://www.fileformat.info/info/unicode/char/26a0/index.htm
					end
				end
			end
			if err then
				break
			end
		end
		if defaultValueRecFld and convertTo == "local" then
			defaultValueRec = {}
			local err2 = dconv.setDefaultValueRecord(result[row], defaultValueRec, query.default_value[tableName], true) -- sets .static, .tag and .code
			err = addError(err, err2)
			if util.tableIsEmpty(defaultValueRec) then
				defaultValueRec = nil
			end
		end

		if allDataTag then
			local localValue
			if type(rec) ~= "table" then
				util.printError("convert field data: record is not a table, record: %s", rec)
			else
				for field, val in pairs(rec) do
					localValue = recData(result[row], allDataTag .. "." .. field)
					if localValue == nil then
						recDataSet(result[row], allDataTag .. "." .. field, val)
					end
				end
			end
		end
	end
	return result, err
end

function convert.convertExternalDataToLocal(dataArr, query, restParam, info)
	if type(dataArr) ~= "table" then
		util.printError("convert external data to local data array type '%s' is not a table", type(dataArr))
		return {}
	end
	if type(query) ~= "table" or type(query.field) ~= "table" then
		util.printError("convert external data to local query field array type '%s' is not a table", type(query and query.field))
		return {}
	end
	local prevWarningChar, err, err2
	if query.parameter.warning_char then
		prevWarningChar = convert.setWarningChar(query.parameter.warning_char)
	end
	local mainData
	local tableArr = query.table or query.save or query.delete
	local mainTable = tableArr[1].main_table or tableArr[1].table
	local mainTablePrefix = tablePrefix(mainTable)
	local schema
	if query.schema then
		schema = query.schema
	else
		schema = dconn.schema()
	end
	local restTag = restParam.query_parameter.full_rest_answer_tag
	if restTag then
		restTag = peg.parseAfterStart(restTag, mainTablePrefix .. ".")
	end
	-- for _, dataArr in ipairs(ret) do
	local recordType
	for tblNum, tblRec in ipairs(tableArr) do
		if tblNum == 1 then -- main table
			recordType = tblRec.record_type[1]
			if tblRec.main_tag then
				if #dataArr > 1 then
					util.printRed("main table '%s' main tag '%s', data array is bigger than 1", tblRec.table, tblRec.main_tag)
				end
				dataArr = dataArr[1][tblRec.main_tag] -- what about other rows
			end
			-- set full ret rec to mainData[i][restTag], restTag is for example ord.json_data.additional_data
			if query.parameter and query.parameter.use_external_field then
				mainData, err2 = dataArr, nil
			else
				mainData, err2 = convertFieldData(dataArr, query, query.field, "local", tblRec.convert, tblRec.table, schema, recordType, restTag)
			end
			err = addError(err, err2)
			if err2 then
				break
			end
			if #mainData ~= #dataArr then
				util.printError("convert convertFieldData() did not return same amount of rows as result data has")
				break
			end
		end
		if tblRec.row_tag and (tblRec.column or tblRec.fields) then
			local prefix = tablePrefix(tblRec.table) -- local schema
			if type(tblRec.column) == "string" then
				tblRec.column = dprf.prf(tblRec.column)
			end
			if tblRec.field == nil and tblRec.column and tblRec.column.column then
				tblRec.field = fn.iter(tblRec.column.column):map(function(rec2)
					local fld = rec2.field and peg.startsWith(rec2.field, prefix .. ".") and rec2.field or nil -- or rec2.variable
					if fld and (rec2.tag or rec2.default_value) then
						tblRec.convert = tblRec.convert or {}
						tblRec.convert[fld] = {tag = rec2.tag, default_value = rec2.default_value}
					end
					return fld
				end):totable()
			end

			-- here dataArr and mainData have same amount of rows in same order, it is ok to use i in both
			if type(tblRec.row_tag) == "table" then
				for i, rowRec in ipairs(dataArr) do
					local arr = {}
					for j, rowTag in ipairs(tblRec.row_tag) do
						if type(rowRec[rowTag]) ~= "table" then
							util.printError("query data row_tag '%s' is empty, preference '%s'", rowTag, tostring(query.name))
						else
							arr[j], err2 = convertFieldData(rowRec[rowTag], query, tblRec.field, "local", tblRec.convert, tblRec.table, schema, recordType)
							err = addError(err, err2)
						end
					end
					mainData[i][prefix] = util.arrayCombine(arr)
				end
			else
				local arr
				if tblRec.field == nil then
					util.printError("field array is nil, preference '%s'", tostring(query.name))
				else
					for row, rowRec in ipairs(dataArr) do
						if #rowRec[tblRec.row_tag] == 0 then
							arr, err2 = convertFieldData({rowRec[tblRec.row_tag]}, query, tblRec.field, "local", tblRec.convert, tblRec.table, schema, recordType)
							mainData[row][prefix] = arr[1]
						else
							arr, err2 = convertFieldData(rowRec[tblRec.row_tag], query, tblRec.field, "local", tblRec.convert, tblRec.table, schema, recordType)
							mainData[row][prefix] = util.arrayCombine({mainData[row][prefix] or {}, arr})
						end
						err = addError(err, err2)
					end
				end
			end
		end
	end
	-- end

	-- set full ret rec to mainData[i][restTag], restTag is for example ord.json_data.additional_data
	restTag = restParam.query_parameter.full_rest_answer_tag
	if restTag then
		restTag = peg.parseAfterStart(restTag, mainTablePrefix .. ".")
		for i, rowRec in ipairs(dataArr) do
			recDataSet(mainData[i], restTag, rowRec) -- json.toJson(rowRec)) -- restTag is usually json_data
		end
	end
	if prevWarningChar then
		convert.setWarningChar(prevWarningChar)
	end
	return {data = mainData, info = info, error = err}
end

local function tableExportFields(tableName, schema, recordType)
	local locaFieldArrGen = dschema.schemaFieldArrayGen(tableName, schema, recordType) -- test all schema fields that have local_field defined
	return locaFieldArrGen:totable()
end

local function convertLocalRecDataToExternal(arr, query, defaultValueRec, tbl, schema, recordType)
	-- local tblPrefix = tablePrefix(tbl)
	local externalFieldDefinition = tableExportFields(tbl, schema, recordType)
	local localFieldName = fn.iter(externalFieldDefinition):map(function(rec)
		return rec.local_field
	end):totable()
	local ret, err = convertFieldData(arr, query, localFieldName, "external", defaultValueRec, tbl, schema, recordType)
	return ret, err
end

local function convertLocalDataToExternal(arr, query)
	-- param query is query or save preference
	local ret = {}
	local tbl, recordType, tblPrefix
	local tableArr = query.table
	local schema
	if query.schema then
		schema = query.schema
	else
		schema = dconn.schema()
	end
	for i, tableInfoRec in ipairs(tableArr) do
		if i == 1 then
			tbl = tableInfoRec.main_table or tableInfoRec.table -- check this
			recordType = tableInfoRec.record_type
			tblPrefix = tablePrefix(tbl)
			ret = convertLocalRecDataToExternal(arr, query, nil, tbl, schema, recordType) -- dsave has already set main table default values
			if tableInfoRec.row_table then
				local rowTable, rowTablePrefix, tagName
				for _, rowInfoRec in ipairs(tableInfoRec.row_table) do
					if type(rowInfoRec) == "table" then
						rowTable = rowInfoRec.table
						rowTablePrefix = tablePrefix(rowInfoRec.table)
						tagName = rowInfoRec.tag_name or rowTablePrefix
					else -- only string
						rowTable = rowInfoRec
						rowTablePrefix = tablePrefix(rowTable)
						tagName = rowTablePrefix
					end
					for rowNum, rowRec in ipairs(arr) do
						if rowRec[rowTablePrefix] then
							-- rowRec[tblPrefix][rowTablePrefix] is row_table array, not a record
							-- local defaultValueRec = rowInfoRec.default_value and rowInfoRec.default_value[rowTable]
							ret[rowNum][tagName] = convertLocalRecDataToExternal(rowRec[rowTablePrefix], query, nil, rowTable, schema, recordType)
						else
							util.printWarning("row_table '%s' data key ['%s']['%s'] does not exit in data[%d], preference '%s'", rowTable, tblPrefix, rowTablePrefix, rowNum, tostring(query.name))
							break
						end
					end
				end
			end
		else
			util.printWarning("check/fix many table -tag setting in this code, preference '%s'", tostring(query.name))
			debug()
		end
	end
	return ret
end
convert.convertLocalDataToExternal = convertLocalDataToExternal

local function paramToData(param, dataType, fileType)
	local fileName, err
	if type(param.convert_definition) == "string" then
		param.convert_definition = convertPreference({convert_name = param.convert_definition})
	end
	if param.convert_definition and dataType == "convert" then -- definition in lua table
		if param.convert_definition.convert == nil then
			err = l("add 'convert' tag to conversion definition")
			return nil, err
		else
			return param.convert_definition
		end
	elseif param.in_data and dataType == "in" then -- data in lua table
		return util.clone(param.in_data)
	elseif param.convert_file and dataType == "convert" then -- definition in file
		fileName = param.convert_file
	elseif param.in_file and dataType == "in" then -- data in file path
		fileName = param.in_file
	else
		err = l("wrong parameter in conversion")
		return nil, err
	end

	local data
	local fileTxt = util.readFile(fileName)
	if fileType == "json" then
		data, err = json.fromJson(fileTxt)
		if err then
			err = l("json error in file '%s'\n", fileName) .. err
			return nil, err
		end
	else -- if fileType == "xml" then
		data = fileTxt
	end
	if not data then
		return nil, err
	end
	return data
end

local function jsonToXml(param)
	local rec, attr
	local function tagStartCallback(currentRec, currentAttr)
		rec = currentRec
		attr = currentAttr
	end
	if not convertJsonToXml then
		convertJsonToXml = require "convert/json-to-xml"
	end
	local convertDefinition, inData, ok, txt, err
	convertDefinition, err = paramToData(param, "convert", "json")
	if not convertDefinition then
		return nil, err
	end
	inData, err = paramToData(param, "in", "json")
	if not inData then
		return nil, err
	end
	-- txt, err = jsonToXml.convert(convertDefinition, inData, param.option, param.data) -- txt is xml text
	if util.fromEditor() then
		txt, err = convertJsonToXml.convert(tagStartCallback, convertDefinition, inData, param.option, param.data) -- txt is xml text
		ok = err == nil
	else
		ok, txt, err = pcall(convertJsonToXml.convert, tagStartCallback, convertDefinition, inData, param.option, param.data) -- txt is xml text
	end
	if ok ~= true then
		local function recError(err2)
			if attr then
				err2 = l("error in attribute '%s'\n%s\ntag '%s' ", attr, err2, rec.tag or "no tag -key")
			else
				err2 = l("error in tag '%s' \n%s", rec.tag or "no tag -key", tostring(err))
			end
			return err2
		end
		if err then
			if rec then
				err = recError(err)
			end
		elseif rec then
			err = recError(txt)
		else
			err = txt
		end
		err = peg.replace(err, ".lua", "") -- don't show we are using lua
	end
	if param.out_file then
		util.writeFile(param.out_file, txt)
	end
	return txt, err -- return xml text
end

local function xmlTextToJson(convertDefinition, xmlText, param)
	if not xml_to_json then
		xml_to_json = require "convert/xml-to-json"
	end
	local ok, err, ret, txt

	if util.fromEditor() then
		ret, err = xml_to_json.convert(convertDefinition, xmlText, param.option) -- , param.data)
		ok = err == nil
	else
		-- ret, err = xml_to_json.convert(convertDefinition, xmlText) -- ret is json table
		ok, ret, err = pcall(xml_to_json.convert, convertDefinition, xmlText, param.option, param.data)
		-- if not ret then
	end
	if not ok then
		if not err then
			err = ret
		end
		local pos = peg.find(err, "\nstack traceback")
		if pos > 0 then
			err = err:sub(1, pos - 1)
		end
		err = peg.replace(err, ".lua", "") -- don't show we are using lua
		return nil, err
	end
	if param.option and param.option.return_table == true then
		return ret, err
	end
	if param.out_file then
		txt, err = json.toJson(ret)
		if not txt then
			return nil, err
		end
		util.writeFile(param.out_file, txt)
	end
	return ret, err -- return json table
end

local function xmlToJson(param)
	local convertDefinition, inData, err
	convertDefinition, err = paramToData(param, "convert", "json")
	if not convertDefinition then
		return nil, err
	end
	inData, err = paramToData(param, "in", "xml") -- xml text
	if not inData then
		return nil, err
	end
	return xmlTextToJson(convertDefinition, inData, param.option) -- , param.data)
end

local function jsonToCsv(convertParam)
	local columnSeparator = convertParam.convert_definition.column_separator
	local columnSeparatorReplace = convertParam.convert_definition.column_separator_replace
	local lineBreak = convertParam.convert_definition.line_break
	local charReplaceArr = convertParam.convert_definition.char_replace
	local allowEmptyValue = convertParam.convert_definition.allow_empty_value
	local charsetConvert = convertParam.convert_definition.charset
	local round = util.roundFunction(convertParam.convert_definition.round_decimals or 15)
	if charsetConvert == "cp1252" then
		charsetConvert = utf.utf8ToCp1252
	elseif charsetConvert == "latin9" then
		charsetConvert = utf.utf8ToLatin9
	elseif charsetConvert then
		util.printError("unknow charset '%s', charset must be 'cp1252' or 'latin9'", tostring(charsetConvert))
		charsetConvert = nil
	end
	local arr = convertParam.in_data
	local convertArr = {}
	for _, conv in ipairs(convertParam.convert_definition.convert) do
		local item = {from = conv.from}
		--[[ if i == 1 and arr[1][item.from] == nil then -- we should test for main table, but for simplicity we set "from": "" to save.json
			item.from = ""
		end ]]
		if type(conv.from_column) == "string" then
			item.column = dprf.prf(conv.from_column)
		else
			item.column = conv.from_column
		end
		convertArr[#convertArr + 1] = item
	end

	local ret = {}
	local function exportCsvRow(item, colPref)
		local data
		for _, col in ipairs(colPref) do
			if col.export ~= false and (col.field or col.variable or col.static_export) then
				if col.static_export then
					data = col.static_export
				else
					data = recData(item, col.field or col.variable, false)
				end
				if data == nil then
					if allowEmptyValue then
						if allowEmptyValue == "warning" then
							if col.field then
								util.printWarning("csv export data was not found from field '%s'", col.field)
							else
								util.printWarning("csv export data was not found from variable '%s'", col.variable)
							end
						end
						data = ""
					else
						if col.field then
							return nil, util.printRed("csv export data was not found from field '%s', record: %s", col.field, item)
						else
							return nil, util.printRed("csv export data was not found from variable '%s', record: %s", col.variable, item)
						end
					end
				else
					if col.format and not col.static_export then
						-- data = formatFieldData(data, col.format)
						if col.type == "date" or col.field and dschema.fieldTypeLua(col.field) then
							data = dt.formatString(data, col.format)
						end
					end
					if type(data) == "string" then
						if data ~= "" then
							if not col.static_export then
								data = pegReplace(data, columnSeparator, columnSeparatorReplace)
								for _, replace in ipairs(charReplaceArr) do
									if pegFound(data, replace.from) then -- easier to debug changes
										data = pegReplace(data, replace.from, replace.to)
									end
								end
							end
							if charsetConvert then
								data = charsetConvert(data)
							end
						end
					elseif type(data) == "number" then
						if data ~= 0 then
							data = tostring(round(data))
						else
							data = tostring(data)
						end
					else
						data = tostring(data)
					end
				end
				ret[#ret + 1] = data .. columnSeparator
			end
		end
		ret[#ret + 1] = lineBreak
	end

	local err
	for _, item in ipairs(arr) do
		for _, conv in ipairs(convertArr) do
			if conv.from == "" then
				err = exportCsvRow(item, conv.column.column)
				if err then
					return nil, err
				end
			else
				local arr2 = recData(item, conv.from)
				if arr2 then
					for _, rowItem in ipairs(arr2) do
						err = exportCsvRow(rowItem, conv.column.column)
						if err then
							return nil, err
						end
					end
				end
			end
		end
	end
	return table.concat(ret)
end

local function exportToType(exportType, convertParam, param, data)
	local start = util.seconds()
	if convertParam then
		if type(convertParam) == "string" then
			-- convertParam = dprf.prf(convertParam)
			local newParam = {}
			newParam.convert_definition = convert.convertPreference({convert_name = convertParam})
			if newParam.convert_definition.main_data_tag then
				newParam.in_data = {[newParam.convert_definition.main_data_tag] = data or param}
			else
				newParam.in_data = data or param
			end
			convertParam = newParam
		end
		local definition = convertParam.convert_definition
		if definition then
			if definition.save_path then
				local ret, err
				if exportType == "xml" then
					ret, err = jsonToXml(convertParam)
				elseif exportType == "csv" then
					ret, err = jsonToCsv(convertParam)
				end
				if ret then -- err == nil
					if definition.save_path_mac and util.isMac() then
						definition.save_path = definition.save_path_mac
					end
					fs.createPath(peg.parseBeforeLast(definition.save_path, "/"))
					local path = peg.parseBeforeLast(definition.save_path, ".") .. "-" .. dt.currentFileString() .. "." .. peg.parseAfterLast(definition.save_path, ".")
					fs.writeFile(path, ret)
					start = util.seconds(start)
					path = util.print("saved %s file to %s, creation time %.3f seconds", exportType, path, start)
					if err then
						err = tostring(err)
						util.printError(err)
						return {message = path, error = err}
					end
					return {message = path}
				end
			end
			return {error = "convert parameter definition save_path does not exist"}
		end
		return {error = "convert parameter definition does not exist"}
	end
	return {error = "convert parameter does not exist"}
end

local function exportXml(convertParam, param, data)
	return exportToType("xml", convertParam, param, data)
end

local function exportCsv(convertParam, param, data)
	return exportToType("csv", convertParam, param, data)
end

local function jsonToJson(param)
	if not json_to_json then
		json_to_json = require "convert/json_to_json"
	end
	local convertDefinition, inData, ok, err, ret
	convertDefinition, err = paramToData(param, "convert", "json")
	if not convertDefinition then
		return nil, err
	end
	inData, err = paramToData(param, "in", "json")
	if not inData then
		return nil, err
	end
	ok, ret, err = pcall(json_to_json.convert, convertDefinition, inData, param.option, param.data)
	-- ret, err = json_to_json.convert(convertDefinition, inData) -- ret is json table
	if not ok then
		if err then
			return nil, err
		end
		return nil, ret -- ret is err
	end
	if param.option and param.option.return_table == true then
		return ret, err
	end

	local pretty
	if param.option and param.option.pretty then
		pretty = param.option.pretty
	else
		pretty = true
	end
	local txt
	if pretty then
		txt, err = json.toJson(ret)
	else
		txt, err = json.toJsonRaw(ret)
	end
	if param.out_file then
		if not txt then
			return nil, err
		end
		util.writeFile(param.out_file, txt)
	end
	return txt, err -- return json table
end

local function tagToArray(arr, tagName, filterTagName, filterTagValue)
	local ret = {}
	for _, rec in ipairs(arr) do
		if rec[tagName] then
			if not filterTagName then
				ret[#ret + 1] = rec[tagName]
			elseif rec[filterTagName] == filterTagValue then
				ret[#ret + 1] = rec[tagName]
			end
		end
	end
	return ret
end

local function filter(arr, filterTagName, filterTagValue)
	local ret = {}
	local isFunc = type(filterTagName) == "function"
	for _, rec in ipairs(arr) do
		if isFunc then
			if filterTagName(rec) then
				ret[#ret + 1] = rec
			end
		elseif rec[filterTagName] == filterTagValue then
			ret[#ret + 1] = rec
		end
	end
	return ret
end

local function convertAndSave(fromTablePrefix, toTablePrefix, rec, convertPrf, extraData) -- pref = mg.constantTbl(), wpaPref = mg.wpa.constantTbl()
	if type(convertPrf) ~= "table" then
		convertPrf = convert.convertPreference({convert_name = convertPrf}) -- "convert/wpa/new_record.json"
	end
	local convertParam = {
		convert_definition = convertPrf,
		in_data = {[fromTablePrefix] = {util.clone(rec)}},
		option = {return_table = true, create_empty_tag = true},
		data = util.clone(extraData)
		-- option.create_not_defined_tag = true
		-- option = {debug = false, create_empty_tag = false},
	}
	local ret, err = convert.jsonToJson(convertParam)
	if err then
		err = l("converting failed, error:") .. "\n" .. err
	end
	if ret[toTablePrefix] ~= nil then
		ret = ret[toTablePrefix]
	end
	return ret, err
end

local function convertPreferenceToFieldTbl(preferenceName, mainTblPrefix, rowTblPrefixArr)
	local preference
	if type(preferenceName) == "string" then
		preference = util.prf(preferenceName)
	end
	if type(preference) ~= "table" or util.tableIsEmpty(preference) then
		return nil, nil, nil, l("convert preference '%s' was not found", tostring(preferenceName))
	end
	if type(preference.convert) ~= "table" then
		return nil, nil, nil, l("preference '%s' has no convert tag", tostring(preferenceName))
	end
	local convertPrf = preference.convert
	mainTblPrefix = mainTblPrefix or {}
	if type(mainTblPrefix) ~= "table" then
		mainTblPrefix = {mainTblPrefix}
	end
	rowTblPrefixArr = rowTblPrefixArr or {}
	if type(rowTblPrefixArr) ~= "table" then
		rowTblPrefixArr = {rowTblPrefixArr}
	end
	local mainTbl = {}
	local rowTbl = {}
	local linkedTbl = {}

	local function addFieldToTbl(tblPrefix, field)
		local function add(tbl, fld)
			tbl[tblPrefix] = tbl[tblPrefix] or {}
			if fn.index(fld, tbl[tblPrefix]) == nil then
				tbl[tblPrefix][#tbl[tblPrefix] + 1] = fld
			end
		end
		if fn.index(tblPrefix, mainTblPrefix) then
			add(mainTbl, field)
		elseif fn.index(tblPrefix, rowTblPrefixArr) then
			add(rowTbl, field)
		else
			add(linkedTbl, field)
		end
	end

	local function fields(tbl)
		for _, rec in ipairs(tbl) do
			if type(rec) == "string" then
				if isField(rec) then
					addFieldToTbl(tablePrefix(rec), rec)
				end
			elseif type(rec) == "table" then
				fields(rec)
			end
		end
	end
	fields(convertPrf)
	return mainTbl, rowTbl, linkedTbl
end

local funcMap = {recToDottedFieldNameArray = recToDottedFieldNameArray, jsonToJson = jsonToJson, jsonToXml = jsonToXml, exportXml = exportXml, exportCsv = exportCsv, xmlToJson = xmlToJson, xmlTextToJson = xmlTextToJson, convertPreference = convertPreference, tagToArray = tagToArray, filter = filter, convertAndSave = convertAndSave, convertPreferenceToFieldTbl = convertPreferenceToFieldTbl}
for key, func in pairs(funcMap) do
	if convert[key] then
		util.printError("convert.%s already exists", key)
	else
		convert[key] = func
	end
end
return convert

--[=[
local function toExternalFieldData(arr, tableName, schema, recordType)
	local result = {}
	if #arr < 1 then
		return result
	end
	local localFieldPartName = {}
	local externalFieldPartName = {}
	local externalFieldType = {}
	local defaultValue = {}
	local fldCount = 0
	local localRec = arr[1]
	local locaFieldArr = tableExportFields(tableName, schema, recordType)
	fn.iter(locaFieldArr):each(function(rec)
		local localFieldArr = dschema.localFieldPart(rec) -- cached peg.splitToArray(rec.local_field, ".")
		local recFirstPart = localRec[localFieldArr[1]]
		if rec.default_value ~= nil or localRec[localFieldArr[2]] ~= nil or (recFirstPart ~= nil and ((#localFieldArr == 2 and recFirstPart[localFieldArr[2]] ~= nil) or (#localFieldArr > 2 and recFirstPart[localFieldArr[2]][localFieldArr[3]] ~= nil))) then
			-- we assume that all records have same fields as not nil -value
			-- look 2 levels for normal fields pr.name
			-- look only 3 levels for tables like pr.json_data.name
			fldCount = fldCount + 1
			defaultValue[fldCount] = rec.default_value
			externalFieldType[fldCount] = rec.field_type -- cached peg.splitToArray(rec.field_name, ".")
			externalFieldPartName[fldCount] = dschema.fieldNamePart(rec) -- cached peg.splitToArray(rec.field_name, ".")
			localFieldPartName[fldCount] = localFieldArr
		end
	end)
	if fldCount > 0 then
		-- todo: this is a general part, combine with convertFieldData()
		local dataPtr, ptr, partName
		for row, localRec2 in ipairs(arr) do
			result[row] = {}
			for fldNum = 1, fldCount do
				-- check for flat record first, must not contain table prefix subrecord
				dataPtr = localRec2[localFieldPartName[fldNum][1]] == nil and localRec2[localFieldPartName[fldNum][2]]
				if dataPtr == nil or type(dataPtr) == "table" then
					local startPartNum = 1
					if type(dataPtr) == "table" then
						startPartNum = 2
					end
					dataPtr = localRec2
					for partNum = startPartNum, #localFieldPartName[fldNum] - 1 do
						partName = localFieldPartName[fldNum][partNum]
						if dataPtr[partName] == nil then
							dataPtr[partName] = {}
						end
						dataPtr = dataPtr[partName]
					end
					if #localFieldPartName[fldNum] > 0 then
						dataPtr = dataPtr[localFieldPartName[fldNum][#localFieldPartName[fldNum]]] or defaultValue[fldNum]
					else
						dataPtr = defaultValue[fldNum]
					end
				end
				if dataPtr ~= nil then
					ptr = result[row]
					for partNum = 1, #externalFieldPartName[fldNum] - 1 do
						partName = externalFieldPartName[fldNum][partNum]
						if ptr[partName] == nil then
							ptr[partName] = {}
						end
						ptr = ptr[partName]
					end
					partName = externalFieldPartName[fldNum][#externalFieldPartName[fldNum]]
					ptr[partName] = formatFieldData(dataPtr, externalFieldType[fldNum])
				end
			end
		end
	end
	dateFormat, timeFormat, dateTimeFormat = nil, nil, nil -- reset for next call because connection may be different
	return result
end
]=]
