--- lib/convert/emf2svg.lua
--
local ffi = require("ffi")
local util = require("util")

local lib
if util.isWin() then
	require "draw/cairo_load_win"
	--[[
	ffi.load("vcruntime140.dll")
	util.loadDll("graphics/emf2svg/brotlicommon.dll")
	util.loadDll("graphics/emf2svg/brotlidec.dll")
	util.loadDll("graphics/emf2svg/bz2.dll")
	util.loadDll("graphics/emf2svg/zlib1.dll")
	util.loadDll("graphics/emf2svg/libpng16.dll")
	util.loadDll("graphics/emf2svg/iconv-2.dll")
	util.loadDll("graphics/emf2svg/libexpat.dll")
	util.loadDll("graphics/emf2svg/freetype.dll")
	util.loadDll("graphics/emf2svg/fontconfig-1.dll")
	lib = util.loadDll("graphics/emf2svg/emf2svg.dll")
	]]
	lib = util.loadDll("graphics/emf2svg")
else
	lib = util.loadDll("graphics/emf2svg")
end
ffi.cdef [[
// structure containing generator param
typedef struct {
    // SVG namespace (the '<something>:' before each fields)
    char *nameSpace;
    // Verbose mode, output fields and fields values if True
    bool verbose;
    // Handle emf+ records or not
    bool emfplus;
    // draw svg document delimiter or not
    bool svgDelimiter;
    // height of the target image
    double imgHeight;
    // width of the target image
    double imgWidth;
} generatoroption;
// convert function
int emf2svg(char *content, size_t length, char **out, size_t *out_length, generatoroption *option);
// check if emf file contains emf+ records
int emf2svg_is_emfplus(char *content, size_t length, bool *is_emfp);
// scan a list of directories to build a font index (index[<FONT_NAME>] =
// <FONT_PATH>)
int emf2svg_gen_font_index(char **font_paths, void *font_index);
// free the font index
int emf2svg_free_font_index(void *font_index);
]]

local function emf2svg(param)
	-- param.width = 0
	-- param.height = 0
	-- param.verbose = false
	-- param.emfplus = true -- Handle emf+ records or not
	-- param.emf = nil -- emf file content as string
	-- param.input = nil -- file path
	-- param.output = nil-- file path
	if param.input == nil and param.emf == nil then
		return nil, util.printRed("missing emf (content) and input (file path) argument")
	end
	if param.output == nil then
		-- return nil, util.printRed("missing --output=FILE argument")
		param.output = param.input:sub(1, -4) .. "svg"
	end
	local content = param.emf
	if content == nil then
		local in_file = io.open(param.input, "r")
		if not in_file then
			return nil, util.printRed("could not open input file '" .. param.input .. "'")
		end
		content = in_file:read("*a")
		in_file:close()
	end
	local svgOut = ffi.new("char *[0]")
	local svgLength = ffi.new("size_t[0]")
	local option = ffi.new("generatoroption")
	-- option.nameSpace = (char *)"svg"
	option.verbose = param.verbose or false
	option.emfplus = param.emfplus or true -- luacheck: ignore
	option.svgDelimiter = true
	option.imgWidth = param.width or 0
	option.imgHeight = param.height or 0
	local ret = lib.emf2svg(ffi.cast("char *", content), #content, svgOut, svgLength, option)
	if ret == 1 then
		local svg = ffi.string(svgOut[0], svgLength[0])
		if param.output then
			local out_file = io.open(param.output, "w")
			if not out_file then
				print("[ERROR] Impossible to open output file '" .. param.output .. "'")
				return 1
			end
			out_file:write(svg)
			out_file:close()
			return util.print("emf2svg, wrote file: " .. param.output)
		end
		return svg
	end
	return nil, util.printRed("emf2svg error: " .. ret)
end

return emf2svg
