--- lib/convert/image-convert.lua
-- test: test/graphics/image-convert-test.lua
-- C++: bin_build_guide/cad-picture/nc-image-convert/nc-image-convert.cpp
--
local ffi = require("mffi")
local util = require("util")
local fs = require("fs")
local peg = require("peg")

local imageConvert = {}
local tempPath

local lib
if util.isWin() then
	util.loadDll("clang/libunwind.dll")
	util.loadDll("clang/libc++.dll")
end
lib = util.loadDll("graphics/libnc-image-convert")
if lib == nil then
	return imageConvert
end
ffi.cdef [[
typedef struct {
	const char* emf_file_path;
	const char* png_file_path;
	char* error_text;
	uint32_t error_length;
	uint32_t max_width;
	uint32_t max_height;
	uint32_t dpi;
	uint32_t original_width;
	uint32_t original_height;
	double aspect_ratio;
	uint32_t higher_res_width;
	uint32_t higher_res_height;
	uint32_t new_width;
	uint32_t new_height;
	uint8_t verbose;
} nc_image_covert_param_t;
void convertEmfToPng(nc_image_covert_param_t* param);
]]

local convertParam = ffi.metatype("nc_image_covert_param_t", {
	__gc = function(struct)
		-- print("convertParam __gc")
		ffi.freeChar(struct.emf_file_path, "no-error")
		ffi.freeChar(struct.png_file_path, "no-error")
		-- lib.convertEmfToPng() did malloc() to struct.error, we must free it
		-- print("convertParam __gc error_length", struct.error_length)
		ffi.freeChar(struct.error_text, "no-error")
	end
})

local function convertEmfToPng(param)
	--[[ if not util.isWin() and not util.isWine() then
		return nil, util.printRed("convert emf to png works only in Windows")
	end ]]
	if param.input == nil then
		return nil, util.printRed("missing input (file path) argument")
	end
	if not param.no_input and fs.fileExists(param.input) == false then
		return nil, util.printRed("input file '%s' does not exist", tostring(param.input))
	end
	if param.output == nil then
		if param.input:sub(-4):lower() ~= ".emf" then
			return nil, util.printRed("input file '%s' does not end with '.emf'", tostring(param.input))
		end
		param.output = param.input:sub(1, -4) .. "png"
	end
	local struct = convertParam() -- create C param struct
	struct.emf_file_path = ffi.mallocChar(param.input)
	struct.png_file_path = ffi.mallocChar(param.output)
	struct.max_width = tonumber(param.max_width) or 200
	struct.max_height = tonumber(param.max_width) or 150
	struct.dpi = tonumber(param.dpi) or 120
	struct.verbose = 0 -- param.verbose == true and 1 or tonumber(param.verbose) or 0 -- command line version is always verbose, but we don't wan C lib to print anything
	lib.convertEmfToPng(ffi.cast("nc_image_covert_param_t *", struct))
	local ret = {input = ffi.string(struct.emf_file_path), output = ffi.string(struct.png_file_path)}
	ret.original_width, ret.original_height, ret.aspect_ratio = struct.original_width, struct.original_height, struct.aspect_ratio -- uint32_t values to lua number
	ret.higher_res_width, ret.higher_res_height, ret.dpi = struct.higher_res_width, struct.higher_res_height, struct.dpi
	ret.new_width, ret.new_height = struct.new_width, struct.new_height
	if struct.error_length < 0 then
		ret.error = util.printRed("emf to png conversion failed from '%s' to '%s', error length %d is negative", ret.input, ret.output, struct.error_length)
	elseif struct.error_length > 0 then
		ret.error = util.printRed("emf to png conversion failed from '%s' to '%s' with error: '%s'.", ret.input, ret.output, ffi.string(struct.error_text, struct.error_length))
	else
		if param.verbose then
			util.print("  - file %s to %s", ret.input, ret.output)
			util.print("  - emf  width: %d px, height: %d px, aspect ratio: %.1f", struct.original_width, struct.original_height, struct.aspect_ratio)
			util.print("  - draw width: %d px, height: %d px, struct.dpi: %d", struct.higher_res_width, struct.higher_res_height, struct.dpi)
			util.print("  - new  width: %d px, height: %d px", struct.new_width, struct.new_height)
		end
	end
	return ret
end

function imageConvert.emfPathToPngPath(emfPath)
	if type(emfPath) ~= "string" or emfPath:lower():sub(-4) ~= ".emf" then
		return {error = util.printRed("emf to png conversion file name '%s' does not end with '.emf'", tostring(emfPath))}
	end
	local output = emfPath:sub(1, -5) .. ".png"
	if fs.fileExists(output) then
		return {output = output}
	end
	if tempPath == nil then
		tempPath = util.mainPath() .. "temp/"
		fs.createFilePath(tempPath)
	end
	output = tempPath .. peg.parseAfterLast(output, "/")
	local ret = convertEmfToPng({input = emfPath, output = output})
	return ret
end

return imageConvert
