--- lib/convert/json-to-xml.lua
-- converts json to xml using json convert table
-- use: dload.selectionToStructureTable(prf.main_table, prf.row_table, prf.linked_table) -- ("inv", {"inr"}, {"co", "top", "soc", "em"})
-- @module json_to_xml
local ffi = require "mffi"
local util = require "util"
local json = require "json"
local peg = require "peg"
local xmlWrite = require "xml/xml_write"
local l = require"lang".l
local dt = require "dt"
local execute = require "execute"
local recData = require"recdata".get
local recDataSet = require"recdata".set -- use set instead of setDot because tags can contain []
-- local fn = require "fn"
local escapeXml = require"xml".escapeXml
local format = string.format

--[[
local function tostring(txt)
	if type(txt) == "table" then
		return "tostring-table"
	elseif type(txt) == "string" then
		return txt
	elseif type(txt) == "number" then
		return "tostring-number"
	else
		return "tostring-type?"
	end
end

local allowedFunctions = {
	round = util.round,
	multiply = util.multiply,
	concatenate = util.concatenate,
	currentUtcString = dt.currentUtcString,
}
]]

local function convert(tagStartCallback, convertDefinition, inJson, option, extraDataRec)
	local convertJson = convertDefinition.convert
	if type(extraDataRec) ~= "table" then
		extraDataRec = {}
	end
	-- add tables to extraDataRec
	if type(inJson) == "table" then
		extraDataRec = util.tableCombine(extraDataRec, inJson, "no-error")
	end
	-- add convertDefinition extra tags to extraDataRec
	for tag, val in pairs(convertDefinition) do
		if tag ~= "convert" and tag ~= "name" and tag ~= "line_break" and not peg.startsWith(tag, "save_path") then
			recDataSet(extraDataRec, tag, val)
		end
	end
	local createEmptyTag
	if convertDefinition.create_empty_tag ~= nil then
		util.print("convertDefinition: %s", convertDefinition.create_empty_tag)
		createEmptyTag = convertDefinition.create_empty_tag
	elseif option and option.create_empty_tag ~= nil then
		createEmptyTag = option.create_empty_tag
	else
		createEmptyTag = false
	end
	local debug = option and option.debug or false
	local sameTagSeparate = true
	if option and option.same_tag_separate ~= nil then
		sameTagSeparate = option.same_tag_separate
	end
	-- local namespace, file
	local endTagStarted = false
	local errorText
	local prevTag
	local count = 0
	local depth = 0
	local outDepth = 0
	local doc = xmlWrite.new()
	local out = {}
	local errorArr = {}
	local errorLoopTag = {}
	local lineBreak = ""

	local startAttr = peg.pattern "[" -- attribute list starts with [
	local endAttr = peg.pattern "]" -- attribute list ends with ]
	local notEndAttr = (1 - endAttr) ^ 0
	local attrPattern = startAttr * notEndAttr * endAttr
	out[#out + 1] = '<?xml version="1.0" encoding="UTF-8"?>\n'

	-- http://luapower.com/genx
	-- http://www.tbray.org/ongoing/genx/docs/Guide.html#Limitations
	-- only UTF8 encoding supported
	-- no empty element tags
	-- no <!DOCTYPE> declarations (write it yourself before calling w:start_doc())
	-- no pretty-printing (add line breaks and indentation yourself with w:text() where needed)
	local mainDataRec
	local function toAttributeArray(dataRec, attribute)
		local ret = {}
		for key, val in pairs(attribute) do
			if type(val) == "string" and peg.found(val, ".") then
				val = recData(dataRec, val) or recData(mainDataRec, val) or val -- should we format data?
			end
			ret[#ret + 1] = key .. '="' .. val .. '"'
		end
		table.sort(ret)
		return ret
	end

	-- if not outFile then
	local space = string.char(9) -- tab, or "  " -- 2 spaces
	doc:start_doc(function(s, sz)
		if type(s) == "cdata" then
			s = sz and ffi.string(s, sz) or ffi.string(s) -- '\n!EOF\n'
		end
		if s == nil then
			util.printWarning(l("json_to_xml callback data type is nil")) -- usually some error?
			-- if there are lots of these, then error may be disabled?
		elseif type(s) ~= "string" then
			util.printError(l("callback data type '%s' is not a string", type(s)))
		else
			-- print(s) -- for trace
			local str
			if s == "</" then -- start of end tag
				endTagStarted = true
				if prevTag == ">" then
					str = "\n" .. space:rep(outDepth - 1) .. s -- this will be changed to "</" at the end of code if end tag is same as start tag
				else
					str = s
				end
			elseif prevTag == ">" and s == "<" then -- start tag
				str = "\n" .. space:rep(outDepth) .. s
				outDepth = outDepth + 1
			elseif endTagStarted and s == ">" then
				outDepth = outDepth - 1
				str = s
				endTagStarted = false
			elseif s == "<" then -- start tag
				outDepth = outDepth + 1
				str = s
			else
				str = s
			end
			prevTag = s
			if str ~= "" then
				out[#out + 1] = str
			end
		end
	end)

	local function tagError(tag, rec, errorType)
		if rec.field then
			tag = rec.field .. " -> " .. tag -- rec.tag
			-- elseif rec.tag then
			-- tag = rec.field.." -> "..rec.tag
		end
		local loop
		if #errorLoopTag > 0 then
			local ret = {}
			for i, rec2 in ipairs(errorLoopTag) do
				ret[i] = l("%s, row %s", tostring(rec2.tag), tostring(rec2.row))
			end
			loop = table.concat(ret, " - ")
		end
		if errorType == "mandatory" then
			if loop then
				errorArr[#errorArr + 1] = l("no data in mandatory tag '%s', %s", tostring(tag), tostring(loop))
			else
				errorArr[#errorArr + 1] = l("no data in mandatory tag '%s'", tostring(tag))
			end
		else -- never here?
			if loop then
				errorArr[#errorArr + 1] = l("'%s' error in tag '%s', %s", errorType, tostring(tag), tostring(loop))
			else
				errorArr[#errorArr + 1] = l("'%s' error in tag '%s'", errorType, tostring(tag))
			end
		end
	end

	local function closeAllTags()
		for _ = 1, depth do
			doc:end_element()
		end
		depth = 0
	end

	local function closeTag(pathArr, start)
		if type(pathArr) == "string" then
			depth = depth - 1
			doc:end_element()
			local prevPath = peg.parseBeforeLast(pathArr, "/") -- pathArr is prevPath string
			if prevPath == pathArr or depth < 1 then
				if pathArr:sub(1, 1) == "/" then
					-- prevPath = pathArr
					return prevPath
				end
				return nil
			end
			return prevPath
		end
		if pathArr == nil then
			util.printError("xml close tag path content is nil")
			return
		else
			for i = start, #pathArr do
				local tag = pathArr[i]
				depth = depth - 1
				if depth < 0 then
					depth = 0
					util.printError(l("closeTag error, depth %d, tag '%s'", depth, tostring(tag)))
				else
					doc:end_element()
				end
			end
		end
	end

	local function debugKey(rec)
		local key
		if rec.from_static then
			key = "from_static"
		elseif rec.from_data then
			key = "from_data '" .. tostring(rec.from_data) .. "'"
		elseif rec.from_preference then
			key = "from_preference '" .. tostring(rec.from_preference) .. "'"
		elseif rec.from_function then
			key = "from_function '" .. tostring(rec.from_function) .. "'"
		elseif rec.field then
			key = "field '" .. tostring(rec.field) .. "'"
		else
			key = "*** error: unknown from ***"
		end
		return key
	end

	local function openTag(rec, prevPath, value, dataRec, closeLastTag)
		tagStartCallback(rec)
		if rec.not_in_use == true then
			return prevPath
		end
		if type(rec.tag) ~= "string" then
			errorArr[#errorArr + 1] = util.printRed("xml definition tag -key '%s' is not a string, definition: %s", tostring(rec.tag), rec)
			return prevPath
		end
		local currentPath = rec.tag
		local pathArr = peg.splitToArray(currentPath, "/")
		if #pathArr > 0 and pathArr[#pathArr]:sub(1, 2) == "{{" then
			-- do not openTag for runner code: {{asd}} = "asd", return prevPath
			return prevPath
		end
		local prevPathArr = peg.splitToArray(prevPath, "/")
		--[[ if currentPath == "ProcessPurchaseOrder/PurchaseOrder/Property/NameValue[@name=\"your\" @unitCode=\"PCE\"]" then
			currentPath = rec.tag
		end ]]
		local sameStart = 0
		for i, tag in ipairs(pathArr) do
			local prevTag2
			if i <= #prevPathArr then
				prevTag2 = prevPathArr[i]
			else
				prevTag2 = ""
			end
			if tag:sub(1, 2) == "{{" then
				tag = tag:sub(1) -- for debug, tag = "{{" -- do not export tags starting with {{
			elseif (i == 1 and tag ~= prevTag2 and prevTag2 ~= "") then
				sameStart = 1
			elseif sameTagSeparate and sameStart < 2 and tag == prevTag2 and i == #pathArr - 1 and #pathArr == #prevPathArr then
				--[[
					- change this:
					<Property>
						<NameValue name="our" unitCode="PCE">T140520/Anne</NameValue>
						<NameValue name="your" unitCode="PCE">IM Blossom pos</NameValue>
					</Property>

					- to this:
					<Property>
						<NameValue name="our" unitCode="PCE">T140520/Anne</NameValue>
					</Property>
					<Property>
						<NameValue name="your" unitCode="PCE">IM Blossom pos</NameValue>
					</Property>
				]]
				local nextTag = pathArr[i + 1]
				local pos1 = peg.find(nextTag, startAttr)
				if pos1 > 0 then
					local prevNextTag = prevPathArr[i + 1]
					local pos2 = peg.find(prevNextTag, startAttr)
					if pos1 == pos2 then
						local noAttr = peg.removePattern(nextTag, attrPattern)
						local prevNoAttr = peg.removePattern(prevNextTag, attrPattern)
						if noAttr == prevNoAttr then
							sameStart = 3
						end
					end
				end
			end
			if tag:sub(1, 2) == "{{" then
				util.printWarning("tag starts with {{")
				-- tag = tag:sub(1) -- for debug
				-- tag = prevTag2
			elseif sameStart < 2 and tag == prevTag2 then
				if sameStart == 0 then
					sameStart = 1
				end
			else
				if sameStart == 1 or sameStart == 3 then
					sameStart = 2
					if i <= #prevPathArr then
						closeTag(prevPathArr, i) -- todo: local ret,err = closeTag()
					end
				end
				-- remove attributes from tags
				local attribute
				local attributePos = peg.find(tag, startAttr)
				if attributePos > 0 then -- contains attribute(s)
					local attrTxt = tag:sub(attributePos + 1, -2) -- remove start and end []
					tag = tag:sub(1, attributePos - 1) -- tag before [
					attrTxt = peg.replace(attrTxt, "@", "") -- remove @ marks, we don't know what they are
					attribute = peg.splitToArray(attrTxt, " ")
				end
				if rec.attribute then
					if attribute == nil then
						attribute = toAttributeArray(dataRec, rec.attribute)
					else
						attribute = util.arrayConcat(attribute, toAttributeArray(dataRec, rec.attribute))
					end
				end
				--[[
				local prevAttributePos = peg.find(prevTag2, startAttr)
				if prevAttributePos > 0 then -- contains attribute(s)
					prevTag2 = prevTag2:sub(1, prevAttributePos - 1) -- prevTag2 is unused
				end ]]
				depth = depth + 1
				doc:start_element(tag)
				if attribute then
					local attribute2 = {}
					local count2
					local prevMismatch = false
					for _, attr in ipairs(attribute) do -- combine quoted attributes like "PRODUCT OPERATIONS"
						count2 = peg.countOccurrence(attr, '"')
						if count2 % 2 == 0 then
							attribute2[#attribute2 + 1] = attr
							prevMismatch = false
						else
							if prevMismatch then
								attribute2[#attribute2] = attribute2[#attribute2] .. " " .. attr
								prevMismatch = false
							else
								attribute2[#attribute2 + 1] = attr
								prevMismatch = true
							end
						end
					end
					attribute = attribute2
					for _, attr in ipairs(attribute) do
						local pos = peg.find(attr, "=")
						local key = attr:sub(1, pos - 1)
						local valueStr = attr:sub(pos + 1)
						local codeStr = peg.parseBetweenWithDelimiter(valueStr, "{{", "}}")
						if codeStr then
							-- {{vat_code}}
							-- local codeStr2 = codeStr:sub(3, -3) -- peg.replace(codeStr:sub(3, -3), "'", '"') -- codeStr:sub(3, -3):gsub('\\"', '"')
							valueStr = recData(dataRec, codeStr) -- execute.runCode(codeStr2, dataRec, nil, option)
							if valueStr == nil then
								-- example: { "tag": "FIELD[FldRef=\"PrdRef\" FldType=\"20\" FldValue=\"{{serial_number}}\"]", "from_data": "serial_number" }
								local codeStr2 = peg.parseBetweenWithoutDelimiter(codeStr, "{{", "}}")
								valueStr = tostring(recData(dataRec, codeStr2))
								if valueStr then -- and rec.result == codeStr2 then
									-- example: {"tag": "FIELD[FldRef=\"Thickness\" FldType=\"100\" FldValue=\"{{result}}\"]", "result": "result", "code": [{ "function": "parse", "parameter": ["pr.subgroup", "*", 1], "result": "result" }]}
									value = nil -- do not set field value, it's used in the attribute
								end
							end
							if type(valueStr) ~= "string" then
								tagError(tag, rec, l("execute code return is not a string: '%s'", tostring(valueStr)))
								valueStr = ""
							end
						end
						local value2 = escapeXml(peg.replace(valueStr, '"', ""))
						-- print(tag, "attr "..i, key, value)
						-- tagStartCallback(rec, valueStr)
						tagStartCallback(rec, value2)
						doc:add_attr(key, value2) -- escaped before
					end
				end
				if i == #pathArr then -- last element
					count = count + 1
					if not rec.content then
						if rec.mandatory then
							if value == nil or value == "" and not rec.from_static then
								tagError(tag, rec, "mandatory")
							end
						end
						if value then
							if createEmptyTag then
								if debug then
									local key = debugKey(rec)
									if type(value) == "string" then
										doc:text(key .. ": '" .. escapeXml(value) .. "'")
									else
										doc:text(key .. ": " .. tostring(value))
									end
								else
									if type(value) == "string" then
										doc:text(escapeXml(value))
									elseif type(value) == "number" then
										doc:text(tostring(value))
									elseif type(value) == "boolean" then
										doc:text(tostring(value))
									else
										local key = debugKey(rec)
										util.printError(l("key '%s' value type '%s' is invalid", key, type(value)))
									end
								end
							elseif type(value) == "string" then
								if value ~= "" then
									if debug then
										local key = debugKey(rec)
										doc:text(key .. ": '" .. escapeXml(value) .. "'")
									else
										doc:text(escapeXml(value))
									end
								end
							else
								if debug then
									local key = debugKey(rec)
									doc:text(key .. ": '" .. escapeXml(tostring(value)) .. "'")
								else
									doc:text(escapeXml(tostring(value)))
								end
							end
						end
					end
				end

			end
		end
		if currentPath:sub(1, 1) ~= "/" and prevPath ~= "" then
			return prevPath .. "/" .. currentPath
		end
		if closeLastTag then
			return closeTag(currentPath) -- todo: local err = closeTag()
		end
		return currentPath
	end

	--[[
	local function closeToPreviousLevel(path, prevPath)
		if path:sub(1, 1) ~= "/" then
			local count = peg.countOccurrence(path, "/") + 1
			for i = 1, count do
				prevPath = closeTag(prevPath)
			end
			return prevPath
		end
		local pathArr = peg.splitToArray(path, "/")
		local prevPathArr = peg.splitToArray(prevPath, "/")
		local start = 0
		-- local startCompare = #prevPathArr - #pathArr
		for i, path2 in ipairs(pathArr) do
			-- if path == prevPathArr[startCompare + i] then
			if path2 == prevPathArr[i] then
				start = start + 1
			else
				break
			end
		end
		if start < 2 then
			errorText = l "closeToPreviousLevel error: json definition can't have content in the first level"
			util.printError(errorText)
			return path
		else
			if start == #pathArr then
				closeTag(pathArr, start) -- closeTag(prevPathArr, start) --
			else
				closeTag(pathArr, start + 1)
			end
			table.remove(pathArr)
		end
		return table.concat(pathArr, "/")
	end
	]]

	local otherDecimalSeparator
	local prevPath = ""
	local closeLastTag = true
	local function createOut(convertArr, dataRec)
		local err

		local function loopArray(convertRec, dataArr)
			--[[ if peg.found(convertRec.tag, "SellerPartyDetails") then
				convertRec.tag = convertRec.tag .. ""
			end ]]
			errorLoopTag[#errorLoopTag + 1] = {tag = convertRec.field}
			local prevDepth = depth
			for j, rec in ipairs(dataArr) do
				if err then
					break
				end
				if depth == 0 then
					mainDataRec = rec
				end
				errorLoopTag[#errorLoopTag].row = j
				local errCount = #errorArr
				prevPath, err = openTag(convertRec, prevPath, dataArr, dataRec, false) -- openTag() will create full path only in first call, other calls will create only latest tag - do not close last tag
				if errCount ~= #errorArr then
					err = errorArr[#errorArr]
				else
					createOut(convertRec.content, rec) -- recursive call
				end
				prevPath = closeTag(prevPath) -- close only 1 latest level
			end
			for _ = 1, depth - prevDepth do -- first openTag() in loop may create more levels but we closed only latest
				prevPath = closeTag(prevPath)
			end
			if prevPath == nil and depth > 0 then -- error
				return nil, l("error in calling code '%s': previous path is nil", tostring(json.toJson(convertRec)))
			end
			table.remove(errorLoopTag)
		end

		for _, convertRec in ipairs(convertArr) do
			if err then
				break
			end

			local function getData(data, from)
				local value = recData(data, from)
				if value == nil and convertRec.code then
					local saveData = dataRec
					err = execute.runCode(convertRec.code, dataRec, saveData, option)
					if err then
						return nil, l("error in calling code '%s':  '%s'", tostring(json.toJson(convertRec.code)), err)
					end
					convertRec._used = true
					value = recData(data, from)
					data[from] = nil
				end
				if type(value) == "number" and convertRec.data_type ~= "date" then
					if convertRec.format then
						value = format(convertRec.format, value)
					else
						value = tostring(value)
					end
					if not peg.found(value, extraDataRec.decimal_separator) then
						if otherDecimalSeparator == nil then
							if extraDataRec.decimal_separator == "," then
								otherDecimalSeparator = "."
							else
								otherDecimalSeparator = ","
							end
						end
						if peg.found(value, otherDecimalSeparator) then
							value = peg.replace(value, otherDecimalSeparator, extraDataRec.decimal_separator)
						end
					end
				elseif convertRec.format then
					if type(value) == "string" and convertRec.data_type == "date" then
						value = dt.formatString(value, convertRec.format)
					elseif type(value) == "number" and convertRec.data_type == "date" then
						value = dt.formatNum(value, convertRec.format)
					else
						errorArr[#errorArr + 1] = util.printRed("json to xml format '%s' is not supported, value: '%s', definition: %s", tostring(convertRec.format), type(value) ~= "table" and tostring(value) or value, convertRec)
					end
				end
				return value -- value can be nil
			end

			--[[ if convertRec.tag and peg.found(convertRec.tag, "RowVatCode") then
				convertRec.tag = convertRec.tag .. ""
			end ]]
			if convertRec.tag and convertRec.tag:sub(1, 1) ~= "/" then
				if prevPath == nil then
					util.printError("convert/json_to_xml createOut() prevPath is nil")
					prevPath = ""
				end
				convertRec.tag = prevPath .. "/" .. convertRec.tag -- openTag() code needs full paths to work correctly
			end
			if convertRec.field == "" then
				convertRec._used = true
				local data = getData(extraDataRec, convertRec.field)
				prevPath, err = openTag(convertRec, prevPath, data, dataRec, closeLastTag)
			elseif convertRec.field then
				local dataValue = getData(dataRec, convertRec.field)
				if dataValue == nil and convertRec.field == "" then
					prevPath, err = openTag(convertRec, prevPath, dataValue, dataRec, false) -- this tag will be closed at the end of convert
				else
					convertRec._used = true
					if err == nil and convertRec.content and type(dataValue) == "table" then
						loopArray(convertRec, dataValue)
					elseif err == nil then
						if (dataValue and dataValue ~= "") or createEmptyTag or convertRec.create_empty_tag then
							prevPath, err = openTag(convertRec, prevPath, dataValue, dataRec, closeLastTag)
						end
					end
				end
			elseif convertRec.from_data then -- not in use in new json-format?
				convertRec._used = true
				local data = getData(extraDataRec, convertRec.from_data)
				prevPath, err = openTag(convertRec, prevPath, data, dataRec, closeLastTag)
				-- end
			elseif convertRec.from_function then
				convertRec._used = true
				local data, funcErr = execute.runFunction(convertRec.from_function, dataRec, convertRec.parameter)
				-- local ok, data = pcall(allowedFunctions[convertRec.from_function])
				-- if ok == false then
				if funcErr then
					return nil, l("error in calling function '%s':  '%s'", tostring(funcErr), convertRec.from_function)
				end
				if data == nil then
					if convertRec.from_default then
						data = convertRec.from_default
					else
						return nil, l("error in calling function '%s'", convertRec.from_function)
					end
				elseif type(data) == "table" then
					if data[1] then
						data = data[1]
					else
						data = ""
					end
				end
				prevPath, err = openTag(convertRec, prevPath, data, dataRec, closeLastTag)
			elseif convertRec.from_preference then
				convertRec._used = true
				-- export_invoice.term_of_delivery[ord.term_of_delivery]
				local data
				data, err = execute.fromPreference(convertRec, dataRec)
				if err then
					return nil, err
				end
				prevPath, err = openTag(convertRec, prevPath, data, dataRec, closeLastTag)
			elseif convertRec.code then
				local saveData = dataRec -- {}
				err = execute.runCode(convertRec.code, dataRec, saveData, option)
				if err then
					return nil, l("error in calling code '%s':  '%s'", tostring(json.toJson(convertRec.code)), err)
				end
				convertRec._used = true
				if saveData[convertRec.result] ~= nil or createEmptyTag or convertRec.create_empty_tag then
					prevPath, err = openTag(convertRec, prevPath, saveData[convertRec.result], dataRec, closeLastTag)
				end
				--[[ for key, val in pairs(saveData) do  -- old code
					local rec = {key, "=", val, tag = convertRec.tag, base_to = convertRec.base_to}
					-- fixConvertRec(rec, dataRec)
					prevPath, err = openTag(rec, prevPath, val, dataRec)
				end ]]
			elseif convertRec.from_static then -- not in use in new json-format?
				convertRec._used = true
				prevPath, err = openTag(convertRec, prevPath, convertRec.from_static, dataRec, closeLastTag)
			elseif type(convertRec.line_break) == "string" then
				lineBreak = convertRec.line_break
			elseif convertRec.content then
				convertRec._used = true
				if err == nil and convertRec.content then
					loopArray(convertRec, {dataRec})
				end
			else
				errorArr[#errorArr + 1] = l("missing allowed data tag in convert record '%s'", convertRec)
			end
			if err ~= nil then
				break
			end
		end -- for i, convertRec in ipairs(convertArr) do
		return err
	end
	local err
	if extraDataRec and not util.tableIsEmpty(extraDataRec) then
		inJson = util.tableCombine(inJson, extraDataRec, "no-error")
	end
	err = createOut(convertJson, inJson)
	closeAllTags()

	local function closeDoc()
		if count > 0 then
			doc:end_doc() -- or without doc:start_element() callback ypu can call = doc:end_doc()
		end
		doc:free()
	end
	if err ~= nil then
		closeDoc()
		return nil, err
	end

	local function checkNotUsedTags(convertJsonArrArr)
		for _, rec in ipairs(convertJsonArrArr) do
			if rec.mandatory then
				if not rec._used then
					local data = rec.from_static or rec.from_data or rec.from_function or rec.field or l "error, no 'from_static', 'from_data', 'from_function' or 'field' -tag"
					local tag = data .. " -> " .. rec.tag
					errorArr[#errorArr + 1] = l("no data found for mandatory tag '%s'", tag)
				end
			end
			if rec.content then
				checkNotUsedTags(rec.content) -- recursive call
			end
		end
	end
	checkNotUsedTags(convertJson)
	closeDoc()
	--[[ if file then
		file:close()
	end ]]
	-- io.flush()
	if extraDataRec.stylesheet then -- and not file then
		if out and out[1] then
			out[1] = out[1] .. extraDataRec.stylesheet .. "\n"
		end
	end
	if extraDataRec.namespace then -- and not file then
		if out and out[3] then
			-- out[1] == '<?xml version="1.0" encoding="UTF-8"?>'
			-- out[2] == "<"
			out[3] = out[3] .. " " .. extraDataRec.namespace
			-- out[4] == ">"
		end
	end
	local prevTag2
	for i = 4, #out do -- remove line breaks between tag and it's end tag, see comment in doc:start_doc(): this will be changed to "</" ...
		if out[i - 1]:sub(-1) == "<" then -- and out[i - 2] == ">" then --
			prevTag2 = out[i]
		elseif out[i] == ">" and out[i - 3] == ">" and out[i - 2]:sub(-2) == "</" and out[i - 1] == prevTag2 then
			out[i - 2] = "</"
		end
	end
	local txt = table.concat(out) .. "\n"
	if lineBreak ~= "" and lineBreak ~= "\n" then
		txt = peg.replace(txt, "\n", lineBreak)
	end
	if #errorArr > 0 then -- if not errorText and #errorArr > 0 then
		errorText = " - " .. table.concat(errorArr, "\n - ")
	end
	return txt, errorText, out
end

return {convert = convert}
