--- convert/text-to-json.lua
-- converts text to json
-- Fixed Width To JSON
-- @module convert
local textToJson = {}

local util = require "util"
local utf = require "utf"
local cp1252ToUtf8 = utf.cp1252ToUtf8
local peg = require "peg"
local dt = require "dt"
local json = require "json"
local l = require"lang".l
local trim = peg.trimFunction(" \t") -- peg.define.space)

local function printError(prf, err)
	if prf.print_error ~= false then
		util.printRed("\n" .. err)
	end
end

local function formatLine(txt, def, prf)
	if def.data_type then
		if def.data_type == "number" then
			local num = tonumber(txt)
			if num == nil then
				if not def.optional then
					util.printWarning("text '%s', as num is nil, definition '%s'", txt, json.toJsonRaw(def))
				end
				return 0
			elseif num == -0 then
				num = 0 -- -0 to 0
			elseif def.decimals then
				num = num / 10 ^ def.decimals
			end
			return num
		elseif def.data_type == "alpha" then
			if prf.remove_double_space then
				if #txt > 70 then
					txt = trim(txt)
				end
				txt = trim(txt)
			else
				txt = peg.removeFromStart(peg.removeFromEnd(txt, " "), " ")
			end
			if prf.win_charset then
				txt = cp1252ToUtf8(txt)
				--[[ if txtTmp ~= txt then
					if not peg.found(txtTmp, "Ä") and not peg.found(txtTmp, "ä") and not peg.found(txtTmp, "Ö") and not peg.found(txtTmp, "ö") then
						txtTmp = txtTmp
					end
					txt = txtTmp
				end ]]
			end
			if prf.char_map then
				for _, rec in ipairs(prf.char_map) do
					txt = peg.replace(txt, rec.from, rec.to)
				end
			end
			return txt
		elseif def.data_type == "timestamp" then
			return dt.parseToString(txt, def.format)
		elseif def.data_type == "date" then
			return dt.parseToString(txt, def.format)
		elseif def.data_type == "time" then
			return dt.parseToString(txt, def.format)
		else
			return txt
		end
	end
	return txt
end

local function textToTextArr(txt, prf)
	if type(txt) == "table" then
		return txt
	end
	txt = peg.removeFromEnd(txt, prf.record_separator) -- prevent extra empty line at the end
	local ret = peg.splitToArray(txt, prf.record_separator)
	return ret
end
textToJson.textToTextArr = textToTextArr

--[[ ---@ return table
local function convertEncoding(tbl)
	local ret = {}
	for key, val in pairs(tbl) do
		if type(val) == "table" then
			ret[key] = convertEncoding(val)
		elseif type(val) == "string" then
			ret[key] = utf.cp1252ToUtf8(val)
		else
			ret[key] = val
		end
	end
	return ret
end
 ]]
local function convert(txtOrArr, prf, definitionTagName)
	local ret = {}
	local err
	local definition = prf.definition and prf.definition[definitionTagName]
	if definition == nil then
		err = l("tag '%s' was not found from preference '%s' definition -tag", definitionTagName, tostring(prf.name)) -- , json.toJsonRaw(prf.definition)
		printError(prf, err)
		return ret, err, {}
	end
	local txtArr = textToTextArr(txtOrArr, prf)
	local startPos, endPos, txt, ok, formatted
	-- local len = 0
	for idx, line in ipairs(txtArr) do
		if err then
			break
		end
		ret[idx] = {}
		startPos = definition.start or 1
		for _, def in ipairs(definition.field) do
			if def.start then
				startPos = def.start
			end
			if def.field_length then
				endPos = startPos + def.field_length - 1
				txt = line:sub(startPos, endPos)
				if #txt < endPos - startPos + 1 then
					if not def.optional then
						err = l("line %d, text field length %d is less than definition field length %d, line '%s', definition '%s'", idx, #txt, endPos - startPos + 1, line, json.toJsonRaw(def))
						printError(prf, err)
						-- break
					end
				end
			else
				txt = line:sub(startPos)
			end
			if def.field_definition then
				local arr2, recordType
				if not def.field_definition.field then
					recordType = ""
				else
					recordType = ret[#ret][def.field_definition.field]
				end
				if not recordType then
					err = l("line %d, field definition field '%s' was not found, definition '%s'", idx, def.field_definition.field, json.toJsonRaw(def))
					printError(prf, err)
					break
				else
					arr2, err = convert({txt}, prf, def.field_definition.tag_start .. recordType)
				end
				if err then
					printError(prf, err)
					-- break
				end
				formatted = arr2[1]
			elseif def.value then
				formatted = def.value
			else
				ok, formatted = pcall(formatLine, def.value or txt, def, prf)
			end
			if ok then
				--[[ if type(formatted) == "string" then
					formatted = cp1252ToUtf8(formatted)
				end ]]
				if not def.optional and def.must_equal and formatted ~= def.must_equal then
					if type(def.must_equal) ~= "table" then
						err = l("line %d, formatted value '%s' is not equal to must equal value '%s', line '%s', definition '%s'", idx, formatted, tostring(def.must_equal), line, json.toJsonRaw(def))
						printError(prf, err)
						break
					elseif not def.must_equal[tostring(formatted)] then
						err = l("line %d, formatted value '%s' is not equal to must equal table key '%s', line '%s', definition '%s'", idx, formatted, json.toJsonRaw(def.must_equal), line, json.toJsonRaw(def))
						printError(prf, err)
						break
					end
				elseif not def.optional and def.convert_value then
					if type(def.convert_value) ~= "table" then
						err = l("line %d, formatted value '%s' is not equal to convert value key '%s', line '%s', definition '%s'", idx, formatted, tostring(def.convert_value), line, json.toJsonRaw(def))
						printError(prf, err)
						break
					elseif not def.convert_value[tostring(formatted)] then
						err = l("line %d, formatted value '%s' is not equal to convert table key '%s', line '%s', definition '%s'", idx, formatted, json.toJsonRaw(def.convert_value), line, json.toJsonRaw(def))
						printError(prf, err)
						break
					end
				elseif def.field and not def.optional and formatted == "" then
					err = l("line %d, formatted value of '%s' is not optional and it is empty, line '%s', definition '%s'", idx, txt, line, json.toJsonRaw(def))
					printError(prf, err)
					break
				end
				if def.field then -- if field is missing then skip data
					if def.convert_value then
						formatted = def.convert_value[tostring(formatted)]
					end
					if not def.optional or (formatted ~= "" and formatted ~= 0) then
						if def.type == "array" then
							if not ret[idx][def.field] then
								ret[idx][def.field] = {}
							end
							local fldRec = ret[idx][def.field]
							fldRec[#fldRec + 1] = formatted
						else
							ret[idx][def.field] = formatted
						end
					end
				end
			else
				err = l("line %d, formatting text '%s' failed with error '%s', line '%s', definition '%s'", idx, txt, formatted, line, json.toJsonRaw(def))
				printError(prf, err)
				break
			end
			startPos = endPos + 1
		end
	end
	return ret, err, definition
end
textToJson.convert = convert

function textToJson.convertByType(txtOrArr, prf, typeTagName)
	local ret = {} -- structured real return array
	local flatRetArr = {} -- flat array, needed for finding previous
	local typeArr, err
	local txtArr = textToTextArr(txtOrArr, prf)
	typeArr, err = convert(txtOrArr, prf, typeTagName)
	if err then
		return ret, err
	end
	local definitionRec = prf.definition
	if err == nil and #typeArr ~= #txtArr then
		err = l("type array size %d is not equal to text array size %d, definition '%s'", #typeArr, #txtArr, json.toJsonRaw(definitionRec))
		printError(prf, err)
	elseif err == nil then
		local arr, definition
		for i, typeRec in ipairs(typeArr) do
			arr, err, definition = convert({txtArr[i]}, prf, typeRec.type)
			if err then
				break
			elseif #arr ~= 1 then
				err = l("convert by type return array size %d is not 1, definition '%s'", #arr, json.toJsonRaw(definitionRec))
				printError(prf, err)
				break
			end
			if definition.content_to then
				local idx = i - 1
				repeat
					if typeArr[idx].type == definition.content_to then
						break
					end
					idx = idx - 1
				until idx < 1
				if idx >= 1 then
					local toRec = flatRetArr[idx]
					if not toRec.content then
						toRec.content = {}
					end
					toRec.content[#toRec.content + 1] = arr[1]
				else
					err = l("previous content of type '%s' was not found, index %d, definition '%s'", definition.content_to, i, json.toJsonRaw(definitionRec))
					printError(prf, err)
					break
				end
			else
				-- util.arrayConcat(ret, arr)
				ret[#ret + 1] = arr[1]
			end
			-- util.arrayConcat(flatRetArr, arr)
			flatRetArr[#flatRetArr + 1] = arr[1]
		end
	end
	-- ret = convertEncoding(ret)
	return ret, err
end

return textToJson
