--- lib/convert/xml-to-json.lua
-- converts cml to json using json convert table
-- see: http://www.xmlviewer.org - xml to json, json format
-- @module xml-to-json
local xmlToJson = {}

local util = require "util"
local l = require"lang".l
local peg = require "peg"
local expat = require "xml/expat"

local startAttr, endAttr, attrPattern
local function setPattern()
	startAttr = peg.pattern "[" -- attribute lista starts with [
	endAttr = peg.pattern "]" -- attribute lista ends with ]
	local notEndAttr = (1 - endAttr) ^ 0
	-- local notStartAttr = (1 - startAttr)^0
	attrPattern = startAttr * notEndAttr * endAttr
end

local function parseToTree(xml, option) -- pm, replace lom.lua
	local anyWhiteSpace = peg.define.anyWhiteSpace
	local tableInsert, tableRemove = table.insert, table.remove
	local stack = {{}}
	local id = option and option.id and 0 or nil
	local callback = {
		attr_list = function(elem, name, attrType, dflt, isRequired)
			-- does not come hre?
			print("xml-to-json parseToTree() attr: ", elem, name, attrType, dflt, isRequired)
		end,
		ref = function(data)
			stack[#stack].ref = peg.removeFromStartEnd(data, " ")
			print("xml-to-json parseToTree() ref: ", data)
		end,
		skipped = function(data)
			stack[#stack].skipped = peg.removeFromStartEnd(data, " ")
			print("xml-to-json parseToTree() skipped: ", data)
		end,
		unknown = function(data)
			stack[#stack].unknown = peg.removeFromStartEnd(data, " ")
			print("xml-to-json parseToTree() unknown: ", data)
		end,
		start_namespace = function(data)
			stack[#stack].namespace = peg.removeFromStartEnd(data, " ")
		end,
		comment = function(data)
			if stack[#stack].comment then
				stack[#stack].comment = stack[#stack].comment .. "\n" .. peg.removeFromStartEnd(data, " ")
			else
				stack[#stack].comment = peg.removeFromStartEnd(data, " ")
			end
		end,
		start_tag = function(name, attrs)
			if id then
				id = id + 1
			end
			if util.tableIsEmpty(attrs) then
				tableInsert(stack, {key = name, id = id})
			else
				tableInsert(stack, {key = name, id = id, attribute = attrs})
			end
		end,
		end_tag = function() -- (name)
			local element = tableRemove(stack) -- move from the end of stack to the element if previous stack element
			if not stack[#stack].element then
				stack[#stack].element = {}
			end
			tableInsert(stack[#stack].element, element)
		end,
		userdata = function(data)
			stack[#stack].userdata = peg.removeFromStartEnd(data, " ")
			print("xml-to-json parseToTree() userdata: ", data)
		end,
		cdata = function(data)
			if peg.replace(data, anyWhiteSpace, "") ~= "" then
				data = peg.removeFromStartEnd(data, '"')
				if stack[#stack].text then
					if data:sub(1, 1) == "]" then
						stack[#stack].text = stack[#stack].text .. peg.removeFromStartEnd(data, '"')
					else
						stack[#stack].text = stack[#stack].text .. "\n" .. peg.removeFromStartEnd(data, '"')
					end
				else
					stack[#stack].text = data -- text could be also named cdata
				end
			end
		end
	}
	local err = expat.parse({string = xml}, callback)
	local xmlJsonTree = stack[1].element and stack[1].element[1]
	if err == nil then
		err = util.getError() -- is this needed?
		if err and peg.found(err, "XML") or xmlJsonTree == nil or util.tableIsEmpty(xmlJsonTree) then
			err = err or "xml convert return is empty"
			err = util.printError(err)
		end
	end
	return xmlJsonTree, err
end
xmlToJson.parseToTree = parseToTree

local function convertToFlat(jsonPrf)
	if attrPattern == nil then
		setPattern()
	end
	local ret = {}
	-- local retArr = {}
	local basePath = ""
	local baseFullPath = ""

	-- local attrCapturePattern = peg.capture(startAttr) * peg.capture(notEndAttr)
	local function flatPath(rec)
		local path, fullPath, field
		if rec.field ~= nil then
			field = peg.removePattern(rec.field, attrPattern) -- remove all attributes
			if field:sub(1, 1) == "/" then
				path = field -- field:sub(2)
				basePath = path .. "/"
				fullPath = rec.field
				baseFullPath = fullPath .. "/"
			else
				path = basePath .. field
				fullPath = baseFullPath .. rec.field
			end
			-- ret contains an array of objects because same base path can have different attributes
			if not ret[path] then
				ret[path] = {rec}
			else
				local arr = ret[path]
				arr[#arr + 1] = rec
			end
			rec.path = fullPath -- save full path with attributes to rec
			if rec.array then
				for _, rec2 in ipairs(rec.array) do
					flatPath(rec2) -- recursive call
				end
			end
			-- else
			-- 	local a -- for trace
		end
	end
	for _, jsonVar in ipairs(jsonPrf) do
		flatPath(jsonVar) -- , jsonVar.field) -- inner local function first call
	end
	return ret
end
-- xmlToJson.convertToFlat = convertToFlat

local function convertToJson(convertDefinition, xmlText, option)
	local convertJson = convertDefinition.convert
	local debug = option and option.debug or false
	local emptyMandatoryToTag = {}
	local errorArr = {}
	setPattern()

	local convert = convertToFlat(convertJson)

	local function parse()
		local ret = {}
		local retParent = {}
		local retTag = ret
		local xmlPath = ""
		local xmlPathAttr = {}
		local xmlPathPrev = {}
		local xmlPathDepth = 0
		local callbackFound = false

		local function matchTag(rec)
			-- check if attributes match
			local pos = peg.find(rec.path, startAttr)
			-- are there any attributes in full definition?
			if pos > 0 then -- contains attributes
				local tagArr = peg.splitToArray(rec.path, "/")
				local tagPath = ""
				for _, tag in ipairs(tagArr) do -- loop all tags between and match attributes
					if tag ~= "" then -- first tag is "" because path starts with /
						pos = peg.find(tag, startAttr)
						if pos < 1 then
							tagPath = tagPath .. "/" .. tag
						else
							local recTag = tag:sub(1, pos - 1)
							tagPath = tagPath .. "/" .. recTag
							if not xmlPathAttr[tagPath] then
								return false -- data does not contain tags, definition does
							end
							local attribute = tag:sub(pos + 1)
							local posEnd = peg.find(attribute, endAttr)
							if posEnd < 1 then
								errorArr[#errorArr + 1] = l("attribute definition '%s' does not contain ']', tag '%s'", attribute, tag)
								return false
							end
							attribute = attribute:sub(1, posEnd - 1) -- remove ]
							attribute = peg.replace(attribute, "@", "") -- get rid of @

							local definitionAttr = {}
							local definitionArray = peg.splitToArray(attribute, " ")
							-- attributes are separated by space
							for _, txt in ipairs(definitionArray) do
								pos = peg.find(txt, "=")
								local key = txt:sub(1, pos - 1)
								local val = txt:sub(pos + 1)
								definitionAttr[key] = peg.replace(val, '\"', "")
								-- get rid of escaped quotes
							end

							-- check that all attributes match
							local found = false
							local dataAttr = xmlPathAttr[tagPath]
							for key, val in pairs(definitionAttr) do
								-- check that all definition attributes are foud from data attributes
								-- data may contain extra attributes
								if val == dataAttr[key] then
									found = true
								else
									found = false
									break
								end
							end
							if not found then
								return false
							end

						end
					end
				end
			end

			return true
		end

		local tagWarningDone
		local function addToTarget(rec, data, attrs) -- (rec, path, data, attrs)
			-- local rec = convert[xmlPath]
			if rec.not_in_use then
				return false
			elseif not rec.tag then
				-- it is ok to have empty 'tag' -key
				if rec.mandatory then
					emptyMandatoryToTag[#emptyMandatoryToTag + 1] = rec.field
				end
				return false
			elseif not matchTag(rec) then
				return false
			end
			local tag = rec.tag
			if tag == nil and ret.to then
				if not tagWarningDone then
					tagWarningDone = true
					util.printWarning("'tag' -key was not defined, please convert 'to' -keys to 'tag', definition '%s'", tostring(convertDefinition.name))
				end
				rec.tag = rec.to
				rec.to = nil
				tag = rec.to
			end

			-- is start tag, not cdata
			if data == nil then
				if rec.array then
					if debug then
						tag = tag .. " - " .. rec.field
						if rec.description then
							tag = tag .. " - '" .. rec.description .. "'"
						end
					end
					retParent[xmlPath] = retTag
					if not retTag[tag] then
						retTag[tag] = {}
					end
					local line = #retTag[tag] + 1
					-- print("add retParent: "..xmlPath..", "..tag..", line "..line)
					retTag[tag][line] = {}
					retTag = retTag[tag][line]
				end

				-- fix this later, what is attrs?
				if rec.data_from_attribute then
					-- print("  match data_from_attribute:", rec.data_from_attribute)
					local val = attrs[rec.data_from_attribute]
					if not val then
						errorArr[#errorArr + 1] = l("attribute '%s' value was not found field data, tag '%s'", rec.data_from_attribute, rec.field)
						return false
					end
					retTag[tag] = val
				end

			else --  ctag data
				if debug then
					tag = tag .. " - " .. rec.field
					if rec.description then
						tag = tag .. " - '" .. rec.description .. "'"
					end
				end
				if tag and tag ~= "" then
					data = data:gsub("\n", "")
					data = data:gsub("\t", "")
					if data ~= "" then
						if rec.type == "boolean" and data == "false" then
							retTag[tag] = false
						elseif rec.type == "boolean" then
							retTag[tag] = true -- all but false is true, is this ok?
						elseif rec.type == "number" then
							retTag[tag] = tonumber(data)
						else
							retTag[tag] = data
						end
						-- print("  tag data:", xmlPath.." \n    "..data)
					end
				end
			end

			return true -- tag ok
		end

		local callbacks = {
			attr_list = function(elem, name, attrType, dflt, isRequired)
				print("attr: ", elem, name, attrType, dflt, isRequired)
			end,
			start_tag = function(name, attrs)
				xmlPathDepth = xmlPathDepth + 1
				xmlPathPrev[xmlPathDepth] = xmlPath
				xmlPath = xmlPath .. "/" .. name
				if not util.tableIsEmpty(attrs) then
					xmlPathAttr[xmlPath] = attrs
					-- this is usually overridden, but it is ok because we go forward in xml data
				elseif xmlPathAttr[xmlPath] then
					xmlPathAttr[xmlPath] = nil -- delete possible previous value, just to be sure
				end
				-- convert[xmlPath] contains an array of objects because same base path can have different attributes
				callbackFound = false
				if convert[xmlPath] then
					for _, rec in ipairs(convert[xmlPath]) do
						callbackFound = addToTarget(rec, nil, attrs) -- (rec, xmlPath, nil, attrs)
						if callbackFound then
							break -- break loop, use only first found attribute-matches record
						end
					end
				end
			end,
			end_tag = function() -- (name)
				if retParent[xmlPath] then
					-- print("del retParent: "..xmlPath)
					retTag = retParent[xmlPath]
					retParent[xmlPath] = nil
				end
				xmlPath = xmlPathPrev[xmlPathDepth]
				xmlPathDepth = xmlPathDepth - 1
			end,
			cdata = function(data)
				if callbackFound then -- and data ~= "\n" then -- convert[xmlPath] then
					if convert[xmlPath] then
						for _, rec in ipairs(convert[xmlPath]) do
							if not rec.array and not rec.data_from_attribute then
								local added = addToTarget(rec, data) -- (rec, xmlPath, data)
								if added then
									break -- break loop, use only first found attribute-matches record
								end
							end
						end
					end
				end
			end
		}

		local err = expat.parse({string = xmlText}, callbacks) -- expat.treeParse({path=xml})
		if err then -- does this ever come?
			print(err)
		end

		local function cleanParent(tbl)
			for k, rec in pairs(tbl) do
				if k == "_parent" then
					tbl[k] = nil
				elseif type(rec) == "table" then
					cleanParent(rec) -- rec may contain other tables, is a recursive call
				end
			end
		end
		cleanParent(ret) -- we must clean parents, otherwise toJson() will fail

		return ret
	end
	local ret, err = parse()

	if #errorArr > 0 then
		if err and err ~= "" then
			err = " - " .. table.concat(errorArr, "\n - ") .. "\n\n" .. err
		else
			err = " - " .. table.concat(errorArr, "\n - ")
		end
	end

	if #emptyMandatoryToTag > 0 then
		local errTxt = " - " .. l("mandatory definition where 'tag' key is not defined:\n")
		if err and err ~= "" then
			err = errTxt .. table.concat(emptyMandatoryToTag, "\n") .. "\n\n" .. err
		else
			err = errTxt .. table.concat(emptyMandatoryToTag, "\n")
		end
	end

	return ret, err
end
xmlToJson.convert = convertToJson

local function setConvertDefField(convertDef, element)
	if convertDef.field == nil and element.local_field then
		convertDef.field = element.local_field
		convertDef.from_static = element.from_static
	end
	if convertDef.from_static == nil and element.value then
		convertDef.from_static = element.value
	end
	if convertDef.mandatory == nil and element.force then
		-- if convertDef.from_static == nil and convertDef.array == nil and convertDef.mandatory == nil and element.force then
		convertDef.mandatory = element.force
	end
	if convertDef.length == nil and element.length then
		convertDef.length = element.length
	end
	if convertDef.format == nil and element.format then
		convertDef.format = element.format
	end
	if convertDef.info == nil and element.info then
		convertDef.info = element.info
	end
	if convertDef.type == nil and element.type then
		convertDef.type = element.type
	end
end

local function fixMandatoryStatic(convertDef)
	if convertDef.array == nil and convertDef.from_static == nil and convertDef.field == nil and convertDef.mandatory == "M" and (convertDef.type == "A" or convertDef.type == "code") then
		convertDef.from_static = ""
	end
	if convertDef.array then
		for _, item in ipairs(convertDef.array) do
			fixMandatoryStatic(item)
		end
	end
end

local tagWarningDone
local function convertToJsonDefinition(xmlJson, convertDef, level)
	if level == nil then
		level = 0
	end
	if xmlJson.key then
		if convertDef.tag == nil and convertDef.to then
			if not tagWarningDone then
				tagWarningDone = true
				util.printWarning("'tag' -key was not defined, please convert 'to' -keys to 'tag'")
			end
			convertDef.tag = convertDef.to
			convertDef.to = nil
		end
		--[[ if parent == nil then
			convertDef.tag = "/" .. xmlJson.key -- absolute path
		else ]]
		convertDef.tag = xmlJson.key -- relative path to parent
		-- end
	end

	local function setContent(item, i)
		local content = item.content[i]
		item.content = nil
		util.recToRec(item, content)
		setConvertDefField(convertDef, item)
	end

	if xmlJson.content then
		if #xmlJson.content == 1 then
			setContent(xmlJson, 1)
		else
			local defineFound = false
			for i, item in ipairs(xmlJson.content) do
				if item.local_field then
					if not defineFound then
						defineFound = true
						setContent(xmlJson, i)
					else
						util.print("   key '%s' content local_field was already found, content: %s", xmlJson.key, item)
					end
				end
			end
			if not defineFound then
				util.print("  key '%s', element count: %d, content count: %d", xmlJson.key, #xmlJson.element, #xmlJson.content)
			end
		end
	end
	if xmlJson.attribute then
		convertDef.attribute = xmlJson.attribute
	end

	local function loopArray(element, i)
		local newDef = convertDef.array[i] or {}
		if convertToJsonDefinition(element, newDef, convertDef, level) == nil then
			return -- error happened
		end
		if convertDef.array[i] == nil and newDef then
			convertDef.array[i] = newDef
		end
	end

	if xmlJson.element then
		local i = 0
		for _, element in ipairs(xmlJson.element) do
			-- "value", "segment", "order", "header", "force", "type", "length", "define", "info", "data", "format", "header", "level", "fill", "start", "end"
			if element.key == "code" then
				convertDef.type = element.key
				if type(element.content) ~= "table" then
					util.printRed("  xml json code element content is not table")
					return
				elseif #element.content ~= 1 then
					util.printRed("  xml json code element content length is not 1")
					return
				end
				setConvertDefField(convertDef, element.content[1])
			elseif element.type == "Field" then
				convertDef.type = "field"
				setConvertDefField(convertDef, element)
			elseif element.type then
				if element.type ~= "A" then
					convertDef.type = element.type
				end
				setConvertDefField(convertDef, element)
			else
				if convertDef.array == nil then
					convertDef.array = {}
				end
				i = i + 1
				loopArray(element, i)
			end
		end
	end
	if level == 0 then
		fixMandatoryStatic(convertDef)
	end
	return true
end
xmlToJson.convertToJsonDefinition = convertToJsonDefinition

return xmlToJson
