-- lib/coro.lua
--
-- @module coroutine
local coro = require "coro-thread"
local treadCreated = coro.treadCreated
local treadClosed = coro.treadClosed
local threadId = coro.threadId
local coroResume = coroutine.resume
local coroutineYield = coroutine.yield
local coroutineRunning = coroutine.running
local currentThread = coro.currentThreadFunction()
local create = coroutine.create
local status = coroutine.status
local concat = table.concat
local mainThread, releaseLock
local coroThreadCount = 0
local defaultCountInterval = 20000
local debugLevel = 0

local useCoro = coro.useCoro()
local util = require "util"
local fromEditor = util.fromEditor()
local dconnConnectionCount

local socketIdx = {}
local socketClosed
do
	local sockNum
	socketClosed = function(sock)
		sockNum = tonumber(sock.socket or sock.closed)
		if sockNum == nil then
			util.printError("-*-ERR: socket '%s' number is nil, %s", tostring(sockNum), sock)
			return
		end
		if socketIdx[sockNum] == nil then
			if not sock.shutdown then
				util.printError("-*-ERR: socket '%s' was not found from the socket index, %s", tostring(sockNum), sock)
			end
		end
		socketIdx[sockNum] = nil
		releaseLock(sock, true, false) -- sock, allowNoLocks, showError
	end

end

local function coroCreate(sock, func)
	coroThreadCount = coroThreadCount + 1
	local oldThread = currentThread()
	local thread = create(func) -- - we don't use coro.wrap because it does not return coroutine status as first return value
	sock.thread = thread
	treadCreated(thread, sock, oldThread)
	return thread
end

local function connectionCount()
	if dconnConnectionCount == nil then
		dconnConnectionCount = require"dconn".connectionCount
	end
	return dconnConnectionCount()
end

local function createSocketCoroutine(sock, loopFunction)
	if sock.thread then
		util.printError("socket '%s' already has a coroutine thread", tostring(sock.socket))
	else
		coroCreate(sock, loopFunction)
		if debugLevel > 0 then
			util.print("  created %s, socket '%s', thread count: %d, connection count: %d", threadId(sock.thread), tostring(sock.socket), coroThreadCount, connectionCount())
		end
		coroResume(sock.thread)
	end
	return sock.thread
end

local coroRunning = coroutine.running
local function currentThreadSocket(thread)
	if useCoro == false then
		return nil
	end
	if thread == nil then
		thread = coroRunning() -- currentThread() returns string in debug mode
	end
	for _, sock in pairs(socketIdx) do
		if sock.thread == thread and not (sock.do_close or sock.closed) then
			return sock, currentThread(thread), thread
		end
	end
	return nil, currentThread(), thread
end

local function threadSocketArr(thread)
	local socketArr = {}
	for _, sock in pairs(socketIdx) do
		if sock.thread == thread and not (sock.do_close or sock.closed) then
			socketArr[#socketArr + 1] = sock
		end
	end
	return socketArr
end

local function setSocket(sock)
	socketIdx[tonumber(sock.socket)] = sock
end

local function coroClose(thread)
	coroThreadCount = coroThreadCount - 1
	if thread == nil then
		util.printError("thread close: thread is nil")
		thread = currentThread()
	end
	if debugLevel > 0 then -- or coroThreadCount < 1
		util.print("closed %s, thread count: %d, connection count: %d", threadId(thread), coroThreadCount, connectionCount())
	end
	treadClosed(thread)
end

local function threadCount()
	return coroThreadCount
end

local function setMainThread(mainThread_, debug)
	debugLevel = debug
	if mainThread_ then
		mainThread = mainThread_
	else
		mainThread = currentThread()
	end
	return mainThread
end

local prevDebugThread = ""
local coThread
local function yieldFunction(countInterval, resume)
	if useCoro == false then
		return function()
		end
	end
	if countInterval == nil or resume == nil then
		util.printError("yield function needs interval and resume parameters")
		resume = true
	end
	local callCount = 0
	local yieldCount = 0
	local time
	if countInterval == nil then
		countInterval = defaultCountInterval
		resume = true
	end
	if util.fromEditor() and countInterval > 10 then
		countInterval = countInterval / 10
	end
	local callCountNext = countInterval
	return function(sock)
		if sock and (sock.do_close or sock.closed) then
			return
		end
		callCount = callCount + 1
		if callCount >= callCountNext then -- should be ok with looping to negative numbers also, max extra coroCountIncrease calls
			yieldCount = yieldCount + 1
			if debugLevel > 0 then
				if mainThread == nil then
					-- this is ok when there is no coroutine at all
					util.printError("yield function main thread is nil")
					return
				end
				if type(sock) == "number" then
					util.printError("  coroutine yield socket '%s' is number, not a table", tostring(sock))
					return
				end
				if sock == nil then
					util.printWarning("yield call without socket")
					sock = currentThreadSocket()
					if sock and (sock.do_close or sock.closed) then
						return
					end
					resume = true
				end
			end
			coThread = coroutineRunning() -- coThread, coRun = coroutineRunning() -- TODO: find out luajit 2 return values
			if coThread ~= sock.thread then
				util.printError(" current coroutine %s is not same as socket '%s' %s", threadId(coThread), tostring(sock.socket), threadId(sock.thread))
				coThread = sock.thread
			end
			callCountNext = callCountNext + countInterval
			if debugLevel > 2 or debugLevel > 0 and prevDebugThread ~= coThread then
				util.printColor("dim white", "   coroutine yield socket '%s', %s, count: %d", tostring(sock.socket), threadId(coThread), yieldCount)
				time = util.seconds()
			end
			if sock then -- if sock and tonumber(sock.socket) > 0 then
				if resume then
					sock.resume = true
				end
				prevDebugThread = coThread
				coroutineYield() --  yield and resume do not send any parameters
				if resume then
					sock.resume = nil
				end
			else
				coroutineYield()
			end
			if debugLevel > 2 or debugLevel > 0 and prevDebugThread ~= coThread then
				util.printColor("dim white", "    coroutine yield after, socket '%s', %s, count: %d, time: %.5f seconds%s", tostring(sock.socket), threadId(coThread), yieldCount, util.seconds(time), sock.do_close and " (will close)" or sock.closed and " (closed)" or "")
			end
		end
	end
end

do
	local function threadIndex(sock)
		if fromEditor then
			return tostring(sock.thread)
		end
		return sock.thread
	end

	local coroYieldLocked = yieldFunction(1, true) -- yield every 1 calls, do resume without real poll event
	local topicLock = {}
	local prevTopicLock = {}
	local threadLock = {}
	local threadIdx

	function coro.releaseLock(sock, allowNoLocks, showError)
		if not useCoro then
			return
		end
		if sock.thread == nil then
			if showError ~= false then
				if sock.closed then
					util.printRed("  lock release socket '%s', %s does not exist", " (closed)", threadId(sock.thread))
				else
					util.printError("  lock release socket '%s', %s does not exist", sock.socket and tostring(sock.socket) or (tostring(sock.closed) .. " (closed)"), threadId(sock.thread))
				end
			end
			return
		end
		threadIdx = threadIndex(sock)
		if threadLock[threadIdx] == nil then
			if allowNoLocks then
				return
			end
			util.printError("  socket '%s', %s has no locks to release", sock.socket and tostring(sock.socket) or (tostring(sock.closed) .. " (closed)"), threadId(sock.thread))
			return
		end
		for i, topic in ipairs(threadLock[threadIdx]) do
			topicLock[topic] = nil
			if prevTopicLock[topic] ~= sock.thread and debugLevel > 0 then
				util.print("  %d. lock '%s' released, socket '%s', %s", i, topic, sock.socket and tostring(sock.socket) or (tostring(sock.closed) .. " (closed)"), threadId(sock.thread))
			end
		end
		threadLock[threadIdx] = nil
		sock.lock_topic = nil
	end
	releaseLock = coro.releaseLock

	local function setLock(sock, topic)
		threadIdx = threadIndex(sock)
		if threadLock[threadIdx] == nil then
			threadLock[threadIdx] = {}
		end
		threadLock[threadIdx][#threadLock[threadIdx] + 1] = topic
		sock.lock_topic = topic
		topicLock[topic] = sock.thread
		prevTopicLock[topic] = sock.thread
	end

	function coro.waitForLock(sock, topic)
		if not useCoro then
			return
		end
		if sock == nil then
			util.printError("  lock set socket is nil")
			return
		end
		if sock.thread == nil then
			util.printError("  lock set socket '%s', %s does not exist", sock.socket and tostring(sock.socket) or (tostring(sock.closed) .. " (closed)"), threadId(sock.thread))
			return
		end
		if topicLock[topic] then
			if topicLock[topic] == sock.thread then
				return -- for example 4drest execute() sets "4drest" lock in consecutive calls, this is ok
			end
			-- sock.resume = true
			--[[ if topicLock[topic].closed then -- now thread, not a socket
				util.printOk("  lock '%s' reserved, lock holding socket '%s' is closed, %s", topic, tostring(sock.socket), threadId(sock.thread))
				setLock(sock, topic)
				return
			end ]]
			if status(topicLock[topic]) == "dead" then
				util.printOk("  lock '%s' reserved, lock holding coroutine is closed, socket '%s', %s", topic, tostring(sock.socket), threadId(sock.thread))
				setLock(sock, topic)
				return
			end
			threadIdx = threadIndex(sock)
			local sockTopic = ""
			if threadLock[threadIdx] then
				sockTopic = ", socket current lock: '" .. concat(threadLock[threadIdx], ", ") .. "', "
				-- first lock locks also next locks, for ex. "calc-server" is the first lock and next locks is "4drest"
			end
			util.printInfo("  waiting for lock '%s'%s, socket '%s', %s", topic, sockTopic, tostring(sock.socket), threadId(sock.thread))
			while topicLock[topic] and not sock.do_close and not sock.closed and status(topicLock[topic]) ~= "dead" do
				--[[ if not topicLock[topic].resume and status(topicLock[topic]) == "suspended" then
					topicLock[topic].resume = true -- this will cause erro 35 in socket read
				end ]]
				coroYieldLocked(sock)
			end
			if sock.do_close or sock.closed then
				util.printOk("  lock '%s' wait ended, socket '%s' will be closed, %s", topic, tostring(sock.socket), threadId(sock.thread))
			elseif type(topicLock[topic]) == "thread" and status(topicLock[topic]) == "dead" then
				topicLock[topic] = sock
				util.printOk("  lock '%s' acquired, lock holding coroutine is closed, socket '%s', %s", topic, tostring(sock.socket), threadId(sock.thread))
			else
				util.printOk("  lock '%s' acquired, socket '%s', %s", topic, tostring(sock.socket), threadId(sock.thread))
			end
		else
			if prevTopicLock[topic] ~= sock and debugLevel > 0 then
				util.printOk("  lock '%s' reserved, socket '%s', %s", topic, tostring(sock.socket), threadId(sock.thread))
			end
		end
		setLock(sock, topic)
	end
end

local func = {
	-- clearLoadedPackages = clearLoadedPackages,
	createSocketCoroutine = createSocketCoroutine,
	currentThread = currentThread,
	setMainThread = setMainThread,
	yieldFunction = yieldFunction,
	yield = coroutineYield,
	setSocket = setSocket,
	socketClosed = socketClosed,
	currentThreadSocket = currentThreadSocket,
	threadSocketArr = threadSocketArr,
	create = coroCreate,
	close = coroClose,
	threadCount = threadCount,
	connectionCount = connectionCount,
	resume = coroResume,
	running = coroutineRunning,
	status = status
	-- wrap = coroutine.wrap,
}
for key, value in pairs(func) do
	coro[key] = value
end
return coro

--[[
-- put at the satrt of the file:
local systemPackageIdx = {}
for key in pairs(package.loaded) do
	systemPackageIdx[key] = true
end
local ioWrite = io.write
local peg = require "peg"
-- local clearPackageIdx = {"dsave", "dqjson", "dprf", "dsql", "dseq", "dconn", "util", "auth", "database", "db", "pg", "system", "plugin", "curl"}
local clearPackageIdx = {} -- {"dconn", "auth"} -- {"d", "system", "plugin", "util", "auth", "qry", "table/", "curl"} --
local keepPackageArr = {} -- {"coro", "coro-thread", "execute", "dschemafld", "doperation", "uuid", "system/poll", "system/server", "system/net", "system/crypto", "system/scrypt"}
local keepPackageIdx = util.invertTable(keepPackageArr)
---

local function clearLoadedPackages(thread)
	if #clearPackageIdx == 0 then
		return
	end
	local time = util.seconds()
	local done = {}
	local i = 0
	util.ioWrite("\n  clear '%s, %s' packages: ", threadId(thread), tostring(thread):sub(#"thread: "))
	for name in pairs(package.loaded) do
		if not systemPackageIdx[name] and not done[name] then
			for _, clearName in ipairs(clearPackageIdx) do
				-- if not done[name] and peg.found(name, clearName) and not keepPackageIdx[name] and not peg.found(name, "ffi") then
				if not done[name] and not keepPackageIdx[name] and not peg.found(name, "ffi") and peg.startsWith(name, clearName) then
					i = i + 1
					done[i] = name
					ioWrite(name .. ", ")
					package.loaded[name] = nil
				end
			end
		end
	end
	--[=[ if not requirePackages then
		ioWrite(string.format("clear count: %d, time: %.3f seconds\n\n", i, util.seconds(time)))
	else ]=]
	util.ioWrite("clear count: ", i)
	util.ioWrite("\n  require '%s' packages: ", thread)
	local ok
	for _, name in ipairs(done) do
		ioWrite(name .. ", ")
		ok = pcall(require, name)
		if not ok then
			util.printError("require thread package '%s' failed", name)
		end
	end
	ioWrite(string.format("require count: %d, time: %.3f seconds\n\n", i, util.seconds(time)))
	-- end
end
--]]

---------------------

--[=[

return {
	-- coroIndex = coroIndex,
	-- coroThreadIndex = coroThreadIndex,
	-- addCoroIdx = addCoroIdx,
	-- removeCoroIdx = removeCoroIdx,
}

local coroIdx = {}
-- local coroThreadArr = {}

local function coroIndex(sock)
	return coroIdx[sock]
end

local function coroThreadIndex(thread)
	for _, value in pairs(coroIdx) do
		if value.thread == thread then
			return value.socket
		end
	end
	return nil
end

-- local threadArr, threadId
local function addCoroIdx(sock, thread)
	if sock.thread then
		util.printError("socket '%s' already has coroutine", sock.socket)
	else
		-- local thread = wrapFunction(sock, "get-thread")
		if thread == mainThread then
			util.printError("socket '%s' thread is main thread", sock.socket)
		end
		coroIdx[sock.socket] = {thread = thread, socket = sock}
		sock.thread = thread
		-- sock.resume = wrapFunction
		-- we don't use coro.wrap because it does not return coroutine status as first return value
		sock.resume = function(...)
			return coroResume(thread, ...)
		end
	end
end

local socketNum
local function removeCoroIdx(sock)
	socketNum = sock.socket or sock.closed
	if sock.thread == nil and coroIdx[socketNum] then
		util.printError("socket '%s' has no coroutine", socketNum)
	elseif coroIdx[socketNum] then
		--[[ threadArr = coroThreadArr[tostring(coroIdx[socketNum].thread)]
		if threadArr == nil then
			util.printError("socket '%s' has no coroutine array", socketNum)
		elseif threadArr.count == 0 then
			util.printError("socket '%s' coroutine thread index size is zero", socketNum)
		else
			threadArr[socketNum] = nil
			threadArr.count = threadArr.count - 1
			if threadArr.count == 0 then
				coroThreadArr[threadId(sock.thread)] = nil
			else
				local num = threadArr.socket.socket
				if num == socketNum or type(num) ~= "number" then
					for key in pairs(threadArr) do
						if type(key) == "number" then
							num = key
							break
						end
					end
				end
				threadArr.socket = coroIdx[num].socket
			end
		end ]]
		coroIdx[socketNum] = nil
		-- sock.thread = nil
	end
end
]=]
