-- curl.lua
-- /Users/pasi/installed/C/tls/curl/include/curl/curl.h
-- see also: https://github.com/LPGhatguy/luajit-request/blob/master/luajit-request.lua
local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local peg = require "peg"
local dt = require "dt"
local l = require"lang".language
local fs = require "fs"
local lcurl = require "curl-ffi"
local gzip

local lparse
local function loadParse()
	lparse = util.loadDll("ftp_list_parse")
	return lparse or {}
end

-- local CURL_CTX = curl_easy_init() -- wrapper func
local receiveData, receiveDataSize
local receiveHeaderData, receiveHeaderDataSize
local sendData = {cdata = nil, bytesSent = 0, size = 0}
local donePrev

--[[
local function yield()
	util.sleep(0.001)
end
]]

--[[
local function curl_pointer_finalizer(pointer)
    -- print "finalizing curl"
    lcurl.curl_easy_cleanup(pointer)
end

function curl_easy_init()
    return ffi.gc(lcurl.curl_easy_init(), curl_pointer_finalizer)
end
]]

local function ftpFileListLine(line)
	local fp = ffi.newNoAnchor("struct ftpparse[1]")
	local len = #line
	lparse = lparse or loadParse()
	local ret = lparse.ftpparse(fp, line, len)
	if ret ~= 1 then
		return nil
	end
	fp = fp[0]

	local name = ffi.string(fp.name, fp.namelen)
	local id = ffi.string(fp.id, fp.idlen)
	local mtime = dt.formatNum(fp.mtime) -- mtime is C time_t struct
	return {name = name, cwd = fp.flagtrycwd, retr = fp.flagtryretr, sizetype = fp.sizetype, size = tonumber(fp.size), mtimetype = fp.mtimetype, mtime = mtime, idtype = fp.idtype, id = id}
end

local function ftpFileList(ret)
	local lines = {}
	--[[ test for linux
	ret = [-[
-rw-rw-r--   1 lujatst  1050         1500 Jan 29 10:20 SyncInventoryBalance_2015.01.29_12.20.31.349.+0200.xml
-rw-rw-r--   1 lujatst  1050         1497 Jan 29 10:20 SyncInventoryBalance_2015.01.29_12.20.31.54.+0200.xml
drwxr-xr-x   2 lujatst  1050          120 Jan 27 20:43 l
]-]
]]
	peg.iterateLines(ret, function(line)
		local rec = ftpFileListLine(line)
		if rec then
			lines[#lines + 1] = rec
		end
	end)
	return lines -- ,err
end

local function initCurl()
	-- sendData = {cdata=nil, bytesSent=0, size=0}
	donePrev = -1
	receiveData = {}
	receiveDataSize = 0
	receiveHeaderData = {}
	receiveHeaderDataSize = 0
	return lcurl.curl_easy_init()
end

local function getReceiveData()
	if util.tableIsEmpty(receiveData) then
		return nil
	end
	local ret = table.concat(receiveData)
	if #ret ~= receiveDataSize then
		print(l("return data sizes do not match %d / %d", #ret, receiveDataSize))
	end
	receiveData = nil
	return ret
end

local function getReceiveHeaderData()
	if util.tableIsEmpty(receiveHeaderData) then
		return nil
	end
	local ret = table.concat(receiveHeaderData)
	if #ret ~= receiveHeaderDataSize then
		print(l("return header data sizes do not match %d / %d", #ret, receiveHeaderDataSize))
	end
	receiveHeaderData = nil
	return ret
end

local function receiveFunction(ptr, size, nmemb, userdata)
	-- WARN: save directly to file if file is big
	-- http://curl.haxx.se/libcurl/c/CURLOPT_WRITEFUNCTION.html
	-- size_t write_callback(char *ptr, size_t size, size_t nmemb, void *userdata);
	-- yield() -- must NOT call or will get "attempt to yield across C-call boundary"
	local bytes = tonumber(size * nmemb)
	if bytes > 0 then
		receiveData[#receiveData + 1] = ffi.string(ptr, bytes)
		receiveDataSize = receiveDataSize + bytes
	end
	return bytes
end
local receiveFunctionPointer = ffi.cast("size_t (*)(char *ptr, size_t size, size_t nmemb, void *stream)", receiveFunction)

local function receiveHeaderFunction(ptr, size, nmemb, userdata)
	-- WARN: save directly to file if file is big
	-- https://curl.haxx.se/libcurl/c/CURLOPT_HEADERFUNCTION.html
	-- size_t write_callback(char *ptr, size_t size, size_t nmemb, void *userdata);
	-- yield()
	local bytes = tonumber(size * nmemb)
	if bytes > 0 then
		receiveHeaderData[#receiveHeaderData + 1] = ffi.string(ptr, bytes)
		receiveHeaderDataSize = receiveHeaderDataSize + bytes
	end
	return bytes
end
local receiveHeaderFunctionPointer = ffi.cast("size_t (*)(char *ptr, size_t size, size_t nmemb, void *stream)", receiveHeaderFunction)

local function readFunction(ptr, size, nmemb, instream)
	-- http://curl.haxx.se/libcurl/c/CURLOPT_READFUNCTION.html
	-- size_t read_callback(char *buffer, size_t size, size_t nitems, void *instream);
	-- This callback function gets called by libcurl as soon as it needs to read data in order to send it to the peer. The data area pointed at by the pointer buffer should be filled up with at most size multiplied with nmemb number of bytes by your function.
	-- Your function must then return the actual number of bytes that it stored in that memory area. Returning 0 will signal end-of-file to the library and cause it to stop the current transfer.
	-- yield()
	local maxBytesToSend = tonumber(size * nmemb)
	local remainingBytes = sendData.size - sendData.bytesSent
	local bytesToSend = remainingBytes
	if bytesToSend > maxBytesToSend then
		bytesToSend = maxBytesToSend
	end
	local sendDataPtr = util.getOffsetPointer(sendData.cdata, sendData.bytesSent)
	ffi.copy(ptr, sendDataPtr, bytesToSend)
	sendData.bytesSent = sendData.bytesSent + bytesToSend
	return bytesToSend
end
local sendFunctionPointer = ffi.cast("size_t (*)(char *ptr, size_t size, size_t nmemb, void *stream)", readFunction)

local function progressFunction(clientp, dltotal, dlnow, ultotal, ulnow)
	-- int http://curl.haxx.se/libcurl/c/CURLOPT_XFERINFOFUNCTION.html
	-- int progress_callback(void *clientp,   curl_off_t dltotal,   curl_off_t dlnow,   curl_off_t ultotal,   curl_off_t ulnow);

	-- old: CURLOPT_PROGRESSFUNCTION
	-- int progress_callback(void *clientp,   double dltotal,   double dlnow,   double ultotal,   double ulnow);
	-- yield()
	local done = -1
	if ultotal > 0 then
		done = math.floor(ulnow / ultotal * 100) -- upload
	elseif dltotal > 0 then
		done = math.floor(dlnow / dltotal * 100) -- download
	end
	if done >= 0 and done ~= donePrev then
		--[[ if ultotal > 0 then
			-- print(l("upload progress %.0f %%, (%d / %d)", done, ulnow, ultotal))
		else
			-- print(l("download progress %.0f %%, (%d / %d)", done, dlnow, dltotal))
		end ]]
		donePrev = done
	end
	return 0 -- Returning a non-zero value from this callback will cause libcurl to abort the transfer and return CURLE_ABORTED_BY_CALLBACK.
end

-- local progressFunctionPointer = ffi.cast("int (*)(void *, curl_off_t, curl_off_t, curl_off_t, curl_off_t)", progressFunction)
local progressFunctionPointer = ffi.cast("int (*)(void *, double, double, double, double)", progressFunction)

local function perform(curl, param, setOptFunc, action)
	local res, err

	if not curl then
		return l("curl library initialization failed")
	end
	lcurl.curl_easy_reset(curl)
	if param.verbose then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_VERBOSE, ffi.cast("long", 1)) -- 1LL fix for basic lua + ffi
	end

	--[[
		* If you want to connect to a site who isn't using a certificate that is
		* signed by one of the certs in the CA bundle you have, you can skip the
		* verification of the server's certificate. This makes the connection
		* A LOT LESS SECURE.
		*
		* If you have a CA cert for the server stored someplace else than in the
		* default bundle, then the CURLOPT_CAPATH option might come handy for
		* you. */
#			ifdef SKIP_PEER_VERIFICATION
			curl_easy_setopt(curl, CURLOPT_SSL_VERIFYPEER, 0L);
#			endif

		* If the site you're connecting to uses a different host name that what
		* they have mentioned in their server certificate's commonName (or
		* subjectAltName) fields, libcurl will refuse to connect. You can skip
		* this check, but this will make the connection less secure. */
#			ifdef SKIP_HOSTNAME_VERFICATION
			curl_easy_setopt(curl, CURLOPT_SSL_VERIFYHOST, 0L);
#			endif
	]]
	if false then
		-- The CURLOPT_CAPATH function apparently does not work in Windows due to some limitation in openssl.
		-- is not built-in in OSX
		local caPath = util.mainPath() .. "bin/ca-bundle.crt"
		if fs.fileExists(caPath) == false then
			util.printError(l("curl, file bin/ca-bundle.crt was not found, parh '%s'", caPath))
		else
			res = lcurl.curl_easy_setopt(curl, C.CURLOPT_CAPATH, caPath)
			-- res 4 = C.CURLE_NOT_BUILT_IN
			-- res 48 = C.CURLE_UNKNOWN_OPTION
			if res ~= 0 and res ~= C.CURLE_NOT_BUILT_IN then
				util.printError(l("curl, setting CURLOPT_CAPATH failed with error %d", tonumber(res)))
			end
		end
	end
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_FOLLOWLOCATION, ffi.cast("long", 1)) -- follow url redirects
	if param.use_ssl then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_USE_SSL, ffi.cast("long", C.CURLUSESSL_ALL)) -- ) -- 0LL fix for basic lua + ffi
	end
	if param.skip_verify_peer then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_SSL_VERIFYPEER, ffi.cast("long", 0)) -- 0LL fix for basic lua + ffi
	end
	if param.skip_verify_host then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_SSL_VERIFYHOST, ffi.cast("long", 0)) -- 0LL fix for basic lua + ffi
	end

	--[[ Define our callback to get called when there's data to be written ]]
	if action == "send" then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_UPLOAD, ffi.cast("long", 1)) -- enable uploading -- 1LL fix for basic lua + ffi
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_READFUNCTION, sendFunctionPointer)
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_INFILESIZE_LARGE, ffi.cast("curl_off_t", sendData.size))
	else
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_WRITEFUNCTION, receiveFunctionPointer)
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_HEADERFUNCTION, receiveHeaderFunctionPointer)
	end
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_NOPROGRESS, ffi.cast("long", 0)) -- will activate progressFunction call -- 0LL fix for basic lua + ffi
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_PROGRESSFUNCTION, progressFunctionPointer)
	-- some servers don't like requests that are made without a user-agent field, so we provide one
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_USERAGENT, "libcurl-agent/1.0")

	if param.url then
		-- param.url = "http://manageri.fi"
		-- local param_c = param.url
		--  lcurl.curl_easy_setopt(curl, C.CURLOPT_URL, param_c)
		lcurl.curl_easy_setopt(curl, C.CURLOPT_URL, param.url)
	end
	if param.username then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_USERNAME, param.username)
	end
	if param.password then
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_PASSWORD, param.password)
		res = lcurl.curl_easy_setopt(curl, C.CURLOPT_HTTPAUTH, C.CURLAUTH_BASIC)
	end
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_FTP_USE_EPSV, ffi.cast("long", 0)) -- 0LL fix for basic lua + ffi
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_ENCODING, "gzip")
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_ENCODING, "")
	res = lcurl.curl_easy_setopt(curl, C.CURLOPT_HTTP_VERSION, ffi.cast("long", C.CURL_HTTP_VERSION_1_1))
	-- res = lcurl.curl_easy_setopt(curl, C.CURLOPT_HTTP_VERSION, ffi.cast("long", C.CURL_HTTP_VERSION_3))
	--[[ Forcing HTTP/3 will make the connection fail if the server isn't
	accessible over QUIC + HTTP/3 on the given host and port.
	Consider using CURLOPT_ALTSVC instead! ]]

	if setOptFunc then
		setOptFunc(curl)
	end
	res = lcurl.curl_easy_perform(curl)
	if res ~= C.CURLE_OK then
		-- error in perform
		local errTxt = ffi.string(lcurl.curl_easy_strerror(res))
		err = "error " .. tostring(res) .. ", '" .. errTxt .. "'"
	end
	lcurl.curl_easy_cleanup(curl)
	return err
end

-- public functions
local function escape(txt)
	if type(txt) == "string" then
		return ffi.string(lcurl.curl_escape(txt, #txt))
	end
	return tostring(txt)
end

local function version()
	return lcurl.curl_version()
end

local function setHeader(curl, param)
	local curl_slist = ffi.newNoAnchor("struct curl_slist *")
	for key, val in pairs(param.header) do
		curl_slist = lcurl.curl_slist_append(curl_slist, key .. ": " .. val) -- escape(val)
	end
	local ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_HTTPHEADER, curl_slist)
	return ret, curl_slist
end

local function getUrl(param, setOptFunc)
	if param.header and type(param.header) ~= "table" then
		local err = util.parameterError("header", "table")
		util.printError(err)
		return nil, err
	end
	local ret, err
	local curl = initCurl()
	local curl_slist
	if setOptFunc == nil and param.header then
		setOptFunc = function()
			if param.header then
				ret, curl_slist = setHeader(curl, param)
			end
		end
	end
	err = perform(curl, param, setOptFunc)
	ret = getReceiveData()
	local retHeader = getReceiveHeaderData()
	if curl_slist then
		lcurl.curl_slist_free_all(curl_slist)
	end
	return ret, err, retHeader
end

local function postUrl(param)
	local ret, err
	if not (type(param.post_data) == "table" or type(param.post_data) == "string") then
		err = util.parameterError("post_data", "table or string")
	elseif param.header and type(param.header) ~= "table" then
		err = util.parameterError("header", "table")
	end
	if err then
		util.printError(err)
		return nil, err
	end
	local curl = initCurl()
	local curl_slist
	local function setOptFunc()
		if param.header then
			ret, curl_slist = setHeader(curl, param)
		end
		-- https://curl.haxx.se/libcurl/c/http-post.html
		-- curl_easy_setopt(curl, CURLOPT_POSTFIELDS, "name=daniel&project=curl")
		if type(param.post_data) == "table" then
			local data = {}
			for key, val in pairs(param.post_data) do
				data[#data + 1] = escape(key) .. "=" .. escape(val)
			end
			ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_POSTFIELDS, table.concat(data, "&"))
		else
			if param.header["Content-Encoding"] == "gzip" then
				if gzip == nil then
					gzip = require "compress/zlib"
				end
				local gzipData = gzip.compress(param.post_data)
				ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_POSTFIELDS, gzipData)
			else
				ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_POSTFIELDS, param.post_data)
			end
		end
	end
	err = perform(curl, param, setOptFunc)
	ret = getReceiveData()
	local retHeader = getReceiveHeaderData()
	if curl_slist then
		lcurl.curl_slist_free_all(curl_slist)
	end
	if ret == nil and err == nil then
		err = l("curl postUrl did not return any data")
	end
	return ret, err, retHeader
end

local function sendUrl(param, setOptFunc)
	local ret, err
	local curl = initCurl()
	err = perform(curl, param, setOptFunc, "send")
	ret = getReceiveData()
	local retHeader = getReceiveHeaderData()
	sendData = {cdata = nil, bytesSent = 0, size = 0}
	return ret, err, retHeader
end

local function deleteFtpFile(param)
	-- http://curl.haxx.se/libcurl/c/CURLOPT_QUOTE.html
	local ret, err
	local curl_slist
	local function setOpt(curl)
		curl_slist = ffi.newNoAnchor("struct curl_slist *")
		curl_slist = lcurl.curl_slist_append(curl_slist, "DELE " .. param.from)
		ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_QUOTE, curl_slist)
	end
	ret, err = getUrl(param, setOpt)
	lcurl.curl_slist_free_all(curl_slist)
	return ret, err
end

local function moveFtpFile(param)
	-- http://curl.haxx.se/libcurl/c/CURLOPT_QUOTE.html
	if not param.from then
		return nil, l "data parameter 'from' does not exist"
	elseif not param.to then
		return nil, l "data parameter 'to' does not exist"
	end
	local ret, err
	local curl_slist
	local function setOpt(curl)
		curl_slist = ffi.newNoAnchor("struct curl_slist *")
		curl_slist = lcurl.curl_slist_append(curl_slist, "RNFR " .. param.from)
		curl_slist = lcurl.curl_slist_append(curl_slist, "RNTO " .. param.to)
		ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_QUOTE, curl_slist)
	end
	ret, err = getUrl(param, setOpt)
	lcurl.curl_slist_free_all(curl_slist)
	return ret, err
end

local function sendFtpFile(param)
	-- local cmd = curlPath .. " -k --ftp-ssl --disable-epsv --ftp-skip-pasv-ip -T " .. dirMg .. "/" .. newZipFileName .. " ftp://g3admin:G3Xmaa@ftp.manageapp.com/drop_box/_manageri/_mg_"..mgVersion.."_ftp/"
	if not param.data then
		return nil, l "data parameter does not exist"
	end
	sendData = {cdata = param.data, bytesSent = 0, size = #param.data}
	return sendUrl(param)
end

local function receiveFtpFile(param)
	-- http://curl.haxx.se/libcurl/c/ftpget.html
	return getUrl(param)
end

local function receiveFtpList(param)
	-- http://curl.haxx.se/libcurl/c/ftpget.html
	local ret, err = getUrl(param)
	if err then
		return ret, err
	end
	return ftpFileList(ret)
end

local function sendEmail(param)
	if not param.from then
		return nil, l "parameter from does not exist"
	elseif not (param.to or param.cc or param.bcc) then
		return nil, l "parameter to, cc or bcc does not exist"
	elseif not param.smtp then
		return nil, l "parameter smtp does not exist"
		-- elseif not param.username then
		-- 	return nil,l"parameter username does not exist"
		-- elseif not param.password then
		-- 	return nil,l"parameter password does not exist"
	elseif not param.body then
		return nil, l "parameter body does not exist"
	end

	local ret
	local curl_slist
	local function setOpt(curl)
		--[[ this is the URL for your mailserver - you can also use an smtps:// URL
     * here ]]
		ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_URL, param.smtp) -- "smtp://mail.example.net."

		--[[ Note that this option isn't strictly required, omitting it will result in
     * lcurl will sent the MAIL FROM command with no sender data. All
     * autoresponses should have an empty reverse-path, and should be directed
     * to the address in the reverse-path which triggered them. Otherwise, they
     * could cause an endless loop. See RFC 5321 Section 4.5.5 for more details.
     ]]
		ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_FROM, param.from)

		--[[ You provide the payload (headers and the body of the message) as the
     * "data" element. There are two choices, either:
     * - provide a callback function and specify the function name using the
     * C.CURLOPT_READFUNCTION option or
     * - just provide a FILE pointer that can be used to read the data from.
     * The easiest case is just to read from standard input, (which is available
     * as a FILE pointer) as shown here.
     ]]

		-- http://stackoverflow.com/questions/2750211/sending-bcc-emails-using-a-smtp-server
		-- curl_slist must contain to, cc and bcc, what is shown is in email message itself
		curl_slist = ffi.newNoAnchor("struct curl_slist *")
		if param.to and param.to ~= "" then
			if type(param.to) == "table" then
				for _, address in ipairs(param.to) do
					curl_slist = lcurl.curl_slist_append(curl_slist, address)
				end
			else
				curl_slist = lcurl.curl_slist_append(curl_slist, param.to)
			end
		end
		if param.cc and param.cc ~= "" then
			if type(param.cc) == "table" then
				for _, address in ipairs(param.cc) do
					curl_slist = lcurl.curl_slist_append(curl_slist, address)
				end
			else
				curl_slist = lcurl.curl_slist_append(curl_slist, param.cc)
			end
		end
		if param.bcc and param.bcc ~= "" then
			if type(param.bcc) == "table" then
				for _, address in ipairs(param.bcc) do
					curl_slist = lcurl.curl_slist_append(curl_slist, address)
				end
			else
				curl_slist = lcurl.curl_slist_append(curl_slist, param.bcc)
			end
		end
		ret = lcurl.curl_easy_setopt(curl, C.CURLOPT_RCPT, curl_slist)
	end
	sendData = {cdata = param.body, bytesSent = 0, size = #param.body}
	if param.debug then
		util.printTable(param, "send email parameters")
	end
	local err, retHeader
	ret, err, retHeader = sendUrl(param, setOpt)
	lcurl.curl_slist_free_all(curl_slist)
	return ret, err, retHeader

	--[[
	param == {
		from = string,
		to = string or string-table,
		cc = string or string-table,
		bcc = string or string-table,
		body = string,
		[user = string,]
		[password = string,]
		[server = string,]
		[port = number,]
		[domain = string,]
	}
	]]

end

-- local CURL_CTX = curl_easy_init() -- wrapper func
return {
	version = version,
	escape = escape,
	getUrl = getUrl,
	postUrl = postUrl,
	sendEmail = sendEmail,
	ftpFileList = ftpFileList, -- returns lua table from ftp file list text
	deleteFtpFile = deleteFtpFile,
	moveFtpFile = moveFtpFile,
	sendFtpFile = sendFtpFile,
	receiveFtpFile = receiveFtpFile,
	receiveFtpList = receiveFtpList
}
