--- database-odbc-ffi.lua
require "db/database-odbc-ffi_h"

local odbcffi = {}
setmetatable(odbcffi, {
	__index = odbcffi
	-- __gc = environment_methods.close
})

local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local utf = require "utf"
local l = require"lang".l

local odbc -- psqlodbc,psql
if util.isWin() then -- ffi.os == "Windows" then
	local ok = false
	--[[ no Easysoft odbc driver - it uses Oracle driver and leaks memory also
	if ok then
    print("load: regex.dll")
    ok = pcall(ffi.load, "C:/Program Files (x86)/Common Files/Easysoft/Shared/regex.dll")
    if ok then
      print("load: essupp.dll")
      ok = pcall(ffi.load, "C:/Program Files (x86)/Easysoft/Easysoft ODBC-Oracle Driver/32-bit/Libs/essupp.dll")
    end
    if ok then
      print("load: esoraset.dll")
      ok = pcall(ffi.load, "C:/Program Files (x86)/Easysoft/Easysoft ODBC-Oracle Driver/32-bit/Libs/qt332.dll")
    end
    if ok then
      print("load: esoraset.dll")
      ok = pcall(ffi.load, "C:/Program Files (x86)/Easysoft/Easysoft ODBC-Oracle Driver/32-bit/Libs/esoraset.dll")
    end
    if ok then
      print("load: esoracle.dll")
      ok, odbc = pcall(ffi.load, "C:/Program Files (x86)/Easysoft/Easysoft ODBC-Oracle Driver/32-bit/Libs/esoracle.dll")
    end
  end
	--]]
	if not ok then
		odbc = ffi.load("C:/WINDOWS/System32/odbc32.dll") -- util.pathBin().."odbc32.dll") -- odbc.dll
		-- odbc = ffi.load("C:/WINDOWS/SYSWOW64/odbc32.dll") -- 32 bit driver in 64 bit windows
	end
elseif util.isMac() then
	-- odbc =  util.loadDll("libodbc.dylib")
	local ok
	ok, odbc = pcall(ffi.load, "libodbc.2.dylib") -- no warning print needed, use pcall
	if ok == false then
		ok, odbc = pcall(ffi.load, "/opt/homebrew/lib/libodbc.2.dylib")
	end
	if ok == false then
		odbc = nil
	end
else -- linux
	odbc = util.loadDll("libodbc.so.2")
end

if odbc == nil then
	-- util.printWarning("libodbc was not found")
	return nil
end
-- local env -- global environment
local environment_methods, connection_methods, cursor_methods
local bufferSize = 4096
local buffer -- cache this buffer
-- local driver = require "luasql.odbc"
-- local env = driver.odbc()

--[[
odbcinst -j
unixODBC 2.3.2
DRIVERS............: /opt/homebrew/Cellar/unixodbc/2.3.2/etc/odbcinst.ini
SYSTEM DATA SOURCES: /opt/homebrew/Cellar/unixodbc/2.3.2/etc/odbc.ini
FILE DATA SOURCES..: /opt/homebrew/Cellar/unixodbc/2.3.2/etc/ODBCDataSources
USER DATA SOURCES..: /Users/manage/.odbc.ini
SQLULEN Size.......: 8
SQLLEN Size........: 8
SQLSETPOSIROW Size.: 8
manage-mbp:~ manage$

use:
~/Library/ODBC/odbcinst.ini
~/.odbc.ini
]]

--[[
odbcinst -j
unixODBC 2.3.2
DRIVERS............: /opt/homebrew/Cellar/unixodbc/2.3.2/etc/odbcinst.ini
SYSTEM DATA SOURCES: /opt/homebrew/Cellar/unixodbc/2.3.2/etc/odbc.ini
FILE DATA SOURCES..: /opt/homebrew/Cellar/unixodbc/2.3.2/etc/ODBCDataSources
USER DATA SOURCES..: /Users/manage/.odbc.ini
SQLULEN Size.......: 8
SQLLEN Size........: 8
SQLSETPOSIROW Size.: 8
manage-mbp:~ manage$

use:
~/Library/ODBC/odbcinst.ini
~/.odbc.ini
]]

-- utility functions
local infoState = ffi.newAnchor("SQLCHAR[6]")
local infoNativeError = ffi.newAnchor("SQLINTEGER[1]")
local infoMsgSize = ffi.newAnchor("SQLSMALLINT[1]")

local function getInfo(ctype, handle)
	local msg = ffi.newNoAnchor("char[?]", C.SQL_MAX_MESSAGE_LENGTH)
	local i = 1 -- Status records are numbered from 1.
	local err = {}
	while (1) do
		local ret = odbc.SQLGetDiagRec(ctype, handle[0], i, infoState, infoNativeError, msg, ffi.sizeof(msg), infoMsgSize)
		if ret == C.SQL_NO_DATA or ret < 0 then -- C.SQL_NO_DATA == 100, SQL_INVALID_HANDLE == -2
			break
		end
		err[i] = utf.latin9ToUtf8(ffi.string(msg, infoMsgSize[0]))
		i = i + 1
	end
	return table.concat(err, "\n")
end

local function sqlPrintInfo(ctype, handle, ret)
	if not ret or ret == 0 then
		util.printWarning(l "  odbc: " .. getInfo(ctype, handle))
	else
		util.printWarning(l "  odbc: " .. getInfo(ctype, handle) .. ": " .. tostring(ret))
	end
end

local function sqlPrintError(errtxt, errnum, doPrint)
	errtxt = l "  odbc: '" .. errtxt .. "'"
	if not errnum or errnum == 0 then
		if doPrint ~= false then
			util.printRed(errtxt)
		end
	else
		errtxt = errtxt .. ": " .. tostring(errnum)
		if doPrint ~= false then
			util.printRed(errtxt)
		end
	end
	return nil, errtxt
end

local function fail(ctype, handle, ret, doPrint)
	local errtxt = getInfo(ctype, handle)
	return sqlPrintError(errtxt, ret, doPrint) -- l"environment does not exist", -1)
end

local function sqlInfo(a)
	return a == C.SQL_SUCCESS_WITH_INFO
end

local function sqlError(a)
	return a ~= C.SQL_SUCCESS and a ~= C.SQL_SUCCESS_WITH_INFO
end

local function sqltypetolua(datatype)
	-- change to Lua table? faster?
	if datatype == C.SQL_UNSPECIFIED -- C.SQL_UNKNOWN_TYPE
	or datatype == C.SQL_CHAR or datatype == C.SQL_VARCHAR or datatype == C.SQL_TYPE_DATE or datatype == C.SQL_TYPE_TIME or datatype == C.SQL_TYPE_TIMESTAMP or datatype == C.SQL_DATE -- or datatype == C.SQL_INTERVAL
	or datatype == C.SQL_TIMESTAMP or datatype == C.SQL_LONGVARCHAR or datatype == C.SQL_WCHAR or datatype == C.SQL_WVARCHAR or datatype == C.SQL_WLONGVARCHAR or datatype == C.SQL_SS_TIMESTAMPOFFSET or datatype == C.SQL_SS_TIME2 then
		return "string"
	end
	if datatype == C.SQL_BIGINT or datatype == C.SQL_TINYINT or datatype == C.SQL_NUMERIC or datatype == C.SQL_DECIMAL or datatype == C.SQL_INTEGER or datatype == C.SQL_SMALLINT or datatype == C.SQL_FLOAT or datatype == C.SQL_REAL or datatype == C.SQL_DOUBLE then
		return "number"
	end
	if datatype == C.SQL_BINARY or datatype == C.SQL_VARBINARY or datatype == C.SQL_LONGVARBINARY then
		return "binary" -- !!!!!! nao seria string?
	end
	if datatype == C.SQL_BIT then
		return "boolean"
	else
		return "unknown: '" .. tostring(datatype) .. "'"
	end
end

-- worker functions

--- Retrieves data from the i_th column in the current row
local push_column
do
	local num = ffi.newAnchor("double[1]")
	local got = ffi.newAnchor("SQLLEN[1]")
	local b = ffi.newAnchor("char[1]")
	push_column = function(coltype, hstmt, i)
		if coltype == "number" then
			local ret = odbc.SQLGetData(hstmt[0], i, C.SQL_C_DOUBLE, num, 0, got)
			if sqlError(ret) then
				fail(C.hSTMT, hstmt)
				return nil
			elseif sqlInfo(ret) then
				sqlPrintInfo(C.hSTMT, hstmt)
			end
			if got[0] == C.SQL_NULL_DATA then
				return "null"
			end
			return num[0]
		elseif coltype == "boolean" then
			local ret = odbc.SQLGetData(hstmt[0], i, C.SQL_C_BIT, b, 0, got)
			if sqlError(ret) then
				fail(C.hSTMT, hstmt)
				return nil
			elseif sqlInfo(ret) then
				sqlPrintInfo(C.hSTMT, hstmt)
			end
			if got[0] == C.SQL_NULL_DATA then
				return "null"
			end
			return b[0] ~= 0
		elseif coltype == "string" or coltype == "binary" then
			-- C.LUAL_BUFFERSIZE in Lua headers
			local retBuffer = {}
			local stype
			if coltype == "string" then
				stype = C.SQL_C_CHAR
			else
				stype = C.SQL_C_BINARY
			end
			if not buffer then
				buffer = ffi.newAnchor("char[?]", bufferSize)
			end
			got[0] = 0
			local ret = odbc.SQLGetData(hstmt[0], i, stype, buffer, bufferSize, got)
			if sqlError(ret) then
				fail(C.hSTMT, hstmt)
				return nil
				-- elseif sqlInfo(ret) then
				-- sqlPrintInfo(C.hSTMT, hstmt)
			end
			if got[0] == C.SQL_NULL_DATA then
				return "null"
			end

			-- concat intermediary chunks
			while ret == C.SQL_SUCCESS_WITH_INFO do
				if got[0] >= bufferSize or got[0] == C.SQL_NO_TOTAL then
					got[0] = bufferSize
					-- get rid of null termination in string block
					if stype == C.SQL_C_CHAR then
						got[0] = got[0] - 1
					end
				end
				retBuffer[#retBuffer + 1] = ffi.string(buffer, got[0])
				ret = odbc.SQLGetData(hstmt[0], i, stype, buffer, bufferSize, got)
			end

			-- concat last chunk
			if ret == C.SQL_SUCCESS then
				if got[0] >= bufferSize or got[0] == C.SQL_NO_TOTAL then
					got[0] = bufferSize
					-- get rid of null termination in string block
					if stype == C.SQL_C_CHAR then
						got[0] = got[0] - 1
					end
				end
				retBuffer[#retBuffer + 1] = ffi.string(buffer, got[0])
			end

			if sqlError(ret) then
				fail(C.hSTMT, hstmt, ret)
				return nil
			end
			-- return everything we got
			return table.concat(retBuffer)
		else
			util.printWarning(l("unknown odbc column type '%s'", tostring(coltype)))
			return "(error)"
		end
	end
end -- do

--- Get another row of the given cursor
local function cur_fetch(cur)
	local ret = odbc.SQLFetch(cur.hstmt[0]) -- odbc.SQLFetchScroll(cur.hstmt[0], C.SQL_FETCH_NEXT, 0) --
	if ret == C.SQL_NO_DATA then
		-- return -1
		return nil
	elseif sqlError(ret) then
		fail(C.hSTMT, cur.hstmt)
		-- return -2
		return nil
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, cur.hstmt)
	end

	local rows = {}
	for i = 1, cur.numcols do
		rows[i] = push_column(cur.coltypes[i], cur.hstmt, i)
	end
	return rows
end

--- Get another row of the given cursor
local function cur_close(cur)
	if cur.closed and cur.closed ~= 0 then
		sqlPrintError(l "cursor was already closed")
		return 1
	end
	-- Nullify structure fields
	cur.closed = 1

	local ret = odbc.SQLCloseCursor(cur.hstmt[0])
	if sqlError(ret) then
		return fail(C.hSTMT, cur.hstmt, ret)
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, cur.hstmt, ret)
	end

	ret = odbc.SQLFreeHandle(C.hSTMT, cur.hstmt[0])
	if sqlError(ret) then
		return fail(C.hSTMT, cur.hstmt, ret)
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, cur.hstmt, ret)
	end

	-- Decrement cursor counter on connection object
	cur.conn.cur_counter = cur.conn.cur_counter - 1
	return nil -- nil = no error
end

--- Returns the table with column names
local function cur_colnames(cur)
	return cur.colnames
end

--- Returns the table with column types
local function cur_coltypes(cur)
	return cur.coltypes
end

local nameBuffer = ffi.newAnchor("SQLCHAR[256]")
local namelen = ffi.newAnchor("SQLSMALLINT[1]")
local datatype = ffi.newAnchor("SQLSMALLINT[1]")

--- Creates two tables with the names and the types of the columns
local function create_colinfo(cur)
	local names = {}
	local types = {}
	for i = 1, cur.numcols do
		local ret = odbc.SQLDescribeCol(cur.hstmt[0], i, buffer, ffi.sizeof(nameBuffer), namelen, datatype, nil, nil, nil)
		-- if (ret == SQL_ERROR) return fail(L, hSTMT, cur.hstmt);
		names[i] = ffi.string(nameBuffer, namelen[0])
		types[i] = sqltypetolua(datatype[0])
	end
	cur.colnames = names
	cur.coltypes = types
	setmetatable(cur, {__index = cursor_methods})
	return cur
end

--- Creates a cursor table
local function create_cursor(conn, hstmt, numcols, numrows, statement)
	conn.cur_counter = conn.cur_counter + 1
	-- fill in structure
	local cur = {}
	cur.closed = 0
	cur.conn = conn -- C.LUA_NOREF
	cur.numcols = tonumber(numcols) -- may be int64
	cur.numrows = tonumber(numrows)
	cur.colnames = {} -- C.LUA_NOREF
	cur.coltypes = {} -- C.LUA_NOREF
	cur.hstmt = hstmt
	cur.statement = statement
	-- make and store column information table
	return create_colinfo(cur)
end

--- Executes a SQL statement.
-- Returns
--   cursor object: if there are results or
--   row count: number of rows affected by statement if no results

local hstmt = ffi.newAnchor("SQLHSTMT[1]")
local numcols = ffi.newAnchor("SQLSMALLINT[1]")
local numrows = ffi.newAnchor("SQLLEN[1]")

local statementBufferSize = 500 * 1000 -- 500 kB
local statementBuffer
local function conn_execute(conn, statement)
	-- fix this
	local ret = odbc.SQLAllocHandle(C.hSTMT, conn.hdbc[0], hstmt)
	if sqlError(ret) then
		return sqlPrintError(l "execute allocation error", ret)
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end

	local len = #statement
	local statement_c
	if len < statementBufferSize then -- must be smaller than because c-string end \0
		if statementBuffer == nil then
			statementBuffer = ffi.createBuffer(statementBufferSize) -- ffi.newNoAnchor("uint8_t[?]",  buffSize)
		end
		statement_c = ffi.copyStringToBuffer(statementBuffer, statement, len)
	else
		statement_c = statement
	end
	-- local statement_c = ffi.newNoAnchor("SQLCHAR[?]", len+1)
	-- ffi.copy(statement_c[0], statement)
	-- statement = statement:gmatch("[^\r\n]+")

	ret = odbc.SQLExecDirect(hstmt[0], ffi.cast("SQLCHAR*", statement_c), ffi.cast("SQLINTEGER", len)) -- , C.SQL_NTS)
	if sqlError(ret) then
		local r, e = fail(C.hSTMT, hstmt, ret)
		odbc.SQLFreeHandle(C.hSTMT, hstmt[0])
		return r, e
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end
	--[[
	ret = odbc.SQLPrepare(hstmt[0], ffi.cast("SQLCHAR*", statement_c), len) -- , C.SQL_NTS)
	if sqlError(ret) then
		sqlPrintError(l"execute allocation error", ret)
		odbc.SQLFreeHandle(C.hSTMT, hstmt[0])
		return ret
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end

	-- execute the statement
	ret = odbc.SQLExecute(hstmt[0])
	if sqlError(ret) then
		fail(C.hSTMT, hstmt)
		odbc.SQLFreeHandle(C.hSTMT, hstmt[0])
		return ret
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end
	]]

	-- determine the number of results
	ret = odbc.SQLNumResultCols(hstmt[0], numcols)
	if sqlError(ret) then
		local r, e = fail(C.hSTMT, hstmt, ret)
		odbc.SQLFreeHandle(C.hSTMT, hstmt[0])
		return r, e
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end
	-- if action has no results (e.g., UPDATE)
	ret = odbc.SQLRowCount(hstmt[0], numrows)
	if sqlError(ret) then
		local r, e = fail(C.hSTMT, hstmt, ret)
		odbc.SQLFreeHandle(C.hSTMT, hstmt[0])
		return r, e
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end

	-- must NOT free hstmt if cursor was created!
	if numcols[0] > 0 then
		--  if there is a results table (e.g., SELECT)
		return create_cursor(conn, hstmt, numcols[0], numrows[0], statement)
	end
	odbc.SQLFreeHandle(C.hSTMT, hstmt[0])
	return numrows[0]
end

--- Commit the transaction
local function conn_commit(conn)
	local ret = odbc.SQLEndTran(C.hDBC, conn.hdbc[0], C.SQL_COMMIT)
	if sqlError(ret) then
		fail(C.hSTMT, conn.hdbc)
		return -1
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end
	return 0
end

--- Rollback the transaction
local function conn_rollback(conn)
	local ret = odbc.SQLEndTran(C.hDBC, conn.hdbc[0], C.SQL_ROLLBACK)
	if sqlError(ret) then
		fail(C.hSTMT, conn.hdbc)
		return -1
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc)
	end
	return 0
end

--- Set autocommit mode
local function conn_setautocommit(conn, setOn)
	local ret
	if setOn then
		setOn = C.SQL_AUTOCOMMIT_ON
	else
		setOn = C.SQL_AUTOCOMMIT_OFF
	end
	ret = odbc.SQLSetConnectAttr(conn.hdbc[0], C.SQL_ATTR_AUTOCOMMIT, ffi.cast("SQLPOINTER", setOn), 0)
	if sqlError(ret) then
		fail(C.hSTMT, conn.hdbc) -- hDBC or hSTMT?
		return -1
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hSTMT, conn.hdbc) -- hDBC or hSTMT?
	end
	return 0
end

--- Closes a connection
local function conn_close(conn)
	if not conn then
		sqlPrintError(l "connection is nil")
		return -2
	end
	if conn.closed == 1 then
		sqlPrintError(l "connection was already closed")
		return 1
	end
	if conn.cur_counter > 0 then
		sqlPrintError(l "connection has open cursors")
		return -1
	end
	-- Decrement connection counter on environment object
	conn.env.conn_counter = conn.env.conn_counter - 1
	-- nullify structure fields
	conn.closed = 1

	local ret = odbc.SQLDisconnect(conn.hdbc[0])
	if sqlError(ret) then
		sqlPrintError(l("SQLDisconnect error: '%s'", tostring(ret)))
		return -2
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hDBC, conn.hdbc)
	end

	ret = odbc.SQLFreeHandle(C.hDBC, conn.hdbc[0])
	if sqlError(ret) then
		fail(C.hDBC, conn.hdbc)
		conn.env.henv = nil
		return ret
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hDBC, conn.hdbc)
	end

	conn.cur_counter = 0
	conn.env = nil
	conn.hdbc = nil
	return 1
end

--- Create a new connection object
local function create_connection(env, hdbc)
	-- fill in structure
	local conn = {}
	conn.closed = 0
	conn.cur_counter = 0
	conn.env = env
	conn.hdbc = hdbc
	if not conn_setautocommit(conn, true) then
		return nil
	end
	env.conn_counter = env.conn_counter + 1

	setmetatable(conn, {__index = connection_methods})
	return conn
end

--- Creates and returns a connection object
-- Lua Input: source [, user [, pass]]
--   source: data source
--   user, pass: data source authentication information
-- Lua Returns:
--   connection object if successfull
--   nil and error message otherwise.

local hdbc = ffi.newAnchor("SQLHDBC[1]")

local function env_connect(env, sourcename, username, password)
	if not env or not env.henv then
		sqlPrintError(l "environment does not exist", 0)
		return nil, -1
	end
	-- tries to allocate connection handle
	local ret = odbc.SQLAllocHandle(C.hDBC, env.henv[0], hdbc)
	if sqlError(ret) then
		sqlPrintError(l "connection allocation error", ret)
		return nil, ret
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hDBC, env.henv)
	end
	-- tries to connect handle
	sourcename = sourcename or ""
	username = username or ""
	password = password or ""
	local sourcename_c = ffi.cast("SQLCHAR*", sourcename)
	local username_c = ffi.cast("SQLCHAR*", username)
	local password_c = ffi.cast("SQLCHAR*", password)
	-- ret = odbc.SQLConnect(hdbc[0], sourcename_c, C.SQL_NTS, username_c, C.SQL_NTS, password_c, C.SQL_NTS)
	ret = odbc.SQLConnect(hdbc[0], sourcename_c, #sourcename, username_c, #username, password_c, #password)
	if sqlError(ret) then
		-- ret = fail(C.hDBC, hdbc)
		local _, err = fail(C.hDBC, hdbc, ret, false) -- false == do not print error text
		odbc.SQLFreeHandle(C.hDBC, hdbc[0])
		return nil, err
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hDBC, hdbc)
	end
	-- success, return connection object
	return create_connection(env, hdbc)
end

--- Closes an environment object
local function env_close(env)
	if not env or not env.henv then
		sqlPrintError(l "environment does not exist", -1)
		return -1
	end
	if env.closed == 1 then
		return 1
	end
	if env.conn_counter > 1 then
		sqlPrintError(l "there are open connections", 0)
	end
	-- env.conn_counter = 0
	buffer = nil
	env.closed = 1
	local ret = odbc.SQLFreeHandle(C.hENV, env.henv[0]) -- C.hDBC
	if sqlError(ret) then
		-- ret = fail(C.hENV, env.henv);
		fail(C.hENV, env.henv)
		env.henv = nil
		return ret
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hENV, env.henv)
	end
	return 1
end

local henv = ffi.newAnchor("SQLHENV[1]")
--- Creates an environment and returns it
local function create_environment()
	local ret = odbc.SQLAllocHandle(C.hENV, ffi.cast("void*", C.SQL_NULL_HANDLE), henv)
	if sqlError(ret) then
		return sqlPrintError(l "error when creating ODBC environment", ret)
	end
	-- local env = ffi.newNoAnchor("struct env_data")

	ret = odbc.SQLSetEnvAttr(henv[0], C.SQL_ATTR_ODBC_VERSION, ffi.cast("void*", C.SQL_OV_ODBC3), 0)
	if sqlError(ret) then
		sqlPrintError(l "error setting SQL version", ret)
		-- local ret2 = odbc.SQLFreeHandle (C.hENV, henv[0])
		odbc.SQLFreeHandle(C.hENV, henv[0])
		return nil, ret
	elseif sqlInfo(ret) then
		sqlPrintInfo(C.hENV, henv)
	end

	-- env = (env_data *)lua_newuserdata (L, sizeof (env_data));
	-- luasql_setmeta (L, LUASQL_ENVIRONMENT_ODBC);
	-- /* fill in structure */
	local env = {closed = 0, conn_counter = 0, henv = henv}
	setmetatable(env, {__index = environment_methods})
	return env
end

--[[
/*
** Create metatables for each class of object.
*/
static void create_metatables (lua_State *L) {
	struct luaL_Reg environment_methods[] = {
		{"__gc", env_close}, /* Should this method be changed? */
		{"close", env_close},
		{"connect", env_connect},
		{NULL, NULL},
	};
	struct luaL_Reg connection_methods[] = {
		{"__gc", conn_close}, /* Should this method be changed? */
		{"close", conn_close},
		{"execute", conn_execute},
		{"commit", conn_commit},
		{"rollback", conn_rollback},
		{"setautocommit", conn_setautocommit},
		{NULL, NULL},
	};
	struct luaL_Reg cursor_methods[] = {
		{"__gc", cur_close}, /* Should this method be changed? */
		{"close", cur_close},
		{"fetch", cur_fetch},
		{"getcoltypes", cur_coltypes},
		{"getcolnames", cur_colnames},
		{NULL, NULL},
	};
	luasql_createmeta (L, LUASQL_ENVIRONMENT_ODBC, environment_methods);
	luasql_createmeta (L, LUASQL_CONNECTION_ODBC, connection_methods);
	luasql_createmeta (L, LUASQL_CURSOR_ODBC, cursor_methods);
	lua_pop (L, 3);
}
]]

environment_methods = {
	-- __gc = env_close,
	close = env_close,
	connect = env_connect
}

connection_methods = {
	-- __gc = conn_close,
	close = conn_close,
	execute = conn_execute,
	commit = conn_commit,
	rollback = conn_rollback,
	setautocommit = conn_setautocommit
}

cursor_methods = {
	-- __gc = cur_close,
	close = cur_close,
	fetch = cur_fetch,
	getcoltypes = cur_coltypes,
	getcolnames = cur_colnames
}

--[[
setmetatable(environment_methods, {__index = environment_methods})
setmetatable(connection_methods, {__index = connection_methods})
setmetatable(cursor_methods, {__index = cursor_methods})
--]]

odbcffi.odbc = create_environment

return odbcffi
