-- lib/db/database-odbc.lua
--[[
brew install unixodbc freetds
odbcinst -j

code ~/nc/nc-server/preference/odbc/usr-local-ect-odbcinst.ini
code ~/nc/nc-server/preference/odbc/usr-local-ect-odbc.ini
code /opt/homebrew/etc/freetds.conf
code /opt/homebrew/etc/odbcinst.ini
# cp ~/nc/nc-server/preference/odbc/usr-local-ect-odbcinst.ini /opt/homebrew/etc/odbcinst.ini
code /opt/homebrew/etc/odbc.ini
# cp ~/nc/nc-server/preference/odbc/usr-local-ect-odbc.ini /opt/homebrew/etc/odbc.ini

OLD:
code ~/nc/nc-server/preference/odbc/usr-local-ect-odbcinst.ini
code ~/nc/nc-server/preference/odbc/usr-local-ect-odbc.ini
code /usr/local/etc/freetds.conf
code /usr/local/etc/odbcinst.ini
# cp ~/nc/nc-server/preference/odbc/usr-local-ect-odbcinst.ini /usr/local/etc/odbcinst.ini
code /usr/local/etc/odbc.ini
# cp ~/nc/nc-server/preference/odbc/usr-local-ect-odbc.ini /usr/local/etc/odbc.ini
# open https://docs.snowflake.com/en/user-guide/odbc-linux.html#simba-snowflake-ini-file-driver-manager-and-logging
# code /opt/snowflake/snowflakeodbc/lib/universal/simba.snowflake.ini

# test: isql -v "SnowflakeDSII" nadacode passw
]] --
local driver = require "db/database-odbc-ffi"
local env
if type(driver) == "table" then
	env = driver.odbc()
end
local util = require "util"
local dconn = require "dconn"
local dschema = require "dschema"
local recDataSet = require"recdata".set
local peg = require "peg"
local import = require "import"
local parseAfter, found, startsWith = import.from(peg, "parseAfter, found, startsWith")
local dconv -- delay load, usually not needed
local lang = require "lang"
local l = lang.l
local utf = require "utf"
local isWin = util.isWin()
-- local icu = require "unicode/unicodeIcu"
-- local icuFrom = "ISO-8859-15"
-- local icuTo = "UTF-8"
local currentCharset
local winCharset = "cp1252" -- "latin9"
local convertCharset = {latin9 = utf.latin9ToUtf8, latin9reverse = utf.utf8ToLatin9, cp1252 = utf.cp1252ToUtf8, cp1252reverse = utf.utf8ToCp1252}
local maxReturnRows = math.huge
local defaultPass = "pps"

local function maxSaveRows()
	return 1
end

local function disconnect(conn)
	conn:close()
	-- env:close() -- should close on last, open on first?
end

local function setReturnRowLimit(rows)
	maxReturnRows = rows
end

local function connect(prf)
	if not env then
		return nil, l "ODBC driver is not loaded", true -- 3. return parameter is driverErr
	end
	local conn, err
	if prf.password then
		conn, err = env:connect(prf.host, prf.database_user, prf.password) -- database == dns name, prf.password or defaultPass
	end
	if conn == nil then
		conn, err = env:connect(prf.host, prf.database_user, defaultPass) -- try with defaultPass
	end
	if err then
		err = err .. l " - preference: " .. prf.name .. l ", host (dns): " .. prf.host
	else
		err = conn:setautocommit(true)
	end
	return conn, err
end

local function execute(sqlExecute, option)
	-- print("  odbc execute: "..sqlExecute)
	local conn = dconn.driverConnection(nil, option.database)
	currentCharset = dconn.currentConnection().charset
	if isWin and currentCharset == nil then
		currentCharset = winCharset
	end
	if currentCharset and currentCharset ~= "" then
		-- execute statements need reverse charset
		if not convertCharset[currentCharset .. "reverse"] then
			local err = l("connection charset must be one of these: %s", table.concat(util.tableKeyToArray(convertCharset), ", "))
			util.printError(err)
		else
			sqlExecute = convertCharset[currentCharset .. "reverse"](sqlExecute)
		end
	end
	local cur, err = conn:execute(sqlExecute)
	if cur == nil then
		-- err = unicode.fromUtf16(err)
		err = l("error in odbc sql execute statement: '%s'\n  " .. sqlExecute, tostring(err))
		return nil, err
	end
	return cur
end

local function columnTypes(cur)
	local ret = cur:getcoltypes()
	return ret
end

local function columnNames(cur)
	local ret = cur:getcolnames()
	return ret
end

--[[
local function columnCount(cur)
	if cur.numcols then
		return cur.numcols
	end
	local ret = columnTypes(cur)
	return #ret
end

--]]

local function rowCount(cur)
	if cur.numrows then
		return cur.numrows
	end
	local ret = cur:numrows()
	return ret
end

local function selectionToArray(cur, fieldNameArray, option, returnTableType)
	if not cur then
		local err = l("cursor is nil")
		util.printError(err)
		return nil, {error = err}
	end
	local colType = columnTypes(cur)
	if #colType < 1 then
		local err = l("column type count < 1")
		util.printError(err)
		return nil, {error = err}
	end
	local colName = columnNames(cur)
	local colCount = #colType -- columnCount(cur)
	local rowCountTotal = rowCount(cur)
	local ret = util.newTable(0, colCount)
	local colTagName = util.newTable(colCount, 0)
	-- print("  -> columns: " ..colCount..", rows: " ..row_count_total..", max returned rows: " ..maxReturnRows)

	local conn = dconn.currentConnection()
	local fieldTypeArray = {} -- util.newTable(colCount, 0)
	for i = 1, colCount do
		if fieldNameArray then
			colTagName[i] = fieldNameArray[i]
		else
			colTagName[i] = colName[i]
		end
		if dschema.isField(fieldNameArray[i]) then
			fieldTypeArray[i] = dschema.fieldTypeLua(fieldNameArray[i], conn.schema) -- ?? we must use "" schema to prevent infinite loop with schema loading
			-- todo: check if schema "" is ok here? should we use external field Lua type? TODO: add recType to call
		else
			fieldTypeArray[i] = ""
		end
	end

	local row = 0
	local rowValue = {}
	rowValue = cur:fetch(rowValue, "n") -- ("n") -- (ret, "n")
	if returnTableType == "record table" then
		-- init colTagName and useRecData -arrays
		local useRecData = {}
		for i = 1, colCount do
			if option and option.table_prefix and startsWith(colTagName[i], option.table_prefix) then
				colTagName[i] = parseAfter(colTagName[i], ".")
			end
			if found(colTagName[i], ".") then
				useRecData[i] = true
			else
				useRecData[i] = false
			end
		end
		-- loop return data
		while rowValue and row < maxReturnRows do
			row = row + 1
			if row % 1000 == 0 then
				if row == 1000 then
					io.write("\n* odbc load row: ")
				end
				io.write(row .. " ")
				io.flush()
			end
			ret[row] = {} -- util.newTable(0, colCount)
			for i = 1, colCount do
				if type(rowValue[i]) == "string" then
					if isWin or currentCharset then
						if convertCharset[currentCharset] then
							rowValue[i] = convertCharset[currentCharset](rowValue[i])
							-- rowValue[i] = icu.convert(rowValue[i], icuFrom, icuTo)
						end
					end
				end
				if useRecData[i] then
					recDataSet(ret[row], colTagName[i], rowValue[i])
				else
					ret[row][colTagName[i]] = rowValue[i]
				end
			end
			rowValue = cur:fetch(rowValue, "n")
		end
	else -- returnTableType == "array table"
		-- init return arrays
		for i = 1, colCount do
			ret[colTagName[i]] = {} -- util.newTable(returnRows, 0)
			-- util.newTable USES MEMORY VERY BADLY!
			-- in ODBC we don't know rowCountTotal, it gives -1
		end
		-- loop return data
		while rowValue and row < maxReturnRows do
			row = row + 1
			if row % 1000 == 0 then
				if row == 1000 then
					io.write("\n* odbc load row: ")
				end
				io.write(row .. " ")
				io.flush()
			end
			for i = 1, colCount do
				if type(rowValue[i]) == "string" then
					if isWin or currentCharset then
						if convertCharset[currentCharset] then
							rowValue[i] = convertCharset[currentCharset](rowValue[i])
							-- rowValue[i] = icu.convert(rowValue[i], icuFrom, icuTo)
						end
					end
				end
				ret[colTagName[i]][row] = rowValue[i]
			end
			rowValue = cur:fetch(rowValue, "n")
		end
	end
	cur:close()
	if row >= 1000 then
		io.write(row .. "\n")
		io.flush()
	end

	local info = {}
	info.column_name = colTagName
	info.row_count = row
	if rowCountTotal < row then
		rowCountTotal = row
	end
	info.row_count_total = rowCountTotal
	info.column_count = colCount
	return ret, info
end

local function selectionToRecordArray(cur, fieldNameArray, option)
	return selectionToArray(cur, fieldNameArray, option, "record table")
end

local function selectionToArrayTable(cur, fieldNameArray, option)
	return selectionToArray(cur, fieldNameArray, option, "array table")
end

local function selectionToRecordTable(cur, fieldNameArray, option)
	local sel, info = selectionToArray(cur, fieldNameArray, option, "array table")
	dconv = dconv or require "dconv"
	return dconv.arrayTableToRecordTable(sel, info, nil, option)
end

return {
	_name = "database-odbc",
	maxSaveRows = maxSaveRows,
	setReturnRowLimit = setReturnRowLimit,
	disconnect = disconnect,
	connect = connect,
	execute = execute,
	selectionToRecordArray = selectionToRecordArray,
	selectionToArrayTable = selectionToArrayTable,
	selectionToRecordTable = selectionToRecordTable
}
