--- lib/db/database-postgre-libpq.lua
-- Database module, postgre.
-- @module database
local util = require "util"
--  ffi = require "mffi"
-- local unicode = require "unicode/unicode4d"
local pg = require "pgproc"
local l = require"lang".l
local peg = require "peg"
local dschema = require "dschema"
local recDataSet = require"recdata".setDot
local dconn = require "dconn"
local dconv = require "dconv"
local import = require "import"
local ioWrite = util.ioWrite
local parseAfter, found, startsWith, endsWith = import.from(peg, "parseAfter, found, startsWith, endsWith")
local execute -- forward declaration
local coro = require "coro"
local useCoro = coro.useCoro()
local coroYield = coro.yieldFunction(1, true) -- yield every 1 calls, do resume without polling
local alwaysSyncCallTable = util.invertTable({"", "preference", "session", "person"})
local useAsync = true

local minMessageRows = 1000 -- -1 -- 1000 -- 10000
local skipCount = 25000
local skip = util.skipFunction(skipCount)

local auth
local function loadLibs()
	if not auth then
		auth = require "auth"
	end
end

local maxReturnRows = math.huge

local function maxSaveRows()
	return 50
end

local function quote(name)
	local driverConn = dconn.driverConnection()
	if not (type(driverConn) == "cdata" or type(driverConn) == "userdata") then
		return nil, l("driver connection is not valid cdata: '%s'", tostring(driverConn))
	end
	return pg.quote(driverConn, name)
end

local function connect(prf)
	if prf.connect_timeout == nil then
		if prf.connection_timeout then
			util.printRed("   connect_timeout does not exist, please fix preference connection.json connection_timeout to connect_timeout")
			prf.connect_timeout = prf.connection_timeout
		else
			util.printRed("   connect_timeout does not exist, please fix preference connection.json")
			prf.connect_timeout = 6
		end
	end
	local connectTimeout = math.ceil(prf.connect_timeout) -- Maximum time to wait while connecting, in seconds (write as a decimal integer, e.g., 10). Zero, negative, or not specified means wait indefinitely. The minimum allowed timeout is 2 seconds, therefore a value of 1 is interpreted as 2.
	local connString = "dbname='" .. prf.database .. "' host='" .. prf.host .. "' port='" .. (prf.port or 5432) .. "' user='" .. prf.database_user .. "' password='" .. prf.password .. "' connect_timeout='" .. connectTimeout .. "'"
	local pgConn, err = pg.connect(connString)
	local driverErr
	if err and found(err:lower(), "fatal") then
		driverErr = err
	end
	if not err then
		loadLibs()
		local _, err2 = execute("SET application_name='nc-server'; SET client_min_messages TO WARNING;", {driver_conn = pgConn})
		if err2 then
			util.printError("query SET application_name='nc-server'; failed, error '%s'", err2)
		end
	end
	return pgConn, err, driverErr
end

local function disconnect(driverConn)
	return pg.close(driverConn)
end

--[[ standard sql92?
function structureQuery()
	return "SELECT column_name, data_type, character_maximum_length FROM information_schema.columns WHERE table_name = "
end
]]

local function columnNames(pgResult)
	local cols = pg.fields(pgResult)
	local ret = {}
	for i = 1, cols do
		ret[i] = pg.field(pgResult, i)
	end
	return ret
end

local function setReturnRowLimit(rows)
	maxReturnRows = rows
end
--[[
local function columnTypes(cursor)
	local ret = cursor:getcoltypes()
	return ret
end
 ]]
execute = function(sqlExecute, option, conn)
	option = option or {}
	-- sqlExecute = sqlExecute:gsub("%.name,", ".name_,")
	-- print(sqlExecute)
	-- loadLibs()
	local pgConn = conn and conn.driver_conn or option.driver_conn or option.organization_id and dconn.driverConnection(nil, option.organization_id)
	if not (type(pgConn) == "cdata" or type(pgConn) == "userdata") then
		pgConn = conn or option.driver_conn or option.organization_id and dconn.driverConnection(nil, option.organization_id)
		return nil, l("driver connection is not valid cdata: '%s'", tostring(pgConn))
	end
	local rows, pgResult, pgStatus, err
	local isSelect = startsWith(sqlExecute, "SELECT ")
	local isUpdate = not isSelect and not startsWith(sqlExecute, "CREATE ") and not startsWith(sqlExecute, "DROP ")
	local socket
	if useCoro then
		socket = coro.currentSocket()
	end
	local async
	if isUpdate then
		-- test: set_config('application.user_id','xxx',true)
		-- local startText = "BEGIN;SELECT set_config('application_name','nc-server',true),set_config('application.user_id','"..auth.currentUserId().."',true);\n"
		local startText = "BEGIN;SET LOCAL application.user_id='" .. auth.currentUserId() .. "'; "
		rows, pgResult, pgStatus, err = pg.querySync(pgConn, startText .. sqlExecute .. ";COMMIT;")
	else
		local time
		local table = option and (option.table or option.query and option.query.table) or ""
		if useAsync and isSelect and socket and table then -- socket means useCoro
			if alwaysSyncCallTable[table] == nil then
				async = true
			end
		end
		local debug = minMessageRows == -1 or minMessageRows == 0 and table ~= "" and not (sqlExecute:find("session") or sqlExecute:find("preference") or sqlExecute:find("person") or sqlExecute:find("sequence"))
		if debug then
			ioWrite(l("\n    executing %s query in PostgreSQL table '%s', connection '%s', query: %s...", async and "async" or "", option and option.query and option.query.table or "", tostring(pgConn), sqlExecute:sub(1, 400)))
			time = util.seconds()
		end
		if isSelect and async then
			rows, pgResult, pgStatus, err = pg.query(pgConn, sqlExecute)
			-- rows, pgResult, pgStatus, err = pg.querySync(pgConn, sqlExecute)
		else
			rows, pgResult, pgStatus, err = pg.querySync(pgConn, sqlExecute)
		end
		if debug then
			ioWrite(l(" in %.4f seconds", util.seconds(time)))
		end
	end
	if err then
		if isUpdate then
			local ret2, _, _, err3 = pg.querySync(pgConn, "ROLLBACK;")
			util.printInfo("Postgres ROLLBACK, return '%s', error '%s'", tostring(ret2), tostring(err3))
		end
		return nil, util.printError("driver '%s', connection '%s', Postgres error: '%s'", tostring(pgConn), conn and conn.info or "unknown", tostring(err))
	end
	local conn2 = conn or dconn.getCurrentConnection()
	local cursor = {error = err, async = async, option = option, conn = conn2, driver_conn = pgConn, pgResult = pgResult, rows = rows, pgStatus = pgStatus, socket = socket}
	return cursor
end

local function clearResult(cursor, pgResult)
	local _
	repeat
		if pgResult then
			pg.reset(pgResult)
		end
		_, pgResult = pg.getResult(cursor.driver_conn)
	until pgResult == nil
end

local function socketCloseError()
	return nil, {error = "socket has been closed"}
end

local function selectionToArray(cursor, fieldNameArray, option, returnTableType)
	if not cursor then
		local err = l("cursor is nil")
		util.printError(err)
		-- pg.reset(pgResult) -- == cursor:close()
		return nil, {error = err}
	end
	local doYield = cursor.async or useCoro and cursor.socket
	if doYield then
		coroYield(cursor.socket)
		if cursor.socket.do_close then
			return socketCloseError()
		end
	end
	local err, pgResult
	local rows = cursor.rows
	if not cursor.async then
		pgResult = cursor.pgResult
	else
		rows, pgResult, err = pg.getResult(cursor.driver_conn)
		if pgResult == nil and err == nil then
			local loop = 0
			local maxAllowedLoop = 500
			while pgResult == nil and err == nil and loop < maxAllowedLoop do -- todo: set timeout
				loop = loop + 1
				if loop % 50 == 1 then
					util.print("    waiting PostgreSQL asynchronous result data, loop: %d", loop)
				end
				coroYield(cursor.socket)
				if cursor.socket.do_close then
					clearResult(cursor, pgResult)
					return socketCloseError()
				end
				rows, pgResult, err = pg.getResult(cursor.driver_conn)
			end
			if loop >= maxAllowedLoop then
				err = l("result data wait maximum loop count reached, loops: %d", loop)
			end
		end
	end
	if err then
		-- err = util.printError("PostgreSQL asynchronous result error '%s'", err) -- does print cause another pg connection?
		err = l("PostgreSQL asynchronous result error '%s'", err)
		clearResult(cursor, pgResult)
		return nil, {error = err}
	end
	if rows < 1 and returnTableType ~= "array table" then
		local info = {column_name = {}, row_count = 0, row_count_total = 0, column_count = 0}
		clearResult(cursor, pgResult)
		return {}, info
	end
	loadLibs()
	option = option or cursor.option
	local conn = cursor.conn
	local colName = columnNames(pgResult)
	local colCount = #colName
	if colCount < 1 then
		err = l("column count < 1")
		util.printError(err)
		clearResult(cursor, pgResult)
		return nil, {error = err}
	end
	local returnRows
	if rows < maxReturnRows then
		returnRows = rows
	else
		returnRows = maxReturnRows
	end
	local ret = {} -- util.newTable(0, colCount)
	local colTagName = {} -- util.newTable(colCount, 0)
	-- io.write("  -> columns: " ..colCount..", rows: " ..rows..", return rows: " ..returnRows)
	-- io.flush()
	local fieldTypeArray = {} -- util.newTable(colCount, 0)
	local fieldTypeArraySql = {} -- util.newTable(colCount, 0)
	for i = 1, colCount do
		if fieldNameArray then
			colTagName[i] = fieldNameArray[i]
		else
			colTagName[i] = colName[i]
		end
		if dschema.isField(fieldNameArray[i]) then
			fieldTypeArray[i], fieldTypeArraySql[i] = dschema.fieldTypeLua(fieldNameArray[i], conn.schema) -- ?? we must use "" schema to prevent infinite loop with schema loading
			-- todo: check if schema "" is ok here? should we use external field Lua type? TODO: add recType to call
		else
			fieldTypeArray[i], fieldTypeArraySql[i] = "", ""
		end
	end

	local rowsCounted = rows
	for i = 1, colCount do
		if option and option.table_prefix and startsWith(colTagName[i], option.table_prefix) then
			colTagName[i] = parseAfter(colTagName[i], ".")
		end
	end
	local rows2
	if returnTableType == "record array" then
		local val
		local useRecData = {}
		for i = 1, colCount do
			if found(colTagName[i], ".") then
				useRecData[i] = true
			else
				useRecData[i] = false
			end
		end
		-- ret = util.newTable(rows, 0) -- is this ok
		local time
		if rows >= minMessageRows then
			ioWrite(l("    loading %d %s of data%s from PostgreSQL table '%s': 0%% ", rows, rows == 1 and "row" or "rows", cursor.async and " asynchronously" or "", option and (option.table or option.table_prefix) or "")) -- must NOT use l() beause it will create another pg connection here
			-- ioWrite("    loading " .. rows .. " rows of data from PostgreSQL table '" .. table .. "': 0%") -- must NOT use l() beause it will create another pg connection here
			time = util.seconds()
		end
		local start = 1
		repeat
			for row = start, rows do
				if rows >= minMessageRows and skip(row) == 0 then
					if doYield then
						coroYield(cursor.socket)
						if cursor.socket.do_close then
							clearResult(cursor, pgResult)
							return socketCloseError()
						end
					end
					ioWrite(l("%.1f%% ", row / rows * 100))
				end
				ret[row] = {} -- util.newTable(0, colCount)
				for i = 1, colCount do
					val = pg.fetch(pgResult, row, i)
					if fieldTypeArray[i] == "number" then
						if val == "null" then
							val = 0
						else
							val = tonumber(val)
						end
					elseif fieldTypeArray[i] == "boolean" then
						if val == "t" then
							val = true -- pg.fetch(pgResult, row, i) return "t" or "f"
						else
							val = false -- "null" is also false
						end
					elseif fieldTypeArraySql[i] == "timestamp" then
						if endsWith(val, ".000000") then -- remove seconds part
							val = val:sub(1, -8)
						end
					end
					if useRecData[i] then
						recDataSet(ret[row], colTagName[i], val)
					else
						ret[row][colTagName[i]] = val
					end
				end
				if row >= returnRows and row < rows then
					rowsCounted = returnRows
					break
				end
			end
			if cursor.async == nil then
				rows2 = nil
			else
				start = rows + 1
				coroYield(cursor.socket)
				if cursor.socket.do_close then
					clearResult(cursor, pgResult)
					return socketCloseError()
				end
				pg.reset(pgResult)
				rows2, pgResult, err = pg.getResult(cursor.driver_conn)
				if rows2 then
					rows = start + rows2
				end
			end
		until rows2 == nil or err

		if rows >= minMessageRows then
			ioWrite(l("100%% in %.6f seconds\n", util.seconds(time)))
		end
	else -- returnTableType == "array table"
		for i = 1, colCount do
			ret[colTagName[i]] = {} -- util.newTable(returnRows, 0) -- newTable USES MEMORY VERY BADLY - if all rows do not come from db?
		end
		local start = 1
		repeat
			for row = start, rows do
				for i = 1, colCount do
					ret[colTagName[i]][row] = pg.fetch(pgResult, row, i)
				end
				if row >= returnRows and row < rows then
					rowsCounted = returnRows
					break
				end
			end
			start = rows + 1
			if cursor.async then
				coroYield(cursor.socket)
				if cursor.socket.do_close then
					clearResult(cursor, pgResult)
					return socketCloseError()
				end
			end
			pg.reset(pgResult)
			rows2, pgResult, err = pg.getResult(cursor.driver_conn)
			if rows2 then
				rows = start + rows2
			end
		until rows2 == nil or err

		local function convertRowType(i, typeLua)
			if typeLua == "number" then
				for j = 1, rows do
					if ret[colTagName[i]][j] == "null" then
						ret[colTagName[i]][j] = 0
					else
						ret[colTagName[i]][j] = tonumber(ret[colTagName[i]][j])
					end
				end
			elseif typeLua == "boolean" then
				for j = 1, rows do
					if ret[colTagName[i]][j] == "t" then -- pg.fetch(pgResult, row, i) return "t" or "f"
						ret[colTagName[i]][j] = true
					else
						ret[colTagName[i]][j] = false -- null == false
					end
				end
			end
		end

		if rows > 0 then
			loadLibs()
			for i = 1, colCount do
				if fieldTypeArray[i] ~= "" then
					convertRowType(i, fieldTypeArray[i])
				end
			end
		end
	end

	clearResult(cursor, pgResult)
	local info = {column_name = colTagName, row_count = rowsCounted, row_count_total = rows, column_count = colCount, error = err}
	return ret, info
end

local function selectionToRecordArray(cursor, fieldNameArray, option)
	return selectionToArray(cursor, fieldNameArray, option, "record array")
end

local function selectionToArrayTable(cursor, fieldNameArray, option)
	return selectionToArray(cursor, fieldNameArray, option, "array table")
end

local function selectionToRecordTable(cursor, fieldNameArray, option)
	local sel, info = selectionToArrayTable(cursor, fieldNameArray, option)
	return dconv.arrayTableToRecordTable(sel, info, nil, option) -- this changes deep tag names to subtags
	-- do
	-- test performance of this, returns object array:
	-- SELECT row_to_json(row) AS data FROM (SELECT * FROM  pg_stat_activity) AS row
	-- https://hashrocket.com/blog/posts/faster-json-generation-with-postgresql
	-- end
end

return {
	_name = "postgre-libpq",
	maxSaveRows = maxSaveRows,
	-- dbType = dbType,
	setReturnRowLimit = setReturnRowLimit,
	quote = quote,
	disconnect = disconnect,
	connect = connect,
	execute = execute,
	selectionToArrayTable = selectionToArrayTable,
	selectionToRecordArray = selectionToRecordArray,
	selectionToRecordTable = selectionToRecordTable
}

--[[
Name	Aliases	Description
bigint	int8	signed eight-byte integer
bigserial	serial8	autoincrementing eight-byte integer
bit [ (n) ]	 	fixed-length bit string
bit varying [ (n) ]	varbit	variable-length bit string
boolean	bool	logical Boolean (true/false)
box	 	rectangular box on a plane
bytea	 	binary data ("byte array")
character [ (n) ]	char [ (n) ]	fixed-length character string
character varying [ (n) ]	varchar [ (n) ]	variable-length character string
cidr	 	IPv4 or IPv6 network address
circle	 	circle on a plane
date	 	calendar date (year, month, day)
double precision	float8	double precision floating-point number (8 bytes)
inet	 	IPv4 or IPv6 host address
integer	int, int4	signed four-byte integer
interval [ fields ] [ (p) ]	 	time span
json	 	JSON data
line	 	infinite line on a plane
lseg	 	line segment on a plane
macaddr	 	MAC (Media Access Control) address
money	 	currency amount
numeric [ (p, s) ]	decimal [ (p, s) ]	exact numeric of selectable precision
path	 	geometric path on a plane
point	 	geometric point on a plane
polygon	 	closed geometric path on a plane
real	float4	single precision floating-point number (4 bytes)
smallint	int2	signed two-byte integer
smallserial	serial2	autoincrementing two-byte integer
serial	serial4	autoincrementing four-byte integer
text	 	variable-length character string
time [ (p) ] [ without time zone ]	 	time of day (no time zone)
time [ (p) ] with time zone	timetz	time of day, including time zone
timestamp [ (p) ] [ without time zone ]	 	date and time (no time zone)
timestamp [ (p) ] with time zone	timestamptz	date and time, including time zone
tsquery	 	text search query
tsvector	 	text search document
txid_snapshot	 	user-level transaction ID snapshot
uuid	 	universally unique identifier
xml	 	XML data
]]
-- Should be compatible with many database engines.
-- Currently supports only Postgre datatypes we use with Memori
