--- database-postgre-pgmoon.lua
-- Database module, postgre.
-- @module database
local util = require "util"
local peg = require "peg"
local pgmoon = require("db/pgmoon/pgmoon")
local l = require"lang".l
local dconn = require "dconn"
local import = require "import"
local parseAfter, startsWith = import.from(peg, "parseAfter, startsWith")
local auth -- forward declaration
local function loadLibs()
	if not auth then
		auth = require "auth"
	end
end

-- local maxReturnRows = math.huge
local function setReturnRowLimit() -- (rows)
	-- maxReturnRows = rows
end

local function maxSaveRows()
	return 50
end

local function quote(name)
	local driverConn = dconn.driverConnection()
	if not (type(driverConn) == "cdata" or type(driverConn) == "userdata") then
		return nil, l("driver connection is not valid cdata: '%s'", tostring(driverConn))
	end
	return driverConn:escape_literal(driverConn, name) -- TODO: escape_identifier()
end

local function connect(prf)
	local host = prf.host == "/tmp" and "127.0.0.1" or prf.host -- todo: handle pipes
	local port = prf.port == "" and 5432 or prf.port or 5432
	local param = {
		host = host, -- You can use a unix domain socket already by just providing something like "unix:/var/run/postgresql/.s.PGSQL.5432" as host in the config object
		-- host = "unix:/tmp/.s.PGSQL.5432", -- does not work in osx with luaSocket
		port = port,
		database = prf.database,
		user = prf.database_user,
		password = prf.password,
		-- connect_timeout = prf.connect_timeout, -- TOD, check this
		application_name = 'nc-server',
		convert_null = false,
		ssl = false -- ssl, ssl_verify, ssl_required
	}
	local pg = pgmoon.new(param)
	local _, err = pg:connect()
	if err then
		return nil, err, err
	end
	pg:settimeout(120 * 1000) -- 2 minute timeout
	return pg
end

local function disconnect(driverConn)
	local ret
	if driverConn then
		if driverConn.disconnect then
			ret = driverConn:disconnect()
		else
			if driverConn.close then
				ret = driverConn:close()
			end
			if driverConn.socket then
				ret = driverConn.socket:close()
			end
		end
		if driverConn.socket then
			driverConn.socket = nil
		end
	end
	util.clearTableKeys(driverConn)
	driverConn.closed = true
	return ret ~= 1 and ret or nil
end

--[[ standard sql92?
function structureQuery()
	return "SELECT column_name, data_type, character_maximum_length FROM information_schema.columns WHERE table_name = "
end
]]

local function execute(sqlExecute, option)
	option = option or {}
	-- sqlExecute = sqlExecute:gsub("%.name,", ".name_,")
	-- print(sqlExecute)
	-- loadLibs()
	local driverConn = option.conn or option.organization_id and dconn.driverConnection(nil, option.organization_id)
	if type(driverConn) ~= "table" or driverConn.sock_type ~= "native" and driverConn.sock_type ~= "luasocket" then
		return nil, l("driver connection is not valid table: '%s'", tostring(driverConn))
	end
	local rows = 0
	local ret, err, queryCount, notifications, notices
	local isSelect = startsWith(sqlExecute, "SELECT ")
	local isUpdate = not startsWith(sqlExecute, "SELECT ") and not startsWith(sqlExecute, "CREATE ") and not startsWith(sqlExecute, "DROP ")
	if isUpdate then
		-- test: set_config('application.user_id','xxx',true)
		-- local startText = "BEGIN;SELECT set_config('application_name','nc-server',true),set_config('application.user_id','"..auth.currentUserId().."',true);\n"
		loadLibs()
		local startText = "BEGIN;SET LOCAL application.user_id='" .. auth.currentUserId() .. "'; "
		ret, err, queryCount, notifications, notices = driverConn:query(startText .. sqlExecute .. ";COMMIT;")
	elseif isSelect == false then
		ret, err, queryCount, notifications, notices = driverConn:query(sqlExecute)
	end
	if type(ret) == "table" then
		for _, val in ipairs(ret) do
			if type(val) == "table" and val.affected_rows then
				rows = rows + val.affected_rows
			end
		end
	end
	if err then
		if isUpdate then
			local ret2, err3 = driverConn:query("ROLLBACK;")
			util.printInfo("Postgres ROLLBACK, return '%s', error '%s'", tostring(ret2), tostring(err3))
		end
		return nil, err
	end
	local cursor = {rows = rows, isSelect = isSelect, sqlExecute = sqlExecute, pq = driverConn, queryCount = queryCount, notifications = notifications, notices = notices}
	return cursor
end

local function selectionToArray(cursor, fieldNameArray, option)
	if not cursor then
		local err = l("cursor is nil")
		util.printError(err)
		-- cursor.pg:reset() -- == cursor:close()
		return nil, {error = err}
	end
	if not cursor.isSelect then
		local err = l("sql cursor type is not select")
		util.printError(err)
		-- cursor.pg:reset() -- == cursor:close()
		return nil, {error = err}
	end
	cursor.pq.option = option or {}
	if option and option.table_prefix then
		local columnNameArray = {}
		for i = 1, #fieldNameArray do
			if startsWith(fieldNameArray[i], option.table_prefix) then
				columnNameArray[i] = parseAfter(fieldNameArray[i], ".")
			else
				columnNameArray[i] = fieldNameArray[i]
			end
			-- columnNameArray[i] = parseAfter(columnNameArray[i], ".")
		end
		cursor.pq.option.fieldNameArray = columnNameArray
	else
		cursor.pq.option.fieldNameArray = fieldNameArray
	end
	local ret, err, queryCount, notifications, notices = cursor.pq:query(cursor.sqlExecute)
	if ret and err == nil then
		cursor.rows = #ret
	else
		cursor.rows = 0
	end
	-- TODO: convert field types if they are given

	--[=[
	loadLibs()
	local colName = columnNames(cursor)
	local colCount = #colName
	if colCount < 1 then
		local err = l("column count < 1")
		util.printError(err)
		-- cursor.pg:reset() -- == cursor:close()
		return nil, {error = err}
	end
	local rows = rowCount(cursor)
	local returnRows
	if rows < maxReturnRows then
		returnRows = rows
	else
		returnRows = maxReturnRows
	end
	local ret = {} -- util.newTable(0, colCount)
	local columnNameArray = {} -- util.newTable(colCount, 0)
	-- io.write("  -> columns: " ..colCount..", rows: " ..rows..", return rows: " ..returnRows)
	-- io.flush()
	local conn = dconn.currentConnection()
	local fieldTypeArray = {} -- util.newTable(colCount, 0)
	for i = 1, colCount do
		if fieldNameArray then
			columnNameArray[i] = fieldNameArray[i]
		else
			columnNameArray[i] = colName[i]
		end
		if dschema.isField(fieldNameArray[i]) then
			fieldTypeArray[i] = dschema.fieldType(fieldNameArray[i], conn.schema) -- ?? we must use "" schema to prevent infinite loop with schema loading
			-- todo: check if schema "" is ok here? should we use external field Lua type? TODO: add recType to call
		else
			fieldTypeArray[i] = ""
		end
	end

	local rowsCounted = rows
	for i = 1, colCount do
		if option and option.table_prefix and startsWith(columnNameArray[i], option.table_prefix) then
			columnNameArray[i] = parseAfter(columnNameArray[i], ".")
		end
	end
	if returnTableType == "record array" then
		local val
		local useRecData = {}
		for i = 1, colCount do
			if found(columnNameArray[i], ".") then
				useRecData[i] = true
			else
				useRecData[i] = false
			end
		end
		-- ret = util.newTable(rows, 0) -- is this ok
		for row = 1, rows do
			ret[row] = {} -- util.newTable(0, colCount)
			for i = 1, colCount do
				val = cursor.pg:fetch(row, i)
				if fieldTypeArray[i] == "number" then
					if val == "null" then
						val = 0
					else
						val = tonumber(val)
					end
				elseif fieldTypeArray[i] == "boolean" then
					if val == "t" then
						val = true -- cursor.pg:fetch(row, i) return "t" or "f"
					else
						val = false -- "null" is also false
					end
				elseif fieldTypeArray[i] == "timestamp" then
					if endsWith(val, ".000000") then
						val = val:sub(1, -8)
					end
				end
				if useRecData[i] then
					recDataSet(ret[row], columnNameArray[i], val)
				else
					ret[row][columnNameArray[i]] = val
				end
			end
			if row >= returnRows and row < rows then
				rowsCounted = returnRows
				break
			end
			-- coroYield(sock)
		end
	else -- returnTableType == "array table"
		for i = 1, colCount do
			ret[columnNameArray[i]] = {} -- util.newTable(returnRows, 0) -- newTable USES MEMORY VERY BADLY - if all rows do not come from db?
		end
		for row = 1, rows do
			for i = 1, colCount do
				ret[columnNameArray[i]][row] = cursor.pg:fetch(row, i)
			end
			-- coroYield(sock)
			if row >= returnRows and row < rows then
				rowsCounted = returnRows
				break
			end
		end

		local function convertRowType(i, typeLua)
			if typeLua == "number" then
				for j = 1, rows do
					if ret[columnNameArray[i]][j] == "null" then
						ret[columnNameArray[i]][j] = 0
					else
						ret[columnNameArray[i]][j] = tonumber(ret[columnNameArray[i]][j])
					end
				end
			elseif typeLua == "boolean" then
				for j = 1, rows do
					if ret[columnNameArray[i]][j] == "t" then -- cursor.pg:fetch(row, i) return "t" or "f"
						ret[columnNameArray[i]][j] = true
					else
						ret[columnNameArray[i]][j] = false -- null == false
					end
				end
			end
		end

		if rows > 0 then
			loadLibs()
			for i = 1, colCount do
				if fieldTypeArray[i] ~= "" then
					convertRowType(i, fieldTypeArray[i])
				end
			end
		end
	end ]=]

	-- cursor.pg:reset() -- == cursor:close()
	local info = {error = err, query_count = queryCount, notifications = notifications, notices = notices}
	info.column_name = fieldNameArray -- columnNameArray
	info.row_count = cursor.rows -- rowsCounted
	info.row_count_total = cursor.rows --  rows
	info.column_count = #fieldNameArray -- colCount
	return ret, info
end

local selectionToRecordArray = selectionToArray
--[[ local function selectionToRecordArray(cursor, fieldNameArray, option)
	return selectionToArray(cursor, fieldNameArray, option) -- , "record array")
end ]]

local function selectionToArrayTable(cursor, fieldNameArray, option)
	return selectionToArray(cursor, fieldNameArray, option) -- , "array table")
	-- todo: convert to array type if this is ever used
end

local function selectionToRecordTable(cursor, fieldNameArray, option)
	-- todo: never use this
	local sel, info = selectionToArrayTable(cursor, fieldNameArray, option)
	local dconv = require "dconv"
	return dconv.arrayTableToRecordTable(sel, info, nil, option) -- this changes deep tag names to subtags
	-- do
	-- test performance of this, returns object array:
	-- SELECT row_to_json(row) AS data FROM (SELECT * FROM  pg_stat_activity) AS row
	-- https://hashrocket.com/blog/posts/faster-json-generation-with-postgresql
	-- end
end

return {
	_name = "postgre-pgmoon",
	maxSaveRows = maxSaveRows,
	-- dbType = dbType,
	setReturnRowLimit = setReturnRowLimit,
	quote = quote,
	disconnect = disconnect,
	connect = connect,
	execute = execute,
	selectionToArrayTable = selectionToArrayTable,
	selectionToRecordArray = selectionToRecordArray,
	selectionToRecordTable = selectionToRecordTable
}

--[[
local function columnTypes(cursor)
	local ret = cursor:getcoltypes()
	return ret
end
]]

--[[
local function rowCount(cursor)
	local ret = cursor.rows
	return ret
end

local function columnCount(cursor)
	return cursor.pg:fields()
end

local function columnNames(cursor)
	local cols = columnCount(cursor)
	local ret = {}
	for i = 1, cols do
		ret[i] = cursor.pg:field(i)
	end
	return ret
end
]]
