--- database-postgre.lua
-- Database module, postgre.
-- @module database
local util = require "util"
local l = require"lang".l
local dschema = require "dschema"
local dconn = require "dconn"
local import = require "import"
local pq
local _name, maxSaveRows, setReturnRowLimit, quote, connect, disconnect, execute, selectionToArrayTable, selectionToRecordArray, selectionToRecordTable

local driverType
local function getDriver()
	return driverType
end

local function setDriver(driver)
	if driver == "libpq" then
		local libpq = require "database-postgre-libpq"
		pq = libpq
		driverType = "libpq"
	else
		local pgmoon = require "database-postgre-pgmoon"
		pq = pgmoon
		driverType = "pgmoon"
	end
	_name, maxSaveRows, setReturnRowLimit, quote, connect, disconnect, execute, selectionToArrayTable, selectionToRecordArray, selectionToRecordTable = import.from(pq, "_name, maxSaveRows, setReturnRowLimit, quote, connect, disconnect, execute, selectionToArrayTable, selectionToRecordArray, selectionToRecordTable")
	--[[ if not util.from4d() then
		util.print("  postgre database driver: '%s'", _name)
	end ]]
end

if pq == nil then

	local function setDriverType(useCoro)
		if useCoro then
			-- setDriver("pgmoon")
			setDriver("libpq")
		else
			-- setDriver("pgmoon")
			setDriver("libpq")
		end
	end

	local coro = require "coro"
	local useCoro = coro.useCoro()
	setDriverType(useCoro)
end

local function roleExists(role, database, conn)
	local cursor = execute(string.format("SELECT COUNT(*) FROM pg_catalog.pg_roles WHERE rolname = '%s';", tostring(role)), {database = database}, conn)
	if cursor == nil then
		return nil, util.printError("function role exists execute failed, cursor is empty")
	end
	local ret, info = selectionToRecordArray(cursor, {"COUNT(*)"}, {database = database, table = "pg_catalog.pg_roles"})
	if not ret then
		return nil, info
	end
	ret = tonumber(ret[1]["COUNT(*)"])
	return ret == 1
end

local tableCountCache
local function tableCount(database)
	if tableCountCache then
		return tableCountCache
	end
	local cursor = execute("SELECT COUNT(*) FROM pg_stat_user_tables", {database = database})
	local ret, info = selectionToRecordArray(cursor, {"COUNT(*)"}, {database = database, table = "pg_stat_user_tables"})
	if not ret then
		return nil, info
	end
	tableCountCache = tonumber(ret[1]["COUNT(*)"])
	return tableCountCache
end

local function fieldCount(tableName)
	if type(tableName) ~= "string" then
		local err = l("table name '%s' is invalid", tostring(tableName))
		util.printError(err)
		return nil, err
	end
	local tableNameEscaped = quote(tableName)
	local database = dconn.database()
	local cursor = execute("SELECT COUNT(*) FROM information_schema.columns WHERE table_name='" .. tableNameEscaped .. "'", {database = database})
	local ret, info = selectionToRecordArray(cursor, {"COUNT(*)"}, {database = database, table = "information_schema.columns"})
	if not ret then
		return nil, info
	end
	return tonumber(ret[1]["COUNT(*)"])
end

local function tableNameArray(organizationId)
	local cursor = execute("SELECT relname, schemaname FROM pg_stat_user_tables", {organization_id = organizationId})
	local ret, info = selectionToRecordArray(cursor, {"table_name", "schema"}, {organization_id = organizationId, table = "pg_stat_user_tables"})
	return ret, info
end

local function fieldNameArray(tableName, organizationId)
	local cursor = execute("SELECT column_name, data_type, character_maximum_length FROM information_schema.columns WHERE table_name = '" .. tableName .. "'", {organization_id = organizationId})
	local ret, info = selectionToRecordArray(cursor, {"field_name", "field_type", "field_length"}, {organization_id = organizationId, table = "information_schema.columns"})
	for _, item in ipairs(ret) do
		item.field_type = dschema.sqlTypeToFieldType(item.field_type)
		item.field_length = item.field_length ~= "" and tonumber(item.field_length) or nil
	end
	return ret, info
end

return {
	_name = _name,
	getDriver = getDriver,
	setDriver = setDriver,
	maxSaveRows = maxSaveRows,
	-- dbType = dbType,
	setReturnRowLimit = setReturnRowLimit,
	quote = quote,
	disconnect = disconnect,
	connect = connect,
	execute = execute,
	selectionToArrayTable = selectionToArrayTable,
	selectionToRecordArray = selectionToRecordArray,
	selectionToRecordTable = selectionToRecordTable,
	roleExists = roleExists,
	tableCount = tableCount,
	fieldCount = fieldCount,
	tableNameArray = tableNameArray,
	fieldNameArray = fieldNameArray
}
