--- database-rest4d.lua
-- tcp rest connection to 4d database
local rest4d = {_name = "database-rest4d"}

local util = require "util"
local socket = require "system/socket"
local json = require "json"
local dt = require "dt"
local l = require"lang".l
local peg = require "peg"
local dschema = require "dschema"
local color = require "ansicolors"
local dconv = require "dconv"
local fn = require "fn"
local dsql = require "dsql"
local coro = require "coro"
local sort = require "table/sort"
local recDataSet = require"recdata".setDot
local print = util.print
local ioWrite = util.ioWrite
local dconn, auth, net -- delay load
local useCompress, useCompressDebug, pattCompress, lz4, readContent -- forward function declaration
local queryMaxRowsLimit = 20 * 10 ^ 6 -- 20 million

local waitForLock = coro.waitForLock
local minMessageRows = 0
local minSortMessageRows = 10000
local formatNum = util.formatNumFunction(0)

local httpStart = [[
POST /rest/nc/query/sql4d HTTP/1.1
Content-Type: application/json
Connection: keep-alive
User-Agent: nc-rest-sql
Authorization: Basic cHJvdGFjb246cmVzdFdoYzEuMA==
]]

local function loadLibs()
	if dconn == nil then
		dconn = require "dconn"
		net = require "system/net"
		auth = require "auth"
		local prf = util.prf("system/option.json").option
		useCompress = prf.compress_rest4d or ""
		useCompressDebug = prf.compress_debug or false
		if useCompress == "lz4" then
			-- print("db/database-rest4d, use compress: "..useCompress)
			lz4 = require "lz4"
			pattCompress = peg.toPattern(useCompress)
			httpStart = httpStart .. "Content-Encoding: " .. useCompress .. "\n"
			httpStart = httpStart .. "Accept-Encoding: " .. useCompress .. "\n"
		end
		httpStart = httpStart:gsub("\n", "\r\n") .. "Content-Length: "
		--[[
		Accept-Language: fi-fi
		Accept-Encoding: lz4, deflate
		]]
		if jit == nil or jit.arch:sub(1, 3) ~= "arm" then
			local compatibilityPref = util.readUpperLevelPreferenceFile("system/4d.json", "no-error").option
			if compatibilityPref and compatibilityPref.use_dconn_4d_version == true then
				util.printInfo("using system/4d.json use_dconn_4d_version, rest calls to: /rest/ma/query/sql")
				httpStart = peg.replace(httpStart, "/rest/nc/query/sql4d", "/rest/ma/query/sql")
			end
		end
	end
end

-- local maxReturnRows = math.huge
--[[
local prfDefault = {}
prfDefault.host = "127.0.0.1"
prfDefault.port = 5949
prfDefault.database = "Manageri v12 local"
prfDefault.database_user = "sql_user1"
prfDefault.pass = "sql_Usrlx1"
prfDefault.connect_timeout = 1
prfDefault.timeout = 60
]]

function rest4d.maxSaveRows()
	return 1
end

function rest4d.dbType()
	return "4d"
end

-- local printSql = false
function rest4d.showSql() -- (val)
	--[[ 	if val == nil then
		printSql = true
	else
		printSql = val
	end ]]
end

local function printErr(conn, txt, details)
	-- local errno = sql.fourd_errno(conn)
	-- local state = ffi.string(sql.fourd_sqlstate(conn))
	-- local errtxt = ffi.string(sql.fourd_error(conn))
	local err = "connection: " .. tostring(conn) .. ", " .. txt -- = string.format(txt..", %s (%d): '%s'", state, errno, errtxt)
	if details then
		err = err .. "\n" .. details:sub(1, 400)
	end
	print(color("%{bright magenta}" .. "\n *** 4D sql error: " .. err))
	return err
	-- disconnect(conn)
end

local function setTimeout(conn, timeout)
	conn.timeout = timeout
end
--[[ local function timeout(conn)
	return conn.timeout -- or 30
end ]]

function rest4d.setReturnRowLimit() -- (rows)
	-- maxReturnRows = rows
end

function rest4d.structureQuery()
	return "SELECT column_name, data_type, character_maximum_length FROM information_schema.columns WHERE table_name = "
	-- select TABLE_NAME, COLUMN_NAME, DATA_TYPE, DATA_LENGTH from _USER_COLUMNS
end

local function disconnect(conn)
	--[[local sqlrRet = -- sql.fourd_close_statement(cursor)
	if sqlrRet ~= 0 then
		print(l"Error in sql close statement:" ..sqlrRet)
	end
	local ret =  = sql.fourd_close(conn)
	if ret ~= 0 then
		print(l"Error in sql library close: "..ret)
	end
	]]
	-- sql.fourd_free(conn)
	if not conn.closed then
		conn:close()
	end
end
rest4d.disconnect = disconnect

local dataRet, dataErr, dataLen
local function readData(sock, readLen)
	dataRet, dataErr, dataLen = sock:receive(readLen) -- sock:receive may call yield()
	if dataErr then
		if type(dataErr) == "string" then
			util.print("call 4D socket receive error: '%s'", dataErr)
		else
			util.print("call 4D socket receive error: '%s'", net.errorText(dataErr))
		end
	end
	return dataRet, dataErr, dataLen
end

local function writeData(sock, data)
	dataRet, dataErr = sock:send(data) -- sock:send may call yield()
	if dataErr then
		if type(dataErr) == "string" then
			util.print("call 4D socket send error: '%s'", dataErr)
		else
			util.print("call 4D socket send error: '%s'", net.errorText(dataErr))
		end
	end
	return dataRet, dataErr
end

local function readHeader(cursor)
	local headerRead = false
	-- local sleepCount = 0
	-- local startTime
	local ret = ""
	local recv, pos, recvLen
	local s, e
	local err
	local conn = cursor.conn
	local sock = dconn.driverConnection(conn)
	local loopCount = 0
	local exit
	repeat
		loopCount = loopCount + 1
		recv, err, recvLen = readData(sock, "\r\n\r\n")
		if err == nil then
			if recvLen and recvLen ~= #recv then
				print("readHeader recvLen ~= #recv")
			end
			ret = ret .. recv
			s, e = ret:find("Content-Length: ", 1, true)
			if s and e and e > 0 then
				s = e
				e = ret:find("\r\n", s + 1, true)
				if e then
					cursor.contentLength = tonumber(ret:sub(s, e))
					pos = ret:find("\r\n\r\n", e, true)
					if pos then
						cursor.header = ret:sub(1, pos - 1)
						cursor.data = ret:sub(pos + 4)
						if useCompress and peg.found(cursor.header, pattCompress) then
							cursor.compress = true
						else
							cursor.compress = false
						end
						headerRead = true
					end
				end
			end
		end
		exit = headerRead or err or loopCount > 1 -- and err < -1
		if not exit then
			util.print("looping readHeader, loop: " .. loopCount)
		end
	until exit
	if err then -- and (type(err) ~= "number" or err < 1) then
		return err
	end
	return -- no error
end

local function uncompress(data)
	local time = util.microSeconds()
	local ret, err = lz4.decompress(data)
	local size = ret and #ret or 0
	if useCompressDebug and size > 0 then
		util.print("lz4.decompress size %d / %d = %d%%, time %d µs", #data, size, math.floor(#data / size * 100), util.microSeconds(time))
	end
	if err then
		err = l("lz4.decompress error: '%s', data size: %d", err, #data)
		util.printRed("call 4D error: '%s'", tostring(err))
	end
	return ret or data, err
end

local function connect(prf, option)
	if prf == nil then
		return nil, util.printWarning("   error in sql call 4D tcp connect, preference is nil")
	end
	-- if conn then
	--   disconnect(conn)
	-- end
	--[[
	prf.host = rec.host or prfDefault.host
	prf.database_user = rec.database_user or prfDefault.database_user
	prf.pass = rec.pass or prfDefault.pass
	prf.database = rec.database or prfDefault.database
	prf.port = rec.port or prfDefault.port
	prf.connect_timeout = rec.connect_timeout or prfDefault.connect_timeout or 2
	prf.timeout = rec.timeout or prfDefault.timeout or 60
	]]
	-- dbInfo = prf.database..", "..prf.host..":"..prf.port
	-- print("  "..dbInfo.."... ")
	loadLibs()
	local sock = socket.connect(prf.host, prf.port, "tcp", prf.connect_timeout, nil, "no-error")
	if not sock then
		-- err = printErr(sock, l("   error in sql call 4D tcp connect, preference: ")..prf.preference)
		return nil, l("call 4D tcp connect failed")
	end
	-- util.printOk("   Sql call 4D tcp connected to address: "..socket.socket_address(sock))
	setTimeout(sock, prf.timeout)
	sock.schema = "4d"
	sock.option = {preference = prf, option = option}
	return sock
end
rest4d.connect = connect

local function reconnect(conn)
	local prf = conn.option.preference
	local option = conn.option.option
	local sock, newErr = connect(prf, option)
	if sock then
		dconn.setDriverConnection(conn, prf.organization_id, sock)
	end
	return sock, newErr
end

local function checkConnection(conn)
	local err
	local sock = dconn.driverConnection(conn)
	if sock == nil or sock.socket == nil or sock.do_close or sock.closed then
		sock, err = reconnect(conn)
	end
	return sock, err
end

local function callRestTcp(conn, sock, cursor, param)
	local err, call
	if useCompress == "lz4" then
		local bodyLen = #param
		local paramCompressed
		paramCompressed, err = lz4.compress(param)
		call = httpStart .. tostring(#paramCompressed) .. "\r\nContent-Uncompressed-Length: " .. bodyLen .. "\r\n\r\n" .. paramCompressed
	else
		call = httpStart .. tostring(#param) .. "\r\n\r\n" .. param -- add content-length + http
	end
	if err then
		util.printError(err)
	else
		local bytesSent
		bytesSent, err = writeData(sock, call)
		--  = sql.fourd_prepare_statement(conn, sqlExecute) -- "SELECT * from _USER_TABLES"
		if bytesSent ~= #call and type(err) == "number" then
			if socket.econnreset(err) then
				conn, err = reconnect(conn)
				if err then
					dconn.disconnect(conn)
					conn, err = checkConnection(conn) -- create new connection
					if err then
						return err
					end
				end
				if conn then
					local err2
					sock = dconn.driverConnection(conn)
					bytesSent, err2 = writeData(sock, call)
					if bytesSent == #call then
						util.printInfo("error in socket send: %d, tcp error: %s, %s, reconnect succeeded", bytesSent, tostring(err), net.errorText(err))
					else
						err = err2
					end
				end
			end
			if bytesSent ~= #call then
				err = l("error in socket send: %d, tcp error: %s, %s", bytesSent, tostring(err), net.errorText(err))
				err = printErr(conn, err) -- , cursor.callParam)
				return err
			end
		end
		if bytesSent > 0 then
			err = readHeader(cursor)
			if not err then
				err = readContent(cursor)
			end
		end
	end
	return err
end

local function callRest(conn, cursor, param)
	if type(param) ~= "string" then
		return nil, l("call 4D tcp parameter type '%s' is not 'string'", type(param))
	elseif not peg.found(param, "SELECT ") and not peg.found(param, "INSERT ") and not peg.found(param, "UPDATE ") then
		-- not peg.found(param, '"local_field"')
		if not peg.found(param, '"sql":"CALL_4D_FUNCTION"') then
			return nil, l("call 4D tcp parameter does not contain valid sql (SELECT/INSERT/UPDATE), parameter: '%s'", type(param))
		end
	end
	--[[ local ret, err
	if conn == nil then
		conn = {}
		conn, err = checkConnection(conn)
	end
	if err then
		return nil, err
	end
	local sock = dconn.driverConnection(conn) ]]
	local sock, err = checkConnection(conn)
	if err then
		return nil, l("call 4D tcp socket is not connected, error: '%s'", tostring(err))
	end
	--[[ if not sock or sock.socket == nil or sock.do_close or sock.closed then
		sock, err = checkConnection(conn)
		if err then
			return nil, l("call 4D tcp socket is not connected, error: '%s'", tostring(err))
		end
	end ]]
	err = callRestTcp(conn, sock, cursor, param)
	if sock.do_close then
		-- disconnect(sock)
		sock.do_close = nil -- one socket for all different thread 4D connections, do not close it
		sock:clearSocketData()
	end
	return err -- real return is in cursor.data
end

local function execute(sqlExecute, option) -- option 2 database is not used in 4D
	loadLibs()
	local prevOrg = dconn.organizationId()
	local currentOrg = prevOrg
	if not peg.found(prevOrg, "-4d") then
		prevOrg = dconn.setCurrentOrganization("4d") -- dconn.currentOrganizationId()
		currentOrg = auth.currentOrganizationId()
	end
	local conn = dconn.connection({organizationId = currentOrg}) -- create connection if needed
	local sock = dconn.driverConnection(conn)
	local err
	if sock == nil or sock.socket == nil then
		sock, err = checkConnection(conn)
		if err then
			return nil, err
		end
	end
	if sock == nil or conn == nil then
		err = l("connection is nil")
		util.printRed("call 4D error: '%s'", tostring(err))
		dconn.setCurrentOrganization(currentOrg)
		return nil, err
	elseif conn.dbtype ~= "4d" then
		err = l("connection is not 4D, connection: %s", conn)
		util.printRed("call 4D error: '%s'", tostring(err))
		dconn.setCurrentOrganization(currentOrg)
		return nil, err
	end
	if not sock.socket then
		err = l("driver connection socket is nil")
		util.printRed("call 4D error: '%s'", tostring(err))
		dconn.setCurrentOrganization(currentOrg)
		return nil, err
	end
	local cursor = {result = {}}
	cursor.conn = conn
	cursor.option = option
	cursor.callParam = sqlExecute
	waitForLock(sock, "4drest")
	if sqlExecute:find("SELECT ", 1, true) ~= 1 then -- SELECT goes to selectionToArrayTable()
		local paramCall = util.clone(option or {})
		paramCall.sql = sqlExecute
		paramCall.auth = dconn.currentAuthTable()
		paramCall.query_name = "new:" .. (option.query_name or tostring(dsql.lastQueryName())) -- call always _lx_ExecuteSqlNew
		local paramCallJson = json.toJsonRaw(paramCall)
		err = callRest(conn, cursor, paramCallJson)
		if err then
			util.printRed("call 4D error: '%s'", tostring(err))
		end
		local data = cursor.data
		if data then
			if cursor.compress and data ~= "" then
				local dataUncompressed
				dataUncompressed, err = uncompress(data:sub(1, cursor.contentLength))
				if err == nil then
					data = dataUncompressed
				else
					err = err .. l("\n - execute content length: %d", cursor.contentLength)
				end
			end
			if data:sub(1, 1) == "{" then -- or data:sub(1, 1) == "[" then -- json text
				data, err = json.fromJson(data)
			elseif not data.error then
				data = {error = data}
			end
		end
		if data == nil then
			local txt = l("could not retrieve data")
			if err then
				err = err .. "\n" .. txt
			else
				err = txt
			end
		elseif data.error or data.info then
			local txt
			dataErr = data.error or data.info
			if type(dataErr) == "table" then
				txt = table.concat(dataErr, "\n")
			else
				txt = tostring(dataErr)
			end
			-- print(data.error)
			if txt ~= "" then
				if err then
					err = err .. "\n" .. txt
				else
					err = txt
				end
			end
		end
		if prevOrg ~= currentOrg then
			dconn.setCurrentOrganization(currentOrg)
		end
		return 0, err, data -- 0 == no cursor
		-- print(cursor.data)
	end
	--[[
	local cursor --  = sql.fourd_exec_statement(state)
	if cursor == nil then
		local  err = l"Error in sql execute statement"
		err = printErr(conn, err, sqlExecute)
		return nil,err
	end
	]]
	dconn.setCurrentOrganization(currentOrg) -- for some reason we need to restore always even if prevOrg == currentOrg
	return cursor
end
rest4d.execute = execute

function rest4d.callRestFunction(param)
	local cursor, err, data
	loadLibs() -- need to call loadLibs() to init httpStart for first time
	local httpStart2 = httpStart -- save for restore
	httpStart = peg.replace(httpStart, "/rest/nc/query/sql4d", "/rest/ma/call_4d_function")
	httpStart = peg.replace(httpStart, "/rest/ma/query/sql4d", "/rest/ma/call_4d_function")
	httpStart = peg.replace(httpStart, "/rest/ma/query/sql", "/rest/ma/call_4d_function")
	local currentOrg = dconn.setCurrentOrganization("4d")
	cursor, err, data = execute("CALL_4D_FUNCTION", param)
	dconn.setCurrentOrganization(currentOrg)
	if cursor ~= 0 then -- cursor will be 0 == no cursor
		util.printRed("CALL_4D_FUNCTION error, cursor is not 0: " .. tostring(cursor))
	end
	httpStart = httpStart2 -- must restore normal httpStart
	if err then -- cursor ~= 0 then
		err = "CALL_4D_FUNCTION error: " .. tostring(err)
		if not param.parameter or param.parameter.print_error ~= false then
			util.printRed("call 4D error: '%s'", tostring(err))
		end
		return nil, err
	end
	if data.columns then -- new system uses singular name 'column', not 'columns'
		data.column = data.columns
		data.columns = nil
	end
	return data, err
end

local function cursorRowCount(cursor)
	local info = cursor.result.info
	if not info then
		return 0
	end
	return info.row_count or info.rowCount -- compatibility for old and new - tonumber(sql.fourd_num_rows(cursor)) -- int
end

local function cursorColumnCount(cursor)
	local info = cursor.result.info
	if not info then
		return 0
	end
	return info.column_count or info.columnCount -- sql.fourd_num_columns(cursor) -- FOURD_LONG8
end

local function columnName(cursor, i)
	local info = cursor.result.info
	if not info then
		return 0
	end
	return info.column_name and info.column_name[i] or info.columnName[i] -- string.lower(ffi.string(sql.fourd_get_column_name(cursor,i-1)))
end

local function columnType(cursor, i)
	local column = columnName(cursor, i)
	if peg.startsWith(column, "COUNT(") then
		return "integer"
	end
	local retStr = dschema.fieldType(column)
	return retStr
end
-- rest4d.columnType = columnType

function rest4d.tableCount()
	return 59 -- 4d table count
end

function rest4d.fieldCount() -- (tableNum)
	return 0
end

readContent = function(cursor)
	local err = 0
	if not cursor.data then
		cursor.data = ""
	elseif not cursor.contentLength then
		cursor.data = ""
	elseif #cursor.data < cursor.contentLength then
		local recv, recvLen
		local readLen = cursor.contentLength - #cursor.data
		-- local sleepCount = 0
		local loopCount = 0
		local conn = cursor.conn
		local sock = dconn.driverConnection(conn)
		local exit
		repeat
			loopCount = loopCount + 1
			recv, err, recvLen = readData(sock, readLen)
			if err == nil then
				if recvLen then
					readLen = readLen - recvLen
					cursor.data = cursor.data .. recv
				elseif recv and #recv > 0 then
					readLen = readLen - #recv
					cursor.data = cursor.data .. recv
				else
					print("recvLen ~= #recv")
				end
			end
			exit = #cursor.data >= cursor.contentLength or err or loopCount >= 6
			if not exit and loopCount > 1 then
				util.print("      call 4D read content loop: " .. loopCount)
			end
		until exit -- and err < -1
	elseif #cursor.data == cursor.contentLength then
		return nil
	elseif #cursor.data > cursor.contentLength then
		cursor.nextdata = cursor.data:sub(cursor.contentLength + 1) -- TODO: use saved nextdata, rename to nextData
		cursor.data = cursor.data:sub(1, cursor.contentLength)
		return nil
	end
	return err
end

local function clearInfo(info)
	info = info or {}
	info.column_name = {}
	info.row_count = 0
	info.row_count_total = 0
	info.column_count = 0
	info.row_count_more = 0
	return info
end

local function clearCursor(cursor)
	util.clearRecord(cursor)
end

local function selectionToArrayTable(cursor, fieldNameArray, option, returnType)
	-- loadLibs() -- execute() must have been called first
	-- waitForLock(sock, "4drest") --- execute() must have been called first
	local sock = dconn.driverConnection(cursor.conn)
	local time
	if minMessageRows >= 0 and not peg.found(cursor.callParam, "SELECT COUNT(") then
		local tbl = option and (option.table or option.table_prefix)
		if tbl then
			tbl = dschema.externalNameSql(tbl, "4d", option.record_type)
		end
		ioWrite(util.color("bright green", "running query in 4D table '%s'... ", tbl or ""))
		time = util.seconds()
	end
	option = option or {}
	if not sock then
		local err = l("connection is nil, cursor '%s'", tostring(cursor))
		util.printRed("call 4D error: '%s'", tostring(err))
		clearCursor(cursor)
		return nil, err
	end
	local sql = cursor.callParam
	local authTbl = dconn.currentAuthTable()
	local connQuery = dconn.query()
	local fldName4d, recType
	if connQuery then
		recType = connQuery.recordType
	end
	if peg.found(fieldNameArray[1], ".") then --- not like 'count(*)'
		fldName4d = {}
		for i, name in ipairs(fieldNameArray) do
			fldName4d[i] = dschema.externalNameSql(name, "4d", recType)
			if fldName4d[i] == nil then
				fldName4d[i] = dschema.externalNameSql(name, "4d", recType) -- debug first
				fldName4d[i] = name -- set name after debug
			end
		end
	else
		fldName4d = fieldNameArray
	end

	local fldType = option.field_type or {}
	if peg.startsWith(sql, "SELECT COUNT(") then
		fldType[1] = "integer"
	end
	for i, name in ipairs(fldName4d) do
		if fldType[1] == nil then -- if i > 1 or fldType[1] == nil then
			fldType[i] = dschema.fieldTypeBasic(name, "4d", recType)
		end
	end

	if returnType == nil then
		returnType = "array table"
		util.printWarningWithCallPath("4d array return type is not set, setting it to '%s'", returnType)
	end
	local param = {
		return_type = returnType,
		sql = sql,
		field = fldName4d,
		local_field = fieldNameArray, -- option.field,
		field_type = fldType,
		table_prefix = option.table_prefix,
		table = option.table,
		auth = authTbl,
		query_name = "new:" .. (option.query_name or tostring(dsql.lastQueryName())) -- call always _lx_ExecuteSqlNew
	}
	if dsql.showSqlOn() then
		param.show_sql = true
	end
	local paramTxt = json.toJsonRaw(param)

	local function call4D()
		local err = callRest(cursor.conn, cursor, paramTxt)
		if err then
			local info = clearInfo()
			info.error = err
			clearCursor(cursor)
			return nil, info
		end

		local ret -- = util.newTable(0, colCount) -- this might cause problems, be careful
		local info = {}
		local data = cursor.data
		if cursor.compress and data ~= "" then
			if #data < cursor.contentLength then
				util.printRed("data length %d < cursor content length  %d", #data, cursor.contentLength)
			elseif #data > cursor.contentLength then
				util.printRed("data length %d > cursor content length  %d", #data, cursor.contentLength)
			end
			local dataUncompressed
			dataUncompressed, err = uncompress(data:sub(1, cursor.contentLength))
			if err == nil then
				data = dataUncompressed
			else
				err = err .. l("\n - call 4D content length: %d", cursor.contentLength)
			end
		end
		if err == nil and data and data:sub(1, 1) == "{" then --  and data:sub(-1) == "}" then -- json text
			ret, err = json.fromJson(data, nil, false) -- do not print json error
			if err then
				ret = {error = err}
			end
		else
			ret = {error = err or data}
		end
		if not ret then -- colCount == nil = result is an error
			info = clearInfo(info)
			info.error = l("cursor.data is nil")
			print(l("call 4D error: '%s'", info.error))
			ret = nil
		elseif ret.error then -- colCount == nil = resut is an error
			if type(ret.error) == "table" then
				info.error = table.concat(ret.error, "\n")
			elseif type(ret.error) == "string" then
				info.error = ret.error
			else
				util.printError(l("ret.error type '%s' is not a table or a string", type(ret.error)))
			end
			print(l("call 4D error: '%s'\n  sql: '%s'", info.error, sql))
			info = clearInfo(info)
			ret = nil
		end
		if ret and ret.data and ret.info and ret.info.return_type then -- and ret.info.return_type == "record array"
			if ret.info and ret.info.columnName then
				ret.info.column_name = ret.info.columnName
				ret.info.columnName = nil
				ret.info.column_count = ret.info.columnCount
				ret.info.columnCount = nil
				ret.info.row_count = ret.info.rowCount
				ret.info.rowCount = nil
				ret.info.row_count_total = ret.info.rowCountTotal
				ret.info.rowCountTotal = nil
				ret.info.query_time = ret.info.queryTime
				ret.info.queryTime = nil
			end
			ret, info = ret.data, ret.info
		elseif ret and ret.data then
			cursor.result = ret
			local colCount = cursorColumnCount(cursor) or 0
			local rowCount = cursorRowCount(cursor) or 0
			local colType = util.newTable(colCount, 0)
			local colName = util.newTable(colCount, 0)
			local colTagName = util.newTable(colCount, 0)
			-- io.write("  -> columns: " ..colCount..", rows: " ..row_count..", max returned rows: " ..maxReturnRows)
			-- io.flush()
			for i = 1, colCount do
				--[[if fldType then
				colType[i] = fldType[i]
			-- elseif fldNum then
				-- colType[i] = db.fieldTypeLua(fldNum[i]) -- or db.fieldType ?
			else]]
				colType[i] = fldType and fldType[i] or columnType(cursor, i)
				-- end
				colName[i] = columnName(cursor, i)
				if fieldNameArray then
					colTagName[i] = fieldNameArray[i]
				else
					colTagName[i] = colName[i]
				end
				if ret.data[colName[i]] and colTagName[i] ~= colName[i] then
					ret.data[colTagName[i]] = ret.data[colName[i]]
					ret.data[colName[i]] = nil
				end
				if colType[i] == "number" then
					if dschema.fieldType(colName[i], "4d", recType, "no-error") == "time" then
						local arr = ret.data[colName[i]]
						for j in ipairs(arr) do
							arr[j] = dt.toTimeString(arr[j])
							--  number to time string
						end
					end
				end
			end
			info = ret.info
			info.column_name = colTagName
			info.row_count = rowCount
			info.row_count_total = info.row_count_total or info.rowCountTotal -- old v12 has rowCountTotal
			-- info.row_count_more = info.row_count_more
			info.column_count = colCount
			info.return_type = ret.info.return_type
			info.column_name_external = fldName4d
		end
		if ret and ret.data then
			ret = ret.data -- v12 compatibility
			return ret, info, true
		end
		return ret, info
	end
	local ret, info = call4D()

	--[[ TODO: loop and call _lx_ExecuteSqlLoad() in a loop again until row_count >= row_count_total ]]
	local rowsFetched = info.row_count
	local rowsTotal = info.row_count_total
	local tbl
	if ret and rowsFetched and rowsFetched > 0 and rowsFetched < rowsTotal and rowsFetched < queryMaxRowsLimit then
		-- call _lx_ExecuteSqlLoad()
		local pos = param.sql:find("FROM ", 1, true) or 0
		local pos2 = param.sql:find(" ", pos + 6, true)
		tbl = param.sql:sub(pos + 5, pos2 - 1)
		if tbl == nil then
			rowsFetched = 0
			util.printError("rest 4D selectionToArrayTable sql table name is empty, parameter: %s", param)
		else
			param.sql = "SELECT 4D_MORE_ROWS FROM " .. tbl
			paramTxt = json.toJsonRaw(param)
			local ret2, info2, isKeyTable
			ioWrite(l("    loading data from 4D table '%s': %.1f%% ", tbl, rowsFetched / rowsTotal * 100))
			repeat
				ret2, info2, isKeyTable = call4D()
				if ret2 == nil or info2.error then
					clearCursor(cursor)
					return ret2, info2
				end
				rowsFetched = rowsFetched + info2.row_count
				-- combine arrays to first set
				if info2.row_count > 0 then
					if isKeyTable then
						for col, val in pairs(ret2) do
							if ret[col] == nil then
								util.printError("column '%s' does not exist in return table", tostring(info.column_name and info.column_name[col]))
							else
								fn.util.concatArray(ret[col], val)
							end
						end
					elseif returnType == "record array" then
						fn.util.concatArray(ret, ret2)
					else
						for key in pairs(ret) do
							fn.util.concatArray(ret[key], ret2[key])
						end
					end
					ioWrite(l("%.1f%% ", rowsFetched / rowsTotal * 100))
				end
			until info2.row_count_more == nil or info2.row_count_more < 1 or rowsFetched >= rowsTotal or rowsFetched >= queryMaxRowsLimit
			if rowsFetched ~= rowsTotal then
				util.printError("rowsFetched %s is not equal to rowsTotal %s, row count more %s", tostring(rowsFetched), tostring(rowsTotal), tostring(info2.row_count_more))
			end
		end
	end
	info.row_count = rowsFetched
	if time and minMessageRows >= 0 then
		if option.fieldSortParam and #option.fieldSortParam > 0 then
			ioWrite(util.color("bright green", "    loaded 100%%, %s %s in %.3f seconds", formatNum(info.row_count or 0), info.row_count == 1 and "row" or "rows", util.seconds(time)))
		else
			util.printOk("    loaded 100%%, %s %s in %.3f seconds\n", formatNum(info.row_count or 0), info.row_count == 1 and "row" or "rows", util.seconds(time))
		end
	end
	clearCursor(cursor)
	return ret, info
end
rest4d.selectionToArrayTable = selectionToArrayTable

function rest4d.selectionToRecordArray(cursor, fieldNameArray, option)
	local sel, info = selectionToArrayTable(cursor, fieldNameArray, option, "record array") --
	-- convert old 4d array to new record array - not needed after new plg4d -call because selectionToArrayTable() handles this
	-- 4d v12 does not return info.return_type, 4D v19 does return info.return_type
	if sel and info and info.return_type ~= "record array" and info.column_name and #info.column_name > 0 then
		local time
		local arrCount = math.huge
		local rowCount = info.row_count
		if rowCount > minSortMessageRows then
			time = util.seconds()
		end
		local key, useRecData, arr, name
		local ret = util.newTable(rowCount)
		for i = 1, rowCount do
			ret[i] = {}
		end
		for col = 1, #info.column_name do
			key = fieldNameArray[col]
			arr = sel[key]
			if option and option.table_prefix and peg.startsWith(key, option.table_prefix) then
				key = peg.parseAfter(key, option.table_prefix .. ".")
			end
			if arr == nil then
				arr = sel[key] -- compatibility with arm Mac with old call or v12
			end
			if peg.found(key, ".") then
				useRecData = true
			else
				useRecData = false
			end
			if arr == nil then
				name = info.column_name_external and info.column_name_external[col] or info.column_name[col]
				arr = sel[name]
			end
			if arr == nil then
				util.printError("column '%s' was not found from call 4D selectionToRecordArray() return", tostring(info.column_name[col]))
			else
				if #arr ~= rowCount then
					if #arr < arrCount or col == 1 then
						arrCount = #arr
					end
					util.printError("column '%s' array count %d differs from returned count %d", tostring(info.column_name[col]), #arr, rowCount)
				end
				if useRecData then
					for i, value in ipairs(arr) do
						recDataSet(ret[i], key, value)
					end
				else
					for i, value in ipairs(arr) do
						ret[i][key] = value
					end
				end
			end
		end
		sel = ret
		if rowCount > minSortMessageRows then
			util.print(" 4D array to record array conversion time: %.3f seconds, %d rows", util.seconds(time), rowCount)
		end
		if rowCount ~= arrCount then
			for i = rowCount, arrCount + 1, -1 do
				table.remove(ret, i)
			end
		end
	end
	if sel and not info.error and option.fieldSortParam and #option.fieldSortParam > 0 then
		local time = util.seconds()
		local sortParam = {} -- {"sort_type", ">", "reff_start_time", "<", "reff_end_time", ">"}
		for i, field in ipairs(option.fieldSortParam) do
			if i % 2 == 1 then
				if option and option.table_prefix and peg.startsWith(field, option.table_prefix) then
					sortParam[i] = peg.parseAfter(field, option.table_prefix .. ".")
				else
					sortParam[i] = field
				end
			else -- sort asc or desc
				if peg.endsWith(field, " lower") then
					sortParam[i] = field:sub(1, -7)
				else
					sortParam[i] = field
				end
			end
		end
		if info.rowCount and info.row_count == nil then
			info.row_count = info.rowCount or 0
		end
		if info.row_count > minSortMessageRows then
			ioWrite(util.color("bright green", " - sorting %s %s", formatNum(info.row_count or 0), info.row_count == 1 and "row" or "rows"))
		end
		sort.sort(sel, sortParam)
		util.printOk(" - sorted %s %s in %.4f seconds\n", formatNum(info.row_count or 0), info.row_count == 1 and "row" or "rows", util.seconds(time))
	end
	return sel, info
end

function rest4d.selectionToRecordTable(cursor, fieldNameArray, option)
	local sel, info = selectionToArrayTable(cursor, fieldNameArray, option, "array table") --
	if info.return_type == "record array" then -- 4d v12 does not return info.return_type, 4D v19 does return info.return_type
		return sel, info
	end
	return dconv.arrayTableToRecordTable(sel, info, fieldNameArray, option)
end

return rest4d
