--- database-sqlite.lua
-- Database module, sqlite3.
-- copied from: https://github.com/SinisterRectus/lit-sqlite3
-- see also original: https://scilua.org/ljsqlite3.html
-- @module database
local util = require "util"
local fs = require "fs"
local l = require"lang".l
local peg = require "peg"
local import = require "import"
local parseAfter, startsWith = import.from(peg, "parseAfter, startsWith")
local dconn = require "dconn"
local dschema = require "dschema"
local recDataSet = require"recdata".set
local sqlite = require "sqlite3"
local SQLITE_ERROR, SQLITE_ROW, SQLITE_DONE, sqlite3_step = sqlite.sql.SQLITE_ERROR, sqlite.sql.SQLITE_ROW, sqlite.sql.SQLITE_DONE, sqlite.sql.sqlite3_step
-- local getType = "k" -- hik

local auth
local function loadLibs()
	if not auth then
		auth = require "auth"
	end
end

local maxReturnRows = math.huge

local function maxSaveRows()
	return 50
end

local function disconnect(driverConn)
	if driverConn._closed then
		return
	end
	return driverConn:close() -- Closes stmt as well
end

local function shutdown()
	util.print("sqlite shutdown")
	sqlite.shutdown()
end

--[[ standard sql92?
function structureQuery()
	return "SELECT column_name, data_type, character_maximum_length FROM information_schema.columns WHERE table_name = "
end
]]

local function setReturnRowLimit(rows)
	maxReturnRows = rows
end

local function execute(sqlExecute, option)
	-- sqlExecute = sqlExecute:gsub("%.name,", ".name_,")
	-- print(sqlExecute)
	-- loadLibs()
	-- local driverConn = option and option.connection or dconn.driverConnection(nil, option.database)
	local driverConn = option and option.driver_conn or dconn.driverConnection(nil, option.database)
	if not (type(driverConn) == "cdata" or type(driverConn) == "userdata") then
		return nil, l("driver connection is not valid cdata: '%s'", tostring(driverConn))
	end
	local statement, err, ret
	local doRollback = false
	local isSelect = peg.startsWith(sqlExecute, "SELECT ") or peg.startsWith(sqlExecute, "PRAGMA ")
	if not isSelect then
		-- local startText = "BEGIN;SET LOCAL application.user_id='" .. auth.currentUserId() .. "'; "
		-- statement, err = driverConn:exec("BEGIN; " .. sqlExecute .. "; COMMIT;") -- cannot start a transaction within a transaction
		-- doRollback = true
		statement, err = driverConn:exec(sqlExecute)
		if statement then
			-- local res, rowCount, err = statement:resultSet()
			ret = sqlite3_step(statement._ptr)
			if ret == SQLITE_ERROR or ret ~= SQLITE_DONE then
				err = sqlite.E_conn(statement._conn, ret, false) -- false = do not print error
				err = util.printRed("sqlite execute error: %s (%s)", tostring(err), tostring(ret))
			end
			statement:close()
			statement = nil
		end
	else
		statement, err = driverConn:exec(sqlExecute)
	end
	--[[
	local stmt
	stmt, err = conn:prepare("INSERT INTO company (company_id, name) VALUES(?, ?)")
	test.is_nil(err)
	for _, rec in ipairs(sel) do
		stmt:reset():bind(rec.company_id, rec.name):step()
	end ]]
	if err then
		if statement then
			statement:close()
		end
		if doRollback then
			local err2
			statement, err2 = driverConn:exec("ROLLBACK;") -- , getType)
			ret = sqlite3_step(statement._ptr)
			if ret == SQLITE_ERROR or ret ~= SQLITE_DONE then
				err = sqlite.E_conn(statement._conn, ret, false) -- false = do not print error
				err2 = util.printRed("sqlite rollback error: %s (%s)", err, tostring(ret))
			end
			statement:close()
			if err2 then
				util.printInfo("Sqlite ROLLBACK statement return '%s', error '%s'", tostring(ret), tostring(err2))
			else
				util.printInfo("Sqlite ROLLBACK")
			end
		end
		return nil, err
	end
	return {statement = statement, sql_execute = sqlExecute, is_select = isSelect, driver = driverConn} -- cursor
end

local function connect(prf)
	local host = fs.filePathFix(prf.host)
	local path = "file:" .. host .. ".sqlite3?cache=shared" -- &mode=memory
	local diskPath = peg.parseBetweenWithoutDelimiter(path, "file:", "?")
	local diskPathExists = fs.fileExists(diskPath)
	prf.file_path = diskPath
	prf.file_path_exists = diskPathExists
	-- path = fs.filePathFix(util.tempPath() .. "sqlite-test.db")
	-- fs.deleteFile(path, nil, false, "no-warning")
	local mode = prf.mode -- red, write, create database
	if mode == nil then
		if diskPathExists then
			mode = "ro"
		else
			mode = "rwc"
		end
	elseif mode == "ro" then
		if not diskPathExists then
			return nil, nil, l("file '%s' does not exist", diskPath)
		end
	elseif mode == "rw" then
		if not diskPathExists then
			mode = "rwc"
		end
	else
		return nil, l("mode '%s' must be rw or ro, file '%s'", tostring(mode), diskPath), nil
	end
	local conn, driverErr = sqlite.open(path, mode, "") -- prf.password or "")
	-- res, n, err = conn:exec("PRAGMA temp_store")
	-- res, n, err = conn:exec("PRAGMA temp_store = 2")
	--[[ if not err then
		loadLibs()
		local _, err2 = execute("SET application_name='nc-server'; SET client_min_messages TO WARNING;", {connection = conn})
		if err2 then
			util.printError("query SET application_name='nc-server'; failed, error '%s'", err2)
		end
	end ]]
	return conn, nil, driverErr
end

--[[
local function rowCount(cursor)
	local ret = cursor.rows
	return ret
end

local function columnTypes(cursor)
	local ret = cursor:getcoltypes()
	return ret
end

local function columnCount(cursor)
	return sqlite.fields()
end

local function columnNames(cursor)
	local cols = columnCount(cursor)
	local ret = {}
	for i = 1, cols do
		ret[i] = sqlite.field(i)
	end
	return ret
end
]]

local function selectionToArray(cursor, fieldNameArray, option, returnTableType)
	if not cursor then
		local err = l("cursor is nil")
		util.printError(err)
		-- cursor.pg:reset() -- == cursor:close()
		return nil, {error = err}
	end
	if not cursor.statement then
		local err = l("selection sql cursor statement is missing")
		util.printError(err)
		return nil, {error = err}
	end
	if not cursor.is_select then
		cursor.statement:close()
		local err = l("selection sql cursor type is not select")
		util.printError(err)
		return nil, {error = err}
	end
	cursor.option = option or {}
	local useRecData = {}
	if option and option.table_prefix then
		local columnNameArray = {}
		for i = 1, #fieldNameArray do
			if startsWith(fieldNameArray[i], option.table_prefix) then
				columnNameArray[i] = parseAfter(fieldNameArray[i], ".")
			else
				columnNameArray[i] = fieldNameArray[i]
			end
			useRecData[i] = peg.found(columnNameArray[i], ".")
		end
		cursor.option.fieldNameArray = columnNameArray
	else
		cursor.option.fieldNameArray = fieldNameArray
	end
	-- TODO: convert field types if they are given
	local ret = {} -- cursor.statement:resultSet(maxReturnRows)
	local err, code
	local statement = cursor.statement
	local ptr = statement._ptr
	fieldNameArray = cursor.option.fieldNameArray
	-- local row, column = statement:_step({}, {})
	local row = 0
	if returnTableType == "array table" then
		for _, name in ipairs(fieldNameArray) do
			ret[name] = {}
		end
	end
	while row < maxReturnRows do
		code = sqlite3_step(ptr)
		if code == SQLITE_ROW then
			row = row + 1
			-- for col = 1, statement:_ncol() do
			if returnTableType == "array table" then
				for col, name in ipairs(fieldNameArray) do
					ret[name][row] = sqlite.get_column(ptr, col - 1)
				end
			else -- if returnTableType == "record array" then
				ret[row] = {}
				for col, name in ipairs(fieldNameArray) do
					if useRecData[col] then
						recDataSet(ret[row], name, sqlite.get_column(ptr, col - 1))
					else
						ret[row][name] = sqlite.get_column(ptr, col - 1)
					end
				end
			end
		elseif code == SQLITE_DONE then -- Have finished now.
			break
		else -- If code not DONE or ROW then it's error.
			err = sqlite.E_conn(statement._conn, code)
			break
		end
		--[[ for col, name in ipairs(cursor.option.fieldNameArray) do
				ret = cursor.rows[col]
				for i = 1, rowCount do
					if col == 1 then
						ret[i] = {}
					end
					if useRecData[i] then
						recDataSet(ret[i], name, rows[i])
					else
						ret[i][name] = rows[i]
					end
				end
			end ]]
	end
	statement:close()
	local info = {error = err}
	info.column_name = fieldNameArray -- columnNameArray
	info.row_count = row -- rowsCounted
	info.row_count_total = row --  rows
	info.column_count = #fieldNameArray -- colCount
	return ret, info
end

local function selectionToRecordArray(cursor, fieldNameArray, option)
	return selectionToArray(cursor, fieldNameArray, option, "record array")
end

local function selectionToArrayTable(cursor, fieldNameArray, option)
	return selectionToArray(cursor, fieldNameArray, option, "array table")
end

local function selectionToRecordTable(cursor, fieldNameArray, option)
	local sel, info = selectionToArray(cursor, fieldNameArray, option, "array table")
	local dconv = require "dconv"
	return dconv.arrayTableToRecordTable(sel, info, nil, option) -- this changes deep tag names to subtags
	-- do
	-- test performance of this, returns object array:
	-- SELECT row_to_json(row) AS data FROM (SELECT * FROM  pg_stat_activity) AS row
	-- https://hashrocket.com/blog/posts/faster-json-generation-with-Sqliteql
	-- end
end

local tableCountCache
local function tableCount() -- (conn)
	if tableCountCache then
		return tableCountCache
	end
	local cursor = execute("SELECT COUNT(*) FROM sqlite_stat_user_tables")
	local ret, info = selectionToRecordArray(cursor, {"COUNT(*)"})
	if not ret then
		return nil, info
	end
	tableCountCache = tonumber(ret["COUNT(*)"][1])
	return tableCountCache
end

local function fieldCount(tableName)
	loadLibs()
	if type(tableName) ~= "string" then
		local err = l("table name '%s' is invalid", tostring(tableName))
		util.printError(err)
		return nil, err
	end
	local driverConn = dconn.driverConnection() -- dconn.driverConnection(nil, database)
	if not (type(driverConn) == "cdata" or type(driverConn) == "userdata") then
		return nil, l("driver connection is not valid cdata: '%s'", tostring(driverConn))
	end
	local tableNameEscaped = sqlite.quote(driverConn, tableName)
	local cursor = execute("SELECT COUNT(*) FROM sqlite_schema.columns WHERE table_name='" .. tableNameEscaped .. "'")
	local ret, info = selectionToRecordArray(cursor, {"COUNT(*)"})
	if not ret then
		return nil, info
	end
	return tonumber(ret["COUNT(*)"][1])
end

local function tableNameArray()
	local conn = dconn.driverConnection()
	local cursor = execute("SELECT name FROM sqlite_schema WHERE type ='table' AND name NOT LIKE 'sqlite_%';", {connection = conn})
	local ret, info = selectionToRecordArray(cursor, {"table_name"})
	return ret, info
end

local function fieldNameArray(tableName, organizationId)
	local cursor = execute("PRAGMA table_info('" .. tableName .. "')", {organization_id = organizationId})
	local ret, info = selectionToRecordArray(cursor, {"cid", "field_name", "field_type"}, {organization_id = organizationId, table = "PRAGMA table_info"}) -- cid name type notnull dflt_value pk, {"cid", "field_name", "field_type", "notnull", "dflt_value", "pk"}
	for _, item in ipairs(ret) do
		item.cid = nil
		item.field_type = item.field_type == "" and "text" or dschema.sqlTypeToFieldType(item.field_type)
		-- item.field_length = item.field_length ~= "" and tonumber(item.field_length) or nil
	end
	return ret, info
end

return {
	_NAME = "sqlite-mc",
	maxSaveRows = maxSaveRows,
	-- dbType = dbType,
	setReturnRowLimit = setReturnRowLimit,
	disconnect = disconnect,
	shutdown = shutdown,
	connect = connect,
	execute = execute,
	selectionToRecordArray = selectionToRecordArray,
	selectionToArrayTable = selectionToArrayTable,
	selectionToRecordTable = selectionToRecordTable,
	tableCount = tableCount,
	fieldCount = fieldCount,
	tableNameArray = tableNameArray,
	fieldNameArray = fieldNameArray
}
