--- dconn.lua
-- Database connections.
-- local dconn = require "dconn"
-- @module db
local dconn = {test = {}} -- test functions, not public

local ffi = require "mffi"
local util = require "util"
local l = require"lang".l
local fn = require "fn"
local dt = require "dt"
local peg = require "peg"
local dsql = require "dsql"
local auth = require "auth"
local dschema -- = require "dschema"
local doperation, pg, sqlite, mysql, odbc, plg4d, rest4d, callRest -- delay load when needed
local connection, disconnect, setCurrentConnection -- forward declaration
local print = util.print
local coro = require "coro"
local useCoro = coro.useCoro(false)
local threadId = coro.threadId
local currentThread = coro.currentThreadFunction()
local currentThreadId = coro.currentThreadId

local debugLevel = 0

local connectionPref = nil
local loc = {}
loc.connectionPrefPath = nil
loc.useDomainSocket = not util.isWin()
loc.defaultDateFormat = "%Y-%m-%d" -- todo: move to connection.json
loc.defaultDateTimeFormat = "%Y-%m-%d %H:%M:%S" -- todo: move to connection.json
loc.defaultDateTime = "%H:%M:%S" -- todo: move to connection.json

loc.driverTbl = {} -- no need for threads
loc.aliasId = {} -- no need for threads
loc.driverConnectionTable = {} -- threads ok
loc.organizationConnection = {} -- threads ok
loc.organizationExtraConnection = {} -- threads ok
if useCoro then
	loc.currentConnection = {} -- threads ok
	loc.currentConnectionArr = {} -- threads ok
	loc.localConnection = {} -- threads ok
else
	loc.currentConnection = nil
	loc.localConnection = nil
end
loc.tableConnectionDropdown = nil
loc.organizationPref = nil
loc.tableRedirectData = nil
loc.connectionTraceInUse = false
loc.queryLimit = 10 ^ 9 -- todo: dbfix
loc.debugConnectionChange = 1
loc.debugConnect = false
loc.debugConnectId = {}
loc.debugDisconnect = false
if util.from4d() then
	loc.debugConnectionChange = false
end

dconn.currentAuthTable = auth.currentAuthTable

local function getCurrentConnection(thread)
	if useCoro then
		thread = thread or currentThread()
		return loc.currentConnection[thread] -- or {}
	else
		if loc.currentConnection and loc.currentConnection.closed then
			loc.currentConnection = nil
		end
		return loc.currentConnection -- or {}
	end
end
dconn.getCurrentConnection = getCurrentConnection

local function setCurrentConnectionArr(conn, thread)
	if useCoro then
		if conn and not conn.closed then
			if loc.currentConnection[thread] == conn then
				return
			end
			if loc.currentConnectionArr[thread] == nil then
				loc.currentConnectionArr[thread] = {}
			end
			local idx = #loc.currentConnectionArr[thread]
			loc.currentConnectionArr[thread][idx + 1] = conn
			loc.currentConnection[thread] = conn
		else -- nil connection or closed connection
			local idx = #loc.currentConnectionArr[thread]
			if idx and idx > 0 then
				loc.currentConnection[thread] = loc.currentConnectionArr[idx - 1]
				loc.currentConnectionArr[thread][idx] = nil
			else
				loc.currentConnection[thread] = nil
			end
		end
	else
		loc.currentConnection = conn
	end
end

function dconn.currentConnection()
	local currentConn = getCurrentConnection()
	if currentConn == nil then
		currentConn = connection()
	end
	return currentConn
end

local function setLocalConnection(conn, thread)
	if useCoro then
		thread = thread or currentThread()
		if loc.localConnection[thread] ~= conn then
			loc.localConnection[thread] = conn
		end
	else
		loc.localConnection = conn -- or {}
	end
end

local function getLocalConnection(thread, warnMissing)
	if useCoro then
		thread = thread or currentThread()
		if warnMissing and loc.localConnection[thread] == nil and not util.tableIsEmpty(loc.localConnection) then
			util.printWarning("local connection is empty, %s\n%s", threadId(thread), debug.traceback())
			return
		end
		return loc.localConnection[thread]
	else
		return loc.localConnection
	end
end
dconn.getLocalConnection = getLocalConnection

local function createLocalConnection(organizationId, thread)
	thread = thread or currentThread()
	local currentConn = getCurrentConnection(thread)
	-- todo: we should find organizationId local connection using redirects
	local localConn = connection({organizationId = organizationId, createLocal = true}) -- connection() uses currentAuthTable an calls setLocalConnection
	if localConn and currentConn and localConn.organization_id ~= currentConn.organization_id then
		setCurrentConnection(currentConn, "create local connection")
	end
	return localConn, currentConn
end

function dconn.localConnectionId()
	local thread = currentThread()
	local localConnection = getLocalConnection(thread)
	if localConnection then
		return localConnection.organization_id, localConnection
	end
	localConnection = createLocalConnection(thread)
	if localConnection and localConnection.is_local then
		return localConnection.organization_id, localConnection
	end
	return "unknown"
end

local function getExtraConnection(thread)
	if useCoro then
		thread = thread or currentThread()
		if loc.organizationExtraConnection[thread] == nil then
			loc.organizationExtraConnection[thread] = {}
		end
		return loc.organizationExtraConnection[thread]
	else
		return loc.organizationExtraConnection
	end
end

local function setExtraConnection(thread, conn)
	if useCoro then
		thread = thread or currentThread()
		if loc.organizationExtraConnection[thread] == nil then
			loc.organizationExtraConnection[thread] = {}
		end
		loc.organizationExtraConnection[thread][conn] = conn.id or -1
	else
		loc.organizationExtraConnection[conn] = conn.id
	end
end

local function setOrganizationConnection(orgId, conn, thread, restore)
	if useCoro then
		thread = thread or currentThread()
		if loc.organizationConnection[thread] == nil then
			loc.organizationConnection[thread] = {}
		end
		if loc.organizationConnection[thread][orgId] == nil then
			loc.organizationConnection[thread][orgId] = conn
		elseif loc.organizationConnection[thread][orgId] ~= conn then
			if restore then
				loc.organizationConnection[thread][orgId] = conn
			elseif loc.organizationConnection[thread][orgId].closed then
				loc.organizationConnection[thread][orgId] = conn
			elseif type(conn) == "table" and loc.organizationConnection[thread][orgId].id ~= conn.id then -- todo: find out what code clones connections?
				util.printRed("trying to change '%s' organization '%s' connection from '%s' to '%s'", threadId(thread), orgId, loc.organizationConnection[thread][orgId].info, conn.info)
			end
		end
	else
		loc.organizationConnection[orgId] = conn
	end
end

local function loadDbConnPreference()
	if connectionPref then
		return -- run only once
	end
	local fs = require "fs"
	local defaultConnPref = util.readUpperLevelPreferenceFile("auth/default-connection.json", "no-db no-error") -- conn pref can't be read from db
	if util.tableIsEmpty(defaultConnPref) then
		util.print("upper level preference file '%s' was not found", "auth/default-connection.json")
		util.closeProgram()
	end
	local envChangeCount = 0
	local defaultEnv = util.readUpperLevelPreferenceFile("auth/default-env.json", "no-error no-db use-default")
	local env = util.readUpperLevelPreferenceFile(".nc-env.json", "no-error no-db use-default")
	if util.tableIsEmpty(env) then
		util.print("upper level preference file '%s' was not found", ".nc-env.json")
	end
	for key, value in pairs(defaultEnv) do
		if env[key] == nil then
			if type(value) ~= "table" or next(value) then
				envChangeCount = envChangeCount + 1
				if key == "default_id" and env.default_password then
					value = env.default_database
					util.printRed("  setting .nc-env.json key '%s' to value '%s', please fix it to be id", tostring(key), tostring(value))
				elseif peg.found(tostring(key), "password") then
					util.print("  setting .nc-env.json key '%s' to value 'xxxxx'", tostring(key))
				elseif type(value) == "table" or type(value) == "string" then
					util.print("  setting .nc-env.json key '%s' to value '%s'", tostring(key), value)
				else
					util.print("  setting .nc-env.json key '%s' to value '%s'", tostring(key), tostring(value))
				end
				env[key] = value
			end
		end
	end
	if env.default_password then
		envChangeCount = envChangeCount + 1
		env.default_database_password = env.default_password
		env.default_password = nil
	end
	if env.default_database then
		envChangeCount = envChangeCount + 1
		if env.default_id == nil then
			env.default_id = env.default_database
		end
		env.default_database = nil
	end
	local path
	connectionPref, loc.connectionPrefPath = util.readUpperLevelPreferenceFile("connection.json", "no-error no-db use-default")
	if util.tableIsEmpty(connectionPref) then
		local authConnection, prefPath = util.readUpperLevelPreferenceFile("auth/connection.json", "no-error no-db")
		local dbConnection = util.readUpperLevelPreferenceFile("table/db_connection_preference.json", "no-error no-db")
		if authConnection and dbConnection then
			prefPath = peg.parseBeforeLast(prefPath, "/")
			prefPath = peg.parseBeforeLast(prefPath, "/") .. "/"
			local fix = require "tool/fix-code/fix-json"
			fix.fixConnection({env = env, auth_connection = authConnection, db_connection_preference = dbConnection, path = prefPath, connection_path = "connection/", redirect_path = "redirect/", organization_path = "organization/"})
			util.printInfo("converted old auth/connection.json and table/db_connection_preference.json -preferences to new format")
			connectionPref, loc.connectionPrefPath = util.readUpperLevelPreferenceFile("connection.json", "no-error no-db use-default")
		else
			util.print("upper level preference file was not found: '%s'", "connection.json")
			path = fs.filePathFix(util.mainPath() .. "../nc-preference/")
			fs.createPath(path)
			connectionPref.name = "nc-preference/connection.json"
			connectionPref.default_language = "en"
			connectionPref.organization = {}
			for i, orgPrefName in ipairs(defaultConnPref.organization) do
				local orgPref = util.readUpperLevelPreferenceFile(orgPrefName, "no-error no-db use-default")
				local name = peg.parseAfterLast(orgPref.name, "/")
				if peg.found(name, "plg4d") and not util.isLinux() then
					connectionPref.organization[i] = orgPref.name -- database/organization/plg4d.json
				else
					orgPref.name = name
					orgPref.database = {orgPref.database[1], orgPref.database[2], orgPref.database[3], orgPref.database[4], orgPref.database[5]} -- max 5 first from demo connection
					connectionPref.organization[i] = name -- part after database/organization/
					fs.writeFile(path .. orgPref.name, orgPref, "log-write")
				end
			end
			fs.writeFile(path .. "connection.json", connectionPref, "log-write")
		end
	end
	if connectionPref.default_id == nil then
		if env.default_id then
			connectionPref.default_id = env.default_id
			env.default_id = nil
			envChangeCount = envChangeCount + 1
		else
			connectionPref.default_id = defaultConnPref.default_id
		end
		path = fs.filePathFix(util.mainPath() .. "../nc-preference/")
		fs.writeFile(path .. "connection.json", connectionPref, "log-write")
	end
	if connectionPref.default_id == nil or peg.countOccurrence(connectionPref.default_id, "-") ~= 2 then
		-- default_database_password
		path = path or fs.filePathFix(util.mainPath() .. "../nc-preference/")
		util.printRed("*** please fix file '%s' key 'default_id' to be in form like 'X-%s-Y' where X is 'your organization prefix' like 'demo' and Y is your organization number (almost always 0), then restart the server ***", path .. "connection.json", connectionPref.default_id or "fi_demo")
		util.closeProgram(false)
	end
	if connectionPref.organization == nil then
		util.printError("connection preference organization '%s' does not exist", connectionPref.organization)
		util.closeProgram()
	end
	if env.default_id then
		env.default_id = nil
		envChangeCount = envChangeCount + 1
	end
	if envChangeCount > 0 then
		path = path or fs.filePathFix(util.mainPath() .. "../nc-preference/")
		if not fs.exists(path) then
			fs.createPath(path)
		end
		env.name = "nc-preference/.nc-env.json"
		local writePath = fs.writeFile(path .. ".nc-env.json", env, "log-write")
		if writePath == nil then
			writePath = fs.writeFile(path .. "nc-env.json", env, "log-write")
			if writePath ~= nil then
				util.printWarning("please rename file '%s' to .nc-enc.json and then restart the server", writePath)
				util.closeProgram()
			end
		end
	end
	for key, value in pairs(defaultConnPref) do
		if connectionPref[key] == nil then
			connectionPref[key] = value
		end
	end
	if env.default_database_password == nil or #env.default_database_password < 10 then
		-- default_database_password
		path = fs.filePathFix(util.mainPath() .. "../nc-preference/")
		util.printRed("*** please fix file '%s' key 'default_database_password' to be at least 10 characters long, then restart the server ***", path .. ".nc-env.json")
		util.closeProgram(false)
	end
	util.recToRec(connectionPref, env, {replace = true, copy_key = {name = false}}) -- copy (override) env keys to connectionPref
	local organizationArr = {}
	for _, conn in ipairs(connectionPref.organization) do
		local org = util.readUpperLevelPreferenceFile(conn, "no-error no-db use-default")
		if util.tableIsEmpty(org) then
			util.printError("connection '%s' preference does not exist", conn)
			util.closeProgram()
		else
			if org.status == "active" then -- not nil or false
				for _, item in ipairs(org.database) do
					item.show = org.show .. item.show
					item.organization_id = org.organization .. "-" .. (item.database_id or item.database) .. "-" .. item.organization_number
				end
				util.arrayConcat(organizationArr, org.database)
			end
		end
	end
	connectionPref.organizationArr = organizationArr
	connectionPref.organizationIdx = fn.util.createIndex(organizationArr, "organization_id")
end

local function aliasId(aliasName)
	if not connectionPref then
		loadDbConnPreference()
	end
	if aliasName == "default" and connectionPref.default_id then
		return connectionPref.default_id
	end
	if connectionPref.aliasId == nil then
		connectionPref.aliasId = fn.util.createIndex(connectionPref.organizationArr, "alias_id", nil, nil, "organization_id")
	end
	local id = connectionPref.aliasId[aliasName] or loc.aliasId[aliasName] -- or aliasName
	if id == "demo-4d-0" and util.isLinux() then -- TODO: use alias pref
		id = "demo-fi_demo4d-0"
	end
	return id or aliasName -- TODO: check that aliasName parameter is valid connection id like 'demo-4d-0'
end
dconn.aliasId = aliasId

local function getOrganizationConnection(orgId_)
	local orgId = aliasId(orgId_)
	if useCoro then
		local thread = currentThread()
		if loc.organizationConnection[thread] == nil then
			return
		end
		return loc.organizationConnection[thread][orgId]
	else
		return loc.organizationConnection[orgId]
	end
end

local function clearOrganizationConnection(thread)
	if useCoro then
		thread = thread or currentThread()
		loc.organizationConnection[thread] = nil
	else
		loc.organizationConnection = {} -- clears all
	end
end

local function clearDriverConnection(orgId, thread)
	if useCoro then
		thread = thread or currentThread()
		loc.driverConnectionTable[thread][orgId] = nil
		loc.organizationConnection[thread][orgId] = nil
	else
		loc.driverConnectionTable[orgId] = nil
		loc.organizationConnection[orgId] = nil
	end
end

local function setDriverConnection(orgId, driverConn, thread)
	if useCoro then
		thread = thread or currentThread()
		if loc.driverConnectionTable[thread] == nil then
			loc.driverConnectionTable[thread] = {}
		end
		local prevDriverConn = loc.driverConnectionTable[thread][orgId]
		if driverConn == prevDriverConn then
			return
		end
		if driverConn == nil then
			loc.driverConnectionTable[thread][orgId] = nil
		elseif driverConn == nil or loc.driverConnectionTable[thread][orgId] == nil then
			loc.driverConnectionTable[thread][orgId] = driverConn
		elseif loc.driverConnectionTable[thread][orgId] ~= driverConn then
			if type(loc.driverConnectionTable[thread][orgId]) == "table" and driverConn.socket then
				if loc.driverConnectionTable[thread][orgId].closed then
					util.printInfo("changed '%s' organization '%s' driver connection closed socket '%s' to '%s'", threadId(thread), orgId, tostring(loc.driverConnectionTable[thread][orgId].closed), tostring(driverConn.socket))
					loc.driverConnectionTable[thread][orgId] = driverConn
					return
				end
			end
			if type(loc.driverConnectionTable[thread][orgId]) == "table" and type(driverConn) == "table" then
				if driverConn.schema ~= "4d" or loc.driverConnectionTable[thread][orgId].schema ~= "4d" then
					util.printRed("changing '%s' organization '%s' driver connection from '%s' to '%s'", threadId(thread), orgId, loc.driverConnectionTable[thread][orgId], driverConn)
				else
					util.printRed("changing '%s' organization '%s' driver connection from '%s' to '%s'", threadId(thread), orgId, tostring(loc.driverConnectionTable[thread][orgId]), tostring(driverConn))
				end
			end
			loc.driverConnectionTable[thread][orgId] = driverConn
		end
	elseif driverConn == nil or loc.driverConnectionTable[orgId] == nil then
		loc.driverConnectionTable[orgId] = driverConn
	elseif loc.driverConnectionTable[orgId] ~= driverConn then
		util.printRed("trying to change organization '%s' driver connection from '%s' to '%s'", orgId, tostring(loc.driverConnectionTable[orgId]), tostring(driverConn))
	end
end

local function getDriverConnection(orgId, thread)
	if useCoro then
		thread = thread or currentThread()
		if loc.driverConnectionTable[thread] == nil or loc.driverConnectionTable[thread][orgId] == nil then
			return
		end
		return loc.driverConnectionTable[thread][orgId]
	else
		return loc.driverConnectionTable[orgId]
	end
end

function dconn.debugConnectionChange(val)
	local prevVal = loc.debugConnectionChange
	if val == nil then
		util.printError("dconn.debugConnectionChange() parameter is nil")
	else
		loc.debugConnectionChange = val
	end
	return prevVal
end

local function loadPreference()
	if loc.organizationPref then
		return
	end
	loc.organizationPref = {}
	loadDbConnPreference()
	if connectionPref == nil then
		return
	end
	loc.tableConnectionDropdown = {}
	local idx = 0
	local connIdx = {}
	local connLoadedIdx = {}
	local redirectIdx = {}
	local allowDatabaseSchemaUpdate = {}
	for i, org in ipairs(connectionPref.organizationArr) do
		local id = org.organization_id
		if loc.organizationPref[id] then -- TODO: should allow different redirectIdx
			util.printError("same database '%s', organization '%s' is more than 1 times in preference '%s' at position %d", org.database, id, loc.connectionPrefPath, i)
			util.closeProgram()
		else
			if org.show and org.status == "active" then -- and (org.status == "active" or org.status == nil) then
				idx = idx + 1
				loc.tableConnectionDropdown[idx] = util.clone(org)
				loc.tableConnectionDropdown[idx].value = id
			end
			org.allow_database_schema_update = org.allow_database_schema_update == true
			if allowDatabaseSchemaUpdate[org.database] == nil then
				if allowDatabaseSchemaUpdate[org.database] ~= nil and allowDatabaseSchemaUpdate[org.database] ~= org.allow_database_schema_update then
					util.printError("same database '%s', organization '%s' is more than 1 times in preference '%s' at position %d", org.database, id, loc.connectionPrefPath, i)
					util.closeProgram()
				elseif allowDatabaseSchemaUpdate[org.database] == nil then
					allowDatabaseSchemaUpdate[org.database] = org.allow_database_schema_update
				end
			end
			if org.redirect ~= nil then
				for _, redirect in ipairs(org.redirect) do
					if redirect ~= "" and redirectIdx[redirect] == nil then
						local redirectPath = "database/redirect/" .. redirect .. ".json"
						redirectIdx[redirect] = util.readUpperLevelPreferenceFile(redirectPath, "no-error no-db use-default")
						if util.tableIsEmpty(redirectIdx[redirect]) then
							util.printError("database '%s', organization '%s', redirect file '%s' was not found", tostring(org.database), tostring(id), tostring(org.connection))
							util.closeProgram()
						elseif redirectIdx[redirect].redirect == nil then
							util.printError("database '%s', organization '%s', redirect file '%s' was not found", tostring(org.database), tostring(id), tostring(org.connection))
							util.closeProgram()
						else
							if redirect == "4d" and util.isLinux() then
								for _, item in ipairs(redirectIdx[redirect].redirect) do
									item.organization_id = peg.replace(item.organization_id, "-4d-", "-fi_demo4d-")
								end
							end
							redirectIdx[redirect] = redirectIdx[redirect].redirect
						end
					end
				end
			end
			if connIdx[id] == nil then
				if connLoadedIdx[org.connection] == nil then
					connLoadedIdx[org.connection] = util.readUpperLevelPreferenceFile(org.connection, "no-error no-db use-default")
				end
				connIdx[id] = connLoadedIdx[org.connection]
			end
			local connRec = connIdx[id]
			if util.tableIsEmpty(connRec) then
				util.printError("database '%s', organization '%s', connIdx file '%s' was not found", tostring(org and org.database), tostring(id), tostring(org and org.connection))
				util.closeProgram()
			end
			org.driver = connRec.driver
			org.dbtype = connRec.dbtype
			loc.organizationPref[id] = org
		end
	end
	connectionPref.allowDatabaseSchemaUpdate = allowDatabaseSchemaUpdate
	connectionPref.connection = connIdx
	connectionPref.redirect = redirectIdx
end

local function traceConnection(conn, action, thread) -- action == "connect" or "disconnect"
	if loc.connectionTraceInUse then
		if useCoro then
			thread = thread or currentThread()
			util.printInfo("%s: name: '%s', %s\n  [%s]", tostring(action), tostring(conn.name), threadId(thread), util.callPath())
		else
			util.printInfo("%s: name: '%s'\n  [%s]", tostring(action), tostring(conn.name), util.callPath())
		end
	end
end

local function tableRedirect(redirectNameArr, organizationId)
	local redirectName = table.concat(redirectNameArr, ", ")
	if loc.tableRedirectData[redirectName] then
		return loc.tableRedirectData[redirectName]
	end
	-- generate combined redirect array anc cache it
	if #redirectNameArr == 1 then
		loc.tableRedirectData[redirectName] = loc.tableRedirectData[redirectName] or {}
	elseif #redirectNameArr > 1 then
		if not util.from4d() then
			util.printInfo("* organization id '%s', combining redirect preferences: '%s'", organizationId, redirectName)
		end
		local prf = util.tableCombine(loc.tableRedirectData[redirectNameArr[1]], loc.tableRedirectData[redirectNameArr[2]], "warning")
		for i = 3, #redirectNameArr do
			util.tableCombine(prf, loc.tableRedirectData[redirectNameArr[i]], "warning", "no-clone")
		end
		loc.tableRedirectData[redirectName] = prf
	end
	return loc.tableRedirectData[redirectName]
end

local function localToOrgId(organizationId)
	local newConn = getOrganizationConnection(organizationId)
	local localConnection
	if newConn == nil then
		localConnection = getLocalConnection()
		if localConnection then
			return localConnection.organization_id
		end
		util.printError("can't set local database, new connection does not exist, current organization '%s', new organization '%s'", getCurrentConnection().organization_id, organizationId)
	elseif loc.organizationPref[organizationId] == nil or not loc.organizationPref[organizationId].is_local then -- newConn.dbtype ~= "postgre" then
		localConnection = getLocalConnection()
		if localConnection == nil then
			localConnection = createLocalConnection(organizationId)
		end
		if localConnection then
			return localConnection.organization_id
		end
		local connId = getCurrentConnection()
		connId = connId and connId.organization_id or "nil"
		util.printError("can't set local database, current organization '%s', new organization '%s'", connId, organizationId)
	else
		return organizationId
	end
end

local function tableRedirectData(redirectNameArr, organizationId)
	if #redirectNameArr == 0 then
		return
	end
	if loc.tableRedirectData then
		return tableRedirect(redirectNameArr, organizationId)
	end
	-- generate all redirects when called the first time
	loc.tableRedirectData = {}
	for redirectId, redirectRecArr in pairs(connectionPref.redirect) do
		loc.tableRedirectData[redirectId] = {}
		for _, redirectRec in ipairs(redirectRecArr) do
			local orgId = redirectRec.organization_id
			if orgId == "local" then
				orgId = localToOrgId(organizationId)
			end
			local dbRec = loc.organizationPref[orgId]
			if dbRec == nil and orgId == "demo-4d-0" then
				for _, item in pairs(loc.organizationPref) do
					if item.database == "4d" and item.driver == "rest4d" then
						loc.organizationPref[orgId] = item -- set loc.organizationPref["demo-4d-0"] to point to local pref
						break
					end
				end
				dbRec = loc.organizationPref[orgId]
			end
			if dbRec == nil then
				util.printError("redirect '%s', organization id '%s' was not found from preference '%s'", redirectId, tostring(orgId), loc.connectionPrefPath)
			else
				local rec = util.clone(redirectRec) -- util.tableCombine(dbRec, redirectRec)
				rec.dbtype = dconn.dbType()
				if type(redirectRec.record_type) == "table" then
					for _, recType in ipairs(redirectRec.record_type) do
						local key = redirectRec.table .. "/" .. recType
						if loc.tableRedirectData[redirectId][key] then
							util.printError("redirect '%s' for table '%s' and record type '%s' already exists", redirectId, redirectRec.table, recType)
						end
						loc.tableRedirectData[redirectId][key] = rec
					end
				else
					local key = redirectRec.table .. "/" .. (redirectRec.record_type or "")
					if loc.tableRedirectData[redirectId][key] then
						util.printError("redirect '%s' for table '%s' and record type '%s' already exists", redirectId, redirectRec.table, redirectRec.record_type or "")
					end
					loc.tableRedirectData[redirectId][key] = rec
				end
			end
		end
	end
	return tableRedirect(redirectNameArr, organizationId)
end

local function redirectRecord(redirectNameArr, tableName, recordType, schema, organizationId)
	local redirect
	if tableName == "currency_rate" then
		tableName = "currency_rate"
	end
	if redirectNameArr then
		local redirectTbl = tableRedirectData(redirectNameArr, organizationId)
		local redirectKey
		if type(recordType) == "table" then
			recordType = recordType[1]
		end
		if recordType == nil then
			util.printWarningWithCallPath("redirect find: record type is nil for table '%s', schema '%s'", tableName, schema)
			recordType = ""
		end
		redirectKey = tableName .. "/" .. recordType
		redirect = redirectTbl and redirectTbl[redirectKey]
		if redirect then
			if redirect.orig_organization_id == "local" then
				if redirect.organization_id ~= organizationId then
					redirect.organization_id = localToOrgId(organizationId)
				end
			elseif redirect.organization_id == "local" then
				redirect.orig_organization_id = redirect.organization_id
				redirect.organization_id = localToOrgId(organizationId)
			end
			if redirect.schema then -- we can't allow redirect if recordType does not match
				schema = redirect.schema or ""
			end
		end
	end
	return redirect, schema
end

local function organizationRedirect(organizationId, tableName, recordType)
	local prf = loc.organizationPref[organizationId]
	if prf == nil then
		util.printError("organization id '%s' was not found from preference '%s'", organizationId, loc.connectionPrefPath)
		return
	end
	return redirectRecord(prf.redirect, tableName, recordType, prf.schema, organizationId)
end

local function isLocalTable(tableOrFldName, schema, recordType)
	if dschema == nil then
		dschema = require "dschema"
	end
	local tableName = dschema.tableName(tableOrFldName, schema, recordType)
	if tableName == nil then
		return false
	end
	if tableName == "currency_rate" then
		tableName = "currency_rate"
	end
	local currentConn = getCurrentConnection()
	if currentConn == nil then
		currentConn = dconn.connection()
	end
	local redirect, redirectSchema
	if currentConn and currentConn.organization_id ~= "" then
		redirect, redirectSchema = organizationRedirect(currentConn.organization_id, tableName, recordType)
	end
	if redirect == nil then
		if currentConn.is_local then
			return true, currentConn
		end
		if redirectSchema ~= "" and redirectSchema == schema then
			return false, currentConn
		end
	end
	local redirectOrgId = redirect and redirect.organization_id or currentConn.organization_id
	local conn = getOrganizationConnection(redirectOrgId)
	-- if useCoro and (conn == nil or not conn.is_local) then
	if conn == nil or not conn.is_local then
		conn = getLocalConnection()
		if conn == nil or not conn.is_local then
			conn = createLocalConnection(redirectOrgId)
		end
		-- redirect = organizationRedirect(authTbl.organization_id, tableName, recordType)
	end
	--[[ if redirect == nil and conn and conn.is_local ~= true then
		authTbl = auth.currentAuthTable()
		redirect = organizationRedirect(authTbl.organization_id, tableName, recordType)
	end ]]
	return conn and conn.is_local or false, conn, redirectOrgId
end
dconn.isLocalTable = isLocalTable

setCurrentConnection = function(conn, reason, thread)
	if useCoro then
		thread = thread or currentThread()
	end
	local currentConn = getCurrentConnection(thread)
	if currentConn == conn then
		return
	end
	if conn then -- and conn.connection -- do not print first change when conn.connection is nil
		-- if conn == nil then
		-- util.printInfo("\n* %s to 'nil'", reason)
		if loc.debugConnectionChange == 1 or loc.debugConnectionChange == 2 then
			loc.debugConnectionChange = loc.debugConnectionChange + 1 -- prevent infinite loop on first call, cause by util.prf()
		elseif loc.debugConnectionChange == 3 then
			loc.debugConnectionChange = 4 -- prevent infinite loop with loc.debugConnectionChange == 3
			loc.debugConnectionChange = util.prf("system/debug.json").debug.connection_change
		end
		if loc.debugConnectionChange then
			if reason == nil then
				reason = ""
			end
			if conn.connection == nil and conn.host and conn.host ~= "" then
				conn.connection = peg.parseAfterLast(conn.host, "/")
			end
			local connName = conn.connection or "error"
			local info2 = conn.database
			if conn.query and conn.query.mainTable then
				info2 = info2 .. ": " .. dschema.recTypeName(conn.query.mainTable, conn.query.recordType)
			end
			if currentConn == nil then
				--[[ if reason == "restore" then
					util.printOk("* connection restored to '%s/%s'", connName, info2)
				else ]]
				if useCoro then
					util.printOk("  connection changed to '%s/%s', %s, %s", connName, info2, threadId(conn.thread), reason)
				else
					util.printOk("  connection changed to '%s/%s', %s", connName, info2, reason)
				end
			elseif currentConn.connection ~= connName or currentConn.database ~= conn.database then
				local info = currentConn.database or ""
				if currentConn.query.mainTable then
					info = info .. ": " .. dschema.recTypeName(currentConn.query.mainTable, currentConn.query.recordType)
				end
				--[[ if reason == "restore" then
					util.printOk("  connection restored from '%s/%s' to '%s/%s'", currentConn.connection, info, connName, info2)
				else ]]
				if useCoro then
					util.printOk("  connection changed from '%s/%s' to '%s/%s', %s, %s", threadId(currentConn.connection), info, connName, info2, threadId(conn.thread), reason)
				else
					util.printOk("  connection changed from '%s/%s' to '%s/%s', %s", tostring(currentConn.connection), info, connName, info2, reason)
				end
			end
		end
	end
	setCurrentConnectionArr(conn, thread)
	if conn == nil then
		return
	end
	setOrganizationConnection(conn.organization_id, conn, thread)
	if conn.is_local then
		setLocalConnection(conn, thread)
		return
	end
	if conn.schema ~= "" and connectionPref.aliasId[conn.schema] == nil then
		loc.aliasId[conn.schema] = conn.organization_id
	end
	local _, conn2, redirectOrgId = isLocalTable("preference", "", "")
	if not (conn2 and conn2.is_local) then
		_, conn2, redirectOrgId = isLocalTable("preference", "", "")
	end
	if conn2 and conn2.is_local then
		local localConnection = getLocalConnection(thread)
		if localConnection == conn2 then
			return
		end
		local info = localConnection.database or ""
		if localConnection.query.mainTable then
			info = info .. ": " .. dschema.recTypeName(localConnection.query.mainTable, localConnection.query.recordType)
		end
		if loc.debugConnectionChange then
			local info2 = conn2.database
			if conn2.query.mainTable then
				info2 = info2 .. ": " .. dschema.recTypeName(conn2.query.mainTable, conn2.query.recordType)
			end
			if useCoro then
				util.printOk("  - local connection changed from '%s/%s' to '%s/%s', %s", localConnection.connection, info, conn2.connection, info2, conn.thread)
			else
				util.printOk("  - local connection changed from '%s/%s' to '%s/%s'", localConnection.connection, info, conn2.connection, info2)
			end
		end
		setLocalConnection(conn2, thread)
		return
	end
	if reason == "connect" then
		if useCoro then
			util.printWarningWithCallPath("setCurrentConnection could not set local connection for '%s' because redirect connection '%s' does not exist yet, %s", conn.organization_id, redirectOrgId or "", conn.thread)
		else
			util.printWarningWithCallPath("setCurrentConnection could not set local connection for '%s' because redirect connection '%s' does not exist yet", conn.organization_id, redirectOrgId or "")
		end
	else
		if useCoro then
			util.printError("setCurrentConnection could not set local connection for '%s', check %s, add \"redirect\": [\"local\"] or other redirect, %s", loc.connectionPrefPath, conn.organization_id, conn.thread)
		else
			util.printError("setCurrentConnection could not set local connection for '%s', check %s, add \"redirect\": [\"local\"] or other redirect", loc.connectionPrefPath, conn.organization_id)
		end
	end
end

local connectionCount = 0
local connectionIdNumber = 0

function dconn.connectionCount()
	return connectionCount
end

local function physicalConnect(driver, connPref, organizationId, useFallback)
	local time = util.seconds()
	local orgPref = loc.organizationPref and loc.organizationPref[organizationId]
	local function dbUser(conn)
		local database = conn.organization_id or conn.database
		local envUser = connectionPref.database and connectionPref.database[database] and connectionPref.database[database].user
		local user = envUser or conn.database_user or connPref.database_user or connectionPref.default_database_user
		if not user or user == "" then
			user = connectionPref.default_database_user
		end
		return user
	end

	local function setConn(prf)
		local conn = util.clone(prf) -- must clone or we will change connPref.connection[i]
		local databaseId = orgPref and orgPref.database or connectionPref.default_id
		conn.database = conn.database or connPref.database or databaseId
		local database = conn.organization_id or conn.database
		conn.driver = connPref.driver
		conn.quote_sql = connPref.quote_sql
		conn.schema = orgPref and orgPref.schema
		conn.is_local = orgPref and orgPref.is_local
		conn.local_group = orgPref and orgPref.local_group
		conn.dbtype = conn.dbtype or connPref.dbtype
		local envPort = connectionPref.database and connectionPref.database[database] and connectionPref.database[database].port
		conn.port = envPort or conn.port or connPref.port
		conn.charset = conn.charset or connPref.charset
		conn.timeout = conn.timeout or connPref.timeout or connectionPref.timeout
		conn.connect_timeout = conn.connect_timeout or connPref.connect_timeout or connectionPref.connect_timeout
		local envPassword = connectionPref.database and connectionPref.database[database] and connectionPref.database[database].password
		conn.password = envPassword or conn.password or connectionPref.default_database_password or ""
		conn.database_user = dbUser(conn)
		conn.organization_id = organizationId
		if conn.host and peg.startsWith(conn.host, "~/nc/") then
			if peg.startsWith(util.mainPath(), "/Volumes/nc/") then
				conn.host = "/Volumes/" .. conn.host:sub(3)
			end
		end
		return conn
	end

	local function dbInfo(conn, info)
		return (info or "") .. conn.name .. ", " .. (conn.host or "") .. (conn.port and conn.port ~= "" and ":" .. conn.port or "") .. ", database: " .. (conn.database or "ERROR - no database") .. ", user: " .. (conn.database_user or "ERROR - no user") -- ..", org: "..conn.organization_number
	end
	local function setFallbackConn(conn)
		if conn.prev_database_user == nil then
			conn.prev_database_user = conn.database_user
		end
		conn.database = connectionPref.fallback_database or conn.fallback_database or connPref.fallback_database or connectionPref.fallback_database
		conn.database_user = connectionPref.fallback_database_user or conn.fallback_database_user or connPref.fallback_database_user or connectionPref.fallback_database_user
		conn.password = connectionPref.fallback_database_password or ""
	end

	local conn, err
	local driverConn, driverErr
	local errTbl = {}
	local os = ffi.os:lower()
	for i = 1, #connPref.connection do -- find working connection and then break
		if not connPref.connection[i].disabled and connPref.connection[i].os == nil or connPref.connection[i].os and connPref.connection[i].os[os] ~= false then
			conn = setConn(connPref.connection[i])
			if useFallback then
				setFallbackConn(conn)
			end
			conn.info = dbInfo(conn, connPref.info)
			if not util.from4d() then
				if not useCoro or loc.debugConnect or loc.debugConnectId[conn.database] == nil then
					util.print("  trying to connect to: '%s'... ", conn.info) -- do not use l() here, it will cause recursion
				end
				-- util.printWarningWithCallPath("  " .. string.format("trying to connect to: '%s'... ", conn.info)) -- do not use l() here, it will cause recursion
			end
			-- util.printTable(conn, "conn")
			if conn.database == nil or conn.database_user == nil or conn.database_user == nil then
				err = "missing database, user, or password"
			else
				if doperation == nil then
					doperation = require "doperation"
				end
				driverConn, err, driverErr = driver.connect(conn)
				if driverConn or err and conn.driver == "postgre" then
					local info
					if useFallback == nil and type(err) == "string" and err:find('database ".+" does not exist') then
						useFallback = true
					end
					if useFallback then
						util.printOk("    connected to fallback database: '%s', %s", conn.info, currentThread())
						err = nil
						setFallbackConn(conn)
						if driverConn == nil then
							driverConn, err = driver.connect(conn)
							if driverConn == nil then
								err = util.printError("    error '%s' in fallback connect, preference: %s, connection: %s, database: %s", tostring(err), conn.organization_id, conn.info, conn.database)
							else
								conn.driver_conn = driverConn
							end
						end
						if err == nil then
							setCurrentConnection(conn, "fallback connect")
							setDriverConnection(conn.organization_id, driverConn) -- set temporarily or will cause infinite loop
							conn.driver_conn = driverConn -- postgre cdata or pgmoon table, needed for createDatabase()
							dsql.clearQuery(conn)
							if conn.prev_database_user and conn.database_user ~= conn.prev_database_user then
								info = doperation.createUser(conn, conn.prev_database_user) -- use fallback conn to check user because new database nay not exist yet
								if info.error then
									util.printError("    " .. string.format("New database user '%s' creation failed, error: '%s'", conn.prev_database_user, info.error))
								elseif not peg.found(info.info, "already exists") then
									util.printOk("    " .. string.format("Created a new database user '%s', result: '%s'", conn.prev_database_user, info.info))
								end
							end
							conn = setConn(connPref.connection[i]) -- set connection before fallback
							info = doperation.createDatabase(conn.database, conn.database_user, conn.organization_id, {no_debug = true})
							util.printOk("    " .. string.format("Created a new database '%s', result: '%s'", conn.database, tostring(info.error or info.info)))
							conn.info = dbInfo(conn, connPref.info)
							print("  " .. string.format("trying to reconnect to: '%s'... ", conn.info))
							disconnect(conn)
							driverConn, err, driverErr = driver.connect(conn) -- doperation.createDatabase() did disconnect, we must connect again
							if driverConn == nil then
								err = util.printError("    error '%s' in connect after database creation, preference: %s, connection: %s, database: %s", tostring(err), conn.organization_id, conn.info, conn.database)
							else
								conn.driver_conn = driverConn
								doperation.checkDatabase(conn, nil, {
									audit_log = conn.schema == "", -- creating audit log to external database fails because
									index = true,
									missing_external_table_warning = true,
									local_table_check = conn.schema == "", -- do not create local tables if schema is set, common tables are created at sync start
									force_check = true,
									backup_database = false
								})
								clearDriverConnection(conn.organization_id) -- driver connection has changed from fallback to real connection
							end
						end
					end
				end
			end
			if err then
				err = "    error in database connect: '" .. err .. "' - preference: " .. conn.organization_id .. ", connection: " .. conn.info .. ", database: " .. (conn.database or conn.dbtype) -- .. ", call path:\n" .. util.callPath()
				errTbl[#errTbl + 1] = err
				if driverErr then
					break
				end
			end
			if driverConn then
				if driver.setReturnRowLimit then
					driver.setReturnRowLimit(loc.queryLimit)
				end
				conn.connected = dt.currentString()
				-- if type(driverConn) ~= "table" then
				conn.driver_conn = driverConn -- postgre cdata or pgmoon table
				-- end
				connectionIdNumber = connectionIdNumber + 1
				connectionCount = connectionCount + 1
				if conn.id then
					util.printError("connection id is already set: %s", tostring(conn.id))
				else
					conn.id = connectionIdNumber
					if type(driverConn) == "table" then
						driverConn.id = conn.id
					end
				end
				conn.info = tostring(conn.id) .. ". " .. conn.info
				if not util.from4d() then
					if not useCoro or loc.debugConnect or loc.debugConnectId[conn.database] == nil then
						util.printOk("    connected to: '%s' %s, connections: %d, driver: %s", conn.info, currentThreadId(), connectionCount, tostring(conn.driver_conn or "")) -- do not use l() here, it will cause recursion
					end
				end
				break
			end
		end
	end
	if conn.call_method == nil or conn.call_method == "" then
		conn.password = ""
	end
	if not driverConn and #errTbl > 0 then
		util.printColor("magenta", table.concat(errTbl, "\n"))
	end
	if not util.from4d() then
		if not useCoro or loc.debugConnect or loc.debugConnectId[conn.database] == nil then
			loc.debugConnectId[conn.database] = {connect = dt.currentString()}
			util.print("    database connect time: %.4f seconds", util.seconds(time))
		end
	end
	traceConnection(conn, "connect")
	if type(driverConn) == "table" and driverConn.option then
		conn.option = util.clone(driverConn.option, {"driver_conn"}) -- skip driver_conn or this will be infinite loop in case of pgmoon
		-- save 4drest socket option to connection, socket will be cleared, must copy or there will be infinite loop
		driverConn.option = nil
	end
	return conn, driverConn
end

local function connect(connPref, organizationId, newConnection)
	local conn
	if rest4d == nil then
		rest4d = require "db/database-rest4d" -- we must load database-rest4d soon because vscode breakpoists do not work on later load
	end
	local driver = connPref.driver
	if driver == "postgre" then
		if not pg then
			pg = require "db/database-postgre"
			loc.driverTbl[driver] = {driver = pg}
		end
	elseif driver == "sqlite" then
		if not sqlite then
			sqlite = require "db/database-sqlite"
			loc.driverTbl[driver] = {driver = sqlite}
		end
	elseif driver == "mysql" then
		if not mysql then
			mysql = require "db/database-mysql"
			loc.driverTbl[driver] = {driver = mysql}
		end
	elseif driver == "odbc" then
		if not odbc then
			odbc = require "db/database-odbc"
			loc.driverTbl[driver] = {driver = odbc}
		end
	elseif driver == "plg4d" then
		if not plg4d then
			plg4d = require "plg4d" -- plg4d must NOT be set anywhere else
			loc.driverTbl[driver] = {driver = plg4d}
		end
	elseif driver == "rest_call" then
		if not callRest then
			callRest = require "call-rest"
			loc.driverTbl[driver] = {driver = callRest}
		end
	elseif driver == "rest4d" then
		if loc.driverTbl[driver] == nil then
			loc.driverTbl[driver] = {driver = rest4d}
		end
	else
		util.printError("Unknown database driver: '%s'", driver)
	end
	-- common for all sql connections
	local driverConn
	if newConnection then
		connPref.info = newConnection .. " "
	end
	if loc.driverTbl[driver] then
		conn, driverConn = physicalConnect(loc.driverTbl[driver].driver, connPref, organizationId)
	end
	local err
	if not (driverConn and loc.driverTbl[driver]) then
		err = util.printWarning("Could not connect to database, preference '%s'", conn and conn.organization_id or organizationId)
		conn, driverConn = physicalConnect(loc.driverTbl[driver].driver, connPref, organizationId, "fallback")
	end
	connPref.info = nil
	if useCoro then
		conn.thread = currentThread()
	end
	if driverConn then
		--[[ if conn.socket == nil and type(driverConn) == "table" and driverConn.socket then
			conn.socket = driverConn.socket
		end ]]
		if newConnection then
			setExtraConnection(conn.thread, conn)
		else
			setDriverConnection(conn.organization_id, driverConn)
			if conn.is_local then
				setLocalConnection(conn, conn.thread)
			end
		end
		return conn
	end
	conn = {error = err}
	return conn
end

disconnect = function(conn, thread)
	if useCoro and thread == nil then
		thread = currentThread()
	end
	traceConnection(conn, "disconnect", thread)
	-- if not util.from4d() or (option and option.force_disconnect) then -- MUST NOT disconnect plg4d ever
	if type(conn) ~= "table" then
		util.printError("disconnect connection is not a table")
		return
	end
	if conn == nil then
		return
	end
	if conn.database == "plg4d" then -- we MUST NOT disconnect plg4d ever
		connectionCount = connectionCount - 1
	else
		if not util.from4d() then
			if useCoro then
				if loc.debugDisconnect or loc.debugConnectId[conn.database] == nil or loc.debugConnectId[conn.database].disconnect == nil then
					loc.debugConnectId[conn.database] = loc.debugConnectId[conn.database] or {connect = dt.currentString()}
					loc.debugConnectId[conn.database].disconnect = dt.currentString()
					util.printInfo("disconnecting from database '%s', driver '%s', %s, connections: %d", tostring(conn and (conn.info or conn.organization_id)), tostring(conn and conn.driver_conn or ""), threadId(conn and conn.thread), connectionCount)
				end
			else
				util.printInfo("disconnecting from database '%s', driver '%s', connections: %d", tostring(conn and (conn.info or conn.organization_id)), tostring(conn and conn.driver_conn or ""), connectionCount)
			end
		end
		setCurrentConnection(nil, "disconnect", thread)
		local driverRec = loc.driverTbl[conn.driver]
		local driver = driverRec and driverRec.driver -- dconn.driver()
		local driverConn = conn.driver_conn or getDriverConnection(conn.organization_id, thread)
		if driver and driverConn then
			if not conn.closed then
				local ret = driver.disconnect(driverConn)
				setDriverConnection(conn.organization_id, nil, thread)
				connectionCount = connectionCount - 1
				if ret and ret ~= 0 then -- ret = 0 from pgmoon (luasocket)
					if useCoro then
						util.printWarning("disconnection warning '%s', database '%s', %s'", tostring(ret), conn.info, threadId(conn.thread))
					else
						util.printWarning("disconnection warning '%s', database '%s'", tostring(ret), conn.info)
					end
				end
			end
		else
			util.printError("disconnection driver was not found, database '%s'", conn.info)
		end
		conn.driver_conn = nil
		conn.connected = nil
		conn.closed = dt.currentString()
	end
end
dconn.disconnect = disconnect

local function updateConnRec(conn) -- action == "connect" or "disconnect"
	-- update auth pref info like redirect and schema to conn
	local dbRec = connectionPref.organizationIdx[conn.organization_id]
	if not dbRec then
		util.printError("connection database '%s' was not found from auth preference", conn.database)
		return
	end
	conn.redirect = dbRec.redirect
	--[[ for key, val in pairs(dbRec) do
		if conn[key] ~= val then
			conn[key] = val
		end
	end ]]
end

function dconn.queryTableParamConvert(tblRec)
	dschema = dschema or require "dschema"
	local ret = {tableRecordType = {}, linkTable = {}}
	if type(tblRec.record_type) == "table" and type(tblRec.record_type[1]) == "table" then
		-- special case when json query table -tag is given as record_type
		tblRec = tblRec.record_type
	elseif type(tblRec[1]) ~= "table" then -- allow table or array of tables
		tblRec = {tblRec}
	end
	for i, rec in ipairs(tblRec) do
		if rec.record_type == nil then
			rec.record_type = {""}
		elseif type(rec.record_type) == "string" then
			rec.record_type = {rec.record_type} -- allow string or array of strings
		end
		if type(rec.table) ~= "string" then
			return nil, l("table -key '%s' is not a string", tostring(rec.table))
		elseif type(rec.record_type) ~= "table" then
			return nil, l("record_type -key '%s' is not an array", tostring(rec.record_type))
		elseif type(rec.record_type[1]) ~= "string" then
			return nil, l("record_type array first key '%s' is not a string", tostring(rec.record_type))
		end
		if rec.linked_table then
			ret.linkedTable = ret.linked_table or {}
			ret.linkedTable[dschema.recTypeName(rec.table, rec.record_type[1])] = rec.linked_table
		end
		if i == 1 then
			ret.table = rec.table
			ret.record_type = rec.record_type[1]
			if not ret.record_type then
				if type(rec.record_type) == "string" or rec.record_type == nil then
					ret.record_type = rec.record_type or ""
				else
					return nil, l("query[%d] record type '%s' is not a table or a string", i, type(rec.record_type))
				end
			end
		end
		if ret.tableRecordType[rec.table] == nil then
			ret.tableRecordType[rec.table] = rec.record_type
		end
		for _, recType in ipairs(rec.record_type) do
			ret.linkTable[#ret.linkTable + 1] = dschema.recTypeName(rec.table, recType)
		end
	end
	if not ret.table then
		return nil, l('there must be at least one record in query table -tag')
	end
	return ret
end

function dconn.setAuthAndRedirect(tableName, recordType, option)
	-- connection()
	-- set currentConn based on current auth
	-- set redirect based on auth redirect, tableName and recordType
	if type(tableName) == "table" then
		local tblRec = dconn.queryTableParamConvert(tableName)
		tableName, recordType = tblRec.table, tblRec.record_type
	end
	local err
	local currentConn = getCurrentConnection()
	local conn = currentConn
	local authTbl = auth.currentAuthTable()
	if conn == nil then
		conn = connection({organizationId = authTbl.organization_id})
		if conn == nil then
			-- should we nill current connection or other system tables?
			return nil, nil, "connection could not be created" -- TODO: return connection error from connection()
		end
		currentConn = getCurrentConnection()
	end
	local organizationId = conn.organization_id
	local redirect, schema
	if tableName and authTbl.organization_id ~= "" then
		redirect, schema = organizationRedirect(authTbl.organization_id, tableName, recordType)
	end
	if redirect then
		schema = redirect.schema or ""
		organizationId = redirect.organization_id
		local currentConnRec = getOrganizationConnection(authTbl.organization_id)
		if currentConnRec and currentConnRec.schema == schema and organizationId ~= authTbl.organization_id then
			organizationId = authTbl.organization_id
		end
		if currentConn and organizationId == currentConn.organization_id then
			conn = currentConn
		elseif currentConnRec and organizationId == currentConnRec.organization_id then
			conn = currentConnRec
			setCurrentConnection(conn, "redirect")
		else
			conn = getOrganizationConnection(organizationId)
			if conn then
				setCurrentConnection(conn, "redirect")
			else
				conn = connection({organizationId = organizationId, reason = "redirect"}) -- sets currentConn
			end
		end
	elseif organizationId ~= authTbl.organization_id then
		organizationId = authTbl.organization_id
		if organizationId == "" then
			organizationId = connectionPref.default_id
		end
		if currentConn and organizationId == currentConn.organization_id then
			conn = currentConn
		else
			local currentConnRec = getOrganizationConnection(organizationId)
			if currentConnRec and organizationId == currentConnRec.organization_id then
				conn = currentConnRec
				setCurrentConnection(conn, "redirect")
			else
				conn, err = connection({organizationId = organizationId, reason = "auth"}) -- sets currentConn
			end
		end
	end
	redirect = {table = tableName, recordType = recordType, redirectId = authTbl.redirect, redirect = redirect, organization_id = organizationId, schema = schema}
	if option and option.new_connection then
		conn = connection({organizationId = organizationId, reason = option.reason or "new connection", new_connection = option.new_connection})
	end
	--[[ if currentConn and currentConn ~= getCurrentConnection() then
		setCurrentConnection(currentConn, "redirect restore")
	end ]]
	if debugLevel > 0 and currentConn.organization_id ~= authTbl.organization_id and (tableName ~= "preference" and tableName ~= "session" and tableName ~= "person") then
		if currentConn.organization_id ~= "demo-fi_demo-0" then
			authTbl = auth.currentAuthTable()
			util.printError("connection organization_id '%s' is not equal to auth organization_id '%s'", conn.organization_id, authTbl.organization_id)
		end
	end
	return conn, redirect, err
end

function dconn.dateFormat()
	if not loc.organizationPref then
		loadPreference()
	end
	connection()
	local connectionRec = connectionPref.connection[getCurrentConnection().connection]
	return connectionRec and connectionRec.date_format or loc.defaultDateFormat
end

function dconn.dateTimeFormat()
	if not loc.organizationPref then
		loadPreference()
	end
	connection()
	local connectionRec = connectionPref.connection[getCurrentConnection().connection]
	return connectionRec and connectionRec.datetime_format or loc.defaultDateTimeFormat
end

function dconn.timeFormat()
	if not loc.organizationPref then
		loadPreference()
	end
	connection()
	local connectionRec = connectionPref.connection[getCurrentConnection().connection]
	return connectionRec and connectionRec.time_format or loc.defaultTimeFormat
end

function dconn.schema()
	connection() -- make default connection if needed, required for authTbl
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.schema or ""
	-- local authTbl = auth.currentAuthTable()
	-- return authTbl.schema -- defaultDatabaseSchema is ""
end

function dconn.recordType()
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.query and currentConn.query.recordType or ""
end

function dconn.defaultDatabaseType()
	if not loc.organizationPref then
		loadPreference()
	end
	local connectionId = loc.organizationPref[connectionPref.default_id].connection
	return connectionPref.connection[connectionId].dbtype
end

function dconn.defaultConnectionId()
	if not connectionPref then
		loadDbConnPreference()
	end
	return connectionPref.default_id
end

function dconn.allConnections()
	return loc.organizationConnection
end

function dconn.databaseName(organizationId)
	if not loc.organizationPref then
		loadPreference()
	end
	local conn = loc.organizationPref[organizationId]
	return conn and conn.database
end

function dconn.setCurrentOrganization(organizationId, prevOrgId, authTbl)
	organizationId = aliasId(organizationId)
	if prevOrgId == nil or authTbl == nil and organizationId ~= "" then -- not called from auth.setCurrentOrganization()
		prevOrgId = auth.setCurrentOrganization(organizationId, true) -- todo: test if we need this call
	else
		local prevAuthTbl = auth.currentAuthTable()
		if prevAuthTbl ~= authTbl then
			auth.setCurrentAuthTable(authTbl)
		end
	end
	local currentConnRec = getOrganizationConnection(organizationId)
	if currentConnRec and not currentConnRec.closed then
		setCurrentConnection(currentConnRec, "set organization")
	else
		connection({organizationId = organizationId})
	end
	return prevOrgId
end

function dconn.restoreConnection(conn, reason) -- first clone connection, then restore it with this
	if conn and conn.organization_id then
		setOrganizationConnection(conn.organization_id, conn, nil, "restore")
		setCurrentConnection(conn, reason or "restore")
		-- we do not need to change authTbl
	end
end

function dconn.checkDatabase(conn, forceCheck, tableNameRecTypeArr)
	-- check database structure
	conn = conn or getCurrentConnection() -- getCurrentConnection() only on first call from nc-server
	if conn == nil then
		if (forceCheck == true or tableNameRecTypeArr) then
			util.printRed("\n***  connection does not exist, there might be another instance of this program or some other program using the port ***  ") -- calling l() causes recursive call here again
			return
		end
		conn = connection() -- create first default connection
	end
	if conn == nil then
		util.printRed("\n***  connection does not exist, there might be another instance of this program or some other program using the port ***  ") -- calling l() causes recursive call here again
		return
	end
	local id = conn.database
	if forceCheck or not doperation.databaseUpdateStatus(id, "check") then -- prevent recursion
		local option = {force_check = forceCheck}
		if conn.schema ~= "" then
			option.missing_external_table_warning = false -- we do not give tableNameRecTypeArr, does not create external tables
		end
		doperation.checkDatabase(conn, tableNameRecTypeArr, option) -- creates default local tables
		-- doperation.databaseUpdateStatus(id, "done")
	end
end

connection = function(parameter) -- creates first default connection if parameter is nil
	if parameter and parameter.createLocal then
		parameter.organizationId = "local" -- will be set later to connectionPref.default_id
	end
	local authTbl = auth.currentAuthTable()
	local currentConn = getCurrentConnection()
	if currentConn and not (parameter and parameter.organizationId) then
		return currentConn
	end
	local newConnection = parameter and parameter.new_connection
	if parameter and parameter.organizationId and not newConnection then
		if not currentConn and parameter.createConnection == false then
			return
		end
		if currentConn and not currentConn.closed and parameter.organizationId == currentConn.organization_id then
			return currentConn
		end
	end
	loadPreference() -- will call loadDbConnPreference() -- must be called only here
	local organizationId, connPref
	if parameter and parameter.organizationId then
		organizationId = parameter.organizationId
	elseif authTbl.organization_id == nil then
		-- create first default connection
		organizationId = connectionPref.default_id
	else
		organizationId = authTbl.organization_id
	end
	if organizationId == "" or organizationId == "local" then
		organizationId = connectionPref.default_id
	end
	organizationId = aliasId(organizationId)
	if authTbl.organization_id == nil then
		auth.setCurrentOrganization(organizationId, "set-id")
	end
	local conn, driverConn
	if not newConnection then
		conn = getOrganizationConnection(organizationId)
		driverConn = conn and (conn.driver_conn or getDriverConnection(conn.organization_id))
	end
	if conn == nil or driverConn == nil or (type(driverConn) == "table" and driverConn.socket == nil and driverConn.connect == nil) then
		connPref = connectionPref.connection and connectionPref.connection[organizationId]
		conn = {}
		if connPref == nil then
			if useCoro then
				conn.thread = currentThread()
			end
			conn.error = l("connection preference '%s' was not found from '%s''", tostring(organizationId), loc.connectionPrefPath)
		else
			if newConnection then
				conn = connect(connPref, organizationId, parameter and parameter.reason or "")
			else
				conn = connect(connPref, organizationId)
			end
			if not conn.error then
				dsql.clearQuery(conn)
			end
		end
	end
	if conn.error then
		util.printRed(conn.error)
		return nil, conn.error
	end
	conn.closed = nil
	currentConn = getCurrentConnection()
	if currentConn and currentConn.closed then
		conn.sql = currentConn.sql
		conn.query = currentConn.query
	end
	updateConnRec(conn)
	if not newConnection then
		setCurrentConnection(conn, parameter and parameter.reason or "connect") -- readonlytable(conn)
	end
	loadPreference() -- must be called only here
	if not authTbl.organization_id then
		auth.setCurrentOrganization(connectionPref.default_id) -- organizationId
	end
	if connectionCount > 1 then
		local prf = loc.organizationPref[organizationId]
		if prf and not prf.database_checked then
			prf.database_checked = dt.currentString()
			conn.database_checked = prf.database_checked
			if conn.dbtype ~= "rest_call" then
				dconn.checkDatabase(conn) -- first connection check here would cause recursive fieldTableStructure creation
			end
		end
	end
	local prevAuthTbl = auth.currentAuthTable()
	if prevAuthTbl ~= authTbl then
		util.printError("currentAuthTable changed")
		auth.setCurrentAuthTable(authTbl)
	end
	return conn
end
dconn.connection = connection

function dconn.allowDatabaseSchemaUpdate(database)
	if not connectionPref then
		loadDbConnPreference()
	end
	return connectionPref.allowDatabaseSchemaUpdate[database]
end

function dconn.tableConnectionDropdown() -- language
	-- todo: use language
	if loc.tableConnectionDropdown == nil then
		loadPreference()
	end
	return loc.tableConnectionDropdown -- fix lang parameter?
end

--[[
function dconn.name(conn)
	if type(conn.name) == "function" then
		return conn.name()
	end
	return conn.name or "-- no connection name --"
end
]]

function dconn.info()
	connection()
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.info or l("invalid connection info")
end

function dconn.database()
	connection()
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.database
end

function dconn.saveJsonAsText()
	return dconn.schema() ~= "" or dconn.dbType() ~= "postgre" -- for example 4d
end

function dconn.sql()
	connection()
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.sql
end

function dconn.query()
	connection()
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.query
end

function dconn.setAuthOrganization(tblName, schema, recordType)
	local conn = connection()
	local prevConnOrgId = conn.organization_id
	local prevOrgId = auth.currentOrganizationId()
	local orgId = prevOrgId
	local isLocal, newConn = isLocalTable(tblName, schema, recordType)
	local currentConn = getCurrentConnection()
	local currentOrgId = currentConn and currentConn.organization_id
	-- if isLocal and schema ~="" then
	if isLocal and conn.is_local then
		orgId = prevConnOrgId
	elseif isLocal and not conn.is_local then
		if newConn == nil then
			util.printError("dconn.setAuthOrganization(), new connection is nil")
		end
		orgId = newConn.organization_id
	elseif newConn and conn ~= newConn then
		orgId = newConn.organization_id
	end
	if orgId ~= currentOrgId then
		dconn.setCurrentOrganization(orgId)
	end
	currentConn = getCurrentConnection()
	return currentConn and currentConn.organization_id, prevConnOrgId -- , prevOrgId
end

function dconn.authOrganizationId()
	connection()
	return auth.currentOrganizationId()
end

function dconn.organizationId()
	connection()
	local currentConn = getCurrentConnection()
	return currentConn and currentConn.organization_id
end

function dconn.organizationIdToSchema(organizationId)
	if organizationId == "" then
		return ""
	end
	local org = connectionPref.organizationIdx[organizationId]
	if org and org.schema then
		return org.schema
	end
	if organizationId ~= "" then
		if connectionPref.organizationNameIdx == nil then
			connectionPref.organizationNameIdx = fn.util.createIndex(connectionPref.organizationArr, "organization_text")
		end
		org = connectionPref.organizationNameIdx[organizationId]
		if org and org.schema then
			return org.schema
		end
	end
	util.printError("organization id '%s' was not found", tostring(organizationId))
	if peg.found(organizationId, "-") then -- fallback to something
		organizationId = peg.parseBetweenWithoutDelimiter(organizationId, "-", "-")
	end
	return organizationId
end

function dconn.defaultConnectionId()
	if not connectionPref then
		loadDbConnPreference()
	end
	return connectionPref.default_id
end

function dconn.validConnection(option)
	local currentConn
	if option and option.organizationId then
		currentConn = getCurrentConnection()
		if currentConn and currentConn.organization_id == option.organizationId then
			return true
		end
	end
	if option == nil or option.option ~= "no-start" then
		currentConn = connection(option) -- create first connection if needed
	end
	return currentConn and currentConn.error == nil or false
end

function dconn.useLowerSlq(conn) -- needs field?
	-- if not (noLowerTable[tableName(fldNum)] or noLowerField[fldName]) then
	-- local orgRec = loc.organizationPref[conn.organization_id]
	conn = conn or getCurrentConnection()
	if conn == nil then
		util.printError("connection is nil")
		return false
	end
	local connRec = connectionPref.connection[conn.organization_id]
	if connRec and connRec.sql_lower == false then
		return false
	end
	return true
	--[[
	local noLowerTable = { -- fix to pref and field, not table
		link = true,
		preference = true,
		session = true,
		user_account = true,
		nomet_machine_data = true,
		nomet_machine_preference = true,
		syntax_error_orders = true,
		orders_from_pps = true,
		mms_storage_view = true,
		mms_order_view = true,
		mms_manufactured_articles = true
	}
	--]]
end

--[[
function dconn.setCurrentDriver(driver)
	connection()
	local currentConn = getCurrentConnection()
	if currentConn then
		currentConn.driver = driver
	end
	util.printError("currentConn was not found, could not set current driver")
end
]]

function dconn.driver(driverName)
	local driverRec = loc.driverTbl[driverName]
	if driverRec == nil then
		connection()
		local currentConn = getCurrentConnection()
		driverRec = loc.driverTbl[driverName or currentConn and currentConn.driver]
	end
	return driverRec and driverRec.driver
end

function dconn.setDriverConnection(conn, organizationId, driverConn)
	connection()
	if organizationId then
		setDriverConnection(organizationId, driverConn)
		return
	end
	if conn then
		setDriverConnection(conn.organization_id, driverConn)
		return
	end
	local currentConn = getCurrentConnection()
	if currentConn and currentConn.organization_id then
		setDriverConnection(currentConn.organization_id, driverConn)
		return
	end
	return util.printError("could not set driver connection, connection was not found, organization id: '%s'", tostring(organizationId))
end

function dconn.driverConnection(conn, organizationId)
	connection()
	if conn and (organizationId == nil or conn.organization_id == organizationId) then
		return conn.driver_conn or getDriverConnection(conn.organization_id)
	end
	if organizationId then
		return getDriverConnection(organizationId)
	end
	local currentConn = getCurrentConnection()
	if currentConn and currentConn.organization_id then
		return currentConn.driver_conn or getDriverConnection(currentConn.organization_id)
	end
	return nil
end

function dconn.disconnectAll(thread, stopApplication)
	--[[ ]]
	--[[ loc.driverConnectionTable = {} -- threads ok
	loc.organizationConnection = {} -- threads ok
	if useCoro then
		loc.currentConnection = {} -- threads ok
		loc.localConnection = {} -- threads ok
	else
		loc.currentConnection = nil
		loc.localConnection = nil
	end ]]
	--[[ if util.tableIsEmpty(loc.organizationConnection) then
		connection()
	end ]]
	-- util.print("disconnect all, database %s", conn.database)
	if useCoro then
		-- special case, accessing loc.organizationConnection directly is ok here
		-- do not clear loc.organizationConnection[thread] while we are looping it with pairs
		if thread then
			local tbl = loc.organizationConnection[thread]
			if tbl then
				for _, conn in pairs(tbl) do
					disconnect(conn, thread)
				end
			end
		else
			for thread2, tbl in pairs(loc.organizationConnection) do
				for _, conn in pairs(tbl) do
					disconnect(conn, thread2)
				end
			end
		end
	else
		for _, conn in pairs(loc.organizationConnection) do
			disconnect(conn)
		end
	end
	local tbl = getExtraConnection(thread)
	if tbl then
		for conn, _ in pairs(tbl) do
			if not conn.closed then
				disconnect(conn, thread)
			end
		end
	end
	setCurrentConnection(nil, "disconnect all")
	clearOrganizationConnection(thread)
	auth.setCurrentOrganization(nil)
	if sqlite and stopApplication then
		sqlite.shutdown()
	end
end

function dconn.dbType()
	connection()
	local currentConn = getCurrentConnection()
	if currentConn then
		if currentConn.error then
			-- util.printWarning("database type was not found, connection has an error: "..conn.error)
			return
		elseif not currentConn.dbtype then
			util.printError("database connection type was not found")
			return
		end
		return currentConn.dbtype
	end
	util.printWarning("database type was not found, connection is nil")
end

function dconn.quoteSql()
	local currentConn = getCurrentConnection()
	if currentConn then
		return currentConn.quote_sql -- nil is falsy value
	end
	util.printWarning("database quote sql was not found, connection is nil")
end

function dconn.database4d() -- dbfix: better name (conn4d?)
	connection()
	if dconn.dbType() == "4d" then
		return true
	end
	return false
end

return dconn

--[[ copy databases:
CREATE DATABASE fi_demo4d WITH -- unix
				TEMPLATE = template0
				OWNER = manage
				ENCODING = 'UTF8'
				LC_COLLATE = 'fi_FI.UTF-8'
				LC_CTYPE = 'fi_FI.UTF-8'
				CONNECTION LIMIT = -1;
CREATE DATABASE fi_demo4d WITH -- windows
				TEMPLATE = template0
				OWNER = manage
				ENCODING = 'UTF8'
				LC_COLLATE = 'Finnish_Finland.1252'
				LC_CTYPE = 'Finnish_Finland.1252'
				CONNECTION LIMIT = -1;
pg_dump -Umanage fi_demo | psql fi_demo4d

CREATE DATABASE fastems WITH
				TEMPLATE = template0
				OWNER = manage
				ENCODING = 'UTF8'
				LC_COLLATE = 'fi_FI.UTF-8'
				LC_CTYPE = 'fi_FI.UTF-8'
				CONNECTION LIMIT = -1;
pg_dump -t nomet_machine_data fi_demo -Umanage | psql fastems
pg_dump -t nomet_machine_preference fi_demo -Umanage | psql fastems
pg_dump -t mms_manufactured_articles fi_demo -Umanage | psql fastems
pg_dump -t mms_order_view fi_demo -Umanage | psql fastems
pg_dump -t mms_storage_view fi_demo -Umanage | psql fastems
pg_dump -t syntax_error_orders fi_demo -Umanage | psql fastems

CREATE DATABASE jyu_ccc WITH
				TEMPLATE = template0
				OWNER = manage
				ENCODING = 'UTF8'
				LC_COLLATE = 'fi_FI.UTF-8'
				LC_CTYPE = 'fi_FI.UTF-8'
				CONNECTION LIMIT = -1;
pg_dump -t jyu_ccc_keyword fi_demo -Umanage | psql jyu_ccc
pg_dump -t jyu_ccc_media fi_demo -Umanage | psql jyu_ccc
--]]
