--- dconv.lua
-- Database (array) conversions.
-- local dconv = require "dconv"
-- @module db
local dconv = {}

local util = require "util"
local l = require"lang".l
local fn = require "fn"
local peg = require "peg"
local recData = require"recdata".get
local recDataSet = require"recdata".set
local json = require "json"
local dschema = require "dschema"
local execute = require "execute"

--[[
local dt = require "dt"
local dconn = require "dconn"
function dconv.localToExternalRec(rec, localPrefix, recordType)
	local newRec = {}
	local newKey
	local schema = dconn.schema()
	for key, val in pairs(rec) do
		newKey = dschema.externalName(key, schema) -- dconv.externalName(key, localPrefix, recordType)
		if newKey == nil then
			newRec[key] = val -- non-field name keys like 'id'
		else
			newRec[newKey] = rec[key]
			if newRec[newKey] and dschema.fieldType(newKey) == "date" then
				newRec[newKey] = dt.toString(dt.hmsAdd(dt.dateParse(newRec[newKey]), 12, 0, 0)) -- add time 12:00:00 to date
			end
		end
	end
	return newRec
end ]]

function dconv.externalToLocalRec(rec, table, fieldName)
	local newTbl = dschema.localNameWithRecType(table)
	if newTbl then
		-- old 4D field names to new structure field names
		local newRec = {}
		local newFld
		if util.isArray(rec) == true then
			fn.iter(rec):each(function(fld)
				newFld = dconv.externalToLocalRec(fld, table)
				if newFld == nil then
					newRec[#newRec + 1] = fld -- non-field name keys like 'id'
				else
					newRec[#newRec + 1] = newFld
				end
			end)
		else
			for fld, val in pairs(rec) do
				newFld = dschema.localNameWithRecType(fld) -- or dconv.externalName(key)
				if newFld == nil then
					newRec[fld] = val -- non-field name keys like 'id'
				else
					newRec[newFld] = val
				end
			end
		end
		if fieldName then
			fieldName = dschema.localNameWithRecType(fieldName) or fieldName -- or dconv.externalName(field_name)
		end
		return newRec, newTbl, fieldName
	end
	return rec, table, fieldName
end

function dconv.arrayTableToRecordTable(sel, info, fldTagName, param)
	if sel == nil then
		return nil, info
	end
	if util.tableIsEmpty(sel) then
		return sel, info
	end
	if fldTagName == nil and param and param.column then
		fldTagName = {}
		local fldTagNameUsed = {}
		local name
		for _, rec in ipairs(param.column) do
			if rec.field then
				name = rec.tag_name or rec.field -- use tag_name name if it exists with field name, otherwise use field name
				if fldTagNameUsed[name] == nil then -- it's ok to have same column many times
					fldTagNameUsed[name] = true
					fldTagName[#fldTagName + 1] = name
				end
			end
		end
		if #fldTagName ~= info.column_count then
			util.printError("parameter column field count is not same as data column count, column:\n  '%s'\nparameter:'%s'", json.toJson(info.column_name), json.toJson(param))
			fldTagName = nil
		end
	end
	local ret = {} -- util.newTable(info.row_count, 0)
	local fields = info.column_name
	local fldName, tagName
	-- local hasSubTable = false
	for col = 1, info.column_count do
		fldName = fields[col]
		if fldTagName then
			tagName = fldTagName[col]
		else
			tagName = fldName
		end
		for row = 1, info.row_count do -- change loop order, this before column
			if not ret[row] then
				ret[row] = {}
			end
			if sel[fldName] == nil then
				if row == 1 then
					util.printWarning(string.format(l("selection[%s] does not exist"), fldName))
					-- break
				end
			elseif sel[fldName][row] ~= nil then
				recDataSet(ret[row], tagName, sel[fldName][row])
				-- ret[row][tagName] = sel[fldName][row]
				--[[ if hasSubTable == false then
						hasSubTable = peg.found(tagName, ".")
					end --]]
				-- else
				-- if row == 1 then
				-- this comes when field value is NULL (ODBC from fastems Oracle)
				-- print(string.format(l"selection[%s][%d] does not exist", fldName, row))
				-- break
				-- end
			end
		end
	end
	if fldTagName then
		info.column_name = fldTagName
	end
	--[[ if hasSubTable and info.row_count > 0 then
		local ret2 = convert.convertFieldData(ret, info.column_name) -- is local field already
		return ret2, info
	end --]]
	return ret, info
end

function dconv.flatRecordArrToStructureArr(arr, mainTablePrefix, aggregateField, rowTableArr)
	local struct = {}
	struct[mainTablePrefix] = {}

	local function findMainRec(rec)
		local recFound --[[ = rec[mainTablePrefix]
		if recFound then
			return recFound
		end ]]
		if aggregateField ~= nil then
			recFound = util.arrayRecord(rec[aggregateField], struct[mainTablePrefix], aggregateField)
		end
		if recFound == nil then
			struct[mainTablePrefix][#struct[mainTablePrefix] + 1] = {}
			recFound = struct[mainTablePrefix][#struct[mainTablePrefix]]
			-- recFound[aggregateField] = rec[aggregateField]
		end
		return recFound
	end
	-- create main tbl
	fn.iter(arr):each(function(rec)
		local mainRec = findMainRec(rec)
		if mainRec then
			for fld, val in pairs(rec) do
				if dschema.tablePrefix(fld) == mainTablePrefix then
					mainRec[fld] = val
				end
			end
		end
	end)
	-- add lower rec
	if type(rowTableArr) == "table" then
		local rowTablePrefixArr = fn.iter(rowTableArr):map(function(tableName)
			return dschema.tablePrefix(tableName)
		end):totable()
		fn.iter(arr):each(function(rec)
			local mainRec = findMainRec(rec)
			if mainRec then
				local rowTbl = {}
				local rowTablePrefixArrNew = {}
				-- check if row table is already in structured format
				for _, tblPrefix in ipairs(rowTablePrefixArr) do
					if rec[tblPrefix] then
						rowTbl[tblPrefix] = rec[tblPrefix]
					else
						rowTablePrefixArrNew[#rowTablePrefixArrNew + 1] = tblPrefix
					end
				end
				--
				for fld, val in pairs(rec) do
					local tblPrefix = dschema.tablePrefix(fld)
					if tblPrefix and fn.index(tblPrefix, rowTablePrefixArrNew) then -- is row fld
						if rowTbl[tblPrefix] == nil then
							rowTbl[tblPrefix] = {}
						end
						rowTbl[tblPrefix][fld] = val
					end
				end
				for tblPrefix, val in pairs(rowTbl) do
					if fn.index(tblPrefix, rowTablePrefixArrNew) then
						if mainRec[tblPrefix] == nil then
							mainRec[tblPrefix] = {}
						end
						mainRec[tblPrefix][#mainRec[tblPrefix] + 1] = val
					elseif type(val[1]) == "table" then
						mainRec[tblPrefix] = val
					else
						mainRec[tblPrefix] = mainRec[tblPrefix] or {}
						mainRec[tblPrefix][#mainRec[tblPrefix] + 1] = val
					end
				end
			end
		end)
	end
	return struct
end

function dconv.combineConversionPreference(conversionArr)
	local ret
	if type(conversionArr) == "table" then
		fn.each(function(convRec)
			if ret == nil then
				ret = convRec
			else
				for tbl, tblRec in pairs(convRec) do -- loop prf table-tags
					if ret[tbl] == nil then -- table-tag not foud from ret-tbl
						ret[tbl] = tblRec
					elseif not (tblRec.array == nil or ret[tbl].array == nil) then
						fn.each(function(rec) -- loop prf array
							local tagFound = false
							for i = #ret[tbl].array, 1, -1 do --  add/replace prf array elements to ret-tbl
								if ret[tbl].array[i].to == rec.to then
									if rec.keep_value == true then
										table.remove(ret[tbl].array, i) -- ret[tbl].array[i].to = ""
									else
										for tag in pairs(rec) do -- copy tags
											ret[tbl].array[i][tag] = rec[tag] -- ret[tbl].array[i].to = rec.to, ret[tbl].array[i].from = rec.from
										end
									end
									tagFound = true
									break
								end
							end
							if tagFound == false and (rec.keep_value == nil or rec.keep_value == false) then
								ret[tbl].array[#ret[tbl].array + 1] = {}
								for tag in pairs(rec) do -- copy tags
									ret[tbl].array[#ret[tbl].array][tag] = rec[tag] -- ret[tbl].array[#ret[tbl].array + 1].to = rec.to ...
								end
							end
						end, tblRec.array)
					end
				end
			end
		end, conversionArr)
	else
		ret = conversionArr
	end
	return ret
end

function dconv.convertTable(rec, prfName, tagName)
	local prf, err
	if type(prfName) == "table" then
		prf = prfName
	else
		prf, err = util.prf(prfName, "get-all no-error") -- field_value.json(cust) + field_value.json(mg)
	end
	if prf and tagName and tagName ~= "" then
		local conversionArr = {}
		if type(tagName) ~= "table" then
			tagName = {tagName}
		end
		fn.each(function(tagRec)
			if prf[tagRec] and not util.tableIsEmpty(prf[tagRec]) then
				conversionArr[#conversionArr + 1] = util.clone(prf[tagRec])
			end
		end, tagName)
		if #conversionArr == 1 then
			prf = conversionArr[1]
		elseif #conversionArr > 1 then
			prf = dconv.combineConversionPreference(conversionArr)
		end
	end
	if prf and err == nil then -- and prf[table]
		local param = {}
		param.convert_definition = {convert = prf}
		param.in_data = util.clone(rec)
		param.option = {}
		param.option.return_table = true
		param.option.create_empty_tag = true
		param.option.create_not_defined_tag = true
		local convert = require "convert/convert"
		local convertRec
		convertRec, err = convert.jsonToJson(param)
		-- convertRec = convertRec[1] or convertRec
		if convertRec ~= nil and util.tableIsEmpty(convertRec) == false and err == nil then
			for fld, val in pairs(convertRec) do
				if type(fld) == "string" and dschema.isField(fld) then
					rec[fld] = val
				end
			end
		end
	end
	return rec
end

function dconv.setDefaultValueRecord(sourceRec, rec, defaultValueRec, override, prf) -- , prf, tableName, schema, recordType)
	if defaultValueRec.static then
		if override then
			util.recToRec(sourceRec, defaultValueRec.static, {replace = override == true, deep = true})
		else
			util.recToRec(rec, defaultValueRec.static, {replace = override == true, deep = true})
		end
	end
	local key = defaultValueRec.copy_value
	if key == nil then
		if defaultValueRec.tag then
			util.printWarning("default value record '%s' has key 'tag', you should use key 'copy_value'", defaultValueRec)
			key = defaultValueRec.tag
		elseif defaultValueRec.static == nil and defaultValueRec.code == nil and defaultValueRec.code_update == nil then
			util.printWarning("default value record '%s' does not have key 'tag', 'copy_value', 'code' or 'code_update'", defaultValueRec)
		end
	end
	if key then
		local warningCount = 0
		for keyTo, keyFrom in pairs(key) do
			local valFrom = recData(sourceRec, keyFrom)
			local valTo = recData(rec, keyTo)
			if valFrom == nil and (valTo == nil or override) then
				if prf == nil or prf.parameter == nil or prf.parameter.default_value_error ~= false then
					warningCount = warningCount + 1
					if warningCount == 1 then
						util.printWarning("default value source record key '%s' does not exist, record '%s'", keyFrom, sourceRec)
					else
						util.printWarning("default value source record key '%s' does not exist", keyFrom)
					end
				end
			elseif (override or valTo == nil) then -- value can be name of some other field
				if valFrom ~= valTo then
					if override then
						recDataSet(sourceRec, keyTo, valFrom)
					else
						recDataSet(rec, keyTo, valFrom)
					end
				end
			end
		end
	end
	local code = defaultValueRec.code
	if prf and prf.do_insert and defaultValueRec.code_insert then
		code = defaultValueRec.code_insert
	end
	if prf and prf.do_update and defaultValueRec.code_update then
		code = defaultValueRec.code_update
	end
	if code then
		-- local saveDataArr = {}
		local err
		if override then
			err = execute.runCode(code, sourceRec, sourceRec, prf)
		else
			err = execute.runCode(code, sourceRec, rec, prf)
		end
		if err then
			util.printError(err)
			return err
			-- else
			-- util.recToRec(rec, saveDataArr)
		end
	end
end

local newRecCache = {}
local syncRecCache = {}
function dconv.newRecord(tableName, option, schema, recordType, addToRec, prf, setFunction) -- option = "no-cache all-fields"
	local err
	schema = schema or ""
	recordType = recordType or "" -- can't be nil, used as table index
	if newRecCache[schema] == nil then
		newRecCache[schema] = {}
		syncRecCache[schema] = {}
	end
	if newRecCache[schema][recordType] == nil then
		newRecCache[schema][recordType] = {}
		syncRecCache[schema][recordType] = {}
	end
	tableName = dschema.tableName(tableName)
	if newRecCache[schema][recordType][tableName] == nil or option and peg.found(option, "no-cache") then
		local newRecValue = {}
		local syncRecValue = {}
		local allFields = option and peg.found(option, "all-fields") or false
		-- use dschema.fieldArray() nad not dschema.localFieldArray() because this is called twice, first with external schema and recordType and then another call  with local schema to overwrite external values
		local fieldArr = dschema.fieldArray(tableName, schema, recordType)
		local val, syncVal
		local added = allFields
		for _, fld in ipairs(fieldArr) do
			val, syncVal = dschema.fieldDefaultValue(fld, schema, recordType, setFunction)
			if val ~= nil then
				added = true
				newRecValue[dschema.fieldName(fld)] = val
			elseif allFields then
				newRecValue[fld] = dschema.fieldDefaultTypeValue(fld)
			end
			if syncVal ~= nil then
				added = true
				syncRecValue[dschema.fieldName(fld)] = syncVal
			end
		end
		if added then
			newRecCache[schema][recordType][tableName] = newRecValue
			syncRecCache[schema][recordType][tableName] = syncRecValue
		else
			newRecCache[schema][recordType][tableName] = false
		end
	end
	local defaultRec
	if (prf and prf.parameter and prf.parameter.sync) then
		defaultRec = syncRecCache[schema][recordType][tableName]
	else
		defaultRec = newRecCache[schema][recordType][tableName]
	end
	if addToRec then
		if prf and prf.parameter and prf.parameter.sync then
			-- is sync record, replace with table json sync_value
			if defaultRec then
				util.recToRec(addToRec, defaultRec, {replace = true})
			end
		else
			-- is new record, not sync record
			local defaultValueRec = dschema.defaultValueRecord(tableName, schema, recordType)
			if defaultValueRec then
				err = dconv.setDefaultValueRecord(addToRec, addToRec, defaultValueRec, false, prf) -- override = false
			end
			if defaultRec then
				util.recToRec(addToRec, defaultRec, {replace = false})
			end
		end
		return nil, err
	end
	return defaultRec and util.clone(newRecCache[schema][recordType][tableName]) or false
end

--[[ default_value/local should be in table
function dconv.duplicateRecord(tableName, rec)
	local duplicated = util.clone(rec)
	local recTbl = {}
	recTbl[tableName] = rec
	duplicated = dconv.convertTable(rec, "default_value/local/field_value_"..tostring(tableName)..".json", {"default", "duplicate"}) -- , "default_value/duplicate_record.json")
	return duplicated[tableName]
end

function dconv.duplicateRecordStructure(tableName, rec)
	local duplicated = util.clone(rec)
	duplicated = dconv.convertTable(rec, "default_value/local/field_value_"..tostring(tableName)..".json", {"default", "duplicate"}) -- "default_value/duplicate_record.json")
	return duplicated
end
]]

function dconv.recordArraysToStructure(selection)
	--[[
	{
	table: "mainTbl"
	mainTbl:[{},{}]
	subTbl1:[{},{}]
	subTbl2:[{},{}]
	...
	}
	]]
	local retTbl = {}
	local linkedSel = {}
	local info
	if selection.table == nil then
		info = {error = l("main table prefix is not defined")}
		return nil, info
	elseif selection[selection.table] == nil then
		info = {error = l("main '%s' table must be included in selection", selection.table)}
		return nil, info
	elseif type(selection[selection.table]) ~= "table" then
		info = {error = l("main '%s' table type is wrong", selection.table)}
		return nil, info
	elseif #selection[selection.table] <= 0 then
		info = {error = l("main '%s' table is empty", selection.table)}
		return nil, info
	end

	-- generate field number arrays
	local fldNumArr = {}

	-- add main tbl to first element
	fldNumArr[#fldNumArr + 1] = {}
	local fldNumRow = fldNumArr[#fldNumArr]
	for fld, _ in pairs(selection[selection.table][1]) do
		fldNumRow[#fldNumRow + 1] = fld
	end
	for prefix, tblArr in pairs(selection) do
		if tblArr and tblArr[1] ~= nil and prefix ~= selection.table then
			fldNumArr[#fldNumArr + 1] = {}
			local fldNumRow2 = fldNumArr[#fldNumArr]
			for fld, _ in pairs(tblArr[1]) do
				fldNumRow2[#fldNumRow2 + 1] = fld
			end
		end
	end
	--
	local mainTblNum = selection.table

	local function insertLinkField(fieldArray, fieldToAdd)
		local insertLinkField2 = true
		for _, fld in ipairs(fieldArray) do
			if fld == fieldToAdd then
				insertLinkField2 = false
			end
		end
		if insertLinkField2 == true then
			fieldArray[#fieldArray + 1] = fieldToAdd
		end
	end

	local mainFieldArray = fldNumArr[1]
	if mainFieldArray and #mainFieldArray > 0 then
		-- insert missing link fields
		if #fldNumArr > 1 then
			for i = 2, #fldNumArr do
				if #fldNumArr[i] > 0 then
					local manyTblNum = dschema.tablePrefix(fldNumArr[i][1])
					local manyFldNum, mainFldNum = dschema.relationFields(manyTblNum, mainTblNum)
					if manyFldNum == nil or mainFldNum == nil then
						info = {error = l("conversion relations do not exist between tables '%s' and '%s'", dschema.tableName(mainTblNum), dschema.tableName(manyTblNum))}
					else
						insertLinkField(mainFieldArray, mainFldNum)
						insertLinkField(fldNumArr[i], manyFldNum)
						linkedSel[#linkedSel + 1] = {}
						linkedSel[#linkedSel].fieldArray = fldNumArr[i]
						linkedSel[#linkedSel].many_table = dschema.tablePrefix(manyTblNum)
						linkedSel[#linkedSel].main_field = mainFldNum
						linkedSel[#linkedSel].main_table = selection.table
						if selection ~= nil and selection.special_link_field ~= nil and selection.special_link_field[linkedSel[#linkedSel].many_table] ~= nil then
							-- set special link field
							linkedSel[#linkedSel].many_field = selection.special_link_field[linkedSel[#linkedSel].many_table]
						else
							linkedSel[#linkedSel].many_field = manyFldNum
						end
					end
				end
			end
		end

		-- add tables to sel-arr
		local ret = selection[selection.table]
		if ret and #ret > 0 then
			for i, rec in ipairs(linkedSel) do
				linkedSel[i][1] = {}
				linkedSel[i][1] = selection[rec.many_table]
			end

			-- join records to main array
			for _, tblRec in ipairs(linkedSel) do
				if tblRec[1] then
					for _, rec in ipairs(tblRec[1]) do
						for j, mainRec in ipairs(ret) do -- find right main record
							if mainRec[tblRec.main_field] == rec[tblRec.many_field] then
								if ret[j][tblRec.many_table] == nil then
									ret[j][tblRec.many_table] = {}
								end
								local recLength = #ret[j][tblRec.many_table] + 1
								ret[j][tblRec.many_table][recLength] = rec
							end
						end
					end
				end
			end
		end
		retTbl[selection.table] = ret
	end
	return retTbl, info
end

function dconv.convertTextToObject(text, delimiter)
	-- "asd.qwe.zxc" -> {asd = {qwe = {zxc = {}}}}
	if type(text) == "string" and peg.found(text, delimiter) then
		local ret = {}
		local prev
		peg.iterateSeparators(text, delimiter, function(tag)
			if prev then
				prev[tag] = {}
				prev = prev[tag]
			else
				ret[tag] = {}
				prev = ret[tag]
			end
		end)
		return ret
	end
	return text
end

function dconv.setDefaultValuesToSaveRec(rec, prf, tableName, schema, recordType)
	-- set default values to save rec
	local err
	local defaultTbl = prf.default_value and prf.default_value[tableName] -- first default values from save preference
	if defaultTbl then
		if type(rec.json_data) == "string" then
			rec.json_data = json.fromJson(rec.json_data)
		end
		local tablePrefix = dschema.tablePrefix(tableName)
		local targetRec = rec[tablePrefix] or rec
		local sourceRec = prf.main_record and prf.main_record[tablePrefix] or targetRec
		local override = sourceRec[tablePrefix] == nil -- sql upsert needs to have override false
		err = dconv.setDefaultValueRecord(sourceRec, targetRec, defaultTbl, override, prf)
	end
	if schema ~= "" then
		dconv.newRecord(tableName, nil, schema, recordType, rec, prf) -- next default values from external schema
	end
	local addFunctionValue = false
	if schema == "" then
		addFunctionValue = true -- todo: better check here?
	end
	dconv.newRecord(tableName, nil, nil, nil, rec, prf, addFunctionValue) -- last default values from default schema, but no default_function

	if prf.table and prf.table[1] then
		if type(prf.table[1].row_table) == "string" then
			prf.table[1].row_table = {table = prf.table[1].row_table}
		end
		if type(prf.table[1].row_table) == "table" then
			for _, rowInfoRec in ipairs(prf.table[1].row_table) do
				local rowTablePrefix = dschema.tablePrefix(rowInfoRec.table)
				if rec[rowTablePrefix] then
					rowInfoRec.default_value = prf.default_value -- copy default values from main rec
					for _, rowRec in ipairs(rec[rowTablePrefix]) do
						-- recursive call, row_table -tag can be recursively like prf.table
						dconv.setDefaultValuesToSaveRec(rowRec, rowInfoRec, rowInfoRec.table, rowInfoRec.schema or schema, rowInfoRec.record_type or recordType)
					end
				end
			end
		end
	end
	return err
end

return dconv
