--- dload.lua
-- (db_selection.lua)
-- Database load operations.
-- local dload = require "dload"
-- @module db
local dload = {}

local util = require "util"
local l = require"lang".l
local peg = require "peg"
local dprf = require "dprf"
local fn = require "fn"
local recData = require"recdata".get
local recDataSet = require"recdata".set
local dschema = require "dschema"
local dsql = require "dsql"
local dqry = require "dqry"
local dconn = require "dconn"
local rest4d, plg4d

local loc = {}
loc.tableNameArray = nil
loc.tableNameIdx = nil
loc.queryTimeWarning = 2.5
loc.rowCountWarning = 20000
loc.bigQueryWarning = true

function dload.setBigQueryWarning(value)
	local prevValue = loc.bigQueryWarning
	loc.bigQueryWarning = value
	return prevValue
end

function dload.recordsInSelection(tbl)
	dconn.query() -- there must be some query initialized when calling recordsInSelection()
	local ret, info = dload.selectionToRecordArray({tbl}, "COUNT(*)") -- sql function COUNT(*) will not clean query, all others do
	local count = ret and ret[1] and tonumber(recData(ret[1], tbl))
	return count, info
end

function dload.recordsInTable(tbl, schema, option)
	local connQuery = dconn.query()
	local driver = dconn.driver()
	if not connQuery or not driver then
		return -1
	end
	local tblName = dschema.tableNameSql(tbl, schema)
	if tblName == nil then
		if option == "no-error" then
			tblName = tbl
		else
			local err = l("table '%s' does not exist , schema '%s', record type '%s'", tostring(tbl), tostring(connQuery and connQuery.schema), tostring(connQuery and connQuery.recordType))
			util.printWarning(err)
			return nil, {error = err}
		end
	end
	local queryText = connQuery.queryText
	connQuery.queryText = "SELECT COUNT(*) FROM " .. tblName .. " WHERE 1=1"
	local cursor, errorText = dsql.sqlQueryExecute({})
	connQuery.queryText = queryText
	if not cursor or errorText then
		-- free cursor?
		return -1
	end
	local ret, info = driver.selectionToRecordArray(cursor, {"COUNT(*)"}, {table = tblName})
	-- dconn.setPrevConnection()
	if not ret or not ret[1] then
		return nil, info
	end
	return tonumber(ret[1]["COUNT(*)"])
end

function dload.linkedRecord(field, fieldValue, linkedFieldArr)
	local linkedFld = dschema.linkedField(field)
	dqry.query("", linkedFld, "=", fieldValue)
	local sel, info = dload.selectionToRecordArray(linkedFieldArr)
	if sel and #sel > 0 then
		sel = sel[1]
	else
		sel = nil
	end
	return sel, info
end

function dload.setLinkedRecord(r, queryName, queryParameter)
	local qry = require "qry"
	local tbl, err, tblOrigPrefix, tblPrefix, tagName
	-- local externalToLocal = false
	local saveRec = r
	local option = queryParameter.option
	if option then
		-- externalToLocal = option.external_to_local
		saveRec = option.rec or r
		tagName = option.tag_name
		if tagName == nil and option.link_field ~= nil then
			tblPrefix = dschema.tablePrefix(option.link_field)
			if tblPrefix == nil then
				return l("link field '%s' table prefix is incorrect", option.link_field)
			end
			tagName = "_link_" .. tblPrefix
		end
		if option.rec_field and option.link_field then
			if saveRec[tagName] and saveRec[tagName][option.link_field] and saveRec[tagName][option.link_field] == saveRec[option.rec_field] then
				return -- linked record already in rec
			end
		end
	end
	tbl, err, tblOrigPrefix = qry.queryJson(queryName, queryParameter)
	if tblOrigPrefix == nil then
		return l("query table is incorrect, query '%s'", queryName)
	end
	tblPrefix = tblOrigPrefix
	--[[
	if externalToLocal == true then
		tblPrefix = dconv.prefixToLocalPrefix(tblOrigPrefix)
	else
		tblPrefix = tblOrigPrefix
	end
	--]]
	if tbl == nil then
		return l("linked table '%s' record was not found, query '%s'", tblPrefix, queryName)
	elseif tbl.error then
		return tbl.error
	elseif err then
		return err
	elseif tbl.data == nil or tbl.data[1] == nil then
		return l("linked table '%s' record was not found, query '%s'", tblPrefix, queryName)
	end
	tagName = tagName or ("_link_" .. tblPrefix)
	saveRec[tagName] = tbl.data[1]
	--[[
	if externalToLocal then
		saveRec[tagName] = dconv.externalToLocalRec(saveRec[tagName], tblOrigPrefix)
	end
	--]]
	if option and option.rec_field and option.link_field then
		if saveRec[tagName][option.link_field] and saveRec[tagName][option.link_field] ~= saveRec[option.rec_field] then
			saveRec[option.rec_field] = saveRec[tagName][option.link_field]
		end
	end
end

local function selectionToTable(returnTableType, field, sqlFunction, option)
	local ret, info
	local time = util.seconds()
	if field == nil then
		local err = l("table %s does not exist", tostring(field))
		util.printError(err)
		return nil, {error = err}
	end
	local connSql = dconn.sql()
	local connQuery = dconn.query()
	if option then
		dsql.sqlSetOption(option)
	else
		option = {}
	end
	local fieldUnique = {}
	if connQuery == nil then
		return {error = util.printError("database connection failed")}
	end
	local queryTable = connQuery.table
	local tableRecordType = connQuery.tableRecordType
	option.table = queryTable
	option.record_type = connQuery.recordType
	option.table_prefix = dschema.localTablePrefix(queryTable, connQuery.schema, connQuery.recordType) -- todo: use prefix_recType and combine product + work to same
	if option.table_prefix == nil then
		util.printError("localTablePrefix was not found for table '%s'", tostring(queryTable))
	end
	if type(field) == "string" then
		fieldUnique = dschema.fieldArray(field)
	else
		-- sql clause contains only unique field names, our tags in call to driver.selectionToArrayTable() must also be unique
		local fldExists = {}
		for _, fldName in ipairs(field) do
			if not fldExists[fldName] then
				fldExists[fldName] = true
				fieldUnique[#fieldUnique + 1] = fldName
			end
		end
	end
	local fieldSortParam = connSql.fieldSortParam
	local err = dsql.sqlQueryBuild(fieldUnique, sqlFunction) -- clears conn's sql and query
	if err then
		return nil, {error = l("sql query build failed: %s", err)}
	end
	if connQuery.schema == "4d" then
		local fieldType = {}
		for i, fldName in ipairs(fieldUnique) do
			if sqlFunction == "COUNT(*)" then
				fieldType[i] = "integer"
			else
				fieldType[i] = dschema.fieldTypeBasic(fldName, connQuery.schema, connQuery.recordType)
			end
		end
		option.field_type = fieldType
	end
	if util.from4d() then -- special case inside 4D, also rest call works but this is hugely faster
		if plg4d == nil then
			plg4d = require "plg4d"
		end
		local fldName4d = {}
		local recType = connQuery.recordType
		for i, name in ipairs(fieldUnique) do
			fldName4d[i] = dschema.externalNameSql(name, "4d", recType)
		end
		option.local_field = fieldUnique
		fieldUnique = fldName4d
		-- util.printTable(fieldUnique, "plg4d returnTableType fieldUnique")
		if returnTableType == "record array" then
			ret, info = plg4d.selectionToRecordArray(connQuery.queryText, fieldUnique, option, returnTableType)
		elseif returnTableType == "array table" then
			util.printError("query '%s', do not use deprecated 'array table', use new 'record array'", connQuery.queryText)
			-- print("plg4d.selectionToArrayTable() returnTableType " .. tostring(returnTableType))
			ret, info = plg4d.selectionToArrayTable(connQuery.queryText, fieldUnique, option, returnTableType)
		elseif returnTableType == "record table" then
			util.printError("query '%s', do not use deprecated 'record table', use new 'record array'", connQuery.queryText)
			ret, info = plg4d.selectionToRecordTable(connQuery.queryText, fieldUnique, option, returnTableType)
		else
			err = l("wrong return table type: '%s'", tostring(returnTableType))
			util.printError(err)
			return nil, {error = err}
		end
		-- util.printTable(ret, "ret plg4d " .. returnTableType)
	else
		local driver = dconn.driver() -- must set driver AFTER dsql.sqlQueryBuild() - it sets option.table_redirect
		if not driver then
			return nil, {error = l("driver is nil")}
		end
		local conn = dconn.currentConnection()
		local driverFunction
		if conn.driver == "rest4d" then
			if rest4d == nil then
				rest4d = require "db/database-rest4d" -- we need this to enable breakpoints in database-rest4d
				driver = rest4d
			end
		end
		if returnTableType == "record array" then
			driverFunction = driver.selectionToRecordArray
			if driverFunction == nil then
				util.printRed("error: driver '%s' has no selectionToRecordArray() defined, using driver.selectionToRecordTable()", conn.driver)
				driverFunction = driver.selectionToRecordTable
			end
		elseif returnTableType == "array table" then
			driverFunction = driver.selectionToArrayTable
		elseif returnTableType == "record table" then
			util.printError("driver '%s', do not use deprecated 'record table', use new 'record array'", conn.driver)
			driverFunction = driver.selectionToRecordTable
		else
			err = l("wrong return table type: '%s'", tostring(returnTableType))
			util.printError(err)
			return nil, {error = err}
		end
		local cursor
		cursor, err = dsql.sqlQueryExecute(option)
		if not cursor then
			if #err < 1000 then
				return nil, {error = l("sql execute failed: '%s', query '%s'", err, connQuery.queryNameFunc())} -- peg.parse(err, "\n", 1))}
			else
				return nil, {error = l("sql execute failed: '%s', query '%s'", peg.parse(err, "\n", 1), connQuery.queryNameFunc())} -- should retun first n lines
			end
		end

		if connSql.containsEmptyArray and not connSql.containsOr then -- if cursor == "containsEmptyArray" then
			ret = {}
			info = {}
			info.column_name = {}
			info.row_count = 0
			info.row_count_total = 0
			info.column_count = 0
			info.info = l("query contains empty array, sql query has been skipped, query: '%s'", connQuery.queryNameFunc())
		elseif conn.driver == "odbc" then -- rest4d
			ret, info = driverFunction(cursor, fieldUnique, option) -- debug only odbc or rest4d -calls
		else
			option.fieldSortParam = fieldSortParam -- rest4d need to know order by fields
			ret, info = driverFunction(cursor, fieldUnique, option)
			option.fieldSortParam = nil
		end
		if type(info) == "table" then
			info.organization_id = conn.organization_id
			info.database = conn.database
		end
		if not ret then
			if type(info) == "table" then
				info.queryText = connQuery.queryText
			end
			return nil, info
		end
		if returnTableType == "record array" then
			info.table = {table = queryTable, table_prefix = option.table_prefix, table_record_type = tableRecordType}
		end
	end

	local function mapToLocal()
		-- value = dschema.fieldSqlMapValue(fld, schema, rec.record_type, value)
		local prefix, prefixLen
		for _, fieldName in ipairs(fieldUnique) do
			if dschema.fieldHasMapValue(fieldName, connQuery.schema, connQuery.recordType) then
				if prefix == nil then
					prefix = info.table.table_prefix .. "."
					prefixLen = #prefix
				end
				local val, val2
				for _, rec in ipairs(ret) do
					val = recData(rec, fieldName)
					val2 = dschema.fieldLocalMapValue(fieldName, connQuery.schema, connQuery.recordType, val)
					if val2 ~= val then
						if fieldName:sub(1, prefixLen) == prefix then
							recDataSet(rec, fieldName:sub(prefixLen + 1), val2)
						else
							recDataSet(rec, fieldName, val2)
						end
					end
				end
			end
		end
	end
	if connQuery.schema ~= "" then
		mapToLocal()
	end

	time = util.seconds(time)
	info.query_time = time -- util.round(time / 1000, 6) -- milliseconds
	if info.row_count == nil then -- todo: !!!! fix this
		info.row_count = 0
	end
	if dsql.showSqlOn() then
		if connSql.containsEmptyArray and not connSql.containsOr then
			util.printOk(l("'%s' query contains empty array (with no OR clause) and was skipped,'%s' result load time: %f seconds, rows: %d\n", connQuery.queryNameFunc(), dconn.info(), info.query_time, info.row_count))
		else
			util.printOk(l("'%s', '%s' result load time: %f seconds, rows: %d\n", connQuery.queryNameFunc(), dconn.info(), info.query_time, info.row_count))
		end
	elseif loc.bigQueryWarning and (info.query_time > loc.queryTimeWarning or info.row_count > loc.rowCountWarning) then
		if option == nil or option.big_query_warning ~= false then
			util.printWarning(l("big sql query: '%s', '%s', %s\nresult load time: %f seconds, rows: %d\n", connQuery.queryNameFunc(), dconn.info(), connQuery.queryText:sub(1, 2000), info.query_time, info.row_count))
		end
	end
	return ret, info, field
end

-- old

function dload.selectionToRecordArray(field, func, option)
	return selectionToTable("record array", field, func, option)
end

function dload.selectionToArrayTable(field, func, option)
	return selectionToTable("array table", field, func, option)
end

function dload.selectionToRecordTable(field, func, option)
	util.printError("do not use deprecated 'record table', use new 'record array'")
	local sel, info = selectionToTable("record table", field, func, option)
	return sel, info
end

--[[
function dload.selectionToIdTable(field, func, option)
	local ret
	local sel, info = dload.selectionToRecordTable(field, func, option)
	if sel and #sel > 0 then
		ret = util.newTable(0, #sel)
		for _, rec in ipairs(sel) do
			local id = rec[ field[1] ]
			if id and not ret[id] then
				ret[id] = {}
			end
			ret[id][#ret[id] + 1] = rec
		end
	end
	return ret, info
end
]]

--[[
local function setFieldArr(tblFieldArr)
	local tblPrefix, fieldArr
	if type(tblFieldArr) == "string" then
		tblPrefix = dschema.tablePrefix(tblFieldArr)
		fieldArr = dschema.fieldArray(tblPrefix)
		-- elseif type(tblFieldArr) == "table" and type(tblFieldArr[tblPrefix]) == "table" then
		-- fieldArr = tblFieldArr[tblPrefix]
	elseif type(tblFieldArr) == "table" then
		tblPrefix = dschema.tablePrefix(tblFieldArr[1])
		fieldArr = util.clone(tblFieldArr)
		--[=[
		tblPrefix = next(tblFieldArr)
		if tblPrefix == nil then
			return nil
		end
		tblPrefix = dschema.tablePrefix(tblPrefix)
		]=]
	end
	-- fieldArr = dsql.changeJsonFieldArrName(fieldArr)
	return fieldArr, tblPrefix
end

local function addMissingFieldToFieldArr(tblPrefix, fldArr, field)
	if field and type(fldArr) == "table" then
		fldArr = fldArr or {}
		if fn.index(field, fldArr) == nil then
			fldArr[#fldArr + 1] = field -- add field to fieldArr if it is missing
		end
	end
	local uuidFld = dschema.uuidField(tblPrefix)
	if uuidFld then
		if fn.index(uuidFld, fldArr) == nil then
			fldArr[#fldArr + 1] = uuidFld -- add record_id field
		end
	end
	local primaryFld = dschema.primaryField(tblPrefix) -- drel2.primaryField(tblPrefix)
	if primaryFld then
		if fn.index(primaryFld, fldArr) == nil then
			fldArr[#fldArr + 1] = primaryFld -- add primary field
		end
	end
end

local function linkingFieldValueArr(tblPrefix, fieldArr, dataTbl)
	-- {linked_field = rec.linked_field, linking_field = rec.linking_field, linking_table = rec.linking_table}
	local linkedFieldTbl
	local linkingArr = dschema.tableLinkingFieldArr(tblPrefix) -- drel2.tableLinkingFieldArr(tblPrefix)
	if linkingArr then
		fn.iter(linkingArr):each(function(rec)
			if dataTbl[rec.linking_table] then
				local valueArr = util.fieldValueToArray(dataTbl[rec.linking_table], rec.linking_field)
				if valueArr and #valueArr > 0 then
					linkedFieldTbl = linkedFieldTbl or {}
					if linkedFieldTbl[rec.linked_field] == nil then
						linkedFieldTbl[rec.linked_field] = valueArr
					else
						linkedFieldTbl[rec.linked_field] = util.arrayCombine({linkedFieldTbl[rec.linked_field], valueArr})
					end
					addMissingFieldToFieldArr(tblPrefix, fieldArr, rec.linked_field)
				end
			end
		end)
	end
	return linkedFieldTbl
end
]]

function dload.linkedToRecordTable(mainRec, linkedTbl)
	local mainTbl, manyTbl, err
	local firstField = next(mainRec)
	mainTbl = dschema.tablePrefix(firstField)
	if mainTbl == nil then
		err = l("relations main table do not exist")
		util.printError(err)
		return nil, err
	end
	manyTbl = linkedTbl
	if manyTbl == nil then
		err = l("relations many table do not exist between tables '%s' and '%s'", dschema.tableName(mainTbl), linkedTbl)
		util.printError(err)
		return nil, err
	end
	local manyFld, mainFld = dschema.relationFields(manyTbl, mainTbl)
	if manyFld == nil or mainFld == nil then
		err = l("relations do not exist between tables '%s' and '%s'", dschema.tableName(mainTbl), dschema.tableName(manyTbl))
		util.printError(err)
		return nil, err
	end
	dqry.query("", manyFld, "=", mainRec[mainFld])
	local sel, info = dload.selectionToRecordTable(manyTbl)
	return sel, info
end

--[=[
function dload.addLinkedDataToStructureTable(structTable, mainLinkField, oneLinkField, fieldArray)
	if not structTable or not mainLinkField or not oneLinkField or not fieldArray then
		return
	end
	local oneQueryArr = {}
	local counter = 0
	local fieldName = mainLinkField
	local tblName = dschema.tablePrefix(mainLinkField)
	for _, rec in ipairs(structTable[tblName]) do
		if rec[fieldName] and not oneQueryArr[rec[fieldName]] then
			counter = counter + 1
			oneQueryArr[rec[fieldName]] = counter
		end
	end
	oneQueryArr = util.invertTable(oneQueryArr)

	if fieldArray[1] ~= oneLinkField then -- order fieldList so that one field is first element
		local fieldArrayCheck = {oneLinkField}
		for _, fld in ipairs(fieldArray) do
			if fld ~= oneLinkField then
				fieldArrayCheck[#fieldArrayCheck + 1] = fld
			end
		end
		fieldArray = fieldArrayCheck
	end

	dqry.query("", oneLinkField, "in", oneQueryArr)
	local oneSel = dload.selectionToIdTable(fieldArray)
	if oneSel then
		for i, rec in ipairs(structTable[tblName]) do
			local oneRec = oneSel[rec[fieldName]][1]
			for _, fld in ipairs(fieldArray) do
				if oneRec[fld] then
					if structTable[tblName][i][fld] then
						structTable[tblName][i][fld.."."..fieldName] = oneRec[fld]
					else
						structTable[tblName][i][fld] = oneRec[fld]
					end
				end
			end
		end
	end
end
]=]

function dload.clearTableNameArray()
	loc.tableNameArray = nil
	loc.tableNameIdx = nil
end

local function fieldNameArray(tblNameRecType, schema)
	local orgId, prevOrgId = dconn.setAuthOrganization(tblNameRecType, schema, "")
	local ret
	local driver = dconn.driver()
	if driver == nil then
		util.printWarning("connection driver is empty")
		return
	end
	if driver.fieldNameArray == nil then -- for example 4d driver does not have driver.fieldNameArray() -function
		local _
		_, ret = dschema.fieldArray(tblNameRecType, schema)
		if ret then
			local ret2 = {}
			for i, item in ipairs(ret) do
				ret2[i] = util.cloneShallow(item)
				ret2[i].table_rec = nil
			end
			ret = ret2
		end
	else
		local conn = dconn.currentConnection()
		local tableName = dschema.tableName(tblNameRecType, schema) -- , recordType
		if tableName == nil then
			ret = {}
		else
			ret = driver.fieldNameArray(tableName, conn.organization_id) -- we MUST send conn
		end
	end
	--[[ if ret then
		loc.fieldNameArray[orgId.. "-" .. tblName] = ret
		-- loc.fieldNameIdx[orgId.. "-" .. tblName] = fn.util.createIndex(ret, "field_name")
	end ]]
	if prevOrgId ~= orgId then
		dconn.setCurrentOrganization(prevOrgId)
	end
	return ret -- , loc.fieldNameIdx[orgId]
end
dload.fieldNameArray = fieldNameArray

local function tableNameArray(tblName, schema, option)
	if loc.tableNameArray == nil then
		loc.tableNameArray = {}
		loc.tableNameIdx = {}
		dprf.registerCache("dload-loc.tableNameArray", loc.tableNameArray, {no_error = true}) -- may be cleared by dload.clearTableNameArray()
	end
	-- do refresh by default
	local driver = dconn.driver()
	local conn, orgId, prevOrgId
	if driver and driver.tableNameArray then
		conn = dconn.currentConnection()
		orgId = conn.organization_id
	end
	if orgId == nil or conn == nil or conn.schema ~= "schema" then
		orgId, prevOrgId = dconn.setAuthOrganization(tblName, schema, "") -- usually we check for preference -table or external table, use record type ""
		if (not option or not option.refresh) and loc.tableNameIdx[orgId] and not util.tableIsEmpty(loc.tableNameIdx[orgId]) then
			if prevOrgId ~= orgId then
				dconn.setCurrentOrganization(prevOrgId)
			end
			return loc.tableNameArray[orgId], loc.tableNameIdx[orgId]
		end
		driver = dconn.driver()
	end
	if driver == nil then
		util.printWarning("connection driver is empty")
		return
	end
	local ret
	if driver.tableNameArray == nil then -- for example 4d driver does not have driver.tableNameArray() -function
		ret = dschema.tableArray(schema)
		if ret then
			local ret2 = {}
			local i = 0
			for _, tableName in ipairs(ret) do
				if not peg.startsWith(tableName, "_timescaledb") then
					i = i + 1
					ret2[i] = {table_name = tableName, schema = schema or ""}
				end
			end
			ret = ret2
		end
	else
		conn = dconn.currentConnection() -- this fails for new external schema database, gives previous local database connection
		dconn.setCurrentOrganization(conn.organization_id)
		ret = driver.tableNameArray(conn.organization_id) -- we MUST send conn
		local ret2 = {}
		if ret == nil then
			util.printError("table name array is nil, organization '%s'", tostring(conn.organization_id))
			driver.tableNameArray(conn.organization_id) -- for debug
		else
			local i = 0
			for _, item in ipairs(ret) do
				if item.schema == nil then
					i = i + 1
					item.schema = "" -- sqlite
					ret2[i] = item
				elseif not peg.startsWith(item.schema, "_timescaledb") then
					i = i + 1
					ret2[i] = item
				end
			end
		end
		ret = ret2
	end
	if ret then
		loc.tableNameArray[orgId] = ret
		loc.tableNameIdx[orgId] = fn.util.createIndex(ret, "table_name")
	end
	if prevOrgId and prevOrgId ~= orgId then
		dconn.setCurrentOrganization(prevOrgId)
	end
	return ret, loc.tableNameIdx[orgId] -- , info
end
dload.tableNameArray = tableNameArray

local function tableInfoRecord(tblName, schema, option)
	schema = schema or ""
	local _, tableNameIdx = tableNameArray(tblName, schema, option)
	if tableNameIdx == nil then
		tableNameIdx = {}
	end
	local tableNameSql = dschema.tableName(tblName, schema)
	if tableNameSql == nil then
		tableNameSql = tblName -- may be external table name or any table name
	end
	tableNameSql = tableNameSql and peg.parseAfter(tableNameSql, ".") -- audit.log -> log
	return tableNameIdx[tableNameSql]
end
-- dload.tableInfoRecord = tableInfoRecord -- enable when needed

function dload.tableExists(tblName, schema, option)
	return tableInfoRecord(tblName, schema, option) ~= nil
end

return dload
