--- lib/db/doperation.lua
--
package.path = "lib/?.lua;lib/?.lx;" .. package.path
package.path = "../lib/?.lua;../lib/?.lx;" .. package.path
require "start"

local doperation = {}

local dschema = require "dschema"
local xxhash = require "xxhash"
local dprf = require "dprf"
local dqry = require "dqry"
local dload = require "dload"
local util = require "util"
local json = require "json"
local fs = require "fs"
local peg = require "peg"
local fn = require "fn"
local qry = require "qry"
local dsave = require "dsave"
local dsql = require "dsql"
local dconn = require "dconn"
local dt = require "dt"
local prefPath = "table"
local print = util.print
local loc = {}
loc.saveLogAfterEverySqlExecution = true
local prfOption = "no-db no-cache"
local tableExistsOption = {refresh = true}

local databaseUpdateStatus = {}
function doperation.databaseUpdateStatus(id, action)
	if not databaseUpdateStatus[id] then -- prevent recursion
		databaseUpdateStatus[id] = action -- "start"
		return false
	end
	if databaseUpdateStatus[id] == "check" and action == "start" then
		return false
	end
	if action ~= "start" then
		databaseUpdateStatus[id] = action -- should be "done"
	end
	return true
end

local function readConsole()
	-- return "yes" -- vscode debug
	return io.read()
end

-- local l = require "lang".l
local function l(txt, ...) -- lang would cause db query and we can't query db until it has been created
	return string.format(txt, ...)
end

local function getError(info)
	if type(info) == "string" and info ~= "" then
		return info
	elseif type(info) == "table" and type(info.error) == "string" and info.error ~= "" then
		return info.error
	end
	return
end

local function addToSqlTextArr(sqlArr, sqlText, rowEndMark) -- endMark = ",\n"
	if sqlText and sqlText ~= "" then
		sqlArr = sqlArr or {}
		if rowEndMark then
			if #sqlArr > 0 then
				sqlArr[#sqlArr] = sqlArr[#sqlArr] .. rowEndMark .. "\n"
			end
		end
		sqlArr[#sqlArr + 1] = sqlText
	end
end

local function toBoolean(field)
	if field == false or field == "false" or field == "0" or field == 0 then
		return false
	end
	return true
end

local function quoteDefaultValue(item)
	local str = item.default_value
	local fldType = item.field_type or item.basic_type or item.lua_type
	if fldType == "double" or fldType == "integer" or fldType == "bigint" or fldType == "number" then
		str = tonumber(str or 0)
	elseif fldType == "boolean" then
		return tostring(toBoolean(str or false))
	else
		str = tostring(str or "")
	end
	if type(str) == "string" then
		if str:sub(1, 1) ~= "'" then
			str = "'" .. str
		end
		if str:sub(-1) ~= "'" then
			str = str .. "'"
		end
	end
	return str
end

local function fieldDefinition(rec, tblRec)
	local indexDef
	if rec.field_type == nil then
		util.printError("table '%s' field '%s' does not have field_type", tblRec.table_name, rec.field_name)
		return
	end
	local def = loc.fieldTypeTbl[rec.field_type]
	if def == nil then
		util.printError("table '%s' field '%s' field_type '%s' is not valid", tblRec.table_name, rec.field_name, tostring(rec.field_type))
		return
	end
	local tableName = dschema.quoteSql(tblRec.table_name)
	local fieldName = dschema.quoteSql(rec.field_name)
	local constraint = rec.constraint and util.clone(rec.constraint) -- clone because we may delete element "index"
	local idx = constraint and fn.index("index", constraint)
	local idxName = tblRec.table_prefix .. "_" .. rec.field_name .. "_idx"
	if idx then
		indexDef = string.format("DROP INDEX IF EXISTS %s; CREATE INDEX %s ON %s(%s)", idxName, idxName, tableName, rec.field_name)
		table.remove(constraint, idx)
		-- example: "CREATE INDEX log_relid_idx ON audit.log(relid)""
	end
	idx = constraint and fn.index("unique", constraint)
	if idx then -- it's better to override previous possible INDEX with UNIQUE INDEX
		indexDef = string.format("DROP INDEX IF EXISTS %s; CREATE UNIQUE INDEX %s ON %s(%s)", idxName, idxName, tableName, rec.field_name)
		table.remove(constraint, idx)
	end
	constraint = constraint and string.upper(table.concat(constraint, " ")) or ""
	-- local foreignKey = ""
	if rec.foreign_key then
		-- see: https://www.shayon.dev/post/2022/17/why-i-enjoy-postgresql-infrastructure-engineers-perspective/
		local constraintArr = {}
		for i, item in ipairs(rec.foreign_key) do
			local onDelete
			if item.link_type == "delete with" then
				onDelete = "ON DELETE CASCADE"
			elseif item.link_type == "prevent delete" then
				onDelete = "ON DELETE RESTRICT"
			elseif item.link_type == "clear linking field" then
				onDelete = "ON DELETE SET NULL" -- possible: ON DELETE SET DEFAULT
			elseif item.link_type == "do nothing" then
				onDelete = "ON DELETE NOT VALID"
			else
				util.printError("unknown rec.foreign_key[%d].link_type '%s'", i, tostring(item.link_type))
				return nil
			end
			local constraintStm = "FOREIGN KEY (" .. fieldName .. ") REFERENCES " .. item.linked_table .. "(" .. item.linked_field .. ")"
			local constraintName = rec.field_name .. "_fkey" -- todo: add constraintName, todo: add postgres constraint NOT VALID and later ALTER TABLE tbl VALIDATE CONSTRAINT constraintName;
			constraintArr[i] = "ALTER TABLE " .. tableName .. " ADD CONSTRAINT " .. constraintName .. " " .. constraintStm .. " ON UPDATE CASCADE " .. onDelete
		end
		rec.constraint = table.concat(constraintArr, "\n")
	end
	local notNull = "NOT NULL"
	if rec.foreign_key then
		notNull = ""
	end
	local defaultValue = ""
	if rec.default_function ~= nil then
		defaultValue = "DEFAULT " .. rec.default_function
	elseif rec.default_value ~= nil then
		local val = quoteDefaultValue(rec)
		if val then
			defaultValue = "DEFAULT " .. val
		end
	elseif def and def.default_value ~= nil then
		local val = quoteDefaultValue(def)
		if val then
			defaultValue = "DEFAULT " .. val
		end
	end
	if rec.allowed_value then
		local allowedValue = rec.allowed_value
		if type(allowedValue) == "string" and peg.found(allowedValue, ".json") then
			allowedValue = dprf.prf(allowedValue)
		end
		if type(allowedValue) ~= "table" then
			util.printError("field allowed_value is not a table, field: %s", rec)
		else
			local allowedArr = allowedValue
			if not util.isArray(allowedValue) then
				allowedArr = util.fieldValueToArray(allowedValue, "value")
			end
			local ok
			if rec.field_type == "varchar" or rec.field_type == "text" then
				ok, allowedArr = pcall(table.concat, allowedArr, "','")
				if ok then
					allowedArr = "'" .. allowedArr .. "'"
				end
			else
				ok, allowedArr = pcall(table.concat, allowedArr, ",")
			end
			if not ok then
				util.printError("field allowed_value is not an array, field: %s", rec)
			else
				rec.constraint = rec.constraint or ""
				rec.constraint = rec.constraint .. "\nALTER TABLE " .. tableName .. "ADD CONSTRAINT check_" .. rec.field_name .. "_val CHECK (" .. rec.field_name .. " in (" .. allowedArr .. "))"
			end
		end
	end
	if notNull ~= "" and defaultValue ~= "" then
		defaultValue = " " .. defaultValue
	end
	if notNull .. defaultValue ~= "" and constraint ~= "" then
		constraint = " " .. constraint
	end
	if rec.field_type == "varchar" then
		if rec.field_length == nil then
			util.printError("table %s field %s does not have field_length", tableName, fieldName)
			return
		end
		return " " .. fieldName .. " " .. def.db_value .. "(" .. rec.field_length .. ") " .. notNull .. defaultValue .. constraint, indexDef
	end
	return " " .. fieldName .. " " .. def.db_value .. " " .. notNull .. defaultValue .. constraint, indexDef
	--[[
			rate_id varchar(80) NOT NULL DEFAULT '',
			main_currency_id varchar(20) NOT NULL DEFAULT '',
			currency_id varchar(20) NOT NULL DEFAULT '',
			type varchar(20) NOT NULL DEFAULT '',
			currency_date date NOT NULL DEFAULT '1970-01-01',
			rate DOUBLE PRECISION NOT NULL DEFAULT 0,
			info text NOT NULL DEFAULT '',

			record_id uuid NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000' UNIQUE,
			create_user varchar(40) NOT NULL DEFAULT '',
			create_time timestamp NOT NULL DEFAULT '1970-01-01 00:00:00+00',

			modify_id uuid NOT NULL DEFAULT '00000000-0000-0000-0000-000000000000',
			modify_user varchar(40) NOT NULL DEFAULT '',
			modify_time timestamp NOT NULL DEFAULT '1970-01-01 00:00:00+00',
			PRIMARY KEY(rate_id)
		)
	]]
end

local function saveChanges(organizationId, newVersion)
	-- todo: use "set statement_timeout = '50ms';" - https://postgres.ai/blog/20210923-zero-downtime-postgres-schema-migrations-lock-timeout-and-retries
	local function saveLog(saveTbl)
		local _, err = dsave.saveJson({preference_name = "save/common/preference/save.json", save_data = saveTbl})
		if err then
			return err
		end
	end

	--[[ -- no uuid generating in database
	if type(loc.extensionSql) == "table" and #loc.extensionSql > 0 then
		for _, sqlText in ipairs(loc.extensionSql) do
			print(sqlText.."\n")
			local _, info = dsql.sqlExecuteUnsafe(sqlText)
			if getError(info) then
				return {error = getError(info)}
			end
		end
		loc.extensionSql = nil
	end ]]

	if type(loc.changeLog) == "table" and #loc.changeLog > 0 then
		local prfNameArr = fn.util.mapFieldValue(loc.changeLog, "prf_name"):toDistinctArr()
		local prfTbl, saveToLog, prfRecordLoaded
		local save = false
		local tableNameIdx = {}
		local prfSaveNum = 0
		for idx, rec in ipairs(loc.changeLog) do
			if rec.save_to_log == nil then
				saveToLog = true
			else
				saveToLog = rec.save_to_log
			end
			if not prfRecordLoaded and saveToLog == true then
				prfRecordLoaded = true
				prfTbl = qry.query({query = "query/common/preference/json_data.json", parameter = {name = prfNameArr}})
				if getError(prfTbl) then
					return {error = getError(prfTbl)}
				end
				prfTbl = prfTbl.data
			end
			prfTbl = prfTbl or {}
			if rec.field_change ~= nil then
				local prfRec = util.arrayRecord(rec.prf_name, prfTbl, "prf.name_id")
				if prfRec == nil then
					prfTbl[#prfTbl + 1] = {name_id = rec.prf_name}
					prfRec = prfTbl[#prfTbl]
				end
				local jsonTbl = {}
				if prfRec.json_data then
					jsonTbl = json.fromJson(prfRec.json_data)
				end
				jsonTbl.field_change = jsonTbl.field_change or {}
				jsonTbl.field_change[#jsonTbl.field_change + 1] = rec.field_change
				jsonTbl.field_change[#jsonTbl.field_change].version = newVersion
				prfRec.json_data = json.toJsonRaw(jsonTbl)
				if type(rec.field_change) == "table" and type(rec.field_change.sql) == "string" and rec.field_change.sql ~= "" then
					print(rec.field_change.sql .. "\n")
					--[[ if not dconn.allowDatabaseSchemaUpdate(dbName) then
						return {error = l("database '%s' structure change is not allowed (alter database"), dbName}
					end ]]
					dconn.setCurrentOrganization(organizationId)
					local conn = dconn.connection()
					if conn.dbtype ~= "postgre" then
						if conn.dbtype == "sqlite" then
							rec.field_change.sql = peg.replace(rec.field_change.sql, "PRIMARY KEY", "UNIQUE")
							rec.field_change.sql = peg.replace(rec.field_change.sql, "NOW()", "CURRENT_TIMESTAMP")
							rec.field_change.sql = peg.replace(rec.field_change.sql, "BIGINT NOT NULL,", "BIGINT NOT NULL DEFAULT 0,") -- without the trigger
						end
						rec.field_change.sql = peg.replace(rec.field_change.sql, "BIGINT NOT NULL GENERATED ALWAYS AS IDENTITY", "INTEGER NOT NULL PRIMARY KEY AUTOINCREMENT")
					end
					local _, info = dsql.sqlExecuteUnsafe(rec.field_change.sql, {}, {})
					if getError(info) then
						return {error = util.printRed("error in executing sql at index %d, see change log '%s':\n", idx, rec.prf_name, getError(info))}
					end
					dconn.disconnect(conn)
					dconn.setCurrentOrganization(organizationId)
					if loc.saveLogAfterEverySqlExecution and saveToLog == true or getError(info) then
						for _, prf in ipairs(prfTbl) do
							prfSaveNum = prfSaveNum + 1
							prf.name_id = prf.name_id .. " - " .. prfSaveNum
						end
						local err = saveLog(prfTbl)
						if getError(err) then
							util.printRed("error in saving change log '%s':\n", rec.prf_name, getError(err))
							-- return {error = l("error in saving change log '%s':\n", rec.prf_name, getError(err))}
						end
					end
				end
				if not loc.saveLogAfterEverySqlExecution then
					save = true
				end
				if not tableNameIdx[rec.field_change.table_name:lower()] then
					tableNameIdx[rec.field_change.table_name:lower()] = {table_name = rec.field_change.table_name, table_prefix = rec.field_change.table_prefix}
				end
			end
		end

		dprf.clearConnection()
		if save and saveToLog == true then
			local err = saveLog(prfTbl)
			if getError(err) then
				return {error = l("error in saving change log '%s':\n", table.concat(prfNameArr), getError(err))}
			end
		end
		--[[
		fn.iter(tableNameTbl):each(function(rec)
			local externalName = dconv.externalName(rec.table_prefix)
			if externalName == nil then -- add only if table does not exist as 4d field number
				f = db.fieldTableAdd(loc.conn, rec.table_name, rec.table_prefix, nil)
			end
		end)
		--]]
	end
end

local function addToChangeLog(param)
	-- param.table_prf
	-- param.field_rec
	-- param.sql_rec
	-- param.sql_txt
	-- param.table_path

	if loc.changeLog == nil then
		loc.changeLog = {}
	end
	local changeRec = {}
	loc.changeLog[#loc.changeLog + 1] = changeRec
	changeRec.prf_name = "version_log_" .. param.table_path .. "_" .. dt.currentString()
	changeRec.field_change = {table_name = param.table_prf.table_name, old_table_name = param.table_prf.old_table_name, table_prefix = param.table_prf.table_prefix, old = param.sql_rec, sql = param.sql_txt, log_type = param.log_type}
	if param.delete then
		changeRec.field_change.delete = param.sql_rec
		changeRec.field_change.old = nil
	else
		changeRec.field_change.new = param.field_rec
	end
	changeRec.save_to_log = param.save_to_log
	addToSqlTextArr(loc.sqlTextArr, param.sql_txt)
end

local function addColumnSqlText(param)
	-- param.table_prf
	-- param.field_rec
	-- param.table_path
	local def = fieldDefinition(param.field_rec, param.table_prf)
	if def == nil then
		return
	end
	local sqlText = "ALTER TABLE \"" .. param.table_prf.table_name .. "\" ADD COLUMN " .. def
	local saveParam = param
	saveParam.sql_txt = sqlText
	saveParam.log_type = "add_field"
	addToChangeLog(saveParam)
	return
end

local function renameColumnSqlText(param)
	-- param.table_prf
	-- param.field_rec
	-- param.sql_rec
	-- param.table_path
	if param.sql_rec.field_name ~= param.field_rec.field_name then
		local sqlText = "ALTER TABLE \"" .. param.table_prf.table_name .. "\" RENAME COLUMN " .. param.sql_rec.field_name .. " TO " .. param.field_rec.field_name
		local saveParam = param
		saveParam.sql_txt = sqlText
		saveParam.log_type = "rename_field"
		addToChangeLog(saveParam)
	end
	return
end

local function renameTableSqlText(param)
	-- param.table_prf
	-- param.table_path
	---[[
	if param.table_prf.old_table_name ~= param.table_prf.table_name then
		local sqlText = "ALTER TABLE \"" .. param.table_prf.old_table_name .. "\" RENAME TO \"" .. param.table_prf.table_name .. '"'
		local saveParam = param
		saveParam.sql_txt = sqlText
		saveParam.log_type = "rename_table"
		addToChangeLog(saveParam)
	end
	-- ]]
end

local function changeColumnSqlText(param)
	-- param.table_prf
	-- param.field_rec
	-- param.sql_rec
	-- param.table_path
	local def
	def = util.arrayRecord(param.field_rec.field_type, loc.fieldType, "change_to_value")
	if def == nil then
		def = util.arrayRecord(param.field_rec.field_type, loc.fieldType, "value")
	end
	if type(def) == "table" then
		param.field_rec.nullable = param.field_rec.nullable or false
		param.sql_rec.nullable = param.sql_rec.nullable or false
		-- if param.sql_rec.field_type ~= def.db_value:lower()
		if param.sql_rec.field_type ~= def.value:lower() or param.sql_rec.nullable ~= param.field_rec.nullable or param.sql_rec.field_length and param.field_rec.field_length and param.field_rec.field_length ~= param.sql_rec.field_length then
			local changeSql = {}
			if param.sql_rec.nullable ~= param.field_rec.nullable then
				if param.field_rec.nullable then -- param.field_rec.foreign_key then
					addToSqlTextArr(changeSql, "ALTER COLUMN " .. param.field_rec.field_name .. " DROP NOT NULL", ",")
				else
					addToSqlTextArr(changeSql, "ALTER COLUMN " .. param.field_rec.field_name .. " SET NOT NULL", ",")
				end
			end
			if param.field_rec.field_type == "varchar" then
				addToSqlTextArr(changeSql, "ALTER COLUMN " .. param.field_rec.field_name .. " TYPE " .. def.db_value .. "(" .. param.field_rec.field_length .. ") USING " .. param.field_rec.field_name .. "::" .. def.db_value:lower(), ",")
			else
				local oldType = param.sql_rec.field_type:lower()
				if oldType == "varchar" then
					addToSqlTextArr(changeSql, "ALTER COLUMN " .. param.field_rec.field_name .. " TYPE " .. def.db_value .. " USING (trim(" .. param.field_rec.field_name .. ")::" .. def.db_value:lower() .. ")", ",")
				else
					addToSqlTextArr(changeSql, "ALTER COLUMN " .. param.field_rec.field_name .. " TYPE " .. def.db_value .. " USING " .. param.field_rec.field_name .. "::" .. def.db_value:lower(), ",")
				end
			end
			if def.default_value ~= nil then
				addToSqlTextArr(changeSql, "ALTER COLUMN " .. param.field_rec.field_name .. " SET DEFAULT " .. tostring(def.default_value), ",")
			end
			local sqlText = "ALTER TABLE \"" .. param.table_prf.table_name .. "\"\n" .. table.concat(changeSql)
			param.sql_rec.change_to_value = param.field_rec.field_type -- set for printed log
			local saveParam = param
			saveParam.sql_txt = sqlText
			saveParam.log_type = "change_field"
			addToChangeLog(saveParam)
		end
	end
	return
end

local function dropColumnSqlText(param)
	-- param.table_prf
	-- param.sql_rec
	-- param.table_path
	local sqlText = "ALTER TABLE \"" .. param.table_prf.table_name .. "\" DROP COLUMN " .. param.sql_rec.field_name
	local saveParam = param
	saveParam.sql_txt = sqlText
	saveParam.delete = true
	saveParam.log_type = "remove_field"
	addToChangeLog(saveParam)
	return
end

--[[
local function dropTableSqlText(param)
	-- param.table_prf
	-- param.table_path
  local tableName = "\""..param.table_prf.table_name.."\""
	local sqlText = "DROP TABLE IF EXISTS "..tableName
  local saveParam = param
  saveParam.sql_txt = sqlText
  saveParam.delete = true
  saveParam.log_type = "remove_table"
  addToChangeLog(saveParam)
	return
end
--]]

local function alterSqlField(prf, defaultTbl, sqlTable, table_path)
	if prf == nil or util.tableIsEmpty(prf) then
		return -- createTableFieldPreference(tablePrefix)
	end
	if prf.table_name == nil then
		util.printError("'table_name'-tag missing from preference '%s'", prf.table_name .. ".json")
		return
	end
	local defaultField = defaultTbl and defaultTbl.field and defaultTbl.field.default_field
	local fieldChange = prf.field_change
	if type(defaultTbl) == "table" and type(defaultTbl.field_change) == "table" and #defaultTbl.field_change > 0 then
		fieldChange = fieldChange or {}
		if not util.tableEqual(prf.field_change, defaultTbl.field_change) then
			fieldChange = util.arrayCombine({fieldChange, defaultTbl.field_change})
		end
	end
	local deleteField = prf.delete_field
	if type(defaultTbl) == "table" and type(defaultTbl.delete_field) == "table" and #defaultTbl.delete_field > 0 then
		deleteField = deleteField or {}
		deleteField = util.arrayCombine({deleteField, defaultTbl.delete_field})
	end
	local prfTable = prf.field
	if type(defaultTbl) == "table" and type(defaultTbl.field) == "table" and #defaultTbl.field > 0 then
		for _, rec in ipairs(defaultTbl.field) do -- add default fields to field arr
			local found = util.arrayRecord(rec.field_name, prfTable, "field_name")
			if found == nil then
				prfTable[#prfTable + 1] = rec
			end
		end
	end
	if defaultField then
		for _, rec in ipairs(defaultField) do -- add default fields to field arr
			local found = util.arrayRecord(rec.field_name, prfTable, "field_name")
			if found == nil then
				prfTable[#prfTable + 1] = rec
			end
		end
	end
	-- add missing fields to sql
	local err -- TODO: do not overwrite err and use return value
	for _, item in ipairs(sqlTable) do
		item.field_name_lower = item.field_name:lower()
	end
	local sqlTableFieldNameIdx = fn.util.createIndex(sqlTable, "field_name_lower")
	local fieldChangeRec, fieldChangeNameIdx
	if fieldChange then
		for _, item in ipairs(fieldChange) do
			item.name_lower = item.name:lower()
		end
		fieldChangeNameIdx = fn.util.createIndex(fieldChange, "name_lower") -- use "name" only in fieldChange -array
	end
	for _, rec in ipairs(prfTable) do
		-- sqlFieldInUse[#sqlFieldInUse + 1] = rec.field_name
		rec.field_name_lower = rec.field_name:lower()
		if fieldChangeNameIdx then
			fieldChangeRec = fieldChangeNameIdx[rec.field_name_lower] -- use "name" only in fieldChange -array
		end
		local sqlRec = sqlTableFieldNameIdx[rec.field_name_lower]
		if fieldChangeRec ~= nil and sqlRec ~= nil then
			if type(fieldChangeRec.old_name) == "table" and fn.index(rec.field_name, fieldChangeRec.old_name) then -- TODO: use lower
				-- update only if old_name = name
				-- check if field have been changed
				err = changeColumnSqlText({table_prf = prf, field_rec = rec, sql_rec = sqlRec, table_path = table_path}) -- change: type, length...
			end
		elseif sqlRec == nil then
			-- ??? do not add missing "defaultField"
			-- ??? if not defaultField or util.arrayRecord(rec.field_name, defaultField, "field_name") == nil then
			if fieldChangeRec == nil then -- create new field
				err = addColumnSqlText({table_prf = prf, field_rec = rec, table_path = table_path})
			elseif type(fieldChangeRec.old_name) == "table" and #fieldChangeRec.old_name > 0 then
				for _, oldName in ipairs(fieldChangeRec.old_name) do
					sqlRec = sqlTableFieldNameIdx[oldName:lower()]
					if sqlRec then
						-- sqlFieldInUse[#sqlFieldInUse] = sqlRec.field_name -- rename to sql field name
						err = renameColumnSqlText({table_prf = prf, field_rec = rec, sql_rec = sqlRec, table_path = table_path}) -- rename field
						if err then
							break
						end
						err = changeColumnSqlText({table_prf = prf, field_rec = rec, sql_rec = sqlRec, table_path = table_path}) -- change: type, length...
						break
					end
				end
				if sqlRec == nil then -- add column if it does not exist in field_change -tag value
					err = addColumnSqlText({table_prf = prf, field_rec = rec, table_path = table_path})
				end
			end
		end
	end

	-- check fields that are deleted from json but exists in sql
	local fieldNameIdx = fn.util.createIndex(prfTable, "field_name_lower")
	if defaultField then
		for _, item in ipairs(defaultField) do
			fieldNameIdx[item.field_name:lower()] = item
		end
	end
	for _, item in ipairs(sqlTable) do
		if fieldNameIdx[item.field_name_lower] == nil then
			if deleteField == nil then
				deleteField = {}
			end
			deleteField[#deleteField + 1] = item.field_name
		end
	end

	if type(deleteField) == "table" and #deleteField > 0 then
		for _, rec in ipairs(sqlTable) do
			if fn.index(rec.field_name, deleteField) then
				util.print("    field '%s' will deleted from table '%s'", rec.field_name, prf.table_name)
				err = dropColumnSqlText({table_prf = prf, sql_rec = rec, table_path = table_path})
			end
		end
	end
	-- createPreference(tablePrefix)
	return err
end

local function checkConnection(organizationId, conn)
	organizationId = tostring(organizationId)
	local validConnection = dconn.validConnection({organizationId = organizationId})
	if not validConnection then -- how to check if connection works?
		repeat
			util.printInfo("\nplease start '%s' database server", dconn.defaultDatabaseType())
			io.write("  - " .. l("press enter to retry after database has been started or close this program if you don't want to continue"))
			readConsole()
			validConnection = dconn.validConnection({organizationId = organizationId})
		until validConnection
	end
	if conn and validConnection then
		return
	end
	local currentConn = dconn.currentConnection()
	if currentConn == nil or currentConn.organization_id ~= organizationId then
		util.printError("current connection '%s' is not same as '%s'", tostring(currentConn and currentConn.organization_id), organizationId)
		util.closeProgram()
	end
	--[[
  local _, info = db.sqlExecuteUnsafeArray(loc.conn, "SELECT version()", {"string"})
  if info.error then
    util.printError(tostring(info.error))
  end
  --]]
end

local function createTriggerAndAuditLog(option)
	local conn = dconn.currentConnection()
	if conn.driver ~= "postgre" then
		return false -- has no trigger
	end
	local ret, info
	if option == nil or option.audit_log ~= false then
		if not dload.tableExists("audit.log", "", {refresh = true}) then -- this fails for new external schema database
			local sqlUpdateTrigger = fs.readFile("table/prf/audit.sql") or ""
			util.print("creating table/prf/audit.sql...")
			ret, info = dsql.sqlExecuteUnsafe(sqlUpdateTrigger, {}, {}, {query_name = "create audit.sql"})
			if getError(info) or getError(ret) then
				return {error = getError(info) or getError(ret)}
			end
			util.printOk("created table/prf/audit.sql")
		end
	end
	ret, info = dsql.sqlExecuteUnsafeRecordArray("SELECT proname FROM pg_catalog.pg_proc WHERE proname = 'updatemodifyid'", {"proname"}, {"string"}, {query_name = "check if trigger function UpdateModifyId() exists"})
	if ret and #ret < 1 or info.error or option.update then
		-- NEW.db_modify_id = nextval(TG_ARGV[0]);
		--[=[ local updateModifyIdSql = [[
			create or replace function UpdateModifyId() returns trigger AS $$
			BEGIN
			NEW.db_modify_id = nextval(TG_ARGV[0]);
			IF TG_OP = 'UPDATE' THEN
				NEW.db_version = OLD.db_version + 1;
			END IF;
			RETURN NEW;
			END;
			$$ language plpgsql;
			]]
		local updateModifyIdSql = [[
			CREATE OR REPLACE FUNCTION UpdateModifyId() RETURNS trigger AS $$
			BEGIN
			NEW.db_modify_id = nextval(TG_ARGV[0]);
			NEW.modify_time = now();
			RETURN NEW;
			END;
			$$ language plpgsql;
			]]
		]=]
		local update = option.update and ret and #ret > 0
		local updateModifyIdSql = option.update_modify_id or fs.readJsonFile(util.preferencePath() .. "system/option.json").trigger.update_modify_id -- must be same in versionUpdate()
		util.print("%s trigger function UpdateModifyId():\n%s", update and "updating" or "creating", updateModifyIdSql)
		ret, info = dsql.sqlExecuteUnsafe(updateModifyIdSql, {}, {}, {query_name = "create trigger function UpdateModifyId()"})
		if getError(info) or getError(ret) then
			return {error = getError(info) or getError(ret)}
		end
		util.printOk("%s trigger function UpdateModifyId()", update and "updated" or "created")
	end
	return true -- has trigger
end

local function createTable(tblRec, tablePath, saveToLog, option, organizationId)
	checkConnection(organizationId)
	createTriggerAndAuditLog(option) -- creates only once
	if saveToLog == nil then
		saveToLog = true
	end
	local sqlTmp = {}
	local sqlIndexTmp = {}
	local dbIdExists = false
	local dbModifyIdExists = false
	for _, rec in ipairs(tblRec.field) do
		-- if hasTrigger then
		local def, indexDef = fieldDefinition(rec, tblRec)
		if def == nil then
			return false -- error here
		end
		if rec.field_name == "db_id" then
			dbIdExists = true
		elseif rec.field_name == "db_modify_id" then
			dbModifyIdExists = true
		end
		addToSqlTextArr(sqlTmp, def, ",")
		if indexDef then
			addToSqlTextArr(sqlIndexTmp, indexDef, ";")
		end
		-- end
	end
	local conn = dconn.currentConnection()
	if conn.driver == "postgre" and (dbIdExists == false or dbModifyIdExists == false) then
		local defaultFieldPrf = dprf.prf("table/prf/default_field.json")
		if dbIdExists == false then
			local rec = util.arrayRecord("db_id", defaultFieldPrf.default_field, "field_name")
			local def, indexDef = fieldDefinition(rec, tblRec)
			if def == nil then
				return false -- error here
			end
			addToSqlTextArr(sqlTmp, def, ",")
			if indexDef then
				addToSqlTextArr(sqlIndexTmp, indexDef, ";")
			end
		end
		if dbModifyIdExists == false then
			local rec = util.arrayRecord("db_modify_id", defaultFieldPrf.default_field, "field_name")
			local def, indexDef = fieldDefinition(rec, tblRec)
			if def == nil then
				return false -- error here
			end
			addToSqlTextArr(sqlTmp, def, ",")
			if indexDef then
				addToSqlTextArr(sqlIndexTmp, indexDef, ";")
			end
		end
	end
	local tableName = dschema.quoteSql(tblRec.table_name)
	local sequenceName = tblRec.table_name .. "_db_modify_seq"
	local triggerName = tblRec.table_name .. "_trigger"
	local sqlText
	if conn.dbtype == "sqlite" then
		sqlText = ""
	else
		sqlText = string.format("DROP SEQUENCE IF EXISTS %s;\nCREATE SEQUENCE %s AS bigint;\n", sequenceName, sequenceName)
	end
	-- sqlText = sqlText .. string.format("DROP TABLE IF EXISTS %s;\n", tableName)
	addToChangeLog({log_type = "remove_table_before_add", delete = true, table_prf = tblRec, sql_txt = sqlText, field_rec = tableName, table_path = tablePath, save_to_log = saveToLog})
	sqlText = sqlText .. "CREATE TABLE " .. tableName .. "(\n" .. table.concat(sqlTmp) .. "\n);\n"
	if #sqlIndexTmp > 0 then
		sqlText = sqlText .. table.concat(sqlIndexTmp, "") .. ";\n\n"
	end
	if tblRec.unique then
		local sqlUniqueIndexTmp = {}
		for _, uniqueArr in ipairs(tblRec.unique) do
			-- local def = string.format("CREATE UNIQUE INDEX %s ON %s (%s)", tblRec.table_prefix.."_"..table.concat(uniqueArr,"_").."_key", tableName, table.concat(uniqueArr,", "))
			local def = string.format("CREATE UNIQUE INDEX %s ON %s (%s)", tblRec.table_prefix .. "_" .. uniqueArr[1] .. "_key", tableName, table.concat(uniqueArr, ", "))
			addToSqlTextArr(sqlUniqueIndexTmp, def, ",")
		end
		sqlText = sqlText .. table.concat(sqlUniqueIndexTmp, ";\n") .. ";\n"
	end
	if conn.driver == "postgre" then
		sqlText = sqlText .. string.format("DROP TRIGGER IF EXISTS %s ON %s;\nCREATE TRIGGER %s BEFORE INSERT OR UPDATE ON %s FOR EACH ROW EXECUTE PROCEDURE UpdateModifyId('%s');\n", triggerName, tableName, triggerName, tableName, sequenceName)
		if tblRec.audit ~= false and (option == nil or option.audit_log ~= false) then
			sqlText = sqlText .. string.format("SELECT audit.audit_table_on('%s');\n", tableName)
			-- else
			-- sqlText = sqlText..string.format("SELECT audit.audit_table_off('%s');\n", tableName)
		end
	end
	util.print("    created sql for table '%s', fields: %d", tblRec.table_name, #tblRec.field)
	addToChangeLog({log_type = "add_table", table_prf = tblRec, sql_txt = sqlText, field_rec = tblRec.field, table_path = tablePath, save_to_log = saveToLog})
	return true
end

local function createTableFromFile(filePath, saveToLog, option, organizationId)
	local defaultTbl
	local tblRec = dprf.prf(filePath, prfOption)
	local defaultField
	if tblRec.default_field and tblRec.default_field ~= "" then
		defaultField = dprf.prf(tblRec.default_field, "no-db")
	else
		defaultField = dprf.prf("table/prf/default_field.json", "no-db") -- code does not come here if schema update is not allowed
		defaultField.default_field = fn.iter(defaultField.default_field):filter(function(item)
			return item.add_always
		end):totable()
	end
	defaultTbl = {field = defaultField, field_change = tblRec.field_change, delete_field = tblRec.delete_field}
	if tblRec and tblRec.table_prefix and tblRec.table_name and tblRec.field then
		local tablePath = peg.parseAfter(filePath, "table/")
		tablePath = peg.parseBeforeLast(tablePath, "/")
		local schema = ""
		if peg.startsWith(tablePath, "external/") then
			schema = peg.parseAfter(tablePath, "external/")
			schema = peg.parseBefore(schema, "/")
		end
		checkConnection(organizationId)
		local sqlTableName = tblRec.table_name
		local tableExists = dload.tableExists(sqlTableName, schema, tableExistsOption)
		if not tableExists and type(tblRec.old_table_name) == "table" and #tblRec.old_table_name > 0 then
			for _, tableOldName in ipairs(tblRec.old_table_name) do
				if dload.tableExists(tableOldName, schema, tableExistsOption) then
					sqlTableName = tableOldName
					tblRec.old_table_name = tableOldName -- change from table to string
					renameTableSqlText({table_prf = tblRec, table_path = tablePath}) -- rename field
					break
				end
			end
			tableExists = dload.tableExists(sqlTableName, schema, tableExistsOption) -- need to refresh
		end
		if tableExists then
			local conn = dconn.currentConnection()
			local sqlTable = dqry.readTableStructure(sqlTableName, "sql") -- "sql" or "lua"
			if sqlTable == nil then
				if sqlTableName ~= "preference" or conn.is_local ~= false then
					util.printRed("Sql table '%s' was not found", tblRec.table_name)
					return false
				end
			else
				alterSqlField(tblRec, defaultTbl, sqlTable, tablePath) -- check table field changes
			end
			if createTriggerAndAuditLog(option) then -- createTriggerAndAuditLog() returns hasTrigger
				local sequenceName = sqlTableName .. "_db_modify_seq"
				local triggerName = sqlTableName .. "_trigger"
				local sqlText = string.format("SELECT sequence_name FROM information_schema.sequences WHERE sequence_name = '%s'", sequenceName)
				local _, info = dsql.sqlExecuteUnsafe(sqlText, {"sequence_name"}, {"text"})
				if info.error then
					util.printError("Sql error '%s', sql: '%s;'", info.error, sqlText)
					return false
				elseif info.row_count == 0 then
					sqlText = string.format("CREATE SEQUENCE %s AS bigint", sequenceName)
					util.printInfo("creating database sequence: " .. sqlText)
					_, info = dsql.sqlExecuteUnsafe(sqlText)
					if info.error then
						util.printError("Sql error '%s', sql: '%s;'", info.error, sqlText)
						return false
					end
				end
				sqlText = string.format("SELECT trigger_name FROM information_schema.triggers WHERE trigger_name = '%s'", triggerName)
				_, info = dsql.sqlExecuteUnsafe(sqlText, {"trigger_name"}, {"text"})
				if info.error then
					util.printError("Sql error '%s', sql: '%s;'", info.error, sqlText)
					return false
				elseif info.row_count == 0 then
					sqlText = string.format("CREATE TRIGGER %s BEFORE INSERT OR UPDATE ON %s FOR EACH ROW EXECUTE PROCEDURE UpdateModifyId('%s')", triggerName, sqlTableName, sequenceName)
					if tblRec.audit ~= false and (option == nil or option.audit_log ~= false) then
						sqlText = sqlText .. string.format("\nSELECT audit.audit_table_on('%s')", sqlTableName)
					end
					util.printInfo("creating database trigger: " .. sqlText)
					_, info = dsql.sqlExecuteUnsafe(sqlText)
					if info.error then
						util.printError("Sql error '%s', sql: '%s;'", info.error, sqlText)
						return false
					end
				end
			end
		else -- table does not exist
			if type(defaultTbl) == "table" and defaultTbl.field then
				for _, rec in ipairs(defaultTbl.field.default_field) do
					tblRec.field[#tblRec.field + 1] = rec
				end
			end
			-- util.printInfo("create table '%s', preference '%s'", tblRec.table_prefix, peg.parseLast(filePath, "/"))
			if not createTable(tblRec, tablePath, saveToLog, option, organizationId) then
				return false
			end
		end
		--[[
		local externalName = dconv.externalName(tblRec.table_prefix)
		if externalName == nil then -- add only if table does not exist as 4d field number
			f = db.fieldTableAdd(loc.conn, tblRec.table_name, tblRec.table_prefix, nil)
		end
    --]]
	end
	return true
end

local function saveVersionPrf(saveRec, versionNumber, prfName)
	saveRec = saveRec or {}
	local jsonTbl
	if saveRec.json_data then
		jsonTbl = json.fromJson(saveRec.json_data)
	end
	jsonTbl = jsonTbl or {}
	jsonTbl.version_number = versionNumber
	jsonTbl.version = jsonTbl.version or {}
	jsonTbl.hash = saveRec.hash
	saveRec.hash = nil
	if jsonTbl.trigger_hash ~= saveRec.trigger_hash then
		createTriggerAndAuditLog({update = true, update_modify_id = saveRec.update_modify_id})
	end
	jsonTbl.trigger_hash = saveRec.trigger_hash
	saveRec.trigger_hash = nil
	saveRec.update_modify_id = nil
	jsonTbl.version[#jsonTbl.version + 1] = {number = versionNumber, update_time = dt.currentString()}
	saveRec.name_id = prfName
	saveRec.json_data = json.toJsonRaw(jsonTbl)
	local _, err = dsave.saveJson({preference_name = "save/common/preference/save.json", save_data = {saveRec}})
	return {error = err and getError(err)}
end

local function versionUpdate(save, orgId, database)
	local prfTbl, prfData
	if not save then
		orgId = dconn.organizationId() -- dconn.authOrganizationId()
	end
	if orgId == "" then
		return {error = "organizationId is empty"}
	end
	local function cleanup()
		dconn.setCurrentOrganization(orgId)
	end
	local newVersion = 0
	local prfName = "table/prf/version_update/" .. orgId .. ".json"
	-- if dload.tableExists("preference", "", tableExistsOption) then
	prfTbl = qry.query({query = "common/preference/json_data.json", parameter = {to_json = true, name = {prfName}, no_error = true}})
	local err = getError(prfTbl.error) or getError(prfTbl)
	if err then
		if type(err) ~= "string" or not peg.found(err, "preference") and not (peg.found(err, "no such table") or (peg.found(err, "relation") and peg.found(err, "does not exist"))) then
			-- sqlite: no such table
			-- postgre: relation ... does not exist
			cleanup()
			return {error = err}
		end
	end
	if prfTbl and prfTbl.data then
		prfData = prfTbl.data
	else
		prfData = {}
		newVersion = 1
	end
	local saveRec, jsonTbl
	local prevVersion = 0
	if util.tableIsEmpty(prfData) then
		newVersion = 0
		jsonTbl = {}
	else
		saveRec = prfData[1]
		if saveRec.json_data == nil then
			err = l("database '%s' -preference json_data is nil in save record: '%s'", prfName, saveRec)
			util.printInfo(err)
			cleanup()
			return
			-- return {error = err}
		end
		jsonTbl = json.fromJson(saveRec.json_data) -- pgmoon returns json as a table, libpq as a string
		newVersion = jsonTbl.version_number or newVersion
		if type(jsonTbl.version) == "table" and #jsonTbl.version > 0 then
			prevVersion = util.arrayFieldMaxValue(jsonTbl.version, "number")
			if not save then
				util.printInfo("  found database version '%s' from database '%s' preference '%s'", tostring(prevVersion), prfTbl.info.database, prfName)
			end
		end
		if newVersion < 1 then
			newVersion = 0
		end
	end
	if newVersion < prevVersion then
		err = l("database '%s' -preference new version number '%s' is smaller than current version number '%s'", prfName, tostring(newVersion), tostring(prevVersion))
		util.printInfo(err)
		cleanup()
		return
		-- return {error = err}
	end
	-- compare schema hash
	if dconn.allowDatabaseSchemaUpdate(database) then
		local schema = dconn.organizationIdToSchema(orgId)
		local dschemafld = require "dschemafld"
		local schemaPref = dschemafld.schemaGroupPreference(schema)
		local updateModifyIdSql = fs.readJsonFile(util.preferencePath() .. "system/option.json") -- must be same in doperation
		if updateModifyIdSql and updateModifyIdSql.trigger and updateModifyIdSql.trigger.update_modify_id then
			updateModifyIdSql = updateModifyIdSql.trigger.update_modify_id
		else
			util.printError("could not load preference 'system/option.json'")
			updateModifyIdSql = nil
		end
		local triggerHash = updateModifyIdSql and xxhash.hash128string(updateModifyIdSql)
		if schemaPref == nil then
			util.printRed("schema group preference was not found for organization '%s', schema '%s'", tostring(orgId), tostring(schema))
		elseif schemaPref.hash == nil then
			util.printRed("schema group preference hash -key was not found for organization '%s', schema '%s'", tostring(orgId), tostring(schema))
		elseif updateModifyIdSql and saveRec and (schemaPref.hash ~= jsonTbl.hash or triggerHash ~= jsonTbl.trigger_hash) then
			saveRec.trigger_hash = triggerHash
			saveRec.update_modify_id = updateModifyIdSql
			saveRec.hash = schemaPref.hash
			if not save then
				if newVersion < 1 then
					newVersion = 1
				else
					newVersion = newVersion + 1
				end
			end
		end
	end

	-- save preference
	if save == true then
		if newVersion < 1 then
			newVersion = 1
		else
			newVersion = newVersion + 1
		end
		local ret = saveVersionPrf(saveRec, newVersion, prfName)
		if getError(ret) then
			cleanup()
			return {error = getError(ret)}
		end
	end
	cleanup()
	if newVersion > prevVersion then
		return {update = true, current_version = prevVersion, new_version = newVersion, save_record = saveRec, hash = jsonTbl.hash, trigger_hash = jsonTbl.trigger_hash}
	end
	return {update = false, current_version = prevVersion, new_version = newVersion, save_record = saveRec, hash = jsonTbl.hash, trigger_hash = jsonTbl.trigger_hash}
end

function doperation.createUser(conn, user)
	local _, info
	local createRole = true
	local driver = dconn.driver(conn.driver)
	user = user or conn.database_user
	if driver.roleExists then
		createRole = not driver.roleExists(user, conn.database, conn)
		if createRole then
			info = {info = string.format("database role '%s' will be created", user)}
		else
			info = {info = string.format("database role '%s' already exists", user)}
		end
	end
	if info == nil then
		createRole = false
		info = {error = "database role could not be queried"}
	end
	if createRole then
		--[[ if not dconn.allowDatabaseSchemaUpdate(driver.database) then
			return {error = l("database '%s' structure change is not allowed", tostring(driver.database))}
		end ]]
		--[[ local sql = "SET password_encryption TO 'scram-sha-256'; CREATE ROLE %s LOGIN ENCRYPTED PASSWORD '%s' NOSUPERUSER INHERIT CREATEDB NOCREATEROLE REPLICATION;"
		local sqlText = string.format(sql, user, '*****')
		util.printInfo("creating database user: " .. sqlText)
		sqlText = string.format(sql, user, conn.password)
		_, info = dsql.sqlExecuteUnsafe(sqlText) ]]
		if info and info.info then
			local sql = "SET password_encryption TO 'scram-sha-256'; CREATE ROLE %s LOGIN ENCRYPTED PASSWORD '%s' NOSUPERUSER INHERIT CREATEDB NOCREATEROLE REPLICATION;" -- 'md5'
			local sqlText = string.format(sql, user, '*****')
			util.printInfo("creating database user: " .. sqlText)
			sqlText = string.format(sql, user, conn.password)
			_, info = dsql.sqlExecuteUnsafe(sqlText, nil, nil, {no_debug = true})
		end
	end
	return info
end

function doperation.createDatabase(dbName, dbUser, organizationId, option)
	--[[ if not dconn.allowDatabaseSchemaUpdate(dbName) then
		return {error = l("database '%s' structure change is not allowed (create database"), dbName}
	end ]]
	-- local sqlText = 'CREATE COLLATION IF NOT EXISTS nc_server FROM "und-x-icu";'
	-- local _, info = dsql.sqlExecuteUnsafe(sqlText)
	local sqlText
	--[=[if false and util.isWin() then
		sqlText = string.format([[CREATE DATABASE %s WITH
		TEMPLATE = template0
		OWNER = %s
		ENCODING = 'UTF8'
		LC_COLLATE = 'und-x-icu'
		LC_CTYPE = 'und-x-icu'
		CONNECTION LIMIT = -1;
		]], dbName, dbUser)
	else --]=]
	sqlText = [[CREATE DATABASE %s WITH
	TEMPLATE = template0
	OWNER = %s
	ENCODING = 'UTF8'
	CONNECTION LIMIT = -1
	]]
	if not util.isWin() then
		sqlText = sqlText .. [[LC_COLLATE = 'en_US.UTF-8'
	LC_CTYPE = 'en_US.UTF-8'
	]]
	end
	local conn = dconn.currentConnection()
	sqlText = string.format(sqlText .. ";", dbName, dbUser)
	util.printInfo("creating database: " .. sqlText)
	local _, info = dsql.sqlExecuteUnsafe(sqlText, nil, nil, option)
	dconn.disconnect(conn) -- conn param is mandatory
	if not info.error then
		conn = dconn.connection({organizationId = organizationId}) -- creates database if - it does not exist - using fallback connection
		if conn == nil then
			return util.printError("database '%s' organization id '%s' connection failed after database creation", dbName, organizationId)
		end
		createTriggerAndAuditLog()
	end
	-- prfName = "table/prf/group.json"
	return info
end

local function duplicateDatabase(currentVersionNumber, database)
	-- if true then return end
	local ret, info, sqlText
	local dbName = database .. "_v" .. tostring(currentVersionNumber)
	--[[
  sqlText = "pg_dump -U manage "..database.." > "..dbName.."_dump.sql"
  print(sqlText.."\n")
	ret, info = db.sqlExecuteUnsafeArray(loc.conn, sqlText)
	if getError(info) or getError(ret) then
		return {error = getError(info) or getError(ret)}
	end
  --]]

	sqlText = "CREATE DATABASE " .. dbName .. " WITH TEMPLATE " .. database
	print(sqlText .. "\n")
	local conn = dconn.connection()
	dprf.closePreferenceConnection()
	dconn.disconnectAll()
	dconn.connection({organizationId = "admin-postgres-0"})
	dprf.closePreferenceConnection() -- connect may cause a new preference connection, close it again
	ret, info = dsql.sqlExecuteUnsafe(sqlText)
	dconn.disconnectAll()
	dconn.connection({organizationId = conn.organization_id})
	if getError(info) or getError(ret) then
		return {error = getError(info) or getError(ret)}
	end
	-- sqlText = "SET search_path TO "..dbName
	-- sqlText = "SELECT COUNT(*) FROM preference"
	-- addToSqlTextArr(sql, sqlText), table.concat(sql)
	sqlText = "SELECT 1 FROM pg_database WHERE datname='" .. dbName .. "'"
	ret, info = dsql.sqlExecuteUnsafe(sqlText, {"count"}, {"integer"})
	if getError(info) or getError(ret) then
		return {error = getError(info) or getError(ret)}
	elseif ret == nil or ret[1] == nil or ret[1].count == nil or tonumber(ret[1].count) ~= 1 then
		return {error = l("duplicated database '%s' does not exist", dbName)}
	end
	return
end

local function printChangeLog(database)
	if type(loc.changeLog) == "table" and #loc.changeLog > 0 then
		local logTextArr = {}
		fn.iter(loc.changeLog):each(function(rec)
			local logText
			if rec.field_change == nil then
				logText = "  " .. l("log table 'field_change' -tag was not found")
			elseif rec.field_change.table_name == nil then
				logText = "  " .. l("log table 'table_name' -tag was not found")
			elseif rec.field_change.log_type == nil then
				logText = "  " .. l("log table 'log_type' -tag was not found")
			elseif rec.field_change.log_type == "remove_table_before_add" then
				logText = ""
			elseif rec.field_change.log_type == "add_table" then
				logText = "  " .. l("add table") .. ": '" .. tostring(rec.field_change.table_name) .. "'"
			elseif rec.field_change.log_type == "rename_table" then -- change table name
				logText = "  " .. l("change table name") .. " '" .. tostring(rec.field_change.table_name) .. "' -> '" .. tostring(rec.field_change.old_table_name) .. "'"
			elseif rec.field_change.log_type == "remove_table" then
				logText = "  " .. l("remove table") .. ": '" .. tostring(rec.field_change.table_name) .. "'"
			elseif rec.field_change.log_type == "add_field" then
				logText = "  " .. l("add field") .. ": '" .. tostring(rec.field_change.table_name) .. "." .. tostring(rec.field_change.new.field_name) .. "'"
			elseif rec.field_change.log_type == "rename_field" then
				logText = "  " .. l("rename field") .. " '" .. tostring(rec.field_change.table_name) .. "." .. tostring(rec.field_change.old.field_name) .. "' -> '" .. tostring(rec.field_change.table_name) .. "." .. tostring(rec.field_change.new.field_name) .. "'"
			elseif rec.field_change.log_type == "change_field" then
				local changeTbl = {}
				local oldType = rec.field_change.old.change_to_value or rec.field_change.old.field_type
				if rec.field_change.new.field_type and oldType and rec.field_change.new.field_type ~= oldType then
					addToSqlTextArr(changeTbl, "  " .. l("type") .. " '" .. tostring(oldType) .. "' -> '" .. tostring(rec.field_change.new.field_type) .. "'", "\n")
				end
				if rec.field_change.new.field_length and rec.field_change.old.field_length and rec.field_change.new.field_length ~= rec.field_change.old.field_length then
					addToSqlTextArr(changeTbl, "  " .. l("length") .. " '" .. tostring(rec.field_change.old.field_length) .. "' -> '" .. tostring(rec.field_change.new.field_length) .. "'", "\n")
				end
				if rec.field_change.new.nullable ~= rec.field_change.old.nullable then
					addToSqlTextArr(changeTbl, "  " .. l("nullable") .. " '" .. tostring(rec.field_change.old.nullable) .. "' -> '" .. tostring(rec.field_change.new.nullable) .. "'", "\n")
				end
				logText = "  " .. l("change field") .. " '" .. tostring(rec.field_change.table_name) .. "." .. tostring(rec.field_change.new.field_name) .. ":\n" .. table.concat(changeTbl)
			elseif rec.field_change.log_type == "remove_field" then
				logText = "  " .. l("remove field") .. ": '" .. tostring(rec.field_change.table_name) .. "." .. tostring(rec.field_change.delete.field_name) .. "'"
			else
				logText = "  " .. l("unknown operation") .. ": " .. tostring(rec.field_change.sql_txt)
			end
			if logText ~= "" then
				logTextArr[#logTextArr + 1] = logText .. "\n"
			end
		end)
		util.printInfo("\ndatabase '%s' changes:", database)
		util.print('%s', table.concat(logTextArr))
	end
end

-- --database-postgre.lua
-- local time = util.seconds()
-- local createCount = 0
function doperation.checkDatabase(conn, tableNameRecTypeArr, option) -- TODO: fix dbPref.postgre.connection
	conn = conn or dconn.currentConnection()
	if conn.dbtype == "4d" or util.from4d() then
		return
	end
	local time = util.seconds()
	option = option or {}
	if option.audit_log == nil then
		option.audit_log = conn.schema == "" -- creating audit log to external database fails
	end
	local dbName = conn and conn.database
	if not dconn.allowDatabaseSchemaUpdate(dbName) then
		if dconn.allowDatabaseSchemaUpdate(dbName) == nil then
			util.printWarning("  check database structure is not allowed for database '%s', please fix preference organization allow_database_schema_update value", dbName)
		else
			util.printWarning("  check database structure is disabled for database '%s', check preference organization allow_database_schema_update value", dbName)
		end
		return
	end
	local dbIdArr = dbName and {[dbName] = true}
	-- local databaseArr = fn.util.mapFieldValue(dbPref.connection[option.connection.connection], "database"):toDistinctArr()
	-- for _, database in ipairs(databaseArr) do
	local checkTableCount = 0
	for database, allow in pairs(dbIdArr) do
		if allow and (doperation.databaseUpdateStatus(database, "start") == false or option.force_check) then
			checkTableCount = checkTableCount + 1
			if checkTableCount == 1 then
				util.printInfo("* check database structure for database '%s'", dbName)
			end
			local saveVersion = false
			local organizationId = conn.organization_id
			loc.changeLog = {}
			loc.sqlTextArr = {}
			checkConnection(organizationId, conn)
			if loc.fieldType == nil then
				loc.fieldType = dprf.prf(prefPath .. "/prf/field_type.json", prfOption).field_type
				loc.fieldTypeTbl = fn.iter(loc.fieldType or {}):reduce(function(acc, item)
					acc[item.value] = item
					return acc
				end, {})
			end

			-- check if version is new, prf/version_update.json vs version_update -record
			local ret = versionUpdate(false, organizationId, database)
			if ret and ret.error then
				util.printError(tostring(ret.error))
				util.printInfo("please close the program with crtl-C twice")
				readConsole()
			elseif ret and ret.current_version == nil then
				util.printError("current version number was not found")
				util.printInfo("please close the program with crtl-C twice")
				readConsole()
			elseif ret and ret.update ~= true and not option.force_check then
				util.printOk("  database '%s', current structure version '%s', new version '%s', version hash '%s'", tostring(database), tostring(ret.current_version), tostring(ret.new_version), tostring(ret.hash))
			elseif ret and (ret.update == true or option.force_check) then
				-- check table changes between current version and new wersion
				--   run create_table_settings.json
				util.printInfo("  database '%s', current structure version '%s', new version '%s', version hash '%s', force check '%s'", tostring(database), tostring(ret.current_version), tostring(ret.new_version), tostring(ret.hash), tostring(option.force_check or false))
				local groupPrfName = "table/prf/group.json"
				local groupPrfLocal = util.readUpperLevelPreferenceFile(groupPrfName, "no-db use-default")
				if not groupPrfLocal.group then
					util.printError("preference file was not found: %s", groupPrfName)
					return
				end
				local schema = conn.schema or ""
				local groupPrfExternal
				if schema ~= "" then
					groupPrfName = "table/external/" .. schema .. "/prf/group.json"
					groupPrfExternal = util.readUpperLevelPreferenceFile(groupPrfName, "no-db use-default")
					if not groupPrfExternal.group then
						util.printError("preference file was not found: %s", groupPrfName)
						return
					end
				end
				local tableNameArr = {}
				local tableNameIdx = {}
				if tableNameRecTypeArr then
					for idx, nameRecType in ipairs(tableNameRecTypeArr) do
						local tblName, recType = dschema.splitRecTypeName(nameRecType)
						if schema ~= "" then
							tableNameArr[idx] = dschema.externalName(tblName, schema, recType) -- or tblName
							if tableNameArr[idx] == nil and option.force_check then
								tableNameArr[idx] = dschema.tableName(tblName, "", recType)
							end
						else
							tableNameArr[idx] = tblName
						end
						tableNameIdx[tableNameArr[idx]:lower()] = idx
					end
				end

				-- create only local schema or common tables for external schema
				if option.local_table_check ~= false and groupPrfLocal.group and #groupPrfLocal.group > 0 then
					for _, item in ipairs(groupPrfLocal.group) do
						local idx
						if tableNameRecTypeArr then
							idx = tableNameIdx[item.table_name:lower()]
						else
							idx = true
						end
						local create = conn.is_local and item.group == "common/" or (schema == "" and idx) or false
						if create == false and conn.local_group then
							create = fn.index(item.group, conn.local_group)
						end
						if create == false and idx and option.force_check then
							if schema == "" then
								create = true
							end
						end
						if create then
							if item.table_name ~= "audit.log" then
								if idx ~= nil or tableNameRecTypeArr == nil then
									local tblName = item.table_name
									local fileName = "table/" .. item.group .. tblName .. ".json"
									util.print("  comparing local database %s table '%s' to file '%s'", peg.parseBeforeLast(item.group, "/"), tblName, fileName)
									if not createTableFromFile(fileName, false, option, organizationId) then
										return false
									end
								elseif tableNameRecTypeArr == nil then
									util.print("    local table '%s' table was not created", item.table_name)
								end
							end
							-- else
							-- 	util.print("    local table '%s', external table was not created", item.table_name)
						end
					end
				end

				if groupPrfExternal and groupPrfExternal.group and (#tableNameArr > 0 or tableNameRecTypeArr == nil) then
					-- create external schema tables
					for _, item in ipairs(groupPrfExternal.group) do
						local extTblName
						local idx = tableNameIdx[item.table_name:lower()]
						if idx == nil then
							extTblName = dschema.externalName(item.table_name, schema)
							idx = tableNameIdx[extTblName:lower()]
						end
						if idx == nil and tableNameRecTypeArr ~= nil and option and option.missing_external_table_warning ~= false then
							util.printWarning("local table '%s', external table was not found", item.table_name)
						else
							local fileName = "table/" .. item.group .. item.table_name .. ".json"
							local tblName, recType
							if tableNameRecTypeArr and idx then
								tblName, recType = dschema.splitRecTypeName(tableNameRecTypeArr[idx])
							else
								tblName, recType = item.table_name, item.record_type or ""
							end
							local extTbl = dschema.externalRec(tblName, schema, recType)
							if extTbl == nil or extTbl.local_table == "" then
								util.printWarning("local table '%s', external table was not found", item.table_name)
							else
								tblName = extTbl.table_name -- find external table name from external group, like sales_order
								local group = util.arrayRecordField(groupPrfExternal.group, "table_name", tblName, "group")
								if group == nil then
									tblName = item.table_name -- find local table name from external group, like order-sales
									group = util.arrayRecordField(groupPrfExternal.group, "table_name", tblName, "group")
								end
								if group == nil then
									util.printWarning("local table '%s', external table was not found form group preference '%s'", item.table_name, groupPrfName)
									fileName = nil
								else
									fileName = "table/external/" .. schema .. "/" .. group .. tblName .. ".json"
								end
							end
							if fileName then
								util.print("  comparing schema '%s', database table '%s' to file '%s'", schema, tblName, fileName)
								if not createTableFromFile(fileName, false, option, organizationId) then
									util.printWarning("    local table '%s', external table '%s' was not created because of error", item.table_name, tostring(extTbl and extTbl.table_name))
								end
							else
								util.print("    local table '%s', external table '%s' was not created", item.table_name, tostring(extTbl and extTbl.table_name))
							end
						end
					end
				end

				if type(loc.changeLog) == "table" and #loc.changeLog > 0 then
					-- print changes
					printChangeLog(database)
					-- ask update
					--[[ repeat
						io.write(l("do you want to update postgre database [yes/no]?") .. " ")
						local confirm = "yes" -- readConsole()
						if confirm ~= "yes" then
							util.printInfo("please close the program with ctrl-C if you don't want to continue")
						end
					until confirm == "yes"
					-- ask logout users & restart server
					dconn.disconnectAll()
					print("\n")
					repeat
						-- util.printInfo("ask users to log out from database and restart postgre server")
						io.write("please ask all users to log out from database and restart postgre server" .. "\n"
								         .. "after you have done it, do you want to continue [yes/no]?" .. " ")
						local confirm = "yes" -- readConsole()
						io.write("\n")
						if confirm ~= "yes" then
							util.printInfo("please close the program with ctrl-C if you don't want to continue")
						end
					until confirm == "yes"
					checkConnection(conn) ]]
					-- todo: pg_dump

					-- backup database
					local saveRet
					if ret.current_version > 0 and option.backup_database ~= false then
						if conn.dbtype ~= "sqlite" then
							saveRet = duplicateDatabase(ret.current_version, database)
						end
					end
					if getError(saveRet) then
						util.printError(getError(saveRet))
						util.closeProgram()
					else
						if ret.new_version == ret.current_version then
							ret.new_version = ret.new_version + 1
						end
						saveRet = saveChanges(organizationId, ret.new_version) -- execute sql statements and save change log
						if getError(saveRet) == nil then
							saveVersion = true
						end
					end
				end
			end
			if saveVersion or ret == nil or ret.current_version == 0 then
				versionUpdate(true, organizationId, database) -- save new version number to prf-data
			end

			-- create table input and output json-files and pug-page model (drivlog_input1.json, drivlog_input1.pug ...)
			-- if false then
			-- util.printInfo(l"creating preferences from 'template' path")
			-- createPreference({"fmsseo"})
			---createPreference({"travelexp"})
			-- createPreference({"travelexp", "travelexpr", "expty", "drivlog"}) -- 2. param "replace" is optional.
			-- createPreference({"travelexp", "travelexpr", "expty", "drivlog"}, "replace")
			-- end
			doperation.databaseUpdateStatus(database, "done")
		end
	end

	if checkTableCount > 0 then
		util.print("  - check database structure time: %.3f", util.seconds(time))
	end
end

return doperation

--[[
if createCount > 0 then
	time = util.seconds(time)
	print()
	util.printInfo("create table run time: %.4f seconds", time)
end

local function createTableFromPath(rec, saveToLog)
	if rec.group == nil then
		return 0
	end
	local createCount = 0
	local path = "table/"..rec.group -- ..rec.table_name..".json"
	if fs.fileExists(path) == true then -- is file
		if createTableFromFile(path, saveToLog) == true then
			createCount = createCount + 1
		end
	else -- is path
		for filename in fs.dirTreeIter(path, {suffix = ".json"}) do
			-- local _, file = peg.splitLast(util.removeEndText(filename, ".json"), "/")
			if createTableFromFile(path..filename, saveToLog) == true then
				createCount = createCount + 1
			end
		end
	end
	return createCount
end

local function createRelation(name)
	return true
end


local function createTableFieldPreference(tablePrefix)
	local path = "table/"..tablePrefix..".json"
	if fs.fileExists(path) then
		return
	end
	local tableName = dschema.tableName(tablePrefix)
	if tableName ~= nil then
		local sqlTable = dqry.readTableStructure(tableName, "sql")
		if sqlTable == nil or #sqlTable == 0 then
			return
		end
		local defaultFieldPrf = dprf.prf("table/prf/default_field.json", prfOption)
		local fieldTypePrf = dprf.prf("table/prf/field_type.json", prfOption)
		local prf = {table_name = tableName, field = {}}
		for _, rec in ipairs(sqlTable) do
			local sqlField
			if defaultFieldPrf.field and #defaultFieldPrf.field > 0 then
				sqlField = util.arrayRecord(rec.field_name, defaultFieldPrf.field, "field_name")
			end
			if sqlField == nil then
				local fldtype = rec.field_type
				local fieldTypePrfRec = util.arrayRecord(rec.field_type, fieldTypePrf.field_type, "value")
				if fieldTypePrfRec and fieldTypePrfRec.change_to_value then
					fldtype = fieldTypePrfRec.change_to_value -- "character varying" -> "varchar"
				end
				prf.field[#prf.field + 1] = {name = rec.field_name, type = fldtype, length = rec.field_length}
			end
		end
		fs.writeFile(path, json.toJson(prf))
		return
	end
	return
end

local function replaceText(source, fromText, toValue)
	local retText
	if type(toValue) == "table" then
		local toValue = json.toJson(toValue)
		retText = peg.replace(source, "{\""..fromText.."\"}", toValue)
		retText = peg.replace(retText, "[\""..fromText.."\"]", toValue)
		retText = peg.replace(retText, '"'..fromText..'"', toValue)
		retText = peg.replace(retText, fromText, toValue)
	elseif type(toValue) == "string" then
		retText = peg.replace(source, '"'..fromText..'"', '"'..toValue..'"')
		retText = peg.replace(retText, fromText, toValue)
	else
		retText = peg.replace(source, '"'..fromText..'"', tostring(toValue))
		retText = peg.replace(retText, fromText, tostring(toValue))
	end
	return retText
end

local function createPreference(tablePrefixTbl, option) -- option = "replace" -> replaces old files
	local convertHtml = require "form/nc/nc-define/convert/convert-html-json"
	-- output
	-- input ma-row -> ma-label-input
	-- query
	-- script
	-- sequence?
	-- route
	-- input page?
	-- convert?
	-- oletusdata, ekaks syötetään, sitten struct talteen
	-- optionaalinen component init?
	-- template jokaiselle muutokselle?
	local fieldTypeTbl = dprf.prf("table/prf/field_type.json", prfOption)
	local templateFolder = util.runPath().."preference/template"

	local destinationCacheArr = {}
	local fieldListCacheArr = {}
	local pugRowCacheArr = {}

	local function create(prf, filename, tablePrefix)
		local pathPrefix = util.runPath().."preference/"
		filename = peg.replace(filename, pathPrefix, "")
		local containingPath, file = peg.splitLast(filename, "/")
		local _, containingFolder = peg.splitLast(containingPath, "/")
		containingPath = containingPath.."/"
		if peg.find(filename, ".json") > 0 and -- only json files
				peg.find(filename, "destination.json") < 1 and -- skip destination.json file
				peg.find(filename, containingFolder.."_field.json") < 1 and -- skip ???_field.json file
				peg.find(filename, "pug_row.json") < 1 then -- skip pug_row.json file

			-- load destination.json
			local destTbl, err
			if destinationCacheArr[containingPath] == nil then
				destTbl, err = dprf.prf(containingPath.."destination.json", prfOption)
				if err then
					util.printError(tostring(err))
					return nil
				end
				local jsonData = json.toJsonRaw(destTbl)
				jsonData = replaceText(jsonData, "{{table_prefix}}", tablePrefix)
				destTbl = json.fromJson(jsonData)
				destinationCacheArr[containingPath] = destTbl
			else
				destTbl = destinationCacheArr[containingPath]
			end

			-- convert "???_field.json"
			-- local fieldListJson = "{}"
			local fieldListArr = {}
			if fieldListCacheArr[containingPath] == nil then
				local path = containingPath..containingFolder.."_field.json"
				if fs.fileExists(pathPrefix..path) then
					local fieldListPrf, err = dprf.prf(path, prfOption)
					if err == nil then
						local jsonData = json.toJsonRaw(fieldListPrf)

						for _, fldRec in ipairs(prf.field) do
							local fileText = replaceText(jsonData, "{{field_name}}", tablePrefix.."."..fldRec.field_name)
							local typeRec = util.arrayRecord(fldRec.field_type, fieldTypeTbl.field_type, "value", 1)
							if typeRec and typeRec.default_width then
								fileText = replaceText(fileText, "{{default_width}}", typeRec.default_width)
							else
								fileText = replaceText(fileText, "{{default_width}}", 60)
							end
							local fieldHeader = tablePrefix.."."..fldRec.field_name
							fileText = replaceText(fileText, "{{field_name_lang}}", fieldHeader)
							local rec = json.fromJson(fileText)
							fieldListArr[#fieldListArr + 1] = rec
						end
						-- fieldListJson = json.toJson(fieldListArr)
					end
				end
				fieldListCacheArr[containingPath] = fieldListArr
			else
				fieldListArr = fieldListCacheArr[containingPath]
			end

			-- convert pug_row.json file
			local pugRowArr = {}
			if pugRowCacheArr[containingPath] == nil then
				local path = containingPath.."pug_row.json"
				if fs.fileExists(pathPrefix..path) then
					local fieldListPrf, err = dprf.prf(path, prfOption)
					if err == nil then
						local jsonData = json.toJsonRaw(fieldListPrf)
						local fileText = replaceText(jsonData, "{{pug_row_index}}", 1)
						fileText = replaceText(fileText, "{{pug_row_parent_index}}", 0)
						fileText = replaceText(fileText, "{{pug_row_child_index}}", 2)
						fileText = replaceText(fileText, "{{pug_row_tag}}", "nc-xmlroot")
						fileText = replaceText(fileText, "{{pug_row_ng_model}}", "")
						fileText = replaceText(fileText, "{{pug_row_value}}", "")
						fileText = replaceText(fileText, "{{pug_row_class_array}}", {})
						local rec = json.fromJson(fileText)
						pugRowArr[#pugRowArr + 1] = rec
						local i = 1
						for index, fldRec in ipairs(prf.field) do
							i = i + 1
							fileText = replaceText(jsonData, "{{pug_row_index}}", i)
							fileText = replaceText(fileText, "{{pug_row_parent_index}}", 1)
							fileText = replaceText(fileText, "{{pug_row_child_index}}", 0)
							fileText = replaceText(fileText, "{{pug_row_tag}}", "  ma-row\n    ma-label-input")
							fileText = replaceText(fileText, "{{pug_row_ng_model}}", "rec."..tablePrefix.."."..fldRec.field_name)
							-- fileText = replaceText(fileText, "{{pug_row_link_array}}", {})
							local fieldHeader = tablePrefix.."."..fldRec.field_name
							fileText = replaceText(fileText, "{{pug_row_value}}", fieldHeader)
							fileText = replaceText(fileText, "{{pug_row_class_array}}", {})
							rec = json.fromJson(fileText)
							pugRowArr[#pugRowArr + 1] = rec
						end
					end
				end
				pugRowCacheArr[containingPath] = pugRowArr
			else
				pugRowArr = pugRowCacheArr[containingPath]
			end

			local jsonData = dprf.prf(filename, prfOption)
			jsonData = json.toJsonRaw(jsonData)
			jsonData = replaceText(jsonData, "{{table_prefix}}", tablePrefix)
			jsonData = replaceText(jsonData, "{{field_array}}", fieldListArr)
			jsonData = replaceText(jsonData, "{{pug_row_array}}", pugRowArr)
			jsonData = replaceText(jsonData, "{{input_page_name}}", tablePrefix.."_input1")
			-- fs.writeFile("preference/test/convert_template.json", jsonData)
			local fileTbl = json.fromJson(jsonData)

			-- save folders and file
			if destTbl and destTbl.copy and destTbl.copy[file] then
				local path = pathPrefix..destTbl.copy[file]
				if option == "replace" or not fs.fileExists(path) then
					local containingPath = peg.splitLast(path, "/").."/"
					fs.createFilePath(containingPath)
					fs.writeFile(path, fileTbl)

					-- constant tag names: "pug.json", "input_pug"
					if true and peg.find(filename, "/pug.json") > 0 and destTbl.copy.input_pug then
						local path = pathPrefix..destTbl.copy.input_pug
						convertHtml.flatJsonToWebPage(fileTbl, path, "form/nc/nc-define/convert/convert-json-to-pug.json")
					end

				end
			end
		end
	end

	fn.iter(tablePrefixTbl):each(function(tablePrefix)
		local prf = dprf.prf(prefPath.."/"..tablePrefix..".json", prfOption)
		if prf and prf.table_name and prf.field then
			for filename1, attr1 in fs.dirTreeIter(templateFolder) do
				if attr1.mode == "directory" then
					if peg.find(filename1, "template/code/") == 0 then -- do not run code-folder when starting nc-server
						for filename, attr in fs.dirTreeIter(filename1) do
							create(prf, filename, tablePrefix)
						end
					end
				end
			end
		end
	end)
end

--]]
