--- lib/db/dprf.lua
-- Database preference reading and saving.
-- local dprf = require "dprf"
-- @module db
local dprf = {}

local util = require "util"
local json = require "json"
local peg = require "peg"
local fn = require "fn"
local execute = require "execute"
local recData = require"recdata".get
local coro = require "coro"
local useCoro = coro.useCoro(false)
local currentThread = coro.currentThreadFunction()
local threadId = coro.threadId
local l, dqry, dload, dsave, auth, dconn, hash, debug -- delay load

local lastPrfName = ""
local namedPreferenceCache = {}
local preferenceCache = {}
local preferenceConnection
if useCoro then
	preferenceConnection = {}
end
local savePref = {table = "preference", record_type = ""}

local function loadLibs()
	if not dqry then
		l = require"lang".l
		dqry = require "dqry"
		dload = require "dload"
		dsave = require "dsave"
		hash = require "hash"
		auth = require "auth" -- later
		dconn = require "dconn"
	end
end

local preferenceConn, thread -- temporary variables
local function getPreferenceConnection()
	if useCoro then
		thread = currentThread()
		return preferenceConnection[thread]
	end
	if preferenceConnection and (preferenceConnection.closed or not preferenceConnection.connected) then
		preferenceConnection = nil
	end
	return preferenceConnection
end

local function closePreferenceConnection()
	if useCoro then
		thread = currentThread()
		if preferenceConnection[thread] then
			if preferenceConnection[thread].connected then
				util.print("  closing preference connection for %s", threadId(thread))
				dconn.disconnect(preferenceConnection[thread])
			end
			preferenceConnection[thread] = nil
		end
	else
		if preferenceConnection then
			dconn.disconnect(preferenceConnection)
		end
		preferenceConnection = nil
	end
end
dprf.closePreferenceConnection = closePreferenceConnection

local function setPreferenceConnection()
	loadLibs()
	local function setConn(conn)
		if useCoro then
			thread = currentThread()
			preferenceConnection[thread] = conn
		else
			preferenceConnection = conn
		end
	end

	preferenceConn = getPreferenceConnection()
	local prevConn = dconn.getCurrentConnection()
	--[[ this not needed when we always create a new preference connection
	if prevConn == nil then
		prevConn = dconn.currentConnection() -- must create first connection before query
	end
	if prevConn == nil or prevConn.driver == nil then
		setConn(false)
		local info = {error = l("could not get default database connection, preference name '%s'", tostring(prfName))}
		util.printRed(info.error)
		return nil, {}, info
	end
	if preferenceConn and prevConn.dbtype ~= "4d" and (preferenceConn.database ~= prevConn.database or preferenceConn.info ~= prevConn.info) then
		setConn(nil)
	end ]]
	if not preferenceConn then
		local conn = dconn.setAuthAndRedirect("preference", "", {new_connection = true, reason = "preference"})
		if conn == nil then
			setConn(false)
			util.printError("could not get default preference database connection")
		end
		if conn and conn.driverTbl == nil then
			conn.driverTbl = dconn.driver(conn.driver)
		end
		setConn(conn)
		-- setConn({info = conn.info, conn = conn, driver = dconn and dconn.driver(), database = conn.database, organization_id = conn.organization_id}) -- , database = conn.database, organization_id = conn.organization_id
	end
	return prevConn
end

local doNotReplaceKey = {name = true, html = true, hdr = true, layout = true, javascript = true}
local function replaceRecParameter(rec, mainRec)
	for key, val in pairs(rec) do
		if not doNotReplaceKey[key] then
			if type(key) == "string" and key:sub(-1) == "-" then -- keys like "run-" are not in use
				rec[key] = nil
			else
				if type(val) == "table" then
					replaceRecParameter(val, mainRec)
				elseif type(val) == "string" then
					local tagName = peg.parseBetweenWithoutDelimiter(val, "{{", "}}")
					while tagName ~= nil and tagName ~= "" do
						if mainRec[tagName] ~= nil and type(mainRec[tagName]) ~= "table" then
							val = peg.replace(val, "{{" .. tagName .. "}}", mainRec[tagName]) -- simple fast case
							rec[key] = val
						else
							local tagNameTbl = peg.splitToArray(tagName, ".")
							local tagNameChanged = tagName
							for i = 2, #tagNameTbl do
								--[[ replace all main level tag names if they exist:
						  rec = {connection="woocommerce", "default_tab": {"woocommerce": "order_row-sales"}, val = {{default_tab.connection}} }
							{{default_tab.connection}} -> default_tab.woocommerce
							val -> "order_row-sales"
						]]
								if mainRec[tagNameTbl[i]] ~= nil and type(mainRec[tagNameTbl[i]]) ~= "table" then
									tagNameChanged = peg.replace(tagNameChanged, tagNameTbl[i], mainRec[tagNameTbl[i]])
								end
							end
							local newVal = recData(mainRec, tagNameChanged)
							if newVal ~= nil then
								val = peg.replace(val, "{{" .. tagName .. "}}", newVal)
								rec[key] = val
							else
								if tagName:sub(1, 1) ~= " " then
									util.printWarning("rec preference tag '%s' replace failed, replace value does not exist", val, lastPrfName) -- todo: test last prf name, maybe full path to tag?
								end
								rec[key] = val
								val = "" -- break loop
							end
						end
						tagName = peg.parseBetweenWithoutDelimiter(val, "{{", "}}")
					end
				end
			end
		end
	end
end
dprf.replaceRecParameter = replaceRecParameter

local preLoadJsonKey = util.invertTable({"rec", "arr", "hdr", "route", "data_query", "filter", "map", "reduce", "menu", "navigation", "shortcut", "output_action", "page", "toolbar", "sidebar", "sidebar2"})
local noPreloadJsonKey = util.invertTable({"name", "info", "query"})
function dprf.queryPreference(param1, option, setHash) -- deep query, load json keys
	if type(param1) ~= "table" and type(param1) ~= "string" then
		return util.parameterError("param", "table or string")
	end
	if param1.develop or param1.parameter and param1.parameter.refresh then
		if option == nil then
			option = "no-cache"
		else
			option = option .. " no-cache"
		end
	end
	local prfName = param1.name or param1
	lastPrfName = prfName
	-- local returnJson = param1["return"] and param1["return"].json
	-- local ret = {}
	local errTxt
	local infiniteLoopPreventArray = {}
	local function queryPref(param, key)
		if type(param) == "string" then
			local keyPos = peg.find(param, ".json/")
			if keyPos > 0 or peg.endsWith(param, ".json") then
				key = param .. " - " .. key
				if infiniteLoopPreventArray[key] ~= nil then
					if key ~= "table/prf/default_field.json - default_field" then -- many tables have default_field.json TODO: add do_not_warn_infinite_loop_key.json
						errTxt = l("preference name '%s', key '%s' has been loaded before", param1.name or "", key)
					end
					return infiniteLoopPreventArray[key], errTxt
				end
				infiniteLoopPreventArray[key] = true
				local prf2, err
				if keyPos > 0 then
					prf2, err = dprf.prf(param:sub(1, keyPos + 4), option, setHash)
					if not err then
						local key2 = param:sub(keyPos + 6)
						if prf2[key2] == nil then
							util.printWarning("preference name '%s', key '%s' does not exist", param, key2)
						end
						prf2 = prf2[key2]
					end
				else
					prf2, err = dprf.prf(param, option, setHash)
				end
				infiniteLoopPreventArray[key] = prf2
				if util.tableIsEmpty(prf2) and keyPos < 1 then
					errTxt = err or l("query preference '%s' was not found", param)
					return nil, errTxt
				end
				return prf2
			end
		end
		return param
	end
	local function queryPreferenceRec(param, key)
		local prf, err = queryPref(param, key)
		if not err and type(prf) == "table" then
			local function queryPreferenceTag(key2, val)
				if type(key2) == "string" and key2:sub(-1) == "-" then -- keys like "run-" are not in use
					return
				end
				if param.name == val then
					util.printError("preference '%s' loads itself in key '%s'", val, key2)
					return
				end
				if type(val) == "string" and not noPreloadJsonKey[key2] and not peg.endsWith(val, "-") and (peg.endsWith(val, ".json") or peg.found(val, ".json/")) then
					if not preLoadJsonKey[key2] then
						local name = type(param) == "table" and param.name or param
						util.printInfo("preloaded json key '%s', preference '%s'", tostring(key2), tostring(name))
					end
					local prf2, recursiveErr = queryPref(val, key2)
					if recursiveErr then
						util.printError(recursiveErr)
						return
					end
					prf[key2] = prf2
					if key2 == "rec" then
						if param1.parameter then -- replace rec keys with call parameter
							util.recToRec(prf[key2], param1.parameter, {copy_key = {name = false}})
						end
						if prf.rec.run then
							err = execute.runCode(prf.rec.run, prf.rec, prf.rec, param)
							if err then
								prf.error = util.printError("error in calling code '%s': '%s', query: '%s'", prf.rec.run, err, prf.name)
							end
						end
						replaceRecParameter(prf.rec, prf.rec)
					elseif key2 == "arr" then
						replaceRecParameter(prf[key2], prf.rec)
					end
					queryPreferenceRec(prf[key2], key2) -- recursive call
				end
			end
			-- rec must be converted first, it's values can be used in other tag replaces
			if prf.rec then
				queryPreferenceTag("rec", prf.rec)
			elseif param1.name == prf.name and param1.parameter and param1.parameter.rec then
				prf = util.tableCombine(prf, param1.parameter.rec, "no-error")
				param1.parameter.rec = prf
			end
			if prf.grid == nil then
				replaceRecParameter(prf, prf) -- replace {{tag}} -values, grid will be handled in queryGrid()
			end
			for key2, val in pairs(prf) do
				if type(key2) == "string" and key2 ~= "rec" and key2 ~= "name" and type(val) == "string" and peg.found(val, ".json") then
					queryPreferenceTag(key2, val)
				end
			end
		end
		return prf
	end
	local ret = queryPreferenceRec(prfName, prfName) -- fn.util.callArrayParamRecursive(param1, queryPreferenceRec, "table", "no-cache")
	if type(ret) == "string" then
		return {error = ret}
	end
	return ret
end

function dprf.printCache()
	local cacheName = {}
	for cacheId in pairs(preferenceCache) do
		for key in pairs(preferenceCache[cacheId]) do
			cacheName[#cacheName + 1] = cacheId .. " - " .. key
		end
	end
	for cacheId in pairs(namedPreferenceCache) do
		for key in pairs(namedPreferenceCache[cacheId]) do
			cacheName[#cacheName + 1] = "named: " .. cacheId .. " - " .. key
		end
	end
	util.print("* cached preference files: %s", table.concat(cacheName, ", "))
end

function dprf.registerCache(name, prfTable, option)
	if type(prfTable) ~= "table" then
		util.printError("dprf.registerCache parameter 1 type '%s' is not a table, cache name '%s'", type(prfTable), tostring(name))
		return
	end
	loadLibs()
	local cacheId = dconn.localConnectionId()
	namedPreferenceCache[cacheId] = namedPreferenceCache[cacheId] or {}
	if namedPreferenceCache[cacheId][name] and not util.tableIsEmpty(namedPreferenceCache[cacheId][name]) then
		if option == nil or not option.no_error then
			util.printError("named peference cache id '%s', name '%s' already exists", cacheId, name)
		end
	end
	namedPreferenceCache[cacheId][name] = prfTable
end

function dprf.clearCache()
	preferenceCache = {}
	local clearName = {}
	local prf
	for cacheId in pairs(preferenceCache) do
		for key in pairs(preferenceCache[cacheId]) do
			clearName[#clearName + 1] = cacheId .. " - " .. key
			prf = preferenceCache[cacheId][key]
			for key2 in pairs(prf) do
				prf[key2] = nil -- todo: test this if it changes the original
			end
			-- preferenceCache[cacheId][key] = nil
		end
		-- preferenceCache[cacheId] = nil
	end
	for cacheId in pairs(namedPreferenceCache) do
		for key in pairs(namedPreferenceCache[cacheId]) do
			clearName[#clearName + 1] = "named: " .. cacheId .. " - " .. key
			prf = namedPreferenceCache[cacheId][key]
			for key2 in pairs(prf) do
				prf[key2] = nil -- todo: test this if it changes the original
			end
			-- namedPreferenceCache[cacheId][key] = nil
		end
		-- namedPreferenceCache[cacheId] = nil
	end
	if #clearName > 0 then
		util.printOk("cleared preference cache tables:\n - %s", table.concat(clearName, "\n - "))
	end
	-- setPreferenceConnection(nil) -- TODO: fix this! must have own prefs for each user
end

function dprf.clearConnection()
	closePreferenceConnection()
	dprf.clearCache()
end

local prevPreferenceName = ""
function dprf.preferenceFromJsonName()
	return prevPreferenceName
end

local function queryPref(prfName, organizationId, option)
	if type(prfName) ~= "string" or prfName == "" then
		local info = {error = l("preference name '%s' is empty", tostring(prfName))}
		util.printError(info.error)
		return nil, {}, info
	end
	local prevConn = setPreferenceConnection()
	preferenceConn = getPreferenceConnection()
	if preferenceConn == false or preferenceConn == nil then
		return
	end
	if prevConn and prevConn.is_local and prevConn.organization_id ~= preferenceConn.organization_id then
		dprf.clearConnection()
		setPreferenceConnection()
		preferenceConn = getPreferenceConnection()
	end
	-- we must use driver.execute() because dqry.query() or dsql.sqlExecuteUnsafe() would cause huge amount of recursive calls
	-- also restoring changed connection is very hard, this is fastest and safest way
	local sql = "SELECT prf.json_data FROM preference prf WHERE prf.name_id = '" .. prfName .. "'"
	--[[ if organizationId then
		sql = sql.." AND prf.organization_id = "..tostring(organizationId)
	end ]]
	local cursor, err = preferenceConn.driverTbl.execute(sql, nil, preferenceConn) -- {organization_id = preferenceConn.organization_id})
	local sel, info
	if cursor == nil then
		if option and peg.found(option, "no-error") then
			info = l("getting preference '%s' from database '%s' table 'preference' failed, error '%s'", prfName, tostring(preferenceConn.info), tostring(err))
		else
			info = util.printError("getting preference '%s' from database '%s' table 'preference' failed, error '%s'", prfName, tostring(preferenceConn.info), tostring(err))
		end
	elseif preferenceConn == nil then
		util.printRed("* preference '%s' connection is nil", tostring(prfName))
	else
		sel, info = preferenceConn.driverTbl.selectionToRecordArray(cursor, {"prf.json_data"}, {table = "preference", table_prefix = "prf"})
	end
	local pref
	if sel and #sel > 0 then
		pref = sel[1].json_data
	end
	if pref then
		if debug == nil then
			debug = util.prf("system/debug.json").debug.dprf
		end
		if debug then
			if organizationId ~= nil then
				util.print("* preference '%s', organization '%s' was loaded from database '%s'", tostring(prfName), tostring(organizationId), tostring(preferenceConn.info))
			else
				util.print("* preference '%s' was loaded from database '%s'", tostring(prfName), tostring(preferenceConn.info))
			end
		end
	end
	if prevConn then
		dconn.restoreConnection(prevConn)
	end
	return pref, sel, info
end

function dprf.prfExistsInDatabase(prfName, organizationId)
	local pref = queryPref(prfName, organizationId)
	return pref
end

function dprf.combine(toTbl, fromTbl, prfName) -- add tags from fromTbl to toTbl
	if type(toTbl) ~= "table" or type(fromTbl) ~= "table" then
		return
	end
	for key, val in pairs(fromTbl) do
		if key == "override" then
			toTbl[key] = val
		elseif type(val) == "table" and type(toTbl[key]) == "table" then
			if util.tableIsEmpty(toTbl[key]) or toTbl[key][1] then
				toTbl[key] = val -- replace whole tag is tbl is array-type
			else
				dprf.combine(toTbl[key], val, prfName)
			end
		elseif type(toTbl[key]) == type(val) then
			toTbl[key] = val
		elseif toTbl[key] == nil then -- only original json can contain allow_undefined_key -tag
			if toTbl.allow_undefined_key == false then
				-- key not found and not assigned
				if key:sub(-1) ~= "-" then -- keys ending with "-" are disabled and not used or warned about
					util.printWarning("preference '%s' contains undefined key '%s' that is not allowed in this preference", tostring(prfName), tostring(key))
				end
			elseif toTbl.allow_undefined_key == true then
				toTbl[key] = val
			else
				if key:sub(-1) ~= "-" then -- keys ending with "-" are disabled and not used or warned about
					util.printWarning("preference '%s' contains undefined key '%s'", tostring(prfName), tostring(key))
					toTbl[key] = val -- key not found, but is still assigned
				end
			end
		else
			-- toTbl[key] = val -- is it ok to replace wrong type tag to toTbl
			util.printWarning("preference '%s' key '%s' type '%s' is wrong. It should be type of '%s'", tostring(prfName), tostring(key), type(val), type(toTbl[key]))
		end
	end
end

local function overrideMemberTbl(prf, devArray, findValue) -- findValue can be array
	if type(devArray) == "table" then
		-- for _,rec in ipairs(devArray) do
		if type(findValue) == "table" then
			for _, val in ipairs(findValue) do
				if fn.index(val, devArray) then
					return prf
				end
			end
		else
			if fn.index(findValue, devArray) then
				return prf
			end
		end
		-- end
	end
end

local function combineOverride(origPrf, options, prfName) -- options: {organization=organizationId, user=userId}
	if origPrf == nil or type(origPrf.override) ~= "table" or options == nil or type(options) ~= "table" then
		return origPrf
	end
	local prf = util.clone(origPrf)
	if options.user then
		loadLibs()
		options.group = auth.userGroup(options.user)
	end
	local overridePrf
	for _, rec in ipairs(prf.override) do
		if type(rec.apply_to) == "table" then
			if rec.apply_to.organization and options.organization then
				overridePrf = overrideMemberTbl(rec.preference, rec.apply_to.organization, options.organization)
				if overridePrf then
					dprf.combine(prf, overridePrf, prfName)
				end
			end
			if rec.apply_to.group and options.group then
				overridePrf = overrideMemberTbl(rec.preference, rec.apply_to.group, options.group)
				if overridePrf then
					dprf.combine(prf, overridePrf, prfName)
				end
			end
			if rec.apply_to.user and options.user then
				overridePrf = overrideMemberTbl(rec.preference, rec.apply_to.user, options.user)
				if overridePrf then
					dprf.combine(prf, overridePrf, prfName)
				end
			end
		end
	end

	return util.clone(prf)
end

function dprf.getFile(prfName, option, hashTbl)
	if type(prfName) ~= "string" then
		local value
		if type(prfName) == "table" then
			value = json.toJsonRaw(prfName)
		else
			value = tostring(prfName)
		end
		return nil, l("preference name type '%s' is not string, value '%s'", type(prfName), value)
	end
	local pref, err = util.getFile("preference", prfName, option, hashTbl) -- will call hash.setHash()
	if pref == nil and err == nil then -- err == nil when file was not found
		pref, err = util.getFile("", prfName, option, hashTbl)
	end
	return pref, err
end

---@return table
function dprf.prf(prfName, option, hashTbl)
	loadLibs()
	prevPreferenceName = prfName
	local tagName
	if peg.found(prevPreferenceName, ".json/") then
		prfName = peg.parseBeforeWithDivider(prevPreferenceName, ".json")
		tagName = peg.parseAfter(prevPreferenceName, ".json/")
	end
	local cacheName = prfName
	local clearCache = false
	if option and peg.found(option, "clear-cache") then
		clearCache = true
	end
	if clearCache then -- option and option:found("clear-cache") then
		preferenceCache = {}
		if not prfName then -- only clear cache
			return {} -- must always return some table
		end
	end
	local cache = true
	if option and peg.found(option, "no-cache") then
		cache = false
	end
	local db = true
	if option and peg.found(option, "no-db") then
		db = false
	end
	local pref, err -- pref: always return {}
	local cacheId
	if cache and db then
		local conn = dconn.getLocalConnection()
		if conn then
			cacheId = conn.organization_id
		else
			cacheId, conn = dconn.localConnectionId() -- this creates a local connection if needed
			local sock = type(conn) == "table" and conn.socket
			if sock then
				if sock.info then
					sock = ", socket: " .. sock.info
				else
					sock = ", socket:" .. tostring(sock)
				end
			end
			util.print("    created preference connection for %s%s", conn and conn.thread and threadId(conn.thread) or "main thread", sock or "")
		end
		if preferenceCache[cacheId] == nil then
			preferenceCache[cacheId] = {}
		end
		pref = preferenceCache[cacheId][cacheName]
		if pref then
			pref = util.clone(pref)
		end
	end
	if not (cache and pref) then
		local showErr = true
		if option and peg.found(option, "no-error") then
			showErr = false
		end
		-- in future code: query all prf beginning same string (like-query) and join them together
		if not option or not peg.found(option, "only-db") then
			pref, err = dprf.getFile(prfName, option, hashTbl)
		end
		local prefData
		if pref == nil then
			if option and peg.found(option, "no-db") then
				prefData = false
			else
				prefData = queryPref(prfName, nil, option) or false -- , organizationId)
			end
			if prefData == false then
				err = err or string.format("preference '%s' was not found", prfName) -- we can't use lang l() here, this is too early in the load chain, it will cause bad load loops
			else
				pref = {} -- combine later with prefData
			end
		elseif type(pref) == "table" and util.tableIsEmpty(pref) then
			err = err or string.format("preference '%s' was invalid json", prfName)
		end
		if err then
			if showErr then
				util.printError(err)
			end
		else
			if option == nil or not peg.found(option, "no-db") then
				if prefData == false then
					prefData = nil
				elseif prefData == nil then
					prefData = queryPref(prfName, nil, option) -- , organizationId
				end
				if type(pref) == "table" then
					if prefData ~= nil then
						prefData = hash.setHash(prefData, prfName, option, hashTbl)
					end
					if prefData and not util.tableIsEmpty(prefData) then
						if prefData.override == true or util.tableIsEmpty(pref) then
							pref = prefData
						else
							dprf.combine(pref, prefData, prfName)
						end
					end
				end
			end
			if not pref then
				if showErr then
					err = l("preference '%s' was found but type was invalid: '%s'", prfName, type(pref))
					util.printWarning(err)
				end
				pref = {} -- must always return some table
			end
		end

		if pref ~= nil and err == nil and cache and cacheId then
			if preferenceCache[cacheId] == nil then
				preferenceCache[cacheId] = {}
			end
			preferenceCache[cacheId][cacheName] = util.clone(pref)
		end
	end -- if cache

	-- dbName = auth.currentDatabase()
	if type(pref) == "table" and db then
		local organizationId = auth.currentOrganizationNumber()
		local userId = auth.currentUserId()
		pref = combineOverride(pref, {organization = organizationId, user = userId}, prfName)
		if false and pref.preload_tag then
			pref = dprf.queryPreference(pref)
		end
	end
	if pref == nil then -- this function has to return table
		pref = {}
	end
	if tagName then
		return pref[tagName]
	end
	return pref, err
end

function dprf.prfKey(prfName, option, hashTbl)
	local name = peg.parseBefore(prfName, ".json/") .. ".json"
	local prf = dprf.prf(name, option, hashTbl)
	prf = recData(prf, prfName:sub(#name + 2))
	return prf
end

function dprf.preferenceFromJson(prfName, option, hashTbl)
	loadLibs()
	prevPreferenceName = prfName
	local cacheName = prfName
	local organizationId = auth.currentOrganizationId()
	local userId = auth.currentUserId()
	local fromData = true
	-- add group here
	if userId ~= "" then
		cacheName = cacheName .. " user=" .. userId
	end
	-- add group here
	if option == nil or option == nil or peg.find(option, "use-organization") <= 0 then
		organizationId = 0
	else
		cacheName = cacheName .. " org=" .. tostring(organizationId)
	end

	local clearCache = false
	if option and peg.find(option, "clear-cache") > 0 then
		clearCache = true
	end
	if clearCache then -- option and option:find("clear-cache") then
		dprf.clearCache()
		if not prfName then -- only clear cache
			return
		end
	end
	local showErr = true
	if option and peg.find(option, "no-error") > 0 then
		showErr = false
	end
	local cache = true
	if option and peg.find(option, "no-cache") > 0 then
		cache = false
	end
	local cacheId
	if cache then
		cacheId = dconn.localConnectionId()
		if preferenceCache[cacheId] and preferenceCache[cacheId][cacheName] and (option == nil or not peg.found(option, "get-all")) then
			-- print("preferenceFromJson fron cache: "..prfName)
			return util.clone(preferenceCache[cacheId][cacheName])
		end
	end
	local pref = {} -- always return {}
	local err, sel, info

	if option and peg.find(option, "get-all") > 0 then
		local getAllOption
		if showErr == false and cache == false then
			getAllOption = "no-error no-cache"
		elseif showErr == false then
			getAllOption = "no-error"
		elseif cache == false then
			getAllOption = "no-cache"
		end
		local prf2, fromPg
		prf2 = dprf.preferenceFromJson(prfName, getAllOption) -- get prf from file, TODO: wrong option? , same line as #702
		if prf2 ~= nil and prf2 ~= {} then
			pref[#pref + 1] = prf2
		end
		prf2, err, fromPg = dprf.preferenceFromJson(prfName, getAllOption) -- get zero organization prf
		if fromPg == true and prf2 ~= nil and prf2 ~= {} then
			pref[#pref + 1] = prf2
		end
		if organizationId ~= 0 then
			local orgOption = "use-organization"
			if getAllOption then
				orgOption = orgOption .. " " .. tostring(getAllOption)
			end
			prf2, err = dprf.preferenceFromJson(prfName, orgOption) -- get organization prf
			if prf2 ~= nil and prf2 ~= {} then
				pref[#pref + 1] = prf2
			end
		end
	else
		pref, sel, info = queryPref(prfName, organizationId)
		if pref ~= nil then
			pref = hash.setHash(pref, prfName, option, hashTbl)
		end
		if sel == nil or util.tableIsEmpty(sel) then
			pref = util.getFile("preference", prfName, option, hashTbl) -- will call hash.setHash()
			if pref == nil then
				pref = util.getFile("", prfName, option, hashTbl)
				-- pref may be nil here
			end
			fromData = false
		end

		if pref == nil or util.tableIsEmpty(pref) and fromData then
			if info and info.error then
				err = l("preference '%s' was not found, error: %s", prfName, info.error)
			else
				err = l("preference name '%s' was not found", prfName)
			end
			if showErr then
				util.printError(err)
			end
		elseif util.tableIsEmpty(pref) and fromData and sel[1] then
			pref = sel[1].json_data -- "{"..sel[1].json_data.."}"
			if type(pref) == "string" then
				pref = hash.setHash(pref, prfName, option, hashTbl)
			end
			if not pref then
				if showErr then
					err = l("preference '%s' was found but type was invalid: '%s'", prfName, type(sel[1].json_data))
					util.print(err)
				end
				pref = {} -- must always return some table
			end
		elseif util.tableIsEmpty(pref) and fromData and #sel ~= 1 then
			err = l("returned selection size is not 1: ") .. #sel
			util.printError(err)
		end

		if pref ~= nil and err == nil and cache then
			preferenceCache[cacheId] = preferenceCache[cacheId] or {}
			preferenceCache[cacheId][cacheName] = pref
		end
	end
	if pref == nil then -- this function has to return table
		pref = {}
	end
	--[[
	if conn4d and util.tableIsEmpty(pref) then
		dprf.copyPrefFrom4dToPostgre(conn4d, c, prfName, organizationId)
	end]]
	return pref, err, fromData
end

function dprf.preferenceToJsonSave(prfName, object)
	local err
	dqry.query("", "prf.name_id", "=", prfName)
	local sel = dload.selectionToRecordArray({"prf.name_id", "prf.json_data"})
	if not sel then
		sel = {}
	end
	if #sel < 1 then
		sel[1] = {name_id = prfName}
		sel[1].json_data = json.toJsonRaw(object)
		-- print("recordTableToSelection new")
		local _
		_, err = dsave.saveToDatabase(sel, savePref) -- dsave.recordTableToSelection(c, sel, "prf.name_id", "insert")
	else
		sel[1] = {name_id = prfName}
		sel[1].json_data = json.toJsonRaw(object)
		-- print("recordTableToSelection old")
		local _
		_, err = dsave.saveToDatabase(sel, savePref) -- dsave.recordTableToSelection(c, sel, "prf.name_id", {sel[1].name_id})
	end
	return err
end

function dprf.save(name, data)
	local err
	if type(data) ~= "table" then
		err = util.printError("dprf.save() data is not a table")
		return err
	end
	err = dprf.delete(name)
	if not err then
		local sel = {}
		sel[1] = {}
		sel[1].name_id = name
		sel[1].json_data = data
		local _
		_, err = dsave.saveToDatabase(sel, savePref)
	end
	if err then
		util.printError(err)
		return err
	end
end

function dprf.cleanPreferenceTableData() -- FIX: - add prf.preference_state-field to not prference table
	local function checkPrf(jsonData, filePrf, cleanedPrf, prfName)
		if filePrf and jsonData then
			for key, val in pairs(jsonData) do
				local fileTagVal = filePrf[key]
				if key == "override" then
					cleanedPrf[key] = val
				elseif fileTagVal == nil then
					util.printWarning(l("preference '%s' key '%s' does not exist", tostring(prfName), tostring(key)))
				elseif type(val) ~= type(fileTagVal) then
					util.printWarning(l("preference '%s' key '%s' type '%s' is wrong. It should be type of '%s'", tostring(prfName), tostring(key), type(fileTagVal), type(val)))
				elseif type(val) == "table" and type(fileTagVal) == "table" then
					local lowerTbl = {}
					if util.tableIsEmpty(fileTagVal) or fileTagVal[1] then -- tbl is array-type - array checking is difficult
						lowerTbl = val
					else
						checkPrf(val, fileTagVal, lowerTbl, prfName)
					end
					if not util.tableIsEmpty(lowerTbl) then
						cleanedPrf[key] = lowerTbl
					end
				elseif val ~= fileTagVal then
					cleanedPrf[key] = val
				end
			end
		end
	end

	-- dqry.clearQuery()
	local sel = dload.selectionToRecordArray({"prf.name_id", "prf.json_data", "prf.record_id"}) -- sel, info =
	-- local notInUse = {}
	local update = {}
	for _, rec in pairs(sel) do
		-- local prf, err = util.prf(rec.name_id)
		local pref = util.getFile("preference", rec.name_id)
		if pref == nil then
			pref = util.getFile("", rec.name_id)
		end
		if pref then
			local jsonData = json.fromJson(rec.json_data)
			local cleanedPrf = {}
			checkPrf(jsonData, pref, cleanedPrf, rec.name_id)
			if cleanedPrf == nil or util.tableIsEmpty(cleanedPrf) then
				-- notInUse[#notInUse + 1] = {record_id = rec.record_id}
				update[#update + 1] = {preference_state = "not in use", record_id = rec.record_id}
			else
				update[#update + 1] = {["prf.json_data"] = json.toJson(cleanedPrf), ["prf.info"] = rec.json_data, ["prf.record_id"] = rec.record_id}
			end
			-- for testing save old json to info-field
		end
	end

	-- save update-arr to prf-data
	-- if update-rec = {} then delete prf from data?
end

function dprf.delete(name)
	dqry.query("", "prf.name_id", "=", name, "")
	local err = dsave.deleteSelection("preference")
	return err
end

return dprf
