--- lib/db/dqjson.lua
-- Database json queries.
-- local dqjson = require "dqjson"
-- @module db
local dqjson = {}

local util = require "util"
local json = require "json"
local peg = require "peg"
local l = require"lang".l
local fn = require "fn"
local dt = require "dt"
local recData = require"recdata".get
local concat = table.concat
local dschema, dsql, qry, dqry, dprf, dconn, dload, execute
local loc = {}
loc.lastQueryJsonName = nil

function dqjson.loadLibs()
	if dprf == nil then
		dschema = require "dschema"
		dsql = require "dsql"
		dqry = require "dqry"
		dprf = require "dprf"
		dconn = require "dconn"
		dload = require "dload"
	end
end

local runQueryCalled = {}
local queryJsonSql
local allowedJoinTypes = util.invertTable({"INNER JOIN", "FULL OUTER JOIN", "LEFT OUTER JOIN", "RIGHT OUTER JOIN"})

local function concatTableValue(val)
	if val[1] and type(val[1]) ~= "string" then
		return concat(val, ",")
	else
		return "'" .. concat(val, "','") .. "'"
	end
end

local function limitAmount(queryTbl, err)
	-- LIMIT
	qry = qry or require "qry"
	local limit = qry.getLimit(queryTbl.limit)
	-- limit and offset could be parameter
	-- if tostring(tonumber(queryTbl.limit)) ~= queryTbl.limit then
	if type(limit) == "string" and queryTbl.parameter and queryTbl.parameter[limit] ~= nil then
		limit = queryTbl.parameter[limit]
	end
	if type(limit) == "string" and limit:lower() == "null" then
		limit = nil -- no limit
	elseif type(limit) ~= "number" then
		util.addError(err, l("query limit is not a number, query: ") .. json.toJsonRaw(queryTbl))
		limit = nil
	elseif limit < 1 then
		util.addError(err, l("query limit is smaller than 1, query: ") .. json.toJsonRaw(queryTbl))
		limit = nil
	end
	return limit
end

local function offsetAmount(queryTbl, err)
	-- OFFSET
	local offset = queryTbl.offset
	if type(offset) == "string" and queryTbl.parameter and queryTbl.parameter[offset] ~= nil then
		offset = queryTbl.parameter[offset]
	end
	if type(offset) == "string" and offset:lower() == "null" then
		offset = nil -- no offset
	elseif type(offset) ~= "number" then
		util.addError(err, l("query offset is not a number, query: ") .. json.toJsonRaw(queryTbl))
		offset = nil
	elseif offset < 0 then
		util.addError(err, l("query offset is smaller than 1, query: ") .. json.toJsonRaw(queryTbl))
		offset = nil
	end
	return offset
end

function dqjson.lastQueryJsonName() -- continue == "and", "or", "in", ?
	return loc.lastQueryJsonName or ""
end

local function runOrder(queryTbl, err, schema)
	if #err == 0 and queryTbl.order then
		if type(queryTbl.order) ~= "table" then
			util.addError(err, "order is not a table")
		else
			local order = queryTbl.order
			if type(order[1]) == "table" then
				order = fn.iter(order):reduce(function(acc, item)
					acc[#acc + 1] = item.field
					acc[#acc + 1] = item.order
					return acc
				end, {})
			end
			if util.fmod(#order, 2) == 1 then
				util.addError(err, l("order array size is not even number, order array: ") .. json.toJsonRaw(order))
			else
				local useExternalField = queryTbl.parameter and queryTbl.parameter.use_external_field
				for i = 1, #order, 2 do
					local field = order[i]
					if not useExternalField and not dschema.isField(field, schema) then
						dschema.isField(field, schema)
						util.addError(err, l("query order field '%s' is not valid", tostring(order[i])))
					else
						local errTbl = dqry.orderBy(field, order[i + 1])
						if errTbl then
							util.addError(err, errTbl)
						end
					end
				end
			end
		end
	end
end

local function sqlAdd(txt)
	queryJsonSql[#queryJsonSql + 1] = txt
end

local function sqlReplace(from, to)
	for i, row in ipairs(queryJsonSql) do
		queryJsonSql[i] = peg.replace(row, from, to)
	end
end

local runArrayQuery -- forward declaration of function
local runQueryByType -- forward declaration of function

local function runSqlQuery(continue, err, queryTbl, parameterTbl, schema)
	-- SELECT fields
	sqlAdd("SELECT") -- select exists if we are here
	if queryTbl.select == "*" then
		sqlAdd("*")
	elseif queryTbl.select then -- may be column
		for i, fld in ipairs(queryTbl.select) do
			if dschema.isField(fld, schema) == nil then
				util.addError(err, l("query select field '%s' is not valid ", fld))
				return
			elseif i < #queryTbl.select then
				sqlAdd(dschema.fieldNamePrefixSql(fld, schema) .. dsql.sqlAlias() .. fld .. ",")
			else
				sqlAdd(dschema.fieldNamePrefixSql(fld, schema) .. dsql.sqlAlias() .. fld)
			end
		end
	end
	-- FROM tbl
	if not queryTbl.from then
		util.addError(err, l("query does not contain from-tag"))
		return
	elseif peg.endsWith(queryTbl.from, ".json") then
		local subQuery, err2 = dprf.prf(queryTbl.from)
		if err2 then
			util.addError(err, l("subquery preference '%s' was not valid, error '%s'"), queryTbl.from, err2)
			return
		end
		subQuery.is_subquery = true
		queryTbl.has_subquery = true
		if not subQuery.field then
			subQuery.field = queryTbl.field -- to prevent errors
		end
		sqlAdd("FROM\n(")
		local ret = dqjson.queryJson(subQuery, parameterTbl, continue) -- recursive call
		if ret.error then
			util.addError(err, l("subquery '%s' returned error '%s'"), queryTbl.from, ret.error)
			return
		end
		sqlAdd("\n)")
	else
		sqlAdd("\nFROM " .. queryTbl.from .. dsql.sqlAlias() .. dschema.tablePrefixSql(queryTbl.from))
	end
	-- AS from_alias
	if queryTbl.from_alias then
		if type(queryTbl.from_alias) ~= "string" then
			util.addError(err, l("from_alias '%s' type is not string"), tostring(queryTbl.from_alias))
			return
		else
			sqlAdd(dsql.sqlAlias():sub(2) .. queryTbl.from_alias)
		end
	end
	-- JOIN
	if queryTbl.join_type then
		if allowedJoinTypes[queryTbl.join_type] == nil then
			util.addError(err, l("join_type '%s' is not valid"), tostring(queryTbl.join_type))
			return
		end
		local tblPrefix = dschema.tablePrefixSql(queryTbl.join_table)
		if tblPrefix == nil then
			util.addError(err, l("join_table '%s' is not valid"), tostring(queryTbl.join_table))
			return
		end
		if type(queryTbl.join_where) ~= "table" then
			util.addError(err, l("join_where '%s' is not a table"), tostring(queryTbl.join_where))
			return
		end
		sqlAdd("\n" .. queryTbl.join_type)
		sqlAdd(queryTbl.join_table .. dsql.sqlAlias() .. tblPrefix)
		dsql.initWhereArr()
		runArrayQuery(queryTbl.join_where, "", err, queryTbl, parameterTbl, schema)
		local where = concat(dsql.getWhereArr()) -- conn.sql.whereArr)
		sqlAdd("\nON " .. where)
	end
	-- WHERE
	if #err == 0 and queryTbl.query then
		dsql.initWhereArr()
		runArrayQuery(queryTbl.query, "", err, queryTbl, parameterTbl, schema)
		local where = concat(dsql.getWhereArr())
		if queryTbl.has_subquery then
			local tbl = dsql.getWhereUsedTableArr()
			for _, tblPrefix in ipairs(tbl) do
				local prefix = dschema.tablePrefixSql(tblPrefix)
				where = peg.replace(where, prefix .. ".", prefix .. "_")
			end
		end
		sqlAdd("\nWHERE " .. where)
	end
	if #err == 0 and queryTbl.join_table then
		local tblPrefix = dschema.tablePrefixSql(queryTbl.join_table)
		sqlReplace(queryTbl.join_table .. ".", tblPrefix .. ".") -- replace long join table names to prefix
	end
	-- ORDER BY
	if #err == 0 and queryTbl.order then
		dsql.initOrderArr() -- conn.sql.orderArr = {}
		runOrder(queryTbl, err)
		local order = concat(dsql.getOrderArr(), ", ") -- conn.sql.orderArr, ", ")
		local tbl = dsql.getOrderUsedTableArr()
		for _, tblPrefix in ipairs(tbl) do
			local prefix = dschema.tablePrefixSql(tblPrefix, schema) -- , recordType
			order = peg.replace(order, prefix .. ".", prefix .. "_")
			local tblName = dschema.tableNameSql(tblPrefix, schema) -- , recordType
			order = peg.replace(order, tblName .. ".", prefix .. "_")
		end
		sqlAdd("\nORDER BY " .. order)
	end
	-- LIMIT
	if #err == 0 and queryTbl.limit then
		local limit
		limit = limitAmount(queryTbl, err)
		if limit then
			sqlAdd("\nLIMIT " .. limit)
		end
	end
	-- OFFSET
	if #err == 0 and queryTbl.offset then
		local offset
		offset = offsetAmount(queryTbl, err)
		if offset then
			sqlAdd("\nOFFSET " .. offset)
		end
	end
end

local function runTableQuery(query, operator, err, queryTbl, parameterTbl, schema)
	if #query ~= 3 then
		util.addError(err, l("query array size is not 3, query: ") .. json.toJsonRaw(query))
	else
		if query[1] ~= 1 and type(query[1]) ~= "string" then -- field name
			util.addError(err, l("query field is not a number 1 or string, query: ") .. json.toJsonRaw(query))
		end
		if type(query[2]) ~= "string" then -- comparison
			util.addError(err, l("query comparison is not a string, query: ") .. json.toJsonRaw(query))
		end
		-- if type(query[3]) ~= ... -- value can be anything
	end

	if #err == 0 and parameterTbl and parameterTbl.skip_empty_parameter then
		if type(parameterTbl.skip_empty_parameter) ~= "table" then
			util.addError(err, l("query '%s' parameter.skip_empty_parameter is not a table, query: %s", tostring(parameterTbl.query_name), json.toJsonRaw(query)))
		else
			for _, key in ipairs(parameterTbl.skip_empty_parameter) do
				if parameterTbl[key] == nil then
					util.addError(err, l("query '%s' parameter.skip_empty_parameter contains key '%s' that does not exist in parameter, query: %s", tostring(parameterTbl.query_name), key, json.toJsonRaw(query)))
					-- util.printWarning("query '%s' parameter.skip_empty_parameter contains key '%s' that does not exist in parameter, query: %s", tostring(parameterTbl.query_name), key, json.toJsonRaw(query))
					return false
				elseif parameterTbl[key] == "" or parameterTbl[key] == 0 or type(parameterTbl[key]) == "table" and util.tableIsEmpty(parameterTbl[key]) then
					util.printWarning("skipping query '%s', parameter '%s' is empty: '%s'", tostring(parameterTbl.query_name), tostring(key), tostring(parameterTbl[key]))
					return false
				end
			end
		end
	end
	if #err == 0 then
		if query[1] ~= 1 and not parameterTbl.use_external_field and dschema.isField(query[1]) == false then -- test only local fild name, dschema.isField(query[1], schema, recordType)
			util.addError(err, l("query '%s' field name '%s' is not valid, query '%s': ", tostring(parameterTbl.query_name), tostring(query[1]), json.toJsonRaw(query)))
			return
		end
		local errTbl, value
		if type(query[3]) == "string" then
			if parameterTbl ~= nil then
				if type(parameterTbl) ~= "table" then
					util.addError(err, l("query '%s' parameter table type is not a table, query '%s': ", tostring(parameterTbl.query_name), json.toJsonRaw(query)))
					return
				end
				-- replace with possible given parameter, parameter must exist in query's parameter -tag
				-- parameterTbl[query[3]] is given parameter, queryTbl.parameter[query[3]] is query json default parameter
				local qryParam = peg.replace(query[3], "%", "")
				if qryParam ~= "" and queryTbl.parameter == nil and queryTbl.parameter[qryParam] == nil then
					util.addError(err, l("query '%s' parameter json tag '%s' was not found, query '%s': ", tostring(parameterTbl.query_name), tostring(qryParam), tostring(json.toJsonRaw(query))))
					return
				end
				value = parameterTbl[qryParam] or (queryTbl.parameter and queryTbl.parameter[qryParam])
				if type(value) == "string" then
					if value:sub(1, 7) == "SELECT " then
						-- replace with possible query default parameter
						if query[3] == "subquery" and parameterTbl.subquery then
							for key, val in pairs(parameterTbl) do
								if key ~= "subquery" and peg.found(parameterTbl.subquery, "{{" .. key .. "}}") then
									if type(val) == "table" then
										val = concatTableValue(val)
									end
									if peg.found(val, "%") then
										val = peg.replace(val, "%", "%%%%") -- peg finds single % but replace value %% must be escaped to %%%%
									end
									value = peg.replace(value, "{{" .. key .. "}}", val)
								end
							end
						end
					elseif value ~= "" and peg.found(value, ".json") then
						local jsonName = peg.parseBeforeWithDivider(value, ".json") -- value
						if jsonName ~= "" then
							local prf = dprf.prf(jsonName)
							local val = recData(prf, value:sub(#jsonName + 2)) -- plugin/nc-calc-server/constant.json/phase_open_state, "/" after .json
							if val == nil then
								if prf.query and prf.field and query[2] == "in" then
									local ret = dqjson.queryJson(jsonName, {return_sql_text = true})
									if ret and ret.sql_text then
										value = ret.sql_text
										--[[ elseif ret and ret.data and ret.info then
										local fieldName = ret.info.column_name[1]
										value = fn.iter(ret.data):map(function(rec)
											return recData(rec, fieldName)
										end):totable() ]]
									else
										util.addError(err, l("query '%s' parameter json tag '%s' was not found, query '%s': ", tostring(parameterTbl.query_name), tostring(value), tostring(json.toJsonRaw(query))))
										return
									end
								else
									util.addError(err, l("query '%s' parameter json tag '%s' was not found, query '%s': ", tostring(parameterTbl.query_name), tostring(value), tostring(json.toJsonRaw(query))))
									return
								end
							else
								value = val
							end
						end
					end
				end
				if value ~= nil then
					if qryParam ~= query[3] then
						value = peg.replace(query[3], qryParam, value)
					end
					-- util.printRed(l("runTableQuery1 %s, %s, %s", tostring(parameterTbl.query_name), tostring(fldNum), query[1]))
					errTbl = dqry.query(operator, query[1], query[2], value, queryTbl.table)
				end
			end
		end
		if value == nil then
			if type(query[3]) == "string" and query[3] ~= "" and query[3] ~= "null" and (queryTbl.parameter and queryTbl.parameter.use_value ~= true) and not dschema.isField(query[3], schema) then
				util.addError(err, l("query '%s' string value '%s' was not found from parameter json, query '%s': ", tostring(parameterTbl.query_name), tostring(query[3]), tostring(json.toJsonRaw(query))))
				return
			end
			-- use given non-string value inside query
			-- util.printRed(l("runTableQuery2 %s, %s, %s", tostring(parameterTbl.query_name), tostring(fldNum), query[1]))
			errTbl = dqry.query(operator, query[1], query[2], query[3], queryTbl.table, queryTbl.name)
		end
		if errTbl then
			util.addError(err, errTbl)
		end
	end
end

local function getQuerySet(queryTbl, param)
	if type(param) ~= "string" then
		return nil
	end
	return queryTbl.set and queryTbl.set[param]
end

local function getQueryJson(jsonPath)
	local jsonQry, err
	if peg.found(jsonPath, ".json") then
		jsonQry, err = dprf.prf("query/" .. jsonPath, "no-error")
		if err or util.tableIsEmpty(jsonQry) then
			jsonQry = dprf.prf(jsonPath, "no-error")
		end
		if type(jsonQry) == "table" and jsonQry.query then
			jsonQry = jsonQry.query
		else
			return
		end
	end
	return jsonQry
end

local function runSetQuery(queryString, operator, err, queryTbl, parameterTbl, schema)
	local set = getQuerySet(queryTbl, queryString)
	if set == nil then
		set = getQueryJson(queryString)
	end
	if set == nil then
		-- try find from preference another query before returning error
		-- call recursively dqjson.queryJson(queryTbl, parameterTbl) - but maybe we need to give extra param "and" so that query continues
		-- how to prevent infinite loops?
		--  save set query names and don't allow dqjson.queryJson if name is in the list
		util.addError(err, l("query set does not exist: ") .. queryString)
	elseif util.fmod(#set, 2) == 0 then
		util.addError(err, l("query set '%s' array size is not an odd number, query set: ", queryString) .. json.toJsonRaw(set))
	else
		dsql.querySetStart()
		runQueryByType(set, operator, err, queryTbl, parameterTbl, schema)
		dsql.querySetEnd()
	end
end

runQueryByType = function(query, operator, err, queryTbl, parameterTbl, schema)
	if type(query) == "string" then -- set name, ok
		runSetQuery(query, operator, err, queryTbl, parameterTbl, schema)
	elseif type(query) == "table" then
		local isTableQuery = true
		if #query == 3 then
			local val
			if type(query[3]) == "string" and parameterTbl[query[3]] then
				val = parameterTbl[query[3]]
			end
			if not (type(query[1]) == "string" or type(query[1]) == "number") then -- field name or number or set
				isTableQuery = false
			elseif not dsql.sqlComparison(query[2], val or query[3]) then -- comparison
				isTableQuery = false
			end
			if isTableQuery then
				-- print("operator: "..operator, json.toJsonRaw(query))
				local ret = runTableQuery(query, operator, err, queryTbl, parameterTbl, schema) -- normal table query
				return ret -- return after query
			end
		end
		runArrayQuery(query, operator, err, queryTbl, parameterTbl, schema) -- loop array of queries
	else
		util.addError(err, l("query element is not a string or an array, query element: ") .. json.toJsonRaw(query))
	end
end

function dqjson.runQueryByTypeCallback(query, operator, err, queryTbl, parameterTbl, schema, queryCallback)
	local sqlQry = dqry.query
	dqry.query = queryCallback
	runQueryByType(query, operator, err, queryTbl, parameterTbl, schema)
	dqry.query = sqlQry
end

runArrayQuery = function(queryParam, operator, err, queryTbl, parameterTbl, schema)
	if queryParam == nil then
		util.addError(err, "query array is nil, query: " .. tostring(queryParam))
		return
	end
	if util.fmod(#queryParam, 2) == 0 then
		util.addError(err, "query array size is not an odd number, query: " .. json.toJsonRaw(queryParam))
		return
	end
	if #queryParam == 3 and type(queryParam[1]) == "string" and getQuerySet(queryTbl, queryParam[1]) == nil and getQueryJson(queryParam[1]) == nil then
		-- can be like this: ["query/local/order/order_row/sales/open.json","and","product_set"] or ["ord.company_id", "=", "KONEEN VALO"]
		runQueryByType(queryParam, operator, err, queryTbl, parameterTbl, schema) -- plain 1 level array, not array of queries
		return
	end
	for i = 1, #queryParam, 2 do
		local query = queryParam[i]
		local isSet = type(query) == "table" and type(query[1]) == "table"
		if not (type(query) == "string" or type(query) == "table") then
			util.addError(err, "query type is not an string or array, query: " .. json.toJsonRaw(queryParam))
			return
		end
		if isSet then
			dsql.querySetStart()
		end
		local ret = runQueryByType(query, operator, err, queryTbl, parameterTbl, schema)
		if isSet then
			dsql.querySetEnd()
		end
		if ret == false then -- query skipped
			util.addError(err, "skipped query") -- special error, not printed
		end
		if #err > 0 then
			return
		end
		if i < #queryParam then
			operator = queryParam[i + 1]
			if type(operator) ~= "string" then
				util.addError(err, "query operator is not a string, operator: " .. tostring(operator) .. ", query: " .. json.toJsonRaw(queryParam))
				return
			end
			if not dsql.sqlOperator(operator) then
				util.addError(err, "query operator is not valid, operator: " .. tostring(operator) .. ", query: " .. json.toJsonRaw(queryParam))
				return
			end
		end
	end
end

local function runStructureQuery()
	util.printError("runStructureQuery() not done")
end

--[[ 	local isSet = type(query) == "table" and type(query[1]) == "table"
		if isSet then
			dsql.querySetStart()
		end
		if isSet then
			dsql.querySetEnd()
		end ]]

local function runQuery(queryTbl, err, parameterTbl, continue, schema) -- (queryTbl) -- continue == "and", "or", "in", ?
	local queryParam = queryTbl.query

	local showSqlOn = type(parameterTbl) == "table" and parameterTbl.show_sql or false
	if not showSqlOn then
		showSqlOn = type(queryTbl.parameter) == "table" and queryTbl.parameter.show_sql or false
	end
	local showSqlPrev
	if showSqlOn then
		showSqlPrev = dsql.showSql(true)
	end

	if queryTbl.structure then
		runStructureQuery()
	elseif queryTbl.select or queryTbl.join_type then
		if not queryTbl.is_subquery then
			queryJsonSql = {}
		end
		runSqlQuery(continue, err, queryTbl, parameterTbl, schema)
	else
		queryJsonSql = nil
		runArrayQuery(queryParam, "", err, queryTbl, parameterTbl, schema)
	end

	if showSqlOn then
		dsql.showSql(showSqlPrev)
	end

end

function dqjson.queryJson(queryTbl, parameterTbl, continue) -- continue == "and", "or", "in", ?
	dqjson.loadLibs()
	dsql.clearQuery()
	loc.lastQueryJsonName = queryTbl.name or parameterTbl.query_name or loc.lastQueryJsonName
	dsql.setQueryName(loc.lastQueryJsonName)
	if type(queryTbl) == "string" then
		if peg.find(queryTbl, ".json") == 0 then
			queryTbl = queryTbl .. ".json"
		end
		if peg.startsWith(queryTbl, "query/") then
			queryTbl = dprf.prf(queryTbl)
		else
			queryTbl = dprf.prf("query/" .. queryTbl)
		end
		-- must be cached or recursion prevention does not work
	end
	if type(queryTbl) ~= "table" then
		return {error = l("query is not a table")}
	else
		if type(parameterTbl) == "table" and parameterTbl.field and #parameterTbl.field > 0 then
			queryTbl.field = parameterTbl.field
		end
		if queryTbl.query == nil and queryTbl.select == nil then
			return {error = l("query '%s' has no tag 'query' or 'select'", tostring(loc.lastQueryJsonName))}
		elseif not queryTbl.field then -- and not parameterTbl.only_query then
			return {error = l("query '%s' has no tag 'field'", tostring(loc.lastQueryJsonName))}
		elseif queryTbl.field and #queryTbl.field < 1 then
			return {error = l("query '%s' tag 'field' count is less than 1", tostring(loc.lastQueryJsonName))}
		end
	end

	if parameterTbl.return_type then
		queryTbl.return_type = parameterTbl.return_type
	elseif queryTbl.return_type == nil then
		queryTbl.return_type = "record array"
	end
	-- queryTbl = fixQueryRecordType(queryTbl, parameterTbl, schema)
	local fieldArr, data, info, warning
	--[[if peg.found(parameterTbl.query_name, "user_group") then
		dqjson.loadLibs() -- debug
	end]]
	if type(queryTbl.table) ~= "table" then
		local err = util.printError("query table -tag is not a table, query '%s', parameter: %s", tostring(parameterTbl.query_name), json.toJson(parameterTbl))
		return {error = err}
	end
	-- local prevConn = dsql.setConnection(queryTbl.table, loc.lastQueryJsonName)-- dsql.setConnection() may change connection
	dsql.setConnection(queryTbl.table, loc.lastQueryJsonName) -- (queryTbl.table or parameterTbl.table, loc.lastQueryJsonName)
	local connQuery = dconn.query()

	local function returnConn(ret)
		--[[ if prevConn ~= dconn.getCurrentConnection() then -- todo: is this needed?
			dconn.restoreConnection(prevConn, "restore dqjson")
		end ]]
		return ret
	end

	if connQuery == nil then
		local err = util.printRed("setting query connection failed, query '%s', parameter: %s", tostring(parameterTbl.query_name), json.toJson(parameterTbl))
		return returnConn({error = err})
	end
	local schema = connQuery.schema
	if queryTbl.organization_id and queryTbl.organization_id ~= "" then
		local dbId = dconn.aliasId(queryTbl.organization_id)
		local currentId = tostring(dconn.organizationId())
		if dbId ~= currentId then
			if queryTbl.organization_id ~= "4d" or currentId ~= "plg4d-plg4d-0" or not util.from4d() then
				warning = util.printWarning("query json '%s' organization id '%s' ('%s') is not same as connection organization id '%s' ('%s')", tostring(parameterTbl.query_name), tostring(queryTbl.organization_id), tostring(dbId), dconn.organizationId(), currentId)
			end
		end
	end
	local errTxt
	if type(queryTbl.map_parameter) == "table" then
		if not execute then
			execute = require "execute"
		end
		parameterTbl = parameterTbl or {}
		errTxt = execute.runCode(queryTbl.map_parameter, parameterTbl, parameterTbl)
		if errTxt then
			return returnConn({error = l("error in query 'map_parameter': ") .. errTxt})
		end
	end
	if parameterTbl.return_sql_text ~= nil then
		queryTbl.return_sql_text = parameterTbl.return_sql_text
	end

	-- ["and", "bka.name", "like", "m%"]
	local err = {}
	if queryTbl.query then -- this really starts query
		local queryTblOrig
		if type(queryTbl.query) == "string" then -- "query": "xxx.json"
			local queryPrf = dprf.prf(queryTbl.query) -- , "no-cache"
			if queryPrf then
				queryTblOrig = util.clone(queryTbl)
				queryTbl.parameter = util.tableCombine(queryTbl.parameter, queryPrf.parameter, "no-error")
				queryTbl.query = queryPrf.query
			end
		end
		if type(queryTbl.query) ~= "table" then
			util.addError(err, "query parameter is not an array, query: " .. tostring(queryTbl.query))
		elseif #queryTbl.query < 1 then
			util.addError(err, l("query array size is less than 1, query: ") .. json.toJsonRaw(queryTbl.query))
		elseif util.fmod(#queryTbl.query, 2) == 0 then
			return returnConn({error = l("query array size is not an odd number, query: " .. json.toJsonRaw(queryTbl.query))})
		elseif runQueryCalled[queryTbl] == true then
			return returnConn({error = l("query is recursive, query: " .. json.toJsonRaw(queryTbl.query))})
		end
		runQueryCalled[queryTbl] = true
		runQuery(queryTbl, err, parameterTbl, continue, schema)
		runQueryCalled[queryTbl] = nil
		if queryTblOrig then
			queryTbl = queryTblOrig
		end
	end

	local ret = {}
	if queryTbl.select then -- sql query
		if not queryTbl.is_subquery and not (parameterTbl.only_query) then
			local queryText = concat(queryJsonSql, " ")
			if queryTbl.return_type == "record array" then
				data, info = dsql.sqlExecuteUnsafeRecordArray(queryText, queryTbl.field)
			elseif queryTbl.return_type == "array table" then
				data, info = dsql.sqlExecuteUnsafeArray(queryText, queryTbl.field)
			elseif queryTbl.return_type == "record table" then
				util.printError("query '%s', do not use deprecated 'record table', use new 'record array'", queryText)
				data, info = dsql.sqlExecuteUnsafe(queryText, queryTbl.field)
			else
				util.printError("unknown queryTbl.return_type")
			end
			ret = {info = info, data = data, error = err}
		end
	else -- normal query
		runOrder(queryTbl, err)
		if #err == 0 and queryTbl.limit then
			-- LIMIT
			local limit
			limit = limitAmount(queryTbl, err)
			if limit then
				dsql.limit(limit)
			end
		end
		if #err == 0 and queryTbl.offset then
			-- OFFSET
			local offset
			offset = offsetAmount(queryTbl, err)
			if offset then
				dsql.offset(offset)
			end
		end
		if #err == 0 and parameterTbl.aggregate then
			-- SUM, AVG, GROUP BY
			dsql.aggregate(parameterTbl.aggregate)
		end
		if queryTbl.extra_join then
			dsql.addExtraJoin(queryTbl.extra_join)
		end
		if #err == 0 and type(queryTbl.field) ~= "table" then
			util.addError(err, l("query json field type '%s' is not an array", tostring(queryTbl.field)))
		elseif #err == 0 then
			local containsSubQuery = {}
			fieldArr = {}
			for i, val in ipairs(queryTbl.field) do
				if type(val) == "string" then
					fieldArr[#fieldArr + 1] = val
				elseif type(val) == "table" then
					containsSubQuery[#containsSubQuery + 1] = i
				else
					util.addError(err, l("query json field array element %d type is not string or table", i))
				end
				if #err > 0 then
					break
				end
			end
			if #err == 0 and continue == nil and #queryTbl.field > 0 then
				if #containsSubQuery > 0 then
					-- dsql.sqlSelectInit()
					-- local fldName = fieldNameArray(conn, fieldArr)
					local sqlQueryReturnField = util.clone(queryTbl.field)
					local sqlQueryReturnFieldType = {}
					for i, val in ipairs(queryTbl.field) do
						sqlQueryReturnFieldType[i] = dschema.fieldTypeLua(val, schema) -- TODO: add recType to call
					end
					local whereTxt, err2, queryTxt, subQueryTxt, subQuery
					queryTxt, err2 = dqry.queryText(fieldArr)
					local whereArr
					if err2 then
						util.addError(err, err2)
					else
						whereTxt, queryTxt = peg.splitWithDivider(queryTxt, "FROM")
						whereArr = peg.splitToArray(whereTxt, ", ")
						for i, pos in ipairs(containsSubQuery) do -- insert from start to keep correct positions
							local rec = queryTbl.field[pos]
							subQuery = dprf.prf(rec.query)
							if util.tableIsEmpty(subQuery) then
								util.addError(err, l("query json field array element %d subquery json '%s' is invalid", i, tostring(rec.query)))
								break
							elseif type(rec.name) ~= "string" or rec.name == "" then
								util.addError(err, l("query json field array element %d subquery name '%s' is invalid", i, tostring(rec.name)))
								break
							elseif type(subQuery.field) ~= "table" or #subQuery.field < 1 then
								util.addError(err, l("query json field array element %d subquery field array is invalid", i))
								break
							end
							subQuery.return_sql_text = true
							local subQueryRet = dqjson.queryJson(subQuery, parameterTbl) -- recursive call
							subQueryTxt, errTxt = subQueryRet.sql_text, subQueryRet.error
							if errTxt then
								util.addError(err, l("error '%s' in subquery '%s'", errTxt, rec.query))
								break
							else
								sqlQueryReturnFieldType[pos] = dschema.fieldTypeLua(subQuery.field[1], schema) -- get field number of subquery first field -- TODO: add recType to call
								sqlQueryReturnField[pos] = rec.name -- return alias name
								table.insert(whereArr, pos, "\n(" .. subQueryTxt .. "\n)" .. dsql.sqlAlias() .. rec.name)
							end
						end
					end
					if #err == 0 and not (parameterTbl.only_query) then
						queryTxt = concat(whereArr, ", ") .. queryTxt
						if queryTbl.return_sql_text then
							ret = {info = "", sql_text = queryTxt, error = err}
						else
							-- sqlQueryReturnFieldType = dschema.fieldType4dArray(sqlQueryReturnFieldType)
							util.print("\n" .. queryTxt .. "\n")
							if queryTbl.return_type == "record array" then
								data, info = dsql.sqlExecuteUnsafeRecordArray(queryTxt, sqlQueryReturnField, sqlQueryReturnFieldType)
							elseif queryTbl.return_type == "array table" then
								data, info = dsql.sqlExecuteUnsafeArray(queryTxt, sqlQueryReturnField, sqlQueryReturnFieldType)
							elseif queryTbl.return_type == "record table" then
								util.printError("query '%s', do not use deprecated 'record table', use new 'record array'", queryTxt)
								data, info = dsql.sqlExecuteUnsafe(queryTxt, sqlQueryReturnField, sqlQueryReturnFieldType)
							else
								util.printError("unknown queryTbl.return_type")
							end
							ret = {info = info, data = data, error = err}
						end
					end
				else
					if queryTbl.return_sql_text then -- or (parameterTbl and parameterTbl.only_query) then
						local queryTxt, err2 = dqry.queryText(fieldArr) -- , queryTbl.aggregate)
						if err2 then
							util.addError(err, err2)
						end
						ret = {info = "", sql_text = queryTxt, error = err}
					elseif not (parameterTbl and parameterTbl.only_query) then
						local sqlKeyword = queryTbl.parameter and queryTbl.parameter.sql_keyword -- "DISTINCT"
						local sqlFunction = queryTbl.parameter and queryTbl.parameter.sql_function -- -- "COUNT(*)" or "COUNT"
						if sqlFunction == nil and sqlKeyword then
							sqlFunction = sqlKeyword
						elseif sqlKeyword then
							sqlFunction = sqlFunction .. "(" .. sqlKeyword .. ")"
						end
						if queryTbl.return_type == "record array" then
							data, info = dload.selectionToRecordArray(fieldArr, sqlFunction, parameterTbl)
						elseif queryTbl.return_type == "array table" then
							data, info = dload.selectionToArrayTable(fieldArr, sqlFunction, parameterTbl)
						elseif queryTbl.return_type == "record table" then
							util.printError("do not use deprecated 'record table', use new 'record array'")
							data, info = dload.selectionToRecordTable(fieldArr, sqlFunction, parameterTbl)
						end
						--[=[
						if data and queryTbl.local_field then
							-- convert data array external names to local names
							local newData = {}
							local oldNameArr = queryTbl.field
							local newNameArr = queryTbl.local_field
							for i, rec in ipairs(data) do
								local newRec = {}
								for j, newName in ipairs(newNameArr) do
									newRec[newName] = rec[oldNameArr[j]]
								end
								newData[i] = newRec
							end
							data = newData
						end
						]=]
						if data == nil and type(info) == "string" then
							if #err > 0 then
								info = info .. "\n" .. table.concat(err, "\n")
							end
							ret = {error = info, data = data} -- error
						else
							if #err > 0 then
								ret = {info = info, data = data, error = err}
							elseif info.error then
								ret = {info = info, data = data, error = l("error in query '%s', %s", tostring(queryTbl.name), tostring(info.error))}
							else
								ret = {info = info, data = data}
							end
						end
					end
				end
			end
		end
	end

	if #err == 0 and ret.error or ret.info and ret.info.error then
		util.addError(err, ret.error or ret.info.error)
	end
	if err[1] == "skipped query" then
		err = nil
		ret.info = {column_count = 0, column_name = {}, query_time = 0, rowCount = 0, row_count_total = 0, info = "skipped query"}
		ret.data = {}
	elseif #err > 0 then
		if parameterTbl.print_error ~= false then
			if peg.found(tostring(err[1]), "socket will be closed") then
				util.printWarning("query '%s' error: '%s'", parameterTbl.query_name or "", table.concat(err, "\n"))
			elseif parameterTbl.no_error ~= true then
				util.printRed("query '%s' error: '%s', sql: '%s'", parameterTbl.query_name or "", table.concat(err, "\n"), tostring(connQuery.queryText))
			end
		end
		util.clearError() -- clears error cache to a new table, next print must not to go to existing err -table
	elseif parameterTbl and parameterTbl.aggregate and parameterTbl.remove_empty_aggregate_row and type(ret.data) == "table" and #ret.data == 1 then
		local remove = true
		for _, value in pairs(ret.data[1]) do
			if value ~= 0 and value ~= "" and value ~= false and value ~= dt.nullDateString() then
				remove = false
				break
			end
		end
		if remove then
			ret.data = {}
		end
	end
	ret.warning = warning
	if err and #err > 0 then
		ret.error = table.concat(err, "\n")
	end
	return returnConn(ret)
end

return dqjson

--[=[
local function fixQueryRecordType(queryTbl, parameterTbl)
	do -- master2/pasi
		-- return queryTbl
	end
	-- _field_converted because dqjson.queryJson can be recursive (?)
	-- util.printTable(queryTbl.field, "queryTbl.field")
	if queryTbl._field_converted then
		return queryTbl
	end
	queryTbl._field_converted = true
	local fld
	local err = false
	if queryTbl.query then
		local queryArr = {}
		local skipNext = false
		for i, rec in ipairs(queryTbl.query) do
			if skipNext == true then
				skipNext = false
			else
				if rec[1] == nil then -- rec = "and", "or"
					queryArr[#queryArr + 1] = rec
				else
					fld = dschema.fieldName(rec[1]) -- fld = dredir2.externalName(rec[1], nil, recordType)
					if not fld then -- if false and not fld then
						if not peg.found(rec[1], ".record_type") then
							util.printWarning("query's query field '%s' could not be changed to external name, query: '%s'", rec[1], parameterTbl.query_name or "")
						end
						if queryArr[#queryArr] ~= nil and (queryArr[#queryArr] == "and" or queryArr[#queryArr] == "or") then
							queryArr[#queryArr] = nil -- delete previous "and" or "or"
						elseif queryArr[#queryArr] == nil and queryTbl.query[i + 1] ~= nil and (queryTbl.query[i + 1] == "and" or queryTbl.query[i + 1] == "or") then
							skipNext = true -- delete next "and" or "or"
						end
					--[[
						err = true
						break
						--]]
					else
						queryArr[#queryArr + 1] = {rec[1], rec[2], rec[3]} -- {fld, rec[2], rec[3]}
					end
				end
			end
		end
		if #err == 0 then
			if #queryArr == 0 then
				util.printError("#queryArr is 0")
			else
				queryTbl.query = queryArr
			end
		end
	end
	return queryTbl
end
]=]
