--- dqry.lua
-- Database json queries.
-- local dqry = require "dqry"
-- @module db
local dqry = {}

local util = require "util"
local peg = require "peg"
local l = require"lang".l
local dschema, dsql, dconn, quoteSql

function dqry.loadLibs()
	if not dconn then
		dschema = require "dschema"
		dconn = require "dconn"
		dsql = require "dsql"
		dqry.limit = dsql.limit -- no need to load dsql in calling code
		dqry.showSql = dsql.showSql -- no need to load dsql in calling code
		dqry.debugSql = dsql.debugSql -- no need to load dsql in calling code
		quoteSql = dschema.quoteSql
	end
end
dqry.loadLibs()

--[[
local qr = {}
function dqry.getQuery()
	return qr
end

function dqry.setQuery(qrNew)
	qr = qrNew
end
]]

function dqry.orderBy(fld, asc)
	asc = dsql.sqlAsc(asc)
	if not asc then
		local err = l("order by direction must be '>' or '<'")
		util.printRed("query error: '%s'", tostring(err))
		return err
	end
	-- order by
	-- dsql.sqlSelectInit() -- sqlSelectStart(conn)
	local fldName = dschema.fieldNamePrefixSql(fld) -- dschema.fieldNamePrefixSql(fld) -- dqry.fieldNameQuery(fldNum, option) -- fieldName(fldNum)
	if fldName == nil then
		local err = l("order by field name '%s' is not correct field", tostring(fld))
		util.printRed("query error: '%s'", tostring(err))
		return err
	end
	local useLower = false
	if peg.found(fldName, "json_data") then -- dschema.fieldType(fldNum) == "json" then
		fldName = "CAST(" .. dsql.changeJsonFieldName(fldName) .. " AS TEXT)" -- does not work in where
		useLower = dconn.useLowerSlq()
	else
		local schema = dconn.schema()
		local fieldType = dschema.fieldType(fld, schema)
		if fieldType == "varchar" or fieldType == "text" then -- or fieldType == "string"
			useLower = dconn.useLowerSlq()
		end
	end
	dsql.addOrder(fldName, asc, useLower, fld)
end

function dqry.queryText(fld, func)
	-- dsql.sqlSelectInit()
	local connQuery = dconn.query()
	local err = dsql.sqlQueryBuild(fld, func)
	local pos = peg.find(connQuery.queryText, "ORDER BY")
	if pos > 0 then
		return connQuery.queryText:sub(1, pos - 1)
	end
	return connQuery.queryText, err
end

function dqry.queryArray(operator, fld, comparison, valueArray, schema, recType)
	if comparison ~= "IN" and comparison ~= "NOT IN" then
		local errTxt = l("query by array comparison must be 'in' or 'not in'")
		util.printRed("query error: '%s'", errTxt)
		return
	end
	local where, fldName
	local connSql = dconn.sql()
	if schema or recType then
		fldName = dschema.fieldNamePrefix(fld, schema or dconn.query().schema, recType)
		if fldName == nil then
			fldName = dschema.fieldNamePrefix(fld)
			if fldName == nil then
				local errTxt = l("query by array field name '%s' is not valid", tostring(fld))
				util.printRed("query error: '%s'", errTxt)
				return
			end
		end
	else
		fldName = fld
	end
	fldName = dsql.changeJsonFieldName(fldName)
	fldName = quoteSql(fldName)
	-- local i = #connQuery.queryArr + 1
	--[[if connQuery.setStart[i] or connQuery.setEnd[i] then
		-- todo: do we ever come gere? setStart and setEnd are handled in dsql
		if operator ~= "" then
			where = connQuery.setStart[i]
		else
			where = dsql.sqlOperator(operator).." "..(connQuery.setEnd[i] or "") -- better to have space between .. and variables
		end
	else]]
	if operator == "" then
		where = ""
	else
		where = dsql.sqlOperator(operator) -- .. " " -- better to have space between .. and variables
	end
	local value
	if type(valueArray) == "table" then
		local luaType = dschema.fieldTypeLua(fld, schema) -- TODO: add recType to call
		-- for i,val in ipairs(valueArray) do
		-- check only q. index for performance reason (can be easily 10000 elements)
		-- no poit to check all elements, will give anyway sql error if is not valid
		-- if tostring(valueArray) == "<generator>" then
		-- valueArray = fn.util.uniqueArray(valueArray.generator, valueArray.field)
		-- end

		if #valueArray > 0 or tostring(valueArray) == "<generator>" then
			local i = 1
			local val
			if #valueArray > 0 then
				val = valueArray[i]
			else
				val = valueArray:head()
			end
			if type(val) ~= luaType then
				local errTxt = l("query by array element type '") .. type(val) .. l("' is not same as field type '") .. luaType .. l("', index: ") .. i .. l(", value: '") .. tostring(val) .. "'"
				util.print(errTxt)
			end
			value = dsql.sqlValueArray(valueArray)
		else
			-- local errTxt = l"query by array contains no values"
			-- util.printRed("query error: '%s'", errTxt)
			-- do not return, search empty array
			where = where .. "(" .. fldName .. " IS NULL AND " .. fldName .. " IS NOT NULL)" -- must return nothing
			connSql.containsEmptyArray = true
			-- value == nil here
		end
	else
		value = valueArray -- string
	end
	if value ~= nil then
		where = where .. fldName .. " " .. comparison .. " (" .. value .. ")"
	end
	dsql.sqlQueryWhereAdd(quoteSql(fld), where, schema, recType)
	return -- no error, all ok
end

function dqry.tableCount()
	local driver = dconn.driver()
	if not (driver and driver.tableCount) then
		util.printRed(l("driver.tableCount() -function does not exist, database '%s'", dconn.database()))
	end
	return driver.tableCount(dconn.database())
end

function dqry.readTableStructure(tableName, dataType)
	local queryText
	local result
	--  "SELECT column_name, data_type, character_maximum_length FROM information_schema.columns WHERE table_name = "
	local dbType = dconn.dbType()
	if dbType == "4d" then
		return
		--[[ queryText = "SELECT COLUMN_NAME, DATA_TYPE, DATA_LENGTH, IS_NULLABLE from _USER_COLUMNS where TABLE_NAME = '" .. tableName:upper() .. "' order by COLUMN_NAME"
		result = dsql.sqlExecuteUnsafeArray(queryText, {"column_name", "data_type", "character_maximum_length", "is_nullable"}, {"string", "number", "number", "string"})
		if result and type(result.data_type) == "table" then
			local typeToTextTbl = {
				[1] = "boolean",
				[3] = "smallint",
				[4] = "integer",
				[5] = "bigint",
				[6] = "double precision",
				[8] = "date",
				[9] = "time without time zone",
				[10] = "character varying",
				[14] = "text",
				[18] = "bytea"
			} -- bytea == blob
			for i, typeNum in ipairs(result.data_type) do
				result.data_type[i] = typeToTextTbl[typeNum]
			end
		end ]]
	elseif dbType == "sqlite" then
		queryText = "PRAGMA table_info('" .. tableName .. "');"
		result = dsql.sqlExecuteUnsafe(queryText, {"cid", "field_name", "field_type", "nullable", "default_value", "primary_key"}, {"number", "string", "string", "number", "string", "number"})
		for _, item in ipairs(result) do
			item.nullable = item.nullable == 0 -- notnull to nullable
		end
	elseif dbType == "oracle" then
		queryText = "SELECT column_name, data_type, data_length, is_nullable  FROM all_tab_columns WHERE table_name = '" .. tableName:upper() .. "'"
		result = dsql.sqlExecuteUnsafe(queryText, {"field_name", "field_type", "length", "nullable"}, {"string", "string", "number", "string"})
		-- http://docs.oracle.com/cd/B28359_01/server.111/b28310/tables014.htm#ADMIN01508
	else
		-- part of sql-92 standard
		-- works at least in new versions of postgre, MySQL, Microsoft SQL Server
		queryText = "SELECT column_name, data_type, character_maximum_length, is_nullable  FROM information_schema.columns WHERE table_name = '" .. tableName .. "'"
		result = dsql.sqlExecuteUnsafe(queryText, {"field_name", "field_type", "length", "nullable"}, {"string", "string", "number", "string"})
	end

	if not result or #result < 1 then
		local err = util.printRed("query error: 'SELECT column_name, ...' query result size is zero, table '%s'", tableName)
		return nil, err
	end
	for _, rec in ipairs(result) do
		if type(rec.nullable) == "string" and rec.nullable:lower() == "yes" then
			rec.nullable = true
		elseif type(rec.nullable) ~= "boolean" then
			rec.nullable = false
		end
		rec.field_name = rec.field_name:lower()
		if peg.find(rec.field_type:lower(), "char") <= 0 then
			rec.length = nil
		end
		if dataType == "lua" then
			rec.field_type = dschema.fieldTypeFromDbToLua(rec.field_type:lower())
		elseif dataType == "sql" then
			rec.field_type = rec.field_type:lower()
		else
			util.printError("wrong data type argument")
			return
		end
	end
	return result
end

function dqry.selectAll(table, record_type)
	-- dqry.loadLibs()
	dsql.setQuery({operator = "", field = 1, comparison = "=", value = 1, record_type = record_type, table = table})
end

function dqry.selectNothing(table, record_type)
	-- dqry.loadLibs()
	dsql.setQuery({operator = "", field = 1, comparison = "=", value = 0, record_type = record_type, table = table})
end

function dqry.query(operator, fld, comparison, value, record_type, queryName)
	-- dqry.loadLibs()
	dsql.setQuery({operator = operator, field = fld, comparison = comparison, value = value, record_type = record_type, query_name = queryName})
end

dqry.sqlValue = dsql.sqlValue
dqry.setQueryName = dsql.setQueryName
dqry.aggregate = dsql.aggregate
dqry.offset = dsql.offset
dqry.limit = dsql.limit
dqry.showSql = dsql.showSql
dqry.clearQuery = dsql.clearQuery
dqry.querySetStart = dsql.querySetStart
dqry.querySetEnd = dsql.querySetEnd
dqry.setConnection = dsql.setConnection
dqry.sqlExecuteUnsafe = dsql.sqlExecuteUnsafe

return dqry
