--- dsave.lua
-- Database save operations.
-- local dsave = require "dsave" - old: dsave = require "dsave"
-- @module db/database-rest4d
local dsave = {}

local util = require "util"
local autil = require "array-util"
local l = require"lang".l
local peg = require "peg"
local fn = require "fn"
local dprf = require "dprf"
local dschema = require "dschema"
local dsql = require "dsql"
local dconv = require "dconv"
local convert = require "convert/convert"
local dconn = require "dconn"
local dqry = require "dqry"
local json = require "json"
local recData = require"recdata".get
local recDataSet = require"recdata".set

local loc = {}
loc.warningChar = convert.warningChar()
loc.checkArr = {}
loc.notifyChangeTbl = {}
-- loc.jsonTabReplaceChar = "  " -- tab to 2 spaces
-- loc.jsonLnReplaceChar = " " -- ln to 1 space
local dseq, tableCode, callRest -- lazy load libs

local function initScript()
	if not tableCode then
		tableCode = require "table-code"
	end
end

local function callRestService(prf, extDataArr, tag_name)
	local ret, err
	if callRest == nil then
		callRest = require "call-rest"
	end
	local conn = dconn.currentConnection()
	--[[
	local restParam = {
		disconnect = true,
		call_method = prf.call_method or conn.call_method,
		call_path = prf.call_path or conn.call_path,
		call_url = prf.call_url or conn.call_url,
		host = {host = conn.host, port = conn.port, connect_timeout = conn.connect_timeout},
		basic_authorization = conn.user and conn.password and base64.encode(conn.user..":"..conn.password),
		debug = prf.parameter and prf.parameter.show_sql or conn.debug or false,
		-- content
	} ]]
	local content
	if prf[tag_name] then
		content = {[prf[tag_name]] = extDataArr} -- batch send all {[prf.call_insert_tag] = extDataArr}
	else
		content = extDataArr[1] -- able to send only 1 record at a time
	end
	local callRet = callRest.authCall(prf, conn, content)
	local saveTag = prf.save_result_tag or conn.save_result_tag
	if type(callRet) ~= "table" then
		err = l("rest call error: ") .. tostring(callRet)
	elseif callRet[prf.error_tag or conn.error_tag] or callRet.data and callRet.data[prf.error_tag or conn.error_tag] then -- error_tag == "code" for woocommerce calls
		err = callRet[prf.error_tag or conn.error_tag] or callRet.data and callRet.data[prf.error_tag or conn.error_tag]
		err = l("rest call error: '%s'", json.toJson(err, 'no-error'))
	elseif prf.save and saveTag and not (callRet[saveTag] or callRet.data and callRet.data[saveTag]) then -- error_tag == "code" for woocommerce calls
		err = callRet.data or callRet
		err = l("rest save -call error: '%s'", json.toJson(err, 'no-error'))
	else
		ret = l("rest call ok: '%s'", json.toJson(callRet))
	end
	return ret, err
end

local prevConn, prevOrg
local function saveConnection(organizationId)
	if prevConn then
		--[[ if organizationId == nil then
			return -- TODO: check this
		end ]]
		util.print("sql save connection, previous connection is not nil")
	end
	if organizationId then
		prevOrg = dconn.setCurrentOrganization(organizationId) -- auth.setCurrentOrganization(prf.organization_id)
	end
	prevConn = dconn.currentConnection()
	-- prevConn = util.clone(prevConn) -- why?
end

local function checkConnection()
	if prevOrg then
		util.printWarningWithCallPath("sql check connection, previous organization is not nil")
		prevOrg = nil
	end
	if prevConn then
		util.printWarningWithCallPath("sql check connection, previous connection is not nil")
		prevConn = nil
	end
end

local function restoreConnection()
	if prevOrg then
		dconn.setCurrentOrganization(prevOrg)
		prevOrg = nil
	end
	if prevConn then
		dconn.restoreConnection(prevConn)
	else
		-- util.printRed("sql restore connection, previous connection is nil")
		util.printWarning("sql restore connection, previous connection is nil")
	end
	prevConn = nil
end

local function saveToDatabase(saveData, prf_)
	local prf = prf_ and util.clone(prf_) or {} -- prf_ is save/xx/save_xxx.json -preference content, usually save/generic_save.json
	if prf.save == nil then
		prf.save = {}
	end
	if type(prf.table) == "string" then
		prf.save.table = prf.table
		prf.table = nil
	end
	if type(prf.record_type) == "string" then
		prf.save.record_type = prf.record_type
		prf.record_type = nil
	end
	-- prf.save.table -- if not defined then uses tbl in data
	-- prf.save_table_prefix -- NOT IN USE, if not defined then save all tables
	-- prf.save_only_not_changed -- NOT IN USE, ("not-updated")
	-- prf.delete
	-- prf.connection_id ("4d") -- NOT IN USE, but should use!
	local rowNumberFieldTbl
	local ret, err
	if type(saveData) ~= "table" then
		err = l("save data -param type is wrong")
		util.printError(err)
		return nil, err
	elseif #saveData <= 0 then
		if next(saveData) then
			err = util.printError("save table is not an array")
			return nil, err
		end
		return nil -- no error, this is ok
	end
	if prf.save.table == nil then
		-- should never come here, add prf param to call
		local fld = next(saveData[1])
		prf.save.table = dschema.tableName(fld) -- todo: we really need to have main table in the rec, next() result is unknown
	end
	if prf.save.table == nil then
		err = l("save prf table -record was not found")
		util.printError(err)
		return nil, err
	end
	local notUpdatedDone = true
	if prf.save_only_not_changed then -- todo: save_only_not_changed is NOT IN USE
		notUpdatedDone = false
	end

	--[[
	local prevChangeIdArr, err = dsave.previousChangeIdArr(saveData, prf)
	if err then
		return nil, err
	end
	--]]
	if not dseq then
		dseq = require "dseq"
	end
	-- tableCode.runSaveTableCode({table = tableName, modify_rec = rec, run_save_script = true})

	local function setRecordType(rec, tableName)
		local tbl, recType = dschema.splitRecTypeName(tableName)
		local recordTypeField = dschema.recordTypeFieldName(tbl)
		if recordTypeField ~= nil then
			if prf.save.record_type then
				rec[recordTypeField] = prf.save.record_type
				recType = rec[recordTypeField]
			elseif recType == "" and rec[recordTypeField] then -- todo: should this be always first option?
				recType = rec[recordTypeField]
			elseif recType ~= "" then
				rec[recordTypeField] = recType
			end
		end
		return tbl, recType
	end

	local function allowSave(tableName, rec)
		if rec._no_save then
			return false
		end
		-- todo: save_table_prefix does not exist
		if prf and prf.save_table_prefix then
			if type(tableName) ~= "string" then
				util.printError("table name type '%s' is not a string when checking tables to save", type(tableName))
				return false
			end
			local prefixArr
			if type(prf.save_table_prefix) == "string" then
				prefixArr = peg.splitToArray(peg.replace(prf.save_table_prefix, " ", ""), ",")
			elseif type(prf.save_table_prefix) == "table" then
				prefixArr = prf.save_table_prefix
			end
			if prefixArr == nil or #prefixArr <= 0 then
				return false
			elseif fn.index(tableName, prefixArr) == nil then
				return false
			end
		end
		return true
	end

	local function setRowNumber(tableName, rec, action)
		-- action == "clear", "increase"
		if rec ~= nil and prf and prf.save and type(prf.save.row_number_field) == "table" then
			if rowNumberFieldTbl == nil then
				rowNumberFieldTbl = util.clone(prf.save.row_number_field)
			end
			if action == "clear" then
				for _, rowRec in ipairs(rowNumberFieldTbl) do
					if dschema.tableName(rowRec.field) == tableName then
						rowRec.next_value = rowRec.start_value
					end
				end
			elseif action == "increase" then
				for _, rowRec in ipairs(rowNumberFieldTbl) do
					if dschema.tableName(rowRec.field) == tableName then
						local fldName = dschema.fieldName(rowRec.field)
						rec[fldName] = rowRec.next_value
						rowRec.next_value = rowRec.next_value + 1
					end
				end
			end
		end
	end

	local function saveRecord(saveRec, tableName, schema, recordType)
		saveConnection(prf.organization_id)
		local fldCount = 0
		local usedField = {}
		local notifyChangeFld = {}
		if recordType == nil then
			local _
			_, recordType = setRecordType(saveRec, tableName)
			util.printWarning("save record type -parameter is nil, using record type '%s'", recordType)
		end
		dsql.setConnection({table = tableName, record_type = recordType}, saveRec) -- does dqry.clearQuery()
		prf.save.record_type = recordType
		local fromSql = dschema.tableName(tableName, schema, recordType)
		if fromSql == nil then
			restoreConnection()
			return nil, l("table '%s', schema '%s', record type '%s', sql name can not be found", tableName, tostring(schema), tostring(recordType))
		end
		schema = schema or dconn.schema()
		local recordIdField = dschema.uuidField(tableName) -- , "", recordType) -- we need local record_id -field
		if recordIdField == nil then
			restoreConnection()
			return nil, l("table '%s' has no 'record_id' -field", fromSql)
		end
		local recordIdFieldName = dschema.fieldName(recordIdField) -- remove table prefix
		local rec = util.clone(saveRec) -- clone original saveRec because we will delete keys from original save data
		local primaryKeyFldRec = dschema.primaryKeyFieldRec(tableName, schema, recordType) -- do we need to check external fields?
		local primaryKeyFieldName = primaryKeyFldRec and primaryKeyFldRec.field_name
		local doInsert = false
		local doUpdate
		if prf.save.on_conflict == "update" then
			doInsert = true
			doUpdate = true
		end
		-- todo: Insert if record id was not found, otherwise update. in pg we should use this: http://www.postgresqltutorial.com/postgresql-upsert/
		-- local saveRecordId = rec[recordIdFieldName]
		rec.db_id = nil
		rec.db_modify_id = nil
		if rec[recordIdFieldName] == nil or rec[recordIdFieldName] == "" or rec[recordIdFieldName] == loc.warningChar then
			if not prf.parameter or prf.parameter.only_update ~= true and not doUpdate then -- todo: check if this is needed?
				doInsert = true
			else
				if primaryKeyFieldName == nil or rec[primaryKeyFieldName] == nil or rec[primaryKeyFieldName] == "" or rec[primaryKeyFieldName] == loc.warningChar then
					doInsert = true
					if prf.save.on_conflict ~= "update" then
						doUpdate = false
					end
				else
					-- we need to set new fields for WHERE query
					if not doUpdate then
						rec[recordIdFieldName] = nil -- delete rec.record_id, need to clear "" or loc.warningChar
					end
					recordIdField = primaryKeyFldRec.local_field or primaryKeyFldRec.field
					recordIdFieldName = primaryKeyFieldName
				end
			end
		end
		if doInsert == false and (rec._do_insert == true or prf.parameter and prf.parameter.always_insert) then
			doInsert = true
		end
		if doUpdate == nil then
			doUpdate = not doInsert
		end
		local updateRec = {}
		if doInsert then
			if not doUpdate and not rec._do_insert and not (prf.parameter and prf.parameter.always_insert) then
				rec[recordIdFieldName] = nil -- must be nil for a save code to set a new sequence, _do_insert or always_insert both prevent this
			end
			if prf.parameter and prf.parameter.set_default_value ~= false then
				prf.do_insert = doInsert
				dconv.setDefaultValuesToSaveRec(rec, prf, tableName, schema, recordType)
				prf.do_insert = nil
				if doInsert and doUpdate then
					updateRec = util.clone(rec)
					prf.do_update = doUpdate
					dconv.setDefaultValuesToSaveRec(updateRec, prf, tableName, schema, recordType)
					prf.do_update = nil
				end
			end
		end
		local externalTable = dschema.isExternalTable(tableName, schema, recordType) -- dschema.externalName(tableName, schema, recordType)
		local externalField, fld
		local tablePrefix = dschema.tablePrefix(tableName) -- local tbl prefix needed
		for fldName, val in pairs(rec) do -- should we clone original rec for debugging?
			fld = tablePrefix .. "." .. fldName
			if externalTable then
				externalField = dschema.externalField(fld, schema, recordType)
				if externalField == nil then
					if type(val) ~= "table" then
						rec[fldName] = nil -- must keep record_type -field here because it contains other fields, we will clear it in loopRecStructure()
						updateRec[fldName] = nil
					end
				end
			end
			if rec[fldName] ~= nil then
				if dschema.fieldIsWriteable(fld, schema, recordType) == false then -- and val ~= nil then
					if dschema.tablePrefix(fldName) ~= fldName then -- do not delete row tables that are named by table prefix
						rec[fldName] = nil
						updateRec[fldName] = nil
					end
				end
			end
		end
		if next(rec) == nil then
			restoreConnection()
			return nil, l("save record is empty")
		end
		prf.prf = "no-cache"
		dsql.setConnection({table = tableName, record_type = recordType}, rec)
		dsql.clearQuery(nil, false) -- clear sql but not table info
		local sequenceArr
		local setUuid = not rec._do_insert and not (prf.parameter and prf.parameter.always_insert)
		if setUuid or rec.record_id == "" then
			sequenceArr, err = dseq.setRecordArraySequence(tableName, schema, recordType, {rec}, doInsert)
			if doInsert and doUpdate and updateRec.record_id == "" then
				updateRec.record_id = rec.record_id
			end
		end
		if sequenceArr == nil and err then
			restoreConnection()
			return nil, err
		end
		dsql.clearQuery(nil, false)
		local scriptResult -- for future tableCode return values
		if type(prf.run_script) == "table" and #prf.run_script > 0 then
			if dschema.localDatabase() then
				-- run tableCode when saving to local postgre database
				initScript()
				scriptResult = tableCode.runSaveTableCode({table = tableName, modify_rec = rec, run_script = prf.run_script})
			elseif not prf.run_script_checked then
				-- change once save-preference run_script -tag content to external names
				--[=[ todo: send only saved table tableCode or in case of structure save we need to fix 4D code too
						-- old plugin4d call code: querySqlAlter()
						-- util.printTable(param, "plg4d/querySqlAlter/param")
						if param.script == nil or param.script.save_preference == nil then
						elseif param.script.save_preference.run_script == nil or #param.script.save_preference.run_script <= 0 then
						elseif param.script.record_id_table == nil or param.script.record_id == nil then
						else
							-- add rec.record_id -tag
							local rec = util.arrayRecord(param.script.record_id_table, param.script.save_preference.run_script, "table", 1)
							if rec then
								rec.record_id = param.script.record_id
							end
							local paramTxt = json.toJsonRaw({run_script = param.script.save_preference.run_script})
							callMethodNoReturn("_lx_JSON_SCRIPT_RUN", paramTxt)
						end
						--[[
						"run_script": [
							{
								"table": "por",
								"record_id": "note -> generate record_id array from data",
								"script_field":["por_product_id", "por_order_amount"]
							},
							{
								"table": "po",
								"record_id": "note-> generate record_id array from data",
								"script_field":["po_supplier_id"],
								"up_calculate": true
							}
						]
						--]]
				]=]
				prf.run_script_checked = true
				prf.prf = "no-error"
				for _, srec in ipairs(prf.run_script) do
					if srec.table and schema == "4d" then
						-- convert to 4d table prefix, 4d code needs local prefix
						local tablePrefix2 = dschema.tablePrefix(srec.table, schema, recordType)
						if tablePrefix2 then
							srec.table = tablePrefix2
						end
					end
					if type(srec.script_field) == "table" and #srec.script_field > 0 then
						local fieldTbl = {}
						for _, prfFieldName in ipairs(srec.script_field) do
							local fieldName = dschema.externalName(prfFieldName, schema, recordType)
							if fieldName then
								fieldTbl[#fieldTbl + 1] = fieldName
							else
								fieldTbl[#fieldTbl + 1] = prfFieldName
							end
						end
						srec.script_field = fieldTbl
					end
					if type(srec.find_linked_record) == "table" and #srec.find_linked_record > 0 then
						local fieldTbl = {}
						for _, prfFieldName in ipairs(srec.find_linked_record) do
							local fieldName = dschema.externalName(prfFieldName, schema, recordType)
							if fieldName then
								fieldTbl[#fieldTbl + 1] = fieldName
							else
								fieldTbl[#fieldTbl + 1] = prfFieldName
							end
						end
						srec.find_linked_record = fieldTbl
					end
				end
				prf.prf = nil
			end
		end
		-- tableCode.runSaveTableCode({table = tableName, modify_rec = rec, run_save_script = true})

		local primaryKeyField = dschema.primaryKeyField(tableName) -- still local rec here, no: (tableName, schema, recordType)
		if primaryKeyField then
			local externalPrimaryKeyField = dschema.externalName(primaryKeyField, schema, recordType)
			primaryKeyField = peg.parseAfter(primaryKeyField, ".")
			if rec[primaryKeyField] == "" then
				if schema == "" or externalPrimaryKeyField ~= nil then
					err = util.printWarning("trying to save empty primary key '%s' to table '%s', schema '%s'", primaryKeyField, tableName, schema)
					-- restoreConnection()
					-- return nil, err
				end
			elseif rec[primaryKeyField] == nil then
				if rec[recordIdFieldName] == "" or rec[recordIdFieldName] == nil then
					if schema == "" or externalPrimaryKeyField ~= nil then
						err = util.printRed("trying to save null primary key '%s' to table '%s' with record id field '%s' value '%s', schema '%s'", primaryKeyField, tableName, recordIdFieldName, tostring(rec[recordIdFieldName]), schema)
						restoreConnection()
						return nil, err
					end
				end
			end
		end

		if rec._save ~= false then
			local function saveAsSql()
				-- old: dqry.clearQuery("clear set"), how sets are handled between queries?
				-- dconn.restoreConnection(conn1)
				local queryTbl = dconn.query()
				-- local prevTbl = queryTbl.table
				-- local prevSchema = queryTbl.schema
				-- local prevRecordType = queryTbl.recordType
				-- dsql.clearQuery(nil, false) -- clear sql but not table info
				dsql.clearQuery() -- clear sql
				queryTbl.table = tableName -- ugly hack, but seems to work
				queryTbl.schema = schema -- ugly hack, but seems to work
				queryTbl.recordType = recordType
				local connSql = dconn.sql() -- MUST be after dsql.setConnection()
				local extTable = dschema.tableName(tableName, schema, recordType)
				dsql.addUsedTable(tableName, extTable, connSql, recordType)
				connSql.from = dschema.quoteSql(fromSql)
				local where, idFldNameSql
				if doUpdate or prf.save.on_conflict then
					-- recordIdFieldName may be primary key field if it exists
					if rec[recordIdFieldName] == nil then
						err = util.printRed("trying to save record that has no id field '%s', record: %s", tostring(recordIdFieldName), json.toJson(rec))
						return err
					end
					if rec[primaryKeyField] and prf.save.on_conflict == nil then
						idFldNameSql = dschema.nameSql(tablePrefix .. "." .. primaryKeyField, schema, recordType) -- returns quoted sql name
					end
					if idFldNameSql == nil then
						idFldNameSql = dschema.nameSql(recordIdField, schema, recordType)
						if prf.save.on_conflict then
							where = "EXCLUDED." .. idFldNameSql .. "=" .. dsql.sqlValue(rec[recordIdFieldName])
						else
							where = idFldNameSql .. "=" .. dsql.sqlValue(rec[recordIdFieldName])
						end
					else
						if prf.save.on_conflict then
							where = "EXCLUDED." .. idFldNameSql .. "=" .. dsql.sqlValue(rec[primaryKeyField])
						else
							where = idFldNameSql .. "=" .. dsql.sqlValue(rec[primaryKeyField])
						end
					end
				end
				prf.prf = "no-error"

				local tblNameNoRecType = dschema.splitRecTypeName(tableName)
				local firstCall = true
				local function createFieldSql(fldName, val)
					local writeable = dschema.fieldIsWriteable(fldName, schema, recordType)
					if writeable == false then
						return
					end
					if dschema.tableName(fldName) == tblNameNoRecType then -- use only own table fields
						local fldNameSql = dschema.nameSql(fldName, schema, recordType)
						if fldNameSql and not usedField[fldNameSql] then
							if fldNameSql == "json_data" then
								--[[ if type(val) == "string" and tableName == "preference" then
									local val2 = json.fromJson(val)
									if type(val2) == "table" then
										val = val2
									end
								end
								if type(val) ~= "table" then
									err = util.printRed("trying to save json_data which is not a table - it is wrong type of '%s'", type(val))
									return err
								end ]]
								if val == "{}" then
									-- empty json_data would override previously saved values
									err = util.printRed("trying to save empty json_data -string to table '%s'", tableName)
									return err
								end
								if type(val) == "table" and util.tableIsEmpty(val) then
									if prf.save.allow_empty_json ~= true then -- TODO: add allow_empty_json to preference input save
										-- empty json_data would override previously saved values
										err = util.printRed("trying to save empty json_data to table '%s'", tableName)
										return err
									end
								end
							end
							usedField[fldNameSql] = 1
							local sqlValue = dsql.fieldValueSql(fldName, val, schema, recordType)
							if sqlValue == nil then
								err = util.printRed("field value is nil, field '%s', table '%s'", fldName, tableName)
								return err
							else
								fldCount = fldCount + 1
								if doInsert then
									connSql.selectArr[fldCount] = fldNameSql
									connSql.insertArr[fldCount] = sqlValue
								end
								if doUpdate then
									local updateValue = fldNameSql .. "=" .. sqlValue
									if (where == updateValue) then
										fldCount = fldCount - 1
									else
										connSql.updateArr[fldCount] = updateValue
									end
									if firstCall then
										firstCall = false
										connSql.whereArr[1] = where
									end
								end
							end
							-- todo: save_only_not_changed is NOT IN USE
							if prf.save_only_not_changed and dschema.fieldName(fldName) == "change_id" then
								if sqlValue == nil then
									notUpdatedDone = false
									-- break
								else
									notUpdatedDone = true
									connSql.whereArr[2] = " AND " .. fldNameSql .. "=" .. dsql.fieldValueSql(fldName, sqlValue, schema, recordType)
								end
							end
							if notUpdatedDone == true then
								if loc.notifyChangeTbl[fldName] then -- notifyChangeTbl is file global table
									notifyChangeFld[#notifyChangeFld + 1] = fldName
								end
							end
						end
					end
				end

				local saveJsonAsText = dconn.saveJsonAsText()
				externalTable = dschema.isExternalTable(tableName, schema, recordType)

				local function loopRecStructure(rec2, fldPrefix)
					for fldName, val in autil.orderedPairs(rec2) do
						if type(rec2[fldName]) == "table" then -- json_data fields, how about more than 1 level of structure?
							local extField = fldName
							if type(fldName) == "number" or dschema.tablePrefix(fldName) ~= fldName then -- not: {ordr = {}}, yes {json_data = {}}, yes json_data.xxx.1.yyy
								if externalTable then
									extField = dschema.externalField(fldPrefix .. fldName, schema, recordType)
								end
								if saveJsonAsText then -- 4d and other non-postgre databases
									if extField ~= nil then -- field exists
										local valText = json.toJsonRaw(val) -- "\n" causes errors
										--[[ local valText = json.toJson(val)
										valText = peg.replace(valText, "\n", loc.jsonLnReplaceChar)
										valText = peg.replace(valText, "\t", loc.jsonTabReplaceChar) -- change tab to 2 spaces to keep json structure ]]
										err = createFieldSql(fldPrefix .. fldName, valText)
									elseif err == nil then
										-- virtual field name in external, continue finding
										err = loopRecStructure(rec2[fldName], fldPrefix .. fldName .. ".")
									end
								else
									if extField ~= nil then -- field exists
										-- todo: json MUST include all fields or missing old fields will be deleted
										err = createFieldSql(fldPrefix .. fldName, val)
										-- save every json_data field separately:
										-- loopRecStructure(rec2[fldName], fldPrefix..fldName..".")
									end
								end
							end
						else
							if externalTable then
								local extField = dschema.externalField(fldPrefix .. fldName, schema, recordType)
								if extField ~= nil then -- field exists
									err = createFieldSql(fldPrefix .. fldName, val) -- needs local field name here
								end
							else
								err = createFieldSql(fldPrefix .. fldName, val)
							end
						end
					end
					return err
				end
				if prf.save.on_conflict == "update" then
					doUpdate = false
				end
				err = loopRecStructure(rec, tablePrefix .. ".")
				if err then
					restoreConnection()
					return nil, err
				end
				if prf.save.on_conflict == "update" then
					doInsert = false
					doUpdate = true
					usedField = {}
					notifyChangeFld = {}
					fldCount = 0
					err = loopRecStructure(updateRec, tablePrefix .. ".")
					doInsert = true
					if err then
						restoreConnection()
						return nil, err
					end
				end
				prf.prf = nil
				if not notUpdatedDone then
					-- dqry.clearQuery("clear set") -- full init, not sqlSelectInit(c)
					-- full init, not sqlSelectInit(c)
					restoreConnection()
					err = l("prf was 'not-updated', but old data did not contain change_id -field")
					util.printError(err)
					return nil, err
				end
				local scriptParam = {
					-- query_name = "save record: " .. (externalTable or tableName) .. ", " .. (rec.name_id or rec.name or rec.record_id or "")
					query_name = "save record: " .. tableName .. ", " .. (rec.name_id or rec.name or rec.record_id or "")
				}
				if prf.name then
					scriptParam.query_name = scriptParam.query_name .. ", " .. prf.name
				end
				if dschema.localDatabase() == false then
					if rec[recordIdFieldName] and rec[recordIdFieldName] ~= "" and prf.run_script then
						scriptParam.script = {}
						scriptParam.script.record_id_table = dschema.tablePrefixSql(tableName)
						scriptParam.script.record_id = {rec[recordIdFieldName]}
						scriptParam.script.save_preference = {name = prf.name, run_script = prf.run_script}
					end
				end
				if doInsert and doUpdate then
					scriptParam.on_conflict = true
				end
				if prf.save.on_conflict == "update" then
					table.insert(connSql.updateArr, 1, "ON CONFLICT (" .. idFldNameSql .. ") DO UPDATE SET")
					connSql.returnArr = {"RETURNING *"}
				elseif prf.save.on_conflict == "nothing" then
					connSql.updateArr = {"ON CONFLICT (" .. idFldNameSql .. ") DO NOTHING"}
				end
				dsql.sqlQueryTextCreate(nil) -- nil is sql function
				local cursor
				cursor, err = dsql.sqlQueryExecute(scriptParam)
				-- queryTbl.table = prevTbl
				-- queryTbl.schema = prevSchema
				-- queryTbl.recordType = prevRecordType
				-- dsql.clearQuery()
				restoreConnection()
				if not cursor or err then
					return nil, err or l("save cursor is nil")
				end
				return rec, err
			end

			local function saveAsJson()
				prf.table = {prf.save} -- convertLocalDataToExternal needs table -tag as array to work
				local extDataArr = convert.convertLocalDataToExternal({rec}, prf)
				prf.table = nil
				if #extDataArr < 1 then
					ret = nil
					err = l("rest call save data is empty array, table '%s', schema '%s'", tostring(tableName), tostring(schema))
				else
					if prf.save.external_mandatory_field then
						local value
						for i, extRow in ipairs(extDataArr) do
							for _, field in ipairs(prf.save.external_mandatory_field) do
								value = recData(extRow, field)
								if value == nil or value == "" then
									return nil, l("rest call save row table filed '%s' mandatory value is empty in row %d, table '%s', schema '%s'", tostring(field), i, tostring(tableName), tostring(schema))
								end
							end
						end
					end
					if prf.save.row_table then
						for _, rowPrf in ipairs(prf.save.row_table) do
							if rowPrf.tag_name and rowPrf.external_mandatory_field then
								local value
								for i, extRow in ipairs(extDataArr) do
									for j, item in ipairs(extRow[rowPrf.tag_name]) do
										for _, field in ipairs(rowPrf.external_mandatory_field) do
											value = recData(item, field)
											if value == nil or value == "" then
												return nil, l("rest call save row table filed '%s' mandatory value is empty in row %d, %s row %d, table '%s', schema '%s'", tostring(field), i, rowPrf.tag_name, j, tostring(tableName), tostring(schema))
											end
										end
									end
								end
							end
						end
					end
					--[[ for i in ipairs(extDataArr) do
						if extDataArr[i].id ~= nil then
							extDataArr[i].id = nil -- todo: set in table definition
						end
					end ]]
					ret, err = callRestService(prf, extDataArr, "call_insert_tag")
					--[[ if not err then
						ret = rec -- TODO, check ret
					end ]]
					-- save returns all records
					--[[
					if ret then
						local result
						if query.field then
							result = convertExternalDataToLocal() -- convert.convertFieldData(ret, query.field, "local")
						else
							result = {}
						end
						ret = {data = result}
					end
					]]
				end
				return ret, err
			end

			if doInsert then
				setRowNumber(tableName, rec, "increase")
			end
			local dbType = dconn.dbType() -- dconn.currentConnection()
			if dbType == "rest_call" then
				ret, err = saveAsJson()
			else
				ret, err = saveAsSql()
			end
			if err == nil then
				-- set changed saved keys to original save, rec may have some fields deleted before save like record_id
				for key, val in pairs(ret) do
					if saveRec[key] ~= val then
						saveRec[key] = val
					end
				end
				ret = saveRec
				if sequenceArr ~= nil then
					dseq.saveSequence(sequenceArr)
				end
				if #notifyChangeFld > 0 then
					dsave.runChangeNotify(notifyChangeFld, prf)
				end
			end
		end
		-- dsql.clearQuery()
		-- if prevOrg then
		-- dconn.setCurrentOrganization(prevOrg) -- auth.setCurrentOrganization(prevOrg)
		-- dsql.setConnection({table = tableName, record_type = recordType}, rec)
		-- dsql.resetQuery()
		-- restoreConnection(prevOrg)
		-- end
		if type(ret) == "table" and type(scriptResult) == "table" and scriptResult.new_data then
			ret._new_data = scriptResult.new_data
		end
		return ret, err
	end

	local function saveTable(recordArray, savePref, foreignFldTbl) -- linkField, linkFieldValue)
		local tableName = savePref.table
		local recordType = savePref.record_type
		local schema
		if savePref.schema then
			schema = savePref.schema
		elseif prf.organization_id then
			schema = dconn.organizationIdToSchema(prf.organization_id)
		else
			schema = dconn.schema()
		end
		if not util.isArray(recordArray) then
			util.isArray(recordArray) -- for debug
			err = l("save array is not an array, table '%s'", tostring(tableName), json.toJson(recordArray, "no-error"))
			util.printError(err)
			return nil, err
		end
		if #recordArray < 1 then
			err = l("no records to save for table '%s'", tostring(tableName))
			util.printError(err)
			return nil, err
		end
		local tablePrefix = dschema.tablePrefix(tableName) -- local tbl prefix needed
		local firstRec = recordArray[1]
		if type(firstRec[tablePrefix]) == "table" then
			if prf.main_record == nil then
				prf.main_record = {}
			end
			prf.main_record[tablePrefix] = firstRec
			firstRec = firstRec[tablePrefix]
		end
		if type(firstRec) ~= "table" then
			err = l("save array does not contain records, table '%s'", tostring(tableName))
			util.printError(err)
			return nil, err
		end
		setRowNumber(tableName, firstRec, "clear")
		local copyValue = prf.default_value and prf.default_value[tableName] and prf.default_value[tableName].copy_value
		local errTbl, rec
		for i, mainRec in ipairs(recordArray) do
			if allowSave(tableName, mainRec) then -- todo: allowSave() does not work
				if type(mainRec[tablePrefix]) == "table" then
					rec = mainRec[tablePrefix] -- change: { pr = {name='example'} } => {name='example'}
				else
					rec = mainRec
				end
				if rec and copyValue then
					local val
					for key, fld in pairs(copyValue) do
						val = recData(rec, fld)
						if val == nil then
							val = recData(mainRec, fld)
						end
						if val == nil then
							util.printWarning("save table copy value field '%s' value does not exist, table '%s', record '%s'", tostring(fld), tostring(tableName), rec)
						else
							recDataSet(rec, key, val)
						end
					end
				end
				if foreignFldTbl and foreignFldTbl.foreignFld and foreignFldTbl.primaryFldValue ~= nil then
					rec[foreignFldTbl.foreignFld] = foreignFldTbl.primaryFldValue
				end
				ret, err = saveRecord(rec, tableName, schema, recordType)
				if err then
					if prf.parameter and prf.parameter.continue_on_error then
						errTbl = errTbl or {}
						errTbl[#errTbl + 1] = "row " .. i .. ": " .. tostring(err)
						if type(prf.parameter.continue_on_error) == "number" and #errTbl >= prf.parameter.continue_on_error then
							return nil, err
						end
					else
						return nil, err
					end
				end
			end

			if err == nil and rec._save_linked ~= false and (prf.save == nil or prf.save.table == nil or prf.save.row_table) then -- (prf.table == nil or prf.table[1] == nil)
				for fld, val in pairs(rec) do
					if type(val) == "table" then
						if dschema.tablePrefix(fld) == fld then -- fld is table prefix
							-- save lower row tables
							local recordType2
							tableName, recordType2 = setRecordType(rec, tableName)
							local tblName = dschema.tableName(fld)
							local foreignFldParam
							local foreignFld, primaryFld = dschema.relationFields(tblName, tableName, recordType2, "no-error")
							if foreignFld and primaryFld then
								local primaryFldName = dschema.fieldName(primaryFld)
								if rec[primaryFldName] ~= nil then
									-- w -> wme: set work number to wme
									-- wme -> w: do not set work number to w
									-- loadDbr()
									if dschema.linkedField(foreignFld) ~= primaryFld then -- just extra check, should not be needed
										err = l("extra check failed: dschema.linkedField(foreignFld) ~= primaryFld, table '%s'", tostring(tableName))
										util.printError(err)
										return nil, err
									else
										local foreignFldName = dschema.fieldName(foreignFld)
										foreignFldParam = {foreignFld = foreignFldName, primaryFldValue = rec[primaryFldName]}
									end
								end
							end
							if tblName then
								local save = true
								if prf.save then
									save = prf.save.table == tblName -- is this ok, send main table as rows too?
									if save == false and prf.save.row_table then
										for _, rowRec in ipairs(prf.save.row_table) do
											if rowRec.table == tblName and not rowRec.tag_name then
												-- if rowRec.tag_name exists in rest save -call then main table rest call will send rows with it and rows should not be sent separately
												save = true
												break
											end
										end
									end
								end
								if save then
									ret, err = saveTable(val, {table = tblName, record_type = recordType2}, foreignFldParam) -- saves lower row table here
								end
							end
							if err then
								return ret, err
							end
						end
					end
				end
			end
			if errTbl and err then
				err = nil
			end
		end
		if errTbl then
			return nil, table.concat(errTbl, "\n")
		end
		return ret, err
	end

	ret, err = saveTable(saveData, prf.save)
	return ret, err
end

function dsave.saveToDatabase(saveData, prf_)
	local prevOrg2
	if prf_.organization_id then
		prevOrg2 = dconn.setCurrentOrganization(prf_.organization_id)
	end
	local ret, err = saveToDatabase(saveData, prf_)
	if prevOrg2 and prevOrg2 ~= prf_.organization_id then
		dconn.setCurrentOrganization(prevOrg2)
	end
	checkConnection()
	return ret, err
end

local function saveJson(param)
	--[[
	 ### param:
		{
			preference_name: "form/local/project-material/save_purchase_order.json"
			save_data: saveData
			delete_data: deleteData
		}
		### prf.preference_name can also be save table - not prf name


		### prf example:
		{
			connection_id: "4d"
			save: [
				{
					"table": "ord",
					"row_table": [{ "table": "order_row" }],
					"aggregate_field": "ord.company_id"
				}
			]
			delete: [
				{
					"table": ["ordr"]
				}
			]
		}
	--]]
	local ret, err, prf
	if param == nil then
		return nil, util.parameterError("param")
	elseif type(param.preference) == "table" then
		prf = util.clone(param.preference)
	elseif param.preference_name == nil and param.table then
		prf = dschema.tableSavePreference({table = param.table})
	elseif param.preference_name == nil then
		return nil, util.parameterError("preference_name or table")
	end
	if type(prf) ~= "table" then
		prf, err = dprf.prf(param.preference_name, "no-cache")
	end
	if prf == nil or util.tableIsEmpty(prf) then
		return nil, l("save preference '%s' error\n - '%s'", tostring(param.preference_name), tostring(err))
	end
	local function save()
		saveConnection()
		if param.delete_data and #param.delete_data > 0 then
			if prf.delete == nil then
				err = l("preference '%s' has no delete -tag", tostring(prf.preference_name))
				return nil, err
			end
			if prf.organization_id then
				dconn.setCurrentOrganization(prf.organization_id) -- auth.setCurrentOrganization(prf.organization_id)
			end
			local saveRet = {}
			for _, rec in ipairs(prf.delete) do
				if rec.table == nil then
					err = l("save preference '%s' delete -tag does not contain table", tostring(prf.preference_name))
					return nil, err
				end
				dsql.setConnection({table = rec.table, record_type = rec.record_type}, rec) -- does dqry.clearQuery()
				local dbType = dconn.dbType() -- dconn.currentConnection()
				if dbType == "rest_call" then
					saveRet, err = callRestService(prf, param.delete_data, "call_delete_tag")
				else
					local recordIdField = dschema.uuidField(rec.table)
					if recordIdField == nil then
						err = l("record_id field was not found from '%s' table", tostring(rec.table))
						return nil, err
					end
					local recordIdArr = fn.util.mapFieldValue(param.delete_data, peg.parseAfter(recordIdField, ".")):toDistinctArr()
					dqry.query("", recordIdField, "in", recordIdArr, rec.record_type)
					err = dsave.deleteSelection(rec.table)
					if err then
						return nil, err
					end
					saveRet[#saveRet + 1] = {table = rec.table, field = recordIdField, data = recordIdArr}
				end
			end
			--[[ 	dsql.clearQuery()
			if prevOrg then
				dconn.setCurrentOrganization(prevOrg) -- auth.setCurrentOrganization(prevOrg)
				local tbl = prf.delete[1] and prf.delete[1].table or "preference"
				local recType = prf.delete[1] and prf.delete[1].record_type or ""
				dsql.setConnection({table = tbl, record_type = recType}, "restore dsave/saveJson() connection") -- todo: is this needed?
				-- dsql.resetQuery()
			end ]]
			restoreConnection()
			if err then
				util.printError(err)
				return saveRet, err
			end
			return saveRet
		end

		if param.save_data and #param.save_data > 0 then
			--[[ prf:
				save: [
					{
					 "table": "ord",
					 "row_table": [{ "table": "order_row" }],
					 "aggregate_field": "ord.company_id"
					}
			--]]
			if prf.save == nil then
				err = l("save preference '%s' save-tag does not exist", tostring(prf.preference_name))
				return nil, err
			end
			local saveArr = prf.save
			for _, rec in ipairs(saveArr) do
				if rec.table == nil then
					err = l("save preference '%s' save-tag does not contain table", tostring(prf.preference_name))
					return nil, err
				end
				prf.save = rec
				--[[
				prf.save.table = rec.table -- allowed tags in table -tag
				prf.save.record_type = rec.record_type
				prf.save.row_table = rec.row_table
				prf.save.row_number_field = rec.row_number_field
				prf.save.aggregate_field = rec.aggregate_field
				]]
				-- todo: aggregate_field on non-flat data, for ex.: need to collect order rows to order by customer_id
				ret, err = dsave.saveToDatabase(param.save_data, prf)
				if err then
					return nil, err
				end
			end
			if err then
				util.printRed("save error: '%s'", tostring(err))
				return nil, err
			end
		end
		return ret, err
	end

	local showSqlPrevValue
	if prf.parameter and prf.parameter.show_sql == true then
		showSqlPrevValue = dsql.showSql(true)
	end
	ret, err = save()
	if prf.parameter and prf.parameter.show_sql == true then
		dsql.showSql(showSqlPrevValue)
	end
	return ret, err
end

function dsave.saveJson(param)
	local ret, err = saveJson(param)
	checkConnection()
	return ret, err
end

local function deleteSelection(tblRecType)
	local err
	if type(tblRecType) == "table" then
		tblRecType = dschema.recTypeName(tblRecType[1].table, tblRecType[1].record_type[1])
	end
	local tbl = dschema.splitRecTypeName(tblRecType)
	if not dschema.isTable(tbl) then
		err = l("wrong table number in delete: '%s'", tostring(tbl))
		util.printError(err)
		return err
	end
	dsql.sqlQueryBuild({})
	local connSql = dconn.sql()
	if connSql == nil then
		err = l("connection to database failed")
		util.printError(err)
		return err
	end
	local connQuery = dconn.query()
	local queryText = connQuery.queryText
	if #connSql.whereArr > 0 then
		local where = table.concat(connSql.whereArr) -- , " ")
		local tableName = dschema.tableName(tbl, connQuery.schema, connQuery.recordType, "quote")
		connQuery.queryText = "DELETE FROM " .. tableName .. " AS " .. dschema.tablePrefix(tbl, connQuery.schema, connQuery.recordType) .. " WHERE " .. where -- sqlite needs AS in delete
	else
		err = l("deleting all records of table without WHERE -statement is not allowed")
		util.printError(err)
		return err
	end
	local cursor
	cursor, err = dsql.sqlQueryExecute({query_name = "dsave.deleteSelection"})
	restoreConnection()
	connQuery.queryText = queryText
	if not cursor or err then
		-- free cursor?
		-- return -1
		err = err or l("sqlQueryExecute cursor is nil")
	end
	return err
end

function dsave.deleteSelection(tblRecType)
	local ret, err = deleteSelection(tblRecType)
	checkConnection()
	return ret, err
end

function dsave.runChangeNotify(fldArr, prf)
	if next(loc.notifyChangeTbl) == nil then
		return
	end
	for _, fld in ipairs(fldArr) do
		if loc.notifyChangeTbl[fld] then
			for cb, exe in pairs(loc.notifyChangeTbl[fld]) do
				if type(exe.callback) == "function" then
					exe.callback(fld, prf)
				else
					loc.notifyChangeTbl[fld][cb] = nil
				end
			end
		end
	end
	-- checkConnection() -- no need
end

return dsave
