--- lib/db/dschema.lua
-- only-local table and field names
-- @module db
local dschema = {}

local util = require "util"
local peg = require "peg"
local pegSplit, pegFound, pegFind = peg.split, peg.found, peg.find
local pegSplitToArray, pegParseBefore, pegReplace, pegStartsWith = peg.splitToArray, peg.parseBefore, peg.replace, peg.startsWith
local pegParseAfter, pegParseLast, pegParseBeforeWithDivider = peg.parseAfter, peg.parseLast, peg.parseBeforeWithDivider
local fn = require "fn"
local fs = require "fs"
local dprf = require "dprf"
local dconn = require "dconn"
local json = require "json"
local dschemafld = require "dschemafld"
local concat = table.concat

local recordTypeSeparator = dschemafld.recordTypeSeparator
dschema.recordTypeSeparator = recordTypeSeparator
dschema.linkSeparator = dschemafld.linkSeparator
-- local schemaSeparator = dschemafld.schemaSeparator
-- dschema.schemaSeparator = schemaSeparator

local defaultDatabaseSchema = ""
local locf
local extf = {}
local loc = {}
loc.fieldTypeIdx = nil
loc.tableNameArr = nil
loc.reservedSqlName = nil
loc.defaultLanguage = "en"
loc.group = nil
loc.postgreExtraFieldName = {db_id = true}

local function splitRecTypeName(name, recordType)
	if name == nil then
		util.printError("splitRecTypeName parameter name is nil")
		return name
	end
	local tbl, recType = pegSplit(name, recordTypeSeparator)
	if recType == "" and recordType then
		if type(recordType) == "table" then
			recType = recordType[1]
		else
			recType = recordType
		end
	end
	return tbl, recType
end
dschema.splitRecTypeName = splitRecTypeName

local function recTypeName(fldOrTbl, recordType)
	if fldOrTbl == nil then
		util.printError("recTypeName param 1 fldOrTbl is nil")
		return fldOrTbl
	end
	if recordType == "" or recordType == nil then
		return fldOrTbl
	end
	if type(recordType) == "table" then
		recordType = recordType[1]
	end
	return fldOrTbl .. recordTypeSeparator .. recordType
end
dschema.recTypeName = recTypeName

local function quoteSql(name)
	if type(name) == "string" then
		if peg.found(name, '"') then
			util.printWarningWithCallPath("field or table name is already quoted: %s", name)
			return name
		end
	elseif name == 1 then
		return name -- field is not a string, query is 1 = 1
	else
		util.printWarningWithCallPath("field or table name type '%s' is  not a string: '%s'", type(name), tostring(name))
		return name
	end
	local dbType = dconn.dbType()
	local quote = dconn.quoteSql()
	if not loc.reservedSqlName then
		local prf = util.prf("table/prf/reserved_name.json", "no-error")
		loc.reservedSqlName = util.invertTable(prf.reserved_sql_name)
		loc.useReservedSqlName = prf.use_reserved_sql_name
		loc.quoteSql = prf.quote_sql
	end
	local pos
	if not quote or quote == "reserved" and loc.useReservedSqlName[dbType] then
		pos = peg.find(name, ".")
		if pos > 0 then
			local name2 = peg.parseAfter(name, ".")
			if not loc.reservedSqlName[name2] and not loc.reservedSqlName[name] then -- quote only reserved names, others can be without quote
				return name
			end
		elseif not loc.reservedSqlName[name] then
			return name
		end
	end
	quote = loc.quoteSql[dbType] or loc.quoteSql.postgre
	local funcStart, funcEnd
	pos = peg.find(name, "(")
	if pos > 0 then
		funcStart = name:sub(1, pos)
		funcEnd = name:sub(pos + 1)
		local pos2 = peg.find(funcEnd, ",")
		if pos2 == 0 then
			pos2 = peg.find(funcEnd, ")")
		end
		name = funcEnd:sub(1, pos2 - 1)
		funcEnd = funcEnd:sub(pos2)
	end
	local nameArr = peg.splitToArray(name, ".")
	if #nameArr == 1 then
		return quote .. name .. quote
	end
	for i = 2, #nameArr do -- first part is always table prefix - except json_data, but jso_data is not external quoted name
		nameArr[i] = quote .. nameArr[i] .. quote
	end
	if funcStart then
		name = concat(nameArr, ".")
		return funcStart .. name .. funcEnd
	end
	return concat(nameArr, ".")
end
dschema.quoteSql = quoteSql

local function unQuoteSql(fld)
	if type(fld) ~= "string" then -- can be quoted name
		if fld ~= 1 then
			util.printError("sql table or field name type '%s' is not a string", tostring(fld))
		end
	elseif peg.found(fld, '"') then -- see table/prf/reserved_name.json for different quotes, should not be other than ' or "
		return peg.replace(fld, '"', "")
	elseif peg.found(fld, "'") then
		return peg.replace(fld, "'", "")
	end
	return fld
end
dschema.unQuoteSql = unQuoteSql

local function schemaStructure(schema)
	if not locf then
		locf = dschemafld.fieldTableStructure()
		extf[""] = locf -- not need to check: if schema and schema ~= "" and extf[schema] == nil then
	end
	local lf = locf
	if schema and schema ~= "" then
		if not extf[schema] then
			extf[schema] = dschemafld.fieldTableStructure(schema)
		end
		lf = extf[schema]
	end
	return lf
end

local function fieldName(field) -- name
	schemaStructure() -- add fields to locf[field].external
	return locf[field] and locf[field].field_name
end
dschema.fieldName = fieldName

local function externalSchemaRec(externalName, schema)
	local pos = pegFind(externalName, "_") -- find first
	if pos > 0 then
		local ef = schemaStructure(schema)
		if ef then
			local externalName2 = externalName:sub(1, pos - 1) .. "." .. externalName:sub(pos + 1)
			return ef[externalName2]
		end
	end
end
dschema.externalSchemaRec = externalSchemaRec

local function isTable(tbl)
	if not locf then
		schemaStructure()
	end
	tbl = splitRecTypeName(tbl)
	local rec = locf[tbl]
	return rec and rec.table_prefix ~= nil or false
end
dschema.isTable = isTable

local function externalRec(fieldOrTable, schema, recordType)
	-- may not have local name, nil return is ok
	if schema == nil or schema == "" then
		return nil
	end
	local fld, recType = splitRecTypeName(fieldOrTable, recordType)
	local ef = schemaStructure(schema) -- add fields to locf[field].external
	local extRec
	local rec = locf[fld]
	if rec and rec.external then
		if recType == nil or recordType == nil then
			recType = ""
			local connQuery = dconn.query()
			if connQuery then
				if type(connQuery.recordType) == "table" then
					recType = recType[1]
				elseif connQuery.recordType then
					recType = connQuery.recordType
				end
			end
		end
		if recType == nil then
			util.printWarningWithCallPath("field default value find: record type is nil for field '%s', schema '%s'", fld, schema)
			recType = ""
		end
		extRec = rec.external[schema .. "/" .. recType]
		if extRec then
			return extRec
		end
		extRec = rec.external[schema .. "/"]
		local connQuery = dconn.query()
		local tableName = connQuery and rec.table_rec and rec.table_rec.table_name
		if extRec then
			if connQuery.table ~= tableName then -- allow return from another table but not same table with empty recType
				return extRec
			end
		end
		if connQuery and connQuery.tableRecordType then -- and connQuery.schema ~= nil
			local recTypeArr = connQuery.tableRecordType[tableName]
			if recTypeArr then
				for _, recType2 in ipairs(recTypeArr) do
					extRec = rec.external[schema .. "/" .. recType2]
					if extRec then
						return extRec
					end
				end
			end
		end
	end
	extRec = ef and ef[fld]
	local extRec1 = extRec
	if extRec then
		local recType2 = extRec.local_record_type or extRec.table_rec and extRec.table_rec.local_record_type
		if recType2 and recType2 ~= recType then
			if isTable(fld) then
				local prefix = fld -- tablePrefix(fld, schema, recType)
				extRec = ef[prefix] -- fix only table
			else
				extRec = nil
			end
		elseif recType2 == nil then
			extRec = nil
		end
	end
	if extRec then
		return extRec
	end
	return extRec1
end
dschema.externalRec = externalRec

local function tablePrefix(field, schema, recordType) -- return local or external table prefix
	local extRec = externalRec(field, schema, recordType)
	if extRec then
		return extRec.table_prefix or extRec.table_rec.table_prefix
	end
	if not locf then
		schemaStructure()
	end
	local field2 = splitRecTypeName(field, recordType)
	local rec = locf[field2]
	return rec and (rec.table_prefix or rec.table_rec and rec.table_rec.table_prefix)
end
dschema.tablePrefix = tablePrefix

local function externalTablePrefix(externalName, schema)
	local ef = schemaStructure(schema)
	if ef then
		local extRec = ef[externalName]
		return extRec and (extRec.table_prefix or extRec.table_rec.table_prefix)
	end
end
dschema.externalTablePrefix = externalTablePrefix

local function nameSql(field, schema, recordType) -- returns quoted sql name
	local extRec = externalRec(field, schema, recordType)
	if extRec then
		return extRec.field_name or extRec.table_name
	end
	local rec = locf[field]
	if rec == nil then
		local field2 = splitRecTypeName(field, recordType)
		rec = locf[field2]
	end
	if rec then
		if dconn.quoteSql() then
			return rec.field_name and quoteSql(rec.field_name) or rec.table_name and quoteSql(rec.table_name)
		end
		return rec.field_name or rec.table_name
	end
end
dschema.nameSql = nameSql

local function fieldNamePrefix(field, schema, recordType) -- name
	local rec, recType, prefix
	field, recType = splitRecTypeName(field, recordType)
	if schema and schema ~= "" then
		rec = externalRec(field, schema, recType)
		-- end
		if rec == nil then
			if dconn.dbType() == "postgre" then
				local name = peg.parseAfter(field, ".")
				if loc.postgreExtraFieldName[name] then
					prefix = peg.parseBefore(field, ".")
					rec = externalRec(prefix, schema, recType)
					if rec then
						local sqlName = nameSql(name, schema, recType)
						if sqlName then
							return rec.table_prefix .. "." .. sqlName -- returns quoted name
						else
							util.printError("sql name is nil for field '%s', schema '%s', record type '%s'", name, schema, recType)
						end
					end
				end
			end
		end
	end
	if rec == nil then
		rec = locf[field]
	end
	if rec then
		prefix = tablePrefix(field, schema, recType)
		return prefix and prefix .. "." .. nameSql(field, schema, recType) -- returns quoted name
	end
end
dschema.fieldNamePrefix = fieldNamePrefix

local function isField(field, schema, allowLocalField)
	if not locf then
		schemaStructure()
	end
	local rec, recType
	field, recType = splitRecTypeName(field)
	if schema and schema ~= "" then
		rec = externalRec(field, schema, recType)
		if rec == nil and allowLocalField ~= false then
			rec = locf[field]
		end
	else
		rec = locf[field]
	end
	--[[ if rec == nil then
		local connQuery = dconn.query()
		schema = schema or connQuery and connQuery.schema and connQuery.schema
		if schema and schema ~= "" then
			rec = externalRec(field, schema, connQuery and connQuery.recordType)
			if rec == nil and allowLocalField ~= false then
				rec = locf[field]
			end
		end
	end ]]
	return rec and rec.field_name ~= nil or false
end
dschema.isField = isField

local function localName(externalName, schema)
	-- may not have local name, nil return is ok
	if schema == nil then
		util.printWarning("local name for external field '%s' could not be found, schema is nil", tostring(externalName))
		return
	end
	local ef = schemaStructure(schema)
	local rec = ef and ef[externalName]
	if rec == nil and ef and not pegFound(externalName, ".") then
		rec = externalSchemaRec(externalName, schema)
	end
	if rec == nil then
		util.printWarning("local name for external name '%s', schema '%s' could not be found", tostring(externalName), tostring(schema))
		return
	end
	if rec then
		if rec.local_field then
			return rec.local_field
		elseif rec.field then -- todo: do we need local_field?
			return rec.field, "no local field found"
		else
			return rec.local_table
		end
	end
end
dschema.localName = localName

local function fieldType(field, schema, recordType, option)
	if type(field) ~= "string" then
		util.printError("field type '%s' is not a string", type(field))
		return
	end
	local fld, recType = splitRecTypeName(field, recordType)
	local rec
	if schema and schema ~= "" then
		rec = externalRec(field, schema, recType)
	end
	--[[ local rec
	if schema and schema ~= "" then
		local connQuery = dconn.query()
		if connQuery and connQuery.recordType then
			if recType == "" and connQuery.recordType ~= "" then
				recType = connQuery.recordType
			end
			rec = externalRec(fld, schema, recType or connQuery.recordType)
		end
	end ]]
	if rec == nil then
		rec = locf[fld]
	end
	if rec then
		if rec.field_type == nil then
			if rec.field_length == nil and (option == nil or not pegFound(option, "no-error")) then
				util.printWarning("field '%s', field type is nil, using default 'varchar' - fix table json to include field_type -tag", tostring(field))
			end
			rec.field_type = "varchar"
		end
		return rec.field_type
	end
end
dschema.fieldType = fieldType

local function loadFieldTypePrf()
	if loc.fieldTypePrf == nil or loc.fieldTypeIdx == nil or util.tableIsEmpty(loc.fieldTypePrf) then
		if loc.fieldTypePrf == nil or loc.fieldTypeIdx == nil or util.tableIsEmpty(loc.fieldTypePrf) then
			loc.fieldTypePrf = dprf.prf("table/prf/field_type.json")
			dprf.registerCache("dschema-loc.fieldTypePrf", loc.fieldTypePrf)
			loc.fieldTypeIdx = fn.util.createIndex(loc.fieldTypePrf.field_type, "value")
		end
	end
end

function dschema.sqlTypeToFieldType(fldType)
	loadFieldTypePrf()
	if loc.fieldSqlTypeIdx == nil then
		local idx = {}
		local val
		for _, rec in ipairs(loc.fieldTypePrf.field_type) do
			val = rec.db_value:lower()
			if val and idx[val] == nil then -- varchar is twice, use first
				idx[val] = rec.change_to_value or rec.value
			end
		end
		for _, rec in ipairs(loc.fieldTypePrf.field_type) do
			val = rec.value:lower()
			if val and idx[val] == nil then
				idx[val] = rec.change_to_value or rec.value
			end
		end
		loc.fieldSqlTypeIdx = idx
	end
	local value = loc.fieldSqlTypeIdx[fldType:lower()]
	if value then
		return value
	else
		util.printWarning("unknown field sql type '%s'", tostring(fldType))
		return fldType:lower()
	end
end

function dschema.fieldTypeSql(field, schema, recordType, option)
	loadFieldTypePrf()
	local fldType = fieldType(field, schema, recordType, option)
	local rec = loc.fieldTypeIdx[fldType]
	if rec then
		return rec.db_value
	else
		util.printError("unknown field sql type '%s'", tostring(fldType))
		return fldType
	end
end

function dschema.defaultOrderBy(table, schema, recordType)
	local rec
	table, recordType = splitRecTypeName(table, recordType)
	if schema and schema ~= "" then
		rec = externalRec(table, schema, recordType)
	end
	local orderArr
	if rec and rec.default_order then
		orderArr = rec.default_order
	else
		rec = locf[table]
		orderArr = rec and rec.default_order
	end
	if orderArr and orderArr[1] and not pegFound(orderArr[1].field, ".") then
		for i, item in ipairs(orderArr) do
			if item.field == nil then
				util.printWarning("table '%s' default_order[%d] field is nil", table, i)
			else
				item.field = (rec.local_table_prefix or rec.table_prefix) .. "." .. item.field
			end
		end
	end
	--[[ for i, item in ipairs(orderArr) do -- TODO: check if field exists, return local primaryKeyField() instead
		if dschema.isField(item.field, schema, recordType) then
		end
	end ]]
	return orderArr
end

function dschema.tableNameLanguage(field, schema, recordType) -- company
	if recordType == nil then
		field, recordType = splitRecTypeName(field, recordType)
	end
	local lf = schemaStructure(schema)
	local rec = lf[field] or locf[field]
	if rec and rec.table_rec then
		rec = rec.table_rec
	end
	return rec and (rec.table_name_language and rec.table_name_language[recordType] or rec.table_name)
end

local function tableName(field, schema, recordType, quote)
	local ret, rec
	local lf = schemaStructure(schema)
	local field2, recType = splitRecTypeName(field, recordType)
	if schema and schema ~= "" then
		rec = externalRec(field, schema, recType)
	end
	if rec then
		ret = rec.table_name or rec.table_rec and rec.table_rec.table_name
	else
		rec = locf[field2]
		if rec then
			ret = rec.table_name or rec.table_rec and rec.table_rec.table_name
		end
	end
	if ret == nil and pegFound(field2, "json_data.") then
		field2 = pegParseBeforeWithDivider(field2, "json_data")
		rec = lf[field2] or locf[field2]
		if rec then
			ret = rec.table_name or rec.table_rec and rec.table_rec.table_name
		end
	end
	if type(ret) == "string" and quote then
		ret = quoteSql(ret)
	end
	return ret
end
dschema.tableName = tableName

function dschema.tableNumber(field, schema, recordType)
	local ret, rec
	local field2, recType = splitRecTypeName(field, recordType)
	if schema and schema ~= "" then
		rec = externalRec(field, schema, recType)
	end
	if rec then
		ret = rec.table_number or rec.table_rec and rec.table_rec.table_number
	else
		rec = locf[field2]
		if rec then
			ret = rec.table_number or rec.table_rec and rec.table_rec.table_number
		end
	end
	return ret
end

function dschema.fieldNumber(field, schema, recordType)
	-- may not have local name, nil return is ok
	if schema == nil then
		util.printWarning("field number for external field '%s' could not be found, schema is nil", tostring(field))
		return
	end
	local extRec = externalRec(field, schema, recordType)
	return extRec and extRec.field_number
end

function dschema.tableNumber(field, schema, recordType)
	-- may not have local name, nil return is ok
	if schema == nil then
		util.printWarning("field number for external field '%s' could not be found, schema is nil", tostring(field))
		return
	end
	local extRec = externalRec(field, schema, recordType)
	return extRec and extRec.table_number
end

function dschema.localNameWithRecType(externalName, schema)
	-- may not have local name, nil return is ok
	if schema == nil then
		util.printWarning("local name for external field '%s' could not be found, schema is nil", tostring(externalName))
		return
	end
	local ef = schemaStructure(schema)
	local rec = ef and ef[externalName]
	if rec == nil and ef then
		if type(externalName) == "number" then
			rec = ef._field_number[externalName]
		end
		if rec == nil and not pegFound(tostring(externalName), ".") then
			rec = externalSchemaRec(externalName, schema)
		end
	end
	if rec == nil then
		util.printWarning("local name for external name '%s', schema '%s' could not be found", tostring(externalName), tostring(schema))
		return
	end
	if rec then
		if rec.local_field then
			if rec.table_rec.local_record_type ~= "" then
				return recTypeName(rec.local_field, rec.table_rec.local_record_type)
			end
			return rec.local_field
		elseif rec.field then -- todo: do we need local_field?
			if rec.table_rec.local_record_type ~= "" then
				return recTypeName(rec.field, rec.table_rec.local_record_type)
			end
			return rec.field, "no local field found"
		else
			if rec.local_record_type ~= "" then
				return recTypeName(rec.local_table, rec.local_record_type)
			end
			return rec.local_table
		end
	end
end

--- localRecordType(externalName, schema) -> "sales"
-- externalName = external_table_prefix, external_table_name, external_table_number, external_field_number, external_field_name
function dschema.localRecordType(externalName, schema, option)
	local showErr = true
	if option and pegFind(option, "no-error") > 0 then
		showErr = false
	end
	local ef = schemaStructure(schema)
	local rec = ef and ef[externalName] -- if schema is 4d then externalName can be a number like 6001
	if rec == nil then
		if ef and type(externalName) == "number" then
			rec = ef._field_number[externalName]
		end
		if rec == nil then
			if showErr then
				util.printError("local record name for external field '%s' could not be found", tostring(externalName))
			end
			return -- nil
		end
	end
	return rec.local_record_type or rec.table_rec and rec.table_rec.local_record_type -- recordType (or nil if type does not exist): ("purchase")
end

function dschema.externalField(field, schema, recordType, schemaVersion)
	-- may not have local name, nil return is ok
	if schema == nil or schema == "" then
		return fieldName(field)
	end
	local extRec = externalRec(field, schema, recordType)
	if extRec then
		if extRec.version then
			if extRec.version[schemaVersion] == "-" then
				return
			end
			for version, action in pairs(extRec.version) do
				if action == "+" and version > schemaVersion then
					return --  16ee (enterprise edition) is bigger than 16ce (community edition) because char e > c
				end
			end
		end
		return extRec.field
	end
	--[[
	local ef = schemaStructure(schema) -- add fields to locf[field].external
	local rec = ef[field]
	if rec then
		return rec.field
	end
	local rec = locf[field]
	if rec and rec.external then
		if type(recordType) == "table" then
			recordType = recordType[1]
		end
		local extRec = rec.external[schema.."/"..recordType]
		if extRec then
			return extRec.field
		end
	end ]]
end

local function externalNameSql(field, schema, recordType)
	local extRec = externalRec(field, schema, recordType)
	if extRec then
		return extRec.field_name and extRec.table_rec.table_prefix .. "." .. extRec.field_name or extRec.table_name
	end
end
dschema.externalNameSql = externalNameSql

function dschema.externalName(field, schema, recordType)
	-- may not have local name, nil return is ok
	--[[
		if schema == nil or schema == "" then
		return fieldName(field)
	end
	schemaStructure(schema) -- add fields to locf[field].external
	local rec = locf[field]
	if type(recordType) == "table" then
		recordType = recordType[1]
	end
	local extRec = rec and rec.external and rec.external[schema.."/"..recordType]
	]]
	local extRec = externalRec(field, schema, recordType or "")
	if extRec then
		return extRec.field_name or extRec.table_name
	end
end

function dschema.localFieldPart(rec)
	if rec.local_field_part == nil then -- cache value on first call
		if rec.local_field == nil then
			rec.local_field_part = {}
		else
			rec.local_field_part = pegSplitToArray(rec.local_field, ".")
		end
	end
	return rec.local_field_part
end

function dschema.fieldNamePart(rec)
	if rec.field_name_part == nil then -- cache value on first call
		rec.field_name_part = pegSplitToArray(rec.field_name, ".")
	end
	return rec.field_name_part
end

function dschema.schemaFieldArrayGen(tblName, schema, recordType)
	if schema == nil or schema == "" then
		util.printError("schema is empty")
	else
		schemaStructure(schema) -- add fields to locf[field].external
		local locTbl = locf[tblName]
		if type(recordType) == "table" then
			recordType = recordType[1]
		end
		local extTbl = locTbl and locTbl.external and locTbl.external[schema .. "/" .. recordType]
		if extTbl then
			return fn.iter(extTbl._field):filter(function(rec)
				return rec.writeable ~= false and (rec.local_field ~= nil or rec.default_value ~= nil)
			end) --[[:map(function(rec)
				return rec -- rec.field_name
			end)]]
		end
	end
	return fn.iter({})
end

--[[ -- use dschema.schemaFieldArrayGen()
function dschema.schemaLocalFieldArray(tblName, schema, recordType)
	local gen = dschema.schemaFieldArrayGen(tblName, schema, recordType)
	return gen:totable()
end
]]

function dschema.localTablePrefix(field, schema, recordType) -- return local or external table prefix
	local field2, recType = splitRecTypeName(field, recordType)
	local lf = schemaStructure(schema) -- add fields to locf[field].external and to extf[schema]
	local rec = locf[field2]
	local extRec = lf[field2]
	if schema and schema ~= "" then -- and recType ~= ""
		if type(recType) == "table" then
			recType = recType[1]
		end
		extRec = rec and rec.external and rec.external[schema .. "/" .. recType] or extRec -- we must use rec.external first because external and local may have same table names
		if extRec then
			return tablePrefix(extRec.local_table or extRec.table_rec.local_table)
		end
	end
	return rec and (rec.table_prefix or rec.table_rec.table_prefix)
end

function dschema.fieldIsWriteable(field, schema, recordType) -- , schema)
	schemaStructure(schema) -- add fields to locf[field].external
	local field2, recType = splitRecTypeName(field, recordType)
	local rec = locf[field2]
	if schema and rec and schema ~= "" then
		if type(recType) == "table" then
			recType = recType[1]
		elseif recType == nil then
			util.printError("field '%s', schema '%s', record type is nil", tostring(field), tostring(schema))
			recType = "" -- prevent error
		end
		local extRec = rec.external and rec.external[schema .. "/" .. recType]
		if extRec then
			return extRec.writeable ~= false
		end
		-- this may be a local table
	end
	return rec and rec.writeable ~= false or false
end

--[[ function dschema.tableGroup(tableNameRecType) -- return save_preference tblName
	if not locf then
		schemaStructure()
	end
	local tblName = splitRecTypeName(tableNameRecType)
	local tblRec = locf[tblName]
	if tblRec == nil then
		return nil, util.printError("table code error, table '%s' is not valid", tostring(tableName))
	end
	local tablePath = pegParseBefore(tblRec.preference_name, tblName .. ".json")
	return peg.parseAfterLast(tablePath:sub(1, -2), "/") -- "table/local/product/" -> product
end ]]

function dschema.tableCodePath(tableNameRecType) -- return save_preference tblName
	if not locf then
		schemaStructure()
	end
	local tblName = splitRecTypeName(tableNameRecType)
	local tblRec = locf[tblName]
	if tblRec == nil then
		return nil, util.printError("table code error, table '%s' is not valid", tostring(tableName))
	end
	local tablePath = pegParseBefore(tblRec.preference_name, tblName .. ".json")
	local path = tablePath .. "code/" .. tableNameRecType .. ".lua"
	if fs.fileExists(path) then
		return path:sub(1, -5)
	end
	path = "code/" .. tableNameRecType .. ".lx"
	if fs.fileExists(path) then
		return path:sub(1, -4)
	end
	path = tablePath .. "code/" .. tblName .. ".lua"
	if fs.fileExists(path) then
		return path:sub(1, -5)
	end
	path = tablePath .. "code/" .. tblName .. ".lx"
	if fs.fileExists(path) then
		return path:sub(1, -4)
	end
end

function dschema.tableSavePreference(param) -- return save_preference table
	-- param.table
	-- param.script_field
	if not locf then
		schemaStructure()
	end
	local tblRec = locf[param.table]
	if tblRec and type(tblRec.save_preference) == "string" and tblRec.save_preference ~= "" then
		return dprf.prf(tblRec.save_preference)
	end
	if param and type(param.table) == "string" and param.table ~= "" then
		local generic = dprf.prf("save/generic_save.json")
		if type(generic) == "table" then
			if param.script_field == nil and tblRec and type(tblRec.save_script) == "table" and #tblRec.save_script > 0 then
				param.script_field = tblRec.save_script
			elseif param.script_field == nil then
				if tblRec and tblRec.script_field then
					param.script_field = tblRec.script_field
				else
					generic.run_script = nil
				end
			end
			local genericTxt = json.toJsonRaw(generic)
			genericTxt = pegReplace(genericTxt, "{{table}}", param.table)
			if generic.run_script then
				if type(param.script_field) ~= "table" then
					param.script_field = {param.script_field}
				end
				local fieldArrTxt = table.concat(param.script_field, '","')
				genericTxt = pegReplace(genericTxt, "{{script_field_array}}", fieldArrTxt)
			end
			return json.fromJson(genericTxt)
		end
	end
	return -- nil
end

function dschema.defaultValueRecord(tbl, schema, recordType)
	--[[
	local lf = schemaStructure(schema)
	local extTbl = dschema.externalName(tbl, schema, recordType)
	local rec = lf[extTbl]
	]]
	tbl, recordType = splitRecTypeName(tbl, recordType)
	local rec = locf[tbl]
	if rec and schema and schema ~= "" then -- if you give schema then recordType is mandatory
		schemaStructure(schema) -- add fields to locf[field].external
		if type(recordType) == "table" then
			recordType = recordType[1]
		end
		rec = rec.external and rec.external[schema .. "/" .. recordType]
	end
	if rec == nil then
		return
	end
	return rec.default_value and rec.default_value[recordType] or rec.default_value
end

local function fieldRec(field, schema, recordType) -- todo: use one function fo all these rec finds
	local rec = locf[field]
	if rec and schema and schema ~= "" then -- if you give schema then recordType is mandatory
		schemaStructure(schema) -- add fields to locf[field].external
		if type(recordType) == "table" then
			recordType = recordType[1]
		end
		if recordType == nil then
			util.printWarningWithCallPath("field default value find: record type is nil for field '%s', schema '%s'", field, schema)
			recordType = ""
		end
		local recExt = rec.external and rec.external[schema .. "/" .. recordType]
		return rec, recExt
	end
	return rec
end

function dschema.fieldDefaultValue(field, schema, recordType, setFunction)
	local rec, recExt = fieldRec(field, schema, recordType)
	local defaultValue, syncValue
	if recExt then
		defaultValue, syncValue = recExt.default_value, recExt.sync_value
	end
	if rec then
		if defaultValue == nil and rec.default_value ~= nil then
			defaultValue = rec.default_value
		end
		if setFunction and defaultValue == nil and rec.default_function ~= nil then
			defaultValue = rec.default_function
		end
		if syncValue == nil and rec.sync_value ~= nil then
			syncValue = rec.sync_value
		end
	end
	return defaultValue, syncValue
end

function dschema.fieldDefaultTypeValue(fld)
	loadFieldTypePrf()
	local fldType = fieldType(fld)
	local rec = loc.fieldTypeIdx[fldType]
	if rec then
		return rec.default_value
	else
		util.printError("unknown field default value type '%s'", tostring(fldType))
		return ""
	end
end

function dschema.isNullField(fld)
	if not locf then
		schemaStructure()
	end
	local rec = locf[fld]
	return rec and rec.nullable == true or false -- rec.field_name and
end

function dschema.recordTypeField(tbl)
	if not locf then
		schemaStructure()
	end
	tbl = splitRecTypeName(tbl)
	local rec = locf[tbl]
	return rec and (rec.table_prefix or rec.table_rec.table_prefix) .. ".record_type" or nil
end

function dschema.recordTypeFieldName(tbl)
	if not locf then
		schemaStructure()
	end
	tbl = splitRecTypeName(tbl)
	local rec = locf[tbl]
	return rec and "record_type" or nil
end

function dschema.fieldCount(tbl)
	if not locf then
		schemaStructure()
	end
	tbl = splitRecTypeName(tbl)
	local rec = locf[tbl]
	if rec == nil then
		util.printError("field count for '%s' could not be found", tostring(tbl))
		return 0
	end
	local rec2 = locf[rec.table_name or rec.table_rec.table_name]
	return rec2 and rec2.field_count
end

function dschema.init(schema)
	schemaStructure(schema)
end
--[[
local function createSubfield(tbl, key)
	local jsonField, jsonSubfield = pegSplit(key, ".")
	if jsonSubfield == "" then
		tbl[key] = key -- maybe?: peg.endsWith(key, ".json_data") then tbl[key] = {} end
	else
		if type(tbl[jsonField]) ~= "table" then
			tbl[jsonField] = {}
		end
		if not pegFound(jsonSubfield, ".") then
			tbl[jsonField][jsonSubfield] = key
		else
			createSubfield(tbl[jsonField], jsonSubfield)
		end
	end
end

function dschema.fieldTable(schema)
	local lf = schemaStructure(schema)
	if lf.fldTable then
		return lf.fldTable
	end
	lf.fldTable = {}
	for _, tblRec in ipairs(lf._table) do
		lf.fldTable[tblRec.table_name] = tblRec.table_name
		lf.fldTable[tblRec.table_prefix] = tblRec.table_prefix
		for _, rec in ipairs(tblRec._field) do
			createSubfield(lf.fldTable, tblRec.table_prefix.."."..rec.field_name)
		end
	end
	return lf.fldTable
end ]]

function dschema.tableArray(schema)
	local lf = schemaStructure(schema)
	local tblArr = {}
	if lf then
		for key in pairs(lf) do -- key == val in f
			if key == tableName(key, schema) then
				tblArr[#tblArr + 1] = key
			end
		end
		table.sort(tblArr)
	end
	return tblArr
end

local function fieldArray(tblName, schema, recType_)
	if tblName == nil then
		util.printError("parameter table is nil")
		return
	end
	local tbl, recType = splitRecTypeName(tblName, recType_)
	if not (schema and recType) then
		local connQuery = dconn.query()
		schema = schema or connQuery.schema
		recType = recType or connQuery.recordType
	end
	local extRec
	if schema or recType then
		extRec = externalRec(tbl, schema, recType)
	end
	if extRec then -- rec is external
		local extSchema = extf[schema]
		local extTbl = extRec.table_name
		if extSchema[extTbl] then
			-- add fieldArr -tag to external schema
			if extSchema.fieldArr and extSchema.fieldArr[extTbl] then
				return extSchema.fieldArr[extTbl]
			end
			local fldArr = extSchema[extTbl]._field
			local arr = {}
			for _, rec in ipairs(fldArr) do
				if rec.field_name ~= "json_data" then
					arr[#arr + 1] = rec.table_rec.table_prefix .. "." .. rec.field_name -- extSchema[extTbl].table_prefix.."."..rec.field_name
				end
			end
			if extSchema.fieldArr == nil then
				extSchema.fieldArr = {}
			end
			if extSchema.fieldArr[extTbl] == nil then
				extSchema.fieldArr[extTbl] = arr
			end
			return arr, fldArr
			-- else
			-- 	util.printError("schema '%s' external table '%s' does not exist", schema, tbl)
			-- 	return {}
		end
	end
	-- loc = local schema, add fieldArr -tag to it
	locf = locf or schemaStructure()
	if not locf[tbl] then
		util.printError("local table '%s' does not exist", tbl)
		return {}
	end
	local tblName2 = locf[tbl].table_name
	if loc.fieldArr == nil then
		loc.fieldArr = {}
		dprf.registerCache("dschema-loc.fieldArr", loc.fieldArr)
	end
	if loc.fieldArr[tblName2] and not util.tableIsEmpty(loc.fieldArr[tblName2]) then
		return loc.fieldArr[tblName2]
	end
	loc.fieldArr[tblName2] = {}
	local fldArr = locf[tblName2]._field
	local arr = {}
	for i, rec in ipairs(locf[tblName2]._field) do
		arr[i] = locf[tblName2].table_prefix .. "." .. rec.field_name
	end
	loc.fieldArr[tblName2] = arr
	return arr, fldArr
end
dschema.fieldArray = fieldArray

function dschema.localFieldArray(tbl, schema, recType)
	local extFldArr = fieldArray(tbl, schema, recType)
	local localFldArr = {}
	local fld
	for _, externalName in ipairs(extFldArr) do
		fld = localName(externalName, schema) -- , recType
		if fld then
			localFldArr[#localFldArr + 1] = localName(externalName, schema)
		end
	end
	return localFldArr
end

local sqlFieldType
local function sqlFieldTypeInit()
	if sqlFieldType == nil then
		local fieldTypePrf = dprf.prf("table/prf/field_type.json", "no-db") -- must not call from db or there will be infinite loop
		sqlFieldType = {}
		fn.iter(fieldTypePrf.field_type):each(function(rec)
			sqlFieldType[rec.value] = {lua_type = rec.lua_type, basic_type = rec.basic_type or rec.lua_type}
		end)
	end
end

function dschema.fieldTypeBasic(field, schema, recType)
	if type(field) ~= "string" then
		util.printError("field type '%s' is not a string", type(field))
		return
	end
	if pegStartsWith(field, "COUNT(") then
		return "integer"
	end
	-- if fldType == nil then
	local fldType = fieldType(field, schema, recType)
	-- end
	if type(fldType) == "string" then
		sqlFieldTypeInit()
		fldType = fldType:lower()
		local basicType = sqlFieldType[fldType].basic_type
		if basicType then
			return basicType
		end
	end
	if fldType ~= nil then
		util.printWarning("unknown field '%s' type '%s', check preference table/prf/field_type.json", tostring(field), tostring(fldType))
	else
		util.printWarning("unknown field type, field '%s'", tostring(field))
	end
	return "string" -- string is default, better than nothing
end

function dschema.fieldTypeLua(field, schema, recType) -- , fldType)
	if type(field) ~= "string" then
		util.printError("field type '%s' is not a string", type(field))
		return
	end
	-- if fldType == nil then
	local fldType = fieldType(field, schema, recType)
	-- end
	if type(fldType) == "string" then
		sqlFieldTypeInit()
		fldType = fldType:lower()
		local luaType = sqlFieldType[fldType] and sqlFieldType[fldType].lua_type
		if luaType then
			return luaType, fldType
		end
	end
	util.printError("unknown field type '%s', check preference table/prf/field_type.json", tostring(fldType))
	return "string", "string" -- string is default, better than nothing
	--[[
		if fldType == "varchar" then
			return "string"
		elseif fldType == "character varying" then
			return "string"
		elseif fldType == "text" then
			return "string"
		elseif fldType == "double" then
			return "number"
		elseif fldType == "integer" then
			return "number"
		elseif fldType == "date" then
			return "string"
		elseif fldType == "time" then
			return "string"
		elseif fldType == "boolean" then
			return "boolean"
		elseif fldType == "blob" then
			return "string"
		elseif fldType == "timestamp" then -- timestamp
			return "string"
		elseif fldType == "timestamp with time zone" then -- timestamp
			return "string"
		elseif fldType == "json" then -- json
			return "string"
		elseif fldType == "jsonb" then -- jsonb
			return "string"
		elseif fldType == "uuid" then -- uuid
			return "string"
		-- elseif fldType == "dp" then
		-- return "number"
		-- elseif fldType == "g" then
		-- return "string"
		end
	--]]
end

function dschema.recordType(field)
	if not locf then
		schemaStructure()
	end
	local rec = locf[field]
	return rec and (rec.record_type or rec.table_rec.record_type)
end

function dschema.isJsonField(field, schema, recType, option)
	local fldType = fieldType(field, schema, recType, option)
	return fldType == "json" or fldType == "jsonb"
end

function dschema.isStringField(field, schema, recType, option)
	local fldType = fieldType(field, schema, recType, option)
	return fldType == "varchar" or fldType == "text"
end

--[[ -- use dload.tableExists(tbl) to know if table exists in database
function dload.tableExists(tbl)
	if not locf then
		schemaStructure()
	end
	return locf[tbl] and locf[tbl].table_name ~= nil or false
end
--]]
local function fieldExists(field)
	if not locf then
		schemaStructure()
	end
	return locf[field] and locf[field].field_name ~= nil or false
end
dschema.fieldExists = fieldExists

local function isExternalTable(tbl, schema, recType)
	local extRec = externalRec(tbl, schema, recType)
	if extRec and extRec.schema ~= "" then --  and extRec.table_name
		return true
	end
	return not dconn.isLocalTable(tbl, schema, recType)
end
dschema.isExternalTable = isExternalTable

function dschema.isLocalTable(tbl, schema, recType)
	return not isExternalTable(tbl, schema, recType)
end

function dschema.externalFieldExists(field, schema, recType)
	local rec
	if schema and schema ~= "" then
		rec = externalRec(field, schema, recType)
		if rec == nil then
			local tbl = tableName(field, schema, recType)
			if not isExternalTable(tbl, schema, recType) then
				return fieldExists(field)
			end
		end
	else
		return fieldExists(field)
	end
	if rec and rec.field_name == pegParseAfter(field, ".") then
		return true
	end
	return false
end

-- dschema original ---
function dschema.localDatabase()
	return dconn.schema() == defaultDatabaseSchema
end

-- relations --

function dschema.linkedField(field)
	local rec = locf[field]
	if not rec then
		util.printError("field '%s' does not exist", tostring(field))
		return -- nil
	end
	if rec.foreign_key == nil or rec.foreign_key[1] == nil then
		util.printError("field '%s' does not have any relation to another field", tostring(field))
		return -- nil
	end
	local prefix = locf[rec.foreign_key[1].linked_table].table_prefix -- locf[rec.foreign_key.table].table_prefix
	return prefix and prefix .. "." .. rec.foreign_key[1].linked_field
end

-- relations end --

local groupPrfName
local function groupArray(schema)
	if schema == nil then
		schema = defaultDatabaseSchema
	end
	schema = string.lower(schema)

	if loc.group == nil then
		loc.group = {}
		dprf.registerCache("dschema-loc.group", loc.group)
	end
	if loc.group[schema] and not util.tableIsEmpty(loc.group[schema]) then
		return loc.group[schema], groupPrfName
	end
	if schema == defaultDatabaseSchema then
		groupPrfName = "table/prf/group.json"
		loc.group[schema] = dprf.prf(groupPrfName).group
		dprf.registerCache(groupPrfName, loc.group[schema])
	else
		schemaStructure(schema)
		groupPrfName = "table/external/" .. schema .. "/prf/group.json"
		local prf = dprf.prf(groupPrfName)
		if prf.group then
			loc.group[schema] = prf.group
			dprf.registerCache(groupPrfName, loc.group[schema])
			for _, rec in ipairs(loc.group[schema]) do
				if rec.table_prefix == nil and rec.local_table ~= nil then
					rec.table_prefix = prf.prefix[rec.local_table]
				end
				if extf[schema][rec.table_name] then
					rec.local_table = extf[schema][rec.table_name].local_table
					rec.table_rec_type = recTypeName(rec.local_table, extf[schema][rec.table_name].local_record_type)
				end
			end
		end
	end
	return loc.group[schema], groupPrfName
end
dschema.groupArray = groupArray

function dschema.tblNameToPath(tblName, schema, recordType)
	if tblName == nil then
		return -- nil
	end
	local groupArr = groupArray(schema)
	if groupArr then
		for _, rec in ipairs(groupArr) do
			if recordType == nil or recordType == "" then
				if rec.table_name == tblName then
					return rec.group .. rec.table_name .. "/"
				elseif rec.local_table == tblName then -- external
					return rec.group .. rec.table_name .. "/"
				end
			elseif rec.record_type == recordType then
				if rec.table_name == tblName then
					return rec.group .. rec.table_name .. "/" .. recordType .. "/"
				elseif rec.local_table == tblName then -- external
					return rec.group .. rec.table_name .. "/" .. recordType .. "/"
				end
			end
		end
	end
	return -- nil
end

function dschema.tablePrefixToPath(tblPrefix, schema, recordType)
	if tblPrefix == nil then
		return -- nil
	end
	local groupArr = groupArray(schema)
	if groupArr then
		for _, rec in ipairs(groupArr) do
			if rec.table_prefix == tblPrefix then
				if recordType == nil or recordType == "" then
					return rec.group .. rec.table_name .. "/"
				elseif rec.record_type == recordType then
					return rec.group .. rec.table_name .. "/" .. recordType .. "/"
				end
			end
		end
	end
end

function dschema.tablePrefixToPrimaryKeyField(tblPrefix, schema, recordType)
	if tblPrefix == nil then
		util.printError("table prefix is nil")
		return -- nil
	end
	schema = schema or ""
	recordType = recordType or ""
	local allowUnique = false
	local primaryKeyField = dschema.primaryKeyField(tblPrefix, schema, recordType, allowUnique)
	return primaryKeyField
end

local function localPrefix(externalName, schema, recType, option)
	-- noes NOT return external prefix if local was not found, not same as localTable()
	local rec
	if schema and schema ~= "" then
		rec = externalRec(externalName, schema, recType)
	end
	if rec then
		return rec.local_table_prefix, rec.local_record_type
	else
		rec = locf[externalName]
	end
	if rec == nil then
		if option == nil or not pegFound(option, "no-error") then
			util.printError("local table prefix for external field '%s' could not be found", tostring(externalName))
		end
		return
	end
	local nameLocal = rec and (rec.local_field or rec.local_table)
	if nameLocal then
		nameLocal = locf[nameLocal] and (locf[nameLocal].table_prefix or locf[nameLocal].table_rec and locf[nameLocal].table_rec.table_prefix)
		if nameLocal then
			return nameLocal
		end
	end
	return rec and (rec.table_prefix or rec.table_rec.table_prefix)
end
dschema.localPrefix = localPrefix

local function localTable(externalName, schema, recType, option)
	-- returns external table if local was not found
	local rec
	if schema and schema ~= "" then
		rec = externalRec(externalName, schema, recType)
		if rec == nil then
			externalName = unQuoteSql(externalName)
			rec = externalRec(externalName, schema, recType)
		end
	end
	if rec == nil then
		rec = locf[externalName]
	end
	if rec == nil then
		local localFld = splitRecTypeName(externalName)
		rec = locf[localFld]
	end
	if rec == nil then
		if option == nil or not pegFound(option, "no-error") then
			util.printError("local table for external field '%s' could not be found", tostring(externalName))
		end
		return -- nil
	end
	local nameLocal = rec and (rec.local_field or rec.local_table)
	if nameLocal then
		local nameLocal2 = locf[nameLocal] and (locf[nameLocal].table_name or locf[nameLocal].table_rec and locf[nameLocal].table_rec.table_name)
		if nameLocal2 then
			return nameLocal2
		end
		return nameLocal
	end
	return rec and (rec.table_name or rec.table_rec and rec.table_rec.table_name)
end
dschema.localTable = localTable

local function toLocal(externalName, schema, option)
	local showErr = true
	if option and pegFind(option, "no-error") > 0 then
		showErr = false
	end
	local ef = schemaStructure(schema)
	local rec = ef and ef[externalName]
	if rec == nil and ef then
		externalName = unQuoteSql(externalName)
		rec = ef[externalName]
	end
	if rec == nil and ef and schema == "4d" then
		rec = ef["_" .. externalName] -- tables like _prf, _rep, ...
	end
	if rec == nil then
		if showErr then
			util.printError("local value for external value '%s' could not be found", tostring(externalName))
		end
		return -- nil
	end
	if rec.table_prefix == externalName then
		local nameLocal = rec and (rec.local_field or rec.local_table)
		local tblRec = nameLocal and locf[nameLocal] and (locf[nameLocal].table_rec or locf[nameLocal])
		if tblRec then
			return tblRec.table_prefix, tblRec and tblRec.local_record_type
		end
		return
	end
	if rec then
		local name = rec.local_field or rec.local_table
		if name then
			return name, rec.table_rec and rec.table_rec.local_record_type or rec.local_record_type
		else
			return rec.field_name or rec.table_name, rec.table_rec and rec.table_rec.record_type or rec.record_type and rec.record_type[1]
		end
	end
end
dschema.toLocal = toLocal

function dschema.isExternalTablePrefix(externalName, schema, option)
	local showErr = true
	if option and pegFind(option, "no-error") > 0 then
		showErr = false
	end
	local ef = schemaStructure(schema)
	local rec = ef and ef[externalName]
	if rec == nil then
		if showErr then
			util.printError("local value for external value '%s' could not be found", tostring(externalName))
		end
		return -- nil
	end
	if rec.table_prefix == externalName then
		return true
	end
	return false
end

function dschema.idField(tblOrFld) -- , schema, recType) -- returns local table prefix
	local prefix = tablePrefix(tblOrFld)
	if prefix == nil then
		return
	end
	local fld = prefix .. ".record_id"
	if isField(fld) then -- , schema) then
		return fld
	end
	--[[ fld = prefix.."json_data.record_id"
	if isField(fld, schema) then
		return fld
	end ]]
	fld = prefix .. ".id"
	if isField(fld) then
		return fld
	end
end

local function modifyIdField(tblOrFld, schema) -- returns local table prefix
	local prefix = tablePrefix(tblOrFld)
	local fld = prefix .. ".modify_id"
	if isField(fld, schema, false) then
		return fld
	end
	--[[ fld = prefix.."json_data.modify_id"
	if isField(fld, schema) then
		return fld
	end ]]
end
dschema.modifyIdField = modifyIdField

function dschema.modifyTimeField(tblOrFld, schema) -- returns local table prefix
	local prefix = tablePrefix(tblOrFld)
	local fld = prefix .. ".modify_time"
	-- todo: use table info about modify_id -field name
	if schema ~= "4d" and isField(fld, schema, false) then --  false == do not allow local field
		return fld
	end
	return modifyIdField(tblOrFld) -- for 4D schema
	--[[ fld = prefix.."json_data.modify_id"
	if isField(fld, schema) then
		return fld
	end ]]
end

function dschema.uniqueArray(tbl)
	if not locf then
		schemaStructure()
	end
	tbl = splitRecTypeName(tbl)
	local rec = locf[tbl]
	if rec then
		if rec.table_rec then
			rec = rec.table_rec
		end
		return rec.unique
	end
end

function dschema.uuidField(tbl, schema, recType)
	if not locf then
		schemaStructure()
	end
	tbl, recType = splitRecTypeName(tbl, recType)
	local rec
	if schema and schema ~= "" then
		rec = externalRec(tbl, schema, recType)
	end
	if rec == nil then
		rec = locf[tbl]
	end
	-- todo FIX name "record_id" (specially for external fields)
	return rec and (rec.table_prefix or rec.table_rec.table_prefix) .. ".record_id" or nil -- rec.field_name == "record_id" and "record_id" or false
end

local function primaryKeyFieldRec(tbl, schema, recType, allowUnique)
	-- finds primary key or first unique field
	if not locf then
		schemaStructure()
	end
	tbl, recType = splitRecTypeName(tbl, recType)
	local rec
	if schema and schema ~= "" then
		rec = externalRec(tbl, schema, recType)
	end
	if rec == nil then
		rec = locf[tbl]
	end
	if rec then
		local idField = dschema.idField(tbl)
		local firstUniqueRec
		if allowUnique == nil then
			allowUnique = true
		end
		for _, fldRec in ipairs(rec._field) do
			if fldRec.constraint then
				for _, constraint in ipairs(fldRec.constraint) do
					if fldRec.field_name ~= "db_modify_id" and fldRec.field_name ~= "modify_id" then
						if constraint == "primary key" then
							return fldRec -- only one primary key
						elseif allowUnique and constraint == "unique" then
							if firstUniqueRec == nil then
								firstUniqueRec = fldRec -- continue, we may find primary key later
							elseif fldRec.field ~= idField and firstUniqueRec.field == idField then
								firstUniqueRec = fldRec -- continue, we may find primary key later, but non record_id wins
							end
						end
					end
				end
			end
		end
		return firstUniqueRec
	end
end
dschema.primaryKeyFieldRec = primaryKeyFieldRec

function dschema.primaryKeyField(tbl, schema, recordType, allowUnique)
	local rec = primaryKeyFieldRec(tbl, schema, recordType, allowUnique)
	return rec and (rec.table_prefix or rec.table_rec.table_prefix) .. "." .. rec.field_name
end

function dschema.relationFields(tbl1, tbl2, recordType, option)
	if not locf then
		schemaStructure()
	end
	local rec1 = locf[tbl1]
	local rec2 = locf[tbl2]
	if rec1 == nil or rec2 == nil then
		util.printError("relationFields between table '%s' and table '%s' is not possible", tostring(tbl1), tostring(tbl2))
		return
	end
	if rec1.table_rec then
		rec1 = rec1.table_rec -- is field, get table
	end
	if not rec1.link then
		if rec2.table_rec then
			rec2 = rec2.table_rec
		end
		rec1, rec2 = rec2, rec1 -- swap
	end
	if not rec1.link then
		util.printError("relationFields links between table '%s' and table '%s' is not possible, neither table has any links", tostring(tbl1), tostring(tbl2))
		return
	end
	local ret1, ret2
	local function findLink()
		for _, linkRec in ipairs(rec1.link) do
			if linkRec.linked_table == rec2.table_name and (linkRec.linked_record_type == recordType or linkRec.linked_record_type == "") then
				if not ret1 then -- or linkRec.default_link then -- use first found or default
					ret1 = linkRec.linking_field
					ret2 = linkRec.linked_field
					-- if linkRec.default_link then
					break
					-- end
				end
			end
		end
	end
	findLink()
	if ret1 == nil then
		rec1, rec2 = rec2, rec1 -- swap and fin the other way
		findLink()
	end
	if ret1 == nil and option ~= "no-error" then
		util.printError("relationFields links between table '%s' and table '%s', record type '%s'  was not found", tostring(tbl1), tostring(tbl2), tostring(recordType))
		return
	end
	return ret1, ret2
end

function dschema.primaryField(tbl)
	if not locf then
		schemaStructure()
	end
	tbl = splitRecTypeName(tbl)
	local rec = locf[tbl]
	if rec == nil then
		util.printError("table '%s' could not be found", tostring(tbl))
		return
	end
	if rec.table_rec then
		rec = rec.table_rec -- is field, get table
	end
	if rec.primary_field == nil then
		return
	end
	return rec.table_prefix .. "." .. rec.primary_field
end

function dschema.tableLinkingFieldArr(tbl)
	if not locf then
		schemaStructure()
	end
	local linkingFldArr = {}
	local tblName = tableName(tbl)
	if locf[tbl].relation and locf[tbl].relation[tblName] then
		for fldName, rec in pairs(locf[tbl].relation[tblName]) do
			linkingFldArr[#linkingFldArr + 1] = {linked_field = rec.linked_field, linking_field = tbl .. "." .. fldName, linking_table = tbl}
			-- linkingFldArr[#linkingFldArr+1] = {linked_field = rec.linked_field, linking_field = rec.linking_field, linking_table = rec.linking_table}
		end
	end
	return linkingFldArr
end

function dschema.linkingTableToArr(tbl)
	if not locf then
		schemaStructure()
	end
	local tblArr
	if locf[tbl] and locf[tbl].relation then
		for tblName, _ in pairs(locf[tbl].relation) do
			if tblArr == nil then
				tblArr = {}
			end
			tblArr[#tblArr + 1] = tablePrefix(tblName)
		end
	end
	return tblArr
end

local function cleanFieldName(fld)
	if fld == nil then
		util.printError("fld is nil")
		return
	end
	local pos = fld:find("json_data.", 1, true)
	if pos and pos > 0 then
		fld = fld:sub(pos + #"json_data.") -- after json_data
	end
	pos = fld:find(".", 1, true)
	if pos and pos > 0 then
		fld = fld:sub(pos + 1) -- remove table prefix
	end
	if fld:sub(-1) == "_" then -- remove trailing "_"
		fld = fld:sub(1, -1)
	end
	fld = fld:gsub("__", " / ") -- change __ to /
	fld = fld:gsub("_", " ") -- change _ to space
	if fld:sub(1, 1) == " " then -- remove trailing "_"
		fld = fld:sub(2)
	end
	fld = string.upper(fld:sub(1, 1)) .. fld:sub(2) -- uppercase first letter
	return fld
end
dschema.cleanFieldName = cleanFieldName

local loadSchemaLanguage
local function fieldLanguageName(fldRecType, language, schema, returnNil)
	if fldRecType == nil then
		util.printError("schema '%s', language '%s', field parameter does not exist", tostring(schema), tostring(language))
		return
	end
	if schema == nil then
		local connQuery = dconn.query()
		schema = connQuery and connQuery.schema
	end
	local lf = schemaStructure(schema)
	local extFld, recType = splitRecTypeName(fldRecType)
	local tblName = localTable(extFld, schema, recType, "no-error")
	if tblName == nil then
		tblName = localTable(extFld, schema, recType, "no-error")
		if returnNil then
			return
		end
	end
	local tblPrefix = tblName and localPrefix(tblName, schema, recType, "no-error")
	if tblPrefix == nil then
		tblPrefix = tablePrefix(fldRecType, schema, "no-error")
		-- is external table with no local_table
	end
	local fld = extFld -- localName(extFld, schema) or tblName -- tblPrefix contains record type in language
	if recType ~= "" then
		tblPrefix = recTypeName(tblPrefix, recType) -- example: "per - customer"
	end
	local langPrf = lf.language and lf.language[language]
	if langPrf == nil or langPrf[schema] == nil then
		langPrf = loadSchemaLanguage(schema, language, fldRecType)
	end
	if langPrf[schema] then
		langPrf = langPrf[schema]
	end
	if langPrf then
		local localFld = fld
		if schema ~= "" then
			fld = externalNameSql(localFld, schema, recType)
		end
		if fld == nil then
			fld = localFld
		else
			if langPrf[fld] then
				return langPrf[fld] -- common filed names like modify_id
			end
			if langPrf[tblPrefix] and langPrf[tblPrefix][fld] then
				return langPrf[tblPrefix][fld]
			end
			if langPrf.common and langPrf.common[fld] then
				return langPrf.common[fld] -- common filed names like modify_id
			end
			local fld2 = pegParseAfter(fld, ".")
			if langPrf[fld2] then
				return langPrf[fld2] -- common field name like record_id
			end
			if langPrf[tblPrefix] and langPrf[tblPrefix][fld2] then
				return langPrf[tblPrefix][fld2]
			end
			if langPrf[fld2] then
				return langPrf[fld2] -- common filed names like modify_id
			end
			if langPrf.common and langPrf.common[fld2] then
				return langPrf.common[fld2] -- common filed names like modify_id
			end
			if schema and schema ~= "" then
				local ret = fieldLanguageName(fldRecType, language, "", true) -- true = returnNil
				if ret then
					return ret
				end
			end
			--[[ fld2 = splitRecTypeName(fld2)
			if langPrf[fld2] then
				return langPrf[fld2] -- common field name like record_id
			end
			if langPrf[tblPrefix] and langPrf[tblPrefix][fld2] then
				return langPrf[tblPrefix][fld2]
			end ]]
		end
	end
	--[[ if schema ~= "" then
		return fieldLanguageName(fld, language, "") -- recursive call, default structure with schema ""
	end ]]
	--[[ local ret = l(fld)
	if ret ~= fld then
		return ret
	end ]]
	if returnNil then
		return
	end
	if not langPrf.error[fldRecType] then
		langPrf.error[fldRecType] = true
		--[[ if false then
			util.print("schema '%s', language '%s', field '%s' name was not found", tostring(schema), tostring(language), tostring(fldRecType))
		end ]]
	end
	return cleanFieldName(fld) -- backup name
end
dschema.fieldLanguageName = fieldLanguageName

loadSchemaLanguage = function(schema, language, tableOrField)
	local lf = schemaStructure(schema)
	if language == nil or language == "" then
		language = loc.defaultLanguage -- "en"
	end
	if not lf.language then
		lf.language = {}
	end
	if not lf.language[language] then
		lf.language[language] = {}
	end
	local prfName, prf
	if not lf.language[language].common then
		prfName = "table/prf/lang/common_" .. language .. ".json"
		prf = dprf.prf(prfName).language
		if prf then
			lf.language[language].common = prf
		else
			util.printWarning("common language file '%s' was not found", prfName)
		end
	end
	if schema and schema ~= "" and lf.language[language] and lf.language[language][schema] == nil then
		prfName = "table/external/" .. schema .. "/prf/lang/common_" .. language .. ".json"
		prf = dprf.prf(prfName, "no-error").language
		if prf then
			lf.language[language][schema] = prf
		else
			lf.language[language][schema] = {}
			util.printWarning("schema language file '%s' was not found", prfName)
		end
		lf.language[language][schema].common = lf.language[language].common
		lf.language[language][schema].error = {}
	elseif not lf.language[language][schema] then
		lf.language[language][schema] = {common = lf.language[language].common, error = {}}
	end
	if tableOrField then
		-- todo: this is ugly code - use standard local field names in calling this
		local tbl, recType = splitRecTypeName(tableOrField)
		tbl = localTable(tbl, schema, recType, "no-error") -- returns external table if local was not found
		if tbl == nil then
			util.printError("table '%s', schema '%s' table was not found", tostring(tableOrField), tostring(schema))
			-- tblPrefix = tbl and tablePrefix(tbl, schema, recType)
			return lf.language[language]
		end
		local tblPrefix = localPrefix(tbl, nil, recType, "no-error")
		if tblPrefix == nil then
			tblPrefix = tbl and tablePrefix(tbl, schema, recType)
		end
		if tblPrefix == nil then
			util.printError("table '%s', schema '%s' table prefix was not found", tostring(tableOrField), tostring(schema))
			-- tblPrefix = tbl and tablePrefix(tbl, schema, recType)
			return lf.language[language]
		end
		local tableRecType = recTypeName(tbl, recType)
		if recType ~= "" then
			tblPrefix = recTypeName(tblPrefix, recType)
		end
		local group
		group, groupPrfName = groupArray(schema)
		local rec = util.arrayRecord(tableRecType, group, "table_rec_type")
		if rec == nil and schema and schema ~= "" then
			rec = util.arrayRecord(tableRecType, group, "local_table")
		end
		if rec == nil then
			rec = util.arrayRecord(tableRecType, group, "table_name")
			if rec == nil and recType == "" and lf[tblPrefix] then
				recType = lf[tblPrefix].record_type_arr[1]
				if recType and recType ~= "" then
					tableRecType = recTypeName(tbl, recType)
					rec = util.arrayRecord(tableRecType, group, "table_name")
				end
			end
		end
		if rec == nil then
			rec = util.arrayRecord(tbl, group, "table_name")
		end
		if rec == nil then
			if tbl ~= "audit.log" then -- TODO: remove this
				util.printError("table '%s', schema '%s' group path was not found from file '%s'", tableRecType, schema, groupPrfName)
			end
			return lf.language[language]
		end
		if schema and schema ~= "" then
			prfName = "table/external/" .. schema .. "/" .. rec.group .. "lang/" .. language .. "/" .. rec.table_name .. ".json"
		else
			prfName = "table/" .. rec.group .. "lang/" .. language .. "/" .. rec.table_name .. ".json"
		end
		prf = dprf.prf(prfName, "no-error").language
		if prf == nil then
			local path = prfName
			local commonName = dprf.prf("table/prf/lang/common_en.json").language
			local oldPrf
			if schema == "4d" and language == "fi" then
				prfName = "table/external/4d/prf/lang/common_fi.json"
				oldPrf = dprf.prf(prfName, "no-error")
				prf = fn.iter(oldPrf.language):filter(function(key)
					return key == tbl or pegStartsWith(key, tblPrefix .. ".")
				end):reduce(function(acc, key, val)
					acc[key] = val
					oldPrf.language[key] = nil
					return acc
				end, {})
				if oldPrf and oldPrf.language and #oldPrf.language > 0 then
					util.print("table language preference '%s', updating file '%s'", prfName, path)
					fs.writeFile(path, oldPrf) -- remove used fields
				end
				prf[tblPrefix] = prf[tbl]
				prf[tbl] = nil
				--[[ 	elseif schema == "4d" then
				oldPrf = dprf.prf("table/external/4d/prf/lang/lang_default.json", "no-error").language
				local externalPrefix = tablePrefix(tblPrefix, schema)
				prf = fn.iter(oldPrf):filter(function(key)
					return key == externalPrefix or pegStartsWith(key, externalPrefix .. ".")
				end):reduce(function(acc, key, val)
					acc[key] = val.name
					return acc
				end, {}) ]]
			else
				local tblRec = locf[tbl]
				local recordType = tblRec.record_type_arr[1]
				local arr = fieldArray(tbl)
				local name, extFld
				local err = {}
				prf = fn.iter(arr):reduce(function(acc, fld)
					name = pegParseLast(fld, ".")
					if not commonName[name] then
						extFld = externalNameSql(fld, schema, recordType)
						if extFld then
							acc[fld] = fieldLanguageName(extFld, language, schema)
						else
							for i = 2, #tblRec.record_type_arr do
								extFld = externalNameSql(fld, schema, tblRec.record_type_arr[i])
								if extFld then
									acc[fld] = fieldLanguageName(extFld, language, schema)
									break
								end
							end
							if extFld == nil then
								acc[fld] = cleanFieldName(fld)
								err[#err + 1] = fld
							end
						end
						if acc[fld] == nil or acc[fld] == name then
							acc[fld] = cleanFieldName(fld)
						end
					end
					return acc
				end, {})
				extFld = externalNameSql(tbl, schema, recordType)
				if extFld then
					prf[tblPrefix] = fieldLanguageName(extFld, language, schema)
				else
					prf[tblPrefix] = cleanFieldName(tbl)
					err[#err + 1] = tblPrefix
				end
				if #err > 0 then
					prf.error = err
				end
				--[[ if prf[tblPrefix] == nil or prf[tblPrefix] == tbl then
					prf[tblPrefix] = cleanFieldName(tbl)
				end ]]
				--[[ 	oldPrf = dprf.prf("table/external/4d/prf/lang_default.json", "no-error").language
				prf = fn.iter(oldPrf):filter(function(key)
					return key == tblPrefix or pegStartsWith(key, tblPrefix..".")
				end):reduce(function(acc, key, val)
					acc[key] = val.name
					return acc
				end, {}) ]]
			end
			if prf and #prf > 0 then
				util.print("table language preference '%s' was not found, creating file '%s'", prfName, path)
				fs.createFilePath(path)
				fs.writeFile(path, {name = prfName, language = prf})
			end
		end
		--[[ if schema and schema ~= "" and not onlyExternal then
			local prfLocal = {}
			local localFld, err
			for extFld, val in pairs(prf) do
				localFld, err = localName(extFld, schema) -- record type is in tblPrefix
				if err then
					util.printWarning("no local field found for external field '%s', schema '%s'", extFld, schema)
				end
				prfLocal[localFld] = val
			end
			prf = prfLocal
		end ]]
		lf.language[language][schema][tblPrefix] = prf
	end
	return lf.language[language]
end

function dschema.tableNameDropdownArr(schema, language)
	language = language or ""
	local cacheId = dconn.organizationId() -- dconn.localConnectionId()
	schema = schema or ""
	local key = schema .. "/" .. language
	if loc.tableNameArr == nil then
		loc.tableNameArr = {[cacheId] = {}}
		dprf.registerCache("dschema-loc.tableNameArr", loc.tableNameArr)
	end
	if loc.tableNameArr[cacheId] == nil or util.tableIsEmpty(loc.tableNameArr[cacheId]) then
		loc.tableNameArr[cacheId] = {}
	elseif loc.tableNameArr[cacheId][key] then
		return loc.tableNameArr[cacheId][key]
	end
	local tableNameArr = {}
	local lf = schemaStructure(schema)
	local i = 0
	local langPrf
	langPrf = lf.language and lf.language[language]
	if not langPrf then
		langPrf = loadSchemaLanguage(schema, language)
		langPrf = langPrf[schema]
	end
	for _, tbl in ipairs(lf._table) do
		for idx, recType in ipairs(tbl.record_type_arr) do
			if idx > 1 and schema ~= "" then
				break
			end
			i = i + 1
			-- if tbl.table_name_language == nil then
			-- util.print("table '%s', record type '%s', table_name_language does not exist", tostring(tbl.table_name), tostring(recType))
			if tbl.table_name_language and tbl.table_name_language[recType] == nil then
				util.printWarning("table '%s', table_name_language does not contain key of record type '%s'", tostring(tbl.table_name), tostring(recType))
			end
			local tblName = tbl.local_table or tbl.table_name
			local tblPrefix = localPrefix(tblName, schema, recType, "no-error")
			if tblPrefix == nil then
				tblPrefix = tbl.table_prefix -- is external table with no local_table
			end
			if recType then -- and recType ~= "" then
				tblPrefix = recTypeName(tblPrefix, recType)
			end
			-- todo: fix ext names and prefixes
			local show = langPrf[tblPrefix] and (langPrf[tblPrefix][tblName] or langPrf[tblPrefix][tbl.table_prefix])
			if show == nil and (langPrf.error == nil or next(langPrf.error) == nil) then -- langPrf.error check is need or loadSchemaLanguage() will be cause infinite loop
				local name = recTypeName(tblName, recType)
				langPrf = loadSchemaLanguage(schema, language, name)
				langPrf = langPrf[schema]
				show = langPrf[tblPrefix] and (langPrf[tblPrefix][tblName] or langPrf[tblPrefix][tbl.table_prefix])
			end
			local localTblName
			if schema ~= "" then
				localTblName = toLocal(tbl.table_name, schema)
			else
				localTblName = tbl.table_name
				recType = ""
			end
			if show == nil then
				if tbl.table_name_language == nil then
					local localTbl = locf[tbl.local_table] or locf[tbl.table_name]
					show = localTbl and localTbl.table_name_language and localTbl.table_name_language[recType] or tbl.table_name
				else
					show = tbl.table_name_language and tbl.table_name_language[recType] or tbl.table_name
				end
				if show:sub(1, 1) == "_" then
					show = show:sub(2) -- _change, _report, _link, "_preference"
				end
				if show:sub(1, 1) ~= show:sub(1, 1):upper() or peg.found(show, "_") then
					show = peg.replace(show:sub(1, 1):upper() .. show:sub(2), "_", " ")
				end
			end
			if recType ~= "" then
				recType = recordTypeSeparator .. recType
			end
			tableNameArr[i] = {show = show, value = localTblName and localTblName .. recType, table_group = tbl.table_group}
			if schema == "" then
				break -- show only first default record type
			end
		end
	end
	table.sort(tableNameArr, function(a, b)
		return a.show < b.show -- todo: test international sorting
	end)
	loc.tableNameArr[cacheId][key] = tableNameArr
	return tableNameArr
	--[[
	if pretty and loc.tablePrettyNameArr and loc.tablePrefixArr then
		return loc.tablePrettyNameArr, loc.tablePrefixArr
	elseif not pretty and loc.tableNameArr and loc.tablePrefixArr then
		return loc.tableNameArr, loc.tablePrefixArr
	end
	if not locf then
		schemaStructure()
	end
	for _, rec in pairs(locf) do
		if rec.table_prefix and rec.table_name then
			if pretty then
				loc.tablePrettyNameArr = loc.tablePrettyNameArr or {}
				loc.tablePrettyNameArr[#loc.tablePrettyNameArr + 1] = cleanFieldName(rec.table_prefix.."_"..rec.table_name)
			else
				loc.tableNameArr = loc.tableNameArr or {}
				loc.tableNameArr[#loc.tableNameArr + 1] = rec.table_name
			end
			loc.tablePrefixArr = loc.tablePrefixArr or {}
			loc.tablePrefixArr[#loc.tablePrefixArr + 1] = rec.table_prefix
		end
	end
	if pretty then
		return loc.tablePrettyNameArr, loc.tablePrefixArr
	end ]]
end

local function mapFieldValue(mapRec, value)
	if type(mapRec) == "table" then
		if type(value) == "table" then
			local newValue = {}
			fn.iter(value):each(function(val)
				newValue[#newValue + 1] = mapRec[val] or val
			end)
			return newValue
		else
			return mapRec[value] or value
		end
	end
	return value
end

function dschema.fieldSqlMapValue(field, schema, recordType, value)
	-- "res.resource_type" -> "4d: wlg_type_": "load_group" -> 0
	local rec = externalRec(field, schema, recordType)
	if rec == nil then
		rec = locf[field]
	end
	if rec and rec.value_map then
		return mapFieldValue(rec.value_map, value)
	end
	return value
end

function dschema.fieldHasMapValue(field, schema, recordType)
	local rec = externalRec(field, schema, recordType)
	if rec == nil then
		rec = locf[field]
	end
	if rec and rec.value_map then
		return true
	end
	return false
end

function dschema.fieldLocalMapValue(field, schema, recordType, value)
	-- "res.resource_type" -> "4d: wlg_type_": "load_group" -> 0
	local rec = externalRec(field, schema, recordType)
	if rec == nil then
		rec = locf[field]
	end
	if rec and rec.value_map_invert then
		return mapFieldValue(rec.value_map_invert, value)
	end
	return value
end

-- copied from dschemasql:
function dschema.fieldNamePrefixSql(field, schema, option) -- was in dschemasql
	schema = schema or dconn.schema()
	local recordType = dconn.recordType()
	local name = fieldNamePrefix(field, schema, recordType)
	if name == nil then
		name = fieldNamePrefix(field)
	end
	if type(name) == "string" and option ~= "no-quote" then
		return quoteSql(name)
	end
	return name
	--[=[ local fld
	local rec = nameChange(field)
	if rec and type(rec.field_name) == "string" then
		if option ~= "no-quote" then
			fld = rec.table_prefix or rec.table_rec.table_prefix .. "." .. quoteSql(rec.field_name)
		else
			fld = rec.table_prefix or rec.table_rec.table_prefix .. "." .. rec.field_name
		end
	end
	if rec == nil then
		fld = dschema.fieldNamePrefix(field)
	end
	-- todo: clean this ugly fix, bka.db_id -> ac.db_id
	local prefix = dschema.tableName(fld)
	prefix = prefix and dschema.tablePrefixSql(prefix)
	if prefix and not peg.found(fld, prefix) then
		fld = prefix .. peg.parseAfterWithDivider(fld, ".")
	end
	return fld
	-- end hack
	--[[local newName = rec and rec.field_name and ((rec.table_prefix or rec.table_rec.table_prefix).."."..rec.field_name)
	if type(newName) == "string" and option ~= "no-quote" then
		return quoteSql(newName)
	end
	return newName
	]] ]=]
end

function dschema.tableNameSql(fieldOrTbl, schema, option) -- was in dschemasql
	schema = schema or dconn.schema()
	local recordType = dconn.recordType()
	local name = tableName(fieldOrTbl, schema, recordType)
	if name == nil then
		name = fieldNamePrefix(fieldOrTbl)
	end
	if type(name) == "string" and option ~= "no-quote" then
		return quoteSql(name)
	end
	return name
	--[[ local rec = nameChange(fieldOrTbl)
	local newName = rec and (rec.table_name or rec.table_rec and rec.table_rec.table_name)
	if type(newName) == "string" and option ~= "no-quote" then
		return quoteSql(newName)
	end
	if newName == nil then
		newName = dschema.tableName(fieldOrTbl)
	end
	return newName ]]
end

function dschema.tablePrefixSql(field) -- was in dschemasql
	local schema = dconn.schema()
	local recordType = dconn.recordType()
	local name = tablePrefix(field, schema, recordType)
	if name == nil then
		name = tablePrefix(field)
	end
	return name
	--[=[
	local rec = nameChange(field, "table_prefix")
	local newName = rec and (rec.table_prefix or rec.table_rec and rec.table_rec.table_prefix)
	--[[if type(newName) == "string" and option ~= "no-quote" then
		return quoteSql(newName)
	end]]
	if rec == nil then
		newName = dschema.tablePrefix(field)
	end
	return newName
	]=]
end

function dschema.reloadSchema(schema)
	return dschemafld.reloadSchema(schema)
end

--[[
function dschema.fieldNameSql(field, schema, recordType, option)
	local rec = externalRec(field, schema, recordType)
	local name = rec and rec.field_name or field
	if type(name) == "string" and option ~= "no-quote" then
		return quoteSql(name)
	end
	return name
	--[=[
	dschema.fieldNameSql(field, option, useLocal) -- was in dschemasql
		local rec = nameChange(field) or {}
		local name = rec.field_name
		if type(name) == "string" then
			if option ~= "no-quote" then
				return quoteSql(name)
			end
		end
		if name == nil and useLocal == nil or useLocal then
			name = dschema.fieldName(field)
		end
		return name ]=]
end ]]

return dschema
