--- lib/db/dschemasql.lua
-- local and external table and field names
-- @module db
local dschemasql = {}

local dconn = require "dconn"
local util = require "util"
local fn = require "fn"
local peg = require "peg"
local dschemafld = require "dschemafld"
local dschema = require "dschema"

local quoteSql = dschema.quoteSql
local recordTypeSeparator = dschemafld.recordTypeSeparator
local schemaSeparator = dschemafld.schemaSeparator
local defaultDatabaseSchema = ""
local loc = {}
loc.debug = nil
local fieldCache = {}
local locf
local extf = {}

--[[ function dschemasql.clearCache()
	fieldCache = {}
end ]]

local function unQuoteSql(fld)
	if type(fld) ~= "string" then -- can be quoted name
		if fld ~= 1 then
			util.printError("sql table or field name type '%s' is not a string", tostring(fld))
		end
		--[[ elseif peg.found(fld, '"') then
		util.printWarning("sql table or field name '%s' is quoted, fixing it to non-quoted name", fld)
		return peg.replace(fld, '"', "")  -- fld:sub(2, -2)
		]]
	elseif (fld:sub(1, 1) == '"' and fld:sub(-1) == '"') or (fld:sub(1, 1) == '"' and fld:sub(-1) == '"') then -- can be quoted name
		-- we should check all quote possibilities from table/prf/reserved_name.json (if there are things like backtick)
		util.printWarning("sql table or field name '%s' is quoted, fixing it to non-quoted name", fld)
		return fld:sub(2, -2)
	end
	return fld
end

local function printConn(rec, extRec, organization_id, cacheKey, redirKey, schemaKey, redir)
	if extRec and rec.field_name ~= extRec.field_name then
		util.printOk("'%s', redirect id '%s', cache key '%s', redirect key '%s', schema key '%s', field '%s', redirect field '%s'", organization_id, redir and redir.organization_id, cacheKey, redirKey, schemaKey, rec.field_name or rec.table_name, extRec.field_name or extRec.table_name)
		util.print(util.callPath():sub(1, 550) .. "...")
	else
		util.printOk("'%s', cache key '%s', redirect key '%s', schema key '%s', field '%s'", organization_id, cacheKey, redirKey, schemaKey, rec.field_name or rec.table_name)
	end
end

local function nameChange(fld_)
	if type(fld_) ~= "string" then
		util.printError("field name change parameter is not a string: '%s'", tostring(fld_))
		return
	end
	if not locf then
		if loc.debug == nil then
			loc.debug = util.prf("system/debug.json").debug.dschemasql
		end
		locf = dschemafld.fieldTableStructure()
	end
	local conn = dconn.currentConnection()
	if conn == nil then
		return
	end
	local connQuery = conn.query
	if not connQuery then
		return
	end
	local schema = connQuery.schema -- or conn.schema
	if schema == nil then
		util.printError("query schema is nil, field '%s'", tostring(fld_))
		return
	end
	if connQuery.recordType == nil then
		util.printError("query recordType is nil")
		return
	end
	local fld, recordType = peg.splitLast(unQuoteSql(fld_), recordTypeSeparator)
	if recordType == "" then
		recordType = connQuery.recordType
	end
	local extRec
	local rec = locf[fld]
	if schema ~= defaultDatabaseSchema then
		if not extf[schema] then -- we must create ext schema so that normal rec contains rec.external -tag
			extf[schema] = dschemafld.fieldTableStructure(schema)
		end
		extRec = rec and rec.external and rec.external[schema .. schemaSeparator .. recordType]
		if extRec == nil and rec and rec.external then
			local tbl = rec.table_name or rec.table_rec.table_name
			local recordType2 = connQuery.tableRecordType and connQuery.tableRecordType[tbl]
			recordType2 = recordType2 and recordType2[1]
			if recordType2 then
				extRec = rec.external[schema .. schemaSeparator .. recordType2]
			end
		end
		if extRec == nil then
			-- find external table first
			extRec = extf[schema][fld]
			if extRec == nil then
				extRec = dschema.externalSchemaRec(fld, schema)
			end
			if extRec == nil and rec and rec.external then
				local tblRec = rec.table_rec or rec
				if tblRec then
					-- get table first record type as default record type
					if type(tblRec.record_type_arr) ~= "table" then
						if peg.found(fld, "json_data.") then
							local field2 = peg.parseBeforeWithDivider(fld, "json_data")
							tblRec = locf[field2]
							tblRec = tblRec.table_rec or tblRec
						end
						if tblRec and tblRec.record_type_arr then
							extRec = rec.external[schema .. schemaSeparator .. recordType]
							-- extRec = rec.external[schema..schemaSeparator..tblRec.record_type_arr[1]]
						else
							util.printError("field '%s' table record_type array type '%s' is not a table", tostring(fld), type(tblRec.record_type_arr))
							extRec = rec.external[schema .. schemaSeparator]
						end
					else
						extRec = rec.external[schema .. schemaSeparator .. recordType]
						-- extRec = rec.external[schema..schemaSeparator..tblRec.record_type_arr[1]]
					end
				end
			end
		end
		if extRec then -- use extRec only if it is found, else use locf[fld]
			rec = extRec
		end
	end
	if not rec then
		fld = tostring(fld)
		if peg.found(fld, '."') then
			util.printWarning("sql name for '%s', record type '%s' could not be found, query '%s'", fld, connQuery.recordType, connQuery.queryNameFunc())
		else
			util.printError("sql name for '%s', record type '%s' could not be found, query '%s'", fld, connQuery.recordType, connQuery.queryNameFunc())
		end
		return
	end
	local recTableName = rec.table_name or rec.table_rec and rec.table_rec.table_name
	if recTableName == nil then
		util.printError("table record for '%s', record type '%s' could not be found, query '%s'", fld, connQuery.recordType, connQuery.queryNameFunc())
		return
	end
	local tableName, tableRecordType = peg.splitLast(recTableName, recordTypeSeparator)
	if recordType == "" then
		if tableRecordType ~= "" then
			recordType = tableRecordType
		elseif connQuery.tableRecordType and connQuery.tableRecordType[tableName] and connQuery.tableRecordType[tableName][1] ~= "" then
			recordType = connQuery.tableRecordType[tableName] and connQuery.tableRecordType[tableName][1]
		elseif connQuery.recordType ~= "" then
			recordType = connQuery.recordType
		end
	end
	local cacheKey = schema .. schemaSeparator .. fld .. recordTypeSeparator .. recordType
	local fieldCacheRec = fieldCache[cacheKey]
	if fieldCacheRec then
		if loc.debug then
			util.print(" - cached '%s', cache key '%s', field/table '%s'", conn.organization_id, cacheKey, rec.field or rec.table_name)
		end
		extRec = fieldCacheRec.extRec
		if extRec then
			rec = extRec
		end
	else
		local err
		if not extRec and schema ~= "" then -- option == nil == use local name
			extRec = rec.external and rec.external[schema .. schemaSeparator .. recordType]
			-- we need to use dschemafld.linkRec() and not search for rec.external[schema..schemaSeparator]
			if not extRec and rec.external and tableName == connQuery.table and recordType == "" then
				-- like tableName == product and record type == ""
				extRec = rec.external[schema .. schemaSeparator .. tableName] -- todo: is it ok to use table name instead of record_type?
			end
			if not extRec then
				local linkRec -- , extTableRec
				linkRec, err = dschemafld.linkRec(tableName) -- connQuery.table
				--[[ linkRec, err, extTableRec = dschemafld.linkRec(tableName) -- connQuery.table
				if linkRec == nil and extTableRec and extTableRec._field then -- and tableRec.local_table == rec.table_name then -- found directly extRec
					extRec = util.arrayRecord(fld, extTableRec._field, "local_field", 1)
				else ]]
				extRec = linkRec and linkRec.linked_record_type and rec.external and rec.external[schema .. recordTypeSeparator .. linkRec.linked_record_type]
				-- end
				if not extRec and rec.external then
					extRec = rec.external[schema .. schemaSeparator] -- TODO: test if this is needed and correct
					if extRec and linkRec then
						util.printWarning("find '%s', rec.external[" .. schema .. schemaSeparator .. linkRec.linked_record_type .. "] using linkRec.linked_record_type was not found but rec.external[" .. schema .. schemaSeparator .. "] was found", fld)
					elseif extRec then
						util.printWarning("find '%s', rec.external[" .. schema .. schemaSeparator .. recordType .. "] was not found but rec.external[" .. schema .. "/] was found", fld)
					end
				end
			end
			if not extRec then
				if err then
					if rec.table_name then
						util.printWarning("find table '%s', external table was not found, query table '%s', record type '%s', error: %s", fld, connQuery.table, recordType, err)
					else
						util.printWarning("find field '%s', external field was not found, query table '%s', record type '%s', error: %s", fld, connQuery.table, recordType, err)
					end
				end
				return
			end
		end
		if loc.debug then
			local redirKey = tableName .. recordTypeSeparator .. recordType
			local redirect = connQuery.redirect
			local schemaKey = schema -- ?
			printConn(rec, extRec, conn.organization_id, cacheKey, redirKey, schemaKey, redirect)
		end
		rec = extRec or rec -- call printConn() before setting rec
		fieldCache[cacheKey] = {extRec = extRec, useCount = 0}
		fieldCacheRec = fieldCache[cacheKey]
	end
	if rec == nil then
		util.printError("sql name for '%s' could not be found, query '%s'", tostring(fld), connQuery.queryNameFunc() or "")
		return
	end
	fieldCacheRec.useCount = fieldCacheRec.useCount + 1
	return rec
end
dschemasql.nameChange = nameChange

return dschemasql
