--- dsql.lua
-- Database sql text creation.
-- local dsql = require "dsql"
-- see: https://github.com/vitaly-t/pg-minify: SELECT*FROM"table"WHERE col=123;
-- @module db
local dsql = {test = {}} -- test functions, not public

local util = require "util"
local peg = require "peg"
local dt = require "dt"
local fn = require "fn"
local dprf = require "dprf"
local utf = require "utf"
local l = require"lang".l
local rest4d -- = require "db/database-rest4d"
local startsWith, parseBefore, parseBeforeLast, parseAfter, splitToArray, removeFromStart, parseBeforeWithDivider = peg.startsWith, peg.parseBefore, peg.parseBeforeLast, peg.parseAfter, peg.splitToArray, peg.removeFromStart, peg.parseBeforeWithDivider
local pegReplace, pegFound, pegFind, pegLower = peg.replace, peg.found, peg.find, peg.lower

local recordTypeSeparator, linkSeparator
local json, dconn, dschema, dqry, dqjson, uuid4d, execute -- delay load
local constantPrefixArr, quoteSql, fieldSqlMapValue
local sqlQuerySqlCreate, sqlSaveSqlCreate, sqlDeleteSqlCreate, fieldNameToSqlFunction -- forward declaration functions
local concat = table.concat
local noLowerField = {record_id = true, modify_id = true, change_id = true}
-- local escapeSingleQuote = "\\'" -- is this standard?
local escapeSingleQuote = "''" -- 4D supports '', Use Two Single Quotes For Every One Quote To Display: Vendors: Oracle, SQL Server, MySQL, PostgreSQL. In MySQL, you can add a backslash before the quote to escape it.

local loc = {}
loc.orderBeforeGroupSql = {["4d"] = true}
loc.lastQueryName = nil
loc.allTableLinkChain = nil
loc.tablePair = nil
loc.printSqlVar = false
loc.printSqlPrevTime = nil
loc.pegPattDot = peg.toPattern(".")
loc.debug = nil -- will be set later from system/debug.json / debug
loc.debugExecute = nil -- will be set later from system/debug.json / sql_execute or sql_execute_4d
loc.sqlDebugLength = 120
loc.sqlSelectDebugLength = 70 -- part before FROM, SELECT fld, fld2, ...
loc.sqlSelectDebugLength2 = 150 -- part after FROM, joins, where, order and limit
loc.maxQueryNameLength = 200
loc.substringLength = 0 -- 200 -- util.prf("connection.json").substring_length or 200
-- todo: fix substringLength back after Otava is done
-- loc.queryMaxRowsLimit = 15 * 10 ^ 6 -- 15 million rows
-- local defaultDatabaseSchema = ""
loc.sqlAlias = " " -- or " AS ", but " " is supported everywhere
-- local defaultOrderBy = {}

local function loadLibs()
	if not dschema then
		-- dsqlOld = require "dsql-old"
		dconn = require "dconn"
		execute = require "execute"
		json = require "json"
		dqry = require "dqry"
		dqjson = require "dqjson"
		dschema = require "dschema"
		quoteSql = dschema.quoteSql
		fieldSqlMapValue = dschema.fieldSqlMapValue
	end
end

function dsql.loadLibs()
	loadLibs()
end

function dsql.setQueryName(name)
	if name == "" then
		name = nil
	end
	loc.lastQueryName = name
end

local function lastQueryName(connQuery)
	connQuery = connQuery or dconn.query()
	if connQuery.queryNameFunc then
		return connQuery.queryNameFunc()
	end
	return loc.lastQueryName or dqjson.lastQueryJsonName() or ""
end
dsql.lastQueryName = lastQueryName

function dsql.sqlAlias()
	return loc.sqlAlias
end

function dsql.showSqlOn()
	return loc.printSqlVar
end

function dsql.showSql(val)
	local prevValue2 = loc.printSqlVar
	if val == nil then
		loc.printSqlVar = true
	else
		loc.printSqlVar = val
	end
	return prevValue2
end

local function debugExecute()
	if loc.debugExecute == nil then
		local debug = util.prf("system/debug.json").debug
		if util.from4d() then
			loc.debugExecute = debug.sql_execute_4d
		else
			loc.debugExecute = debug.sql_execute
		end
		loc.sqlDebugLength = debug.sql_execute_length
	end
	return loc.debugExecute
end

function dsql.debugSql(val)
	local prevValue = loc.debugExecute
	if val == nil then
		loc.debugExecute = nil
		loc.debugExecute = debugExecute() -- re-set to preference
	else
		loc.debugExecute = val
	end
	return prevValue
end

function dsql.addExtraJoin(arr)
	local conn = dconn.currentConnection() -- dschema call allowed only here, will add table_redirect -rec to conn.connection
	if not conn then
		return l("connection does not exist")
	elseif conn.error then
		return conn.error
	end
	conn.query.extraJoinArr = arr
end

local function defaultQueryNameFunc()
	return "unknown query name"
end

local function initWhereArr(connSql)
	connSql = connSql or dconn.sql()
	connSql.whereArr = {}
	connSql.returnArr = {}
	connSql.whereUsedTableArr = {}
	connSql.whereUsedTable = {}
end
dsql.initWhereArr = initWhereArr

function dsql.getWhereArr()
	local connSql = dconn.sql()
	return connSql.whereArr
end

local function addUsedTable(localTable, sqlTable, connSql, recType)
	if localTable == nil then
		util.printError("local table is nil when adding used table")
		return
	elseif sqlTable == nil then
		util.printError("external table is nil when adding used table")
		return
	end
	connSql = connSql or dconn.sql()
	local connQuery = dconn.query()
	if #connSql.usedTableArr == 0 then
		local schema = dconn.schema()
		local localTable2, recType2, sqlTable2
		for _, localTbl in ipairs(connQuery.linkTable) do
			-- for _, recType in ipairs(connQuery.tableRecordType[localTbl]) do
			localTable2, recType2 = dschema.splitRecTypeName(localTbl)
			sqlTable2 = dschema.tableName(localTable2, schema, recType2)
			if sqlTable2 == nil then
				util.printError("adding used table '%s', schema '%s', record type '%s', external table is nil", tostring(localTbl), tostring(schema), tostring(recType))
			elseif connSql.usedTable[sqlTable2] == nil then
				connSql.usedTable[sqlTable2] = true
				connSql.usedTableArr[#connSql.usedTableArr + 1] = {sql_table = sqlTable2, local_table = localTable2, record_type = recType2, link_field = connQuery.option and connQuery.option.link_field}
			else
				util.printWarning("adding used table '%s', schema '%s', record type '%s', external table '%s' was already defined", tostring(localTbl), tostring(schema), tostring(recType), tostring(sqlTable2))
			end
			-- end
		end
	end

	if connSql.usedTable[sqlTable] == nil then
		connSql.usedTable[sqlTable] = true
		if sqlTable == "session" and #connSql.usedTableArr > 0 then
			connSql.usedTableArr = connSql.usedTableArr
		end
		connSql.usedTableArr[#connSql.usedTableArr + 1] = {sql_table = sqlTable, local_table = localTable, record_type = recType, link_field = connQuery.option and connQuery.option.link_field}
	end
end
dsql.addUsedTable = addUsedTable

local function sqlQueryFromAdd(connQuery, connSql)
	-- NOTE: this is called only once, there can't be more than one table in FROM if using joins
	local recType = connQuery.recordType or ""
	local tblName = connQuery.mainTable
	local tblNameSql = dschema.tableName(tblName, connQuery.schema, recType)
	addUsedTable(tblName, tblNameSql, connSql, recType)
	local prefixSql = dschema.tablePrefix(tblName, connQuery.schema, recType)
	tblNameSql = quoteSql(tblNameSql)
	connSql.from = tblNameSql .. loc.sqlAlias .. prefixSql -- -- tblName.." AS "..prefix
end

function dsql.getWhereUsedTableArr()
	local connSql = dconn.sql()
	return connSql.whereUsedTableArr
end

local function initOrderArr(connSql)
	connSql = connSql or dconn.sql()
	connSql.orderArr = {}
	connSql.fieldSortParam = {}
end
dsql.initOrderArr = initOrderArr

function dsql.getOrderArr()
	local connSql = dconn.sql()
	return connSql.orderArr
end

function dsql.getOrderUsedTableArr()
	local connSql = dconn.sql()
	return connSql.orderUsedTableArr or {}
end

function dsql.addOrder(fld, asc, useLower, localField)
	local connSql = dconn.sql()
	local connQuery = dconn.query()
	local fieldNameSql = dschema.fieldNamePrefix(fld, connQuery.schema, connQuery.recordType) -- returns quoted field name
	if fieldNameSql == nil then
		fieldNameSql = fld
	end
	if useLower and noLowerField[parseAfter(fld, ".")] then
		useLower = false
	end
	if useLower then
		-- connSql.orderArr[#connSql.orderArr + 1] = "lower(" .. quoteSql(fieldNameSql) .. ")" .. asc
		connSql.orderArr[#connSql.orderArr + 1] = "lower(" .. fieldNameSql .. ")" .. asc
		connSql.fieldSortParam[#connSql.fieldSortParam + 1] = localField
		connSql.fieldSortParam[#connSql.fieldSortParam + 1] = asc == " DESC" and "<" or ">"
	else
		-- connSql.orderArr[#connSql.orderArr + 1] = quoteSql(fieldNameSql) .. asc
		connSql.orderArr[#connSql.orderArr + 1] = fieldNameSql .. asc
		connSql.fieldSortParam[#connSql.fieldSortParam + 1] = localField
		connSql.fieldSortParam[#connSql.fieldSortParam + 1] = asc == " DESC" and "<" or ">"
	end
	local tblName = dschema.tableName(fld)
	if tblName == nil then
		tblName = dschema.localTable(fld, connQuery.schema, connQuery.recordType)
		if tblName == nil then
			tblName = dschema.toLocal(parseBefore(fld, "."), connQuery.schema, connQuery.recordType)
		end
	end
	local tblNameSql = dschema.tableName(fld, connQuery.schema, connQuery.recordType)
	if tblNameSql == nil then
		local fld2 = dschema.unQuoteSql(fld)
		tblNameSql = dschema.tableName(fld2, connQuery.schema, connQuery.recordType)
	end
	addUsedTable(tblName, tblNameSql, connSql) -- do not guess record type
end

local function clearQuerySetLimitOffset(conn)
	-- conn.query.setStart and conn.query.setEnd may be totally different - they are preserved between queries and cleaned after dsql.sqlQueryBuild()
	conn.query = conn.query or {}
	conn.query.setStart = {}
	conn.query.setEnd = {}
	conn.sql.offset = -1
	conn.sql.limit = -1
end

local function clearQuerySql(conn, clearOrder)
	conn.sql = conn.sql or {}
	local connSql = conn.sql
	connSql.selectArr = {}
	connSql.selectArrUsed = {}
	connSql.selectArrUsedTable = {}
	connSql.from = nil
	-- connSql.fromArr = {}
	-- connSql.fromUsedTable = {}
	-- connSql.fromUsedTableArr = {}
	connSql.joinArr = {}
	connSql.insertArr = {}
	connSql.updateArr = {}
	initWhereArr(connSql)
	if clearOrder then
		connSql.usedTable = {}
		connSql.usedTableArr = {}
		initOrderArr(connSql)
		connSql.aggregate = nil
	else
		if connSql.orderArr == nil or #connSql.orderArr == 0 then
			-- TODO: test this and remove if needed
			connSql.usedTable = {}
			connSql.usedTableArr = {}
		end
		connSql.orderArr = connSql.orderArr or {}
		-- connSql.orderArrUsedTable = connSql.orderArrUsedTable or {}
		-- connSql.aggregate = nil
	end
	connSql.containsEmptyArray = nil
	connSql.containsOr = nil
	if conn.query == nil then
		clearQuerySetLimitOffset(conn)
	end
	-- offset and limit are cleared in clearQuerySetLimitOffset()
	if type(connSql.offset) ~= "number" then
		connSql.offset = -1
	end
	if type(connSql.limit) ~= "number" then
		connSql.limit = -1
	end
end

function dsql.clearQuerySet(connQuery)
	connQuery.setStart = {} -- these are preserved between queries
	connQuery.setEnd = {}
end

function dsql.clearQuery(conn, clearTable)
	--[[
	if not conn then
		local all = dconn.allConnections()
		for _, conn2 in pairs(all) do
			dsql.clearQuery(conn2)
		end
	end --]]
	loadLibs()
	conn = conn or dconn.currentConnection()
	if conn == nil then
		return
	end
	-- loc.lastQueryName = nil
	clearQuerySql(conn, true)
	if conn.query == nil then
		conn.query = {}
	end
	local connQuery = conn.query
	if clearTable == nil or clearTable then
		connQuery.table = nil
		connQuery.mainTable = nil
		connQuery.linkTable = {}
		connQuery.tableRecordType = nil
		connQuery.redirectId = nil
		connQuery.redirect = nil
		connQuery.schema = nil
		connQuery.recordType = nil
	end
	connQuery.option = nil
	connQuery.queryArr = {}
	connQuery.deleteArr = {}
	connQuery.saveArr = {}
	connQuery.extraJoinArr = {}
	connQuery.query_count = 0
	if type(connQuery.queryNameFunc) ~= "function" then
		connQuery.queryNameFunc = defaultQueryNameFunc -- return previous query name by default even when query has been cleared
	end
	connQuery.queryText = ""
	connQuery.setStart = connQuery.setStart or {} -- these are preserved between queries
	connQuery.setEnd = connQuery.setEnd or {}
end

function dsql.sqlSetOption(option)
	local connQuery = dconn.query()
	if connQuery then
		connQuery.option = option
	end
end

local function qryRecNameFunc(qryRec)
	loc.lastQueryName = nil
	if type(qryRec.query_name) == "string" then
		return function()
			return qryRec.query_name
		end
	elseif type(qryRec.value) == "table" and #qryRec.value > 10 then -- don't make too big string
		return function()
			local qryRec2 = util.clone(qryRec)
			qryRec2.value = fn.iter(qryRec2.value):take(3):totable()
			qryRec2.value = {table.concat(qryRec2.value, ", ") .. "... " .. l("%d values to show", #qryRec.value) .. " ..."}
			return json.toJsonRaw(qryRec2)
		end
	end
	return function()
		return json.toJsonRaw(qryRec)
	end
end

local function setConnection(tblRec_, queryName)
	dsql.loadLibs()
	local prevConn = dconn.getCurrentConnection() -- dconn.connection({createConnection = false})
	local queryNameFunc
	if type(queryName) == "function" then
		queryNameFunc = queryName
	elseif type(queryName) == "table" then
		queryNameFunc = function()
			local txt = json.toJsonRaw(queryName)
			if #txt > loc.maxQueryNameLength then
				txt = txt:sub(1, loc.maxQueryNameLength) .. "..."
			end
			return txt
		end
	else
		queryNameFunc = function()
			if type(queryName) == "function" then
				return tostring(queryName())
			end
			return tostring(queryName)
		end
	end

	if type(tblRec_) ~= "table" then
		util.printError("dsql.setConnection() parameter is not a table, query '%s', parameter: %s", queryNameFunc(), tostring(tblRec_))
		return prevConn
	end
	local tblRec, err = dconn.queryTableParamConvert(tblRec_) -- sets tblRec.table and tblRec.record_type
	if err then
		util.printError("dsql.setConnection() error: %s, query '%s', parameter: %s", err, queryNameFunc(), json.toJson(tblRec_))
		return prevConn
		--[[elseif tblRec_.table and tblRec.table ~= tblRec_.table then
		-- this case is ok when in query json main table is different than query's query field, like searching order_row data using order's field
		util.printWarning(
			"dsql.setConnection() error: query table '%s' is not same as converted query table '%s', query '%s', parameter: %s",
			tblRec.table,
			tblRec_.table,
			queryNameFunc(),
			json.toJson(tblRec_)
		) ]]
	end
	-- local tableName = dschema.tableName(tblRec.table) -- TODO: remove this, never use tbl prefix
	local conn, redirect
	conn, redirect, err = dconn.setAuthAndRedirect(tblRec.table, tblRec.record_type) -- redirect can be nil == default database, but prevConn can be different
	if err then
		util.printRed("dconn.setAuthAndRedirect returned error '%s', query '%s', parameter: %s", err, queryNameFunc(), json.toJson(tblRec_))
	end
	if conn then
		-- dsql.clearQuery(conn)
		util.recToRec(conn.query, redirect)
		conn.query.mainTable = tblRec.table
		conn.query.linkTable = tblRec.linkTable
		conn.query.linkedTable = tblRec.linkedTable
		conn.query.tableRecordType = tblRec.tableRecordType
		conn.query.recordType = tblRec.record_type
		conn.query.queryNameFunc = queryNameFunc
	end
	return prevConn
end
dsql.setConnection = setConnection

function dsql.setQuery(qryRec)
	loadLibs()
	local conn = dconn.currentConnection()
	if not conn then
		return --- util.printError("query connection does not exist")
	end
	local connQuery = conn.query
	if not connQuery then
		return --- util.printError("query connection does not exist")
	else
		if qryRec.operator == "" then
			dsql.clearQuery() -- start a new clean query
			local newQueryTable
			if qryRec.field == 1 then
				newQueryTable = connQuery.mainTable
			else
				newQueryTable = qryRec.table or qryRec.record_type and qryRec.record_type[1] and qryRec.record_type[1].table or dschema.tableName(qryRec.field) -- we need local table here, otherwise params: (qryRec.field, conn.schema, qryRec.record_type)
			end
			if connQuery == nil or connQuery.table ~= newQueryTable or connQuery.recordType ~= qryRec.record_type then
				-- lib/db/dqjson.lua has called dsql.setConnection(), so call only when we know that this must be a new query
				setConnection({table = newQueryTable, record_type = qryRec.record_type}, qryRecNameFunc(qryRec)) -- qryRec is table, will be converted to name
				connQuery = dconn.query() -- dsql.setConnection may have changed connection so we MUST set connQuery again
				if not connQuery then
					return --- util.printError("query connection does not exist")
				end
			else
				connQuery.queryArr = {}
				if connQuery.queryNameFunc == nil then
					connQuery.queryNameFunc = qryRecNameFunc(qryRec)
				end
			end
		end
		connQuery.queryArr[#connQuery.queryArr + 1] = qryRec
	end
end

local function createJoin(connSql, connQuery, schema, queryTbl)
	-- create sql joins
	-- todo: use dschema.externalName(queryTbl, schema, recType)
	local queryTblSql = dschema.tableName(queryTbl, schema, connQuery.recordType)
	local usedTableArr = connSql.usedTableArr
	if #usedTableArr < 1 then
		return
	end
	local joinDoneTbl = {}
	local function addJoin(tablePrefixSql, tblNameSql, where, reverse)
		if not joinDoneTbl[tblNameSql] then
			joinDoneTbl[tblNameSql] = true
			-- local tablePrefixSql = dschema.tablePrefix(tblNameSql)-- parseBefore(fieldNameSql, ".")
			connSql.joinArr[#connSql.joinArr + 1] = {tablePrefixSql = tablePrefixSql, tblNameSql = tblNameSql, where = where, reverse = reverse}
		end
	end

	local function addLinkRecJoin(linkRec, linkedTblSql)
		local prefixSql, where
		if linkRec.reverse then
			prefixSql = parseBefore(linkRec.linked_field, ".")
			where = linkRec.linked_field .. " = " .. linkRec.linking_field
		else
			prefixSql = parseBefore(linkRec.linking_field, ".")
			where = linkRec.linking_field .. " = " .. linkRec.linked_field
		end
		local where1 = where
		if linkRec.extra_link_field then
			for _, rec2 in ipairs(linkRec.extra_link_field) do
				local where2 = rec2.linking .. " = " .. rec2.linked
				local where3 = rec2.linked .. " = " .. rec2.linking
				if where2 ~= where1 and where3 ~= where1 then -- prevent duplicates
					where = where .. " AND " .. where2
				end
			end
		end
		addJoin(prefixSql, linkRec.linking_table_sql or linkedTblSql, where, linkRec.reverse)
		linkRec.reverse = nil
	end

	if not loc.tablePair then
		loc.tablePair = dprf.prf("table/prf/table_pair.json", "no-error")
	end

	local function findLinkRec(linkingTbl, linkedTbl, findTableLink)
		local linkRec, key1
		local pairRec = loc.tablePair[schema]
		-- recordTypeSeparator, schemaSeparator, linkSeparator
		if pairRec == nil or pairRec.pair == nil then
			util.printError("table/prf/table_pair.json does not contain schema '%s' key 'pair'", tostring(schema))
			return
		end
		-- example of key: "work_order_schedule-schedule > work"
		local warn, linkedTblSql
		local linkingTblSql = linkingTbl.sql_table
		local linkingRecordType = linkingTbl.record_type or connQuery.recordType
		local linkField = linkingTbl.link_field
		if linkField then
			local prefix = parseBefore(linkField, ".")
			if prefix ~= linkField and prefix ~= "json_data" then
				linkField = linkField:sub(#prefix + 2) -- remove prefix
				local linkingTblPrefix = dschema.tablePrefix(linkingTbl.local_table)
				if prefix ~= linkingTblPrefix then
					util.printRed("query link field '%s' prefix '%s' is not same as linking local table prefix '%s'", linkField, prefix, linkingTblPrefix)
				end
			end
			linkField = "." .. linkField -- like: work_material_estimated-estimated > product.product_id instead of work_material_estimated-estimated > product
		else
			linkField = ""
		end
		local linkedRecordType
		if linkedTbl == nil then
			-- use main table as linked table
			linkedRecordType = connQuery.recordType
			linkedTblSql = queryTblSql
			if connQuery.linkedTable then
				local id = dschema.recTypeName(linkingTbl.local_table, linkingRecordType)
				if connQuery.linkedTable[id] then
					linkedTblSql = connQuery.linkedTable[id]
					linkedTblSql = dschema.tableName(linkedTblSql, schema, linkedRecordType)
				end
			end
		else
			linkedTblSql = linkedTbl.sql_table
			-- linkedRecordType = linkedTbl.record_type
			if linkingTbl.record_type == nil then
				local linking = tostring(linkingTbl.local_table)
				linkingRecordType = connQuery.tableRecordType[linking]
				linkingRecordType = linkingRecordType and linkingRecordType[1]
				if linkingRecordType == nil then
					warn = l("query does not contain linking record type for schema '%s', linking table '%s', linked table '%s'", schema, tostring(linkingTblSql), tostring(linkedTblSql))
				else
					warn = l("query join find did not have record type defined, used record type 1 '%s' for schema '%s', linking table '%s', linked table '%s'", tostring(linkingRecordType), schema, tostring(linkingTblSql), tostring(linkedTblSql))
				end
			end
		end
		--[[ if linkingTbl.record_type == nil then
			util.printWarning(
					"query join find did not have linking table record type defined, used main table record type for schema '%s', linking table '%s', linked table '%s'",
					schema, tostring(linkingTblSql), tostring(linkedTblSql))
		end ]]
		key1 = dschema.recTypeName(linkingTblSql, linkingRecordType) .. linkSeparator .. linkedTblSql .. linkField -- dschema.recTypeName(linkedTblSql, linkedRecordType)
		linkRec = pairRec.pair[key1]
		if linkRec == nil and findTableLink then
			key1 = dschema.recTypeName(linkingTblSql, linkedRecordType) .. linkSeparator .. linkedTblSql
			linkRec = pairRec.pair[key1]
		end
		--[[ 	if linkRec == nil then
			key1 = dschema.recTypeName(linkingTblSql, linkedRecordType) .. linkSeparator .. linkedTblSql
			linkRec = pairRec.pair[key1]
		end
		if linkRec == nil then
			key1 = linkingTblSql .. linkSeparator .. dschema.recTypeName(linkedTblSql, linkedRecordType)
			linkRec = pairRec.pair[key1]
		end ]]
		if linkRec == nil then
			-- do reverse search
			local linked -- , linkedRecordType
			if linkedTbl then
				linked = linkedTbl.local_table
				linkedRecordType = linkedTbl.record_type
				--[[ elseif linkingTbl.local_table == linkedTblSql then
				linked = linkedTblSql ]] -- test this if it is needed
			else
				linked = dschema.toLocal(linkedTblSql, schema)
				-- linkedRecordType = linkingTbl.record_type
			end
			if linkedRecordType == nil then
				local linkedRecordTypeArr = connQuery.tableRecordType[linked]
				linkedRecordType = linkedRecordTypeArr and linkedRecordTypeArr[1]
				if linkedRecordType and #linkedRecordTypeArr > 1 then
					warn = l("query join find did not have record type defined, used record type 1 for schema '%s', linking table '%s', linked table '%s'", schema, tostring(linkingTblSql), tostring(linkedTblSql))
				end
			end
			if linkedRecordType == nil then
				warn = l("query does not contain record type for schema '%s', linking table '%s', linked table '%s'", schema, tostring(linkingTblSql), tostring(linkedTblSql))
			end
			local key = dschema.recTypeName(linkedTblSql, linkedRecordType) .. linkSeparator .. linkingTblSql .. linkField
			linkRec = pairRec.pair[key]
			if linkRec == nil and linkedRecordType ~= "" then -- todo: check if this causes problems
				key = dschema.recTypeName(linkedTblSql, "") .. linkSeparator .. linkingTblSql
				linkRec = pairRec.pair[key]
			end
			--[[ if linkRec == nil then
					util.printWarning("table/prf/table_pair.json does not contain schema '%s' pair key '%s'", schema, key1)
				end ]]
			if linkRec then
				linkRec.reverse = true
			end
		end
		if linkRec then
			linkRec.linking_table_sql = linkingTblSql
			if warn then
				util.printWarning(warn)
			end
		end
		return linkRec
	end

	local function findLinkingChildTable()
		local linkRec, tblRec
		local linkOrder = 0
		-- loop from 2 because first table is main table by definition
		for i = 2, #usedTableArr do
			-- find first direct connections to main table, they must be first joins
			tblRec = usedTableArr[i]
			linkRec = findLinkRec(tblRec)
			if linkRec then
				linkOrder = linkOrder + 1
				connSql.usedTable[tblRec.sql_table] = linkOrder -- mark as link done with order number
				addLinkRecJoin(linkRec, queryTblSql) -- tblRec.sql_table)
			end
		end

		for i = 2, #usedTableArr do

			local function loopFind(tblRec1)
				for j = 2, #usedTableArr do
					if i ~= j then
						linkRec = findLinkRec(tblRec1, usedTableArr[j])
						if linkRec then
							linkOrder = linkOrder + 1
							connSql.usedTable[usedTableArr[i].sql_table] = linkOrder
							-- connSql.whereUsedTable[usedTableArr[i].sql_table] = linkOrder -- force inner join
							break
						end
					end
				end
			end

			-- find next other ones that do not have direct connections to main table, find connections to other tables
			tblRec = usedTableArr[i]
			if connSql.usedTable[tblRec.sql_table] == true then -- marked as link done with order number, test only for true
				linkRec = findLinkRec(tblRec)
				if linkRec == nil then
					loopFind(tblRec)
				end
				if linkRec == nil and schema == "" then
					linkRec = findLinkRec(tblRec, nil, true) -- find plain table link without record type
					if linkRec == nil then
						loopFind(tblRec)
					end
				end
				if linkRec then
					addLinkRecJoin(linkRec, tblRec.sql_table)
				else -- old find
					local linkedTblSql = tblRec.sql_table
					util.printWarning("query link find did not work using table pairs, schema '%s', table '%s', linked table '%s'", schema, queryTblSql, linkedTblSql)
					findLinkRec(tblRec) -- for debug
					tblRec = usedTableArr[i]
					findLinkRec(tblRec) -- for debug
					--[=[
						local err
						linkRec, err = dsqlOld.findLinkRecOld(linkedTblSql, queryTblSql, loc)
						if type(linkRec) == "table" then
							dsqlOld.addLinkRecJoinOld(linkRec.default.link_array, queryTblSql, addJoin)
						else
							-- break
							local extraJoin = table.concat(connQuery.extraJoinArr, "\n")
							if not pegFound(extraJoin, linkedTblSql) then
								if queryTblSql == linkedTblSql then
									return
								end
								--[[ if connQuery.linkTable and #connQuery.linkTable == 1 and connQuery.linkTable[1] == connQuery.mainTable then
									return -- wrong to no to error but it's better to return sql error, it tess more what is wrong
								end ]]
								if connQuery.linkTable then
									local fld = fn.iter(connSql.selectArr):filter(function(tblRec)
										local tbl = dschema.tableName(tblRec, schema)
										return tbl == linkedTblSql
									end):totable()
									err = l("schema '%s', relations do not exist between tables '%s' and '%s' using link table '%s', link fields: '%s', error: '%s'", schema, queryTblSql, linkedTblSql, connQuery.linkTable, table.concat(fld, ", ") or "", err or "")
								else
									err = l("schema '%s', relations do not exist between tables '%s' and '%s', error: '%s'", schema, queryTblSql, linkedTblSql, err or "")
								end
								util.printError(err .. "\n" .. tostring(linkRec)) -- linkRec is error string
								err = nil -- it's better to return sql error, it tells more what is wrong
							end
						end
					--]=]
				end
			end
		end
	end
	findLinkingChildTable()
end

function dsql.sqlQueryBuild(fldNameArr, func)
	if not recordTypeSeparator then
		recordTypeSeparator = dschema.recordTypeSeparator
		linkSeparator = dschema.linkSeparator
	end
	local conn = dconn.currentConnection() -- dschema call allowed only here, will add table_redirect -rec to conn.connection
	if not conn then
		return l("connection does not exist")
	elseif conn.error then
		return conn.error
	end
	local err
	local connSql = conn.sql
	local connQuery = conn.query
	if connQuery == nil then
		return
	end
	local schema = connQuery.schema
	local queryTbl = connQuery.table
	local recordType = connQuery.recordType
	clearQuerySql(conn, false)
	if not connQuery.queryNameFunc then
		util.printError("conn.query.queryNameFunc is nil")
		connQuery.queryNameFunc = function()
			return dsql.lastQueryName -- dqjson.lastQueryJsonName
		end -- function
	end
	if connQuery.mainTable == nil then
		return util.printError("query main table is nil")
	end
	sqlQueryFromAdd(connQuery, connSql) -- only main table may be in sql FROM

	local function debug(queryArr)
		if loc.debug == nil then
			loc.debug = util.prf("system/debug.json", "no-error").debug.dsql
		end
		if loc.debug then
			local value, operator
			local maxArrCount = 10
			local query = fn.iter(queryArr):map(function(rec)
				local function toStr(val)
					if type(val) == "table" then
						value = fn.iter(val):take(maxArrCount):reduce(function(acc, rec2)
							if acc == "" then
								return toStr(rec2)
							end
							return acc .. ", " .. toStr(rec2)
						end, "")
						value = "[" .. value .. "]" -- json.toJsonRaw(rec.value)
					elseif type(val) == "string" then
						value = "'" .. val .. "'"
					else
						value = tostring(val)
					end
					return value
				end
				value = toStr(rec.value)
				if type(rec.value) == "table" and #rec.value > maxArrCount then
					value = value:sub(1, -2) .. ",... (more than " .. maxArrCount .. " values)]"
				end
				if rec.operator == "" then
					operator = ""
				elseif rec.operator == "or" then
					operator = "         or  "
				else
					operator = "         " .. rec.operator .. " "
				end
				return operator .. tostring(rec.field) .. " " .. tostring(rec.comparison) .. " " .. tostring(value)
			end):totable()
			local qry = "\n  table: " .. queryTbl .. "\n  field: " .. table.concat(fldNameArr, ", ") .. "\n  query: " .. table.concat(query, "\n")
			util.print("\ndsql.sqlQueryBuild (%s/%s), name: %s%s", conn.database, recordType, connQuery.queryNameFunc(), qry)
		end
	end

	if #connQuery.queryArr > 0 then
		debug(connQuery.queryArr)
		err = sqlQuerySqlCreate(conn)
		if func ~= "COUNT(*)" then -- sql function COUNT(*) will not clean query, all others do
			connQuery.queryArr = {} -- why empty here?
		end
		if err then
			util.printError(err)
			return err
		end
	end
	if #connQuery.saveArr > 0 then -- how about update, same as save? - insert + update
		debug(connQuery.saveArr)
		err = sqlSaveSqlCreate(conn)
		connQuery.saveArr = {}
		if err then
			util.printError(err)
			return err
		end
	end
	if #connQuery.deleteArr > 0 then
		debug(connQuery.deleteArr)
		err = sqlDeleteSqlCreate(conn)
		connQuery.deleteArr = {}
		if err then
			util.printError(err)
			return err
		end
	end

	-- create select array
	local prevUsedTable
	local useExternalField = connQuery.option and connQuery.option.use_external_field
	for _, localFld in ipairs(fldNameArr) do
		local fld
		if func == "COUNT(*)" then
			fld = dschema.tableName(localFld, schema, recordType)
		elseif useExternalField then
			fld = localFld
		else
			fld = dschema.fieldNamePrefix(localFld, schema, recordType)
		end
		if fld == nil then
			if schema == nil or schema == "" or dschema.isLocalTable(localFld, schema, recordType) then
				fld = localFld
			else
				err = l("could not convert field '%s' to schema '%s', main table '%s', record type '%s'", tostring(localFld), tostring(schema), tostring(connQuery.table), tostring(recordType))
				util.printRed("sql generation error: '%s'", tostring(err))
				dschema.fieldNamePrefix(localFld, schema, recordType) -- for debug
				return err
			end
		end
		-- fldNameArrSql[i] = fld
		if fld and not connSql.selectArrUsed[fld] then
			local idx = #connSql.selectArr + 1
			connSql.selectArrUsed[fld] = true
			if fld == queryTbl then
				connSql.selectArr[idx] = fld
			else
				connSql.selectArr[idx] = fieldNameToSqlFunction(fld, connQuery.option, connQuery.schema)
			end
			if not useExternalField then
				local sqlTbl = dschema.tableName(fld, schema, connQuery.recordType)
				local locaTbl = dschema.localTable(localFld, schema, connQuery.recordType)
				if sqlTbl and not connSql.selectArrUsedTable[sqlTbl] then
					connSql.selectArrUsedTable[sqlTbl] = true
				end
				if locaTbl == nil then
					err = l("local table of field '%s' is nil", tostring(localFld))
					return err
				end
				if sqlTbl == nil then
					err = l("sql table of field '%s' is nil", tostring(localFld))
					return err
				end
				if locaTbl .. "/" .. sqlTbl ~= prevUsedTable then
					prevUsedTable = locaTbl .. "/" .. sqlTbl
					addUsedTable(locaTbl, sqlTbl, connSql) -- do not guess record type
				end
			end
		end
	end

	createJoin(connSql, connQuery, schema, queryTbl)

	dsql.sqlQueryTextCreate(func)
	if func == "COUNT(*)" then -- sql function COUNT(*) will not clean query, all others do
		dsql.clearQuerySet(conn.query)
	else
		clearQuerySetLimitOffset(conn)
	end
	-- connQuery.table = nil -- is this needed?
	return err
end

function dsql.sqlQueryExecute(option)
	-- local loc.printSqlVar = true
	option = option or {}
	local conn = dconn.currentConnection()
	local connSql = conn.sql
	local connQuery = conn.query
	loc.lastQueryName = option.query_name or lastQueryName(connQuery)
	if loc.debug then
		util.printOk("\ndsql.sqlQueryExecute (%s/%s): %s, %s", conn.database, connQuery.recordType, connQuery.queryText, loc.lastQueryName)
	else
		local len2, txt2
		if not option.no_debug and debugExecute() then
			local info = conn.database
			if conn.query.mainTable then
				info = info .. ": " .. dschema.recTypeName(conn.query.mainTable, conn.query.recordType)
			end
			if #connQuery.queryText > loc.sqlSelectDebugLength and startsWith(connQuery.queryText, "SELECT ") then
				local txt
				if #connQuery.queryText <= loc.sqlSelectDebugLength + loc.sqlSelectDebugLength2 then
					txt = connQuery.queryText
				else
					txt = connQuery.queryText:sub(1, loc.sqlSelectDebugLength)
					local pos = pegFind(txt, "FROM ")
					len2 = loc.sqlSelectDebugLength2
					if pos > 0 then
						txt = connQuery.queryText:sub(1, pos - 1)
					else
						txt = parseBeforeLast(txt, ",") .. ", ... "
						pos = pegFind(connQuery.queryText, "FROM ")
					end
					if #txt < loc.sqlSelectDebugLength then
						len2 = len2 + (loc.sqlSelectDebugLength - #txt)
					end
					txt2 = connQuery.queryText:sub(pos, pos + len2 - 1)
					if #connQuery.queryText - pos <= len2 then
						txt = txt .. txt2
					else
						txt = txt .. txt2 .. " ..."
					end
				end
				txt = peg.replace(txt, "\n", "")
				util.printInfo("  - sql execute '%s/%s', query: '%s'\n    %s", conn.connection or conn.info or "", info, tostring(loc.lastQueryName), txt)
			else
				util.printInfo("  - sql execute '%s/%s', query: '%s'\n    %s%s", conn.connection or conn.info or "", info, tostring(loc.lastQueryName), pegReplace(connQuery.queryText:sub(1, loc.sqlDebugLength), "\n", ""), #connQuery.queryText > loc.sqlDebugLength and "..." or "")
			end
		end
	end
	--[[
	if pegFind(connQuery.queryText, "'\0'") > 0 then
		local pos = pegFind(connQuery.queryText, "'\0'")
		util.print("\n*** sql error, data contains char(0) in position "..pos..", data: "..connQuery.queryText.."\n\n")
		connQuery.queryText = pegReplace(connQuery.queryText, "'\0'", "''")
	end
	]]
	connQuery.query_count = connQuery.query_count + 1
	-- print("queryText: "..connQuery.queryText)
	if loc.printSqlVar and not util.from4d() then
		local showSqlChars = 2500 -- todo: get from preference
		if #connQuery.queryText > showSqlChars and not peg.startsWith(connQuery.queryText, "SELECT ") then
			util.printInfo("sqlQueryExecute '%s', db-name: %s, '%s', sql:\n %s;", loc.lastQueryName, conn.name .. "/" .. conn.database, connQuery.queryNameFunc(), connQuery.queryText:sub(1, showSqlChars) .. " ...")
		else
			util.printInfo("sqlQueryExecute '%s', db-name: %s, '%s', sql:\n %s;", loc.lastQueryName, conn.name .. "/" .. conn.database, connQuery.queryNameFunc(), connQuery.queryText)
		end
	end
	local time = util.milliSeconds()
	local cursor, err
	if connSql.containsEmptyArray and not connSql.containsOr then
		cursor = "containsEmptyArray"
	else
		-- fix: dsql.sqlQueryExecute() needs main table/field
		local driver = dconn.driver()
		--[[
		if util.from4d() then
			util.print("dsql.sqlQueryExecute driver.execute() connQuery.queryText: '%s'", tostring(connQuery.queryText))
			util.printTable(option, "dsql.sqlQueryExecute driver.execute() option")
		end -- ]]
		option.organization_id = conn.organization_id
		if conn.driver == "rest4d" then
			if rest4d == nil then
				rest4d = require "db/database-rest4d" -- we need this to enable breakpoints in database-rest4d
			end
			driver = rest4d
		end
		if driver.execute == nil then
			err = util.printWarning("\n*** sql error: driver '%s' has no function execute()", conn.info or conn.name or "")
			return nil, err
		end
		cursor, err = driver.execute(connQuery.queryText, option, conn)
	end
	-- if not count then
	-- print(connQuery.query_count..". #"..#connQuery.queryText..", "..connQuery.queryText)
	-- end --]]
	if err then
		if pegFind(connQuery.queryText, "SELECT 1 FROM ") ~= 1 then -- tableExists() does not need to print error
			if not option.no_error then
				util.printWarning("\n*** sql error: '%s', connection: '%s', sql:\n %s\n\n", tostring(err), conn.info or "conn.info is missing", parseBeforeWithDivider(connQuery.queryText, "PASSWORD"))
			end
		end
		return nil, err
	end
	if not cursor and not err then
		local driver = dconn.driver()
		cursor, err = driver.execute(connQuery.queryText, option, conn) -- or option as nil, is this ok?
	end
	if loc.printSqlVar and not util.from4d() then
		if loc.printSqlPrevTime == nil then
			loc.printSqlPrevTime = time
		end
		util.print(" sqlQueryExecute, delay from previous: %d ms, %s", time - loc.printSqlPrevTime, dt.currentStringMillisecond())
		loc.printSqlPrevTime = time
	end
	-- dsql.clearQuery() -- same query may have save and load and update
	return cursor, err
end

local numberFormulaPattern = peg.set("+-*/")
local function fieldValueSql(fld, fldValue_, schema, recType) -- old dschema.fieldValueChangedTo(conn, fld, fldValue)
	dsql.loadLibs()
	local dbType = dconn.dbType()
	local fldValue = fieldSqlMapValue(fld, schema, recType, fldValue_)
	local valueType = type(fldValue)
	local database4d = dbType == "4d" -- dconn.database4d()
	if database4d and fldValue == "0000-00-00" then
		return "'1970-01-01'" -- "NULL" -- in 4d sql we must use NULL, not '0000-00-00'
	elseif database4d and fldValue == "1970-01-01" then
		return "NULL"
	elseif not database4d and (fldValue == "0000-00-00" or fldValue == "0000-00-00 00:00:00.000") then
		return "'1970-01-01'"
	elseif valueType == "boolean" then
		return tostring(fldValue)
	elseif valueType == "table" then
		fldValue = json.toJsonRaw(fldValue)
		valueType = "string"
	elseif valueType == "nil" then
		util.printInfo("field '%s' save value is '%s'", fld, tostring(fldValue))
		-- use fieldDefaultValue() instead of NULL?
		return "NULL" -- works in postgre, other databases?
	elseif database4d and fld == dschema.uuidField(fld, schema, recType) then
		if fldValue ~= "" then
			return "'" .. fldValue .. "'"
		end
		if not uuid4d then
			uuid4d = require "uuid-4d"
		end
		return "'" .. uuid4d.uuid(fld) .. "'" -- set record_id field value if connected to 4d
	end

	local fldType = dschema.fieldType(fld, schema, recType)
	if valueType == "number" and fldType == "time" then
		return "'" .. dt.secondsToTimeString(fldValue) .. "'"
	elseif (dbType == "postgre" or dbType == "oracle") and (fldType == "date" or startsWith(fldType, "timestamp")) then -- odbc == oracle
		-- elseif database4d and fldType == "date" then
		-- return "TO_DATE('"..fldValue.."','YYYY-MM-DD')"
		if startsWith(fldType, "timestamp") or valueType == "number" then -- fldType == "ts" then
			-- http://www.techonthenet.com/oracle/functions/to_timestamp.php
			-- http://www.postgresql.org/docs/9.3/static/functions-formatting.html
			if valueType == "number" then
				if dbType == "postgre" then
					-- return "TO_TIMESTAMP('"..dt.secondsToTimestampStringMicrosecond(fldValue).."','YYYY-MM-DD HH24:MI:SS.US')"
					return "TO_TIMESTAMP('" .. dt.secondsToTimestampString(fldValue) .. "','YYYY-MM-DD HH24:MI:SS')"
				else
					return "TO_TIMESTAMP(" .. fldValue .. ")"
				end
			elseif fldValue:sub(11, 11) == "T" then
				return "TO_TIMESTAMP('" .. fldValue .. "','YYYY-MM-DDTHH24:MI:SS')"
			elseif pegFound(fldValue, "(") then -- not a function like NOW()
				valueType = "function" -- prevent quote
			else
				return "TO_TIMESTAMP('" .. fldValue .. "','YYYY-MM-DD HH24:MI:SS')"
			end
		elseif fldType == "date" and dbType == "oracle" then
			-- oracle date contains also seconds
			-- http://docs.oracle.com/cd/B28359_01/server.111/b28318/datatype.htm#i1847
			return "TO_DATE('" .. fldValue .. "','YYYY-MM-DD HH24:MI:SS')"
		elseif fldType == "date" then
			return "TO_DATE('" .. fldValue .. "','YYYY-MM-DD')"
		end
	elseif dbType == "sql server" and (fldType == "date" or startsWith(fldType, "timestamp")) then
		return "CONVERT(DATETIME, '" .. fldValue .. "')"
	elseif fldType == "date" then
		if valueType == "number" then
			fldValue = dt.toDateString(fldValue) -- toDateString gives only date part
		else
			fldValue = fldValue:sub(1, 10) -- remove time part
		end
	elseif valueType == "string" and (fldType == "integer" or fldType == "double") then
		if startsWith(fldValue, "(SELECT ") then
			-- valueType = "string"
			return fldValue
		else
			if tonumber(fldValue) == nil then
				if peg.found(fldValue, numberFormulaPattern) then
					return fldValue
				end
				util.printError("field '%s' save value '%s' must be a number, zero-value will be used", dschema.fieldName(fld, schema, recType), fldValue)
				return 0
			end
			return tonumber(fldValue)
		end
	elseif valueType ~= "string" and (fldType == "varchar" or fldType == "text") then
		fldValue = tostring(fldValue)
		valueType = "string"
	end
	if fldType == "float" or fldType == "double" then
		fldValue = tonumber(fldValue)
		valueType = "number"
	elseif fldType == "integer" or fldType == "bigint" then
		fldValue = tonumber(fldValue) or 0
		fldValue = math.floor(fldValue) -- or math.ceil for negatives?
		valueType = "number"
	end
	if valueType == "string" then
		fldValue = dsql.sqlValue(fldValue, fld)
		if database4d then
			if fldType == "text" then -- dschema.fieldType(fld) == "text" then
				fldValue = pegReplace(fldValue, "\n", "\r")
			elseif fldType == "varchar" then
				fldValue = pegReplace(fldValue, "\n", " ")
				fldValue = pegReplace(fldValue, "\t", " ")
			end
			-- elseif (fldType == "text" or fldType == "varchar") and pegFound(fldValue, "\\") then -- not json, jsonb
			-- 	fldValue = pegReplace(fldValue, "\\", "\\\\")
		end
		if fldValue == "''" then
			if dschema.isNullField(fld) then
				return "NULL"
			end
		end
	end
	return fldValue
end
dsql.fieldValueSql = fieldValueSql

local fieldSeparator = ", " -- or ","
local function fieldArrSqlConcat(fieldArr, quote)
	if quote then
		local quotedArr = {}
		for i, fldNameSql in ipairs(fieldArr) do
			quotedArr[i] = quoteSql(fldNameSql)
		end
		return concat(quotedArr, fieldSeparator)
	end
	return concat(fieldArr, fieldSeparator)
end

function dsql.sqlQueryTextCreate(func)
	local conn = dconn.currentConnection()
	local connSql = conn.sql
	local connQuery = conn.query
	local schema, recordType = connQuery.schema, connQuery.recordType
	local quote = dconn.quoteSql()
	local sqlExecute = {}
	local notAggregatedField
	-- local isSelect
	local i = 1
	if #connSql.insertArr > 0 then
		sqlExecute[i] = "INSERT INTO " .. connSql.from
		i = i + 1
		sqlExecute[i] = "(" .. fieldArrSqlConcat(connSql.selectArr, quote) .. ")" -- column names
		i = i + 1
		for j = #connSql.insertArr, 1, -1 do
			if type(connSql.insertArr[j]) == "userdata" then
				util.printError("connSql.insertArr contains userdata: '%s'", tostring(connSql.insertArr[j]))
				table.remove(connSql.insertArr, j)
			end
		end
		-- should not send null values, clean them and warn instead of crash
		sqlExecute[i] = "VALUES (" .. concat(connSql.insertArr, fieldSeparator) .. ")"
		i = i + 1
		if #connSql.updateArr > 0 then -- prf.save.on_conflict
			if peg.found(connSql.updateArr[1]:lower(), "on conflict") then
				sqlExecute[i] = connSql.updateArr[1]
				i = i + 1
				table.remove(connSql.updateArr, 1)
			end
			sqlExecute[i] = concat(connSql.updateArr, fieldSeparator) -- "ON CONFLICT DO "
			i = i + 1
		end
	elseif #connSql.updateArr > 0 then
		sqlExecute[i] = "UPDATE " .. connSql.from
		i = i + 1
		sqlExecute[i] = "SET " .. concat(connSql.updateArr, fieldSeparator)
		i = i + 1
	else
		local selectArr = connSql.selectArr
		if quote then
			selectArr = {}
			for j, fldNameSql in ipairs(connSql.selectArr) do
				selectArr[j] = quoteSql(fldNameSql)
			end
		end
		if connSql.aggregate then
			local aggregateField = {}
			notAggregatedField = {}
			for key, val in pairs(connSql.aggregate) do
				for _, fldNameSql in ipairs(val) do
					local fldSqlName = dschema.fieldNamePrefix(fldNameSql, schema, recordType)
					if quote then
						fldSqlName = quoteSql(fldNameSql)
					end
					aggregateField[fldSqlName] = key:upper()
				end
			end
			for j, fldNameSql in ipairs(selectArr) do
				if aggregateField[fldNameSql] then
					selectArr[j] = aggregateField[fldNameSql] .. "(" .. selectArr[j] .. ")"
				else
					notAggregatedField[#notAggregatedField + 1] = fldNameSql
				end
			end
		end
		if func then
			func = func:upper()
			if func == "DISTINCT" then
				sqlExecute[i] = "SELECT " .. func .. " " .. concat(selectArr, fieldSeparator)
			elseif func == "COUNT(*)" then
				sqlExecute[i] = "SELECT COUNT(*)"
			elseif func == "COUNT(DISTINCT)" then
				sqlExecute[i] = "SELECT COUNT(DISTINCT " .. selectArr[1] .. ")"
			elseif type(func) == "string" then
				sqlExecute[i] = "SELECT " .. func .. "(" .. concat(selectArr, fieldSeparator) .. ")"
			else
				util.printWarning("unknown sql function '%s'", tostring(func))
				sqlExecute[i] = "SELECT " .. concat(selectArr, fieldSeparator)
			end
		else
			sqlExecute[i] = "SELECT " .. concat(selectArr, fieldSeparator)
		end
		i = i + 1
		sqlExecute[i] = "FROM " .. connSql.from -- fieldArrSqlConcat(connSql.fromArr, quote)
		i = i + 1
		if #connSql.joinArr > 0 then
			local queryTable = connSql.usedTableArr[1].sql_table
			for _, rec in ipairs(connSql.joinArr) do
				if connSql.selectArrUsedTable[rec.tblNameSql] and rec.tblNameSql ~= queryTable and not connSql.whereUsedTable[rec.tblNameSql] then -- if rec.reverse then
					if quoteSql(rec.tblNameSql) == rec.tablePrefixSql then
						sqlExecute[i] = "LEFT OUTER JOIN " .. rec.tablePrefixSql .. " ON " .. rec.where
					else
						sqlExecute[i] = "LEFT OUTER JOIN " .. quoteSql(rec.tblNameSql) .. loc.sqlAlias .. rec.tablePrefixSql .. " ON " .. rec.where
					end
				else
					if quoteSql(rec.tblNameSql) == rec.tablePrefixSql then
						sqlExecute[i] = "INNER JOIN " .. rec.tablePrefixSql .. " ON " .. rec.where
					else
						sqlExecute[i] = "INNER JOIN " .. quoteSql(rec.tblNameSql) .. loc.sqlAlias .. rec.tablePrefixSql .. " ON " .. rec.where
					end
				end
				i = i + 1
			end
		end
		if #connQuery.extraJoinArr > 0 then
			for _, rec in ipairs(connQuery.extraJoinArr) do
				sqlExecute[i] = rec
				i = i + 1
			end
		end
	end
	if #connSql.whereArr > 0 then
		-- if #connSql.whereArr ~= 1 or connSql.whereArr[1] ~= "1 = 1" then -- missing WHERE is suspicious
		sqlExecute[i] = "WHERE " .. concat(connSql.whereArr)
		i = i + 1
	end
	if #connSql.returnArr > 0 then
		if #connSql.returnArr > 0 then
			sqlExecute[i] = concat(connSql.returnArr)
			i = i + 1
		end
	end
	if notAggregatedField and #notAggregatedField > 0 then
		sqlExecute[i] = "GROUP BY " .. fieldArrSqlConcat(notAggregatedField, quote)
		i = i + 1
	end
	if #connSql.orderArr > 0 then
		sqlExecute[i] = "ORDER BY " .. concat(connSql.orderArr, fieldSeparator)
		if notAggregatedField and #notAggregatedField > 0 and loc.orderBeforeGroupSql[conn.dbtype] then
			sqlExecute[i - 1], sqlExecute[i] = sqlExecute[i], sqlExecute[i - 1] -- 4d needs ORDER BY before GROUP BY, swap statements
		end
		i = i + 1
	end
	if connSql.limit > 0 then
		sqlExecute[i] = "LIMIT " .. connSql.limit
		i = i + 1
	end
	if connSql.offset > 0 then
		sqlExecute[i] = "OFFSET " .. connSql.offset
		-- i = i + 1 -- last does not need this
	end
	connQuery.queryText = concat(sqlExecute, "\n ")
end

local function sqlQueryWhereAdd(field, where, schema, recType)
	dsql.loadLibs()
	local connSql = dconn.sql()
	local pos = #connSql.whereArr + 1
	if pos > 1 then
		local startsWithSet = where:sub(1, 1)
		local prevEnd = connSql.whereArr[pos - 1]:sub(-1)
		if startsWithSet ~= ")" and prevEnd ~= " " and prevEnd ~= "(" then
			where = " " .. where
		end
	end
	if field ~= 1 then
		local tbl = dschema.tableName(field, schema, recType)
		if tbl == nil then
			util.printError("could not convert field '%s', schema '%s', record type '%s' to table", tostring(field), tostring(schema), tostring(recType))
		elseif connSql.whereUsedTable[tbl] == nil then
			connSql.whereUsedTable[tbl] = true
			connSql.whereUsedTableArr[#connSql.whereUsedTableArr + 1] = tbl
		end
	end
	connSql.whereArr[pos] = where
end
dsql.sqlQueryWhereAdd = sqlQueryWhereAdd

function dsql.sqlAsc(asc)
	if asc == ">" then
		return "" -- " ASC"
	elseif asc == "<" then
		return " DESC"
	end
	return nil
end

local pattJsonArrow = peg.toPattern("->>")
fieldNameToSqlFunction = function(fldNameSql, option, schema)
	local fldNameSqlJson = dsql.changeJsonFieldName(fldNameSql) --   ordr.json_data->TO_CHAR('first_conf_delivery_date','YYYY-MM-DD'),
	local dbType = dconn.dbType()
	local fldType
	if dbType == "postgre" or dbType == "oracle" then -- odbc == oracle
		-- todo: move all database formats to json file, remove all "postgre" and "oracle" and "4d"
		fldType = dschema.fieldType(fldNameSql, schema)
		if fieldNameToSqlFunction == nil then
			util.printError("field '%s' type was not found, schema '%s'", tostring(fldNameSql), tostring(schema))
			fldType = "varchar"
		end
		if fldType and startsWith(fldType, "timestamp") then
			return "TO_CHAR(" .. fldNameSqlJson .. ",'YYYY-MM-DD HH24:MI:SS.US')"
		elseif fldType == "date" and dbType == "oracle" then
			-- oracle stores dates WITH seconds
			-- http://docs.oracle.com/cd/B28359_01/server.111/b28318/datatype.htm#i1847
			-- http://www.techonthenet.com/oracle/functions/to_char.php
			return "TO_CHAR(" .. fldNameSqlJson .. ",'YYYY-MM-DD HH24:MI:SS')"
		elseif fldType == "date" then -- postgre
			if not pegFound(fldNameSqlJson, pattJsonArrow) then -- not like: ord.json_data->>'first_delivery_date'
				return "TO_CHAR(" .. fldNameSqlJson .. ",'YYYY-MM-DD')"
			end
		end
	end
	if option and loc.substringLength > 0 and pegFound(option, "substring") then
		fldType = fldType or dschema.fieldType(fldNameSql)
		if fldType == "text" then
			return "SUBSTRING(" .. fldNameSqlJson .. ",1," .. loc.substringLength .. ") " .. pegReplace(fldNameSqlJson, loc.pegPattDot, "_") -- alias name at the end
		elseif fldType == "json" or fldType == "jsonb" then
			return "SUBSTRING(CAST(" .. fldNameSqlJson .. " AS TEXT),1," .. loc.substringLength .. ") " .. pegReplace(fldNameSqlJson, loc.pegPattDot, "_") -- alias name at the end
		end
	end
	return fldNameSqlJson
end

sqlSaveSqlCreate = function() -- fldName, func, conn)
end

sqlDeleteSqlCreate = function() -- fldName, func, conn)
end

local function queryValueChange(value)
	local continue = false
	if not constantPrefixArr then
		constantPrefixArr = util.prf("constant/local/constant_prefix.json", "no-cache").prefix or false
	end
	if constantPrefixArr then -- prevent recursive error in util.prf("constant/local/constant_prefix.json"
		for _, prefix in ipairs(constantPrefixArr) do
			if startsWith(value, prefix) then
				continue = true
				break
			end
		end
		if continue then
			local prfName = "constant/mg_constant" .. (dconn.dbType() and "_" .. dconn.dbType() or "") .. ".json"
			local constant = util.prf(prfName)
			local ret = execute.getTagValue(value, constant) or value
			return ret ~= nil and ret or value
		end
	end
	return value
end

sqlQuerySqlCreate = function(conn)
	local connSql = conn.sql
	local connQuery = conn.query
	local schema = connQuery.schema
	local err = nil
	-- local dbType = dconn.dbType()
	for i, rec in ipairs(connQuery.queryArr) do
		local fld = rec.field -- {operator = operator, field = fld, comparison = comparison, value = value}
		if fld == nil then
			util.printError(l("fld is nil, query '%s'", connQuery.queryNameFunc()))
			return
		elseif fld == "" then
			util.printError(l("fld is empty, query '%s'", connQuery.queryNameFunc()))
			return
		end
		local err2
		local fldNameSql, extTable
		local localTbl, recTypeArr
		local useExternalField = connQuery.option and connQuery.option.use_external_field
		if fld == 1 then -- WHERE 1 = 1
			localTbl = connQuery.mainTable
			recTypeArr = {connQuery.recordType}
		elseif useExternalField then
			localTbl = connQuery.mainTable
			recTypeArr = {connQuery.recordType}
		else
			localTbl = dschema.tableName(fld)
			recTypeArr = connQuery.tableRecordType[localTbl]
		end
		if recTypeArr == nil then
			-- util.printWarning("record type is not valid for table '%s', query '%s'", tostring(localTbl), connQuery.queryNameFunc()) todo: check if we need this warning
			recTypeArr = {""}
		end
		local recTypeCount = #recTypeArr
		local operator = rec.operator
		for recTypeLoop, recType in ipairs(recTypeArr) do
			-- if localTbl == connQuery.table and recType ~= connQuery.recordType then
			-- recType = connQuery.recordType
			-- end
			local value = rec.value
			local comparison = rec.comparison
			local function queryAsText()
				return tostring(operator) .. " " .. tostring(fld) .. " " .. tostring(comparison) .. " " .. tostring(value) .. "\n - query: " .. connQuery.queryNameFunc()
			end
			local function recTypeOperator(oper)
				if recTypeCount > 1 and recTypeLoop == 1 then
					if oper == "" then
						return "("
					else
						return " ("
					end
				elseif oper ~= "" then
					return " "
				end
				return ""
			end
			if type(value) == "string" then
				value = queryValueChange(value)
			end
			if fld == 1 then -- WHERE 1 = 1
				fldNameSql = 1
			else
				if type(rec.record_type) == "string" then
					value = fieldSqlMapValue(fld, schema, rec.record_type, value)
					extTable = dschema.tableName(localTbl, schema, rec.record_type)
					addUsedTable(localTbl, extTable, connSql, rec.record_type)
					fldNameSql = dschema.externalNameSql(fld, schema, rec.record_type)
				else
					value = fieldSqlMapValue(fld, schema, recType, value)
					extTable = dschema.tableName(localTbl, schema, recType)
					addUsedTable(localTbl, extTable, connSql, recType)
					fldNameSql = dschema.externalNameSql(fld, schema, recType)
				end
				if fldNameSql == nil then
					fldNameSql = dschema.fieldNamePrefix(fld, schema, recType)
					if fldNameSql == nil then
						fldNameSql = dschema.fieldNamePrefix(fld)
					end
				end
				if not fldNameSql then
					util.addError(err, l("query field is not valid: ") .. queryAsText())
					return err
				end
				local pos = pegFind(fldNameSql, loc.pegPattDot)
				local fldNameNoPrefix = fldNameSql:sub(pos + 1)
				if pegFound(fldNameNoPrefix, loc.pegPattDot) then -- json_data.xxx
					fldNameSql = dsql.changeJsonFieldName(fldNameSql)
					if type(value) ~= "string" then
						local fldTypeSql = dschema.fieldTypeSql(fld)
						-- if fldTypeSql ~= "varchar" and fldTypeSql ~= "text" then
						fldNameSql = "CAST(" .. fldNameSql .. " AS " .. fldTypeSql .. ")"
						-- end
					end
				end
			end
			fldNameSql = quoteSql(fldNameSql)
			comparison, err2 = dsql.sqlComparison(comparison, value)
			if err2 then
				util.addError(err, err2 .. l("\nquery comparison is not valid: ") .. queryAsText())
				return err
			end
			if comparison == "LIKE" or comparison == "NOT LIKE" or comparison == "STARTS" then
				if not dschema.isStringField(fld, schema) then
					fldNameSql = "CAST(" .. fldNameSql .. " AS TEXT)"
				end
			end
			if operator ~= "" and #connSql.whereArr == 0 then
				-- elseif operator == "" and #connSql.whereArr > 0 then
				-- util.addError(err, l"continue query operator must not be empty, operator: "..tostring(operator))
				util.addError(err, l("first query operator must be empty, operator: ") .. queryAsText())
			elseif operator ~= "" then
				operator = dsql.sqlOperator(operator)
				if not operator then
					util.addError(err, l("query operator is not valid: ") .. queryAsText())
				elseif operator == "OR" then
					connSql.containsOr = true
				end
			end
			if err then
				util.clearError()
				util.printError(err)
				return err
			end
			local useNativeValue = false
			local arrayQuery = false -- not an array query
			local field2, fieldType
			if fld == 1 then
				fieldType = "number"
			else
				fieldType = dschema.fieldType(fld, schema, recType)

				local function changeValueType()
					if type(value) == "string" and (fieldType == "double" or fieldType == "integer" or fieldType == "bigint") then
						value = tonumber(value) or 0 -- TODO: cleaner conversion, clean letters like '1e5' (for exponent) first?, fix also convert.lua formatFieldData()
						if fieldType == "integer" or fieldType == "bigint" then
							value = math.floor(value) -- cut decimals, do not round
						end
						useNativeValue = true
					elseif type(value) == "string" and fieldType == "boolean" then
						if value:lower() == "true" then
							value = "true"
						elseif tonumber(value) ~= nil and tonumber(value) > 0 then -- only positive numbers are true
							value = "true"
						else
							value = "false"
						end
						useNativeValue = true
					elseif fieldType == "boolean" then
						if type(value) == "number" then
							if comparison == ">" then
								if value >= 0 then
									value = "true"
								else
									value = "false"
								end
							elseif comparison == ">=" then
								if value >= 1 then
									value = "true"
								else
									value = "false"
								end
							elseif comparison == "<" then
								if value <= 1 then
									value = "false"
								else
									value = "true"
								end
							elseif comparison == "<=" then
								if value <= 0 then
									value = "false"
								else
									value = "true"
								end
							end
						end
						if comparison ~= "#" then
							comparison = "="
						end
						if value ~= "false" and value ~= "true" then
							if value == false then
								value = "false"
							else
								value = "true"
							end
						end
						useNativeValue = true
					elseif type(value) == "boolean" then
						-- use: dschema.fieldType(fldNum, schema, recType) == "boolean" ?
						-- or use 'WHERE field' for true and 'WHERE not field' for false
						value = tostring(value)
						useNativeValue = true
					elseif fieldType == "date" then
						if type(value) == "number" then
							value = fieldValueSql(fld, dt.toDateString(value), schema, recType)
						else
							value = fieldValueSql(fld, value, schema, recType)
						end
						if value == "NULL" then
							if comparison == "=" then
								comparison = "IS"
							elseif comparison == "<>" then
								comparison = "IS NOT"
							else
								util.printError(l("when searching for null date comparison must be '=' or '<>'"))
							end
						end
						useNativeValue = true
					elseif startsWith(fieldType, "timestamp") then
						if type(value) == "number" then
							value = fieldValueSql(fld, dt.toString(value), schema, recType)
						else
							value = fieldValueSql(fld, value, schema, recType)
						end
						useNativeValue = true
					elseif fieldType == "jsonb" or fieldType == "json" then
						fldNameSql = "CAST(" .. fldNameSql .. " AS TEXT)" -- dschema.fieldType(fldNum) == "json" then
					elseif pegFound(fldNameSql, "json_data") then -- json_data.send_type
						fldNameSql = fldNameSql .. "" -- debug, was: "CAST("..fldNameSql.." AS TEXT)"
					elseif value ~= "NULL" and dschema.fieldTypeLua(fld, schema) ~= type(value) then -- TODO: add recType to call
						if not (type(value) == "table" and (comparison == "IN" or comparison == "NOT IN")) then
							util.addError(err, l("query field type is not same as value type, field '%s', field type '%s', value type '%s', query '%s'", fldNameSql, dschema.fieldTypeLua(fld, schema), type(value), queryAsText())) -- TODO: add recType to call
						end
					end
				end

				if value == "NULL" or fld == 1 then
					useNativeValue = true
				elseif value == "null" then
					value = "NULL"
					useNativeValue = true
				elseif type(value) == "string" and dschema.isField(value) then
					field2 = value
				elseif not (comparison == "LIKE" or comparison == "NOT LIKE" or comparison == "STARTS") then
					changeValueType()
					-- elseif type(value) == "string" and dbType ~= "4d" and pegFound(value, "\\") then
					-- 	value = pegReplace(value, "\\", "\\\\")
				end

				if field2 ~= nil then -- allow queries where fields are compared
					-- queryJson / sql needs this for joins
					value = dschema.externalNameSql(field2, schema, recType)
					useNativeValue = true
				else
					if type(value) == "table" and not (value.y and value.m and value.d) then -- not a date table
						arrayQuery = true
					elseif type(value) == "table" and value.y and value.m and value.d then
						if fieldType ~= "date" then
							util.addError(err, l("query field is not date, value is date table"))
						end
					elseif type(value) == "string" and (comparison == "IN" or comparison == "NOT IN") then -- subquery
						arrayQuery = true
					end
					if arrayQuery then
						local recTypeOp = recTypeOperator(operator)
						if connQuery.setStart[i] and recTypeLoop == 1 then
							sqlQueryWhereAdd(1, operator .. recTypeOp .. connQuery.setStart[i])
							operator = ""
							recTypeOp = ""
							connQuery.setStart[i] = nil -- mark used
						end
						dqry.queryArray(operator .. recTypeOp, fld, comparison, value, schema, recType)
						if connQuery.setEnd[i] and recTypeLoop == recTypeCount then -- add last missing ')' when last query is inside parentheses and last quey is IN -query
							sqlQueryWhereAdd(1, connQuery.setEnd[i])
							connQuery.setEnd[i] = nil -- mark used
						end
					end
				end
			end
			if err then
				return err
			end
			-- run
			if arrayQuery == false then --  value ~= "%"  like "%" -> no search at all
				local useLower = false
				if fieldType == "varchar" or fieldType == "text" then -- or fieldType == "string"
					useLower = dconn.useLowerSlq(conn)
				end
				local where
				local escape = ""
				if useNativeValue == false then
					value = dsql.sqlValue(value, fld)
				end
				if comparison == "STARTS" then
					comparison = "LIKE"
					value = value:sub(1, -2) .. "%'"
				end
				if comparison == "LIKE" or comparison == "NOT LIKE" then
					if pegFind(value, "_") > 0 then
						value = pegReplace(value, "_", "\\_")
						escape = " ESCAPE '\\'"
					end
				end
				if useLower and noLowerField[parseAfter(fld, ".")] then
					useLower = false
				end
				if useNativeValue == false and useLower then
					if type(value) == "string" and value ~= "'%'" and value ~= "''" then
						fldNameSql = "lower(" .. fldNameSql .. ")"
						if pegFound(value, "%") then
							value = pegLower(value)
						else
							value = "lower(" .. value .. ")"
						end
					end
				end
				if value == nil then
					util.addError(err, l("query value is nil, using value ''"))
					value = "''"
				end
				local recTypeOp = recTypeOperator(operator)
				if connQuery.setStart[i] or connQuery.setEnd[i] then
					if operator == "" then -- i == 1
						where = recTypeOp .. (connQuery.setStart[i] or "") .. fldNameSql .. " " .. comparison .. " " .. value .. escape .. (recTypeLoop == recTypeCount and connQuery.setEnd[i] or "")
					else
						where = operator .. recTypeOp .. (connQuery.setStart[i] or "") .. fldNameSql .. " " .. comparison .. " " .. value .. escape .. (recTypeLoop == recTypeCount and connQuery.setEnd[i] or "")
					end
					if recTypeLoop == 1 then
						connQuery.setStart[i] = nil -- mark used
					end
					if recTypeLoop == recTypeCount then
						connQuery.setEnd[i] = nil -- mark used
					end
				elseif operator == "" then
					where = recTypeOp .. fldNameSql .. " " .. comparison .. " " .. value .. escape
				else
					where = operator .. recTypeOp .. fldNameSql .. " " .. comparison .. " " .. value .. escape
				end
				sqlQueryWhereAdd(fld, where, schema, recType)
			end

			if recTypeCount > 1 then
				if recTypeLoop == recTypeCount then
					connSql.whereArr[#connSql.whereArr] = connSql.whereArr[#connSql.whereArr] .. ")"
				else
					connSql.whereArr[#connSql.whereArr] = connSql.whereArr[#connSql.whereArr] .. " OR "
					operator = ""
				end
			end
		end
	end
	local i = #connQuery.setEnd
	if connQuery.setEnd[i] then -- add last missing ')' when last query is inside parentheses and last quey is IN -query
		sqlQueryWhereAdd(1, connQuery.setEnd[i])
	end
end
dsql.sqlQuerySqlCreate = sqlQuerySqlCreate

function dsql.showRemoteSql(val)
	local driver = dconn.driver()
	if driver and driver.showSql then
		driver.showSql(val)
	end
end

function dsql.aggregate(aggregateTbl)
	local connSql = dconn.sql()
	if connSql then
		connSql.aggregate = aggregateTbl
	end
end

function dsql.offset(num)
	local connSql = dconn.sql()
	if connSql then
		connSql.offset = num
	end
end

function dsql.limit(num)
	local connSql = dconn.sql()
	if connSql then
		connSql.limit = num
	end
end

function dsql.sqlText()
	local connQuery = dconn.query()
	return connQuery and connQuery.queryText
end

function dsql.querySetStart()
	local connQuery = dconn.query()
	if connQuery then
		local i = #connQuery.queryArr + 1
		connQuery.setStart[i] = (connQuery.setStart[i] or "") .. "("
	end
end

function dsql.querySetEnd()
	local connQuery = dconn.query()
	if connQuery then
		local i = #connQuery.queryArr
		if connQuery.setEnd[i] and connQuery.setEnd[i]:sub(-1) == " " then
			connQuery.setEnd[i] = connQuery.setEnd[i]:sub(1, -2) .. ")"
		else
			connQuery.setEnd[i] = (connQuery.setEnd[i] or "") .. ")" -- next will come AND/OR/NOT
		end
	end
end

function dsql.arrayToSqlArr(arr)
	if type(arr[1]) == "number" then
		return "(" .. table.concat(arr, ",") .. ")"
	end
	return "('" .. table.concat(arr, "','") .. "')"
end
--- old

function dsql.dateYmdToSqlString(y, m, d)
	local dateStr = {}
	if y >= 1000 then
		dateStr[1] = y
	elseif y >= 0 and y < 9 then
		dateStr[1] = "000" .. y
	elseif y < 100 then
		dateStr[1] = "00" .. y
	elseif y < 1000 then
		dateStr[1] = "0" .. y
	end
	dateStr[2] = "-"
	if m > 9 then
		dateStr[3] = m
	else
		dateStr[3] = "0" .. m
	end
	dateStr[4] = "-"
	if d > 9 then
		dateStr[5] = d
	else
		dateStr[5] = "0" .. d
	end
	return table.concat(dateStr)
end

function dsql.sqlValue(fldValue, field) -- , starts)
	if type(fldValue) == "table" and fldValue.y then
		return "'" .. dsql.dateYmdToSqlString(fldValue.y, fldValue.m, fldValue.d) .. "'"
	elseif type(fldValue) == "string" then
		--[=[ if starts then
			-- fldValue = fldValue..[[\'%_]]
			fldValue = fldValue:gsub([[\]], [[\\]])
			fldValue = fldValue:gsub([[']], [[\']])
			fldValue = fldValue:gsub([[%%]], [[\%%]])
			fldValue = fldValue:gsub([[_]], [[\_]])
			-- print("'"..fldValue.."%'")
			return "'"..fldValue.."%'"
		else ]=]
		local pos = pegFind(fldValue, "\000")
		if pos > 0 then
			local fldValueFixed = pegReplace(fldValue, "\000", "")
			-- fldValueFixed = pegReplace(fldValueFixed, "'", "")
			fldValueFixed = utf.fixUTF8(fldValueFixed, "") -- second param is replacement char
			util.print("sql value contained char(0) in position %d, value '%s', length %d, fixed value '%s', fixed length %d, field '%s'", pos, fldValue, #fldValue, fldValueFixed, #fldValueFixed, tostring(field))
			fldValue = fldValueFixed
		end
		if pegFound(fldValue, "'") then
			fldValue = pegReplace(fldValue, "'", escapeSingleQuote) -- escape ' with double '' or \'
		end
		if pegFound(fldValue, "\n") then
			if dschema.fieldType(field) ~= "text" and not dschema.isJsonField(field) then
				util.printRed("field '%s' type is not text, but it contains new line character, field value: '%s'%s", field, fldValue:sub(1, 200), #fldValue > 200 and "..." or "")
				fldValue = pegReplace(fldValue, "\n", "\\n")
			end
		end
		-- end
		if not pegFound(fldValue, "CURRENT_DATE") then
			fldValue = "'" .. fldValue .. "'"
		end
		return fldValue
	end
	return fldValue
end

function dsql.sqlValueArray(valueArray)
	if #valueArray < 1 then
		return ""
	end
	if type(valueArray[1]) == "table" and valueArray[1].y then
		local ret = ""
		for _, rec in ipairs(valueArray) do
			ret = ret .. "'" .. dsql.dateYmdToSqlString(rec.y, rec.m, rec.d) .. "'"
		end
		return ret
	elseif type(valueArray[1]) == "string" or type(valueArray[1]) == "boolean" then
		local valueArrayDistinct, notDistinct = util.arrDistinctValues(valueArray)
		for i, val in ipairs(valueArrayDistinct) do
			if pegFound(val, "'") then
				valueArrayDistinct[i] = pegReplace(val, "'", escapeSingleQuote) -- escape ' with double '' or \'
			end
			if pegFound(val, "\n") then
				valueArrayDistinct[i] = pegReplace(val, "\n", "\\n")
			end
		end
		local ret = "'" .. table.concat(valueArrayDistinct, "','") .. "'"
		if #notDistinct > 0 then
			util.printWarning(l("query '%s' array does not contain distinct values, more than once values: %s", dsql.lastQueryName(), table.concat(notDistinct, ", "):sub(1, 400)))
		end
		return ret
	end
	local valueArrayDistinct = util.arrDistinctValues(valueArray)
	return table.concat(valueArrayDistinct, ",")
end

-- do
local validComparison = util.invertTable({"=", "<>", ">", ">=", "<", "<=", "LIKE", "NOT LIKE", "IN", "NOT IN", "STARTS", "IS", "IS NOT", "BETWEEN"})
function dsql.sqlComparison(comparison, value) -- global local function
	comparison = comparison:upper()
	if (comparison == "IN" or comparison == "NOT IN") and not (type(value) == "table" or type(value) == "string") then -- and type(value) == "table" and not(value.y and value.m and value.d) then -- not a date table
		local err = l("query comparison is ") .. comparison .. l(" and value is not a string or not a table")
		return nil, err
	elseif (comparison == "LIKE" or comparison == "NOT LIKE" or comparison == "STARTS") and type(value) ~= "string" then
		local err = l("query comparison is ") .. comparison .. l(" and value is not a string")
		return nil, err
	elseif (comparison == "IS" or comparison == "IS NOT") and value:upper() ~= "NULL" then
		local err = l("query comparison is ") .. comparison .. l(" and value is not 'NULL'")
		return nil, err
	elseif comparison == "=" and type(value) == "string" and value:upper() == "NULL" then
		comparison = "IS"
	elseif comparison == "<>" and type(value) == "string" and value:upper() == "NULL" then
		comparison = "IS NOT"
	elseif comparison == "BETWEEN" and (type(value) ~= "table" or #value ~= 2) then
		local err = l("query comparison is ") .. comparison .. l(" and value is not a two element table")
		return nil, err
	elseif not validComparison[comparison] then
		local err = l("query comparison is not valid: ") .. comparison
		return nil, err
	end
	return comparison
end

local validOperator = {"AND", "OR", "NOT", "AND NOT", "OR NOT"} -- "" is not valid here
-- , "AND (", "AND ", "OR ", "OR (" are non-harmful special cases for multi record_type queries
validOperator = util.invertTable(validOperator)
function dsql.sqlOperator(operator) -- global local function
	operator = operator:upper()
	if not validOperator[operator] then
		if operator:sub(-1) == " " and validOperator[operator:sub(1, -2)] then
			return operator
		end
		if operator:sub(-2) == " (" and validOperator[operator:sub(1, -3)] then
			return operator
		end
		util.printError("query operator '%s' is not valid", tostring(operator))
		return "" -- prevent concat errors, probably sql will fail, that's ok
	end
	return operator
end

function dsql.sqlQueryWherePart(sql)
	local _, endPos = sql:find(" WHERE ", 1, true)
	if endPos then
		local query = sql:sub(endPos + 1)
		query = query:gsub("\n", "") -- more clean for 4d debug
		return query
	end
	-- util.printError(l"Query did not contain WHERE part: "..sql)
	return ""
end

local function sqlExecuteUnsafe(queryText, fieldNameArray, option, returnTableType)
	loadLibs()
	local data = {}
	local driver = dconn.driver()
	local conn = dconn.currentConnection()
	-- local connSql = conn.sql
	local connQuery = conn.query
	if not connQuery then
		return data, {error = l("sql connection could not be established")}
	end
	option = option or {}
	local queryName = option.query_name or ""
	if queryName == "" and loc.lastQueryName then
		queryName = tostring(loc.lastQueryName)
	end
	queryText = removeFromStart(pegReplace(queryText, "\t", ""), " ")
	local queryStart = parseBefore(queryText, " "):upper()
	if util.from4d() then
		if queryStart == "DROP" then
			return nil, l("sqlExecuteUnsafeArray DROP is not supported, query '%s', '%s'", queryName, queryText)
		elseif queryStart == "ALTER" then
			return nil, l("sqlExecuteUnsafeArray ALTER is not supported, query '%s', '%s'", queryName, queryText)
		elseif queryStart == "CREATE" then
			return nil, l("sqlExecuteUnsafeArray CREATE is not supported, query '%s', '%s'", queryName, queryText)
		elseif queryStart == "TRUNCATE" then
			return nil, l("sqlExecuteUnsafeArray TRUNCATE is not supported, query '%s', '%s'", queryName, queryText)
		end
	end
	if not fieldNameArray and queryStart == "SELECT" then
		if not queryText:find("AS SELECT ", 1, true) then -- CREATE TABLE x AS SELECT * FROM y...
			-- close cursor?
			return nil, l("sqlExecuteUnsafeArray parameter fieldNameArray is missing for SELECT statement, query '%s'", queryName)
		end
	end
	-- UPDATE and INSERT INTO is ok
	connQuery.queryText = queryText
	--[[ dsql.sqlQueryExecute() does this debug
		if debugExecute() then
		local info = conn.database
		if conn.query.mainTable then
			info = info..": "..dschema.recTypeName(conn.query.mainTable, conn.query.recordType)
		end
		util.printInfo("  - sql execute unsafe '%s/%s', query: '%s'\n    %s%s", conn.connection, info, queryName, pegReplace(connQuery.queryText:sub(1, loc.sqlDebugLength), "\n", ""), #connQuery.queryText > loc.sqlDebugLength and "..." or "")
	end ]]
	local time = util.seconds()
	--[[ util.printTable(connQuery, "sqlExecuteUnsafeArray: connQuery")
		local conn = dconn.currentConnection()
		util.printTable(conn, "sqlExecuteUnsafeArray: conn")
	]]
	option.query_name = option.query_name or queryName
	local cursor, err = dsql.sqlQueryExecute(option)
	-- if err then
	-- close cursor?
	-- 	return err
	-- end
	if queryText:find("SELECT ", 1, true) ~= 1 and queryText:find("PRAGMA ", 1, true) ~= 1 or cursor == 0 or err then -- PRAGMA table_info(company); fpr sqlite
		time = util.seconds(time)
		local info = {query_time = time}
		if err then
			info.error = err
		elseif (pegFound(queryText, "SELECT ") or pegFound(queryText, "(SELECT ")) and pegFind(queryText, "SELECT ") > 2 and not pegFound(queryText, "SELECT audit.audit_table_") then
			-- elseif cursor == 0 then -- cursor is 0 witn INSERT INTO, UPDATE and so on...
			-- info.error = "query cursor is 0"
			info.error = "SELECT query does not start with SELECT"
		else
			info.info = "ok"
		end
		-- close cursor?
		return data, info
	end

	if type(cursor) == "table" and loc.printSqlVar then
		cursor.show_sql = true -- show sql in 4d
		-- local showSqlPrev = driver.showSql(true) -- or use this?
	end
	local ret, info
	-- util.print("sqlExecuteUnsafeArray, returnTableType: '%s'", tostring(json.toJsonRaw(returnTableType, "no-error")))
	if returnTableType == "record array" then
		ret, info = driver.selectionToRecordArray(cursor, fieldNameArray, option)
	else
		ret, info = driver.selectionToArrayTable(cursor, fieldNameArray, option)
	end
	if not ret then
		-- dconn.setPrevConnection()
		return nil, info
	end
	--[[
	local rowsFetched = info.row_count
	local rowsTotal = info.row_count_total
	local sql = connQuery.queryText
	while false and rowsFetched and rowsFetched > 0 and rowsFetched < rowsTotal and rowsFetched < loc.queryMaxRowsLimit do
		connQuery.queryText = sql .. "\n LIMIT " .. loc.queryMaxRowsLimit .. "\n OFFSET " .. rowsFetched .. " "
		cursor = dsql.sqlQueryExecute(option)
		local ret2, info2
		if returnTableType == "record array" then
			ret2, info2 = driver.selectionToRecordArray(cursor, fieldNameArray, option)
		else
			ret2, info2 = driver.selectionToArrayTable(cursor, fieldNameArray, option)
		end
		if not ret2 then
			-- dconn.setPrevConnection()
			return nil, info2
		end
		rowsFetched = rowsFetched + info2.row_count
		if rowsFetched >= rowsTotal then
			util.printError("rowsFetched %d >= rowsTotal %d", rowsFetched, rowsTotal)
		end
		-- combine arrays to first set
		for tag, arr2 in pairs(ret2) do
			local arr1 = ret[tag]
			local j = 0
			local from = #arr1 + 1
			local to = #arr1 + #arr2
			for i = from, to do
				j = j + 1
				arr1[i] = arr2[j]
			end
		end
	end
	-- dconn.setPrevConnection()
	if rowsFetched and rowsFetched >= loc.queryMaxRowsLimit then
		util.printError(l("Query returned >= max rows: ") .. rowsFetched .. " / " .. loc.queryMaxRowsLimit)
	end
	info.row_count = rowsFetched
	]]
	time = util.seconds(time)
	info.query_time = time
	return ret, info, fieldNameArray
end

function dsql.sqlExecuteUnsafeArray(queryText, fieldNameArray, fieldTypeArray, option)
	option = option or {}
	option.field_type = fieldTypeArray
	return sqlExecuteUnsafe(queryText, fieldNameArray, option, "array table")
end

function dsql.sqlExecuteUnsafeRecordArray(queryText, fieldNameArray, fieldTypeArray, option)
	option = option or {}
	option.field_type = fieldTypeArray
	return sqlExecuteUnsafe(queryText, fieldNameArray, option, "record array")
end

function dsql.sqlExecuteUnsafe(queryText, fieldNameArray, fieldTypeArray, option)
	option = option or {}
	option.field_type = fieldTypeArray
	if type(option) == "string" then
		util.printError("sqlExecuteUnsafe: option is string, use table instead")
		option = {query_name = option}
	else
		option = option or {}
		option.query_name = option.query_name or "dsql.sqlExecuteUnsafe"
	end
	return sqlExecuteUnsafe(queryText, fieldNameArray, option, option and option.return_type or "record array")
end

--[[
function dsql.pushSql()
	if not dconn then
		dconn = require "dconn"
	end
	local connSql = dconn.sql()
	if connSql == nil then
		return false
	end
	if loc.sqlPushArray == nil then
		loc.sqlPushArray = {}
	end
	loc.sqlPushArray[#loc.sqlPushArray + 1] = util.clone(connSql)
	return true
end

function dsql.popSql()
	if type(loc.sqlPushArray) ~= "table" or #loc.sqlPushArray <= 0 then
		return
	end
	local sql = table.remove(loc.sqlPushArray) -- Remove an element from a table. If a position is specified the element at that the position is removed. The remaining elements are reindexed sequentially and the size of the table is updated to reflect the change. The element removed is returned by this function.
	local conn = dconn.currentConnection()
	conn.sql = sql
end
--]]

function dsql.createTableByCopy(tableName, newTableName, where)
	if not where then
		where = "WHERE 1=0"
	end
	local ret, info = sqlExecuteUnsafe("CREATE TABLE " .. newTableName .. " AS SELECT * FROM " .. tableName .. " " .. where)
	if info.error then
		util.print(tostring(info.error))
		return nil
	end
	return ret
end

function dsql.changeJsonFieldName(fld)
	-- see: http://www.postgresqltutorial.com/postgresql-json/
	local tbl = splitToArray(fld, loc.pegPattDot)
	if #tbl > 2 then
		local newField = tbl[1] .. "." .. tbl[2]
		-- we need tbl.json_data->'rec'->'subRec'->'subSubRec'->>'subSubRecField'
		for i = 3, #tbl do
			if i < #tbl then -- i == 3
				newField = newField .. "->'" .. tbl[i] .. "'" -- -> operator returns a JSON object
			else
				newField = newField .. "->>'" .. tbl[i] .. "'" -- ->> operator returns JSON object field
			end
		end
		return newField
	end
	return fld
end

function dsql.changeJsonFieldArrName(fieldArr)
	local newArr = fn.iter(fieldArr):map(function(fld)
		return dsql.changeJsonFieldName(fld)
	end):totable()
	return newArr
end

function dsql.test.setRecordType(recType)
	local connQuery = dconn.query()
	connQuery.recordType = recType or ""
end

return dsql
