--- lib/db/duckdb/duckdb.lua
-- see: /Users/pasi/ma-git/other/c/database/duckdb/duckdb-duck.C
local ffi = require "mffi"
local C = ffi.C
local ffiString = ffi.string
local isNotNull = ffi.isNotNull
local util = require "util"
local def = require "duckdb/duckdb-ffi"
local lib = def.lib
local func = def.func

-- local duckArray = {}
-- local duckIdx = {}
local duck = {}
duck.__index = duck

local function printNotConnected(duck)
	if duck.not_connected_error then
		duck.not_connected_error = true
		util.printError("not connected to duck database")
	end
	return nil
end

local function printDuckError(self, errFmt, errText)
	local ret
	if self.result and isNotNull(self.result) then
		-- local count = duck:result_chunk_count(result[0]) -- this is of if there is no error, but crashes if there is an error
		local res = lib.duckdb_result_error(self.result) -- const char duckdb_result_error(duckdb_result *result)
		if isNotNull(res) then
			ret = util.printRed(errFmt .. ", DuckDB error:\n%s", errText, ffi.string(res))
		else
			ret = util.printRed(errFmt .. ", DuckDB error text was NULL")
		end
	end
	-- duck.closeConnection(self)
	return ret
end

function duck.newConnection(configTable)
	local self = {}
	local err
	if configTable and next(configTable) then
		self.config = ffi.newNoAnchor("duckdb_config[1]")
		err = lib.duckdb_create_config(self.config) -- duckdb_state duckdb_create_config(duckdb_config *out_config)
		if err == C.DuckDBError then
			self.config = nil
			util.printError("failed to create DuckDB config")
			return nil
		end
		for name, option in pairs(configTable) do
			err = lib.duckdb_set_config(self.config, name, option) -- duckdb_state duckdb_set_config(duckdb_config config, const char *name, const char *option)
			if err == C.DuckDBError then
				util.printError("failed to set DuckDB config '%s': '%s'", tostring(name), tostring(option))
				duck.closeConnection(self)
				return nil
			end
		end
	end
	self.database = ffi.newNoAnchor("duckdb_database[1]")
	self.connection = ffi.newNoAnchor("duckdb_connection[1]")
	self.result = ffi.newNoAnchor("duckdb_result[1]")
	err = lib.duckdb_open(nil, self.database) -- duckdb_state duckdb_open(const char *path, duckdb_database *out_database)
	if err == C.DuckDBError then
		util.printError("failed to open DuckDB database")
		duck.closeConnection(self)
		return nil
	end
	self.db = self.database[0]
	err = lib.duckdb_connect(self.db, self.connection) -- duckdb_state duckdb_connect(duckdb_database database, duckdb_connection *out_connection)
	if err == C.DuckDBError then
		util.printError("failed to set DuckDB config '%s': '%s'", tostring(name), tostring(option))
		duck.closeConnection(self)
		return nil
	end
	self.con = self.connection[0]
	--[[ local tbl = setmetatable(self, duck)
	duckArray[#duckArray + 1] = tbl
	duckIdx[tbl] = tbl ]]
	-- self.state = lib.duckdb_open_ext(path, out_database, config, out_error) -- duckdb_state duckdb_open_ext(const char *path, duckdb_database *out_database, duckdb_config config, char **out_error)
	setmetatable(self, duck)
	return self
end

function duck.closeConnection(self)
	if self.database ~= nil then
		lib.duckdb_close(self.database) -- void duckdb_close(duckdb_database *database)
		self.db = nil
		self.database = nil
	end
	if self.connection ~= nil then
		lib.duckdb_disconnect(self.connection) -- void duckdb_disconnect(duckdb_connection *connection)
		self.con = nil
		self.connection = nil
	end
	if self.config ~= nil then
		lib.duckdb_destroy_config(self.config) -- void duckdb_destroy_config(duckdb_config *config)
		self.config = nil
	end
	if self.result ~= nil then
		lib.duckdb_destroy_result(self.result) -- void duckdb_destroy_result(duckdb_result *result)
		self.result = nil
	end
end

function duck.library_version()
	return ffiString(lib.duckdb_library_version()) -- const char duckdb_library_version()
end

function duck:config_count()
	return lib.duckdb_config_count() -- size_t duckdb_config_count()
end

function duck:get_config_flag(index, out_name, out_description)
	return lib.duckdb_get_config_flag(index, out_name, out_description) -- duckdb_state duckdb_get_config_flag(size_t index, const char **out_name, const char **out_description)
end

-- not done:

function duck:set_config(config, name, option)
end

function duck:query(query)
	if self.connection == nil then
		return printNotConnected(self)
	end
	-- duckdb_state duckdb_query(duckdb_connection connection, const char *query, duckdb_result *out_result)
	if lib.duckdb_query(self.con, query, self.result) == C.DuckDBError then
		return printDuckError(self, "failed to query DuckDB database, query: '%s'", query)
	end
end

function duck:column_name(col)
	if self.connection == nil then
		return printNotConnected(self)
	end
	return ffiString(lib.duckdb_column_name(self.result, col - 1)) -- const char duckdb_column_name(duckdb_result *result, idx_t col)
end

function duck:column_type(col)
	if self.connection == nil then
		return printNotConnected(self)
	end
	return lib.duckdb_column_type(self.result, col - 1) -- duckdb_type duckdb_column_type(duckdb_result *result, idx_t col)
end

function duck:column_logical_type(col)
	if self.connection == nil then
		return printNotConnected(self)
	end
	return lib.duckdb_column_logical_type(self.result, col - 1) -- duckdb_logical_type duckdb_column_logical_type(duckdb_result *result, idx_t col)
end

function duck:column_count()
	if self.connection == nil then
		return printNotConnected(self)
	end
	return tonumber(lib.duckdb_column_count(self.result)) -- idx_t duckdb_column_count(duckdb_result *result)
end

function duck:row_count()
	if self.connection == nil then
		return printNotConnected(self)
	end
	return tonumber(lib.duckdb_row_count(self.result)) -- idx_t duckdb_row_count(duckdb_result *result)
end

function duck:rows_changed()
	return tonumber(lib.duckdb_rows_changed(self.result)) -- idx_t duckdb_rows_changed(duckdb_result *result)
end

function duck:column_data(result, col)
	return lib.duckdb_column_data(result, col - 1) -- void duckdb_column_data(duckdb_result *result, idx_t col)
end

function duck:nullmask_data(result, col)
	return lib.duckdb_nullmask_data(result, col - 1) -- _Bool duckdb_nullmask_data(duckdb_result *result, idx_t col)
end

-- values

function duck:value_varchar(col, row)
	if self.connection == nil then
		return printNotConnected(self)
	end
	if self.result and isNotNull(self.result) then
		local duckStrPtr = lib.duckdb_value_varchar(self.result, col - 1, row - 1) -- char duckdb_value_varchar(duckdb_result *result, idx_t col, idx_t row)
		if isNotNull(duckStrPtr) then
			local str = ffiString(duckStrPtr)
			lib.duckdb_free(duckStrPtr)
			return str
		end
		return "NULL"
	end
	return "-DuckDB value_varchar(), result is not valid pointer-"
end

function duck:value_boolean(col, row)
	if self.connection == nil then
		return printNotConnected(self)
	end
	if self.result and isNotNull(self.result) then
		return lib.duckdb_value_boolean(self.result, col - 1, row - 1) -- _Bool duckdb_value_boolean(duckdb_result *result, idx_t col, idx_t row)
	end
end

function duck:value_int8(col, row)
	if self.connection == nil then
		return printNotConnected(self)
	end
	if self.result and isNotNull(self.result) then
		return lib.duckdb_value_int8(self.result, col - 1, row - 1) -- int8_t duckdb_value_int8(duckdb_result *result, idx_t col, idx_t row)
	end
end

function duck:value_int16(col, row)
	return lib.duckdb_value_int16(self.result, col - 1, row - 1) -- int16_t duckdb_value_int16(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_int32(col, row)
	return lib.duckdb_value_int32(self.result, col - 1, row - 1) -- int32_t duckdb_value_int32(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_int64(col, row)
	return lib.duckdb_value_int64(self.result, col - 1, row - 1) -- int64_t duckdb_value_int64(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_hugeint(col, row)
	return lib.duckdb_value_hugeint(self.result, col - 1, row - 1) -- duckdb_hugeint duckdb_value_hugeint(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_decimal(col, row)
	return lib.duckdb_value_decimal(self.result, col - 1, row - 1) -- duckdb_decimal duckdb_value_decimal(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_uint8(col, row)
	return lib.duckdb_value_uint8(self.result, col - 1, row - 1) -- uint8_t duckdb_value_uint8(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_uint16(col, row)
	return lib.duckdb_value_uint16(self.result, col - 1, row - 1) -- uint16_t duckdb_value_uint16(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_uint32(col, row)
	return lib.duckdb_value_uint32(self.result, col - 1, row - 1) -- uint32_t duckdb_value_uint32(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_uint64(col, row)
	return lib.duckdb_value_uint64(self.result, col - 1, row - 1) -- uint64_t duckdb_value_uint64(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_float(col, row)
	return lib.duckdb_value_float(self.result, col - 1, row - 1) -- float duckdb_value_float(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_double(col, row)
	return lib.duckdb_value_double(self.result, col - 1, row - 1) -- double duckdb_value_double(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_date(col, row)
	return lib.duckdb_value_date(self.result, col - 1, row - 1) -- duckdb_date duckdb_value_date(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_time(col, row)
	return lib.duckdb_value_time(self.result, col - 1, row - 1) -- duckdb_time duckdb_value_time(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_timestamp(col, row)
	return lib.duckdb_value_timestamp(self.result, col - 1, row - 1) -- duckdb_timestamp duckdb_value_timestamp(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_interval(col, row)
	return lib.duckdb_value_interval(self.result, col - 1, row - 1) -- duckdb_interval duckdb_value_interval(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_string(col, row)
	return lib.duckdb_value_string(self.result, col - 1, row - 1) -- duckdb_string duckdb_value_string(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_varchar_internal(col, row)
	return ffiString(lib.duckdb_value_varchar_internal(self.result, col - 1, row - 1)) -- char duckdb_value_varchar_internal(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_string_internal(col, row)
	return lib.duckdb_value_string_internal(self.result, col - 1, row - 1) -- duckdb_string duckdb_value_string_internal(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_blob(col, row)
	return lib.duckdb_value_blob(self.result, col - 1, row - 1) -- duckdb_blob duckdb_value_blob(duckdb_result *result, idx_t col, idx_t row)
end

function duck:value_is_null(col, row)
	return lib.duckdb_value_is_null(self.result, col - 1, row - 1) -- _Bool duckdb_value_is_null(duckdb_result *result, idx_t col, idx_t row)
end

return duck
