--- lib/db/pgproc-ffi.lua
--
-- © 2010, 2013 David J Goehrig <dave@dloh.org>
--
-- Copyright (c) 2010, 2013, David J Goehrig <dave@dloh.org>
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
-- following conditions are met:
--
-- Redistributions of source code must retain the above copyright notice, this list of conditions and the following
-- disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
-- the following disclaimer in the documentation and/or other materials provided with the distribution.
--
-- Neither the name of the project nor the names of its contributors may be used to endorse or promote products derived
-- from this software without specific prior written permission.  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
-- CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
-- MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
-- CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
-- (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
-- (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.
--
local ffi = require "mffi"
local util = require "util"

ffi.cdef([[
typedef enum { CONNECTION_OK, CONNECTION_BAD, CONNECTION_STARTED, CONNECTION_MADE, CONNECTION_AWAITING_RESPONSE, CONNECTION_AUTH_OK, CONNECTION_SETENV, CONNECTION_SSL_STARTUP, CONNECTION_NEEDED } ConnStatusType;

typedef enum { PGRES_EMPTY_QUERY = 0, PGRES_COMMAND_OK, PGRES_TUPLES_OK, PGRES_COPY_OUT, PGRES_COPY_IN, PGRES_BAD_RESPONSE, PGRES_NONFATAL_ERROR, PGRES_COPY_BOTH } ExecStatusType;

typedef struct pg_conn PGconn;
typedef struct pg_result PGresult;
typedef struct pg_cancel PGcancel;
typedef char pqbool;

typedef struct pgNotify { char *relname; int be_pid; char *extra; struct pgNotify *next; } PGnotify;

typedef void (*PQnoticeReceiver) (void *arg, const PGresult *res);
typedef void (*PQnoticeProcessor) (void *arg, const char *message);

PGconn *PQconnectdb(const char *conninfo);
void PQfinish(PGconn *conn);
void PQclear(PGresult *res);
void PQfreemem(void *ptr);
PGresult *PQexec(PGconn *conn, const char *query);
int PQsendQuery(PGconn *conn, const char *command);
PGresult *PQgetResult(PGconn *conn);
ExecStatusType PQresultStatus(const PGresult *res);
int PQgetisnull(const PGresult *res, int tup_num, int field_num);
char *PQgetvalue(const PGresult *res, int tup_num, int field_num);
char *PQerrorMessage(const PGconn *conn);
char *PQresultErrorMessage(const PGresult *res);
ExecStatusType PQresultStatus(const PGresult *res);
int PQntuples(const PGresult *res);
int PQnfields(const PGresult *res);
char *PQfname(const PGresult *res, int field_num);
size_t PQescapeStringConn(PGconn *conn,char *to, const char *from, size_t length, int *error);
ConnStatusType PQstatus(const PGconn *conn);

char *PQencryptPassword(const char *passwd, const char *user);
char *PQencryptPasswordConn(PGconn *conn, const char *passwd, const char *user, const char *algorithm);
int	PQlibVersion(void);
]])

local sql
if util.isWin() then
	util.loadDll("graphics/libiconv-2.dll") -- x64
	util.loadDll("graphics/libintl-8.dll")
	util.loadDll("libeay32.dll")
	util.loadDll("ssleay32.dll")
	if ffi.arch == "x64" then
		ffi.loadMsvcr("120")
	end
	sql = util.loadDll("libpq.dll")
elseif util.isMac() then
	sql = util.loadDll("libpq.dylib")
else
	sql = util.loadDll("libpq.so.5", "no-error")
	if sql == nil then
		sql = util.loadDll("pq")
	end
	if sql == nil and ffi.os == "Linux" then
		util.printWarning("use 'sudo apt-get install libpq5' to install Linux libpq dependencies")
	end
end

---@return userdata
return sql
