-- pgproc module
--
-- © 2010, 2013 David J Goehrig <dave@dloh.org>
--
-- Copyright (c) 2010, 2013, David J Goehrig <dave@dloh.org>
-- All rights reserved.
--
-- Redistribution and use in source and binary forms, with or without modification, are permitted provided that the
-- following conditions are met:
--
-- Redistributions of source code must retain the above copyright notice, this list of conditions and the following
-- disclaimer. Redistributions in binary form must reproduce the above copyright notice, this list of conditions and
-- the following disclaimer in the documentation and/or other materials provided with the distribution.
--
-- Neither the name of the project nor the names of its contributors may be used to endorse or promote products derived
-- from this software without specific prior written permission.  THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND
-- CONTRIBUTORS "AS IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED TO, THE IMPLIED WARRANTIES
-- MERCHANTABILITY AND FITNESS FOR A PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT HOLDER OR
-- CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL, SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES
-- (INCLUDING, BUT NOT LIMITED TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR PROFITS; OR
-- INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT
-- (INCLUDING NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS SOFTWARE, EVEN IF ADVISED OF THE
-- POSSIBILITY OF SUCH DAMAGE.
--
local pg = {}

local ffi = require "mffi"
local lib = require "db/pgproc-ffi"
local util = require "util"
local isNull = ffi.isNull
local isNotNull = ffi.isNotNull

-- pg.resultArr = {}

function pg.reset(pgResult)
	if pgResult == nil then
		util.printError("pgResult should exist on reset")
		return
	end
	lib.PQclear(pgResult)
	--[=[ if #pg.resultArr > 0 then
		table.remove(pg.resultArr) -- returns the removed element
		--[[ if #pg.resultArr > 1 then
			util.printInfo("pgResult array removed, size: %d", #pg.resultArr)
		end ]]
	end ]=]
end

local function pgResultError(pgResult)
	return ffi.string(lib.PQresultErrorMessage(pgResult))
end

local function pgError(pgConn)
	return ffi.string(lib.PQerrorMessage(pgConn)):sub(1, -2) -- remove last \n
end

function pg.connect(connectString)
	local pgConn = lib.PQconnectdb(connectString)
	if lib.PQstatus(pgConn) ~= lib.CONNECTION_OK then
		local err = pgError(pgConn)
		return nil, err
	end
	-- local rows, pgResult, status, err = pg.querySync(pgConn, "SELECT 1")
	local _, pgResult, _, err = pg.querySync(pgConn, "SELECT 1")
	pg.reset(pgResult)
	if err then
		return nil, err
	end
	return pgConn
end

function pg.querySync(pgConn, sqlText)
	if not pgConn then
		pgConn = pg.connect()
	end
	local pgResult = lib.PQexec(pgConn, sqlText)
	--[=[ if pgResult then
		pg.resultArr[#pg.resultArr + 1] = pgResult
		--[[ if #pg.resultArr > 1 then
			util.printInfo("pgResult array added, size: %d", #pg.resultArr)
		end ]]
	end ]=]
	-- async: http://www.postgresql.org/docs/9.1/static/libpq-async.html
	local status = lib.PQresultStatus(pgResult)
	if status == lib.PGRES_TUPLES_OK then
		return lib.PQntuples(pgResult), pgResult, status
	end
	if status == lib.PGRES_EMPTY_QUERY or status == lib.PGRES_COMMAND_OK then
		pg.reset(pgResult)
		return 0, pgResult, status
	end
	local err = pgResultError(pgResult)
	if isNotNull(pgResult) then
		pg.reset(pgResult)
		pgResult = nil
	end
	return nil, pgResult, status, err
end

function pg.getResult(pgConn)
	local pgResult = lib.PQgetResult(pgConn)
	if isNull(pgResult) then
		return
	end
	local status = lib.PQresultStatus(pgResult)
	if status == lib.PGRES_TUPLES_OK then
		return lib.PQntuples(pgResult), pgResult
	end
	if status == lib.PGRES_EMPTY_QUERY or status == lib.PGRES_COMMAND_OK then
		pg.reset(pgResult)
		return 0
	end
	local err = pgResultError(pgResult)
	pg.reset(pgResult)
	return 0, nil, err
end

function pg.query(pgConn, sqlText)
	if not pgConn then
		pgConn = pg.connect()
	end
	local ret = lib.PQsendQuery(pgConn, sqlText)
	if ret ~= 1 then
		return nil, nil, ret, pgError(pgConn)
	end
	-- async: https://www.postgresql.org/docs/current/libpq-async.html
	return nil, nil, 1 -- lib.PGRES_COMMAND_OK == 1
end

--[[ function pg.rows(pgResult, status)
	if status == lib.PGRES_TUPLES_OK then
		return lib.PQntuples(pgResult)
	end
	return -1
end ]]

function pg.fields(pgResult)
	if not pgResult then
		return 0
	end
	return lib.PQnfields(pgResult)
end

function pg.field(pgResult, index)
	if not pgResult then
		return nil
	end
	return ffi.string(lib.PQfname(pgResult, index - 1))
end

function pg.fetch(pgResult, row, column)
	if not pgResult then
		return nil
	end
	--[[
	if lib.PQgetisnull(pgResult, row-1, column-1) == 1 then
	  return "null"
	end
	]]
	return ffi.string(lib.PQgetvalue(pgResult, row - 1, column - 1))
end

function pg.close(pgConn) -- , pgResult)
	-- pg.reset(pgResult)
	local ret = lib.PQfinish(pgConn)
	return ret
end

function pg.quote(pgConn, str)
	if not pgConn then
		return nil
	end
	local err = ffi.newNoAnchor('int[1]')
	local buffer = ffi.newNoAnchor('char[?]', 2 * #str + 1) -- +1 for \0
	local len_c = lib.PQescapeStringConn(pgConn, buffer, str, #str, err)
	if err[0] then
		print("Failed to escape " .. str)
		return nil
	end
	return ffi.string(buffer, len_c)
end

--[[
function pg.bind(schema)
	local query = "select proc.proname::text from pg_proc proc join pg_namespace namesp on proc.pronamespace = namesp.oid where namesp.nspname = '" .. schema .. "'"
	_G[schema] = {}
	local rows = pg.querySync(query)
	if rows < 0 then
		print(pgResultError(pgResult))
		return -1
	end
	local i = 0
	local rowCount
	while i < rows do
		local proc = pg.fetch(pgResult, i, 0);
		local F = function()
			-- local sql = "select * from  " .. schema .. "." .. proc .. "('"
			return function(...)
				local Q = query .. table.concat({...}, "','") .. "')"
				rowCount = pg.querySync(Q)
				local R = {}
				local k, j = 0, 0
				while k < rowCount do
					while j < pg.fields() do
						local key = pg.field(pgResult, j)
						local value = pg.fetch(pgResult, k, j)
						R[key] = value
						j = j + 1
					end
					k = k + 1
				end
				return R
			end
		end
		_G[schema][proc] = F()
		i = i + 1
	end
end
 ]]
return pg
