--- lib/db/plg4d.lua
local plg4d = {}

local ffi = require "mffi"
local bit = require "bit"
local json = require "json"
-- require "table.new" -- does not work in athlon
local C = ffi.C
local rshift, band = bit.rshift, bit.band
local c4d, dsql, util, skip, peg, dconv, recDataSet
local l, toUnichar, fromUnistring, toUnistring

local longHandleType
if ffi.arch == "x86" then
	require "plg4d_public_types"
	-- require "private_types"
	require "plg4d_api"
	longHandleType = "long**"
else
	require "plg4d_public_types_x64"
	require "plg4d_api_x64"
	longHandleType = "PA_long32**"
end

local printSql = false

local function loadLibs()
	if c4d == nil then
		local start = require "start"
		local loc = start.getLocal()
		c4d = loc.c4d
		local unicode4d = require "unicode/unicode4d"
		-- we must set these functions before calling other require's because they may print something
		fromUnistring = unicode4d.fromUnistring
		toUnistring = unicode4d.toUnistring
		toUnichar = unicode4d.toUnichar
		util = require "util"
		skip = util.skip
		dsql = require "dsql"
		peg = require "peg"
		dconv = require "dconv"
		recDataSet = require"recdata".set
		-- c4d = util.c4d()
		l = require"lang".l

		if not util.from4d() then
			util.printError("plg4d must not be loaded outside 4D")
		end
		-- fromUnichar = unicode4d.fromUnichar
		-- unicharLen = unicode4d.unicharLen
		-- dschema = require "dschema"
		-- date = require "dt"
	end
end

local function printLog(txt, ...)
	if printSql then
		-- loadLibs()
		return util.print(txt, ...)
	end
end

local function printError(txt, ...)
	return util.printError(txt, ...)
end

function plg4d.showSql(val)
	local printSqlPrev = printSql
	if val == nil then
		printSql = true
	else
		printSql = val
	end
	return printSqlPrev
end

function plg4d.connect() -- (conn)
	return plg4d
end

function plg4d.c4d()
	loadLibs()
	return c4d
end

local clearVariableCount = 0
local function clearVariable() -- paVariable
	clearVariableCount = clearVariableCount + 1
	-- if clearVariableCount % 10 == 0 then
	--	printLog("clearVariableCount: "..clearVariableCount)
	-- end
	-- printError("c4d.PA_ClearVariable(paVariable)")
	-- c4d.PA_ClearVariable(paVariable)
end

-- === Variables ===

local function getString(paVariable)
	--[[local value = fromUnistring(c4d.PA_GetStringVariable(variable))
	local eb = ffi.newNoAnchor("EngineBlock")
	eb.fHandle = ffi.cast("PA_Handle", variable)
	eb.fUniString1.fLength = 0
	eb.fUniString1.fString = ffi.cast("PA_Unichar*", 0)
	eb.fUniString1.fReserved1 = 0
	eb.fUniString1.fReserved2 = 0
	c4d.Call4D( C.EX_VARIABLE_TO_STRING, eb )
	local value = fromUnistring(eb.fUniString1)
	]]
	local value = ""
	if paVariable.fType == C.eVK_Unistring then
		-- value = fromUnistring(paVariable.uValue.fString)
		value = peg.replace(fromUnistring(paVariable.uValue.fString), "\r", "\n")
	end
	return value
end

local function getLong(paVariable)
	local value = 0
	if paVariable.fType == C.eVK_Longint then
		value = tonumber(paVariable.uValue.fLongint)
	end
	return value
end

local function getReal(paVariable)
	local value = 0.0
	if paVariable.fType == C.eVK_Real then
		value = tonumber(paVariable.uValue.fReal)
	end
	return value
end

local function getTime(paVariable)
	local value = 0
	if paVariable.fType == C.eVK_Time then
		value = tonumber(paVariable.uValue.fTime)
	end
	return value
end

local function getDate(paVariable)
	local y, m, d = 0, 0, 0
	if paVariable.fType == C.eVK_Date then
		y = tonumber(paVariable.uValue.fDate.fYear)
		m = tonumber(paVariable.uValue.fDate.fMonth)
		d = tonumber(paVariable.uValue.fDate.fDay)
	end
	return y, m, d
end

local function getBoolean(paVariable)
	local value = 0
	if paVariable.fType == C.eVK_Boolean then
		value = paVariable.uValue.fBoolean
	end
	return value == 1
end

--[[ local function getPointerName(paPointer)
	local pointerToVariable = paPointer.uValue.fVariable.fName
	local strLen = pointerToVariable[0] -- first char is string length
	return ffi.string(pointerToVariable+1, strLen) -- get from second, max 1+31 chars
end ]]

local function executeFunction(param)
	local paVariable = c4d.PA_ExecuteFunction(toUnistring(param))
	return paVariable -- MUST use c4d.PA_ClearVariable(paVariable) in calling function
end

function plg4d.applicationVersion()
	local paVariable = executeFunction("Application version")
	local ret = getString(paVariable)
	clearVariable(paVariable)
	return ret
end

local functionId = {} -- create local func name cache tbl
local function callMethod(method, ...)
	loadLibs()
	-- === Parameters ===
	-- printLog("c4d: " .. tostring(c4d))
	local function setLongParam(param)
		local ret = ffi.newNoAnchorNoTrace("PA_Variable") -- prevent mffi trace
		c4d.PA_SetLongintVariable(ret, param)
		return ret
	end

	local function setRealParam(param)
		local ret = ffi.newNoAnchorNoTrace("PA_Variable") -- prevent mffi trace
		c4d.PA_SetRealVariable(ret, param)
		return ret
	end

	local function setStringParam(param)
		-- local ret = ffi.newNoAnchor("PA_Variable")
		local ret = ffi.newNoAnchorNoTrace("PA_Variable") -- prevent mffi trace infinite loop
		c4d.PA_SetStringVariable(ret, toUnistring(param))
		return ret
	end

	--[[
	local function setDateParam(year, month, day)
		local ret = ffi.newNoAnchorNoTrace("PA_Variable") -- prevent mffi trace
		c4d.PA_SetDateVariable(ret, day, month, year)
		-- void PA_SetDateVariable( PA_Variable* variable, short day, short month, short year )
		return ret
	end

	local function setTimeParam(param)
		local ret = ffi.newNoAnchorNoTrace("PA_Variable") -- prevent mffi trace
		c4d.PA_SetTimeVariable(param)
		-- void PA_SetDateVariable( PA_Variable* variable, short day, short month, short year )
		return ret
	end
 ]]

	local function setBooleanParam(param)
		local ret = ffi.newNoAnchorNoTrace("PA_Variable") -- prevent mffi trace
		if param then
			c4d.PA_SetBooleanVariable(ret, 1)
		else
			c4d.PA_SetBooleanVariable(ret, 0)
		end
		-- void PA_SetDateVariable( PA_Variable* variable, short day, short month, short year )
		return ret
	end
	-- void PA_SetTableFieldVariable( PA_Variable* variable, short table, short field )
	-- void PA_SetBlobVariable( PA_Variable* variable, void* blob, long len )
	-- void PA_SetBlobHandleVariable( PA_Variable* variable, PA_Handle hblob )
	-- void PA_SetPictureVariable( PA_Variable* variable, PA_Picture picture )

	if not c4d then
		printError("c4d is null")
		return nil
	end
	local arg = {...}
	local argCount = #arg
	-- local params = ffi.newNoAnchor("PA_Variable[?]", argCount)
	local params = ffi.newNoAnchorNoTrace("PA_Variable[?]", argCount) -- prevent mffi trace infinite loop
	for i, p in ipairs(arg) do
		local ptype = type(p)
		if ptype == "cdata" then -- or userdata
			params[i - 1] = p
		elseif ptype == "string" then
			params[i - 1] = setStringParam(p)
		elseif ptype == "number" then
			if math.floor(p) == p then
				params[i - 1] = setLongParam(p)
			else
				params[i - 1] = setRealParam(p)
			end
		elseif ptype == "boolean" then
			params[i - 1] = setBooleanParam(p)
		else
			printError("parameter type is not supported: " .. ptype)
			return nil
		end
	end
	local methodId = functionId[method]
	if not methodId then
		methodId = c4d.PA_GetMethodID(toUnichar(method))
		if methodId < 1 then
			printError("method was not found: " .. method)
			return nil
		end
		functionId[method] = methodId
		-- elseif method ~= "_err LOG" then -- _err LOG causes infinite loop
		-- printLog("cached method id: "..method..": "..methodId)
	end
	local paVariable = c4d.PA_ExecuteMethodByID(methodId, params, argCount)
	for i = 1, #arg do
		clearVariable(params[i - 1])
	end
	return paVariable
end

local function callMethodNoReturn(...)
	local paVariable = callMethod(...)
	clearVariable(paVariable)
end
plg4d.callMethodNoReturn = callMethodNoReturn

function plg4d.currentProcessNumber()
	return c4d.PA_GetCurrentProcessNumber()
end

local function sleep(delayInTicks)
	-- local process = ...
	-- c4d.PA_PutProcessToSleep(process, delayInTicks)
	callMethodNoReturn("_lx_WAIT", delayInTicks)
end
plg4d.sleep = sleep

local function yield()
	c4d.PA_Yield()
end
plg4d.yield = yield

function plg4d.yieldAbsolute()
	c4d.PA_YieldAbsolute()
end

local function getVariable(variable)
	return c4d.PA_GetVariable(toUnichar(variable))
end
plg4d.getVariable = getVariable

local function getStringVariable(variable)
	return getString(c4d.PA_GetVariable(toUnichar(variable)))
end
plg4d.getStringVariable = getStringVariable

--[[ does not work
function plg4d.setLongVariable(varName, value)
	loadLibs()
	local var = getVariable(varName) -- prevent mffi trace
	c4d.PA_SetLongintVariable(var, value)
	return var
end ]]

local function getLongVariable(variable)
	loadLibs()
	local paVariable = c4d.PA_GetVariable(toUnichar(variable))
	return getLong(paVariable)
end
plg4d.getLongVariable = getLongVariable

local function getRealVariable(variable)
	return getReal(c4d.PA_GetVariable(toUnichar(variable)))
end
plg4d.getRealVariable = getRealVariable

local function getTimeVariable(variable)
	return getTime(c4d.PA_GetVariable(toUnichar(variable)))
end
plg4d.getTimeVariable = getTimeVariable

local function getDateVariable(variable)
	return getDate(c4d.PA_GetVariable(toUnichar(variable))) -- returns 3 values
end
plg4d.getDateVariable = getDateVariable

local function getBooleanVariable(variable)
	return getBoolean(c4d.PA_GetVariable(toUnichar(variable)))
end
plg4d.getBooleanVariable = getBooleanVariable

-- ===
local function callMethodReturnString(...)
	local paVariable = callMethod(...)
	local ret = getString(paVariable)
	clearVariable(paVariable)
	return ret
end
plg4d.callMethodReturnString = callMethodReturnString

local function callMethodReturnLong(...)
	local paVariable = callMethod(...)
	local ret = getLong(paVariable)
	clearVariable(paVariable)
	return ret
end
plg4d.callMethodReturnLong = callMethodReturnLong

local function callMethodReturnReal(...)
	local paVariable = callMethod(...)
	local ret = getReal(paVariable)
	clearVariable(paVariable)
	return ret
end
plg4d.callMethodReturnReal = callMethodReturnReal

local function callMethodReturnTime(...)
	local paVariable = callMethod(...)
	local ret = getTime(paVariable)
	clearVariable(paVariable)
	return ret
end
plg4d.callMethodReturnTime = callMethodReturnTime

local function callMethodReturnDate(...)
	local paVariable = callMethod(...)
	local ret = getDate(paVariable)
	clearVariable(paVariable)
	return ret
end
plg4d.callMethodReturnDate = callMethodReturnDate

local function callMethodReturnBoolean(...)
	local paVariable = callMethod(...)
	local ret = getBoolean(paVariable)
	clearVariable(paVariable)
	return ret
end
plg4d.callMethodReturnBoolean = callMethodReturnBoolean

local function getArrayPointerElement(ar, i) -- long PA_GetLongintInArray( PA_Variable ar, long i )
	local value = nil
	if ar.fType == C.eVK_ArrayPointer and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( *(PointerBlock**) ar.uValue.fArray.fData ) [ i ];
		local arr = ffi.cast("PointerBlock**", ar.uValue.fArray.fData)[0]
		value = arr[i]
	end
	return value
end

--[[
local function setArrayPointerElement(ar, i, value) -- long PA_GetLongintInArray( PA_Variable ar, long i )
	if ar.fType == C.eVK_ArrayPointer and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( *(PointerBlock**) ar.uValue.fArray.fData ) [ i ];
		local arr = ffi.cast("PointerBlock**", ar.uValue.fArray.fData)[0]
		arr[i] = value
	end
	return value
end

local function setArrayIntElement(ar, i, value) -- long PA_GetLongintInArray( PA_Variable ar, long i )
	if ar.fType == C.eVK_ArrayInteger and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( * (short**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("short**", ar.uValue.fArray.fData)[0]
		arr[i] = value
	end
	return value
end

local function setArrayLongintElement(ar, i, value)
	if ar.fType == C.eVK_ArrayLongint and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- ( * (long**) (ar.uValue.fArray.fData) )[i] = value;
		local arr = ffi.cast(longHandleType, ar.uValue.fArray.fData)[0]
		arr[i] = value
	end
end

local function setArrayRealElement(ar, i, value) -- double PA_GetRealInArray( PA_Variable ar, long i )
	if ar.fType == C.eVK_ArrayReal and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( * (long**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("double**", ar.uValue.fArray.fData)[0]
		arr[i] = value
	end
	return value
end

 ]]

local function getArrayIntElement(ar, i) -- long PA_GetLongintInArray( PA_Variable ar, long i )
	local value = 0
	if ar.fType == C.eVK_ArrayInteger and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( * (short**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("short**", ar.uValue.fArray.fData)[0]
		value = tonumber(arr[i])
	end
	return value
end

local function getArrayLongElement(ar, i) -- long PA_GetLongintInArray( PA_Variable ar, long i )
	local value = 0
	if ar.fType == C.eVK_ArrayLongint and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( * (long**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast(longHandleType, ar.uValue.fArray.fData)[0]
		value = tonumber(arr[i])
	end
	return value
end
plg4d.getArrayLongElement = getArrayLongElement

local function getArrayRealElement(ar, i) -- double PA_GetRealInArray( PA_Variable ar, long i )
	local value = 0
	if ar.fType == C.eVK_ArrayReal and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( * (long**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("double**", ar.uValue.fArray.fData)[0]
		value = tonumber(arr[i])
	end
	return value
end
plg4d.getArrayRealElement = getArrayRealElement

-- === Arrays ===
--[[
local function getArrayType(ar)
	if ar.fType == C.eVK_ArrayLongint then
		return "long"
	elseif ar.fType == C.eVK_ArrayReal then
		return "real"
	elseif ar.fType == C.eVK_ArrayUnicode then
		return "string"
	elseif ar.fType == C.eVK_ArrayDate then
		return "date"
	elseif ar.fType == C.eVK_ArrayBoolean then
		return "boolean"
	elseif ar.fType == C.eVK_ArrayInteger then
		return "integer"
	elseif ar.fType == C.eVK_ArrayPointer then
		return "pointer"
	end
	printError("not supprted array type: "..ar.fType)
	return nil
end
--]]

-- PA_GetPointerInArray
-- void PA_ResizeArray( PA_Variable *ar, long nb )
local tabTrue = {0x01, 0x02, 0x04, 0x08, 0x10, 0x20, 0x40, 0x80}
-- local tabFalse= { 0xFE, 0xFD, 0xFB, 0xF7, 0xEF, 0xDF, 0xBF, 0x7F }
local function getBitInTable(bitTable, index)
	local a = bitTable + rshift(index, 3) -- a[0] == *( bitTable + ( index >> 3 )
	local b = tabTrue[band(index, 7) + 1] -- tabTrue[ index & 7 ], +1 because Lua index starts at 1
	return band(a[0], b) ~= 0
	--[[static char GetBitInTable( unsigned char *bitTable, long index)
	{
		return (char) ( ( *( bitTable + ( index >> 3 ) ) & tabTrue[ index & 7 ] ) ? 1 : 0 );
	}
	]]
end

local function getArrayBooleanElement(ar, i) -- double PA_GetBooleanInArray( PA_Variable ar, long i )
	local value = false
	if ar.fType == C.eVK_ArrayBoolean and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- value = ( * (unsigned char**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("unsigned char**", ar.uValue.fArray.fData)[0]
		value = getBitInTable(arr, i) -- c4d.GetBitInTable(arr, i)
	end
	return value
end
plg4d.getArrayBooleanElement = getArrayBooleanElement

local function getArrayStringElement(ar, i) -- PA_GetStringInArray( arr, i )
	local value = ""
	if ar.fType == C.eVK_ArrayUnicode and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- str = ( * (PA_Unistring**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("PA_Unistring**", ar.uValue.fArray.fData)[0]
		-- pointer to start of array == arr[0] == * (PA_Unistring**)
		-- see: http:--luajit.org/ext_ffi_tutorial.html, list of common C idioms and their translation to the LuaJIT FFI:
		-- printLog("PA_Unistring"..tostring(arr[i]))
		-- value = fromUnistring(arr[i])
		value = peg.replace(fromUnistring(arr[i]), "\r", "\n")
	end
	return value
end
plg4d.getArrayStringElement = getArrayStringElement

local function setArrayStringElement(ar, i, value) -- PA_GetStringInArray( arr, i )
	if ar.fType == C.eVK_ArrayUnicode and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		if type(value) ~= "string" then
			util.printError("plg4d setArrayStringElement() value '%s' is not a string", tostring(value))
			value = ""
		end
		-- str = ( * (PA_Unistring**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("PA_Unistring**", ar.uValue.fArray.fData)[0]
		-- util.print("setArrayStringElement arr %s, value %s", tostring(arr), tostring(value))
		-- pointer to start of array == arr[0] == * (PA_Unistring**)
		-- see: http:--luajit.org/ext_ffi_tutorial.html, list of common C idioms and their translation to the LuaJIT FFI:
		-- printLog("PA_Unistring"..tostring(arr[i]))
		arr[i] = toUnistring(value)
	end
	return value
end

local function getArrayDateElementYMD(ar, i) -- void PA_GetDateInArray( PA_Variable ar, long i, short* day, short* month, short* year )
	if ar.fType == C.eVK_ArrayDate and ar.uValue.fArray.fData and i >= 0 and i <= ar.uValue.fArray.fNbElements then
		-- ( * (PA_Date**) (ar.uValue.fArray.fData) )[i];
		local arr = ffi.cast("PA_Date**", ar.uValue.fArray.fData)[0]
		return arr[i].fYear, arr[i].fMonth, arr[i].fDay
	end
	printError(l("date array index '%s' is invalid", tostring(i)))
	return 0, 0, 0
end

local function getArrayDateElementStr(ar, i)
	loadLibs()
	local y, m, d = getArrayDateElementYMD(ar, i)
	local dateStr = dsql.dateYmdToSqlString(y, m, d)
	return dateStr
end

--[[
local function setArrayDateElementYMD(ar, i, y, m, d)
	if ar.fType == C.eVK_ArrayDate
		and ar.uValue.fArray.fData
		and i >= 0
		and i <= ar.uValue.fArray.fNbElements
		then
			-- date = ( * (PA_Date**) (ar.uValue.fArray.fData) )[i];
			local arr = ffi.cast("PA_Date**", ar.uValue.fArray.fData)[0]
			arr[i].fYear = y
			arr[i].fMonth = m
			arr[i].fDay = d
	end
end

local function setArrayDateElementStr(ar, i, dateStr)
	local y = tonumber(dateStr:sub(1, 4)) -- :sub is faster than match
  local m = tonumber(dateStr:sub(6, 7))
  local d = tonumber(dateStr:sub(9, 10))
	setArrayDateElementYMD(ar, i, y, m, d)
end
]]

local function getSizeOfArray(ar) -- PA_GetArrayNbElements
	if not ar then
		printError("ar-parameter is nil")
		return -1
	end
	if (ar.fType >= C.eVK_ArrayOfArray and ar.fType <= C.eVK_ArrayBoolean) or ar.fType == C.eVK_ArrayUnicode then
		return ar.uValue.fArray.fNbElements
	end
	return -1
end
plg4d.getSizeOfArray = getSizeOfArray

local function getArrayElement(ar, i)
	if ar.fType == C.eVK_ArrayLongint then
		return getArrayLongElement(ar, i)
	elseif ar.fType == C.eVK_ArrayReal then
		return getArrayRealElement(ar, i)
	elseif ar.fType == C.eVK_ArrayUnicode then
		return getArrayStringElement(ar, i)
	elseif ar.fType == C.eVK_ArrayDate then
		return getArrayDateElementStr(ar, i) -- getArrayDateElementYMD or getArrayDateElementLong ?
	elseif ar.fType == C.eVK_ArrayBoolean then
		return getArrayBooleanElement(ar, i)
	elseif ar.fType == C.eVK_ArrayInteger then
		return getArrayIntElement(ar, i)
	elseif ar.fType == C.eVK_ArrayPointer then
		return getArrayPointerElement(ar, i)
	end
	printError("not supprted array type: " .. ar.fType)
	return nil
end
plg4d.getArrayElement = getArrayElement

--- return 4D arrays so that array name is return table index and values are in array
-- ret = {"por_asProduct_id":["RUUVI M15","PUT 18MM TERÄS","LE 2MM 100*200 SFS2001"], "fld2":[1,2,3]}
local function array4dToArrayTable(arrayNames, tagNames, option, returnType)
	local arr = {}
	tagNames = option.local_field or tagNames
	-- util.printTable(tagNames, "tagNames - plg4d array4dToArrayTable")
	-- util.printTable(option, "option - plg4d array4dToArrayTable")
	-- util.print("'%s' returnType - plg4d array4dToArrayTable", tostring(returnType))
	if #arrayNames ~= #tagNames then
		printError(l("name array size %d is not same as tag array size %d", #arrayNames, #tagNames))
	end
	for i, name in ipairs(arrayNames) do
		if option.trace then
			printLog("array name: " .. name .. ", tag name: " .. tostring(tagNames[i]))
		end
		arr[i] = getVariable(name)
	end
	local firstArrSize = getSizeOfArray(arr[1])
	local retTbl = util.newTable(firstArrSize, 0) -- retTbl = {}
	local useRecData
	local name = tagNames[1] or arrayNames[1] -- in 4d first field must be from main table
	local tablePrefix = option.table_prefix
	local tablePrefixField = peg.parseBefore(name, ".")
	if tablePrefix and tablePrefixField then
		tablePrefixField = tablePrefixField .. "."
	end
	for i = 1, #arrayNames do
		yield()
		local arrSize = getSizeOfArray(arr[i])
		name = tagNames[i] or arrayNames[i]
		if tablePrefix and peg.startsWith(name, tablePrefixField) then
			name = peg.parseAfter(name, tablePrefixField)
		end
		if peg.found(name, ".") then
			useRecData = true
		else
			useRecData = false
		end
		if arrSize ~= firstArrSize then
			printError("array size is not equal to first array size: " .. name .. " " .. arrSize .. " / " .. (tagNames[1] or arrayNames[1]) .. " " .. firstArrSize)
		elseif name == nil then
			printError(l("array name is nil, tag names: '%s'", "'" .. table.concat(tagNames, "', '") .. "'"))
		else
			if returnType == "record array" then
				for idx = 1, arrSize do
					retTbl[idx] = retTbl[idx] or {}
					if useRecData then
						recDataSet(retTbl[idx], name, getArrayElement(arr[i], idx))
					else
						retTbl[idx][name] = getArrayElement(arr[i], idx)
					end
					if arrSize >= 2500 and skip(idx, 500) == 0 then
						yield()
					end
				end
			elseif "array table" then
				retTbl[name] = util.newTable(arrSize, 0)
				for idx = 1, arrSize do
					retTbl[name][idx] = getArrayElement(arr[i], idx)
					if arrSize >= 2500 and skip(idx, 500) == 0 then
						yield()
					end
				end
			end
		end
	end
	return retTbl, firstArrSize
end

--- return 2 arrays as Lua table so that first array is Lua table index and second array is value
-- local constantTbl = plg4d.array4dToIndexTable("_res_atConstantName", "_res_alConstantNumber")
function plg4d.array4dToIndexTable(idx, value)
	local idxVariable = getVariable(idx)
	local valueVariable = getVariable(value)
	local idxArrSize = getSizeOfArray(idxVariable)
	local valueArrSize = getSizeOfArray(valueVariable)
	if idxArrSize ~= valueArrSize then
		printError("array sizes are not equal: " .. idxArrSize .. " / " .. valueArrSize)
	end
	local retTbl = {}
	for i = 1, idxArrSize do
		local idxName = getArrayElement(idxVariable, i)
		retTbl[idxName] = getArrayElement(valueVariable, i)
	end
	return retTbl
end

local valueArray, executeArray, nameArray

local function queryInit()
	callMethodNoReturn("_lx_QUERY_PARAM_RESIZE", 1)
	-- we MUST get pointers with getVariable() to arrays again after call to _lx_QUERY_PARAM_RESIZE
	valueArray = getVariable("_lx_atQueryValueParam")
	executeArray = getVariable("_lx_atQueryExecuteParam")
	nameArray = getVariable("_lx_atQueryNameParam")
end

local function executeSqlIn4d(queryText, numCount, textCount, dateCount, booleanCount, longCount, option)
	local trace, executeJson, fieldNameArray = option.trace, option.script, option.field
	-- util.printTable(fieldNameArray, "plg4d fldName executeSqlIn4d")
	-- util.printTable(option, "plg4d option executeSqlIn4d")
	if not valueArray then
		queryInit()
	end
	queryText = tostring(queryText)
	executeJson = executeJson and json.toJson(executeJson) or ""
	local queryName = option and option.query_name or "plg4d.executeSqlIn4d"
	queryName = tostring(queryName)
	printLog("plg4d executeSqlIn4d, queryText: " .. queryText)
	setArrayStringElement(valueArray, 1, queryText)
	setArrayStringElement(executeArray, 1, executeJson)
	setArrayStringElement(nameArray, 1, queryName)
	--[[
	if trace then
		-- printLog("plg4d/executeSqlIn4d, trace: "..tostring(trace))
		-- printLog(util.callPath())
	end
	-- ]]
	local traceSql = trace and 1 or -1
	local fld = fieldNameArray and table.concat(fieldNameArray, ",") or ""
	local err
	if queryName:sub(1, 4) == "new:" then
		err = callMethodReturnLong("_lx_ExecuteSqlNew", numCount, textCount, dateCount, booleanCount, longCount, traceSql, fld)
	else
		err = callMethodReturnLong("_lx_ExecuteSql", numCount, textCount, dateCount, booleanCount, longCount, traceSql, fld)
	end
	local errTxt = getStringVariable("_lx_tErr")
	if errTxt == "" then
		---@diagnostic disable-next-line: cast-local-type
		errTxt = nil
	else
		-- errTxt = l("4d execute sql error %s, %s\n %s", err, errTxt, queryText)
		errTxt = l("4d execute sql error %s: %s", err, errTxt)
		-- printLog(errTxt) -- 4d _lx_ExecuteSqlNew will print this error
		util.addError(nil, errTxt)
	end
	return err, errTxt
end

local function executeSql(queryText, fieldNameArray, option)
	loadLibs()
	if queryText == nil then
		printError("plg4d executeSql queryText: " .. tostring(queryText))
		return
	end
	option = option or {}
	local fieldType = option.field_type
	if fieldType == nil then
		local errTxt = l("field type is nil, sql:\n %s;\nfields:\n %s", tostring(queryText), table.concat(fieldNameArray or {}, ", "))
		printError(errTxt)
		return nil, util.addError(nil, errTxt)
	end
	local trace = option.trace or false
	local numCount = 0
	local textCount = 0
	local dateCount = 0
	local booleanCount = 0
	local longCount = 0
	local arrName = util.newTable(#fieldNameArray)
	-- util.printTable(fieldNameArray, "fieldNameArray - plg4d executeSql")
	for i, fld in ipairs(fieldNameArray) do
		if fld == nil then
			local errTxt = l("field is nil at index %d, sql:\n %s;\nfields:\n %s", i, tostring(queryText), table.concat(fieldNameArray, ", "))
			printError(errTxt)
			return nil, util.addError(nil, errTxt)
		end
		local fldType = fieldType[i] -- dschema.fieldTypeBasic(fld)
		if fldType == "number" or fldType == "double" then
			numCount = numCount + 1
			arrName[i] = "_lx_arExe" .. numCount
		elseif fldType == "integer" or fldType == "time" or fldType == "bigint" then
			--[[ 	if compatibilityPref then
				numCount = numCount+1
				arrName[i] = "_lx_arExe"..numCount
			else ]]
			longCount = longCount + 1
			arrName[i] = "_lx_alExe" .. longCount
			-- end
		elseif fldType == "date" or fldType == "timestamp" then
			dateCount = dateCount + 1
			arrName[i] = "_lx_adExe" .. dateCount
		elseif fldType == "boolean" then
			booleanCount = booleanCount + 1
			arrName[i] = "_lx_afExe" .. booleanCount
		elseif fldType == "string" then
			textCount = textCount + 1
			arrName[i] = "_lx_atExe" .. textCount
		else
			local errTxt = l("field type '%s' is not supported, sql:\n %s;", tostring(fldType), queryText)
			printError(errTxt)
			return nil, util.addError(nil, errTxt)
		end
		--[[ if trace then
			util.print("**** "..fld.." - fldType: "..fldType.." - arrName: "..arrName[i])
		end ]]
	end
	-- util.printTable(arrName, "arrName - plg4d executeSql")
	queryText = queryText .. "\n INTO :" .. table.concat(arrName, ",:")
	option.trace = trace
	option.field = fieldNameArray
	-- option.field_type = table.concat(fieldType, ",")
	if queryText:find("SELECT 4D_MORE_ROWS FROM", 1, true) then
		local rows = callMethodReturnLong("_lx_ExecuteSqlLoad")
		return arrName, nil, rows
	end
	local err, errText = executeSqlIn4d(queryText, numCount, textCount, dateCount, booleanCount, longCount, option)
	if err ~= 0 then
		return nil, errText
	end
	return arrName, errText
end

--- returns Lua table so that array name is return table index and values are in array
-- ret = {"por_asProduct_id":["RUUVI M15","PUT 18MM TERÄS","LE 2MM 100*200 SFS2001"], "fld2":[1,2,3]}
local function selectionToArrayTable(sql, fieldNameArray, option, returnType)
	option = option or {}
	option.trace = option.trace or sql.show_sql
	returnType = returnType or option.return_type
	local info = {}
	info.return_type = returnType
	-- util.printTable(option, "option - plg4d selectionToArrayTable")
	-- util.print("plg4d selectionToArrayTable, returnType: " .. tostring(returnType))
	-- printLog("plg4d selectionToArrayTable call path: "..util.callPath())
	-- util.printTable(sql, "sql plg4d.selectionToArrayTable")
	--[[ if option.trace then
		util.printTable(fieldNameArray, "plg4d fldName selectionToArrayTable")
		util.printTable(fldType, "plg4d fldType selectionToArrayTable")
		printLog("plg4d returnType: "..tostring(returnType))
	end ]]
	local queryText
	if not sql then
		local err = "sql parameter is null"
		info.error = err
		printLog(err) -- printError(err)
		return nil, info
	elseif sql.sql then
		queryText = sql.sql -- sql is cursor table created earlier by execute()
	elseif type(sql) == "string" then
		queryText = sql
	else
		queryText = sql.queryText
	end
	if option.trace then
		printLog("plg4d selectionToArrayTable:\n ", queryText) -- , json.toJsonRaw(fieldNameArray), json.toJsonRaw(fieldType))
	end
	local arrName, err, moreRows = executeSql(queryText, fieldNameArray, option)
	if err and err ~= "" then
		info.error = err
	end
	if not arrName then
		return nil, info
	end
	local ret, rowCount = array4dToArrayTable(arrName, fieldNameArray, option, returnType) -- create array table from array names
	--[[
	if option.trace then
		util.printTable(arrName, "plg4d.selectionToArrayTable: arrName")
		util.printTable(fieldNameArray, "plg4d.selectionToArrayTable: fieldNameArray")
		util.printTable(ret, "plg4d.selectionToArrayTable: ret")
	end
	--]]
	info.column_name = fieldNameArray
	info.column_count = #fieldNameArray
	info.row_count_more = moreRows
	info.row_count = rowCount
	local rowCountTotal = getLongVariable("_lx_lResultSetSize")
	info.row_count_total = rowCountTotal > 0 and rowCountTotal or rowCount
	return ret, info
end
plg4d.selectionToArrayTable = selectionToArrayTable

function plg4d.selectionToRecordArray(sql, fieldNameArray, option)
	-- util.printTable(option, "plg4d.selectionToRecordArray option")
	local sel, info = selectionToArrayTable(sql, fieldNameArray, option, "record array")
	-- util.printTable(sel, "plg4d.selectionToRecordArray sel")
	return sel, info
end

function plg4d.selectionToRecordTable(sql, fieldNameArray, option)
	-- option.trace = true
	-- util.printTable(option, "plg4d.selectionToRecordTable option")
	local sel, info = selectionToArrayTable(sql, fieldNameArray, option, "array table")
	sel = dconv.arrayTableToRecordTable(sel, info) -- no (sel, info, nil, param) here because trace is 3. parameter
	return sel, info
end

local function execute(sqlExecuteText, option) -- , showSql, executeJson, queryName, fieldArray)
	-- printLog("plg4d execute sql: ", sqlExecuteText) -- call from 4d
	-- util.printTable(option, "option - plg4d execute")
	-- option.trace = true
	if sqlExecuteText:find("SELECT ", 1, true) == 1 then
		local cursor = {sql = sqlExecuteText}
		return cursor
	elseif (sqlExecuteText:find("INSERT INTO ", 1, true) == 1 or sqlExecuteText:find("UPDATE ", 1, true) == 1 or sqlExecuteText:find("DELETE FROM ", 1, true) == 1 or sqlExecuteText == "START" or sqlExecuteText == "ROLLBACK" or sqlExecuteText == "COMMIT") then
		if printSql then
			printLog("plg4d execute:\n " .. sqlExecuteText)
		end
		local err, errText = executeSqlIn4d(sqlExecuteText, 0, 0, 0, 0, 0, option)
		if err ~= 0 then
			return nil, errText
		end
		return 0, errText -- cursor number 0 == don't run anything else
	end
	-- when this comes?
	local errText = printError("plg4d execute other than SELECT, UPDATE, INSERT INTO or DELETE FROM (in uppercase) is not supported, sql:\n '%s;'", sqlExecuteText)
	--[[ local err, errText = executeSqlIn4d(sqlExecuteText, 0, 0, 0, 0, 0, option)
	if err ~= 0 then
		return nil, errText
	end ]]
	return nil, errText -- cursor number 0 == don't run anything else
end
plg4d.execute = execute

function plg4d.selectPath(prompt)
	if not prompt then
		prompt = ""
	end
	callMethodNoReturn("_doc FolderChoose", prompt)
	local path = getStringVariable("_doc_tPrevPath")
	if util.isWin() then
		path = peg.replace(path, "\\", "/")
	elseif util.isMac() then
		path = peg.replace(path, ":", "/")
	end
	return path
end

return plg4d

--[=[
function plg4d.querySqlAter(param, data, info)
	-- util.printTable(param, "plg4d/querySqlAter/param")
	if param.script == nil or param.script.save_preference == nil then
	elseif param.script.save_preference.run_script == nil or #param.script.save_preference.run_script <= 0 then
	elseif param.script.record_id_table == nil or param.script.record_id == nil then
	else
		-- add rec.record_id -tag
		local rec = util.arrayRecord(param.script.record_id_table, param.script.save_preference.run_script, "table", 1)
		if rec then
			rec.record_id = param.script.record_id
		end
		local paramTxt = json.toJsonRaw({run_script = param.script.save_preference.run_script})
		callMethodNoReturn("_lx_JSON_SCRIPT_RUN", paramTxt)
	end
	return nil
	--[[
		"run_script": [
			{
				"table": "por",
				"record_id": "muistiin-> generoi record_id array datasta",
				"script_field":["ordr.product_id", "ordr.order_amount"]
			},
			{
				"table": "po",
				"record_id": "muistiin-> generoi record_id array datasta",
				"script_field":["ord.company_id"],
				"up_calculate": 1
			}
		]
		--]]

	-- callMethodNoReturn("_app_ JSON_SCRIPT_RUN_LUA", paramTxt)
	-- callMethodNoReturn("_doc FolderChoose", prompt)
	-- local path = getStringVariable("_doc_tPrevPath")
end
]=]

--[==[

--[[
	DOES not work with normal variables
function getVariableName(paVariable)
	local value = paVariable.uValue.fVariableDefinition.fName
	value = ffi.string(value)
	return "**-"..value.."-**" -- "**-"..fromUnichar(value).."-**" --fromUnichar(value) --value --toUnichar(value)
end
]]

--[=[
do
	local saveFieldArray, saveIdArray, saveValueArray, saveScriptFieldArray
	local maxFieldCount = -100

	function arrayTableToSelection(fldValueTbl, idFld, idFieldArray, scriptFieldArray)
		local errTxt, idField --, idFldName
		idField = idFld
		if not idField then
			errTxt = l"plg4d.arrayTableToSelection: unique id field is missing"
		else
			local _,firstValueArr = next(fldValueTbl)
			if #firstValueArr ~= #idFieldArray then
				errTxt = l"value array length is not same as id array length"
				printLog(errTxt.." "..#firstValueArr.."/"..#idFieldArray)
				return errTxt
			end
			local fldCount = 0
			local fldArrName = {}
			for fldName,_ in pairs(fldValueTbl) do -- for fldName,fldValueArrX
				fldCount = fldCount +1
				fldArrName[fldCount] = fldName
			end

			if #idFieldArray > maxFieldCount then -- maxFieldCount is negative in first call, will go here
				maxFieldCount = math.abs(maxFieldCount)
				if #idFieldArray > maxFieldCount then
					maxFieldCount = #idFieldArray
				end
				if fldCount > maxFieldCount then
					maxFieldCount = fldCount
				end
				callMethodNoReturn("_lx_ARR_TO_SEL_PARAM_RESIZE", maxFieldCount)
				saveFieldArray = getVariable("_lx_atSaveFieldParam")
				saveIdArray = getVariable("_lx_atSaveIdParam")
				saveScriptFieldArray = getVariable("_lx_atScriptFieldArray")
			end
			for i,val in ipairs(fldArrName) do
				setArrayStringElement(saveFieldArray, i, val)
			end
			for i,val in ipairs(idFieldArray) do
				setArrayStringElement(saveIdArray, i, tostring(val))
			end

			if fldCount > 0 then
				callMethodNoReturn("_lx_ARR_TO_SEL_PARAM_RESIZE", -(fldCount * #fldValueTbl[fldArrName[1]]))
				saveValueArray = getVariable("_lx_atSaveValueParam")
				local i = 0
				for _,arrayName in ipairs(fldArrName) do-- loop all arrays and all elements, make big value array size = arrays*rows
					for _,val in ipairs(fldValueTbl[arrayName]) do
						i = i+1
						setArrayStringElement(saveValueArray, i, tostring(val))
					end
				end
			end

			if scriptFieldArray then
				if scriptFieldArray == true then
					scriptFieldArray = fldArrName
				end
				for i,val in ipairs(scriptFieldArray) do
					setArrayStringElement(saveScriptFieldArray, i, val)
				end
			end
			local err = callMethodReturnLong("_lx_ARR_TO_SEL", idField, fldCount, #idFieldArray)
			if err ~= 0 then
				printLog("plg4d.lua - arrayTableToSelection: idField, fldCount, #idFieldArray: ", tostring(idField), tostring(fldCount), tostring(#idFieldArray))
				errTxt = getStringVariable("_lx_tErr")
				printLog("*** _lx_ARR_TO_SEL error: "..err..", "..errTxt)
			end
		end
		return errTxt
	end
end
]=]

--[[ Structure Access
-- http:--sources.4d.com/trac/4d_4dpluginapi/wiki/XKU00034.HTM


function fieldStructureName(tableNum, fieldNum)
	local fieldNameArray = ffi.newNoAnchor("PA_Unichar[32]")
	c4d.PA_GetFieldName(tableNum, fieldNum, fieldNameArray)
	local ret = fromUnichar(fieldNameArray)
	clearVariable(fieldNameArray)
	return ret
end

function tableStructureName(tableNum)
	local tableName = ffi.newNoAnchor("PA_Unichar[32]")
	c4d.PA_GetTableName(tableNum, tableName)
	local ret = fromUnichar(tableName)
	clearVariable(tableName)
	return ret
end

do
	local tableCountCache
	function tableCount()
		if not tableCountCache then
			tableCountCache = c4d.PA_CountTables() -- plg4d.callMethodReturnLong("_tbl_ Count")
		end
		return tableCountCache
	end
end

do
	local fieldCountCache = {}
	function fieldCount(tableNum)
		if not fieldCountCache[tableNum] then
			fieldCountCache[tableNum] = c4d.PA_CountFields(tableNum)
		end
		return fieldCountCache[tableNum]
	end
end
]]

--- Get variable
--[[
function getPointer(paVariable)
	if paVariable.fType == C.eVK_Pointer	then
		return paVariable.uValue.fPointer -- PA_Pointer
	end
	printError("parameter 1 was not a pointer to variable")
	return nil
end

function getPointerArray(variable)
	local paVariable = getVariable(variable)
	if paVariable.fType == C.eVK_ArrayPointer then
		return paVariable
	end
	printError("parameter 1 was not a pointer to variable")
	return nil
end
]]

--[[
function isCompiled()
	local ret = c4d.PA_IsCompiled(1)
	if tonumber(ret) == 1 then
		return true
	end
	return false
end

function executeMethod(param)
	c4d.PA_ExecuteMethod(toUnistring(param))
end
]]

-- dbfix: f
local function fieldNameToNumber(fldName)
	local ret = f[tostring(fldName)] or f["_"..tostring(fldName)] -- or try 4D cha -> _cha
	if not ret and type(fldName) == "string" then
		local pos = peg.find(fldName, ".")
		if pos > 0 then
			local fld1 = fieldNameToNumber(fldName:sub(1, pos - 1))
			local fld2 = fieldNameToNumber(fldName:sub(pos + 1))
			if fld1 and fld2 then
				ret = {fld1, fld2}
			end
		end
	end
	return ret
end

function plg4d.confirm(prompt)
	if not prompt then
		prompt = ""
	end
	callMethodNoReturn("_info_ Confirm", prompt)
	local ok = getLongVariable("_info_lbYes")
	return ok == 1
end

function constantNumber(constantName)
	if constantName == "_el_kmOutput" then
		return 1100005
	elseif constantName == "_el_kmModify" then
		return 1100004
	elseif constantName == "_el_kmBrowse" then
		return 1100003
	elseif constantName == "Into set" then
		return 1
	elseif constantName == "Into named selection" then
		return 2
	else
		return nil
	end
end
--]]

--[[
function plg4d.connect(prf, option)
	return {prf = prf, option = option}
end

function plg4d.disconnect() -- disconnect(conn)
	return
end
]]

--[[ Selections
function recordsInSelection(tbl)
	return c4d.PA_RecordsInSelection(tbl)
end

function recordsInTable(tbl)
	return c4d.PA_RecordsInTable(tbl)
end

function deleteSelection(tbl)
	local err = l"plg4d deleteSelection() has not been implemented, table: "..tostring(tbl)
	printLog(err)
	-- callMethodNoReturn("_sel DELETE", tbl)
	return err
end
]]

--[=[
local queryParamCount = 0
local orderParamCount = 0
-- local queryBySql
do
	local opArray, fldArray, compArray, valueArray, executeArray, nameArray --,arrArray
	local orderFldArray, orderDirArray
	local maxQueryParamCount = 1000
	local function queryInit()
		--[[
		ARRAY TEXT(_lx_atQueryArrayParam;$1)
		ARRAY TEXT(_lx_atQueryOpParam;$1)
		ARRAY LONGINT(_lx_alQueryFldParam;$1)
		ARRAY TEXT(_lx_atQueryCompParam;$1)
		ARRAY TEXT(_lx_atQueryValueParam;$1)

		ARRAY LONGINT(_lx_alOrderFldParam;$1)
		ARRAY LONGINT(_lx_alOrderDirParam;$1)
		]]
		callMethodNoReturn("_lx_QUERY_PARAM_RESIZE", maxQueryParamCount)
		-- we MUST get pointers to arrays again after call to _lx_QUERY_PARAM_RESIZE
		opArray = getVariable("_lx_atQueryOpParam")
		fldArray = getVariable("_lx_alQueryFldParam")
		compArray = getVariable("_lx_atQueryCompParam")
		valueArray = getVariable("_lx_atQueryValueParam")
		-- arrArray = getVariable("_lx_atQueryArrayParam")
		orderFldArray = getVariable("_lx_alOrderFldParam")
		orderDirArray = getVariable("_lx_alOrderDirParam")
		executeArray = getVariable("_lx_atQueryExecuteParam")
		nameArray = getVariable("_lx_atQueryNameParam")
	end

	function query(operator, fieldNum, comparison, value)
		printLog("query", operator, fieldNum, comparison, json.toJson(value))
		if not opArray then
			queryInit()
		end
		if operator == "" then
			queryParamCount = 0
			orderParamCount = 0
		end
		queryParamCount = queryParamCount+1
		if queryParamCount > maxQueryParamCount then
			maxQueryParamCount = queryParamCount
			callMethodNoReturn("_lx_QUERY_PARAM_RESIZE", maxQueryParamCount)
		end

		local i = queryParamCount
		if operator == "and" then
			operator = "&"
		elseif operator == "or" then
			operator = "|"
		end
		if comparison == "<>" then
			comparison = "#"
		end

		if type(value) == "table" then
			if value.y and value.m and value.d then -- date table
				value = date.ymdToDateString(value.y, value.m, value.d)
			else
				local arrayOperator
				if comparison=="in" then
					arrayOperator = "|"
					comparison = "="
				elseif comparison=="not in" then
					arrayOperator = "&"
					comparison = "#"
				end
				-- recursive calls
				queryParamCount = queryParamCount-1
				for j,val in ipairs(value) do
					if j==1 then
						query(operator, fieldNum, comparison, val)
					else
						query(arrayOperator, fieldNum, comparison, val)
					end
				end
				return
				--[[
				value = "query array"
				callMethodNoReturn("_lx_QUERY_PARAM_RESIZE", -(#value))
				arrArray = getVariable("_lx_atQueryArrayParam")
				for j,val in ipairs(value) do
					setArrayStringElement(arrArray, j, tostring(val))
				end
				]]
				-- printError("Query value of type table can be only date")
				-- queryParamCount = queryParamCount-1
			end
		end
		setArrayStringElement(opArray, i, operator)
		setArrayLongintElement(fldArray, i, fieldNum)
		setArrayStringElement(compArray, i, comparison)
		setArrayStringElement(valueArray, i, tostring(value))
		setArrayStringElement(executeArray, i, "")
	end

	function orderBy(fieldNum, asc)
		if not opArray then
			queryInit()
		end
		orderParamCount = orderParamCount+1
		if orderParamCount > maxQueryParamCount then
			maxQueryParamCount = orderParamCount
			callMethodNoReturn("_lx_QUERY_PARAM_RESIZE", maxQueryParamCount)
		end
		local i = orderParamCount
		if asc == ">" then
			setArrayLongintElement(orderDirArray, i, 1)
		elseif asc == "<" then
			setArrayLongintElement(orderDirArray, i, -1)
		else
			local errTxt = l"Order operator must be > or <"
			printLog(errTxt) -- printError(errTxt)
		end
		setArrayLongintElement(orderFldArray, i, fieldNum)
	end

	queryBySql = function(sql, fieldNum, executeJson)
		loadLibs()
		-- query by sql
		-- printLog("plg4d query by sql: "..sql, json.toJsonRaw(fieldNum))
		if not opArray then
			queryInit()
		end
		if #fieldNum > maxQueryParamCount then
			maxQueryParamCount = #fieldNum
			callMethodNoReturn("_lx_QUERY_PARAM_RESIZE", maxQueryParamCount)
		end

		local query = dsql.sqlQueryWherePart(sql)
		-- printLog("plg4d query by sql query: "..query)
		setArrayStringElement(valueArray, 1, query)
		setArrayStringElement(executeArray, 1, executeJson or "")

		local firstFldNum = fieldNameToNumber(fieldNum[1])
		if type(firstFldNum) ~= "number" or firstFldNum == 0 then
			local errTxt = l"QuerySql error, first field number is not valid: "..tostring(firstFldNum)..", sql: "..sql..";"
			printLog(errTxt) -- printError(errTxt)
		end
		local err = callMethodReturnLong("_lx_QuerySql", firstFldNum)
		if err ~= 0 then
			local errTxt = getStringVariable("_lx_tErr")
			errTxt = l"QuerySql error: "..err..", "..errTxt
			printLog(errTxt) -- printError(errTxt)
		end
	end
end

do
	local longArray
	local maxFieldCount = -100
	function selectionToArray4d(fieldNum)
		if #fieldNum > maxFieldCount then -- maxFieldCount is negative in first call, will go here
			maxFieldCount = math.abs(maxFieldCount)
			if #fieldNum > maxFieldCount then
				maxFieldCount = #fieldNum
			end
			callMethodNoReturn("_lx_SEL_TO_ARR_PARAM_RESIZE", maxFieldCount)
			longArray = getVariable("_lx_alCallbackParam")
		end
		for i,val in ipairs(fieldNum) do
			setArrayLongintElement(longArray, i, val)
		end
		callMethodNoReturn("_lx_SEL_TO_ARR", #fieldNum)
	end
end

--- 4D array to Lua Table
function array4dToTable(...)
	local arg = {...} -- array names
	if type(arg[1]) == "table" then
		arg = arg[1]
	end
	local arr = {}
	for i=1,#arg do  --i,p in ipairs(arg) do
		arr[i] = {}
		arr[i].name = arg[i]
		arr[i].variable = getVariable(arr[i].name)
	end
	local firstArrSize = getSizeOfArray(arr[1].variable)
	local retTbl = {}
	for i=1,#arg do
		local arrSize = getSizeOfArray(arr[i].variable)
		local name = arr[i].name
		if arrSize ~= firstArrSize then
			printError("array size is not equal to first array size: "..name.." "..arrSize.." / "..arr[1].name.." "..firstArrSize)
		end
		retTbl[name] = {}
		for idx=1,arrSize do
		  retTbl[name][idx] = getArrayElement(arr[i].variable, idx)
			-- printLog(retTbl[name][idx])
		end
	end
	return retTbl
end

--- return 4D arrays so that array name is return table record field name
-- ret = [{"por_asProduct_id":"RUUVI M15","fld2":1},{"por_asProduct_id":"PUT 18MM TERÄS","fld2":2}]
function array4dToRecordTable(...)
	local arg = {...} -- array names
	--[[
	local stringReplaceCallback
	if type(arg[1]) == "table" then
		stringReplaceCallback = arg[2] -- must get 2 before 1
		arg = arg[1] -- arg is now table at arg[1]
	end
	]]
	local arg2
	if arg[2] and type(arg[2]) == "table" then
		arg2 = arg[2] -- tag names, mut get before arg = arg[1]
		arg = arg[1]  -- 4d array names
	end
	local arr = {}
	for i,argValue in ipairs(arg) do
		arr[i] = getVariable(argValue)
	end
	local firstArrSize = getSizeOfArray(arr[1])
	local retTbl = util.newTable(firstArrSize, 0)
	for idx=1,firstArrSize do
		retTbl[idx] = {}
	end
	for i=1,#arg do
		yield()
		local arrSize = getSizeOfArray(arr[i])
		local name
		if arg2 then
			name = arg2[i]
		else
			name = arg[i]
		end
		if arrSize ~= firstArrSize then
			local firstArrName
			if arg2 then
				firstArrName = arg2[1]
			else
				firstArrName = arg[1]
			end
			printError("array size is not equal to first array size: "..name.." "..arrSize.." / "..firstArrName.." "..firstArrSize)
		else
			-- local arrType = getArrayType(arr[i])
			for idx=1,arrSize do
				retTbl[idx][name] = getArrayElement(arr[i], idx)
				--[[
				if stringReplaceCallback and arrType == "string" then
					retTbl[idx][name] = stringReplaceCallback(retTbl[idx][name])
				end
				]]
				if arrSize >= 2500 and skip(idx, 500) == 0 then
					yield()
				end
			end
		end
	end
	return retTbl,firstArrSize
end
--]==]

--[=[ PA_GetStringField() and others do not work in 4Dv19

local ffi = require "mffi"
local C = ffi.C

local function fieldKind(kind)
	--[[
	typedef enum
	{
		eFK_InvalidFieldKind	= -1,
		eFK_AlphaField		= 0,	//  Alphanumeric field (from 2 to 80 characters)
		eFK_RealField			= 1,	//  Numeric field (Double or Extended value)
		eFK_TextField			= 2,	//  Text field (up to 32000 characters)
		eFK_PictureField		= 3,	//  Picture field (virtually any block of data)
		eFK_DateField			= 4,	//  Date field
		eFK_BooleanField		= 6,	//  Boolean field
		eFK_SubfileField		= 7,	//  Subfile field
		eFK_IntegerField		= 8,	//  Integer field (-32768..32767)
		eFK_LongintField		= 9,	//  Long Integer field (-2^31..(2^31)-1)
		eFK_TimeField			= 11,	//	Time field
		eFK_Long8				= 25,
		eFK_BlobField			= 30,	//	Blob field
		eFK_FloatField		= 35,	//  Float
		eFK_ObjectField		= 38	//  Object field
	} PA_FieldKind;
	]]
	if kind == C.eFK_AlphaField then
		return "Alpha"
	elseif kind == C.eFK_RealField then
		return "Real"
	elseif kind == C.eFK_TextField then
		return "Text"
	elseif kind == C.eFK_PictureField then
		return "Picture"
	elseif kind == C.eFK_BooleanField then
		return "Boolean"
	elseif kind == C.eFK_IntegerField then
		return "Integer"
	elseif kind == C.eFK_LongintField then
		return "Longint"
	elseif kind == C.eFK_TimeField then
		return "Time"
	elseif kind == C.eFK_BlobField then
		return "Blob"
	elseif kind == C.eFK_FloatField then
		return "Float"
	elseif kind == C.eFK_SubfileField then
		return "Subfile"
		-- elseif kind == C.eFK_Long8 then
		-- 	return "Long8"
		-- elseif kind == C.eFK_ObjectField then
		-- 	return "Object"
	else
		return "no-in-use"
	end
	-- return tostring(kind)
end

local function fieldValue(table, field, fieldType)
	--[[
	PA_Unistring PA_GetStringField     ( short table, short field );
	long         PA_GetBlobField       ( short table, short field, void* blob );
	PA_Handle    PA_GetBlobHandleField ( short table, short field );
	PA_Picture   PA_GetPictureField    ( short table, short field );
	double       PA_GetRealField       ( short table, short field );
	long         PA_GetLongintField    ( short table, short field );
	short        PA_GetIntegerField    ( short table, short field );
	long         PA_GetTimeField       ( short table, short field );
	void         PA_GetDateField       ( short table, short field, short* day, short* month, short* year );
	char         PA_GetBooleanField    ( short table, short field );
	]]
	if fieldType == "Alpha" or fieldType == "Text" then
		local str = C.PA_GetStringField(table, field)
		return fromUnistring(str)
	elseif fieldType == "Real" then
		return C.PA_GetRealField(table, field)
	elseif fieldType == "Longint" then
		return C.PA_GetLongintField(table, field)
	elseif fieldType == "Integer" then
		return C.PA_GetIntegerField(table, field)
	elseif fieldType == "Time" then
		return C.PA_GetTimeField(table, field)
	elseif fieldType == "Date" then
		return C.PA_GetDateField(table, field)
	elseif fieldType == "Boolean" then
		return C.PA_GetBooleanField(table, field) == 0 and false or true
	else
		return nil -- fieldType
	end
end

local function getRecord(table, fieldCount)
	--[[
	void  PA_GetFieldProperties ( short table, short field, PA_FieldKind* kind, short* stringlength, char* indexed, long* attributes );
	]]
	local kind = ffi.newNoAnchorNoTrace("PA_FieldKind[1]") -- prevent mffi trace
	local stringLength = ffi.newNoAnchorNoTrace("short[1]")
	local indexed = ffi.newNoAnchorNoTrace("char[1]")
	local attributes = ffi.newNoAnchorNoTrace("uint32_t[1]")
	local fieldType, value
	-- fieldCount = 3
	for field = 1, 1 do -- fieldCount do
		c4d.PA_GetFieldProperties(table, field, kind, stringLength, indexed, attributes)
		fieldType = fieldKind(kind[0])
		value = fieldValue(table, field, fieldType) -- PA_GetPackedRecord
		util.print("field %d, value '%s', kind %s, stringLength %s, indexed %s, attributes %s.  ", field, value, fieldType, tostring(stringLength[0]), tostring(indexed[0]), tostring(attributes[0]))
		-- value = fieldValue(table, field, fieldType)
	end
	return {}
end
]=]
