--- lib/db/qry.lua
--
-- @module db
local qry = {}

-- local argcheck = require "argcheck"
local dprf = require "dprf"
local dsql = require "dsql"
local dconv = require "dconv"
local dqry = require "dqry"
local dsave = require "dsave"
local convert = require "convert/convert"
local recdata = require "recdata"
local recData, recDataSet = recdata.get, recdata.set
local l = require"lang".l
local fn = require "fn"
local util = require "util"
local json = require "json"
local dqjson = require "dqjson"
local dschema = require "dschema"
local dconn = require "dconn"
local peg = require "peg"
local selection = require "selection"
local execute = require "execute"
local tableCode = require "table-code" -- script_old
local dload = require "dload"
local col = require "convert/col"
local rest, auth, callRest
local queryPreference = dprf.queryPreference
local replaceRecParameter = dprf.replaceRecParameter
local lastModifyTimeFieldWidth = 100

---@return table -- , string
local function answerError(err)
	if rest == nil then
		rest = require "rest"
	end
	return rest.answerError(err)
end

local function getLimit(limit)
	if type(limit) == "string" and peg.found(limit, ".json/") then
		local limit2, err2 = dprf.queryPreference(limit) -- queryPreference() handles name.json/tag -case
		if err2 then
			util.addError(err2, l("subquery preference '%s' was not valid, error '%s'"), limit, err2)
			limit = nil
		else
			limit = tonumber(limit2)
		end
	end
	return limit
end
qry.getLimit = getLimit

local function resultToArray(result)
	if result.error == nil and result.data and not util.isArray(result.data) then
		result.data = {result.data}
	end
end

function qry.queryPreference(queryJsonOrName) -- , param)
	local query, queryJsonName, err
	if type(queryJsonOrName) == "table" then
		query = queryJsonOrName
	else
		-- if parameter then and parameter.show_sql == true then
		-- 	queryJsonName = query
		-- end
		queryJsonName = queryJsonOrName
		if not peg.startsWith(queryJsonName, "query/") and not peg.found(queryJsonName, "form/") then
			query, err = util.prf("query/" .. queryJsonName, "no-cache no-error")
		end
		if query == nil or err then
			query, err = util.prf(queryJsonName, "no-cache") -- in new system query/ -folder will be deleted eventually
		end
		if err then
			return answerError(err)
		end
	end
	return query, err, queryJsonName
end

local function setColumnOption(column, queryParam)
	column.option = column.option or {}
	if queryParam.return_type and queryParam.return_type ~= "record array" then
		column.option.return_type = queryParam.return_type
	end
	column.option.table = queryParam.table and queryParam.table[1].table or ""
	if column.option.table ~= "" then
		column.option.record_type = queryParam.table[1].record_type and queryParam.table[1].record_type[1]
		column.option.table_prefix = dschema.tablePrefix(column.option.table)
	end
end

---@return table, string
function qry.queryJson(queryJsonOrName, param)
	local origQueryParam = param
	local parameter = param and param.parameter or param or {}
	local origParam = parameter
	local query, err, queryJsonName = qry.queryPreference(queryJsonOrName) -- , param)
	-- if query and query.field
	if not query or util.tableIsEmpty(query) then
		return answerError(l("query '%s' was not found or json was invalid", tostring(origQueryParam)))
	elseif not (query.field or parameter and (parameter.field or parameter.column)) then
		if query.grid then
			return {} -- grid will be handled later
		end
		return answerError(util.parameterError("field or column"))
	elseif query.query == nil then
		return answerError(util.parameterError("query"))
	elseif not (parameter and parameter.column) and (not query.field or #query.field < 1) and (not parameter or not parameter.field or #parameter.field < 1) then
		return answerError(l("parameter field count is smaller than 1"))
	end
	if parameter then
		parameter = util.clone(parameter) -- do not change the original parameter
	end
	if parameter.organization_id then
		query.organization_id = parameter.organization_id
	end
	-- copy query.parameter keys to parameter
	if query.parameter then
		if not parameter then
			parameter = query.parameter
		else
			for key, val in pairs(query.parameter) do
				if parameter[key] == nil then
					parameter[key] = val
				end
			end
			if type(queryJsonOrName) == "table" then
				query = util.clone(query) -- just to be sure that we do not change the original query
			end
			query.parameter = parameter
		end
	end

	-- parameter can contain column-definition
	-- local columnPrfName
	if parameter and parameter.column then -- and not query.field, -- parameter always overrides field -tag
		query = util.clone(query) -- do not change the original query
		if type(parameter.column) == "string" then
			-- columnPrfName = parameter.column
			local prf = util.prf(parameter.column, "no-cache")
			parameter.column = prf.column
		end
		if parameter.column then
			parameter.field = util.objectFieldToArr(parameter.column, "field")
		end
	end

	if parameter.last_modify_time and parameter.modify_time_field then
		if util.arrayItemFound(param.parameter.modify_time_field, parameter.field) == false then
			-- util.printWarning("please add column '%s' to preference '%s'", param.parameter.modify_time_field, columnPrfName or queryJsonName)
			parameter.field[#parameter.field + 1] = param.parameter.modify_time_field
			parameter.column[#parameter.column + 1] = {field = param.parameter.modify_time_field, name = "Modify time", width = lastModifyTimeFieldWidth}
		end
		if parameter.last_modify_time ~= "" then
			query.query = {{param.parameter.modify_time_field, ">", "last_modify_time"}, "and", #query.query == 1 and query.query[1] or query.query}
			-- parameter.show_sql = true
		end
	end
	-- copy query.option keys to option
	-- if type(query.option) == "table" and type(option) == "table" then
	--	 option = util.tableCombine(option, query.option, "no-error")
	-- end

	if query.run then
		if not parameter then
			parameter = {}
		end
		err = execute.runCode(query.run, {parameter = parameter}, nil, origParam) -- or {parameter = param.parameter}?
		if err then
			util.printError(l("error in calling code '%s':  '%s'", tostring(json.toJson(query.run)), err))
		end
	end

	if type(query.map_parameter) == "table" then
		parameter = parameter or {}
		err = execute.runCode(query.map_parameter, parameter, parameter, origParam)
		if err then
			return answerError(l("error in query 'map_parameter': ") .. err)
		end
	end

	if parameter then
		if parameter.show_sql == true then
			if queryJsonName == nil then
				queryJsonName = dprf.preferenceFromJsonName() -- our best guess
			end
			-- io.write(l("- query json '%s'", queryJsonName))
			dsql.showSql(true)
		end
		if type(parameter.selection_id) == "string" and parameter.selection_id ~= "" then
			local sel = selection.loadSelection(parameter.selection_id)
			if sel and sel.record_id then
				parameter.selection = sel.record_id
			else
				return answerError(l("selection id '%s' was not found", parameter.selection_id))
			end
		end
	end
	if not parameter then
		parameter = {}
	end
	parameter.query_name = query.name or queryJsonName -- parameter.query_name = query.name or queryJsonName
	-- get data
	local ret
	-- local restCall = false
	if query.organization_id == nil then
		ret = dqjson.queryJson(query, parameter) -- MUST be dqjson.queryJson(), not local queryJson()
		if ret.error then
			err = ret.error
		else
			err = util.getError()
		end
		--[[ if err then
			local trace
		end ]]
	else
		if auth == nil then
			auth = require "auth"
		end
		local prevOrgId = auth.setCurrentOrganization(query.organization_id)
		dsql.setConnection(query.table, parameter.query_name)
		local conn = dconn.currentConnection()
		if conn == nil then
			err = l("query connection failed, query '%s', organization '%s'", tostring(query.name), auth.currentOrganizationId())
		else
			if conn.driver ~= "rest_call" then
				ret = dqjson.queryJson(query, parameter) -- MUST be dqjson.queryJson(), not local queryJson()
			else
				-- restCall = true
				if callRest == nil then
					callRest = require "call-rest"
				end
				if conn.auth_preference and (type(conn.auth_preference) == "string" or util.tableIsEmpty(conn.auth_preference)) then
					conn.auth_preference = dprf.prf(conn.auth_preference)
				end
				local info = {callCount = 1}
				info.call_time = util.seconds()
				local limit = getLimit(query.parameter.limit)
				if query.parameter and query.parameter.per_page and limit and limit > 0 and limit < query.parameter.per_page then
					query.parameter.per_page = limit
				end
				local result, restParam = callRest.authCall(query, conn)
				resultToArray(result)
				err = result.error
				ret = result.data
				local pageCountHeaderName = query.page_count_header_name or conn.page_count_header_name
				if ret ~= nil and err == nil and restParam.query_parameter then
					if restParam.query_parameter.page and result.header and pageCountHeaderName then
						local totalPageCount = peg.parseBetweenWithoutDelimiter(result.header, pageCountHeaderName .. ":", "\13\n")
						totalPageCount = tonumber(totalPageCount) or 0
						local pageCount = 1
						local limit2 = getLimit(restParam.query_parameter.limit)
						if limit2 and limit2 > 0 then
							if #ret >= limit2 then
								pageCount = totalPageCount -- prevent loop
							end
						end
						while pageCount < totalPageCount do
							pageCount = pageCount + 1 -- this loop starts with page 2
							restParam.query_parameter.page = pageCount
							result = callRest.call(query, restParam, conn)
							resultToArray(result)
							err = result.error
							if err == nil and type(result.data) == "table" then
								local count
								if limit and limit > 0 then
									count = #ret + #result.data
									if count >= limit then
										pageCount = totalPageCount -- end loop
										count = limit - #ret -- count to copy from new result to ret
									end
								end
								util.arrayConcat(ret, result.data, count) -- adds to ret, count can be nil
							else
								pageCount = totalPageCount -- end while loop
							end
						end
					elseif restParam.query_parameter.per_page and restParam.query_parameter.offset and restParam.content and restParam.content.offset and #ret >= restParam.query_parameter.per_page then
						local endLoop
						local loopCount = 0
						restParam.query_parameter.offset = restParam.content.offset + #ret
						repeat
							loopCount = loopCount + 1
							endLoop = true
							restParam.content.offset = restParam.query_parameter.offset
							result = callRest.call(query, restParam, conn)
							resultToArray(result)
							err = result.error
							if err == nil and type(result.data) == "table" then
								resultToArray(result)
								if #result.data >= restParam.query_parameter.per_page then
									endLoop = false
									restParam.query_parameter.offset = restParam.query_parameter.offset + #result.data
								end
								local count
								if limit and limit > 0 then
									count = #ret + #result.data
									if count >= limit then
										endLoop = true
										count = limit - #ret -- count to copy from new result to ret
									end
								end
								util.arrayConcat(ret, result.data, count) -- adds to ret, count can be nil
							end
						until endLoop
					end
				end
				if query.field == nil and parameter.field then
					query.field = parameter.field
				end
				if type(query.field) ~= "table" then
					err = l("rest call error: query.field is not a table")
					ret = {data = {}, info = {}}
				else
					info.call_time = util.seconds(info.call_time)
					info.column_count = #query.field
					info.column_name = query.field
					info.row_count = ret and #ret or 0
					info.row_count_total = info.row_count
					info.organizationId = query.organization_id
					if type(ret) ~= "table" then
						err = l("rest call error: ") .. tostring(err) .. ", " .. tostring(ret)
						ret = {data = {}, info = {}}
					else
						if info.row_count > 0 then
							if ret[1][query.error_tag or conn.error_tag] then -- error_tag == "code" for woocommerce calls
								err = l("rest call error: '%s'", json.toJson(ret))
								ret = {data = {}, info = {}}
							elseif type(ret[1]) == "string" and #ret == 1 and peg.found(ret[1], "html>") then -- error_tag == "code" for woocommerce calls
								err = l("rest call error: 'result is html, not data'")
								ret = {data = {}, info = {}}
							else
								ret = convert.convertExternalDataToLocal(ret, query, restParam, info)
							end
						end
					end
				end
			end
		end
		if prevOrgId then
			auth.setCurrentOrganization(prevOrgId)
		end
	end

	if parameter and parameter.show_sql == true then
		dsql.showSql(false)
	end

	if ret == nil or ret.data == nil or type(ret.data) ~= "table" then -- or util.tableIsEmpty(ret.data) == true then
		if ret == nil then
			ret = {}
		end
		ret.data = {}
	end
	-- if ret.info and ret.info.error then
	-- 	print(l("query json returned error '%s'", ret.info.error))
	-- end
	if type(query["return"]) == "table" then
		for key, val in pairs(query["return"]) do
			ret[key] = val
		end
	end

	if ret and parameter.to_json and ret.data and #ret.data > 0 then
		fn.iter(ret.data):each(function(rec)
			for key, val in pairs(rec) do
				if peg.endsWith(key, ".json_data") then
					rec[key] = json.fromJson(val)
				end
			end
		end)
	end
	-- create structured data if query.structure -tag exist
	if ret and query.structure and ret.data and #ret.data > 0 then
		local tablePrefix = next(ret.data[1]) -- tablePrefix = main data first field, todo: use query main table tag
		tablePrefix = dschema.tablePrefix(tablePrefix)
		local subParam = util.clone(parameter) -- safer to clone, do not change the original
		if #query.structure > 0 then
			local recordIdArr = {}
			fn.iter(query.structure):each(function(subQuery)
				local subTbl = {}
				subTbl[1] = ret.data -- newData[tablePrefix] -- original main data
				local maxDepth = subQuery.depth or 1
				local subQueryName = subQuery.query
				for depth = 1, maxDepth do -- do queries as long as there is data or maxDepth has been reached
					-- set query parameter
					if subQuery.parameter then
						fn.iter(subQuery.parameter):each(function(param2)
							subParam[param2.tag] = nil
							if param2.tag and param2.field then -- later param2.value or something else
								recordIdArr[depth] = util.fieldValueToArray(subTbl[depth], param2.field)
								if #recordIdArr[depth] > 0 then
									subParam[param2.tag] = recordIdArr[depth]
								end
							end
						end)
					end

					-- do query
					subParam.column = subQuery.column
					ret, err, tablePrefix = qry.queryJson(subQueryName, subParam) -- tablePrefix changes to query table
					subTbl[depth + 1] = ret and ret.data -- ret.data may be nil
					if not subTbl[depth + 1] then
						break
					end
				end -- end: for depth = ...

				-- link lower level records to upper level records
				for depth = #subTbl, 2, -1 do
					fn.iter(subTbl[depth]):each(function(rec)
						local parentTbl = subTbl[depth - 1]

						--[[ create row table array for every parent even if it does not contain any sub records
						fn.iter(parentTbl):each(function(parentRec)
							if not parentRec[tablePrefix] then
								parentRec[tablePrefix] = {json.jsonNull()} -- how to force this to be an json array instead of object?
							end
						end)
						--]]
						local parentRec = util.arrayRecord(rec[subQuery.child_field], parentTbl, subQuery.parent_field)
						if not parentRec then
							util.printError(l("parent record was not found, id '%s'", tostring(rec[subQuery.child_field])))
						else
							if not parentRec[tablePrefix] then
								parentRec[tablePrefix] = {}
							end
							parentRec[tablePrefix][#parentRec[tablePrefix] + 1] = rec
						end
					end)
				end
			end)

			-- newData, ret.error = db.recordArraysToStructure(newData)
		end
		ret.table = tablePrefix
	end
	local queryTable
	if type(query.table) ~= "table" then
		util.printError("query.table type is not a 'table', query '%s'", tostring(query.name))
	else
		local rec = query.table and query.table[1]
		if rec then
			queryTable = rec.table
		end
	end
	if queryTable == nil then
		util.printError("query table was not found, query '%s'", tostring(query.name))
		if query.field and query.field[1] then
			queryTable = dschema.tablePrefix(query.field[1])
		end
	end
	ret = ret or {}
	if ret.info == nil then
		ret.info = {}
	end
	if err then
		ret.error = err
	end
	ret.info.query_table = queryTable
	ret.info.query_table_prefix = dschema.tablePrefix(queryTable)
	return ret, err, queryTable -- clean down to 1 ret param
end

function qry.queryRec(query, parameter)
	local rec
	-- local tag = (parameter and parameter.return_tag) or "rec"
	-- option = option or parameter and parameter.option or nil
	local qryRet, err, tbl = qry.queryJson(query, parameter)
	if err then
		return nil, err
	elseif qryRet and qryRet.error then
		return nil, qryRet.error
	elseif qryRet == nil or qryRet.data == nil or type(qryRet.data) ~= "table" or qryRet.data[1] == nil then
		if tbl and parameter and parameter.option and peg.find(parameter.option, "empty-record") > 0 then
			local newRec = dconv.newRecord(tbl, "all-fields")
			if newRec then
				rec = newRec
			else
				return {}
			end
		else
			return {}
		end
	elseif qryRet.data[1] then
		rec = qryRet.data[1]
	else
		return {} -- or nil?, warn of no-data?
	end
	if parameter and parameter.default then -- override default values
		if type(parameter.default) ~= "table" then
			err = l("parameter.default type '%s' is not a table", type(parameter.default))
			util.printError(err)
		else
			for key, val in pairs(parameter.default) do
				rec[key] = val
			end
		end
	end
	-- "script": ["timec.employee_id"]
	if parameter and parameter.script then
		tableCode.runTableCode({table = tbl, data = {rec}, field_name = parameter.script})
		--[[
		local ret = tableCode.runTableCode({rec = rec, table = tbl, field_name = parameter.script})
		if ret.rec then
			rec = ret.rec
		end
		]]
	end
	rec.__keep_value = true
	return {rec = rec, error = err}
end

function qry.queryJsonData(query, parameter)
	local ret = qry.queryJson(query, parameter)
	if ret and ret.data ~= nil and type(ret.data) == "table" then -- and util.tableIsEmpty(ret.data) == false then
		return ret.data, ret and ret.error
	else
		-- info
		-- return answerError(l"parameter field count is smaller than 1")
		return {}, ret and ret.error
	end
end

function qry.queryRecord(table, recordId)
	if table == nil then
		return nil, l("'table' parameter is missing")
	elseif recordId == nil then
		return nil, l("'recordId' parameter is missing")
	end
	local ret
	local recordIdField = dschema.uuidField(table)
	if type(recordId) == "string" then
		dqry.query("", recordIdField, "=", recordId)
		ret = dload.selectionToRecordArray(table)
		ret = ret[1]
	elseif type(recordId) == "table" then
		dqry.query("", recordIdField, "in", recordId)
		ret = dload.selectionToRecordArray(table)
	end
	return ret
end

local function getSessionParameter(param)
	local ret = {}
	if param.parameter then
		-- if type(param.parameter.record_id) == "string" and param.parameter.record_id ~= "" then
		if type(param.parameter.session_id) == "string" and param.parameter.session_id ~= "" then
			local sessionJson = selection.loadSelection(param.parameter.session_id)
			if sessionJson and type(sessionJson.parameter) == "table" then
				sessionJson = util.tableCombine(sessionJson, sessionJson.parameter) -- combine session.parameter to main level
				sessionJson.parameter = nil
			end
			sessionJson.parameter = nil
			--[[if ret then
				ret = util.tableCombine(ret, sessionJson, "no-error")
			else
				ret = sessionJson
			end]]
			ret = sessionJson
			ret.session_record_id = param.parameter.session_id
			if sessionJson and sessionJson.table and sessionJson.record_id and #sessionJson.record_id >= 1 then
				ret[sessionJson.table .. "_record_id"] = sessionJson.record_id[1] -- set first record id
			end
		end
	end
	return ret
end

local function queryGrid(_grid, queryGridParam)
	if type(_grid) ~= "table" and type(_grid) ~= "string" then
		return util.parameterError("grid", "table or string")
	end
	if _grid.grid == nil and type(_grid.name) == "string" then
		_grid = dprf.prf(_grid.name)
	end
	local ret = {name = _grid.name, run = _grid.run, run_after = _grid.run_after}
	local function queryGridRec(grid)
		if grid.grid then
			return fn.util.callArrayParamRecursive(grid.grid, queryGridRec, "table", "no-cache")
		end
		if grid.column == "-" then
			grid.column = {}
		end
		if type(grid.name) ~= "string" then
			return util.parameterError("name", "string")
		elseif grid.name == "" then
			return l("preference '%s' grid name is empty", dprf.preferenceFromJsonName())
		elseif type(grid.query) ~= "string" and type(grid.query) ~= "table" then
			return util.parameterError("query", "string or table")
		elseif grid.query == "" then
			return l("preference '%s' grid query is empty", dprf.preferenceFromJsonName())
		elseif type(grid.column) ~= "string" and type(grid.column) ~= "table" then
			return util.parameterError("column", "string or table")
		elseif grid.column == "" then
			return l("preference '%s' grid column is empty", dprf.preferenceFromJsonName())
		end
		if grid.query == "-" then
			grid.query = ""
		end
		queryGridParam = queryGridParam or {}
		local returnGrid = queryGridParam["return"] and queryGridParam["return"].grid
		replaceRecParameter(grid, queryGridParam.rec or queryGridParam)
		if returnGrid and type(returnGrid) ~= "string" then
			local useGrid = false
			for _, rec in ipairs(returnGrid) do
				if grid.name == rec.name and (grid.tab == nil or grid.tab == rec.tab) then
					useGrid = true
					break
				end
			end
			if useGrid == false then
				return -- exit, do not add to ret[grid.name]
			end
		end
		local column, queryParam, err
		if type(grid.column) == "table" then
			if grid.column.column then
				column = grid.column
			else
				column = {column = grid.column}
			end
		else
			column, err = util.prf(grid.column, "no-cache")
		end
		if type(column) == "table" and column.column then
			-- TODO: fix columns widths to em decimal number
			for _, item in ipairs(column.column) do
				if type(item.value_map) == "string" then
					item.value_map = dprf.prfKey(item.value_map)
				end
			end
			--[[
			if grid.data and #grid.data > 0 and not grid.data[1].id then
				for i, rec in ipairs(grid.data) do
					rec.id = i
				end
			end ]]
		end
		if util.tableIsEmpty(column) then
			return err or l("preference '%s' parameter column '%s' preference was not found", queryGridParam.name or dprf.preferenceFromJsonName(), grid.column)
		end
		if type(grid.query) == "table" then
			queryParam = grid.query
		elseif grid.query == "" then
			queryParam = {empty = true}
		else
			queryParam, err = util.prf(grid.query, "no-cache")
		end
		if util.tableIsEmpty(queryParam) then
			return err or l("preference '%s' parameter query '%s' preference was not found", queryGridParam.name or dprf.preferenceFromJsonName(), grid.query)
		end
		local prevQueryOrgId
		if returnGrid then
			for _, item in ipairs(returnGrid) do
				if queryParam.organization_id == nil and item.organization_id and item.tab == grid.tab and item.name == grid.name then
					prevQueryOrgId = queryParam.organization_id
					queryParam.organization_id = item.organization_id
					break
				end
			end
		end
		local field, loadFieldChange = col.fieldChange(column.column, queryParam.organization_id)
		queryParam.field = field

		local sel
		if queryParam.empty then
			sel = {}
		else
			sel = qry.queryJson(queryParam, queryGridParam)
		end
		if prevQueryOrgId then
			queryParam.organization_id = prevQueryOrgId
		end
		if sel == nil or type(sel) == "string" or sel.error then
			local db = queryParam.database or dconn.database() or ""
			local prfName = queryGridParam.name or dprf.preferenceFromJsonName()
			if sel == nil then
				return l("database '%s', preference '%s' query error: 'query returned nil'", db, prfName)
			end
			if sel.data == nil then
				return l("database '%s', preference '%s' query error: %s", db, prfName, sel.error or sel)
			end
		end
		if loadFieldChange and sel.data and #sel.data > 0 then -- change loaded data field name back to original
			col.convertFieldChange(sel.data, loadFieldChange)
		end
		setColumnOption(column, queryParam)
		ret[grid.name] = {data = sel.data, info = sel.info, column = column.column, option = column.option} -- , data2 = sel.data2
		if sel.error and ret.error == nil then
			ret.error = sel.error
		end
		if grid.run_after then
			err = execute.runCode(grid.run_after, ret, ret) -- err = execute.runCode(runAfter, {data = ret, parameter = param.parameter}, ret)
			if err then
				ret.error = ret.error or util.printRed("error in grid '%s' run after code  when calling code '%s':  '%s'", grid.name, grid.run_after, err)
			end
		end
	end
	local err = fn.util.callArrayParamRecursive(_grid, queryGridRec, "table", "no-cache")
	if type(err) == "string" then
		return {error = err}
	end
	return ret
end

---@return table
function qry.query(param)
	-- db.preferenceNameToDropdownTable()
	local prf, err
	local ret = {}
	if param.name == nil and param.query then
		if type(param.query) == "table" then
			param.name = param.query.name
		else
			param.name = param.query
		end
	end
	if type(param.name) ~= "string" then
		return util.parameterError("name", "string")
	end
	local queryName = tostring(param.name) or tostring(param)
	local sessionParam = getSessionParameter(param)
	if param.parameter then
		param.parameter = util.tableCombine(param.parameter, sessionParam, "no-error")
	end
	if param.query_json then
		param.name, err = json.fromJson(param.query_json)
		-- elseif not peg.endsWith(param.name, ".json") then
		-- param.name = param.name..".json"
	end
	if param.table_query then -- query is given as table, not as parameter name that will be loaded from prf
		prf = param.table_query
	elseif type(param.query) == "table" then -- query is given as table
		prf = param.query
	elseif not param.query then
		prf = queryPreference(param) -- recursively loads all json files, replaces rec fields with param.parameter
		if prf and prf.error then
			return prf
		end
	end
	if prf and util.tableIsEmpty(prf) then
		return err or l("query '%s' was not found", queryName)
	end
	prf = prf or {}
	if prf.json then
		ret.json = prf.json -- preference as json text
	end
	if prf.data then
		ret.data = prf.data -- parse possible data tags json?
	end
	if prf.run then
		-- main level run tag may set values only to data or parameter
		--[[ ret.data = ret.data or {}
		err = execute.runCode(prf.run, {data = ret.data, parameter = param.parameter}, nil, param) -- error here? ]]
		--[[ new working, but do we need this? ]]
		util.printWarning("running prf.run, is this needed?, query '%s'", queryName)
		if ret.data then
			err = execute.runCode(prf.run, {parameter = param.parameter}, ret.data, param) -- err = execute.runCode(prf.run, {data = ret.data, parameter = param.parameter})
		elseif param.parameter.rec then
			err = execute.runCode(prf.run, {parameter = param.parameter}, param.parameter.rec, param)
		else
			util.printError("run tag did not have data or rec to set values to, query '%s'", queryName)
		end
		if err then
			return {error = l("error in query '%s' when calling code '%s':  '%s'", queryName, tostring(json.toJson(prf.run)), err)}
		end
	end

	if ret.hdr == nil then
		if prf.hdr and type(prf.hdr) ~= "table" then
			util.printWarning(l("prf.hdr is not a table ('%s'), query '%s', parameter '%s'", tostring(prf.hdr), tostring(ret.query), tostring(ret.parameter)))
			prf.hdr = nil
		end
		if prf.hdr ~= nil then
			ret.hdr = prf.hdr
			ret.hdr.__keep_value = true
		elseif ret.hdr and ret.rec.hdr ~= nil then -- get ret.hdr from rec.hdr, needed for dynamic hdr name
			ret.hdr = ret.rec.hdr
			ret.hdr.__keep_value = true
		end
	end

	if ret.arr == nil then
		if prf.arr ~= nil then
			ret.arr = prf.arr
		elseif ret.rec and ret.rec.arr ~= nil then -- get ret.arr from rec.arr, needed for dynamic arr name
			ret.arr = ret.rec.arr
		end
	end

	local retParam = param["return"] or prf.rec and prf.rec["return"]
	if retParam then
		local function setReturnParam(var, tag)
			if param.to_json == true then
				prf = json.toJson(prf)
			end
			if var ~= "grid" and ret[var] == nil then
				ret[var] = {}
			end
			if var == "rec" then
				if param.parameter and param.parameter.rec then
					ret.rec = param.parameter.rec
				else
					util.printError("param.return.rec is not supported yet")
				end
			elseif var == "grid" then
				if type(tag) == "table" then -- and prf.rec and prf.rec.grid then
					-- if tag is not a string it may be array like in form/nc/erp-sync/rec.json "grid": [{name:"", tab:""},{}]
					-- later ret.grid = ret.rec.grid when prf.rec has been assigned to ret.rec
					return
				else -- if type(tag) == "string" then
					ret.grid = ret.grid or {}
					ret.grid.name = tag
					--[[ for key, val in pairs(prf) do
						ret.grid[key] = val
					end ]]
					if param.query then -- override with rest call query
						ret.grid.query = param.query
					end
					if param.column then
						ret.grid.column = param.column
					end
					if param.option then
						ret.grid.option = param.option
					end
				end
			elseif var == "json" then
				ret[var].__keep_value = true
				ret[var][tag] = prf
			end
		end
		if type(retParam) == "string" then -- table not supported yet, grid works
			local var, tag
			var = peg.parse(retParam, ".", 1)
			tag = peg.parse(retParam, ".", 2)
			setReturnParam(var, tag)
		else
			for key in pairs(retParam) do -- search for some other key than retParam.grid
				if key ~= "rec" and key ~= "grid" and key ~= "json" then --
					util.printWarning("param.return type '%s' is not supported (rec, grid and json -keys do work)", type(retParam))
					break
				else
					setReturnParam(key, retParam[key])
				end
			end
		end
	end

	if param.rec_field ~= nil then
		ret.rec = {}
		ret.rec[param.rec_field] = json.toJson(prf)
		ret.rec.__keep_value = true
	else
		-- if prf.hdr ~= nil then
		-- do this?
		-- end

		if ret.rec == nil and prf.rec ~= nil then
			ret.rec = prf.rec -- queryPreference(prf.rec, "rec")
		elseif ret.rec == nil and param.rec ~= nil then
			ret.rec = queryPreference(param.rec)
		end
		if ret.rec then
			ret.rec = util.tableCombine(ret.rec, sessionParam, "no-error")
		end
		--[[ done in: queryPreference(param)
		if ret.rec and param.parameter then -- PM: 2017-06-08, add ret.rec to parameters, do not override previous call parameters
			-- param.parameter = util.tableCombine(ret.rec, param.parameter, "no-error")
			ret.rec = util.tableCombine(ret.rec, param.parameter, "no-error")
		end ]]
		if param.query then
			-- ret = qry.queryJson(param.query, param.parameter, param, option)
			ret = qry.queryJson(param.query, param)
		end

		local dataQuery = prf.data_query
		if dataQuery ~= nil then
			if dataQuery and dataQuery.data_query then
				dataQuery = dataQuery.data_query -- json name, containing json with data_query -tag
			end
			if type(dataQuery) ~= "table" then
				ret.error = l("error in query '%s', tag 'data_query' is not an array", queryName)
			else
				local dataParam = util.tableCombine(ret.rec or {}, param.parameter or {}, "no-error")
				local runResult = {parameter = param.parameter} -- or dataParam?
				for _, dataQuery2 in ipairs(dataQuery) do
					if dataQuery2.run then
						if type(dataQuery2.run) ~= "table" then
							ret.error = l("error in query '%s', tag 'data_query' array object 'run' is not a table", queryName)
							break
						end
						if type(ret.rec) ~= "table" then
							ret.error = l("error in query '%s', tag 'rec' is not a table", queryName)
							break
						end
						err = execute.runCode(dataQuery2.run, {data = ret.data, parameter = dataParam}, runResult, param)
						if err then
							ret.error = l("error '%s' in in query '%s', calling code '%s': ", err, queryName, tostring(json.toJson(dataQuery2.run)))
							break
						elseif dataQuery2.tag and runResult[dataQuery2.tag] then -- result tag is rec, arr, ...
							if dataQuery2.tag == "all-keys" then
								util.recToRec(ret, runResult[dataQuery2.tag], {replace = true, deep = true, copy_key = {name = false}})
							else
								ret[dataQuery2.tag] = runResult[dataQuery2.tag]
							end
						end
					elseif type(dataQuery2.tag) ~= "string" then
						ret.error = l("error in query '%s', tag 'data_query' array object does not contain 'tag'", queryName)
						break
					end
					if dataQuery2.query then
						-- data contains query-tag and field-tag (json is already expanded)
						-- todo: this should handle also possible column tag
						if dataQuery2.column then
							dataParam.column = dataQuery2.column -- util.prf(rec.column).column
						end
						local ans
						ans, err = qry.queryJson(dataQuery2.query, dataParam)
						if err or ans == nil or ans.error then
							ret.error = l("database '%s', error in preference '%s', data_query -subquery '%s':\n%s'", tostring(dconn.database()), tostring(queryName), tostring(dataQuery2.query), tostring(err or ans and ans.error or "answer is nil"))
							break
						end
						ret.data = ret.data or {}
						ret.data[dataQuery2.tag] = ans
					end
				end
				-- param.parameter = runResult.parameter -- this is already the same, just for debug
			end
		end

		if ret.rec ~= nil then
			if ret.rec.error then
				ret.error = ret.rec.error -- .."\nrec: "..tostring(prf.rec)}
			end
			if type(ret.rec) ~= "table" then
				ret.error = l("type of ret.rec is not a table, value: '%s'", tostring(ret.rec))
			end
			-- set parameter values to rec
			for key in pairs(ret.rec) do
				if type(ret.rec[key]) == "string" and peg.found(ret.rec[key], "parameter.") then
					ret.rec[key] = execute.getTagValue(ret.rec[key], ret.rec)
				end
			end

			--[[ rec.run has been executed in dprf queryPreferenceRec(), it must be executed before loading arr -tag to get arr tag mustache {{name}} names before loading arr -tag
			if ret.rec.run then
				err = execute.runCode(ret.rec.run, ret.rec, ret.rec, param)
				if err then
					ret.error = l("error in calling code '%s': '%s', query: '%s'", tostring(json.toJson(ret.rec.run)), err, queryName)
				end
			end ]]

			if type(ret.rec["return"]) == "table" then
				for key, val in pairs(ret.rec["return"]) do
					if type(val) == "string" and peg.endsWith(val, ".json") then
						prf, err = queryPreference(val)
						if prf and not err then
							ret.rec[key] = prf
						end
					end
				end
			end

			if ret.rec.query and #ret.rec.query > 0 then
				local function addRetRecQuery(jsonName)
					local qryRet
					qryRet, err = qry.queryJson(jsonName, ret.rec)
					if err and not ret.error then
						ret.error = err
						-- return
					end
					if qryRet and qryRet.data and qryRet.data[1] and qryRet.error == nil then
						ret.rec = util.tableCombine(ret.rec, qryRet.data[1], "no-error")
					end
				end
				if type(ret.rec.query) == "table" then
					for _, jsonName in ipairs(ret.rec.query) do
						addRetRecQuery(jsonName)
					end
				else
					addRetRecQuery(ret.rec.query)
				end
			end

			if ret.rec.__load_json then
				for key, val in pairs(ret.rec.__load_json) do
					local jsonPrf = queryPreference(val)
					if jsonPrf.error then
						ret.error = jsonPrf.error
						break
					end
					ret.rec[key] = json.toJson(jsonPrf)
				end
				ret.rec.__load_json = nil
			end
		end
	end

	if ret.arr ~= nil then
		if ret.arr.error then
			ret.error = ret.arr.error -- .."\ndropdown: "..tostring(prf.arr)}
		elseif type(ret.arr) ~= "table" then
			util.printWarning(l("arr is not a table ('%s'), query '%s', parameter '%s'", tostring(ret.arr), tostring(ret.query), tostring(ret.parameter)))
		else
			-- arr can contain query and it's parameter
			for key, rec in pairs(ret.arr) do
				if type(rec) == "table" and rec.query then
					local content
					content, err = qry.queryJson(rec.query, rec.parameter or rec)
					if err then
						ret.error = err
						-- elseif type(content.data) ~= "table" then
						-- util.printWarning(l("arr is nil, query '%s', parameter '%s'", tostring(rec.query), tostring(rec.parameter)))
					else
						local arr = {}
						for i, rec2 in ipairs(content.data) do
							local show -- show tag can be text or array: ["fld1", " - ", "fld2"]
							if type(rec.show) == "table" then
								show = {}
								for j = 1, #rec.show, 2 do
									show[j] = recData(rec2, rec.show[j])
									if j + 1 <= #rec.show then
										show[j + 1] = rec.show[j + 1] -- add divider
									end
								end
								show = table.concat(show)
							else
								show = recData(rec2, rec.show)
							end
							local dropdown = {show = show}
							for key2, val in pairs(rec) do
								if key2 ~= "query" and key2 ~= "show" then
									recDataSet(dropdown, key2, recData(rec2, val))
								end
							end
							arr[i] = dropdown
						end
						ret.arr[key] = arr
					end
				end
			end
		end
	end

	if ret.grid == nil then
		if prf.grid ~= nil then
			ret.grid = prf.grid
		elseif ret.rec and ret.rec.grid ~= nil then -- get ret.grid from rec.grid, needed for dynamic grid name
			ret.grid = ret.rec.grid
		end
	end
	if ret.grid ~= nil then
		local queryGridParam = util.tableCombine(ret.rec or {}, param.parameter or {}, "no-error") -- new: ret.rec has already param.parameter in it and al {{tag}} has been changed and parameter.xxx has been changed, but data_query run may have set new tags to param.parameter
		--[[old:
		if param.parameter and ret.rec then
			param.parameter overrides keys in ret.rec, no error reporting
			queryGridParam = util.tableCombine(ret.rec, param.parameter, "no-error")
		elseif param.parameter then
			queryGridParam = param.parameter
		elseif ret.rec then
			queryGridParam = ret.rec
		end
		]]
		if param["return"] and not queryGridParam["return"] then
			queryGridParam["return"] = param["return"]
		end
		ret.grid = queryGrid(ret.grid, queryGridParam) -- queryGrid(ret.grid, queryGridParam, param, option)
		if ret.grid.error then
			ret.error = ret.grid.error -- .."\ngrid: "..tostring(prf.grid)}
		end
	end
	if ret.grid then -- add missing data array
		for _, rec in pairs(ret.grid) do
			if type(rec) == "table" and not rec.data then
				rec.data = {}
			end
		end
		if ret.grid.run == nil and prf.run then -- call of grid.json that has run -tag
			ret.grid.run = prf.run
		end
	end
	if ret.grid and ret.grid.run and not ret.grid.error then
		err = execute.runCode(ret.grid.run, ret, ret, param)
		if err then
			ret.error = l("error in calling code '%s': '%s', query: '%s'", tostring(json.toJson(ret.grid.run)), err, queryName)
		end
		--[[ local func, err = execute.getFunction(ret.grid.run)
		if err then
			ret.error = ret.error or err
		else
			local err = func(ret) -- function mutates ret
			if err then
				ret.error = ret.error or err
			end
		end ]]
	end
	local runAfter = prf.run_after or ret.grid and ret.grid.run_after
	if runAfter and not ret.error and not ret.grid.error then
		if ret.rec == nil then
			ret.rec = param.parameter and param.parameter.rec or prf.rec or param.parameter or prf -- ret.grid.run function needs .rec
		end
		err = execute.runCode(runAfter, ret, ret, param) -- err = execute.runCode(runAfter, {data = ret, parameter = param.parameter}, ret)
		if err then
			ret.error = l("error in query '%s' when calling code '%s':  '%s'", queryName, tostring(json.toJson(runAfter)), err)
		end
	end
	return ret
end

function qry.deleteJson(query, parameter)
	local err
	if type(query) == "string" then
		query, err = qry.queryPreference(query) -- , parameter)
	end
	if not query then
		return util.printError("parameter query is nil, error: '%s'", tostring(err))
	end
	local param = util.tableCombine(parameter, {only_query = true})
	local ret = qry.queryJson(query, param) -- we must do query before calling dsave.deleteSelection()
	if ret.error ~= nil then
		util.printError("delete json query error: '%s', query: '%s', parameter: '%s'", tostring(ret.error), query, param)
		return ret.error
	end
	local tbl
	if query.table then
		tbl = query.table
	elseif query.field and #query.field > 0 then
		tbl = dschema.tableName(query.field[1]) -- was: dschema.tablePrefix
	else
		err = l("preference '%s' contains no table or field -tag", tostring(query))
		util.printError(err)
		return err
	end
	if tbl then
		err = dsave.deleteSelection(tbl)
		if err then
			return err
		end
	end
	-- end
end

function qry.changeName(table, queryName)
	local prfTbl, err = util.prf("query/generate/query_name.json")
	if not err then
		if type(prfTbl[table]) == "table" then
			for newName, oldName in pairs(prfTbl[table]) do
				if oldName == queryName then
					newName = peg.replace(newName, ".json", "") .. ".json"
					return table .. "/" .. newName
				end
			end
		end
	end
	return queryName
end

return qry
