-- lib/db/sqlite3-ffi.lua
local ffi = require "ffi"
local bit = require "bit"
local util = require "util"
local lib = util.loadDll("libsqlite3mc") -- ffi.load("sqlite3") --

-- Codes -----------------------------------------------------------------------
local sqlconstants = {} -- SQLITE_* and OPEN_* declarations.
local codes = {
	[0] = "OK",
	"ERROR",
	"INTERNAL",
	"PERM",
	"ABORT",
	"BUSY",
	"LOCKED",
	"NOMEM",
	"READONLY",
	"INTERRUPT",
	"IOERR",
	"CORRUPT",
	"NOTFOUND",
	"FULL",
	"CANTOPEN",
	"PROTOCOL",
	"EMPTY",
	"SCHEMA",
	"TOOBIG",
	"CONSTRAINT",
	"MISMATCH",
	"MISUSE",
	"NOLFS",
	"AUTH",
	"FORMAT",
	"RANGE",
	"NOTADB",
	[100] = "ROW",
	[101] = "DONE"
} -- From 0 to 26.

do
	local types = {"INTEGER", "FLOAT", "TEXT", "BLOB", "NULL"} -- From 1 to 5.

	local opens = {
		READONLY = 0x00000001,
		READWRITE = 0x00000002,
		CREATE = 0x00000004,
		DELETEONCLOSE = 0x00000008,
		EXCLUSIVE = 0x00000010,
		AUTOPROXY = 0x00000020,
		URI = 0x00000040,
		MAIN_DB = 0x00000100,
		TEMP_DB = 0x00000200,
		TRANSIENT_DB = 0x00000400,
		MAIN_JOURNAL = 0x00000800,
		TEMP_JOURNAL = 0x00001000,
		SUBJOURNAL = 0x00002000,
		MASTER_JOURNAL = 0x00004000,
		NOMUTEX = 0x00008000,
		FULLMUTEX = 0x00010000,
		SHAREDCACHE = 0x00020000,
		PRIVATECACHE = 0x00040000,
		WAL = 0x00080000
	}

	local t = sqlconstants
	local pre = "static const int32_t SQLITE_"
	for i = 0, 26 do
		t[#t + 1] = pre .. codes[i] .. "=" .. i .. ";\n"
	end
	for i = 100, 101 do
		t[#t + 1] = pre .. codes[i] .. "=" .. i .. ";\n"
	end
	for i = 1, 5 do
		t[#t + 1] = pre .. types[i] .. "=" .. i .. ";\n"
	end
	pre = pre .. "OPEN_"
	for k, v in pairs(opens) do
		t[#t + 1] = pre .. k .. "=" .. bit.tobit(v) .. ";\n"
	end
end

-- Cdef ------------------------------------------------------------------------
-- SQLITE_*, OPEN_*
ffi.cdef(table.concat(sqlconstants))

-- sqlite3*, sqlite3_*
ffi.cdef [[
// Typedefs.
typedef struct sqlite3 sqlite3;
typedef struct sqlite3_stmt sqlite3_stmt;
typedef void (*sqlite3_destructor_type)(void*);
typedef struct sqlite3_context sqlite3_context;
typedef struct Mem sqlite3_value;

/*
** Specify the key for an encrypted database.
** This routine should be called right after sqlite3_open().
**
** Arguments:
**   db       - Database to be encrypted
**   zDbName  - Name of the database (e.g. "main")
**   pKey     - Passphrase
**   nKey     - Length of passphrase
*/
int sqlite3_key(sqlite3* db, const void* pKey, int nKey);
int sqlite3_key_v2(sqlite3* db, const char* zDbName, const void* pKey, int nKey);

/*
** Change the key on an open database.
** If the current database is not encrypted, this routine will encrypt
** it.  If pNew==0 or nNew==0, the database is decrypted.
**
** Arguments:
**   db       - Database to be encrypted
**   zDbName  - Name of the database (e.g. "main")
**   pKey     - Passphrase
**   nKey     - Length of passphrase
*/
int sqlite3_rekey(sqlite3* db, const void* pKey, int nKey);
int sqlite3_rekey_v2(sqlite3* db, const char* zDbName, const void* pKey, int nKey);

/*
** Specify the activation key for a SEE database.
** Unless activated, none of the SEE routines will work.
**
** Arguments:
**   zPassPhrase  - Activation phrase
**
** Note: Provided only for API compatibility with SEE.
** Encryption support of SQLite3 Multi Cipher is always enabled.
*/
void sqlite3_activate_see(const char* zPassPhrase);

/*
** Define functions for the configuration of the wxSQLite3 encryption extension
*/
int sqlite3mc_config(sqlite3* db, const char* paramName, int newValue);
int sqlite3mc_config_cipher(sqlite3* db, const char* cipherName, const char* paramName, int newValue);
unsigned char* sqlite3mc_codec_data(sqlite3* db, const char* zDbName, const char* paramName);
const char* sqlite3mc_version();

// ---------------------------------------------------

/*
** If a database contains the SQLITE_USER table, then the
** sqlite3_user_authenticate() interface must be invoked with an
** appropriate username and password prior to enable read and write
** access to the database.
**
** Return SQLITE_OK on success or SQLITE_ERROR if the username/password
** combination is incorrect or unknown.
**
** If the SQLITE_USER table is not present in the database file, then
** this interface is a harmless no-op returnning SQLITE_OK.
*/
int sqlite3_user_authenticate(
  sqlite3 *db,           /* The database connection */
  const char *zUsername, /* Username */
  const char *aPW,       /* Password or credentials */
  int nPW                /* Number of bytes in aPW[] */
);

/*
** The sqlite3_user_add() interface can be used (by an admin user only)
** to create a new user.  When called on a no-authentication-required
** database, this routine converts the database into an authentication-
** required database, automatically makes the added user an
** administrator, and logs in the current connection as that user.
** The sqlite3_user_add() interface only works for the "main" database, not
** for any ATTACH-ed databases.  Any call to sqlite3_user_add() by a
** non-admin user results in an error.
*/
int sqlite3_user_add(
  sqlite3 *db,           /* Database connection */
  const char *zUsername, /* Username to be added */
  const char *aPW,       /* Password or credentials */
  int nPW,               /* Number of bytes in aPW[] */
  int isAdmin            /* True to give new user admin privilege */
);

/*
** The sqlite3_user_change() interface can be used to change a users
** login credentials or admin privilege.  Any user can change their own
** login credentials.  Only an admin user can change another users login
** credentials or admin privilege setting.  No user may change their own
** admin privilege setting.
*/
int sqlite3_user_change(
  sqlite3 *db,           /* Database connection */
  const char *zUsername, /* Username to change */
  const char *aPW,       /* New password or credentials */
  int nPW,               /* Number of bytes in aPW[] */
  int isAdmin            /* Modified admin privilege for the user */
);

/*
** The sqlite3_user_delete() interface can be used (by an admin user only)
** to delete a user.  The currently logged-in user cannot be deleted,
** which guarantees that there is always an admin user and hence that
** the database cannot be converted into a no-authentication-required
** database.
*/
int sqlite3_user_delete(
  sqlite3 *db,           /* Database connection */
  const char *zUsername  /* Username to remove */
);

// ---------------------------------------------------

int sqlite3_initialize(void);
int sqlite3_shutdown(void);
// Get informative error message.
const char *sqlite3_errmsg(sqlite3*);

// Connection.
int sqlite3_enable_shared_cache(int);
// int sqlite3_open(const char *filename, sqlite3 **ppDb);
int sqlite3_open_v2(const char *filename, sqlite3 **ppDb, int flags, const char *zVfs);
int sqlite3_close(sqlite3*);
int sqlite3_busy_timeout(sqlite3*, int ms);

// Statement.
int sqlite3_prepare_v2(sqlite3 *conn, const char *zSql, int nByte,
  sqlite3_stmt **ppStmt, const char **pzTail);
int sqlite3_step(sqlite3_stmt*);
int sqlite3_reset(sqlite3_stmt *pStmt);
int sqlite3_finalize(sqlite3_stmt *pStmt);

// Extra functions for SELECT.
int sqlite3_column_count(sqlite3_stmt *pStmt);
const char *sqlite3_column_name(sqlite3_stmt*, int N);
int sqlite3_column_type(sqlite3_stmt*, int iCol);

// Get value from SELECT.
int64_t sqlite3_column_int64(sqlite3_stmt*, int iCol);
double sqlite3_column_double(sqlite3_stmt*, int iCol);
int sqlite3_column_bytes(sqlite3_stmt*, int iCol);
const unsigned char *sqlite3_column_text(sqlite3_stmt*, int iCol);
const void *sqlite3_column_blob(sqlite3_stmt*, int iCol);

// Set value in bind.
int sqlite3_bind_int64(sqlite3_stmt*, int, int64_t);
int sqlite3_bind_double(sqlite3_stmt*, int, double);
int sqlite3_bind_null(sqlite3_stmt*, int);
int sqlite3_bind_text(sqlite3_stmt*, int, const char*, int n, void(*)(void*));
int sqlite3_bind_blob(sqlite3_stmt*, int, const void*, int n, void(*)(void*));

// Clear bindings.
int sqlite3_clear_bindings(sqlite3_stmt*);

// Get value in callbacks.
int sqlite3_value_type(sqlite3_value*);
int64_t sqlite3_value_int64(sqlite3_value*);
double sqlite3_value_double(sqlite3_value*);
int sqlite3_value_bytes(sqlite3_value*);
const unsigned char *sqlite3_value_text(sqlite3_value*); //Not used.
const void *sqlite3_value_blob(sqlite3_value*);

// Set value in callbacks.
void sqlite3_result_error(sqlite3_context*, const char*, int);
void sqlite3_result_int64(sqlite3_context*, int64_t);
void sqlite3_result_double(sqlite3_context*, double);
void sqlite3_result_null(sqlite3_context*);
void sqlite3_result_text(sqlite3_context*, const char*, int, void(*)(void*));
void sqlite3_result_blob(sqlite3_context*, const void*, int, void(*)(void*));

// Persistency of data in callbacks (here just a pointer for tagging).
void *sqlite3_aggregate_context(sqlite3_context*, int nBytes);

// Typedefs for callbacks.
typedef void (*sqlite3_cbstep)(sqlite3_context*,int,sqlite3_value**);
typedef void (*sqlite3_cbfinal)(sqlite3_context*);

// Register callbacks.
int sqlite3_create_function(
  sqlite3 *conn,
  const char *zFunctionName,
  int nArg,
  int eTextRep,
  void *pApp,
  void (*xFunc)(sqlite3_context*,int,sqlite3_value**),
  void (*xStep)(sqlite3_context*,int,sqlite3_value**),
  void (*xFinal)(sqlite3_context*)
);
]]

return {lib = lib, codes = codes}
