-- lib/draw/draw-array.lua
local draw_array = {}

local draw = require "draw/draw" -- see: report.lua
local util = require "util"
local peg = require "peg"
local json = require "json"
local dt = require "dt"
local fn = require "fn"
local dschema = require "dschema"
local execute = require "execute"
local l = require"lang".l
local recData = require"recdata".get
local recDataSet = require"recdata".set -- use set instead of setDot because tags can contain []
local print = util.print
local dbPref, dbPrefName
local tablePrefix, tablePrefixLength
local object, line, pageBreak = draw.object, draw.line, draw.pageBreak
local stringFormat = string.format
local fromJson = json.fromJson
local pegFound = peg.found
local pegReplace = peg.replace
local getFunction = execute.getFunction
local unpack = table.unpack

local function preference(name)
	local pos = peg.find(name, ".")
	if pos < 1 then
		util.printWarning("preference '%s' must contain tag name after dot", name)
		return nil
	end
	if dbPrefName ~= name:sub(1, pos - 1) then
		dbPrefName = name:sub(1, pos - 1) -- cache last pref, same is usually used many times
		dbPref = util.prf(dbPrefName, "no-cache")
	end
	name = name:sub(pos + 1)
	local ret = dbPref[name] or dbPref
	-- use: util.tableDottedNameValue(tbl, name)
	pos = peg.find(name, ".")
	while pos > 0 do
		ret = ret[name:sub(1, pos - 1)]
		if ret == nil then
			break
		end
		name = name:sub(pos + 1)
		ret = ret[name] or ret
		pos = peg.find(name, ".")
	end
	if type(ret) == "table" then -- prf was not found
		return ""
	end
	return ret
end

local function formatDate(value, fmt)
	if value == nil then
		return "" -- "nil" for debug
	elseif type(value) == "table" and (value[1] == "1970-01-01" or value[1] == "0000-00-00") then
		return ""
	elseif value == "1970-01-01" or value == "0000-00-00" then
		return ""
	end
	return type(value) == "table" and dt.formatString(value[1], fmt:sub(2)) or dt.formatString(value, fmt:sub(2))
end

local function formatRecord(calc, rec, value)
	local fmt
	if rec.format_number then -- rec.format_number = {positive, negative, zero}
		if type(value) == "table" and #value > 0 then -- default rec.format is {"%'.2f"} and value has been converted to table {value}
			value = value[1]
		end
		if rec.field_type == "number" then -- tonumber(value) == value then
			if value > 0 or #rec.format_number < 2 then
				fmt = rec.format_number[1]
			elseif value < 0 or #rec.format_number < 3 then
				fmt = rec.format_number[2]
			else -- == 0
				fmt = rec.format_number[3]
			end
		end
	end
	if fmt == nil then
		fmt = rec.format
		if fmt == nil and rec.style and rec.style.format then
			fmt = rec.style.format
		end
		if type(fmt) ~= "table" then
			return stringFormat(value) -- this uses locale (see setlocale), could be replaced by tostring
		elseif #fmt < 1 then
			util.printWarning("format table length is zero, value: '%s'", tostring(value))
			return stringFormat(value) -- this uses locale (see setlocale), could be replaced by tostring
		end
		if type(fmt[1]) ~= "string" then
			util.printWarning("unexpected format type[1] '%s', value: '%s', format '%s'", type(fmt[1]), tostring(value), fmt)
			fmt = tostring(fmt[1])
		else
			fmt = fmt[1]
		end
	end
	if type(fmt) == "string" and fmt:sub(1, 1) == "d" then -- in style: row_data_date{ "format": ["d%d.%m.%y"], ...}
		return formatDate(value, fmt)
	end
	local ret
	local thousandsSep = peg.find(fmt, "%'")
	if thousandsSep > 0 then
		fmt = fmt:sub(1, thousandsSep) .. fmt:sub(thousandsSep + 2) -- get rid of '
	end
	if type(value) == "table" then
		local ok
		ok, ret = pcall(stringFormat, fmt, unpack(value))
		if ok ~= true then
			ret = ""
		end
	elseif value == nil then
		ret = "" -- "nil" for debug
	elseif pegFound(fmt, ".-") then
		ret = stringFormat(pegReplace(fmt, ".-", "."), value)
	elseif type(fmt) == "string" then
		ret = stringFormat(fmt, value)
	else
		util.printWarning("unsupported format type: '%s', format: '%s', value: '%s'", type(fmt), tostring(fmt), tostring(value))
		return value
	end
	if thousandsSep > 0 then
		local decSepChar = ","
		local decSep = peg.find(ret, decSepChar)
		local sep = calc.formulaTotal and calc.formulaTotal.thousands_separator or " " -- " " is Finland and most of europe
		if decSep < 1 then
			decSepChar = "."
			decSep = peg.find(ret, ".")
			-- sep = "," -- us, use locale info from machine or pref
		end
		local decPart, intPart
		if decSep > 0 then
			decPart = ret:sub(decSep)
			intPart = ret:sub(1, decSep - 1)
			if calc.formulaTotal and calc.formulaTotal.decimal_separator and calc.formulaTotal.decimal_separator ~= decSepChar then
				decPart = calc.formulaTotal.decimal_separator .. decPart:sub(2)
			end
		else
			decPart = ""
			intPart = ret
		end
		if #intPart <= 3 then
			ret = intPart .. decPart
		else
			intPart = intPart:reverse()
			local formatted = ""
			for i = 1, #intPart do
				if util.fmod(i, 3) == 0 and intPart:sub(i, i) ~= "-" then
					formatted = formatted .. intPart:sub(i, i) .. sep
				else
					formatted = formatted .. intPart:sub(i, i)
				end
			end
			ret = formatted:reverse() .. decPart
		end
	end
	return ret
end

local function formulaTotalValue(calc, rec, subtotal, fieldName)
	-- This function returns the total value for a formula field based on the subtotal or subtotal-operator
	if rec.subtotal and rec.subtotal_operator and calc.formulaTotal[fieldName] then -- report json contains subtotal
		return calc.formulaTotal[fieldName][rec.subtotal][rec.subtotal_operator]
	end
	if subtotal then
		if calc.breakField[subtotal] and calc.formulaTotal[subtotal][fieldName] and calc.breakField[subtotal][fieldName] then
			return calc.formulaTotal[subtotal][fieldName][calc.breakField[subtotal][fieldName]]
		end
		--[=[ elseif calc.formulaTotal.total[fieldName] then -- TODO: fix, this was not needed before
		return calc.formulaTotal.total[fieldName][calc.breakField.total[fieldName]] ]=]
	end
end

local function getTagValue(tag, data)
	local value
	if type(tag) == "number" then
		return tag
	elseif type(tag) == "boolean" then
		return tag
	elseif type(tag) == "string" then
		value = data[tag]
		if value == nil then
			if pegFound(tag, ".") or pegFound(tag, "[") then
				value = recData(data, tag) -- getTableData(tag, data)
			else
				value = tag
			end
		end
	end
	-- util.print("getTagValue %s, %s, %s", tag, tostring(value), data)
	return value
end

local function fixFieldName(fieldName)
	if tablePrefixLength and fieldName:sub(1, tablePrefixLength) == tablePrefix then
		return fieldName:sub(tablePrefixLength + 1)
	end
	return fieldName
end

local function getArrayData(fieldName, dataArr)
	local ret
	-- util.print("getArrayData %s, size %d, %s", fieldName, #dataArr, dataArr)
	if tablePrefixLength then
		fieldName = fixFieldName(fieldName)
	end
	for _, data in ipairs(dataArr) do
		if pegFound(fieldName, ".") then --  or pegFound(tag, "[")
			ret = getTagValue(fieldName, data)
		else
			ret = data[fieldName]
		end
		if ret ~= nil then
			break
		end
	end
	-- util.print("getArrayData %s, size %d, ret %s", fieldName, #dataArr, tostring(ret))
	return ret
end

local function getData(calc, rec, dataArr, subtotal, fieldName) -- subtotal, fieldName are optional
	local ret
	fieldName = fieldName or rec.id -- default is rec.id
	if calc.formulaTotal then
		ret = formulaTotalValue(calc, rec, subtotal, fieldName) -- see if total value key exists
	end
	if ret == nil then
		ret = getArrayData(fieldName, dataArr) -- normal row data, not subtotal
	end
	return ret
end

local function formatData(calc, rec, value)
	local fmt
	if rec.format_number then -- rec.format_number = {positive, negative, zero}
		if type(value) == "table" and #value > 0 then -- default rec.format is {"%'.2f"} and value has been converted to table {value}
			value = value[1]
		end
		if rec.field_type == "number" then -- tonumber(value) == value then
			if value > 0 or #rec.format_number < 3 then
				fmt = rec.format_number[1]
			elseif value < 0 then
				fmt = rec.format_number[2]
			else -- == 0
				fmt = rec.format_number[3]
			end
		end
	else
		if type(rec.format) ~= "table" or #rec.format < 1 then
			return string.format(value) -- this uses locale (see setlocale), could be replaced by tostring
		end
		fmt = rec.format[1]
		if fmt:sub(1, 1) == "d" then -- in style: row_data_date{ "format": ["d%d.%m.%y"], ...}
			return formatDate(value, fmt)
		end
	end
	local ret
	local thousandsSep = peg.find(fmt, "%'")
	if thousandsSep > 0 then
		fmt = fmt:sub(1, thousandsSep) .. fmt:sub(thousandsSep + 2) -- get rid of '
	end
	if type(value) == "table" then
		local ok
		ok, ret = pcall(string.format, fmt, unpack(value))
		if ok ~= true then
			ret = ""
		end
	else
		ret = string.format(fmt, value)
	end
	if thousandsSep > 0 then
		local decSepChar = ","
		local decSep = peg.find(ret, decSepChar)
		local sep = calc.formulaTotal and calc.formulaTotal.thousands_separator or " " -- " " is Finland and most of europe
		if decSep < 1 then
			decSepChar = "."
			decSep = peg.find(ret, ".")
			-- sep = "," -- us, use locale info from machine or pref
		end
		local decPart, intPart
		if decSep > 0 then
			decPart = ret:sub(decSep)
			intPart = ret:sub(1, decSep - 1)
			if calc.formulaTotal and calc.formulaTotal.decimal_separator and calc.formulaTotal.decimal_separator ~= decSepChar then
				decPart = calc.formulaTotal.decimal_separator .. decPart:sub(2)
			end
		else
			decPart = ""
			intPart = ret
		end
		if #intPart <= 3 then
			ret = intPart .. decPart
		else
			intPart = intPart:reverse()
			local formatted = ""
			for i = 1, #intPart do
				if util.fmod(i, 3) == 0 and intPart:sub(i, i) ~= "-" then
					formatted = formatted .. intPart:sub(i, i) .. sep
				else
					formatted = formatted .. intPart:sub(i, i)
				end
			end
			ret = formatted:reverse() .. decPart
		end
	end
	return ret
end

local function formatRec(calc, rec, amount)
	if not rec.format and not rec.format_number and amount == 0 then
		return "" -- format 0 to ""
	end
	if rec.format or rec.style.format then
		if amount == nil then
			return "" -- "nil" for debug
		end
		return formatRecord(calc, rec, amount)
	end
	if amount then
		return formatData(calc, rec, amount)
	end
end

local function prepareFormulaTotal(pref, formulaTotal, calculateField)
	local function addPrf(break_field, arr)
		-- if operator == "" then
		--	defaultValue = ""
		-- end
		--[[
		if operator == "+" then
			operator = "add"
		elseif operator == "-" then
			operator = "subtract"
		elseif operator == "*" then
			operator = "multiply"
		elseif operator == "/" then
			operator = "divide"
		end
		]]
		for _, item in ipairs(arr) do
			if #item < 2 then
				util.print("\nreport calculate field '%s' is not valid", item)
			else
				local field = item[1]
				local operator = item[2]
				local defaultValue = 0
				if operator == "*" then
					defaultValue = 1
				elseif operator == "/" then
					defaultValue = 1
				end
				if not formulaTotal.break_printed then
					formulaTotal.break_printed = {}
				end
				if not formulaTotal[break_field] then
					formulaTotal[break_field] = {}
				end
				if not formulaTotal[break_field][field] then
					formulaTotal[break_field][field] = {}
				end
				if not calculateField[field] then
					calculateField[field] = {}
				end
				if not calculateField[field][break_field] then
					calculateField[field][break_field] = {}
				end
				formulaTotal[break_field][field][operator] = defaultValue
				calculateField[field][break_field] = formulaTotal[break_field][field]
			end
		end
	end

	if pref.subtotal then
		if pref.subtotal.subtotal then
			for _, rec in ipairs(pref.subtotal.subtotal) do
				if rec.calculate then
					addPrf(rec.field, rec.calculate)
				end
			end
		end
		if pref.subtotal.total and pref.subtotal.total.calculate then
			addPrf("total", pref.subtotal.total.calculate)
		end
	end
end

local calculateCommand = util.invertTable({"", "+", "-", "*", "/"})
local function command(cmd, val, amount)
	if cmd == "formula" then
		return nil
	end
	if type(amount) ~= "number" then
		util.print("calculation, amount '%s' type '%s' is not number", tostring(amount), type(amount))
		return nil
	end
	if cmd == "+" then -- or cmd == "add" then
		if val == nil then
			val = amount
		else
			val = val + amount
		end
	elseif cmd == "-" then -- or cmd == "subtract" then
		if val == nil then
			val = -amount
		else
			val = val - amount
		end
	elseif cmd == "*" then -- or cmd == "multiply" then
		if val == nil then
			val = amount
		else
			val = val * amount
		end
	elseif cmd == "/" then -- or cmd == "divide" then
		if amount ~= 0 then
			if val == nil then
				val = amount
			else
				val = val / amount
			end
		else
			-- util.print("report calculation, division by zero")
			val = 0
		end
	elseif cmd == "" then
		val = amount -- return last row value as it is, may be text
	else
		util.print("report calculation command '%s' is not valid", cmd)
	end
	return val
end

local function fromFunction(functionName, param)
	local func = functionName and getFunction(functionName) -- func = checkAllowedFunction(functionName)
	if func ~= nil then
		local ok, data
		if param and type(param) == "table" and #param > 0 then
			ok, data = pcall(func, unpack(param))
		else
			ok, data = pcall(func)
		end
		if ok == true then
			return data
		end
		local err = l("error in calling function '%s'", functionName)
		util.printError(err)
		return nil, err
	end
	return nil
end

local function calculateTotal(calc, rec, value)
	if calc.formulaTotal and not rec.subtotal and calc.calculateField[rec.id] then -- rec.subtotal may be false?
		for key, operatorRec in pairs(calc.calculateField[rec.id]) do
			for cmd, subtotalValue in pairs(operatorRec) do
				if value == nil then
					value = 0
				end
				if type(value) == "number" and cmd ~= "formula" then
					calc.formulaTotal[key][rec.id][cmd] = command(cmd, subtotalValue, value) -- nil == 0 here
				end
			end
		end
	end
	--[[
	if type(value) ~= "number" then
		-- print(l("report subtotal calculation, id '%s' value type '%s' is not number", tostring(rec.id), type(value)))
		-- this has been (hopefully) reported before
		return
	end
	{
  "subtotal": [
    {
      "field": "pr.product_group",
      "formula": null,
      "page_break": false,
			"calculate": [
				["work_self_cost", "+"],
				["work_self_cost", "max"],
				["material_self_cost", "+"],
				["self_cost", "+"],
				["pro.json_data.sales_price__unit", "+"],
				["sales_price", "+"]
			]
    },
    {
      "field": "pro.production_order_id"
    }
  ],
  "total": {
		"calculate": [
			["work_self_cost", "+"],
			["work_self_cost", "max"],
			["material_self_cost", "+"],
			["self_cost", "+"],
			["pro.json_data.sales_price__unit", "+"],
			["sales_price", "+"]
		]
  }
}
	local break_field = "pr.product_group"
	calc.formulaTotal[break_field] = {}
	calc.formulaTotal[break_field]["work_self_cost"]["+"] = 12
	calc.formulaTotal[break_field]["work_self_cost"]["max"] = 15
	calc.formulaTotal["total"] = {}
	calc.formulaTotal["total"]["work_self_cost]["+"] = 15
	calc.formulaTotal["total"]["work_self_cost]["max"] = 15
	]]
end

local function calculateSubtotal(calc, ret, contentRec, dataArr, subtotal)
	if ret == nil then
		ret = getData(calc, contentRec, dataArr, subtotal)
	end
	if ret ~= nil and contentRec.subtotal ~= false then
		if subtotal == nil then
			calculateTotal(calc, contentRec, ret)
		end
		if dataArr and #dataArr == 1 then
			local id = contentRec.id -- fixFieldName(contentRec.id)
			if pegFound(id, ".") then
				recDataSet(dataArr[1], id, ret)
			elseif dataArr[1][id] ~= ret then
				dataArr[1][id] = ret
			end
		end
	end
	return ret
end

local function calculate(calc, formula, rec, dataArr, subtotal)
	local ret
	local cmd = formula[1]
	if cmd == "replace" then
		local value = getData(calc, rec, dataArr, nil)
		local replaceTbl = formula[2]
		ret = replaceTbl[tostring(value)] -- keys in josn are always strings
		if ret == nil then
			ret = formula[3] or value -- formula[3] is possible else value when replace array does not match
		end
	elseif calculateCommand[cmd] then
		for i = 2, #formula do -- get formula field values
			local field = formula[i]
			local amount
			if util.isNumber(field) then -- is a number
				amount = tonumber(field)
			else
				amount = getData(calc, rec, dataArr, subtotal, field)
			end
			if amount == nil then
				amount = 0 -- nil is same as zero
				if not calc.formulaError.nullAllowed[field] then
					local err = l("id '%s' field '%s' is not valid in formula '%s'", tostring(rec.id), tostring(field), formula)
					if not calc.formulaError[err] then -- print error only once
						calc.formulaError[err] = true
						print(err)
					end
				end
			end
			ret = command(cmd, ret, amount)
		end
	else
		local param = {}
		for i = 2, #formula do
			param[i - 1] = formula[i]
		end
		ret = fromFunction(cmd, param) -- ret, err, but we don't need err
	end
	calculateSubtotal(calc, ret, rec, dataArr, subtotal)
	return ret
end

local function calculateFormula(calc, rec, dataArr, subtotal)
	local ret
	if rec.formula then
		if type(rec.formula) ~= "table" then
			local err = l("report id '%s' formula type '%s' is not a table", tostring(rec.id), type(rec.formula))
			if not calc.formulaError[err] then -- print error only once
				calc.formulaError[err] = true
				print(err)
			end
		else
			if subtotal and calc.calculateField[rec.id] and calc.calculateField[rec.id][subtotal] then
				local cmd = next(calc.calculateField[rec.id][subtotal])
				if cmd ~= "formula" then
					ret = getData(calc, rec, dataArr, subtotal, rec.id) -- TODO: check this
					return ret
				end
			end
			if type(rec.formula[1]) == "table" then
				for _, item in ipairs(rec.formula) do
					ret = calculate(calc, item, rec, dataArr, subtotal)
				end
			else
				ret = calculate(calc, rec.formula, rec, dataArr, subtotal)
			end
		end
	else
		return calculateSubtotal(calc, nil, rec, dataArr, subtotal)
	end
	return ret
end

local function clearBreakField(calc, breakField)
	local breakRec = calc.formulaTotal[breakField]
	for field, rec in pairs(breakRec) do
		local key = next(rec)
		local defaultValue = 0
		if key == "*" then
			defaultValue = 1
		elseif key == "/" then -- separate check for debugging
			defaultValue = 1
		end
		breakRec[field][key] = defaultValue
	end
	calc.formulaTotal.break_printed[breakField] = false
end

local function drawObject(calc, contentRec, dataArr, subtotal)
	local prevText = contentRec.text
	if contentRec.type == "text" then
		if not contentRec.id then
			util.printWarning("id is missing from element '%s'", contentRec)
			-- elseif subtotal and not (calc.breakField[subtotal] and calc.breakField[subtotal][contentRec.id]) then
		elseif subtotal and calc.breakField[subtotal] and calc.breakField[subtotal][contentRec.id] == nil then
			if contentRec.id ~= subtotal then
				return -- skip draw
			end
			-- draw subheader text
			local value = getData(calc, contentRec, dataArr, nil, nil)
			if value then
				if contentRec.format or contentRec.style.format then -- type(amount) == "number" then
					contentRec.text = formatRec(calc, contentRec, value) or value
				else
					contentRec.text = value
				end
				-- contentRec.text = l("Total for: ") .. contentRec.text
			end
		elseif contentRec.preference then
			contentRec.text = preference(contentRec.preference)
		elseif contentRec.subtotal == nil and calc.calculateField[contentRec.id] then
			local amount = calculateFormula(calc, contentRec, dataArr, subtotal)
			if amount == nil then
				contentRec.text = "" -- data does not contain contentRec.id -field, for example dataArr[1].calc or dataArr[1].calc.stock_value is nil
			else
				contentRec.text = formatRec(calc, contentRec, amount)
			end
		elseif contentRec.formula then
			local amount = calculateFormula(calc, contentRec, dataArr, subtotal)
			contentRec.text = formatRec(calc, contentRec, amount)
		elseif contentRec.subtotal == nil and calc.formulaTotal[contentRec.id] then
			local amount = calculateFormula(calc, contentRec, dataArr, subtotal)
			contentRec.text = formatRec(calc, contentRec, amount)
		else
			local txt = calc.pref.language and calc.pref.language[contentRec.id]
			if txt then
				contentRec.text = txt
			else
				local amount = getData(calc, contentRec, dataArr, nil, nil)
				if amount then
					if contentRec.format or contentRec.style.format then -- type(amount) == "number" then
						contentRec.text = formatRec(calc, contentRec, amount)
					else
						--[[ if contentRec.field_type_sql == "date" then
								contentRec.text = dt.localDateStringFormat(amount, calc.pref.style.date_format) -- in style: row_data_date{ "format": ["d%d.%m.%y"], ...}
							else]]
						contentRec.text = amount
						-- end
					end
				elseif contentRec.format then
					if amount == nil and not calc.formulaError.nullAllowed[contentRec.id] then
						util.printWarning("field amount is nil in report, field id '%s'", contentRec.id)
					end
					-- amount = getData(calc, contentRec, dataArr, nil, nil) -- for debug
					contentRec.text = formatRec(calc, contentRec, nil)
				elseif amount == nil and contentRec.id == contentRec.text then
					if not calc.formulaError.nullAllowed[contentRec.id] then
						util.printWarning("field amount is nil in report, field id '%s'", contentRec.id)
					end
					contentRec.text = ""
				end
			end
		end
	end
	if subtotal then
		local fontWeight, fontStyle = contentRec.style["font-weight"], contentRec.style["font-style"]
		contentRec.style["font-weight"] = "bold"
		-- contentRec.style["font-style"] = "underline" -- underline is not supported in Cairo, need to use Pango or something other than cairo
		local y = object(calc.doc, contentRec, calc.rowY)
		if subtotal == "total" and calc.breakField["total"] then
			-- contentRec.style = pref.style.total_text  "oblique"
			if not calc.breakField["total"].total_text_drawn then
				-- draw "Report total" text to the beginning of the row
				calc.breakField["total"].total_text_drawn = true
				local item = util.clone(contentRec)
				item.field_type = "string"
				item.text = l("Report total")
				item.id = item.text
				item.left = 0
				-- item.width = nil
				-- item.style.no_width_missing_error = true
				item.style.format = nil -- util.clone() has cloned style too, we can change it without changing default style
				item.style["margin-right"] = 0
				item.style["text-align"] = "left"
				y = object(calc.doc, item, calc.rowY)
				-- draw total line
				-- y = y + (2 * calc.pref.style.row_data["row-height"]) + calc.pref.style.row_data["font-size"] - 5
				y = y + (calc.pref.style.row_data["row-height"] or 14) - 3 --  + calc.pref.style.row_data["font-size"] - 5
				line(calc.doc, {style = calc.pref.style.total_line}, 0, y, calc.pref.paper.full_width, y)
				-- calc.formulaTotal.break_printed_count = calc.formulaTotal.break_printed_count + 1
				calc.formulaTotal.total_printed = true
			end
		elseif not calc.formulaTotal.break_printed[subtotal] then
			-- draw subtotal line
			-- calc.formulaTotal.break_printed_count = calc.formulaTotal.break_printed_count + 1
			calc.formulaTotal.break_printed[subtotal] = true
			-- y = y + (2 * calc.pref.style.row_data["row-height"]) + calc.pref.style.row_data["font-size"] - 5
			y = y + (calc.pref.style.row_data["row-height"] or 14) - 3 -- + calc.pref.style.row_data["font-size"] - 5
			line(calc.doc, {style = calc.pref.style.subtotal_line}, contentRec.left + calc.pref.paper.left, y, calc.pref.paper.full_width, y) -- + 20
		end
		contentRec.style["font-weight"], contentRec.style["font-style"] = fontWeight, fontStyle
		return y
	end
	if contentRec.text ~= "" then
		local ret = object(calc.doc, contentRec, calc.rowY)
		if prevText ~= contentRec.text then
			contentRec.text = prevText
		end
		return ret
	end
end

local function sortRowData(calc, data)
	-- sort data first by subtotal.subtotal and then by subtotal.order
	if data[1] then
		if calc.doc.subtotal then
			local ordFld = fn.chain(calc.doc.subtotal.subtotal or {}, calc.doc.subtotal.order or {}):reduce(function(acc, rec)
				acc[#acc + 1] = fixFieldName(rec.field)
				acc[#acc + 1] = rec.sort_direction or ">"
				return acc
			end, {})
			-- ordFld = fn.chain(unpack(ordFld)):totable() -- if ordFld was: {{"sort_type", ">"}, {"reff_start_time", "<"}, ...}
			-- intersperse("x", {"a", "b"}) -> {"a", "x", "b", "x"}. https://luafun.github.io/transformations.html
			if ordFld and #ordFld > 0 then
				local sort = require "table/sort"
				-- for _, tbl in ipairs(doc.row_table) do
				sort.sort(data, ordFld) -- sort.sort(data, {"sort_type", ">", "reff_start_time", "<", "reff_end_time", ">"})
				-- end
			end
		end
		if data[1].idx == nil and util.arrayRecord("idx", calc.content.row_data, "id") then
			for i, item in ipairs(data) do -- set index for each row, web page does this normally, but we need it for the report print
				item.idx = i
			end
		end
	end
end

local drawArray, drawArrayOne -- forward declaration of functions
local function drawObjectOrder(calc, excludeKey, content, dataArr, subtotal)
	for _, key in ipairs(content.object_order) do
		if excludeKey == nil or key ~= excludeKey then
			local contentRec = content[key]
			if contentRec == nil then
				util.printWarning("object '%s' was not found from report content", key)
			elseif type(contentRec) == "table" then
				if next(contentRec) == nil then
					util.printWarning("object '%s' is empty in report content", key)
				else
					drawArrayOne(calc, key, contentRec, dataArr, subtotal)
				end
			end
		end
	end
end

local function drawRow(calc, contentRec, data, subtotal, rowNumber)
	calc.rowYMax = 0
	drawArray(calc, contentRec, {data}, subtotal)
	local rowHeightMax = calc.rowYMax - calc.rowYPrev
	if calc.rowY > 0 and rowHeightMax > calc.rowPrf.height then
		calc.lineHeight = calc.rowPrf.height -- rowHeightMax -- variable size row
	else
		calc.lineHeight = calc.rowPrf.height
	end
	--[[ if not calc.formulaTotal.break_printed_count then
				calc.formulaTotal.break_printed_count = 0
			end ]]
	if calc.lineHeight + calc.rowYMax > calc.pageBreakY and rowNumber <= #calc.rowData and calc.formulaTotal.total_printed == false then
		subtotal = nil -- page header did not draw if previous page had subtotal as last line
		pageBreak(calc.doc)
		calc.rowYPrev = 0
		calc.rowY = 0
		calc.pageNumber = calc.pageNumber + 1
		calc.formulaPageNumber[""] = calc.pageNumber
		if calc.content.object_order then -- draw everything else but not "row_data"
			drawObjectOrder(calc, "row_data", calc.content, data, subtotal)
		end
	else
		calc.rowY = calc.rowY + calc.lineHeight
	end
end

drawArrayOne = function(calc, key, contentRec, dataArr, subtotal)
	local pref = calc.pref
	if pref.row_table and #pref.row_table > 0 and type(key) == "string" and peg.find(key, "row_data") == 1 then
		if calc.rowArea[key] then
			calc.rowPrf = calc.rowArea[key]
		else
			calc.rowAreaCount = calc.rowAreaCount + 1
			calc.rowPrf = pref.row and pref.row[calc.rowAreaCount]
			if not calc.rowPrf then
				calc.rowPrf = {}
			end
			calc.rowArea[key] = calc.rowPrf
		end
		if not calc.rowPrf.height then
			calc.rowPrf.height = 14 -- some style font height + x value here?
		end
		-- is row data
		calc.rowDataCount = calc.rowDataCount + 1
		local rowKey = pref.row_table[calc.rowDataCount]
		if rowKey then
			-- print(calc.rowDataCount..". print row data, key: "..key)
			local rowData = dataArr[1] -- dataArr[1] is always main data
			calc.rowYPrev = 0
			calc.rowY = 0
			rowData = rowData[rowKey] or rowData or {}
			local prefix = dschema.tablePrefix(rowKey)
			if prefix then
				tablePrefix = prefix .. "."
				tablePrefixLength = #tablePrefix -- example: ord.
			end

			sortRowData(calc, rowData)
			calc.rowData = rowData
			for rowNumber, data in ipairs(rowData) do
				-- print(calc.rowDataCount, row)
				calc.formulaRowNumber[""] = rowNumber
				drawRow(calc, contentRec, data, nil, rowNumber)
				if calc.breakField then -- #pref.subtotal.subtotal > 0 then
					for i = #pref.subtotal.subtotal, 1, -1 do -- loop backwards because we need to print last subtotals first
						local field = pref.subtotal.subtotal[i].field
						if rowNumber == #rowData then -- total row
							drawRow(calc, contentRec, data, field, rowNumber)
							if i == 1 then
								field = "total"
								drawRow(calc, contentRec, data, field, rowNumber)
							end
						else
							local shortField = fixFieldName(field)
							local needRecData = pegFound(shortField, ".")
							if needRecData then
								if recData(rowData[rowNumber], field) ~= recData(rowData[rowNumber + 1], field) then
									drawRow(calc, contentRec, data, field, rowNumber)
									clearBreakField(calc, field)
								end
							elseif rowData[rowNumber][shortField] ~= rowData[rowNumber + 1][shortField] then
								drawRow(calc, contentRec, data, field, rowNumber)
								clearBreakField(calc, field)
							end
						end
					end
				end
			end
		end
	else
		if contentRec[1] then
			drawArray(calc, contentRec, dataArr, subtotal) -- recursive function
		else
			if contentRec.type == "calculate" then -- just calculate
				calculateFormula(calc, contentRec, dataArr, subtotal)
			else -- we can't exclude anything here because drawObject() will get value from dataArr or calculated (subtotal) value
				local rowHeight = drawObject(calc, contentRec, dataArr, subtotal)
				--[[ if not rowHeight then
					calc.rowYMax = calc.rowYMax + 0 -- for debugging
				elseif ]]
				if rowHeight and rowHeight > calc.rowYMax then
					calc.rowYMax = rowHeight
				end
			end
		end
	end
end

drawArray = function(calc, content, dataArr, subtotal)
	if content.object_order then
		drawObjectOrder(calc, nil, content, dataArr, subtotal)
	else
		for key, contentRec in pairs(content) do
			drawArrayOne(calc, key, contentRec, dataArr, subtotal)
		end
	end
end

function draw_array.draw(pref)
	local calc = {pref = pref}
	calc.rowDataCount = 0
	calc.pageNumber = 1
	calc.breakField = nil
	calc.calculateField = nil
	calc.formulaRowNumber = nil
	calc.formulaPageNumber = nil
	calc.rowYMax = 0
	calc.rowYPrev = 0
	calc.rowY = 0
	calc.pageBreakY = 0
	calc.lineHeight = 0
	calc.rowAreaCount = 0
	calc.rowArea = {}
	calc.formulaError = {}
	calc.page = {}
	calc.rowPrf = nil
	calc.doc = nil

	if pref.data and type(pref.data) ~= "table" then
		pref.data = fromJson(pref.data)
	end
	local mainData = pref.main_table and pref.data[pref.main_table] or {{}}
	tablePrefix, tablePrefixLength = nil, nil
	if pref.row_table and pref.row_table[1] then
		tablePrefix = dschema.tablePrefix(pref.row_table[1])
		if tablePrefix then
			tablePrefix = tablePrefix .. "."
			tablePrefixLength = #tablePrefix -- example: ord.
		end
	end
	if pref.subtotal and pref.subtotal.subtotal then
		calc.breakField = {}
		for _, subtotal in ipairs(pref.subtotal.subtotal) do
			if subtotal.calculate then
				for _, item in ipairs(subtotal.calculate) do
					if not calc.breakField[subtotal.field] then
						calc.breakField[subtotal.field] = {}
					end
					calc.breakField[subtotal.field][item[1]] = item[2]
					-- calc.breakField["pr.product_group"]["work_self_cost"] = "+"
				end
			end
		end
	end
	if pref.subtotal and pref.subtotal.total then
		if calc.breakField == nil then
			calc.breakField = {}
		end
		if pref.subtotal.total.calculate then
			for _, item in ipairs(pref.subtotal.total.calculate) do
				if not calc.breakField["total"] then
					calc.breakField["total"] = {total_text_drawn = false}
				end
				calc.breakField["total"][item[1]] = item[2]
			end
		end
	end

	local locale = os.setlocale()
	if util.isWin() then
		os.setlocale("Finnish_Finland.1252") -- TODO: set current system locale, os.setlocale() returns current locale "C"
	else
		-- https://stackoverflow.com/questions/661935/how-to-detect-current-locale-in-mac-os-x-from-the-shell
		-- defaults read -g AppleLocale
		-- defaults read -g AppleLanguages
		os.setlocale("fi_FI")
	end

	pref.printer_name = pref.printer_name or pref.print and pref.print.printer_name
	local doc = pref.document or draw.newDocument(pref)
	calc.formulaError.nullAllowed = {}
	if pref.null_allowed then
		for _, field in ipairs(pref.null_allowed) do
			calc.formulaError.nullAllowed[field] = true
		end
	end
	-- util.print("mainData: %d, %s", #mainData, mainData)
	-- nore: mainData is array of printed documents array data, if there is more than one row in mainData it will print multiple (pdf) documents or with page break if pref.separate_document is not set
	for i, mainRowData in ipairs(mainData) do
		-- drop include table keys to main level
		-- local dataArr = mainRowData
		local dataArr = {}
		dataArr[#dataArr + 1] = mainRowData
		for _, val in pairs(mainRowData) do
			if type(val) == "table" then
				if #val == 1 then -- there is only 1 record in linked table, or may be row table
					dataArr[#dataArr + 1] = val[1] -- save linked rec for data replace later
				end
			end
		end

		local formulaTotal = {}
		local calculateField = {}
		prepareFormulaTotal(pref, formulaTotal, calculateField)
		formulaTotal.total_printed = false
		formulaTotal.decimal_separator = pref.style and pref.style.decimal_separator -- ",", this passed to execute, goow way to pass some pref too
		formulaTotal.thousands_separator = pref.style and pref.style.thousands_separator -- " "
		-- local rec = {id="rowNumber"}
		local key = ""
		formulaTotal["rowNumber"] = {}
		formulaTotal["rowNumber"][key] = {}
		calc.formulaRowNumber = formulaTotal["rowNumber"][key]

		formulaTotal["pageNumber"] = {}
		formulaTotal["pageNumber"][key] = {}
		calc.formulaPageNumber = formulaTotal["pageNumber"][key]
		calc.formulaPageNumber[""] = calc.pageNumber

		formulaTotal["pageNumberTotal"] = {}
		formulaTotal["pageNumberTotal"][key] = {}
		formulaTotal["pageNumberTotal"][key][""] = 0

		if dataArr and dataArr[1] and pref.row_table and pref.row_table[1] and dataArr[1][pref.row_table[1]] then
			formulaTotal["rowCount"] = {}
			formulaTotal["rowCount"][key] = {}
			formulaTotal["rowCount"][key][""] = #dataArr[1][pref.row_table[1]] -- fix this, may be many row data areas
		end
		calc.formulaTotal = formulaTotal
		calc.calculateField = calculateField
		-- calc.rowDataCount = 0
		-- calculateArray(pref.content, dataArr, subtotal) -- collect here all formula totals and set text values
		calc.rowDataCount = 0
		calc.pageBreakY = pref.content.page_break or draw.height(doc.paper)
		calc.doc = doc
		calc.content = pref.content
		drawArray(calc, pref.content, dataArr, nil)
		if i < #mainData then
			if pref.separate_document then
				calc.page[#calc.page + 1] = draw.toPicture(doc)
				doc = draw.newDocument(pref)
			else
				pageBreak(doc)
			end
		end
	end
	calc.page[#calc.page + 1] = draw.toPicture(doc)
	os.setlocale(locale) -- restore old

	dbPrefName = nil -- TODO: fix: must clear or will use old values from another database
	dbPref = nil
	return calc.page
end

return draw_array

--[[
	local function calculateArray(content, dataArr, subtotal)
		local calc.rowPrf
		for key,rec in pairs(content) do
			if pref.row_table and #pref.row_table > 0 and type(key) == "string" and peg.find(key, "row_data") == 1 then
				if calc.rowArea[key] then
					calc.rowPrf = calc.rowArea[key]
				else
					calc.rowAreaCount = calc.rowAreaCount + 1
					calc.rowPrf = pref.row and pref.row[calc.rowAreaCount]
					if not calc.rowPrf then
						 calc.rowPrf = {}
					else
						--
					end
					calc.rowArea[key] = calc.rowPrf
				end
				-- is row data
				calc.rowDataCount = calc.rowDataCount + 1
				local rowKey = pref.row_table[calc.rowDataCount]
				if rowKey then
					local rowData = dataArr[1]
					rowData = rowData[rowKey] or {} -- dataArr[1] is always main data
					for _,data in ipairs(rowData) do
						calculateArray(rec, {data}, subtotal)
					end
				end
			else
				if rec[1] then
					calculateArray(rec, dataArr, subtotal) -- recursive function
				elseif rec.formula or calc.formulaTotal[rec.id] then
					calculateFormula(calc, rec, dataArr, subtotal)
				end
			end
		end
	end
--]]

--[=[
local function getTableData(id, data, pref)
	-- local validJson = json.validJson
	if id == nil or data == nil or type(id) == "number" then
		return id
	elseif data[id] then
		return data[id]
	end
	local dotPos = peg.find(id, ".")
	-- local posBracket = peg.find(id, "[")
	if dotPos == 0 then
		return nil
	end
	local tagTbl = peg.splitToArray(id, ".")
	local val = data
	for i, key in ipairs(tagTbl) do
		if type(val) == "string" and #val > 0 and i < #tagTbl then
			val = fromJson(val)
		end
		if val == nil then
			return nil
		end
		local keyToNumber = tonumber(key)
		if val[key] == nil and keyToNumber and val[keyToNumber] then
			key = keyToNumber
		end
		if i == 1 and type(val[key]) == "string" and validJson(val[key]) then
			val[key] = fromJson(val[key])
		end
		if i == #tagTbl and pref then
			if val[key] == nil and pref.default_missing_tag then
				return "default_missing_tag"
			elseif pref[val[key]] == nil and pref.default_no_match then
				return "default_no_match"
			end
		end
		val = val[key]
	end
	return val
	--[[
	for key, val in pairs(data) do
		if type(val) == "table" then
			local ret2 = getTableData(id, val) -- recursive call
			if ret2 ~= nil then
				ret = ret2
				break
			end
		elseif key == id then
			ret = val
			break
		end
	end
	return ret
	]]
end
]=]

--[=[
local function formatRec(calc, rec, ret, dataArr, subtotal)
	if rec.format_number then
		ret = getData(calc, rec, dataArr, subtotal)
		return formatData(calc, rec, ret, subtotal)
	end
	if rec.style.format and not rec.format then
		rec.format = rec.style.format
	end
	if type(rec.format) ~= "table" or #rec.format < 1 then
		return nil
	end
	local param = {}
	if #rec.format < 2 then
		if not ret then
			ret = getData(calc, rec, dataArr, subtotal)
		end
		param[#param + 1] = ret -- get field id
	else
		for i = 2, #rec.format do
			--[[ "format": [
				"%'.2f %s",
				"vat_price_total",
				"po_currency_id"
			] --]]
			if i == 2 and ret then
				param[#param + 1] = ret
			else
				local fieldName = rec.format[i]
				local ret = getData(calc, rec, dataArr, subtotal, fieldName)
				if ret then
					param[#param + 1] = ret
				else
					param[#param + 1] = fieldName
				end
			end
		end
	end
	return formatData(calc, rec, param, subtotal)
end
]=]
