--- lib/draw/draw.lua
-- http://cairographics.org/tutorial/#L1tips
-- @module draw
-- local svg = require "EzSVG"
local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local fs = require "fs"
local peg = require "peg"
local json = require "json"
local dprf = require "dprf"
local drawUtil = require "draw/draw-util"
local emfPathToPngPath = require"convert/image-convert".emfPathToPngPath
local drawArray -- = require "draw/draw-array"
local cairo = require "draw/cairo"
local isNull = ffi.isNull
local svg, zint, rsvg, rsvgError
local currentPaper

-- util.print("Cairo draw version: %s", cairo.cairo_version_string())

local refId = 0
local gdi, gdiPlusToken -- win print dll
local scaleX = 1
local scaleY = 1
local defaultFont = "Arial" -- Arial, 'sans-serif'
local defaultFontSize = 14
local wordSeparatorChar, wordSeparatorPattern, maxRowsToFit

local draw = {colorToRgba = drawUtil.colorToRgba, colorToRgbaName = drawUtil.colorToRgbaName, colorToRgbaTable = drawUtil.colorToRgbaTable}

function draw.bottom(paper)
	return paper.bottom
end

function draw.right(paper)
	return paper.right
end

function draw.width(paper)
	return paper.width
end

function draw.height(paper)
	return paper.height
end

function draw.left(paper)
	return paper.left
end

function draw.top(paper)
	return paper.top
end

--[[
local backgroundStyle = {
	fill_color = "white",
	stroke_color = "white",
	stroke_width = 0,
}

local rowYMultiplier = 1.20
local function rowY(style)
	return math.floor(style["font-size"] * rowYMultiplier)
	-- return util.round(style["font-size"] * rowYMultiplier, 0)
end
]]

local function setPageBorder(paper, pageSize)
	if pageSize[3] then
		paper.border_left = pageSize[3]
	end
	if pageSize[4] then
		paper.border_top = pageSize[4]
	end
	if pageSize[5] then
		paper.border_right = pageSize[5]
	end
	if pageSize[5] then
		paper.border_bottom = pageSize[6]
	end
end

function draw.paperWidthHeight(paperType, dpi, paper)
	local fullWidth, fullHeight
	local fullDpi = 1200
	if paperType == "A4" then -- http://www.a4papersize.org/a4-paper-size-in-pixels.php
		-- A4 - 1200 dpi 9921 Pixels	14031 Pixels
		fullWidth = 9921
		fullHeight = 14031
	elseif paperType == "A4 landscape" then
		fullWidth = 14031
		fullHeight = 9921
	elseif paperType == "A5" then -- http://www.a5papersize.org/a5-paper-size-in-pixels.php
		-- A5 - 1200 PPI	7004 Pixels	9916 Pixels
		fullWidth = 7004
		fullHeight = 9916
	elseif paperType == "A5 landscape" then
		fullWidth = 9916
		fullHeight = 7004
	else
		local pageSize = dprf.prf("plugin/vue-renderer/renderer/page-size.json").page_size
		--	"info": "using 72 PPI/DPI, see also https://pixelcalculator.com/en, http://www.a4papersize.org/a4-paper-size-in-pixels.php"
		if pageSize and pageSize[paperType] then
			pageSize = pageSize[paperType]
			fullWidth = pageSize[1]
			fullHeight = pageSize[2]
			setPageBorder(paper, pageSize)
			fullDpi = 72
		elseif pageSize and pageSize[peg.parseBefore(paperType, " landscape")] then
			pageSize = pageSize[peg.parseBefore(paperType, " landscape")]
			fullWidth = pageSize[2]
			fullHeight = pageSize[1]
			setPageBorder(paper, pageSize)
			fullDpi = 72
		else
			util.printError("draw: unknown paper type '%s', please add it to plugin/vue-renderer/renderer/page-size.json", paperType)
			return
		end
	end
	-- print("draw.paperWidthHeight", paperType, dpi)
	fullWidth = fullWidth / (fullDpi / dpi)
	fullHeight = fullHeight / (fullDpi / dpi)
	return fullWidth, fullHeight
end

function draw.setPaper(paper, paperType, docType, dpi)
	---@diagnostic disable-next-line
	if docType == "gl" or docType == "pdf" or docType == "svg" or dpi == nil then
		dpi = 72
	end
	-- dpi = *72
	currentPaper = paper
	paper.paperType = paperType
	paper.docType = docType
	paper.dpi = dpi

	if paperType ~= "custom" then -- custom must contain full_width, full_height
		paper.full_width, paper.full_height = draw.paperWidthHeight(paperType, dpi, paper)
		if paper.full_width == nil then
			return
		end
	end
	-- paper.full_width = util.round(paper.full_width, 2)
	-- paper.full_height = util.round(paper.full_height, 2)
	scaleX = dpi / 72
	scaleY = dpi / 72
	--[[
	print("scaleX: %d, scaleY: %d", scaleX, scaleY)
	]]
	if not paper.border_left then
		paper.border_left = 0
	end
	if not paper.border_top then
		paper.border_top = 0
	end
	if not paper.border_right then
		paper.border_right = 0
	end
	if not paper.border_bottom then
		paper.border_bottom = 0
	end
	paper.left = paper.border_left
	paper.top = paper.border_top
	paper.bottom = paper.full_height - paper.border_bottom
	paper.height = paper.bottom - paper.border_top
	paper.right = paper.full_width - paper.border_right
	paper.width = paper.right - paper.border_left
	-- util.printTable(paper, "draw paper")
	return paper
end

function draw.margin(px)
	return {left = px, top = px, right = px, bottom = px}
end

local function clipRect(ref, style, option, x, y, width, height)
	if ref.document_type == "svg" then
		return svg.clipRect(ref, style, option, x, y, width, height)
	end
	local cr = ref.cr
	if option == "restore" then
		cr:restore()
		return
	elseif option == "set" then
		cr:save() -- set x, y, width, height
	elseif option ~= "default" then
		util.printError("draw: clipRect parameter '%s' is not 'set', 'restore' or 'default'", option)
		return
	end
	cr:rectangle(x, y, width, height)
	cr:clip()
end

local function clipText(ref, style, option, x, y, width, height)
	-- local cr = ref.cr
	if x then
		x = x - 1 --  - 0.5
		y = y - 1 -- - 0.5
		-- width = width - 1 -- 0.5
		-- height = height - 1 -- 0.5
	end
	return clipRect(ref, style, option, x, y, width, height)
end

local function rect(ref, obj, x, y, width, height)
	local style = obj.style
	if ref.document_type == "svg" then
		return svg.rect(ref, obj, x, y, width, height)
	end
	if not style then
		style = ref.style["default-rect"]
	end
	local cr = ref.cr
	-- svgObject = rect(svgGroupRef, horizPos, top, thickness, pixelsDown, 0, 0, "black", "black", borderWidth)
	--[[
		SVG_New_rect (parentSVGObject; x; y; width; height{; roundedX{; roundedY{; strokeColor{; backgroundColor{; strokeWidth}}}}}) -> SVG_Ref

		Parameter	Type		Description
		parentSVGObject	SVG_Ref		Reference of parent element
		x	Number		X of upper left corner
		y	Number		Y of upper left corner
		width	Number		Width of rectangle
		height	Number		Height of rectangle
		roundedX	Number		Horizontal curve
		roundedY	Number		Vertical curve
		strokeColor	String		Color or gradient name
		backgroundColor	String		Color or gradient name
		strokeWidth	Real		Line thickness
		Function result	SVG_Ref		Reference of rectangle
	]]
	-- local roundedX = 0
	-- local roundedY = 0
	if style["border-color"] or style["border-width"] then
		if style["border-color"] then
			local r, g, b, a = drawUtil.colorToRgba(style["border-color"])
			cr:set_source_rgba(r, g, b, a or 1)
		end
		if style["border-width"] and style["border-width"] > 0 then
			local ux = ffi.newNoAnchor("double[1]")
			local uy = ffi.newNoAnchor("double[1]")
			ux[0] = style["border-width"] * scaleX -- scale?
			uy[0] = ux[0]
			cr:device_to_user_distance(ux, uy)
			if ux[0] < uy[0] then
				ux[0] = uy[0]
			end
			cr:set_line_width(ux[0]) --  * scaleX
			-- cr:set_line_width(style["border-width"] * scaleX)
			-- http://cairographics.org/tutorial/#L1tips
			--[[
				When you're working under a uniform scaling transform, you can't just use pixels for the width of your line. However it's easy to translate it with the help of cairo_device_to_user_distance() (assuming that the pixel width is 1):

				double ux=1, uy=1;
				cairo_device_to_user_distance (cr, &ux, &uy);
				if (ux < uy)
						ux = uy;
				cairo_set_line_width (cr, ux);
			]]
		end
		-- http://cairographics.org/samples/set_line_cap/
		-- cr:set_line_cap(0) -- CAIRO_LINE_CAP_BUTT=0, CAIRO_LINE_CAP_ROUND=1, CAIRO_LINE_CAP_SQUARE=2
		cr:rectangle(x, y, width, height)
		cr:stroke()
	end

	if style["color"] then
		cr:rectangle(x, y, width, height)
		local r, g, b, a = drawUtil.colorToRgba(style["color"])
		cr:set_source_rgba(r, g, b, a or 1)
		cr:fill()
		cr:set_source_rgba(0, 0, 0, 1) -- back to black
	end

	return -- svg ref?
end

local function pict(ref, obj, x, y, width, height, fileName, image)
	local style = obj.style
	local filePath = fileName or style.src
	if image == nil then
		if filePath == nil then
			util.printRed("draw: picture path is nil")
			return
		end
		filePath = fs.filePathFix(filePath)
		if not fs.fileExists(filePath) then -- try possible full path first
			if peg.startsWith(filePath, "/img/") then
				filePath = util.mainPath() .. peg.parseAfter(filePath, "/img/")
				filePath = peg.replace(filePath, "/nc-server/", "/nc-preference/")
			else
				filePath = filePath
			end
		end
		if not fs.fileExists(filePath) then
			util.printRed("draw: picture '%s' was not found", filePath)
			return
		end
	end
	--[[ if ref.use_hpdf then
		return pictHpdf(ref, obj, x, y, width, height, filePath)
	end ]]
	if filePath and peg.endsWith(filePath:lower(), ".emf") then
		local ret = emfPathToPngPath(filePath)
		if ret.error then
			-- util.printRed("draw: %s", ret.error) -- emfPathToPngPath() prints errors
			return
		end
		filePath = ret.output
	end
	if ref.document_type == "svg" then
		return svg.pict(ref, obj, x, y, width, height, fileName)
	end
	if not style then
		style = ref.style and ref.style["default-pict"]
	end
	local cr = ref.cr
	clipRect(ref, style, "set", x, y, width, height) -- cr:save()
	if style and (style["border-color"] or style["color"]) then
		rect(ref, obj, x, y, width, height)
	end
	-- print("util.runPath(): "..util.runPath())
	-- add clipping
	-- http://cairographics.org/manual/cairo-PNG-Support.html
	-- use cairo_image_surface_create_from_png_stream () instead of filename
	if not image then
		image = cairo.cairo_image_surface_create_from_png(filePath)
	end
	local w = image:get_image_width()
	local h = image:get_image_height()
	cr:translate(x, y) -- basically same as cr:move_to(x, y)
	if style then
		local scaleW = 1
		local scaleH = 1
		if style["object-fit"] == "contain" then
			scaleW = width / w
			scaleH = height / h
			if scaleW > scaleH then
				scaleW = scaleH
			else
				scaleH = scaleW
			end
		end
		if style["justify-content"] == "center" then
			local scaledWidth = w * scaleW
			local moveX = (width - scaledWidth) / 2
			cr:translate(moveX, 0) -- basically same as cr:move_to(x, y)
		end
		if style["align-self"] == "center" then -- or style["align-items"] == "center" then
			local scaledHeight = h * scaleH
			local moveY = (height - scaledHeight) / 2
			cr:translate(0, moveY) -- basically same as cr:move_to(x, y)
		end
		if style["object-fit"] == "contain" then
			cr:scale(scaleW, scaleH) -- use style pict margin to fit on to box
		end
	end
	cr:set_source_surface(image, 0, 0)
	cr:paint()
	cairo.cairo_surface_destroy(image)
	clipRect(ref, style, "restore")
end

svg = function(ref, obj, x, y, width, height)
	do
		util.printWarning("draw: svg rendering library LibRsvg is not in use, svg pictures are not supported")
		return
	end
	local style = obj.style
	if rsvg == nil then
		rsvg = require "draw/rsvg"
		if type(rsvg) ~= "userdata" then
			util.printRed("draw: svg rendering library load failed")
			return
		end
		rsvg.rsvg_init()
		rsvgError = ffi.newAnchor("GError*[1]") -- create only once and anchor it
	elseif type(rsvg) ~= "userdata" then
		return
	end
	-- local pixelsInEm = 14 * 0.52
	local scale = height / style.height
	local imgWidth = style.width * scale
	local imgHeight = style.height * scale
	-- local data = string.format('<svg xmlns="%s" x="%f" y="%f" width="%f" height="%f" preserveAspectRatio="%s" viewBox="%s">%s</svg>', style.xmlns, x, y, imgWidth, imgHeight, style.preserveAspectRatio, style.viewBox, style.innerHTML)
	-- local data = string.format('<svg xmlns="%s" width="%f" height="%f" preserveAspectRatio="%s" viewBox="%s">%s</svg>', style.xmlns or "http://www.w3.org/2000/svg", imgWidth, imgHeight, style.preserveAspectRatio or "xMidYMid meet", style.viewBox or "0 0 " .. imgWidth .. " " .. imgHeight, style.innerHTML or "")
	-- data = string.format('<svg xmlns="http://www.w3.org/2000/svg" width="%d" height="%d"></svg>', 0, 0, '<circle cx="' .. imgWidth .. '" cy="' .. imgHeight .. '" r="' .. imgHeight / 2 .. '" />')
	-- data = '<svg xmlns="http://www.w3.org/2000/svg" width="48" height="48" viewBox="0 0 48 48"><path d="M42 6H6c-2.21 0-4 1.79-4 4v28c0 2.21 1.79 4 4 4h36c2.21 0 4-1.79 4-4V10c0-2.21-1.79-4-4-4zm0 32H6V10h20v8h16v20z"/></svg>'
	local data = '<svg xmlns="http://www.w3.org/2000/svg" width="8" height="8" viewBox="0 0 8 8"><line x1="0" y1="0" x2="7" y2="7" /></svg>'
	local handle = rsvg.rsvg_handle_new_from_data(ffi.cast("const guint8 *", data), ffi.cast("gsize", #data), rsvgError)
	if isNull(handle) then
		util.printWarning("draw: rsvg_handle_new_from_data failed, error: %s", tostring(rsvgError[0].message))
	end
	local cr = ref.cr
	clipRect(ref, style, "set", x, y, width, imgHeight)
	cr:translate(x, y) -- cr:move_to(x, y)
	local bool = rsvg.rsvg_handle_render_cairo(handle, cr)
	rsvg.rsvg_handle_free(handle)
	clipRect(ref, style, "restore")
	-- util.print('svg: x="%f" y="%f" width="%f" height="%f"', x, y, style.width, style.height)
	if bool ~= 1 then
		util.printWarning("draw: svg rendering failed")
	end
end

local function barcode(ref, obj, x, y, width, height)
	local style = obj.style or {}
	if style.value == nil or style.value == "" then
		return
	end
	if zint == nil then
		zint = require "barcode/zint"
		util.print("Zint barcode version: %s", zint.version())
	end
	local barcodeHeight = tonumber(style["code-height"]) or height
	-- util.print("code-height: %s, code-width %s, height %d, width %d, scaleY %d, dpi %s", tostring(style["code-height"]), tostring(style["code-width"]), height, width, scaleY, tostring(currentPaper.dpi))
	if height < barcodeHeight then
		barcodeHeight = height
	end
	barcodeHeight = barcodeHeight * scaleY
	local scale = tonumber(style["code-width"]) or 1
	if currentPaper.dpi == 72 and scale > 2 then
		scale = scale - 2 -- code-width = pixel width 2 is normal in web page
	else
		scale = scale + math.floor(currentPaper.dpi / 72)
	end
	local barcodeHeight2
	if currentPaper.dpi == 72 then
		barcodeHeight2 = math.floor(barcodeHeight / (300 / 72))
		barcodeHeight2 = barcodeHeight2 - 6 -- TODO: why this formula, why zint makes so tall codes?
	else
		barcodeHeight2 = 25
	end
	y = y + 2
	-- http://zint.org.uk/Manual.aspx?type=p&page=5
	local zintParam = {height = barcodeHeight2, scale = scale, border_width = 6} -- , outfile = "barcode.png"
	-- util.printTable(zintParam, "zintParam")
	local bitmap, bitmapWidth, bitmapHeight = zint.encode(style.value, zintParam)
	if bitmap == nil then
		util.printRed("draw: could not create barcode, model '%s',  value '%s', format '%s'", style.model or "", style.value or "", style.format or "")
		return
	end
	-- see: https://stackoverflow.com/questions/43127823/cairo-draw-an-array-of-pixels
	local image = cairo.cairo_image_surface_create(C.CAIRO_FORMAT_ARGB32, bitmapWidth, bitmapHeight)
	if cairo.cairo_surface_status(image) ~= C.CAIRO_STATUS_SUCCESS then
		util.printError("draw: could not create cairo_image_surface, style: %s", json.toJson(style))
		return
	end
	cairo.cairo_surface_flush(image) -- A call to cairo_surface_flush() is required before accessing the pixel data to ensure that all pending drawing operations are finished
	local imageRow = cairo.cairo_image_surface_get_data(image)
	local stride = cairo.cairo_image_surface_get_stride(image)
	local imageRowPos = ffi.cast('intptr_t', imageRow)
	local mem
	local rows = bitmapHeight - 1
	local cols = bitmapWidth - 1
	local idx = 0
	-- http://wiki.luajit.org/ffi-knowledge
	-- This returns a signed result, which is cheaper to handle and turns into a no-op, when JIT-compiled:
	-- tonumber(ffi.cast('intptr_t', ffi.cast('void *', ptr)))
	-- [[
	for _ = 0, rows do
		mem = ffi.cast("uint32_t *", ffi.cast("intptr_t", imageRowPos))
		for col = 0, cols do
			mem[col] = bitmap[idx]
			idx = idx + 1
		end
		imageRowPos = imageRowPos + stride
	end
	cairo.cairo_surface_mark_dirty(image) -- A call to cairo_surface_mark_dirty() is required after the data is modified.
	-- util.printInfo("barcode width: %d, height: %d, stride: %d", bitmapWidth, bitmapHeight, stride)
	-- util.printInfo("x: %d, y: %d, width: %d, height: %d", x, y, width, height)
	pict(ref, obj, x, y, width, height, nil, image) -- pict() will call cairo.cairo_surface_destroy(image)
end

local function wordSeparatorPreference()
	local pref = util.prf("draw/word_separator.json", "no-cache")
	local ret = pref.char or " "
	if pref and type(pref.char_code) == "table" then
		for _, code in pairs(pref.char_code) do
			if type(code) == "number" and code >= 0 then
				ret = ret .. string.char(code)
			else
				util.printError("draw: preference draw/word_separator.json char_code element is not a positive number")
			end
		end
	end
	wordSeparatorChar, maxRowsToFit = ret, pref.max_rows_to_fit
	wordSeparatorPattern = peg.set(wordSeparatorChar)
	return ret
end
draw.wordSeparatorPreference = wordSeparatorPreference

-- http://www.cairographics.org/manual/cairo-cairo-scaled-font-t.html#cairo-text-extents-t
local textExtents, fontExtents, fontSize -- fontSize is not set on every call
local function text(ref, obj, x, y, txt, width, height)
	if txt == "" then
		return
	end
	local style = obj.style
	if not style then
		style = ref.style and ref.style["default-text"] or {}
	end
	local cr = ref.cr
	local add = 0
	if not txt then
		txt = "--" --  no text data --"
	end
	txt = peg.replace(tostring(txt), "\t", " ") -- tab to space, tabs draw as box in win and linux
	-- txt = peg.replace(txt, "\t", string.char(0xA0)) -- NO-BREAK SPACE, http://www.fileformat.info/info/unicode/char/00a0/index.htm
	local txtArr = peg.splitToArray(txt, "\n")

	if type(style) == "string" then
		style = ref.style[style] -- it is safe to change string to table
	end
	if style and (style["border-color"] or style["color"]) then
		if x and y and width and height then
			rect(ref, obj, x, y, width, height)
		end
	end

	local txtHeight, lineHeightBottom, lineHeight
	local function fitText(origTxtArr)
		local txtArr2 = util.clone(origTxtArr)
		if style then
			-- local fontId = string.format("%s-%s-%s-%s", style["font"] or defaultFont, tostring(style["font-size"] or defaultFontSize), tostring(style["font-weight"]), tostring(style["font-style"]))
			-- if fontId ~= prevFont then -- not a good optimization
			-- 	prevFont = fontId
			if style["font"] or style["font-weight"] or style["font-style"] then
				local fontWeight, fontStyle
				if style["font-weight"] == "bold" then
					fontWeight = C.CAIRO_FONT_WEIGHT_BOLD
				else
					fontWeight = C.CAIRO_FONT_WEIGHT_NORMAL
				end
				if style["font-style"] == "italic" then
					fontStyle = C.CAIRO_FONT_SLANT_ITALIC
				elseif style["font-style"] == "oblique" then
					fontStyle = C.CAIRO_FONT_SLANT_OBLIQUE
				else
					fontStyle = C.CAIRO_FONT_SLANT_NORMAL
				end
				cr:select_font_face(style["font"] or defaultFont, fontStyle, fontWeight)
			end
			if style["font-size"] then
				fontSize = style["font-size"]
				cr:set_font_size(fontSize * scaleX)
			end
			fontExtents = cr:font_extents(fontExtents) -- fontExtents can be nil on first call
			-- end
		end
		if type(txt) ~= "string" then
			txt = tostring(txt)
		end
		textExtents = cr:text_extents(txt, textExtents) -- textExtents can be nil on first call
		txtHeight = fontExtents and fontExtents.height or fontSize or defaultFontSize
		lineHeightBottom = fontExtents.descent
		if style and style["line-height"] then
			lineHeight = style["line-height"]
			if peg.found(lineHeight, "%") then
				lineHeight = tonumber(peg.parseBefore(lineHeight, "%")) / 100
				if lineHeight == nil then
					util.printRed("draw line-height '%s' percent is not a valid number", tostring(style["line-height"]))
				elseif lineHeight ~= 1 then
					txtHeight = txtHeight * lineHeight
					lineHeightBottom = lineHeightBottom * lineHeight
				end
			else
				if peg.found(lineHeight, "px") then
					lineHeight = tonumber(peg.parseBefore(lineHeight, "px"))
				elseif type(lineHeight) == "number" or tostring(tonumber(lineHeight)) == lineHeight then
					lineHeight = tonumber(lineHeight)
				else
					util.printRed("draw line-height '%s' unis is not supported, line-height supports currently only plain numbers, px and %", tostring(style["line-height"]))
					lineHeight = "error"
				end
				if lineHeight == nil then
					util.printRed("draw line-height '%s' is not a valid number", tostring(style["line-height"]))
				elseif lineHeight ~= "error" then
					txtHeight = lineHeight
					lineHeightBottom = 0
				end
			end
		end
		if txtHeight > 32 and fontExtents then
			txtHeight = txtHeight - fontExtents.descent * 0.8
			-- print(txt, fontSize, txtHeight, fontExtents.height, fontExtents.ascent, fontExtents.descent, fontExtents.max_x_advance, fontExtents.max_y_advance)
			-- print(txt, fontSize, txtHeight, fontExtents.height, textExtents.x_bearing, textExtents.x_advance, textExtents.y_bearing, textExtents.y_advance)
			-- txtHeight = fontSize -- fontSize is not set on every call -- fontExtents.height
		end
		-- txtHeight = fontSize or defaultFontSize -- TODO: must be in default style
		-- lineHeightBottom = textExtents.y_advance -- textExtents.y_advance
		-- see: http://cairographics.org/samples/
		if style then
			--[[ if style["align-items"] and style["align-self"] == nil then
				style["align-self"] = style["align-items"]
			end ]]
			if style["align-self"] then -- vertical align
				if not height then
					util.printError("draw: align-self vertical aligned text '%s' does not contain height", txt)
				elseif style["align-self"] == "center" then
					local allLinesHeight = (txtHeight + lineHeightBottom) * #txtArr2 -- - textExtents.y_bearing
					-- if allLinesHeight > height and #txtArr2 > 1 then
					-- 	add = 0 -- cut from the last lines
					-- else
					add = (height - allLinesHeight) / 2
					-- end
				elseif style["align-self"] == "flex-end" then -- bottom
					add = height - (txtHeight + lineHeightBottom) * #txtArr2 -- - textExtents.y_bearing
				elseif style["align-self"] ~= "flex-start" then -- top
					util.printWarning("draw: vertical text align align-self '%s' is not 'flex-start', 'center' or 'flex-end'", tostring(style["align-self"]))
				end
			end
		end

		local i = 0
		local textMaxWidth = 0
		local textMaxWidth2 = 0
		local textMaxWidthRow
		local rowTextExtents, rowTxt
		while i < #txtArr2 do -- we may add new rows to txtArr2 while drawing
			i = i + 1
			rowTxt = txtArr2[i]
			if style then
				if style["text-align"] or style["justify-content"] then -- horizontal align
					if not width then
						-- if style.no_width_missing_error ~= true then
						util.printWarning("draw: horizontal aligned text '%s' does not contain width", rowTxt)
						-- end
					elseif style["text-align"] == "center" or style["justify-content"] == "center" then
						if textExtents.width < width then -- do not move x if text is too wide to fit rect
							x = x + (width - textExtents.width) / 2 -- - textExtents.x_bearing
						end
					elseif style["text-align"] == "right" or style["justify-content"] == "flex-end" then -- right
						x = x + width - textExtents.width - textExtents.x_bearing - 2 -- TODO: -1, -2 or other pixels like 1.3 or 3?
					elseif style["text-align"] == "left" or style["justify-content"] == "flex-start" then -- left
						x = x + 0 -- no move
					else
						util.printWarning("draw: horizontal text-align '%s' is not 'left ', 'center' or 'right' or justify-content '%s' is not 'flex-start', 'center' or 'flex-end'", tostring(style["text-align"]), tostring(style["justify-content"]))
					end
				end
			end
			rowTextExtents = cr:text_extents(rowTxt, rowTextExtents)
			textMaxWidth2 = textMaxWidth
			if rowTextExtents.width > textMaxWidth then
				textMaxWidthRow = i
				textMaxWidth = rowTextExtents.width
			end
			if i == 1 then
				textMaxWidth2 = textMaxWidth
			end
		end

		-- cut only longest line to smaller
		if width and style and style["word-wrap"] and textMaxWidth > width then
			i = textMaxWidthRow
			rowTxt = txtArr2[i]
			textMaxWidth = textMaxWidth2
			if wordSeparatorChar == nil then
				wordSeparatorPreference()
			end
			-- util.printWarning("draw: too long text row: "..rowTxt)
			-- style["word-wrap"] == "break-word" means brake anywhere, not in word separators
			-- there is also "overflow-wrap", but it is not user here
			if wordSeparatorChar ~= "" then
				local newLine = ""
				repeat
					local pos = peg.findFromEnd(rowTxt, wordSeparatorPattern) -- find first separator from end
					if pos > 0 then
						newLine = rowTxt:sub(pos) .. newLine
						rowTxt = rowTxt:sub(1, pos - 1)
						rowTextExtents = cr:text_extents(rowTxt, rowTextExtents)
					end
				until rowTxt == "" or rowTextExtents.width <= width or pos < 1
				if newLine ~= "" and rowTxt ~= "" then
					txtArr2[i] = rowTxt
					rowTextExtents = cr:text_extents(rowTxt, rowTextExtents)
					if rowTextExtents.width > textMaxWidth then
						textMaxWidth = rowTextExtents.width
					end
					--[[ -- remove separator from start of cut line - do not use because it's better to show separator as indent to previous line
					if peg.find(newLine, wordSeparatorPattern) == 1 then
						newLine = newLine:sub(2)
					end ]]
					rowTextExtents = cr:text_extents(newLine, rowTextExtents)
					if rowTextExtents.width > textMaxWidth then
						textMaxWidth = rowTextExtents.width
					end
					table.insert(txtArr2, i + 1, newLine) -- may insert in the middle
				end
			end
		end
		return txtArr2, textMaxWidth, txtHeight
	end

	local doClipText = false
	local textHeight, textMaxWidth
	if style["fit-text"] == nil and style.parent_node and style.parent_node.style["fit-text"] then
		style["fit-text"] = style.parent_node.style["fit-text"] -- util.recToRec(style, style.parent_node.style, {replace = false}) -- todo: is there copy to chid somewhere in the code? if then add to copy list
	end
	if (width == nil or width == 0) and ---@diagnostic disable-next-line
	(height == nil or height == 0) then
		txtArr, textMaxWidth, txtHeight = fitText(txtArr)
	elseif not (style and style["font-size"] and style["fit-text"]) then
		doClipText = true
		clipText(ref, style, "set", x, y, width, height)
		txtArr, textMaxWidth, txtHeight = fitText(txtArr)
	else
		doClipText = true
		clipText(ref, style, "set", x, y, width, height)
		local txtArr2
		local fontSizeOrig = style["font-size"]
		local i = 0
		local textFitsHeight, resizeHeight, textFitsWidth, resizeWidth, resize, prevX
		repeat
			i = i + 1
			prevX = x
			txtArr2, textMaxWidth, txtHeight = fitText(txtArr) -- will calculate txtHeight and longest text in array = textMaxWidth
			if i > 1 and x > prevX then
				x = prevX -- x moves to the right with every fitText() loop call
			end
			textHeight = #txtArr2 * (txtHeight + lineHeightBottom)
			textFitsHeight = textHeight <= height
			resizeHeight = height / textHeight
			textFitsWidth = textMaxWidth <= width
			resizeWidth = width / textMaxWidth
			if i == 1 and textFitsHeight and textFitsWidth and style["fit-text"] == "grow" then
				if resizeWidth < resizeHeight then
					resize = resizeWidth
				else
					resize = resizeHeight
				end
				style["font-size"] = util.round(style["font-size"] * resize, 2)
				textFitsHeight = false -- continue loop
			elseif resizeWidth < resizeHeight then
				resize = resizeWidth
				if maxRowsToFit == nil then
					wordSeparatorPreference()
				end
				if i <= maxRowsToFit and #txtArr2 > #txtArr and style["fit-text"] == "grow" then
					txtArr = txtArr2
					textFitsHeight = false
				end
			else
				resize = resizeHeight
			end
			if textFitsHeight == false or textFitsWidth == false then
				if resize < 1 then
					style["font-size"] = util.round(style["font-size"] * resize, 2)
				end
				if style["font-size"] > 80 then
					style["font-size"] = style["font-size"] - 4
				elseif style["font-size"] > 60 then
					style["font-size"] = style["font-size"] - 3
				elseif style["font-size"] > 40 then
					style["font-size"] = style["font-size"] - 1.5
				else
					style["font-size"] = style["font-size"] - 0.5
				end
			end
		until textFitsHeight and textFitsWidth or style["font-size"] <= 0
		txtArr = txtArr2
		style["font-size"] = fontSizeOrig -- restore font-size
	end
	for _, rowTxt in ipairs(txtArr) do
		add = add + txtHeight
		if ref.document_type == "svg" then
			if style then
				local tmp = style["font-size"]
				style["font-size"] = fontSize
				svg.text(ref, obj, rowTxt, x, y + add)
				style["font-size"] = tmp
			else
				svg.text(ref, obj, rowTxt, x, y + add)
			end
		else
			local radian
			if style["transform"] then -- we only handle rotate for now, other params are used to position text in web page
				local cmdArr = peg.splitToArray(style["transform"], " ")
				local param
				for _, cmd in ipairs(cmdArr) do
					cmd, param = peg.split(cmd, "(")
					if cmd == "rotate" then
						param = peg.parseBefore(param, "deg")
						local angle = tonumber(param)
						if angle and angle ~= 0 then
							radian = angle * math.pi / 180
							break
						end
					end
				end
			end
			if radian then
				local y2 = y + height / 2
				y2 = y2 + textMaxWidth / 2
				cr:move_to(x + width / 2 + add / 2, y2)
				cr:rotate(radian)
			else
				cr:move_to(x, y + add)
			end
			cr:show_text(rowTxt)
		end
		add = add + lineHeightBottom
	end
	if doClipText then
		clipText(ref, style, "restore")
	end
	return add
end

local function line(ref, obj, x, y, x2, y2)
	local style = obj.style
	if ref.document_type == "svg" then
		return svg.line(ref, obj, x, y, x2, y2)
	end
	if not style then
		style = ref.style["default_line"] or {}
	end
	local cr = ref.cr
	-- draw.line(svgGroupRef, 0, pixelsDown, horizontalPos, pixelsDown, "white", borderWidth)
	--[[
		SVG_New_line (parentSVGObject; startX; startY; x2; y2{; color{; strokeWidth}})  SVG_Ref

		Parameter	Type		Description
		parentSVGObject	SVG_Ref		Reference of parent element
		startX	Number		Horizontal start position
		startY	Number		Vertical start position
		x2	Number		Horizontal end position
		y2	Number		Vertical end position
		color	String		Color or gradient name
		strokeWidth	Real		Line thickness
		Function result	SVG_Ref		Reference of line
	]]

	local color = style["border-color"] -- or "black"
	local strokeWidth = style["border-width"] or 1
	if strokeWidth == 0 then
		util.printWarning("draw: stroke width is 0, using 1")
		strokeWidth = 1
	end

	if color then
		local r, g, b, a = drawUtil.colorToRgba(color)
		cr:set_source_rgba(r, g, b, a)
	end
	cr:move_to(x, y)
	cr:line_to(x2, y2)
	if strokeWidth then
		cr:set_line_width(strokeWidth * scaleX)
	end
	cr:stroke()
	cr:set_source_rgba(0, 0, 0, 1) -- back to black
	return -- svg ref?
end
draw.line = line

local function object(ref, obj, rowY)
	local left = obj.left
	local top = obj.top
	local width = obj.width
	local height = obj.height or obj.style and obj.style["row-height"]
	if left == nil or top == nil or width == nil or height == nil then -- or (width == nil and not (obj.style and ob.style.no_width_missing_error))
		util.printWarning("draw.object() position is nil, object: %s", obj) -- for debug
		return
	end
	--[[ if obj.style and obj.style["print-scale"] then -- false and
		-- left = left * obj.style.scale
		-- top = top * obj.style.scale
		width = width * obj.style["print-scale"]
		height = height * obj.style["print-scale"]
	end ]]
	if ref.paper.left then
		left = left + ref.paper.left
	end
	if ref.paper.top then
		top = top + ref.paper.top
	end
	local x = left
	local y = top
	local x2 = left + width
	local y2 = top + height
	if obj.y_offset then
		y = obj.y_offset
	end

	if obj.style then
		if obj.style.margin then
			if obj.style["margin-left"] == nil and obj.style.margin.left then
				obj.style["margin-left"] = obj.style.margin.left
			end
			if obj.style["margin-right"] == nil and obj.style.margin.right then
				obj.style["margin-right"] = obj.style.margin.right
			end
			if obj.style["margin-top"] == nil and obj.style.margin.top then
				obj.style["margin-top"] = obj.style.margin.top
			end
			if obj.style["margin-bottom"] == nil and obj.style.margin.bottom then
				obj.style["margin-bottom"] = obj.style.margin.bottom
			end
		end
		if obj.style["margin-left"] then
			local margin = obj.style["margin-left"] * scaleX
			if margin then
				x = x + margin
				if width then
					width = width - margin
				elseif x2 then
					x2 = x2 - margin
				end
			end
		end
		if obj.style["margin-right"] then
			local margin = obj.style["margin-right"] * scaleX
			if width then
				width = width - margin
			elseif x2 then
				x2 = x2 - margin
			end
		end
		if obj.style["margin-top"] then
			local margin = obj.style["margin-top"] * scaleY
			y = y + margin
			if height then
				height = height - margin
			elseif y2 then
				y2 = y2 - margin
			end
		end
		if obj.style["margin-bottom"] then
			local margin = obj.style["margin-bottom"] * scaleY
			if height then
				height = height - margin
			elseif y2 then
				y2 = y2 - margin
			end
		end
		if obj.style["width"] == "100%" and x2 > ref.paper.right then
			x2 = ref.paper.right
		end
	end

	if rowY and rowY > 0 and obj.y_offset then -- and rowY and rowY > 0 then
		if obj.type == "line" then
			y2 = y
		end
	end

	if y and rowY and rowY > 0 then
		y = y + rowY
	end
	if y2 and rowY and rowY > 0 then
		if obj.type ~= "line" then
			y2 = y2 + rowY
		end
	end
	local addY
	if obj.type == "line" then
		-- x2 = left + x2
		-- y2 = top + y2
		if x2 > ref.paper.right then
			local printObj
			if obj.style and obj.style.parent_node then
				printObj = util.cloneShallow(obj)
				printObj.style.parent_node = nil -- style contains
			else
				printObj = obj
			end
			util.printWarning("draw: position over paper right=%.2f, x=%.2f, object='%s'", ref.paper.right, x2, printObj)
			x2 = ref.paper.right
		end
		if y2 > ref.paper.bottom then
			local printObj
			if obj.style and obj.style.parent_node then
				printObj = util.cloneShallow(obj)
				printObj.style.parent_node = nil -- style contains
			else
				printObj = obj
			end
			util.printWarning("draw: position over paper bottom=%d, y=%d, object='%s'", ref.paper.bottom, y2, printObj)
			y2 = ref.paper.bottom
		end
		line(ref, obj, x, y, x2, y2)
	elseif obj.type == "rect" then
		rect(ref, obj, x, y, width, height)
	elseif obj.type == "pict" then
		pict(ref, obj, x, y, width, height, obj.source)
	elseif obj.type == "svg" then
		svg(ref, obj, x, y, width, height)
	elseif obj.type == "barcode" then
		barcode(ref, obj, x, y, width, height)
	end
	if obj.type == "text" then
		addY = text(ref, obj, x, y, obj.text, width, height)
	elseif obj.text then -- print debug text in addition to line or rect
		text(ref, obj, x, y, obj.text)
	end
	return y, addY
end
-- http://cairographics.org/manual/cairo-PNG-Support.html#cairo-write-func-t

local pictureData = {}
local function writeFunction(id, data, length) -- (closure, data, length)
	id = ffi.string(id)
	if not pictureData[id] then
		pictureData[id] = {}
	end
	-- print("writeFunction: "..length, ffi.string(pictureData, length))
	pictureData[id][#pictureData[id] + 1] = ffi.string(data, length)
	-- TODO: save to file / tcp stream and start new pictureData[id] table
	return C.CAIRO_STATUS_SUCCESS
end
local writeFunctionPointer = ffi.cast("cairo_status_t (*)(void *closure, const unsigned char *pictureData, unsigned int length)", writeFunction)

local function printWindowsBegin(printerName, fileName)
	local err, status --  = 1 -- 1 == no error
	local hdcPrint
	fileName = fileName or "nc-print-file" -- todo: change to come from pref
	if gdi == nil then
		gdi = ffi.load("gdiplus") -- Gdi32.dll --util.loadDll
	end
	if not gdiPlusToken then
		gdiPlusToken = ffi.newNoAnchor("ULONG_PTR[1]") -- ULONG_PTR
	end
	local startup = ffi.newNoAnchor("GdiplusStartupInput")
	startup.GdiplusVersion = 1
	gdi.GdiplusStartup(gdiPlusToken, startup, nil)
	-- local gdiStruct = gdi.GdiplusStartup(gdiPlusToken, startup, nil)
	-- if gdiStruct ~= nil then
	-- special case, it is impossible to know an error because if 3. param is null then will return enum: 0
	-- GdiplusStartup error here
	-- else
	-- printerName = "\\\\printserver\\print1"
	-- print("printWindowsBegin", tostring(printerName), tostring(fileName))
	hdcPrint = C.CreateDCA(nil, printerName, nil, nil)
	if hdcPrint == nil then
		status = 0 -- we dont know the error
		util.printError("draw: Windows printing error code '%d', error in '%s'. Opening windows printer '%s' failed. Please check first that you can print other documents to printer '%s'", status, "CreateDCA", printerName, printerName)
	else
		local docInfo = ffi.newNoAnchor("DOCINFO")
		docInfo.cbSize = ffi.sizeof(docInfo)
		docInfo.lpszDocName = fileName
		status = C.StartDocA(hdcPrint, docInfo)
		if status < 1 then
			err = util.printError("draw: Windows printing error code '%d', error in '%s'. Creating windows print file '%s' failed. Please check first that you can print other documents to printer '%s'", status, "StartDocA", fileName, printerName)
			C.DeleteDC(hdcPrint) -- we need no error here
		else
			status = C.StartPage(hdcPrint)
			if status < 1 then
				err = util.printError("draw: Windows printing error code '%d', error in '%s'. Please check first that you can print other documents to printer '%s'", status, "StartPage", printerName)
				C.EndDoc(hdcPrint) -- we need no error here
			end
		end
	end
	if status < 1 or hdcPrint == nil then
		gdi.GdiplusShutdown(gdiPlusToken) -- returns nothing
		return nil, err
	end
	return hdcPrint
end

local function printWindowsEnd(hdcPrint, printerName)
	-- print has been done before this
	local err
	local status = C.EndPage(hdcPrint)
	if status < 1 then
		err = util.printError("draw: Windows printing error code '%d', error in '%s'. Please check first that you can print other documents to printer '%s'", status, "EndPage", printerName)
	end
	status = C.EndDoc(hdcPrint)
	if status < 1 and not err then
		err = util.printError("draw: Windows printing error code '%d', error in '%s'. Please check first that you can print other documents to printer '%s'", status, "EndDoc", printerName)
	end
	status = C.DeleteDC(hdcPrint)
	if status < 1 and not err then
		err = util.printError("draw: Windows printing error code '%d', error in '%s'. Please check first that you can print other documents to printer '%s'", status, "DeleteDC", printerName)
	end
	-- end
	gdi.GdiplusShutdown(gdiPlusToken) -- returns nothing
	if status < 1 then
		return err or status -- error code
	end
	return nil
end

local function documentType(docType)
	if docType == "print_win" and not util.isWin() then
		docType = "pdf"
	elseif type(docType) == "table" then
		docType = docType.document_type or docType.paper and docType.paper.docType or "pdf"
		-- elseif util.fromEditor() then
		-- docType = "pdf"
	end
	return docType
end

function draw.newDocument(docType, style, fileName, paper) -- , printerName
	local ref
	if type(docType) == "table" then
		ref = docType
		paper = ref.paper
	else
		ref = {style = style, fileName = fileName, paper = paper}
	end
	refId = refId + 1
	ref.refId = tostring(refId)
	local cs, cr
	pictureData[ref.refId] = nil
	local paperWidth = paper.full_width -- / scale
	local paperHeight = paper.full_height -- / scale
	ref.document_type = documentType(docType)
	if ref.document_type == "print_win" and not util.isWin() then
		ref.document_type = "pdf"
	end
	if ref.document_type == "svg" then
		if svg == nil then
			svg = require "svg"
		end
		ref.svg = svg.newDocument(ref)
		-- we still need cairo for text drawing, text extents, font size and so on
		cs = cairo.cairo_svg_surface_create_for_stream(writeFunctionPointer, ffi.cast("void*", ref.refId), paperWidth, paperHeight)
		cairo.cairo_svg_surface_restrict_to_version(cs, C.CAIRO_SVG_VERSION_1_2)
	elseif ref.document_type == "cairo-svg" then
		-- cs = cairo.cairo_svg_surface_create(fileName, w, h)
		cs = cairo.cairo_svg_surface_create_for_stream(writeFunctionPointer, ffi.cast("void*", ref.refId), paperWidth, paperHeight)
		cairo.cairo_svg_surface_restrict_to_version(cs, C.CAIRO_SVG_VERSION_1_2)
		-- http://www.cairographics.org/manual/cairo-SVG-Surfaces.html#cairo-svg-surface-create-for-stream
		--[[x
		cairo_surface_t *
		cairo_svg_surface_create_for_stream(
				cairo_write_func_t write_func,
				void *closure,
				double width_in_points,
				double height_in_points);

		cairo_status_t
		(*cairo_write_func_t) (
				void *closure,
				const unsigned char *pictureData,
				unsigned int length);
		]]

	elseif ref.document_type == "gl" then
		--[[
		cairo_surface_t *
		cairo_gl_surface_create (cairo_device_t *abstract_device, cairo_content_t content, int width, int height);

		cairo_surface_t *
		cairo_gl_surface_create_for_texture (cairo_device_t *abstract_device, cairo_content_t content, unsigned int tex, int width, int height);
		]]
		local device = nil
		local content = nil
		cs = cairo.cairo_gl_surface_create(device, content, paperWidth, paperHeight)
	elseif ref.document_type == "pdf" then
		-- cs = cairo.cairo_pdf_surface_create(fileName, w, h)
		cs = cairo.cairo_pdf_surface_create_for_stream(writeFunctionPointer, ffi.cast("void*", ref.refId), paperWidth, paperHeight)
	elseif ref.document_type == "png" then
		cs = cairo.cairo_image_surface_create(C.CAIRO_FORMAT_ARGB32, paperWidth, paperHeight)
	elseif ref.document_type == "print_win" then
		local hdc = printWindowsBegin(tostring(ref.printer_name), tostring(ref.file_name)) -- hdc, err = ...,returns error, should we return it?
		if hdc ~= nil then
			ref.hdc = hdc
			cs = cairo.cairo_win32_printing_surface_create(hdc) -- cairo.cairo_win32_surface_create_with_dib(C.CAIRO_FORMAT_ARGB32, paperWidth, paperHeight)
		end
	end
	-- local finalScale = 1 / scale
	-- cs:set_device_scale(finalScale, finalScale)
	cr = cairo.cairo_create(cs)
	cr:select_font_face(defaultFont, C.CAIRO_FONT_SLANT_NORMAL, C.CAIRO_FONT_WEIGHT_NORMAL)
	cr:set_font_size(defaultFontSize) --  * scaleX
	-- cr:scale(scaleX, scaleY)
	-- cr:scale(1/scaleX, 1/scaleY)
	ref.cs = cs
	ref.cr = cr
	cr:set_source_rgb(0, 0, 0)
	cr:set_line_width(1.0)
	clipRect(ref, ref.style, "default", ref.paper.left, ref.paper.top, ref.paper.width, ref.paper.height) -- set clip area so that everything outside inner area is clipped

	ref.cs = cs
	ref.cr = cr
	return ref
end

function draw.pageBreak(ref)
	ref.cr:show_page()
end

function draw.toPicture(ref)
	--[[ if ref.use_hpdf then
		return toPictureHpdf(ref)
	end ]]
	local ret
	local cs = ref.cs
	local cr = ref.cr

	--[[
	local left = ref.paper.width - ref.paper.border_right
	local top = 0
	local width = ref.paper.border_right
	local height = ref.paper.height
	-- rect(ref, obj, left, top, width, height, backgroundStyle) -- draw white rect over right border to cut text and other drawing

	left = 0
	top = ref.paper.height - ref.paper.border_bottom
	width = ref.paper.width
	height = ref.paper.border_bottom
	-- rect(ref, obj, left, top, width, height, backgroundStyle) -- draw white rect over bottom border to cut text and other drawing
	]]
	-- cr:scale(1/scale, 1/scale)
	-- local finalScale = 1 / scale
	-- cs:set_device_scale(finalScale, finalScale)
	if ref.document_type == "svg" then
		ret = svg.draw(ref)
	else
		cr:show_page()
	end
	if ref.document_type == "png" then
		if ref.fileName then
			cairo.cairo_surface_write_to_png(cs, ref.fileName)
		else
			cairo.cairo_surface_write_to_png_stream(cs, writeFunctionPointer, ffi.cast("void*", ref.refId))
			--[[
			local ihdr = pictureData[3] -- PNG IHDR data
			local width1, width2, width3, width4 = ihdr:byte(1, 4)
			local height1, height2, height3, height4 = ihdr:byte(5, 8)
			local lshift = bit.lshift
			local rshift = bit.rshift
			local band = bit.band
			local c = string.char
			local width = lshift(width1, 24) + lshift(width2, 16) + lshift(width3, 8) + width4
			local height = lshift(height1, 24) + lshift(height2, 16) + lshift(height3, 8) + height4
			width = width / scaleX
			-- width = math.floor(width)
			height = height / scaleY
			-- height = math.floor(width)
			-- http://www.libpng.org/pub/png/book/chapter11.html#png.ch11.div.8
			-- 600 dpi is equal to 23,622.05 pixels per meter, so both the x and y values would be 23,622, and the unit specifier would be 1.
			width = 23622
			height = 23622
			local unit = 1
			width1, width2, width3, width4 = band(rshift(width, 24), 0xff), band(rshift(width, 16), 0xff), band(rshift(width, 8), 0xff), band(width, 0xff)
			height1, height2, height3, height4 = band(rshift(height, 24), 0xff), band(rshift(height, 16), 0xff), band(rshift(height, 8), 0xff), band(height, 0xff)
			table.insert(pictureData, 8, c(0)..c(0)..c(0)..c(9).."pHYs")
			table.insert(pictureData, 9, c(width1)..c(width2)..c(width3)..c(width4)..c(height1)..c(height2)..c(height3)..c(height4)..c(unit))
			print(width, height)
			--]]
		end
	end

	if cs then
		cairo.cairo_surface_destroy(cs)
	end
	if cr then
		cairo.cairo_destroy(cr)
	end

	ref.cs = nil
	ref.cr = nil
	if ref.hdc then -- windows print
		printWindowsEnd(ref.hdc, tostring(ref.printer_name)) -- returns error, should we return it?
	end
	if pictureData[ref.refId] and ref.document_type ~= "svg" then
		ret = table.concat(pictureData[ref.refId])
	end
	pictureData[ref.refId] = nil
	return ret
end

draw.object = object
-- draw.rect = rect
-- draw.text = text
-- draw.pict = pict

function draw.drawArray(pref)
	drawArray = require "draw/draw-array"
	return drawArray.draw(pref)
end

return draw

--[[ function draw.documentDrawArea(ref)
	return ref.cr
end ]]
