--- dt.lua (date_time)
-- see: https://stackoverflow.com/questions/26370688/convert-a-julian-date-to-regular-date-in-javascript/36073807
-- https://www.sqlite.org/quirks.html, uses julian
-- @module dt
-- http://www.cs.tufts.edu/~nr/drop/lua/cal.lua
-- http://stackoverflow.com/questions/4460902/unix-time-to-humanreadable-date-and-time
-- local ma = require "lib/ma_lib"
-- local date = require "date"
local dt = {}
local ffi = require "mffi"
local C = ffi.C
local util = require "util"
local peg = require "peg"
local l = require"lang".l
local osDate = os.date
local osTime = os.time
local strFormat = string.format
local fmod, floor, abs, modf = util.fmod, math.floor, math.abs, math.modf
local timeParse, toString, daysAdd -- forward declarations

if util.isWin() then
	C = ffi.loadMsvcr()
end

--[[
local bit = require "bit"
local lshift = bit.lshift
local rshift = bit.rshift
local bits = 17
local bitNum = 0x1FFFF
]]

--[[
local function gettimeofday()
  local t = ffi.newNoAnchor("timeval")
  ffi.C.gettimeofday(t, nil)
  return tonumber(t.tv_sec)
end

print("time of day is " .. gettimeofday())
]]

--[[
local function dateStringFormat(txtDate,dateFormat)
	--txtDate: "yyyy-mm-dd"
	--dateFormat:"dd.mm.yyyy"
	local year,month,day = dateParseToYmd(txtDate)
	if year == 0 and month == 0 and day == 0 then
		return txtDate
	end
	dateFormat = dateFormat:lower()
	dateFormat = peg.replace(dateFormat, 'yyyy', "yy")
	dateFormat = peg.replace(dateFormat, "yy", tostring(year))
	dateFormat = peg.replace(dateFormat, "mm", tostring(month))
	dateFormat = peg.replace(dateFormat, "dd", tostring(day))
	return dateFormat
end
dt.dateStringFormat = dateStringFormat
]]

local secondsInDay = 24 * 60 * 60 -- = 86400
local daysInMonthArr = {31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}
local monthTable = {{31, 28, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}, {31, 29, 31, 30, 31, 30, 31, 31, 30, 31, 30, 31}} -- days in each month
local dayNameShort = {"Mon", "Tue", "Wed", "Thu", "Fri", "Sat", "Sun"}
local dayName = {"Monday", "Tuesday", "Wednesday", "Thursday", "Friday", "Saturday", "Sunday"}

--[[
local function localeConv()
	-- os.setlocale("fi_FI")
	local lc = C.localeconv()
	return {
		decimal_point = ffi.string(lc.decimal_point),
		thousands_sep = ffi.string(lc.thousands_sep),
		grouping = ffi.string(lc.grouping),
		int_curr_symbol = ffi.string(lc.int_curr_symbol),
		currency_symbol = ffi.string(lc.currency_symbol),
		mon_decimal_point = ffi.string(lc.mon_decimal_point),
		mon_thousands_sep = ffi.string(lc.mon_thousands_sep),
		mon_grouping = ffi.string(lc.mon_grouping),
		positive_sign = ffi.string(lc.positive_sign),
		negative_sign = ffi.string(lc.negative_sign),
		int_frac_digits = lc.int_frac_digits,
		frac_digits = lc.frac_digits,
		p_cs_precedes = lc.p_cs_precedes,
		p_sep_by_space = lc.p_sep_by_space,
		n_cs_precedes = lc.n_cs_precedes,
		n_sep_by_space = lc.n_sep_by_space,
		p_sign_posn = lc.p_sign_posn,
		n_sign_posn = lc.n_sign_posn
	}
end ]]

--[[
-- returns the day of week integer and the name of the weekday
-- Compatible with Lua 5.0 and 5.1.
-- from sam_lie
local function dayOfWeek(yy, mm, dd)
	if mm == nil then
		if type(yy) == "string" then
			yy, mm, dd = dateParseToYmd(yy)
		elseif type(yy) == "number" then
			yy, mm, dd = numToYmd(yy)
		end
	end
	if type(yy) == "string" and type(mm) == "string" and type(dd) == "string" then
		yy = tonumber(yy)
		mm = tonumber(mm)
		dd = tonumber(dd)
	end
	local mmx = mm
	if (mm == 1) then
		mmx = 13;
		yy = yy - 1
	end
	if (mm == 2) then
		mmx = 14;
		yy = yy - 1
	end
	local val8 = dd + (mmx * 2) + floor(((mmx + 1) * 3) / 5) + yy + floor(yy / 4) - floor(yy / 100) + floor(yy / 400) + 2
	local val9 = floor(val8 / 7)
	local dw = val8 - (val9 * 7)
	if dw == 0 then
		dw = 7
	end
	return dw, toWeekdayShortName(dw)
end
dt.dayOfWeek = dayOfWeek
 ]]
local function dayOfWeekSun0Sat6(year, month, day)
	if month < 3 then
		month = month + 12
		year = year - 1
	end
	local weekDay = (day + floor((13 * (month + 1)) / 5) + year + floor(year / 4) - floor(year / 100) + floor(year / 400)) % 7
	if weekDay == 0 then
		return 6
	end
	return weekDay - 1
end
dt.dayOfWeekSun0Sat6 = dayOfWeekSun0Sat6

local function daylightSavingsTime(year, month, day, hour)
	-- http://stackoverflow.com/questions/5590429/calculating-daylight-saving-time-from-only-date
	-- Calculate day of week for Daylight savings time.
	local weekDay = dayOfWeekSun0Sat6(year, month, day)
	if month > 3 and month < 11 or month == 3 and day > 21 and weekDay == 0 and hour >= 3 or -- DST starts last Sunday of March;  3am
	month == 11 and day <= 21 and weekDay > 0 or month == 11 and day <= 21 and weekDay == 0 and hour < 3 -- DST ends last Sunday of November; 3am
	then
		return true
	end
	return false
end
dt.daylightSavingsTime = daylightSavingsTime

local function isLeapYear(y)
	return ((y % 4) == 0) or (((y % 100) == 0) and ((y % 400) == 0)) == true
end
dt.isLeapYear = isLeapYear

local function leapYearIndex(y)
	if isLeapYear(y) then
		return 2
	else
		return 1
	end
end

local bufSize = 4096 -- max return size of format return
local buf = ffi.newNoAnchor("char[?]", bufSize)
local time_c
if util.isWin() and ffi.arch == "x64" then
	time_c = ffi.newAnchor("__time64_t[1]")
else
	time_c = ffi.newAnchor("time_t[1]")
end

local tm, len
local function formatNum(num, format)
	format = format or "%Y-%m-%d %H:%M:%S"
	time_c[0] = num
	if util.isWin() then
		if ffi.arch == "x64" then
			tm = C._localtime64(time_c)
		else
			tm = C._localtime32(time_c)
		end
	else
		tm = C.localtime(time_c)
	end
	len = C.strftime(buf, bufSize, format, tm)
	return ffi.string(buf, len)
end
dt.formatNum = formatNum

local function htmlDate()
	return os.date("!%a, %d %b %Y %H:%M:%S %Z") -- "Fri, 02 Nov 2018 02:49:15 UTC"
end
dt.htmlDate = htmlDate

--[=[
 -- this is slower than htmlDate()
local htmlDateFormat = "%a, %d %b %Y %H:%M:%S %Z"
ffi.cdef[[
time_t time(time_t *timer);
struct tm *gmtime(const time_t *timer);
]]
local function htmlDate2()
  time_c[0] = 0
  time_c[0] = C.time(time_c) -- time_t now = time(0);
  tm = C.gmtime(time_c) -- struct tm tm = *gmtime(&now);
	len = C.strftime(buf, bufSize, htmlDateFormat, tm)
	return ffi.string(buf, len)
end
dt.htmlDate2 = htmlDate2
]=]

local function round(val, decimal)
	if decimal then
		return floor((val * 10 ^ decimal) + 0.5) / (10 ^ decimal)
	else
		return floor(val + 0.5)
	end
end

local function idiv(a, b)
	-- return (a - fmod(a, b)) / b -- (a - a % b) / b, not safe with negative numbers
	return floor(a / b)
end

local unixDateInJulian = 2440588 -- 2440587.5 is correct, but our calculation needs non-decimal number -- ymdToJulian(1970, 1, 1)

local function julianToNum(julian)
	return (julian - unixDateInJulian) * secondsInDay
end
dt.julianToNum = julianToNum

local function numToJulian(num)
	return (num / secondsInDay) + unixDateInJulian
end
dt.numToJulian = numToJulian

local ymdToJulian
do
	local a, y, m
	ymdToJulian = function(year, month, day)
		-- see: https://craftofcoding.files.wordpress.com/2013/07/cs_langjuliandates.pdf
		if not year or not month or not day then
			util.printWarning("ymdToJulian -function year, month or day parameter is nil")
			return 0
		end
		-- local a = idiv((14 - month), 12)
		a = floor((14 - month) / 12)
		y = year + 4800 - a
		m = month + 12 * a - 3
		-- local num = day + idiv((153 * m + 2), 5) + 365 * y + idiv(y, 4) - idiv(y, 100) + idiv(y, 400) - 32045
		return day + floor((153 * m + 2) / 5) + 365 * y + floor(y / 4) - floor(y / 100) + floor(y / 400) - 32045
	end
end
dt.ymdToJulian = ymdToJulian

local function ymdToNum(year, month, day)
	if year == 0 and month == 0 and day == 0 then -- we have agreed that 00.00.00 is 1970.01.01
		return 0
	end
	return julianToNum(ymdToJulian(year, month, day))
end
dt.ymdToNum = ymdToNum

--[[
local defaultTimeZoneOffset = timeZoneOffset(osTime({year = 2000, month = 1, day = 1, hour = 12})) -- no DaylightSavings in january
local function isDaylightSavings(dt)
	local offset = timeZoneOffset(osTime(dt))
	if offset ~= defaultTimeZoneOffset then
		return true
	end
	return false
end
local function ymdToNum1(y, m, d)
	local dt = {year = y, month = m, day = d, hour = 0, min = 0, sec = 0} -- MUST contain hour=0
	-- dt.isdst = isDaylightSavings(dt)
	local num = osTime(dt)
	-- num = num + timeZoneOffset(num)
	-- local txt = toString(num)
	return num
end
]]

local function numToUnix(num)
	return round(num * secondsInDay, 0)
end
dt.numToUnix = numToUnix

local julianToYmd
do
	local a, b, c, d, e, m, day, month, year
	julianToYmd = function(num)
		a = num + 32044
		-- local b = idiv((4 * a + 3), 146097)
		-- local c = a - idiv((146097 * b), 4)
		-- local d = idiv((4 * c + 3), 1461)
		-- local e = c - idiv((1461 * d), 4)
		-- local m = idiv((5 * e + 2), 153)
		-- local day = e - idiv((153 * m + 2), 5) + 1
		-- local month = m + 3 - 12 * idiv(m, 10)
		-- local year = 100 * b + d - 4800 + idiv(m, 10)
		b = floor((4 * a + 3) / 146097)
		c = a - floor((146097 * b) / 4)
		d = floor((4 * c + 3) / 1461)
		e = c - floor((1461 * d) / 4)
		m = floor((5 * e + 2) / 153)
		day = e - floor((153 * m + 2) / 5) + 1
		month = m + 3 - 12 * floor(m / 10)
		year = 100 * b + d - 4800 + floor(m / 10)
		return year, month, day
	end
end

local function hmsAdd(num, h, m, s)
	if not num or not h or not m or not s then
		util.printWarning("hmsAdd -function num, h, m or s parameter is nil")
		return 0
	end
	return num + (h * 3600) + (m * 60) + s
end
dt.hmsAdd = hmsAdd

local function removeTimePart(num)
	if num and type(num) ~= "number" then
		util.printError("removeTimePart -function num parameter is not a number")
		return num
	end
	local secs = num % secondsInDay
	return num - secs
end
dt.removeTimePart = removeTimePart

local function numToYmd(num)
	num = removeTimePart(num)
	return julianToYmd(numToJulian(num))
end
dt.numToYmd = numToYmd

local julianToWeek
do
	local d4, L, d1
	julianToWeek = function(num)
		d4 = fmod(fmod(fmod((num + 31741 - fmod(num, 7)), 146097), 36524), 1461) -- (num + 31741 - (num % 7)) % 146097 % 36524 % 1461
		-- local L = idiv(d4, 1460)
		L = floor(d4 / 1460)
		d1 = fmod((d4 - L), 365) + L -- ((d4 - L) % 365) + L
		-- local numberOfWeek = idiv(d1, 7) + 1
		return floor(d1 / 7) + 1 -- return numberOfWeek
	end
end
--[[
local function timeZoneOffset(num)
	local utcdate = osDate("!*t", num)
	local localdate = osDate("*t", num)
	localdate.isdst = false -- this is the trick
	local offset = os.difftime(osTime(localdate), osTime(utcdate))
	return offset
end ]]

local function currentDate()
	local d = osDate("*t")
	return ymdToNum(d.year, d.month, d.day)
end
dt.currentDate = currentDate

local function currentCentury()
	local d = osDate("*t")
	return floor(d.year / 100) * 100
end
dt.currentCentury = currentCentury

local function currentYear()
	return numToYmd(currentDate())
end
dt.currentYear = currentYear

local function timeZoneOffset(num)
	if num <= 0 then
		return 0
	end
	local utcdate = osDate("!*t", num)
	local localdate = osDate("*t", num)
	localdate.isdst = false -- this is the trick
	return osTime(localdate) - osTime(utcdate) -- offset in seconds
end
dt.timeZoneOffset = timeZoneOffset

local function hoursToNum(hours)
	if type(hours) == "string" then
		hours = timeParse(hours)
	end
	return hours * 3600
end
dt.hoursToNum = hoursToNum

local function minutesToNum(minutes)
	if type(minutes) == "string" then
		minutes = timeParse(minutes)
	end
	return minutes * 60
end
dt.minutesToNum = minutesToNum

local function secondsToNum(seconds)
	if type(seconds) == "string" then
		seconds = timeParse(seconds)
	end
	return seconds
end
dt.secondsToNum = secondsToNum

local function hoursToMillisecond(hour)
	return hour * 3600000 -- 1000 ms * 60*60 secs in hour
end
dt.hoursToMillisecond = hoursToMillisecond

local function ymdsToNum(y, m, d, s) -- we have agreed that 00.00.00 is 1970.01.01
	if y == 0 and m == 0 and d == 0 then
		return 0
	end
	return ymdToNum(y, m, d) + secondsToNum(s)
end
dt.ymdsToNum = ymdsToNum

local function numToTable(num)
	local y, m, d = numToYmd(num)
	return {y = y, m = m, d = d}
end
dt.numToTable = numToTable

local toWeek
do
	local year
	toWeek = function(num)
		year = numToYmd(num) -- year, month, day = numToYmd(num)
		return year * 100 + julianToWeek(numToJulian(num))
	end
end
dt.toWeek = toWeek

local weekToYmd
do
	local year, week, month, day, monthIndex, yearFirst, jumpYear, offset
	weekToYmd = function(weeknum, offsetFromMonday) -- returns monday of the week
		if weeknum <= 0 then
			return 0, 0, 0
		end
		offset = offsetFromMonday or 0
		year = idiv(weeknum, 100)
		month = 0
		week = weeknum - (year * 100)
		--[[ -- don't tolerate bad code
	if (year < 1000) then
		if (year < 60) then
			year = year + 2000
		else
			year = year + 1900
		end
	end
	]]
		day = 1 --  monday
		monthIndex = year - 1
		yearFirst = (monthIndex + idiv(monthIndex, 4) - idiv(monthIndex, 100) + idiv(monthIndex, 400) + 1) % 7
		if yearFirst == 0 then
			yearFirst = 7
		elseif yearFirst >= 1 and yearFirst <= 4 then
			week = week - 1
		end
		day = day + (7 * week) - yearFirst + 1 + offset -- offset from monday
		if fmod(year, 100) ~= 0 then
			jumpYear = fmod(year, 4) == 0
		else
			jumpYear = fmod(year, 400) == 0
		end
		for monthNum = 1, 12 do
			local daysInMonth = daysInMonthArr[monthNum]
			if jumpYear and monthNum == 2 then
				daysInMonth = daysInMonth + 1
			end
			if day <= daysInMonth then
				month = monthNum
				break
			else
				day = day - daysInMonth
			end
		end
		if month == 0 then
			month = 1
			year = year + 1
		end
		return year, month, day
	end
end
dt.weekToYmd = weekToYmd

local ymdToWeekdayNum
do
	local yearLastDigit, yearFirstDigit, dayNum -- we do not want to create garbage on every call
	ymdToWeekdayNum = function(year, month, day) -- returns 0-6 as Saturday - Friday
		-- https://www.tutorialspoint.com/day-of-the-week-in-cplusplus
		if month <= 2 then -- for march to december month code is same as month
			month = 12 + month -- for Jan and Feb, month code will be 13 and 14
			year = year - 1 -- decrease year for month Jan and Feb
		end
		yearLastDigit = year % 100 -- last two digit
		yearFirstDigit = idiv(year, 100) -- first two digit
		dayNum = (day + floor((13 * (month + 1)) / 5) + yearLastDigit + floor(yearLastDigit / 4) + floor(yearFirstDigit / 4) + (5 * yearFirstDigit)) % 7
		if dayNum <= 1 then
			return 6 + dayNum -- 0, 1 to 6, 7 Saturday-Sunday
		end
		return dayNum - 1 -- 2-6 to 1-5 Monday-Friday
	end
end
dt.ymdToWeekdayNum = ymdToWeekdayNum

local function weekToYmdFriday(weeknum) -- returns friday of the week - usually end of the week is needed
	return weekToYmd(weeknum, 4)
end
dt.weekToYmdFriday = weekToYmdFriday

local function weekToNum(weeknum, offsetFromMonday)
	return ymdToNum(weekToYmd(weeknum, offsetFromMonday))
end
dt.weekToNum = weekToNum

local function weekToNumFriday(weeknum)
	return ymdToNum(weekToYmdFriday(weeknum))
end
dt.weekToNumFriday = weekToNumFriday

local function weekDaynumToNum(week, dayNum) -- dayNum 1 = monday, 7 = sunday, week in format 201423
	-- local offsetFromMonday = dayNum - 1
	return weekToNum(week, dayNum - 1)
end
dt.weekDaynumToNum = weekDaynumToNum

-- local dateMatchPattern = "(%d+)%p(%d+)%p(%d+)"
-- "(%d%d%d%d)%p(%d%d)%p(%d%d)"--"(%d+)%p(%d+)%p(%d+)" -- %p: represents all punctuation characters.
local year, month, week, day
local function dateParseToYmd(txtDate)
	if txtDate == nil or txtDate == "" then
		return 0, 0, 0
	end
	year = tonumber(txtDate:sub(1, 4)) -- :sub is faster than match
	month = tonumber(txtDate:sub(6, 7))
	day = tonumber(txtDate:sub(9, 10))
	return year, month, day
end
dt.dateParseToYmd = dateParseToYmd

local function dateParseToNum(txtDate) -- usel peg here to parse any delim.
	if type(txtDate) == "number" then
		return txtDate
	end
	if txtDate == nil or type(txtDate) ~= "string" or #txtDate < 10 then -- bad fix
		local err = l("date parse invalid date '%s'", tostring(txtDate))
		util.printWarning(err)
		return currentDate(), err
	end
	year = tonumber(txtDate:sub(1, 4))
	month = tonumber(txtDate:sub(6, 7))
	day = tonumber(txtDate:sub(9, 10))
	if year == 0 and month == 0 and day == 0 then
		return 0
	end
	if year == 0 or month == 0 or day == 0 then
		local err = l("date parse invalid date '%s'", txtDate)
		util.printWarning(err)
		return currentDate(), err
	end
	return ymdToNum(year, month, day)
end
dt.dateParseToNum = dateParseToNum

local function dateParseToUnix(txtDate)
	return dateParseToNum(txtDate) * 1000
end
dt.dateParseToUnix = dateParseToUnix

local function dateParse(txtDate)
	--[[if not txtDate then
		ma.err("txtDate is nil")
		return 0
	end]]
	if txtDate == "" then
		return 0
	end
	year, month, day = dateParseToYmd(txtDate)
	if year == 0 and month == 0 and day == 0 then
		return 0
	end
	return ymdToNum(year, month, day)
end
dt.dateParse = dateParse

local hour, min, sec
timeParse = function(txtTime)
	hour = tonumber(txtTime:sub(1, 2))
	min = tonumber(txtTime:sub(4, 5))
	sec = tonumber(txtTime:sub(7, 8))
	if hour == nil then
		hour = 0
	end
	if min == nil then
		min = 0
	end
	if sec == nil then
		sec = 0
	end
	return hmsAdd(0, hour, min, sec)
end
dt.timeParse = timeParse

local function removeDatePart(num)
	return num % secondsInDay
end
dt.removeDatePart = removeDatePart

local function isDateString(str)
	if type(str) == "string" then
		return #str == 10
	end
	return false
end
dt.isDateString = isDateString

local function isIsoDateString(str)
	if type(str) == "string" then
		return #str == 10 and str:find("%d%d%d%d%-%d%d%-%d%d") ~= nil
	end
	return false
end
dt.isIsoDateString = isIsoDateString

local function toDateString(num, isEndDate) -- add format and default format
	if type(num) == "string" then
		return num:sub(1, 10) -- 2017-01-01 == first 10 chars
	end
	if isEndDate then
		local secs = removeDatePart(num)
		if secs == 0 then
			num = daysAdd(num, -1)
		end
	end
	--[[local txt = osDate("%Y-%m-%d", num)]]
	year, month, day = numToYmd(num)
	if month < 10 then
		---@diagnostic disable-next-line: cast-local-type
		month = "0" .. month
	end
	if day < 10 then
		day = "0" .. day
	end
	return year .. "-" .. month .. "-" .. day
end
dt.toDateString = toDateString

local function ymdToDateString(y, m, d)
	return toDateString(ymdToNum(y, m, d))
end
dt.ymdToDateString = ymdToDateString

local function secondsAdd(num, seconds)
	return num + seconds
end
dt.secondsAdd = secondsAdd

daysAdd = function(num, d)
	if d == 0 then
		return num
	end
	local origDate
	local isString = type(num) == "string"
	if isString then
		origDate = num
		num = dateParseToNum(num)
	end
	sec = removeDatePart(num)
	year, month, day = numToYmd(num)
	day = day + d
	local num2 = ymdToNum(year, month, day)
	num2 = num2 + sec -- ]]
	if isString then
		if origDate and #origDate < 11 and sec == 0 then
			return toDateString(num2) -- return only date part - not time
		else
			return toString(num2)
		end
	end
	return num2
end
dt.daysAdd = daysAdd

local function yearsAdd(num, years)
	if years == 0 then
		return num
	end
	local origDate
	local isString = type(num) == "string"
	if isString then
		origDate = num
		num = dateParseToNum(num)
	end
	sec = removeDatePart(num)
	year, month, day = numToYmd(num)
	year = year + years
	local num2 = ymdToNum(year, month, day)
	num2 = num2 + sec -- ]]
	if isString then
		if origDate and #origDate < 11 and sec == 0 then
			return toDateString(num2) -- return only date part - not time
		else
			return toString(num2)
		end
	end
	return num2
end
dt.yearsAdd = yearsAdd

local function daysSubtract(num, d)
	if d == 0 then
		return num
	end
	return daysAdd(num, -d)
end
dt.daysSubtract = daysSubtract

local function daysToCurrentDateStringAdd(num)
	-- return daysAdd(num, currentDateString())
	return toDateString(daysAdd(currentDate(), num))
end
dt.daysToCurrentDateStringAdd = daysToCurrentDateStringAdd

local function weeksAdd(num, weeks)
	return daysAdd(num, weeks * 7)
end
dt.weeksAdd = weeksAdd

local function isMidnight(num)
	if num % secondsInDay == 0 then
		return true
	end
	return false
end
dt.isMidnight = isMidnight

local function toSecondsFromDayStart(num, seconds)
	local time = removeTimePart(num)
	time = secondsAdd(time, seconds)
	return time
end
dt.toSecondsFromDayStart = toSecondsFromDayStart

local function toSecondsFromDayEnd(num, seconds)
	local time = removeTimePart(num)
	time = daysAdd(time, 1)
	time = secondsAdd(time, seconds)
	return time
end
dt.toSecondsFromDayEnd = toSecondsFromDayEnd

local function toHours(r, decimals) -- 6 decimals seems to be ok?
	if decimals then
		return round(r / 3600, decimals)
	end
	return r / 3600
end
dt.toHours = toHours

local function toMinutes(r, decimals)
	if decimals then
		return round(r / 60, decimals)
	end
	return r / 60
end
dt.toMinutes = toMinutes

local function toSeconds(r) -- , decimals)
	return r
end
dt.toSeconds = toSeconds

local function toTime(num)
	sec = removeDatePart(num)
	-- local hours = idiv(sec, 3600)
	local hours = floor(sec / 3600)
	-- local minutes = idiv(sec, 60) - (hours * 60)
	local minutes = floor(sec / 60) - (hours * 60)
	local seconds = sec - (hours * 3600) - (minutes * 60)
	return hours, minutes, seconds
end
dt.toTime = toTime

local function formatString(str, formatStr)
	local num = dateParse(str)
	return formatNum(num, formatStr)
end
dt.formatString = formatString

local function format(num, formatStr)
	if type(num) == "string" then
		num = dateParseToNum(num)
	end
	if type(formatStr) == "table" then
		return formatNum(num, formatStr[1])
	end
	return formatNum(num, formatStr)
end
dt.format = format

local function localDateStringFormat(str, formatStr)
	local num = dateParse(str)
	return formatNum(num, formatStr or "%d.%m.%Y")
end
dt.localDateStringFormat = localDateStringFormat

local function toRfc2822(num)
	if num < 0 then
		num = 0
	end
	return osDate("%a, %d %b %Y %X %z", num) -- changed "%a, %d %b %Y %T %z" to "%a, %d %b %Y %X %z"
end
dt.toRfc2822 = toRfc2822

local function toWeekdayNum(num) -- return 1-7
	if num <= 0 then
		num = 4 -- 1970-01-01 is thursday
	end
	return ymdToWeekdayNum(numToYmd(num))
end
dt.toWeekdayNum = toWeekdayNum

local function toWeekdayNumLua(num) -- return 1-7
	if num <= 0 then
		num = 4 -- 1970-01-01 is thursday
	end
	local weekday = tonumber(osDate("%w", num)) -- [0-6 = Sunday-Saturday]
	if weekday == 0 then
		return 7 -- 1-7 = Monday-Sunday
	end
	return weekday
end
dt.toWeekdayNumLua = toWeekdayNumLua

local function toWeekdayShortName(num)
	local weekday = toWeekdayNum(num)
	return dayNameShort[weekday]
end
dt.toWeekdayShortName = toWeekdayShortName

local function toWeekdayName(num)
	local weekday = toWeekdayNum(num) -- [0-6 = Monday-Sunday]
	return dayName[weekday]
end
dt.toWeekdayName = toWeekdayName

local function ywdStringToDate(str) -- yyyy(w)wdd -> date number
	year = str:sub(1, 4)

	if (#str == 6) then
		week = str:sub(5, 5)
		day = str:sub(6, 6)
	else
		week = str:sub(5, 6)
		day = str:sub(7, 7)
	end

	return weekDaynumToNum(tonumber(year) * 100 + tonumber(week), tonumber(day))
end
dt.ywdStringToDate = ywdStringToDate

local function dayDifference(start_time, end_time)
	if type(start_time) == "string" then
		start_time = dateParse(start_time)
	end
	if type(end_time) == "string" then
		end_time = dateParse(end_time)
	end
	start_time = removeTimePart(start_time)
	end_time = removeTimePart(end_time)
	--[[start_time = idiv(start_time, secondsInDay)
		start_time = start_time - unixDateInJulian
		end_time = idiv(end_time, secondsInDay)
		end_time = end_time - unixDateInJulian
		]]
	local days = end_time - start_time
	-- days = idiv(days, secondsInDay)
	days = floor(days / secondsInDay)
	return days
end
dt.dayDifference = dayDifference

local function weekDifference(start_time, end_time)
	-- we need actual weeks between to dates, and not the number of seven days between them
	local startWeekDay = toWeekdayNum(start_time) - 1 -- toWeekdayNum returns 1-7, we want to remove 0-6 days from the data to get monday on that week
	local endWeekDay = toWeekdayNum(end_time) - 1
	local days = dayDifference(start_time - startWeekDay * secondsInDay, end_time - endWeekDay * secondsInDay)
	return days / 7 -- ceil(days / 7) -- days / 7 should be always integer without decimals
end
dt.weekDifference = weekDifference

local function hourDifference(start_time, end_time)
	sec = end_time - start_time
	local hours = sec / 3600
	return hours
end
dt.hourDifference = hourDifference

local function unionRecHours(start_dt, end_dt, rec, start_field_name, end_field_name)
	if rec[start_field_name] and rec[end_field_name] then
		if end_dt > start_dt and rec[end_field_name] > rec[start_field_name] and rec[end_field_name] > start_dt and end_dt > rec[start_field_name] then
			if rec[start_field_name] < start_dt and end_dt < rec[end_field_name] then -- start before & end after time period
				return hourDifference(start_dt, end_dt), start_dt, end_dt
			elseif rec[start_field_name] < start_dt then -- end inside time period
				return hourDifference(start_dt, rec[end_field_name]), start_dt, rec[end_field_name]
			elseif end_dt < rec[end_field_name] then -- start inside time period
				return hourDifference(rec[start_field_name], end_dt), rec[start_field_name], end_dt
			else -- whole md inside time period
				return hourDifference(rec[start_field_name], rec[end_field_name]), rec[start_field_name], rec[end_field_name]
			end
		end
	end
	return 0
end
dt.unionRecHours = unionRecHours

--[[ local function unionTableHours(start_dt, end_dt, tbl, start_field_name, end_field_name)
	local hours = 0
	for _, rec in ipairs(tbl) do
		hours = hours + unionRecHours(start_dt, end_dt, rec, start_field_name, end_field_name)
	end
	return hours
end ]]

local function secondsBetween(start_time, end_time)
	return abs(end_time - start_time)
end
dt.secondsBetween = secondsBetween

local function secondDifference(start_time, end_time)
	return end_time - start_time -- secondsBetween(start_time, end_time)
end
dt.secondDifference = secondDifference

local function timeToNum(hours, minutes, seconds)
	return hours * 3600 + minutes * 60 + seconds
end
dt.timeToNum = timeToNum

local function timeStringToNum(str)
	local hours = tonumber(peg.parse(str, ":", 1))
	local minutes = tonumber(peg.parse(str, ":", 2))
	local seconds = tonumber(peg.parse(str, ":", 3))
	if hours < 0 then
		return -(abs(hours) * 3600 + minutes * 60 + seconds)
	end
	return hours * 3600 + minutes * 60 + seconds
end
dt.timeStringToNum = timeStringToNum

local function currentTime()
	local d = osDate("*t")
	return timeToNum(d.hour, d.min, d.sec)
	--[[

	-- use also timezone?
	-- 	https://developer.apple.com/library/mac/documentation/Darwin/Reference/ManPages/man2/gettimeofday.2.html

	local ffi = require "mffi"

	ffi.cdef[-[
		typedef long time_t;

		typedef struct timeval {
			time_t tv_sec;
			time_t tv_usec;
		} timeval;

		int gettimeofday(struct timeval* t, void* tzp);
	]-]

	local gettimeofday_struct = ffi.newNoAnchor("timeval")
	local function gettimeofday()
		 ffi.C.gettimeofday(gettimeofday_struct, nil)
		 return tonumber(gettimeofday_struct.tv_sec) * 1000000 +
			 tonumber(gettimeofday_struct.tv_usec)
	end
	]]
end
dt.currentTime = currentTime

-- Compute the difference in seconds between local time and UTC.
local function currentTimeZoneSeconds()
	local now = osTime()
	local tlocal = osDate("*t", now)
	local t = osDate("!*t", now)
	local time = os.difftime(now, osTime(t))
	if tlocal.isdst then -- PM: added this
		time = time + 3600 -- does not work always aroud the globe!
	end
	return time
end
dt.currentTimeZoneSeconds = currentTimeZoneSeconds

local function toTimeString(num, isEndDate)
	if type(num) == "string" then
		return num
	end
	sec = removeDatePart(num)
	if isEndDate and sec == 0 then
		return "24:00:00"
	end
	--[[local txt = osDate("%H:%M:%S", num)
	]]
	local hours, minutes, seconds = toTime(num)
	if hours < 10 then
		---@diagnostic disable-next-line: cast-local-type
		hours = "0" .. hours
	end
	if minutes < 10 then
		---@diagnostic disable-next-line: cast-local-type
		minutes = "0" .. minutes
	end
	seconds = floor(seconds)
	if seconds < 10 then
		seconds = "0" .. seconds
	end
	local txt = hours .. ":" .. minutes .. ":" .. seconds
	return txt
end
dt.toTimeString = toTimeString

local function currentTimeString()
	return toTimeString(currentTime())
end
dt.currentTimeString = currentTimeString

local function millisecondToString(ms)
	return toString(tonumber(ms) / 1000)
end
dt.millisecondToString = millisecondToString

local function millisecondToDateString(ms)
	return toDateString(tonumber(ms) / 1000)
end
dt.millisecondToDateString = millisecondToDateString

local function millisecondToTimeString(ms)
	return toTimeString(tonumber(ms) / 1000)
end
dt.millisecondToTimeString = millisecondToTimeString

local function millisecondToLocalTimeString(ms)
	return toTimeString(tonumber(ms) / 1000 + currentTimeZoneSeconds())
end
dt.millisecondToLocalTimeString = millisecondToLocalTimeString

local function currentDateTime()
	local d = osDate("*t")
	return ymdToNum(d.year, d.month, d.day) + timeToNum(d.hour, d.min, d.sec)
end
dt.currentDateTime = currentDateTime

local function currentDateTimeUtc()
	local d = osDate("!*t") -- If format starts with '!', then the date is formatted in Coordinated Universal Time UTC
	return ymdToNum(d.year, d.month, d.day) + timeToNum(d.hour, d.min, d.sec) -- -currentTimeZoneSeconds()
end
dt.currentDateTimeUtc = currentDateTimeUtc

local function currentFormat(formatStr)
	local time = currentDateTime()
	time = time - currentTimeZoneSeconds()
	if type(formatStr) == "table" then
		return formatNum(time, formatStr[1])
	end
	return formatNum(time, formatStr)
end
dt.currentFormat = currentFormat

local function currentStringLocal(formatStr)
	return currentFormat(formatStr)
end
dt.currentStringLocal = currentStringLocal

local function currentDateTimeMs()
	local d = osDate("*t")
	local ms = util.seconds()
	ms = ms - floor(ms) -- take only decimal part
	return ymdToNum(d.year, d.month, d.day) + timeToNum(d.hour, d.min, d.sec) + ms
end
dt.currentDateTimeMs = currentDateTimeMs

local function currentDateTable()
	local num = currentDate()
	year, month, day = numToYmd(num)
	return {y = year, m = month, d = day}
end
dt.currentDateTable = currentDateTable

local function currentString()
	local num = currentDateTime()
	return toString(num)
end
dt.currentString = currentString

local function currentStringMs()
	local num = currentDateTimeMs()
	return toString(num)
end
dt.currentStringMs = currentStringMs

local function currentDayOffsetDateString(dayOffset)
	local num = currentDateTime()
	num = daysAdd(num, dayOffset)
	return toDateString(num)
end
dt.currentDayOffsetDateString = currentDayOffsetDateString

local function currentHmsOffsetString(h, m, s)
	local num = currentDateTime()
	num = hmsAdd(num, h, m, s)
	return toString(num)
end
dt.currentHmsOffsetString = currentHmsOffsetString

local function nullDateString()
	return "0000-00-00" -- "1970-01-01"
end
dt.nullDateString = nullDateString

local function currentTimeZone(hhmmSeparator)
	-- Return a timezone string in ISO 8601:2000 standard form (+hhmm or -hhmm)
	-- http://lua-users.org/wiki/TimeZone
	local function get_tzoffset(tz)
		local h, m = modf(tz / 3600)
		if hhmmSeparator then
			return strFormat("%+.2d" .. hhmmSeparator .. "%.2d", h, 60 * m)
		end
		return strFormat("%+.4d", 100 * h + 60 * m)
	end
	local timezone = currentTimeZoneSeconds()
	local tzoffset = get_tzoffset(timezone)
	return tzoffset
end
dt.currentTimeZone = currentTimeZone

local function currentTimeZoneString()
	local timestamp = currentString()
	local timezone = currentTimeZone()
	timezone = peg.replace(timezone, ":", "")
	return timestamp .. timezone
end
dt.currentTimeZoneString = currentTimeZoneString

local function isZeroDate(date)
	return date == "1970-01-01 00:00:00" or date == 0
end
dt.isZeroDate = isZeroDate

local dateStr
local function isNullDate(date)
	-- NOTE: any timestamp starting 1970-01-01 or less WITH ANY time part is null date
	-- this is because timezone can be part of date and null date with timezone can be for ex.: 1970-01-01 02:00:00
	local ret = false
	if type(date) == "string" then
		dateStr = date
	elseif type(date) == "number" then
		dateStr = toDateString(date)
	end
	if dateStr == nil then
		ret = true -- ???
	elseif dateStr == "0000-00-00" or dateStr:sub(1, 10) <= "1970-01-01" then
		ret = true
	end
	return ret
end
dt.isNullDate = isNullDate

local function subsecond() -- not actually nanoseconds, just most accurate you can get
	local ns = util.seconds()
	return ns - floor(ns) -- take only decimal part
end
dt.subsecond = subsecond

local function milliseconds()
	local ms = subsecond() -- take only decimal part
	ms = round(ms * 1000, 0) -- return value 0-999
	return ms
end
dt.milliseconds = milliseconds
--[[
local currentMilliseconds = 99
local function milliseconds()
	currentMilliseconds = currentMilliseconds + 1
	if currentMilliseconds > 999 then
		currentMilliseconds = 100
	end
	return currentMilliseconds
end
dt.milliseconds = milliseconds
]]

local function stripTime(date)
	return date:sub(1, 10)
end
dt.stripTime = stripTime

local function currentStringPicosecond()
	local num = currentDateTime()
	local str = toString(num) .. tostring(subsecond()):sub(2) .. "000000000000" -- sub(2) = 0.xxx -> .xxx
	return str:sub(1, 32) -- truncs to 12 decimals
end
dt.currentStringPicosecond = currentStringPicosecond

local function currentStringNanosecond()
	local num = currentDateTime()
	local str = toString(num) .. tostring(subsecond()):sub(2) .. "000000000" -- sub(2) = 0.xxx -> .xxx
	return str:sub(1, 29) -- truncs to 9 decimals
end
dt.currentStringNanosecond = currentStringNanosecond

local function currentStringMicrosecond()
	local num = currentDateTime()
	local str = toString(num) .. tostring(subsecond()):sub(2) .. "000000" -- sub(2) = 0.xxx -> .xxx
	return str:sub(1, 26) -- truncs to 6 decimals
end
dt.currentStringMicrosecond = currentStringMicrosecond

local function currentStringMillisecond()
	local num = currentDateTime()
	local str = toString(num) .. tostring(subsecond()):sub(2) .. "000" -- sub(2) = 0.xxx -> .xxx
	return str:sub(1, 23) -- truncs to 6 decimals
end
dt.currentStringMillisecond = currentStringMillisecond

local function secondsToTimestampStringMicrosecond(num)
	local str = toString(num) .. "000000"
	return str:sub(1, 26) -- truncs to 6 decimals
end
dt.secondsToTimestampStringMicrosecond = secondsToTimestampStringMicrosecond

local function secondsToTimestampString(num)
	num = floor(num) -- remove second decimals
	return toString(num)
end
dt.secondsToTimestampString = secondsToTimestampString

local function currentDateString()
	local num = currentDateTime()
	return toDateString(num)
end
dt.currentDateString = currentDateString

-- function currentUtcStringToFilename()
-- local timestamp = currentUtcString()
-- return currentUtcString()
-- end

local function ymdAdd(num, y, m, d)
	local origDate
	local isString = type(num) == "string"
	if isString then
		origDate = num
		num = dateParseToNum(num)
	end
	sec = removeDatePart(num)
	year, month, day = numToYmd(num)
	year = year + y
	month = month + m
	day = day + d
	local num2 = ymdToNum(year, month, day)
	num2 = num2 + sec
	if isString then
		if origDate and #origDate < 11 and sec == 0 then
			return toDateString(num2) -- return only date part - not time
		else
			return toString(num2)
		end
	end
	return num2
end
dt.ymdAdd = ymdAdd

local function ymdAddDateString(str, y, m, d)
	local num = dateParseToNum(str)
	sec = removeDatePart(num)
	year, month, day = numToYmd(num)
	year = year + y
	month = month + m
	day = day + d
	local num2 = ymdToNum(year, month, day)
	num2 = num2 + sec
	return toDateString(num2)
end
dt.ymdAddDateString = ymdAddDateString

local minute, second, retNum
local function parseToString(txtDate, dateFormat)
	minute, second = 0, 0
	if txtDate == nil or txtDate == "" then
		return ""
	elseif dateFormat then
		local yearStart = peg.find(dateFormat, "YYYY")
		if yearStart > 0 then
			year = tonumber(txtDate:sub(yearStart, yearStart + 3))
		else
			yearStart = peg.find(dateFormat, "YY")
			if yearStart > 0 then
				year = currentCentury() + tonumber(txtDate:sub(yearStart, yearStart + 1))
			end
		end
		local monthStart = peg.find(dateFormat, "MM")
		if monthStart > 0 then
			month = tonumber(txtDate:sub(monthStart, monthStart + 1))
		end
		local dayStart = peg.find(dateFormat, "DD")
		if dayStart > 0 then
			day = tonumber(txtDate:sub(dayStart, dayStart + 1))
		end
		local hourStart = peg.find(dateFormat, "hh")
		if hourStart > 0 then
			hour = tonumber(txtDate:sub(hourStart, hourStart + 1))
			local minuteStart = peg.find(dateFormat, "mm")
			if minuteStart > 0 then
				minute = tonumber(txtDate:sub(minuteStart, minuteStart + 1))
			end
			local secondStart = peg.find(dateFormat, "ss")
			if secondStart > 0 then
				second = tonumber(txtDate:sub(secondStart, secondStart + 1))
			end
		end
	else
		year = tonumber(txtDate:sub(1, 4))
		month = tonumber(txtDate:sub(6, 7))
		day = tonumber(txtDate:sub(9, 10))
	end
	if year then
		retNum = ymdToNum(year, month, day)
	end
	if hour then
		retNum = hmsAdd(retNum, hour, minute, second)
	end
	if year and hour then
		return toString(retNum)
	elseif hour then
		return toTimeString(retNum)
	elseif year then
		return toDateString(retNum)
	else
		return ""
	end
end
dt.parseToString = parseToString

local function toLocalString(num, isEndDate)
	num = num + timeZoneOffset(num)
	local txt = toDateString(num, isEndDate) .. " " .. toTimeString(num, isEndDate)
	return txt
end
dt.toLocalString = toLocalString

toString = function(num, isEndDate)
	return toDateString(num, isEndDate) .. " " .. toTimeString(num, isEndDate)
end
dt.toString = toString

--[[ function toStringMillisecond(num, isEndDate)
	local txt = toDateString(num, isEndDate) .. " " .. toTimeString(num, isEndDate) .. "." .. (num - floor(num)) -- take only decimal part
	return txt
end ]]

local function secondsToTimeString(s)
	-- sec = secondsAdd(0, s)
	return toTimeString(s)
end
dt.secondsToTimeString = secondsToTimeString

local dec
local function formatTime(seconds, decimals)
	sec = floor(seconds)
	dec = round(seconds, decimals) - sec
	---@diagnostic disable-next-line: cast-local-type
	dec = strFormat("%." .. decimals .. "f", dec)
	return toTimeString(sec) .. dec:sub(2) -- take from dot
end
dt.formatTime = formatTime

local function toUtcString(num)
	return toDateString(num) .. "T" .. toTimeString(num) -- .."000Z"
end
dt.toUtcString = toUtcString

local function currentUtcStringMillisecond()
	local num = currentDateTime()
	return toUtcString(num) .. "_" .. tostring(milliseconds())
end
dt.currentUtcStringMillisecond = currentUtcStringMillisecond

local function toFileString(num, isEndDate)
	local txt
	if type(num) == "string" then
		txt = num
	else
		txt = toString(num, isEndDate)
	end
	txt = peg.replace(txt, " ", "_")
	return peg.replace(txt, ":", ".")
end
dt.toFileString = toFileString

local function currentFileString()
	return toFileString(currentDateTime())
end
dt.currentFileString = currentFileString

local function currentUtcString()
	return toUtcString(currentDateTime())
end
dt.currentUtcString = currentUtcString

local function currentUtcStringZ()
	return currentUtcString() .. "Z"
end
dt.currentUtcStringZ = currentUtcStringZ

-- 1970-01-01 00:00:00
local function ISO_8601DateTimeNow()
	local unixTime = osTime()
	local date = osDate("%Y", unixTime) .. "-" .. osDate("%m", unixTime) .. "-" .. osDate("%d", unixTime)
	local time = osDate("%H", unixTime) .. ":" .. osDate("%M", unixTime) .. ":" .. osDate("%S", unixTime)
	return date .. " " .. time
end
dt.ISO_8601DateTimeNow = ISO_8601DateTimeNow

-- 1970-01-01 00:00:00+00
local function ISO_8601DateTimeWithTimezoneNow()
	local unixTime = osTime()
	local date = osDate("%Y", unixTime) .. "-" .. osDate("%m", unixTime) .. "-" .. osDate("%d", unixTime)
	local time = osDate("%H", unixTime) .. ":" .. osDate("%M", unixTime) .. ":" .. osDate("%S", unixTime)
	return date .. " " .. time .. "+02"
end
dt.ISO_8601DateTimeWithTimezoneNow = ISO_8601DateTimeWithTimezoneNow

local function dateTimeWithTimeZoneParse(txtDateTimeWithTimezone)
	local txtDate = txtDateTimeWithTimezone:sub(1, 10) -- "2014-01-22T22:33:44"
	local txtTime = txtDateTimeWithTimezone:sub(12, 19)
	local tzoffsetOperation = txtDateTimeWithTimezone:sub(20, 20)
	local tzoffset = txtDateTimeWithTimezone:sub(21, 22)
	tzoffset = tonumber(tzoffset)
	local ret = dateParse(txtDate)
	ret = ret + timeParse(txtTime)
	if tzoffsetOperation == "+" then
		ret = ret - (tzoffset * 60 * 60)
	elseif tzoffsetOperation == "-" then
		ret = ret + (tzoffset * 60 * 60)
	end
	if ret < 0 then -- for incase
		return 0
	end
	return ret
end
dt.dateTimeWithTimeZoneParse = dateTimeWithTimeZoneParse

local function dateTimeParse(txtDateTime)
	if type(txtDateTime) ~= "string" then
		util.printError("dateTimeParse parameter type '%s' is not string", type(txtDateTime))
		return 0
	end
	local txtDate = txtDateTime:sub(1, 10) -- "2014-01-22T22:33:44"
	local txtTime = txtDateTime:sub(12, 19)
	local ret = dateParse(txtDate)
	ret = ret + timeParse(txtTime) -- secondsAdd(ret, timeParse(txtTime))
	return ret
end
dt.dateTimeParse = dateTimeParse

local function hourDifferenceTimestamp(startTs, endTs)
	if type(startTs) == "string" then
		startTs = dateTimeParse(startTs)
	end
	if type(endTs) == "string" then
		endTs = dateTimeParse(endTs)
	end
	sec = endTs - startTs
	return sec / 3600 -- seconde to hours
end
dt.hourDifferenceTimestamp = hourDifferenceTimestamp

local function dateAndTimeToNum(txtDate, txtTime)
	local ret
	if txtDate == nil or txtTime == nil then
		return 0
	end
	if type(txtDate) == "number" then
		ret = txtDate
	else
		ret = dateParse(txtDate)
	end
	if type(txtTime) == "number" then -- time in seconds
		ret = ret + txtTime
	else
		ret = ret + timeParse(txtTime) -- secondsAdd(ret, timeParse(txtTime))
	end
	return ret
end
dt.dateAndTimeToNum = dateAndTimeToNum

local date2000
local function to4dReal(num)
	if not date2000 then
		date2000 = ymdToNum(2000, 1, 1)
	end
	sec = num - date2000
	return sec / secondsInDay
end
dt.to4dReal = to4dReal

local function timestampToDateString(timestamp)
	local y, m, d = dateParseToYmd(timestamp)
	return ymdToDateString(y, m, d)
end
dt.timestampToDateString = timestampToDateString

local function setTimestampToFile(filename, prefix, divider, setTimezone)
	if prefix == nil then
		prefix = ""
	end
	if divider == nil then
		divider = "."
	end
	if setTimezone == nil then
		setTimezone = true
	end
	local timestamp = currentUtcString()
	timestamp = peg.replace(timestamp, "-", divider)
	timestamp = peg.replace(timestamp, ":", divider)
	timestamp = peg.replace(timestamp, "/", divider)
	local millisec = tostring(milliseconds())
	millisec = peg.replace(millisec, ":", "")
	local timezone = currentTimeZone()
	timezone = peg.replace(timezone, ":", "")
	if setTimezone == true then
		filename = filename .. timestamp .. divider .. millisec .. divider .. timezone .. prefix -- ".xml"
	else
		filename = filename .. timestamp .. divider .. millisec .. prefix
	end
	return filename
	-- currentUtcStringMillisec
end
dt.setTimestampToFile = setTimestampToFile

local function firstDateAndTimeIsLater(date1, time1, date2, time2)
	if date1 == date2 then
		if time1 > time2 then
			return true
		end
	elseif date1 > date2 then
		return true
	end
	return false
end
dt.firstDateAndTimeIsLater = firstDateAndTimeIsLater

local function secondsToYmdHms(secs) -- , tz
	local y, m, d = numToYmd(secs)
	hour = floor((secs % 86400) / 3600)
	if daylightSavingsTime(y, m, d, hour) then
		hour = hour + 1 -- daylight savings time currently in effect, add one hour
		-- print("  daylight savings time is currently in effect")
		if hour == 24 then
			hour = 0
			d = d + 1
			if d > monthTable[leapYearIndex(y)][m] then
				d = 1
				m = m + 1
				if m > 12 then
					m = 1
					y = y + 1
				end
			end
		end
		-- else
		--   print("  daylight savings time is not currently in effect")
	end
	return y, m, d, hour, (secs % 3600) / 60, secs % 60 -- year, month, day, hour, minute, second
end
dt.secondsToYmdHms = secondsToYmdHms

return dt
