-- fnutil.lua
-- Utility functions for functional programming and fn.lua.
-- module fn
local l, util, peg, dprf, dschema, sort, recData -- delay load

local function loadLibs()
	if not util then
		l = require"lang".l
		util = require "util"
		peg = require "peg"
		dprf = require "dprf"
		dschema = require "dschema"
		sort = require "table/sort"
		recData = require"recdata".get
	end
end

local fn
local function setFn(fn_) -- to prevent circular errors
	fn = fn_
end

local function objectToArray(groupResult, key1Sum, key2Sum, key2Idx, resultOrderArr)
	local row, total, idx
	local totalAll = 0
	for key, val in pairs(groupResult) do
		row = {[resultOrderArr[1]] = key}
		total = 0
		idx = 0
		for key2, val2 in pairs(val) do
			idx = idx + 1
			for key3, val3 in pairs(val2) do
				if key2Idx[key2] == nil then
					key2Idx[key2] = {[resultOrderArr[3]] = key2, [key3] = 0, group = {[key] = {[key3] = 0}}}
					key2Sum[#key2Sum + 1] = key2Idx[key2]
				end
				if key2Idx[key2][key3] == nil then
					key2Idx[key2][key3] = 0
				end
				key2Idx[key2][key3] = key2Idx[key2][key3] + val3
				if key2Idx[key2].group[key] == nil then
					key2Idx[key2].group[key] = {[key3] = 0}
				end
				if key2Idx[key2].group[key][key3] == nil then
					key2Idx[key2].group[key][key3] = 0
				end
				key2Idx[key2].group[key][key3] = key2Idx[key2].group[key][key3] + val3
				total = total + val3
			end
		end
		totalAll = totalAll + total
		row[resultOrderArr[2]] = total
		key1Sum[#key1Sum + 1] = row
	end
	sort.sort(key1Sum, {resultOrderArr[2], "<", resultOrderArr[1], ">"})
	sort.sort(key2Sum, {resultOrderArr[4], "<", resultOrderArr[3], ">"})
	return totalAll
end

local function groupByCalculate(arr, result, groupFieldArr, sumDefinitionArr)
	loadLibs()
	local group, ret
	for _, item in ipairs(arr) do
		for i, groupField in ipairs(groupFieldArr) do
			if i == 1 then
				ret = result
			end
			group = item[groupField] or recData(item, groupField) or ""
			if ret[group] == nil then
				ret[group] = {}
			end
			ret = ret[group] -- move to next level group
		end
		for _, def in ipairs(sumDefinitionArr) do
			if ret[def.result] == nil then
				ret[def.result] = 0
			end
			for _, field in ipairs(def.field) do
				if item[field] and item[field] > 0 then
					if def.operator == "+" then
						ret[def.result] = ret[def.result] + item[field]
					elseif def.operator == "*" then
						ret[def.result] = ret[def.result] * item[field]
					elseif def.operator == "-" then
						ret[def.result] = ret[def.result] - item[field]
					elseif def.operator == "/" then
						ret[def.result] = ret[def.result] / item[field]
					else
						local err = util.printRed("group by calculate operator '%s' is not supported", tostring(def.operator))
						return err
					end
				end
			end
		end
	end
end

local function uniqueToArr(acc, x) -- fn.reduce function
	if not fn.index(x, acc) then -- x was not found from acc array
		acc[#acc + 1] = x
	end
	return acc
end

local function uniqueArray(gen, keyName)
	return fn.reduce(uniqueToArr, {}, fn.map(function(rec)
		return rec[keyName]
	end, gen))
end

local function uniqueConcatArray(gen, keyName, separator)
	loadLibs()
	return fn.reduce(uniqueToArr, {}, fn.map(function(rec)
		return fn.reduce(function(acc, key) -- combine keys to one unique string
			if recData(rec, key) == nil then
				return nil
			end
			if acc == "" then -- first call
				return acc .. tostring(recData(rec, key))
			else
				return acc .. separator .. tostring(recData(rec, key))
			end
		end, "", keyName)
	end, gen))
end

local function addUniqueToArray(arr)
	-- local unique = {}
	return function(val)
		-- if val and not unique[val] then
		--   unique[val] = true
		if val and fn.index(val, arr) == nil then
			arr[#arr + 1] = val
		end
	end
end

local function arrayKeyIsUnique(arr, fieldName)
	-- this should be done without unique table with fn.xxx?
	loadLibs()
	local unique = {}
	local manyFields = type(fieldName) == "table"
	local val
	-- we must use ipairs because return does not break fn.each() loop
	for _, rec in ipairs(arr) do -- fn.each(function(rec)
		if manyFields then
			val = fn.reduce(function(acc, key) -- combine keys to one unique string
				return acc .. tostring(recData(rec, key)) .. string.char(0) -- string.char(0) is safer than "\t"
			end, "", fieldName)
		else
			val = rec[fieldName]
		end
		if unique[val] then
			return false
		else
			unique[val] = true
		end
	end -- , arr)
	return true
end

local function recordFilter(generator, rec, filterFunction)
	return fn.filter(function(generatorRec)
		return filterFunction(rec, generatorRec)
	end, generator)
end

local function callArrayParamRecursive(param, recFunction, paramType, jsonOption)
	loadLibs()
	if paramType ~= "string" and type(param) == "string" and peg.endsWith(param, ".json") then
		local prf, err = dprf.preferenceFromJson(param, jsonOption)
		if util.tableIsEmpty(prf) then
			return err or l("preference '%s' was not found or preference was invalid json", param)
		end
		return callArrayParamRecursive(prf, recFunction, paramType)
	end
	if type(param[1]) == "table" then
		-- return fn.each(function(rec) -- this can't return errors
		for _, rec in ipairs(param) do
			local err = callArrayParamRecursive(rec, recFunction)
			if err ~= nil then
				return err
			end
		end
		-- end, tbl)
	elseif paramType and type(param) == "table" and paramType ~= "table" then
		for _, rec in ipairs(param) do
			local err = recFunction(rec)
			if err ~= nil then
				return err
			end
		end
	elseif paramType and type(param) == paramType then
		return recFunction(param)
	elseif not paramType then
		return recFunction(param)
	else
		return l("parameter type is not '%s'", paramType)
	end
end

local function uniqueFieldValueGen(recArr, key)
	loadLibs()
	local iter = fn.iter(recArr)
	local function mapDistinct(rec)
		local found = iter:filter(function(loopRec) -- find first where key is same or return self
			return rec == loopRec or recData(rec, key) == loopRec[key]
		end):head()
		if found == rec then
			return recData(rec, key)
		end
		-- == return nil
	end
	return iter:map(mapDistinct):filter(function(rec) -- do not return nils
		return rec ~= nil
	end)
end

local function distinctFieldValueArray(recArr, key, returnArray, returnArrayIdx)
	loadLibs()
	local count = 0
	local ret = returnArray or {}
	local distinct = returnArrayIdx or {}
	local useRecData = peg.found(key, ".")
	local val
	for _, rec in ipairs(recArr) do
		if useRecData then
			val = recData(rec, key)
		else
			val = rec[key]
		end
		if distinct[val] == nil then
			distinct[val] = true
			count = count + 1
			ret[count] = val
		end
	end
	return ret
end

local function mapFieldValue(recArr, key)
	loadLibs()
	local function mapDistinct(rec)
		return recData(rec, key)
	end
	return fn.iter(recArr):map(mapDistinct) -- :filter(nill pois)
end

local function mapRecordArrayGen(valueToFind, recordArray, fieldName)
	loadLibs()
	return fn.iter(recordArray):filter(function(rec)
		if recData(rec, fieldName) == valueToFind then
			return true
		end
		return false
	end):map(function(rec)
		return rec
	end)
end

local function setIdx(data)
	for idx, item in ipairs(data) do
		item.idx = idx
	end
end

local function createIndex(data, key, exclude, maxWarnCount, keyOrBoolean)
	local ret = {}
	local warnKey
	local warnCount = 0
	if maxWarnCount == false then
		maxWarnCount = -1
	elseif type(maxWarnCount) ~= "number" then
		maxWarnCount = math.huge
	end
	if data == nil then
		loadLibs()
		util.printWarning("fn.util.createIndex data is nil, key: '%s'", key)
	else
		if type(keyOrBoolean) == "string" then
			loadLibs()
		end
		local val
		for idx, item in ipairs(data) do
			val = item[key]
			if val then
				if exclude == nil or exclude[val] == nil or not (exclude[val]) then
					if ret[val] ~= nil then
						warnCount = warnCount + 1
						if warnCount <= maxWarnCount then
							if warnKey == nil then
								warnKey = {}
							end
							if warnKey[val] == nil then
								warnKey[val] = 1
								loadLibs()
								util.printWarning("fn.util.createIndex key '%s' value '%s' at index %d is not unique", key, val, idx)
							else
								warnKey[val] = warnKey[val] + 1
							end
						end
					elseif type(keyOrBoolean) == "string" then
						ret[val] = recData(item, keyOrBoolean)
					elseif type(keyOrBoolean) == "boolean" then
						ret[val] = true
					else
						ret[val] = item
					end
				end
			end
		end
	end
	return ret, warnKey
end

local function concatArray(arr1, arr2)
	local i = #arr1
	for _, item in ipairs(arr2) do
		i = i + 1
		arr1[i] = item
	end
end

local function createIndexArray(data, key, exclude, include, valueKey)
	-- exclude are values you want to skip {[""] = true}
	-- include are values you only want to include {[""] = true}
	local ret = {}
	loadLibs()
	if data == nil then
		util.printWarning("fn.util.createIndexArray data is nil, key: '%s'", key)
	else
		local val
		local useRecData = false
		if peg.found(key, ".") then
			useRecData = true
		end
		local useValueRecData = false
		if valueKey and peg.found(valueKey, ".") then
			useValueRecData = true
		end
		for _, item in ipairs(data) do
			if useRecData then
				val = recData(item, key)
			else
				val = item[key]
			end
			if val ~= nil then
				if (include == nil or include[val]) and (exclude == nil or exclude[val] == nil or not (exclude[val])) then
					if ret[val] == nil then
						ret[val] = {}
					end
					if valueKey then
						if useValueRecData then
							ret[val][#ret[val] + 1] = recData(item, valueKey)
						else
							ret[val][#ret[val] + 1] = item[valueKey]
						end
					else
						ret[val][#ret[val] + 1] = item
					end
				end
			end
		end
	end
	return ret
end

local function fieldKey(data, fldOrTbl, multiple)
	loadLibs()
	local idxTbl = {}
	-- local schema, recordType
	if next(data) == nil then
		return idxTbl
	end
	local idxFld
	if data[1] == nil or data[1][fldOrTbl] == nil then
		idxFld = dschema.primaryKeyField(fldOrTbl) -- , schema, recordType)
	else
		idxFld = fldOrTbl
	end
	if idxFld == nil then
		local tbl = dschema.tableName(fldOrTbl)
		if tbl then
			util.printError("table '%s' has no primary keyy, field '%s'", tbl, tostring(fldOrTbl))
		else
			util.printError("table has no primary key, field '%s' ", tostring(fldOrTbl))
		end
	else
		idxFld = peg.parseAfter(idxFld, ".") -- pr.product_id -> product_id
		if multiple then
			local val
			for _, item in ipairs(data) do
				val = item[idxFld]
				if idxTbl[val] == nil then
					idxTbl[val] = {}
				end
				idxTbl[val][#idxTbl[val] + 1] = item
			end
		else
			for _, item in ipairs(data) do
				idxTbl[item[idxFld]] = item
			end
		end
		return idxTbl
	end
end

return {
	-- addToArray = addToArray, -- use fn.totable()
	objectToArray = objectToArray,
	groupByCalculate = groupByCalculate,
	setIdx = setIdx,
	createIndex = createIndex,
	concatArray = concatArray,
	createIndexArray = createIndexArray,
	fieldKey = fieldKey,
	setFn = setFn,
	uniqueArray = uniqueArray,
	uniqueConcatArray = uniqueConcatArray,
	addUniqueToArray = addUniqueToArray,
	arrayKeyIsUnique = arrayKeyIsUnique,
	recordFilter = recordFilter,
	callArrayParamRecursive = callArrayParamRecursive,
	uniqueFieldValueGen = uniqueFieldValueGen,
	distinctFieldValueArray = distinctFieldValueArray,
	mapFieldValue = mapFieldValue,
	mapRecordArrayGen = mapRecordArrayGen
}
