--- lib/fs.lua
-- fs = filesystem
-- https://keplerproject.github.io/luafilesystem/manual.html#reference
-- todo: fs.readFile vs fs.getFile, not both, combine them
local fs = {}

local util = require "util"
local lfs = require "lfs_load"
local peg = require "peg"
local fn = require "fn"
local json = require "json"
local l = require"lang".l
local utf = require "utf"
local endsWith, startsWith, replace, replaceFirst, removeFromEnd, found = peg.endsWith, peg.startsWith, peg.replace, peg.replaceFirst, peg.removeFromEnd, peg.found

local winDriveLetter
local filePathChange
local loc = {}
loc.prettierExists = nil
loc.defaultKeyNameArr = nil
local maxReadFileSize = 50 * 1024 * 1024 -- 50 Mb

if type(lfs) == "table" then
	for key, val in pairs(lfs) do -- return all lfs functions + some new
		if type(val) == "function" then
			fs[key] = val
		end
	end
end

local function executeError(ret, err)
	if ret ~= true then
		util.printError("os.execute error: " .. tostring(ret) .. ", " .. tostring(err))
	end
end

local function fileNameAllowed(fileName)
	return fileName ~= "." and fileName ~= ".." and fileName ~= ".DS_Store" and fileName:sub(1, 1) ~= "$" -- fileName:sub(1, 1) ~= "."
end

function fs.setWindowsDrive(drive)
	winDriveLetter = drive
end

function fs.filePath(fileName)
	local lastIndex
	local sep = "/"
	local p = string.find(fileName, sep, 1)
	lastIndex = p
	while p do
		p = string.find(fileName, sep, p + 1)
		if p then
			lastIndex = p
		end
	end
	if lastIndex then
		return fileName:sub(1, lastIndex)
	end
	return nil
end

local allowedPath
-- clean ../ and ./ in path and check allowed path
---@return string
local function cleanPath(fileName, option)
	if type(fileName) ~= "string" then
		return fileName
	end
	if not peg.found(fileName, "./") then
		return fileName
	end
	if allowedPath == nil then
		allowedPath = util.mainPath()
		allowedPath = peg.parseBeforeLast(allowedPath, "/") -- nc/nc-server
		allowedPath = peg.parseBeforeLast(allowedPath, "/") .. "/" -- nc/
		if peg.found(allowedPath, "/nc/") then
			allowedPath = peg.parseBeforeLast(allowedPath, "/nc/") .. "/"
		end
	end
	if fileName:sub(1, 2) == "./" then
		fileName = util.mainPath() .. fileName
	elseif fileName:sub(1, 3) == "../" then
		fileName = util.mainPath() .. fileName
	end
	local pathArray = {}
	local parts = peg.splitToArray(fileName, "/")
	for _, part in ipairs(parts) do
		if part == ".." then
			table.remove(pathArray)
		elseif part ~= "." then
			table.insert(pathArray, part)
		end
	end
	local path = table.concat(pathArray, "/")
	if not peg.startsWith(path, allowedPath) then
		if option == nil or not peg.found(option, "no-error") then
			util.printError("path '%s' is not allowed, allowed path is '%s'", path, allowedPath)
		end
		return ""
	end
	return path
end

---@return string
function fs.filePathFix(fileName, option)
	if fileName == nil then
		return ""
	end
	if filePathChange == nil then
		filePathChange = false -- prevent recursive loop
		if option == nil or not peg.found(option, "no-db") then -- fix only ../nc-plugin -paths if directory ../nc-plugin exists
			local dprf = require "dprf" -- we must prevent loading of dprf on start before useCoro is set
			filePathChange = dprf.getFile("system/path.json", option)
			local pathChange
			local pathChangeUsed = "path_change"
			local pluginPath = "../nc-plugin"
			local attr = lfs.attributes(pluginPath)
			if not (attr and attr.mode == "directory") then
				pluginPath = util.mainPath() .. "../nc-plugin"
				attr = lfs.attributes(pluginPath)
			end
			if attr and attr.mode == "directory" then
				pathChange = filePathChange.path_change_dev
				pathChangeUsed = "path_change_dev"
			else
				pluginPath = peg.replace(util.mainPath(), "/nc-server/", "/nc-server-dist/") .. "../nc-plugin"
				attr = lfs.attributes(pluginPath)
				if attr and attr.mode == "directory" then
					pathChange = filePathChange.path_change_dev
					pathChangeUsed = "path_change_dev"
				elseif filePathChange then
					pathChange = filePathChange.path_change
				end
			end
			if not pathChange then
				filePathChange = false
				util.printRed("system/path.json is missing or path_change is missing from system/path.json")
			else
				local commonPath = pathChange.common
				local osPath = pathChange[util.os()]
				filePathChange = commonPath or false
				if commonPath then
					osPath = osPath or {}
					filePathChange = osPath
					local osPathIdx = fn.util.createIndex(osPath, "from")
					for _, item in ipairs(commonPath) do
						for _, osItem in ipairs(osPath) do
							if peg.startsWith(item.to, osItem.from) then
								item.to = peg.replaceFirst(item.to, osItem.from, osItem.to)
							end
						end
						if not osPathIdx[item.from] then
							filePathChange[#filePathChange + 1] = item -- overwrite commonPath
						end
					end
				end
				util.print("* using path change: system/path.json.%s, first change: %s", tostring(pathChangeUsed), json.toJsonRaw(filePathChange[1] or {}))
			end
		end
	end
	if filePathChange then
		local fileName2
		for _, item in ipairs(filePathChange) do
			if startsWith(fileName:lower(), item.from:lower()) then
				fileName2 = item.to .. fileName:sub(#item.from + 1)
				if fileName2:sub(-1) == "/" then
					fileName2 = fileName2:sub(1, -2) -- lfs.attributes() does not work with trailing path in Windows
				end
				if lfs.attributes(fileName2) then
					fileName = fileName2
					break
				end
			end
		end
	end
	if util.isWin() then
		if winDriveLetter == nil then
			local attr = lfs.attributes("Y:")
			if attr and attr.mode == "directory" then
				winDriveLetter = "Y:"
			else
				winDriveLetter = "C:"
			end
		end
		if util.isWine() and fileName:sub(1, 1) == "~" then
			fileName = replace(fileName, "~", "Z:/Users/Pasi/")
		else
			fileName = replaceFirst(fileName, "~", winDriveLetter)
		end
		if fileName:sub(1, 1) == "/" then -- unix to win
			fileName = winDriveLetter .. fileName
		end
		if option == nil or not peg.found(option, "no-char-conversion") then
			return utf.utf8ToLatin9(cleanPath(fileName, option))
		end
		return cleanPath(fileName, option)
	elseif fileName:sub(1, 1) == "~" then
		fileName = replace(fileName, "~", util.homeDirectory())
		if util.isLinux() then
			fileName = replace(fileName, "/users/", "/home/")
		end
	end
	return cleanPath(fileName, option)
end

local function filePathFix(path)
	if path == nil then
		return nil
	end
	local path2 = fs.filePathFix(path, "no-char-conversion")
	if path2:sub(-1) == "/" and path2:sub(-2) ~= ":/" then -- lfs does not handle last /, but windows paths like C:/ must be allowed
		path2 = removeFromEnd(path2, "/")
	end
	if util.isWine() then
		path2 = replace(path2, "C:/", "Z:/")
	end
	return path2
end

function fs.mainPath()
	return util.mainPath()
end

function fs.fixSaveFileName(filename)
	-- https://stackoverflow.com/questions/1976007/what-characters-are-forbidden-in-windows-and-linux-directory-names
	--[[< (less than)
			> (greater than)
			: (colon - sometimes works, but is actually NTFS Alternate Data Streams)
			" (double quote)
			/ (forward slash)
			\ (backslash)
			| (vertical bar or pipe)
			? (question mark)
			* (asterisk)
	]]
	filename = filePathFix(filename)
	-- filename = replace(filename, "-", "_") -- this is allowed and ok
	filename = replace(filename, ":", "_") -- extra: old macintosh, 4D
	filename = replace(filename, "<", "_")
	filename = replace(filename, ">", "_")
	filename = replace(filename, '"', "_") -- should we do also "'"?
	filename = replace(filename, "/", "_")
	filename = replace(filename, "\\", "_")
	filename = replace(filename, "|", "_")
	filename = replace(filename, "?", "_")
	return replace(filename, "*", "_")
end

function fs.attributes(fs_)
	return lfs.attributes(fs_)
end

function fs.symlinkattributes(fs_)
	return lfs.symlinkattributes(fs_)
end

local function fileSize(path)
	path = filePathFix(path)
	local size = lfs.attributes(path, "size") -- returns nil or size in bytes
	return size or 0 -- returns size in bytes or 0
end
fs.fileSize = fileSize

function fs.openFile(fileName, delay)
	fileName = fs.filePathFix(fileName)
	if util.isWin() then
		fileName = fileName:gsub("/", "\\")
		os.execute('start "" "' .. fileName .. '"') -- second param is win title, "" here
	else
		if not found(fileName, '"') then
			fileName = '"' .. fileName .. '"'
		end
		local cmd
		if util.isLinux() then
			if delay then
				cmd = "/bin/bash -c 'sleep " .. delay .. " && xdg-open " .. fileName .. "'&"
			else
				cmd = "xdg-open " .. fileName
			end
		else
			if delay then
				cmd = "/bin/bash -c 'sleep " .. delay .. " && open " .. fileName .. "'&" -- run as background task
			else
				cmd = "open " .. fileName
			end
		end
		-- print("fs.openFile: "..cmd)
		os.execute(cmd)
	end
end

function fs.getFile(fileName, option, hashTbl)
	return util.getFile(fileName, nil, option, hashTbl)
end

local lastReadFilePath
function fs.lastReadFilePath()
	return lastReadFilePath
end

local prevPath
function fs.readFile(fileName, mode, option)
	-- print("readFile: ", fileName, data, util.callPath())
	if fileName == nil then
		return nil
	end
	fileName = fs.filePathFix(fileName, option)
	if fileName == "" then
		return nil
	end
	if mode and mode == "binary" then
		mode = "rb"
	else
		mode = "r"
	end
	local filePtr = io.open(fileName, mode)
	if filePtr == nil and not peg.found(fileName, ":/") then
		local fileName2 = util.mainPath() .. fileName
		filePtr = io.open(fileName2, mode)
		if filePtr == nil then
			fileName2 = util.pluginPath() .. fileName
			filePtr = io.open(fileName2, mode)
		end
		fileName = fileName2
	end
	if filePtr == nil then
		return nil
	end
	local size = fileSize(fileName)
	if size > maxReadFileSize then
		util.printRed("file '%s' was not read because it's size %.2f Mb is bigger than maximum allowed read file size %.2f Mb", fileName, size / 1024 / 1024, maxReadFileSize)
		return nil
	end
	local fileData = filePtr:read("*all")
	filePtr:close()
	if prevPath ~= fileName and fileName:sub(-5) == ".json" and fileData:sub(-5) == ".json" then -- is symbolic link
		prevPath = fileData -- prevent infinite loop
		fileName = fileData
		fileData = fs.readFile(fileName, mode, option)
		prevPath = nil
	end
	lastReadFilePath = fileName
	return fileData -- , fileName -- , filePtr
end

function fs.readJsonFile(fileName, option)
	local ret = fs.readFile(fileName, option)
	if ret and ret ~= "" then
		if option and option == "no-error" then
			return json.fromJson(ret, fileName, false)
		end
		return json.fromJson(ret, fileName)
	end
	-- return {}
end

function fs.writeFile(fileName, data, log)
	if not data then
		local err = l("data is nil")
		util.printError(err)
		return nil, err
	end
	-- print("writeFile: ", fileName, data, util.callPath())
	fileName = fs.filePathFix(fileName)
	local f = io.open(fileName, "wb")
	if f == nil and fileName:sub(1, 1) ~= "/" then
		local fileName2 = util.mainPath() .. fileName
		f = io.open(fileName2, "wb")
		if f == nil then
			fileName2 = util.preferencePath() .. fileName
			f = io.open(fileName2, "wb")
		end
		if f == nil then
			fileName2 = util.pluginPath() .. fileName
			f = io.open(fileName2, "wb")
		end
		if f == nil then
			fileName2 = util.backendPath() .. fileName
			f = io.open(fileName2, "wb")
		end
		if f then
			fileName = fileName2
		end
	end
	if f == nil then
		local err = string.format("write fs to path '%s' failed", tostring(fileName))
		util.printError(err)
		return nil, err
	end
	if false and loc.prettierExists == nil then
		loc.defaultKeyNameArr = json.defaultKeyNameArr()
		loc.defaultKeyNameArr = table.concat(loc.defaultKeyNameArr, ",")
		local ret = fs.runCommandLineReturn("prettier-package-json --version")
		util.printInfo("prettier-package-json version: '%s'", ret)
		local prettierVersion = tonumber(ret:sub(1, 1))
		loc.prettierExists = prettierVersion and prettierVersion >= 1
	end
	if type(data) == "table" then
		-- loadJson()
		if loc.prettierExists then
			data = json.toJson(data)
		else
			data = json.toJsonKeySorted(data)
		end
	end
	f:write(data)
	f:close()
	if loc.prettierExists then
		loc.defaultKeyNameArr = ""
		local cmd = "prettier-package-json --use-tabs --write '" .. fileName .. "'"
		local ret = fs.runCommandLineReturn(cmd) -- successful command does not return any errors
		if ret ~= "" then
			local err = util.printError("prettier command \"%s\" error '%s'", cmd, ret)
			return nil, err
		end
	end
	if log then
		util.printOk("saved file '%s' to disk", fileName)
	end
	return fileName
end

function fs.runCommandLineReturn(command, option)
	local tmpFile = os.tmpname() -- util.mainPath().."tmp.txt" -- os.tmpname() -- get a temporary file name
	os.execute(command .. " > " .. tmpFile) -- execute and redirect result to tmpFile
	local ret = fs.readFile(tmpFile)
	os.remove(tmpFile) -- remove temporary file
	if option ~= "allow-linefeed" then
		ret = ret:gsub("\r", "")
		ret = ret:gsub("\n", "")
	end
	return ret
end

function fs.runCommandLine(command)
	return util.runCommandLine(command)
end

function fs.appendFile(fileName, data)
	fileName = fs.filePathFix(fileName)
	local f = io.open(fileName, "a+b")
	f:write(data)
	f:close()
	return fileName
end

function fs.upperPath(path)
	if path:sub(-1) == "/" then
		path = path:sub(1, -2)
	end
	return util.removeEndPart(path, "/") .. "/" -- util.upperPath() does not add "/" to the end
end

function fs.createPath(path)
	local exe
	path = filePathFix(path) .. "/"
	if util.isWin() then
		exe = 'mkdir "' .. replace(path, "/", "\\") .. '"' -- https://www.computerhope.com/mdhlp.htm
		-- MKDIR creates any intermediate directories in the path, if needed
	else
		exe = 'mkdir -p "' .. path .. '"'
	end
	local ret, err = os.execute(exe) -- todo: add ret checking and print warnings
	if not (found(err, "already exists") or (util.isWin() and ret == nil)) then
		executeError(ret, err)
	end
	return ret
end

function fs.move(from, to)
	local exe
	from = filePathFix(from)
	to = filePathFix(to)
	if util.isWin() then
		exe = 'move "' .. from .. '" "' .. to .. '"'
		exe = replace(exe, "/", "\\")
	else
		exe = 'mv "' .. from .. '" "' .. to .. '"'
	end
	local ret, err = os.execute(exe) -- todo: add ret checking and print warnings
	executeError(ret, err)
	return ret
end

function fs.copyPath(from, to)
	local exe
	from = filePathFix(from) -- .."/"
	to = filePathFix(to) -- .."/"
	if util.isWin() then
		exe = 'xcopy "' .. from .. '" "' .. to .. '"' -- https://www.computerhope.com/xcopyhlp.htm
		exe = replace(exe, "/", "\\") .. " /E /I /Q"
	else
		exe = 'cp -r "' .. from .. '" "' .. to .. '"'
	end
	local ret, err = os.execute(exe) -- todo: add ret checking and print warnings
	executeError(ret, err)
	return ret
end

function fs.movePath(from, to)
	from = filePathFix(from)
	to = filePathFix(to)
	fs.copyPath(from, to)
	lfs.rmdir(from)
end

function fs.currentPath()
	return replace(lfs.currentdir(), "\\", "/") .. "/"
	--[[ -- something strange ins osx, c-lpeg replace returns ""
	local dir = lfs.currentdir()
	if found(dir, "\\") then
		return replace(lfs.currentdir(), "\\", "/").."/"
	end
	-- local path = replace(dir, "\\", "/")
	-- print(path)
	return dir.."/"
	]]
end

function fs.exists(path, printWarning)
	local err
	if not path then
		err = "path is missing" -- l"path is missing"
	elseif path == "" then
		err = "path is empty" -- l"path is empty"
	end
	if not err then
		path = filePathFix(path)
		local attr
		attr, err = lfs.attributes(path)
		if attr then -- and attr.mode == "file" or attr.mode == "directory" then -- or attr.mode == "other" then
			return true
		else
			err = string.format("path '%s' is not valid, attribute '%s', error '%s'", tostring(path), tostring(attr and json.toJsonRaw(attr)), tostring(err)) -- l("path '%s' is not valid", tostring(path))
		end
	end
	if printWarning then
		if found(printWarning, "warning") then
			util.printWarning(err)
		elseif found(printWarning, "error") then
			util.printError(err)
		elseif found(printWarning, "info") then
			util.printInfo(err)
		end
	end
	return false, err
end

function fs.fileExists(path, printWarning)
	local err
	if not path then
		err = "path is missing" -- l"path is missing"
	elseif path == "" then
		err = "path is empty" -- l"path is empty"
	end
	if not err then
		path = filePathFix(path)
		local attr
		attr, err = lfs.attributes(path)
		if attr and attr.mode == "file" then -- or attr.mode == "other" then
			return true
		else
			err = string.format("file path '%s' is not valid, attribute '%s', error '%s'", tostring(path), tostring(attr and json.toJsonRaw(attr)), tostring(err)) -- l("path '%s' is not valid", tostring(path))
		end
	end
	if printWarning then
		if found(printWarning, "warning") then
			util.printWarning(err)
		elseif found(printWarning, "error") then
			util.printError(err)
		elseif found(printWarning, "info") then
			util.printInfo(err)
		end
	end
	return false, err
end

--[[ -- not in use yet
function fs.tempDirPath()
	local path = filePathFix("~/MA/")
	fs.createFilePath(path)
	return path
end
--]]

--[[
function fs.deletePath(path)
	path = filePathFix(path)
	local attr, err = lfs.attributes(path)
	if attr then -- path exists
		if attr.mode ~= "directory" then
			util.printWarning(l("path '%s' is not a directory", tostring(path))
		else
			os.remove(path)
		end
	end
end
]]

function fs.deletePathContentRecursive(path, option)
	path = filePathFix(path)
	local attr, err = lfs.attributes(path)
	if err then
		err = string.format("directory path '%s' is not valid, error '%s'", tostring(path), tostring(err))
		util.printError(err)
	end
	if attr then -- path exists
		if attr.mode ~= "directory" then
			err = string.format("directory path '%s' is not valid directory", tostring(path))
			util.printWarning(err)
		else
			for fileName in lfs.dir(path) do
				if fileNameAllowed(fileName) then
					local filePath = path .. '/' .. fileName
					attr, err = lfs.attributes(filePath)
					if err then
						err = string.format("file path '%s' is not valid, error '%s'", tostring(filePath), tostring(err))
						util.printError(err)
					end
					if attr.mode == "directory" then
						fs.deletePathContentRecursive(filePath)
						lfs.rmdir(filePath)
					else
						os.remove(filePath)
					end
				end
			end
			if option == "delete-main-path" then
				lfs.rmdir(path)
			end
		end
	end
	return err
end

function fs.deletePath(path)
	-- return fs.deletePathContentRecursive(path, "delete-main-path")
	local exe
	path = filePathFix(path) .. "/"
	if not fs.dirExists(path) then
		return
	end
	if util.isWin() then
		exe = 'rmdir /S /Q "' .. replace(path, "/", "\\") .. '"' -- https://www.computerhope.com/rmdirhlp.htm
	else
		exe = 'rm -rf "' .. path .. '"'
	end
	local ret, err = os.execute(exe)
	executeError(ret, err)
	return ret
end

function fs.dirExists(dir, printWarning)
	local err
	if not dir then
		err = "path is missing" -- l"path is missing" -- l causes infinite loop
	elseif dir == "" then
		err = "path is empty" -- l"path is empty"
	end
	if not err then
		dir = filePathFix(dir)
		local attr
		attr, err = lfs.attributes(dir)
		if attr == nil and util.isWin() and #dir == 2 and dir:sub(2, 2) == ":" then
			attr, err = lfs.attributes(dir .. "/") -- win C:/ must have / at the end
		end
		if attr and (attr.mode == "directory" or attr.mode == "other") then -- attr.mode == "other" in Debian, (attr.mode == "fs")
			return true
		end
		err = string.format("directory path '%s' is not valid, attribute '%s', error '%s'", tostring(dir), tostring(attr and json.toJsonRaw(attr)), tostring(err)) -- l("path '%s' is not valid", tostring(dir))
	end
	if printWarning then
		if found(printWarning, "warning") then
			util.printWarning(err)
		elseif found(printWarning, "error") then
			util.printError(err)
		elseif found(printWarning, "info") then
			util.printInfo(err)
		end
	end
	return false, err
end

function fs.createFilePath(path)
	if not lfs then
		lfs = require "lfs_load"
	end
	path = filePathFix(path)
	if endsWith(path, ".json") then
		path = fs.upperPath(path)
	end
	local attr = lfs.attributes(path)
	if not attr then -- path does not exist
		lfs.mkdir(path)
		attr = lfs.attributes(path)
		if not attr then -- path does not exist -> create lower paths
			local pathTbl = peg.splitToArray(path, "/")
			local makePath = ""
			for i, folder in ipairs(pathTbl) do
				if i > 1 then
					makePath = makePath .. "/"
				end
				makePath = makePath .. folder
				if makePath ~= "" and i > 1 then -- do not make root-folder: "C:"
					attr = lfs.attributes(makePath)
					if not attr then -- path does not exist
						lfs.mkdir(makePath)
					end
				end
			end
		end
	end
	--[[
	local attr = lfs.attributes(path)
	if not attr then  -- path does not exist
		lfs.mkdir(path)
	end
	]]
end

function fs.deleteFile(path, fileName, recursive, option)
	local err
	if fileName == nil then
		fileName = filePathFix(path)
		path = ""
	else
		fileName = filePathFix(fileName)
	end
	if path ~= "" and fileName and fileName:find("*", 1, true) then
		local dirPath = lfs.dir(path)
		for dir in dirPath:entries() do
			local name = dir.name
			if dir.mode == "directory" and recursive and name:sub(1, 1) ~= "." then
				fs.deleteFile(path .. name .. "/", fileName, recursive)
			elseif dir.mode == "fs" then
				-- http://peterodding.com/code/lua/lfs/docs/#lfs.fnmatch
				local match_ = lfs.fnmatch(fileName, name, true)
				if match_ then
					local _
					_, err = os.remove(path .. name)
					if err and option ~= "no-warning" then
						-- util.printWarning(l("delete of fs '%s' failed with error '%s': ", path..name, err))
						util.printWarning(string.format("delete of fs '%s' failed with error '%s': ", path .. name, err))
					end
				end
			end
		end
	else
		local name = path .. (fileName or "")
		if path ~= "" and fileName ~= "" then
			name = path .. "/" .. (fileName or "")
		end
		--[[if not found('"', name) then
			name = '"'..name..'"'
		end]]
		local _
		_, err = os.remove(name)
		if err and option ~= "no-warning" then
			-- util.printWarning(l("delete of fs '%s' failed with error '%s': ", fileName, err))
			util.printWarning(string.format("delete of fs '%s' failed with error '%s': ", name, err))
		end
	end
	return err
end

function fs.copyFile(pathFrom, pathTo)
	pathFrom = filePathFix(pathFrom)
	pathTo = filePathFix(pathTo)
	local exe
	if util.isWin() then
		pathFrom = pathFrom:gsub("/", "\\")
		pathTo = pathTo:gsub("/", "\\")
		exe = 'COPY "' .. pathFrom .. '" "' .. pathTo .. '"'
	else
		exe = 'cp "' .. pathFrom .. '" "' .. pathTo .. '"'
	end
	local ret, err = os.execute(exe) -- todo: add ret checking and print warnings
	executeError(ret, err)
	return ret
end

-- http://lua-users.org/wiki/DirTreeIterator
function fs.dirTreeIter(dirParam, option) -- old: depth, suffix, disallowSuffix, directory (allow directory)
	-- {suffix = ".json"}
	-- {suffix = {".json", ".lua"}}
	local maxLowerDepth, allowedSuffix, notAllowedSuffix, allowDirectory = option.depth, option.suffix, option.disallowSuffix, option.directory -- followSymlinks
	local dirOrig = filePathFix(dirParam)
	local dirExist, err = fs.dirExists(dirOrig)
	if not dirExist then
		util.printWarning(err)
		return function()
			return nil
		end
	end
	local func, cdata = lfs.dir(dirOrig)
	local dirIter = {{func = func, iter = cdata}}
	local dirs = {}
	local allowedSuffixIsString = type(allowedSuffix) == "string"
	local allowedSuffixIsTable = type(allowedSuffix) == "table"
	local notAllowedSuffixIsString = type(notAllowedSuffix) == "string"
	local notAllowedSuffixIsTable = type(notAllowedSuffix) == "table"
	if allowDirectory == nil then
		allowDirectory = false
		if allowedSuffix == nil then -- allowedSuffix can be nil
			allowDirectory = true
		elseif allowedSuffixIsString then -- allowedSuffix can be nil
			allowDirectory = allowedSuffix == "directory"
		elseif allowedSuffixIsTable then
			for _, rec in ipairs(allowedSuffix) do
				if rec == "directory" then
					allowDirectory = true
					break
				end
			end
		end
	end

	local function testYield(entry, isDirectory)
		local doYield = true
		-- allowedSuffix is not valid for directories
		-- notAllowedSuffix is valid for directories
		if not isDirectory and allowedSuffixIsString then -- allowedSuffix can be nil
			doYield = endsWith(entry, allowedSuffix)
		elseif not isDirectory and allowedSuffixIsTable then
			doYield = false
			for _, val in ipairs(allowedSuffix) do
				doYield = allowedSuffix == val or endsWith(entry, val)
				if doYield == true then
					break
				end
			end
		elseif notAllowedSuffixIsString then -- notAllowedSuffix can be nil
			doYield = endsWith(entry, notAllowedSuffix) == false
		elseif notAllowedSuffixIsTable then
			doYield = true
			for _, val in ipairs(notAllowedSuffix) do
				doYield = endsWith(entry, val) == false
				if doYield == false then
					break
				end
			end
		end
		return doYield
	end

	return function()
		local subPath, fileName, attr
		repeat
			local iter = dirIter[#dirIter]
			local entry = iter.func(iter.iter)
			if entry and entry ~= "" then
				if fileNameAllowed(entry) then -- entry ~= "." and entry ~= ".." then
					-- $ is used in win, like $Recycle.Bin
					if #dirs > 0 then
						subPath = table.concat(dirs, "/") .. "/" .. entry
					else
						subPath = entry
					end
					fileName = dirOrig .. "/" .. subPath
					attr, err = lfs.attributes(fileName)
					if attr == nil then
						if util.isWin() and not (peg.endsWith(fileName, ".sys") or peg.endsWith(fileName, ".Msi") or peg.found(fileName, "DumpStack")) then
							local linkAttr = lfs.symlinkattributes(fileName)
							if not linkAttr or linkAttr.mode ~= "link" then
								util.printError("dirTreeIter path is not valid for fs: '%s', error '%s'", tostring(fileName), tostring(err))
							end
						end
					else
						if attr and attr.mode == "file" then
							local doYield
							if allowDirectory == "directory" then
								doYield = false
							else
								doYield = testYield(entry)
							end
							if doYield then
								-- if not doYield then continue repeat and iter to next fs, else return to caller
								return subPath, attr, #dirs
							end
						else -- must be directory
							if maxLowerDepth == nil or #dirs < maxLowerDepth then
								if testYield(entry, "directory") then
									func, cdata = lfs.dir(fileName)
									table.insert(dirIter, {func = func, iter = cdata})
									table.insert(dirs, entry)
									if allowDirectory then
										-- if not doYield then repeat and iter to next fs, else return to caller
										return subPath, attr, #dirs
									end
								end
							else
								if allowDirectory then
									if testYield(entry) then
										-- if not doYield then repeat and iter to next fs, else return to caller
										return subPath, attr, #dirs
									end
								end
							end
						end
					end
				end
			else
				table.remove(dirIter)
				table.remove(dirs)
			end
		until #dirIter == 0
	end
end

--[=[
-- http://lua-users.org/wiki/DirTreeIterator
function fs.dirTreeIterCoroutine(dir, depth, suffix)
	dir = filePathFix(dir)
	-- if endsWith(filePathName, suffix) then
	--[[ example usage:
		for filename, attr in fs.dirTreeIter(".") do
			print(attr.mode, filename)
		end
	]]
  if dir:sub(-1) == "/" then
    dir = dir:sub(1, -2)
  end
	local level = -1 -- first call to yieldTree() will set it to 0 in level = level + 1

	local dirExist, err = fs.dirExists(dir)
  if not dirExist then
		util.printWarning(err)
		return function() return nil end
	end

  local function yieldTree(dir, subPath, nextPath)
		level = level + 1
		if nextPath ~= "" then
			dir = dir.."/"..nextPath
			subPath = subPath..nextPath.."/"
		end
		-- print(" * yieldTree: "..level, depth, dir, subPath)
		for entry in lfs.dir(dir) do
			if entry:sub(1, 1) ~= "." then -- entry ~= "." and entry ~= ".." then
				local attr = lfs.attributes(dir.."/"..entry) or {}
				local doYield = true
				if type(suffix) == "string" then
					doYield = endsWith(entry, suffix)
				elseif type(suffix) == "table" then
					doYield = false
					for _, val in ipairs(suffix) do
						doYield = endsWith(entry, val)
						if doYield == true then
							break
						end
					end
				end
				if doYield then
					coroutine.yield(subPath..entry, attr, level)
				end
				if (depth == nil or level < depth) and attr.mode == "directory" then
					yieldTree(dir, subPath, entry)
				end
			end
		end
		level = level - 1
	end

	--[[
	add require('mobdebug').on() call to that coroutine, which will enable debugging for that particular coroutine, or
	add require('mobdebug').coro() call to your script, which will enable debugging for all coroutines created using coroutine.create later in the script.
	--]]
  return coroutine.wrap(function() yieldTree(dir, "", "") end)
end
--]=]

return fs
