--- hardware.lua
-- test hardware in runtime, https://github.com/anrieff/libcpuid
-- https://gcc.gnu.org/onlinedocs/gcc-4.9.2/gcc/X86-Built-in-Functions.html
local hardware = {}

local util -- = require "util"
local ffi = require "mffi"
local C = ffi.C
local peg = require "peg"
local features = nil
local cpuid

ffi.cdef [[ // hardware
int cpuinfo(char* buffer, int maxLen);
]]

ffi.cdef [[ // cpuid
static const int VENDOR_STR_MAX = 16;
static const int BRAND_STR_MAX = 64;
static const int CPU_FLAGS_MAX	 = 128;
static const int MAX_CPUID_LEVEL	 = 32;
static const int MAX_EXT_CPUID_LEVEL = 32;
static const int MAX_INTELFN4_LEVEL = 8;
static const int MAX_INTELFN11_LEVEL = 4;
static const int MAX_INTELFN12H_LEVEL = 4;
static const int MAX_INTELFN14H_LEVEL = 4;
static const int CPU_HINTS_MAX	 = 16;
static const int SGX_FLAGS_MAX	 = 14;

typedef enum {
	CPU_FEATURE_FPU = 0,	/*!< Floating point unit */
	CPU_FEATURE_VME,	/*!< Virtual mode extension */
	CPU_FEATURE_DE,		/*!< Debugging extension */
	CPU_FEATURE_PSE,	/*!< Page size extension */
	CPU_FEATURE_TSC,	/*!< Time-stamp counter */
	CPU_FEATURE_MSR,	/*!< Model-specific regsisters, RDMSR/WRMSR supported */
	CPU_FEATURE_PAE,	/*!< Physical address extension */
	CPU_FEATURE_MCE,	/*!< Machine check exception */
	CPU_FEATURE_CX8,	/*!< CMPXCHG8B instruction supported */
	CPU_FEATURE_APIC,	/*!< APIC support */
	CPU_FEATURE_MTRR,	/*!< Memory type range registers */
	CPU_FEATURE_SEP,	/*!< SYSENTER / SYSEXIT instructions supported */
	CPU_FEATURE_PGE,	/*!< Page global enable */
	CPU_FEATURE_MCA,	/*!< Machine check architecture */
	CPU_FEATURE_CMOV,	/*!< CMOVxx instructions supported */
	CPU_FEATURE_PAT,	/*!< Page attribute table */
	CPU_FEATURE_PSE36,	/*!< 36-bit page address extension */
	CPU_FEATURE_PN,		/*!< Processor serial # implemented (Intel P3 only) */
	CPU_FEATURE_CLFLUSH,	/*!< CLFLUSH instruction supported */
	CPU_FEATURE_DTS,	/*!< Debug store supported */
	CPU_FEATURE_ACPI,	/*!< ACPI support (power states) */
	CPU_FEATURE_MMX,	/*!< MMX instruction set supported */
	CPU_FEATURE_FXSR,	/*!< FXSAVE / FXRSTOR supported */
	CPU_FEATURE_SSE,	/*!< Streaming-SIMD Extensions (SSE) supported */
	CPU_FEATURE_SSE2,	/*!< SSE2 instructions supported */
	CPU_FEATURE_SS,		/*!< Self-snoop */
	CPU_FEATURE_HT,		/*!< Hyper-threading supported (but might be disabled) */
	CPU_FEATURE_TM,		/*!< Thermal monitor */
	CPU_FEATURE_IA64,	/*!< IA64 supported (Itanium only) */
	CPU_FEATURE_PBE,	/*!< Pending-break enable */
	CPU_FEATURE_PNI,	/*!< PNI (SSE3) instructions supported */
	CPU_FEATURE_PCLMUL,	/*!< PCLMULQDQ instruction supported */
	CPU_FEATURE_DTS64,	/*!< 64-bit Debug store supported */
	CPU_FEATURE_MONITOR,	/*!< MONITOR / MWAIT supported */
	CPU_FEATURE_DS_CPL,	/*!< CPL Qualified Debug Store */
	CPU_FEATURE_VMX,	/*!< Virtualization technology supported */
	CPU_FEATURE_SMX,	/*!< Safer mode exceptions */
	CPU_FEATURE_EST,	/*!< Enhanced SpeedStep */
	CPU_FEATURE_TM2,	/*!< Thermal monitor 2 */
	CPU_FEATURE_SSSE3,	/*!< SSSE3 instructionss supported (this is different from SSE3!) */
	CPU_FEATURE_CID,	/*!< Context ID supported */
	CPU_FEATURE_CX16,	/*!< CMPXCHG16B instruction supported */
	CPU_FEATURE_XTPR,	/*!< Send Task Priority Messages disable */
	CPU_FEATURE_PDCM,	/*!< Performance capabilities MSR supported */
	CPU_FEATURE_DCA,	/*!< Direct cache access supported */
	CPU_FEATURE_SSE4_1,	/*!< SSE 4.1 instructions supported */
	CPU_FEATURE_SSE4_2,	/*!< SSE 4.2 instructions supported */
	CPU_FEATURE_SYSCALL,	/*!< SYSCALL / SYSRET instructions supported */
	CPU_FEATURE_XD,		/*!< Execute disable bit supported */
	CPU_FEATURE_MOVBE,	/*!< MOVBE instruction supported */
	CPU_FEATURE_POPCNT,	/*!< POPCNT instruction supported */
	CPU_FEATURE_AES,	/*!< AES* instructions supported */
	CPU_FEATURE_XSAVE,	/*!< XSAVE/XRSTOR/etc instructions supported */
	CPU_FEATURE_OSXSAVE,	/*!< non-privileged copy of OSXSAVE supported */
	CPU_FEATURE_AVX,	/*!< Advanced vector extensions supported */
	CPU_FEATURE_MMXEXT,	/*!< AMD MMX-extended instructions supported */
	CPU_FEATURE_3DNOW,	/*!< AMD 3DNow! instructions supported */
	CPU_FEATURE_3DNOWEXT,	/*!< AMD 3DNow! extended instructions supported */
	CPU_FEATURE_NX,		/*!< No-execute bit supported */
	CPU_FEATURE_FXSR_OPT,	/*!< FFXSR: FXSAVE and FXRSTOR optimizations */
	CPU_FEATURE_RDTSCP,	/*!< RDTSCP instruction supported (AMD-only) */
	CPU_FEATURE_LM,		/*!< Long mode (x86_64/EM64T) supported */
	CPU_FEATURE_LAHF_LM,	/*!< LAHF/SAHF supported in 64-bit mode */
	CPU_FEATURE_CMP_LEGACY,	/*!< core multi-processing legacy mode */
	CPU_FEATURE_SVM,	/*!< AMD Secure virtual machine */
	CPU_FEATURE_ABM,	/*!< LZCNT instruction support */
	CPU_FEATURE_MISALIGNSSE,/*!< Misaligned SSE supported */
	CPU_FEATURE_SSE4A,	/*!< SSE 4a from AMD */
	CPU_FEATURE_3DNOWPREFETCH,	/*!< PREFETCH/PREFETCHW support */
	CPU_FEATURE_OSVW,	/*!< OS Visible Workaround (AMD) */
	CPU_FEATURE_IBS,	/*!< Instruction-based sampling */
	CPU_FEATURE_SSE5,	/*!< SSE 5 instructions supported (deprecated, will never be 1) */
	CPU_FEATURE_SKINIT,	/*!< SKINIT / STGI supported */
	CPU_FEATURE_WDT,	/*!< Watchdog timer support */
	CPU_FEATURE_TS,		/*!< Temperature sensor */
	CPU_FEATURE_FID,	/*!< Frequency ID control */
	CPU_FEATURE_VID,	/*!< Voltage ID control */
	CPU_FEATURE_TTP,	/*!< THERMTRIP */
	CPU_FEATURE_TM_AMD,	/*!< AMD-specified hardware thermal control */
	CPU_FEATURE_STC,	/*!< Software thermal control */
	CPU_FEATURE_100MHZSTEPS,/*!< 100 MHz multiplier control */
	CPU_FEATURE_HWPSTATE,	/*!< Hardware P-state control */
	CPU_FEATURE_CONSTANT_TSC,	/*!< TSC ticks at constant rate */
	CPU_FEATURE_XOP,	/*!< The XOP instruction set (same as the old CPU_FEATURE_SSE5) */
	CPU_FEATURE_FMA3,	/*!< The FMA3 instruction set */
	CPU_FEATURE_FMA4,	/*!< The FMA4 instruction set */
	CPU_FEATURE_TBM,	/*!< Trailing bit manipulation instruction support */
	CPU_FEATURE_F16C,	/*!< 16-bit FP convert instruction support */
	CPU_FEATURE_RDRAND,     /*!< RdRand instruction */
	CPU_FEATURE_X2APIC,     /*!< x2APIC, APIC_BASE.EXTD, MSRs 0000_0800h...0000_0BFFh 64-bit ICR (+030h but not +031h), no DFR (+00Eh), SELF_IPI (+040h) also see standard level 0000_000Bh */
	CPU_FEATURE_CPB,	/*!< Core performance boost */
	CPU_FEATURE_APERFMPERF,	/*!< MPERF/APERF MSRs support */
	CPU_FEATURE_PFI,	/*!< Processor Feedback Interface support */
	CPU_FEATURE_PA,		/*!< Processor accumulator */
	CPU_FEATURE_AVX2,	/*!< AVX2 instructions */
	CPU_FEATURE_BMI1,	/*!< BMI1 instructions */
	CPU_FEATURE_BMI2,	/*!< BMI2 instructions */
	CPU_FEATURE_HLE,	/*!< Hardware Lock Elision prefixes */
	CPU_FEATURE_RTM,	/*!< Restricted Transactional Memory instructions */
	CPU_FEATURE_AVX512F,	/*!< AVX-512 Foundation */
	CPU_FEATURE_AVX512DQ,	/*!< AVX-512 Double/Quad granular insns */
	CPU_FEATURE_AVX512PF,	/*!< AVX-512 Prefetch */
	CPU_FEATURE_AVX512ER,	/*!< AVX-512 Exponential/Reciprocal */
	CPU_FEATURE_AVX512CD,	/*!< AVX-512 Conflict detection */
	CPU_FEATURE_SHA_NI,	/*!< SHA-1/SHA-256 instructions */
	CPU_FEATURE_AVX512BW,	/*!< AVX-512 Byte/Word granular insns */
	CPU_FEATURE_AVX512VL,	/*!< AVX-512 128/256 vector length extensions */
	CPU_FEATURE_SGX,	/*!< SGX extensions. Non-autoritative, check cpu_id_t::sgx::present to verify presence */
	CPU_FEATURE_RDSEED,	/*!< RDSEED instruction */
	CPU_FEATURE_ADX,	/*!< ADX extensions (arbitrary precision) */
	/* termination: */
	NUM_CPU_FEATURES,
} cpu_feature_t;

struct cpu_raw_data_t {
	uint32_t basic_cpuid[MAX_CPUID_LEVEL][4];
	uint32_t ext_cpuid[MAX_EXT_CPUID_LEVEL][4];
	uint32_t intel_fn4[MAX_INTELFN4_LEVEL][4];
	uint32_t intel_fn11[MAX_INTELFN11_LEVEL][4];
	uint32_t intel_fn12h[MAX_INTELFN12H_LEVEL][4];
	uint32_t intel_fn14h[MAX_INTELFN14H_LEVEL][4];
};

typedef enum {
	VENDOR_INTEL = 0,  /*!< Intel CPU */
	VENDOR_AMD,        /*!< AMD CPU */
	VENDOR_CYRIX,      /*!< Cyrix CPU */
	VENDOR_NEXGEN,     /*!< NexGen CPU */
	VENDOR_TRANSMETA,  /*!< Transmeta CPU */
	VENDOR_UMC,        /*!< x86 CPU by UMC */
	VENDOR_CENTAUR,    /*!< x86 CPU by IDT */
	VENDOR_RISE,       /*!< x86 CPU by Rise Technology */
	VENDOR_SIS,        /*!< x86 CPU by SiS */
	VENDOR_NSC,        /*!< x86 CPU by National Semiconductor */
	NUM_CPU_VENDORS,   /*!< Valid CPU vendor ids: 0..NUM_CPU_VENDORS - 1 */
	VENDOR_UNKNOWN = -1,
} cpu_vendor_t;

struct cpu_sgx_t {
	uint32_t present;
	uint8_t max_enclave_32bit;
	uint8_t max_enclave_64bit;
	uint8_t flags[SGX_FLAGS_MAX];
	int num_epc_sections;
	uint32_t misc_select;
	uint64_t secs_attributes;
	uint64_t secs_xfrm;
};

struct cpu_id_t {
	char vendor_str[VENDOR_STR_MAX];
	char brand_str[BRAND_STR_MAX];
	cpu_vendor_t vendor;
	uint8_t flags[CPU_FLAGS_MAX];
	int32_t family;
	int32_t model;
	int32_t stepping;
	int32_t ext_family;
	int32_t ext_model;
	int32_t num_cores;
	int32_t num_logical_cpus;
	int32_t total_logical_cpus;
	int32_t l1_data_cache;
	int32_t l1_instruction_cache;
	int32_t l2_cache;
	int32_t l3_cache;
	int32_t l4_cache;
	int32_t l1_assoc;
	int32_t l2_assoc;
	int32_t l3_assoc;
	int32_t l4_assoc;
	int32_t l1_cacheline;
	int32_t l2_cacheline;
	int32_t l3_cacheline;
	int32_t l4_cacheline;
	char cpu_codename[64];
	int32_t sse_size;
	uint8_t detection_hints[CPU_HINTS_MAX];
	struct cpu_sgx_t sgx;
};

const char* cpuid_error(void);
int cpuid_present(void);
int cpuid_get_raw_data(struct cpu_raw_data_t* data);
int cpu_identify(struct cpu_raw_data_t* raw, struct cpu_id_t* data);
int cpu_clock_by_os(void);
int cpu_clock_measure(int millis, int quad_check);
int sysctlbyname(const char *, void *, size_t *, void *, size_t);
]]

if ffi.os == "Windows" then -- util must be delay-loaded
	ffi.cdef [[
  typedef struct _OSVERSIONINFOW {
    ULONG dwOSVersionInfoSize;
    ULONG dwMajorVersion;
    ULONG dwMinorVersion;
    ULONG dwBuildNumber;
    ULONG dwPlatformId;
    WCHAR szCSDVersion[128];
  } OSVERSIONINFOW, *POSVERSIONINFOW, *LPOSVERSIONINFOW, RTL_OSVERSIONINFOW, *PRTL_OSVERSIONINFOW;

  typedef uint32_t NTSTATUS;

  NTSTATUS RtlGetVersion(
    PRTL_OSVERSIONINFOW lpVersionInformation
  );
  ]]
end

local function init() -- hardware.cpuSupports("sse4.2")
	if features then
		return features
	end

	util = require "util"
	local hw
	if util.isWin() then
		util.loadDll("graphics/libwinpthread-1.dll")
		if ffi.arch == "x64" then
			util.loadDll("graphics/libgcc_s_seh-1.dll")
		else
			-- ffi.loadMsvcr() --util.loadDll("msvcr71.dll") -- for cpuid
			util.loadDll("graphics/libgcc_s_sjlj-1.dll")
			util.loadDll("graphics/libgcc_s_dw2-1.dll")
		end
		util.loadDll("graphics/libstdc++-6.dll")
		hw = util.loadDll("hardware")
	elseif util.isMac() then
		hw = util.loadDll("hardware.dylib")
	else
		hw = util.loadDll("hardware.so")
	end
	if not hw then
		features = ""
		return
	end
	cpuid = util.loadDll("cpuid", "no-error") -- arm has no cpuid
	local buffSize = 512
	local buff = ffi.newNoAnchor("char[?]", buffSize)
	local ret = hw.cpuinfo(buff, buffSize)
	if ret > buffSize then
		util.printError("cpuinfo buffer size is too small, it  shoud be at least " .. tostring(ret))
	end
	--[[features = "vender:GenuineIntel idmax:13 step:9 model:58 family:6 type:0 fpu:1 mmx:1 sse:1 sse2:1 htt:1 ia64:0 sse3:1 vmx:1 ssse3:1 fma:0 sse4.1:1 sse4.2:1 popcnt:1 aes:1 avx:1 hypervisor:0 bmi1:0 avx2:0 bmi2:0 avx512f:0 avx512dq:0 avx512ifma:0 avx512pf:0 avx512er:0 avx512cd:0 sha:0 avx512bw:0 avx512vl:0 avx512vbmi:0 "]]
	features = ffi.string(buff, ret)

	--[[
	local handle = io.popen("sysctl machdep.cpu.features")
	--[=[machdep.cpu.features: FPU VME DE PSE TSC MSR PAE MCE CX8 APIC SEP MTRR PGE MCA CMOV PAT PSE36 CLFSH DS ACPI MMX FXSR SSE SSE2 SS HTT TM PBE SSE3 DTES64 MON DSCPL VMX EST TM2 SSSE3 CX16 TPR PDCM SSE4.1 XSAVE]=]
	features = handle:read("*a")
	handle:close()
	features = peg.replace(features, "", " ") -- last  to space
	features = peg.replace(features, "machdep.cpu.features: ", "")
	]]
end

function hardware.processIsTranslated()
	if cpuid then -- not arm
		return "native"
	end
	local ret = ffi.newNoAnchor("int[1]")
	local size = ffi.newNoAnchor("size_t[1]")
	local val = nil
	size[0] = ffi.sizeof("int");
	if (C.sysctlbyname("sysctl.proctranslated", ret, size, nil, 0) == -1) then
		if (ffi.errno() == C.ENOENT) then
			val = 0 -- native
		else
			val = -1 -- unknown
		end
	else
		val = ret[0] -- 1 == rosetta
	end
	if val == 0 then
		return "native"
	elseif val == 1 then
		return "rosetta"
	end
	return "unkown"
end

function hardware.cpuSupports(feature) -- hardware.cpuSupports("sse4.2")
	init()
	return peg.found(features, feature:lower() .. ":1 ")
end

function hardware.printInfo() -- hardware.cpuSupports("sse4.2")
	init()
	if cpuid == nil then
		return
	end
	if cpuid.cpuid_present() ~= 1 then
		util.printWarning("CPU id present: false")
	elseif jit then
		-- this code causes ffi crashes with Lua 5.1 ffi, use only with luajit
		local raw = ffi.newNoAnchor("struct cpu_raw_data_t[1]") -- contains only raw data
		local ret = cpuid.cpuid_get_raw_data(raw)
		if ret < 0 then -- obtain the raw CPUID data
			util.printWarning("Can not get the CPUID raw data, error '%s'", ffi.string(cpuid.cpuid_error()))
		end
		local data = ffi.newNoAnchor("struct cpu_id_t[1]")
		ret = cpuid.cpu_identify(raw, data)
		if ret < 0 then -- obtain the raw CPUID data
			util.printWarning("CPU identification failed, error '%s'", ffi.string(cpuid.cpuid_error()))
		end
		if ret >= 0 then
			local cpuInfo = string.format("CPU %s, %s", ffi.string(data[0].brand_str), ffi.string(data[0].cpu_codename))
			-- print out the CPU brand string, CPU code name (e.g. `Pentium 4 (Northwood)')
			-- print out the vendor string (e.g. `GenuineIntel'): xffi.string(data[0].vendor_str)
			-- [[
			cpuInfo = cpuInfo
					          .. string.format("\n  - %d cores and %d logical processors, %dK L1 cache and %dK L2 cache", data[0].num_cores,
					                           data[0].num_logical_cpus, data[0].l1_data_cache, data[0].l2_cache) -- print out CPU cores information, cache size information
			--[[ Here we basically exploit the data and print out some stuff. We can't really assume we have all the data filled in - but if something is not recognized (e.g. cache size is not filled in), there will be safe "uninitialized" values for the fields (e.g. empty strings, -1 for cache sizes, etc.).
			]]
			--[[util.printInfo("Supported multimedia instruction sets:")
			util.printInfo("  MMX         : %s", data[0].flags[C.CPU_FEATURE_MMX] > 0 and "present" or "absent")
			util.printInfo("  MMX-extended: %s", data[0].flags[C.CPU_FEATURE_MMXEXT] > 0 and  "present" or "absent")
			util.printInfo("  SSE         : %s", data[0].flags[C.CPU_FEATURE_SSE] > 0 and  "present" or "absent")
			util.printInfo("  SSE2        : %s", data[0].flags[C.CPU_FEATURE_SSE2] > 0 and  "present" or "absent")
			util.printInfo("  3DNow!      : %s", data[0].flags[C.CPU_FEATURE_3DNOW] > 0 and  "present" or "absent")
			]]

			--[[ We check for presence of specific CPU features. There are two points to be made here:
				If you run the above code on an Intel CPU, it will report absence of the MMX-extended set, even if it has SSE. Actually, the MMX-extended is a proper subset of the SSE (these are the integer SSE instructions, which extend the original MMX set - added by Intel with SSE and also supported by some AMD CPUs, which do not otherwise support the full SSE set). This means, that, for Intel CPUs, presence of the SSE set (or later) implies the presence of MMX-ext. The general rule is that libcpuid reports just what the CPU says - it doesn't try to be smart in cases like this.
				If you need this kind of feature detection, it is recommended to execute cpu_identify() only once and save the results somewhere; calling cpu_identify() or cpuid_get_raw_data() too many times (e.g. several hundred times per second) can hinder performance significantly.
			]]

			-- cpuInfo = cpuInfo..string.format("\n  - %d MHz (according to your OS), %d MHz (tested)", cpuid.cpu_clock_by_os(), cpuid.cpu_clock_measure(200, 0)) -- print out the CPU clock, according to the OS, -- print out the CPU clock, measured with RDTSC.
			cpuInfo = cpuInfo .. string.format("\n  - %d MHz (according to your OS)", cpuid.cpu_clock_by_os()) -- print out the CPU clock, according to the OS, -- print out the CPU clock, measured with RDTSC.
			util.printInfo(cpuInfo)
		end
	end
	local emulation = hardware.processIsTranslated()
	if emulation ~= "native" then
		util.print("System emulation: " .. emulation)
	end
	util.printInfo("CPU features: " .. features)
end

local function windowsVersion()
	if ffi.os ~= "Windows" then
		return ""
	end
	local hntdll = C.GetModuleHandleA("ntdll.dll")
	local RtlGetVersion = C.GetProcAddress(hntdll, "RtlGetVersion")
	if ffi.isNull(RtlGetVersion) then
		return 4 -- this works in win 2000 == 5.0
	end
	local ntdll = ffi.load("ntdll.dll")
	local versInfo = ffi.newNoAnchor("OSVERSIONINFOW[1]")
	versInfo[0].dwOSVersionInfoSize = ffi.sizeof("OSVERSIONINFOW")
	local status = ntdll.RtlGetVersion(versInfo)
	if status ~= 0 then
		util.printError("Windows RtlGetVersion() function return value '%s' is not SUCCESS == 0", tostring(status))
	end
	return tonumber(versInfo[0].dwMajorVersion .. "." .. versInfo[0].dwMinorVersion)
end

local function winVersionNumber(ver)
	-- https://en.wikipedia.org/wiki/List_of_Microsoft_Windows_versions
	if ver == "2000" then
		ver = 5.0
	elseif ver == "5.2" then
		ver = 5.2 -- Windows XP Professional x64 Edition, Wine uses this if it was set in winecfg
	elseif ver == "7" then
		ver = 6.1
	elseif ver == "8" then
		ver = 6.2
	elseif ver == "8.1" then
		ver = 6.3
	elseif tostring(ver):lower() == "xp" then
		ver = 5.1
	elseif tostring(ver):lower() == "vista" then
		ver = 6.0 -- Windows Vista
	end
	return tonumber(ver)
end

function hardware.windowsVersionString()
	if ffi.os ~= "Windows" then -- util must be delay-loaded
		return ""
	end
	local ver = windowsVersion()
	if ver == 5.0 then
		ver = "2000"
	elseif ver == 5.1 then
		ver = "XP"
	elseif ver == 6.0 then
		ver = "Vista"
	elseif ver == 6.1 then
		ver = "7"
	elseif ver == 6.2 then
		ver = "8"
	elseif ver == 6.3 then
		ver = "8.1"
	end
	return tostring(ver)
end

function hardware.windowsMinVersion(ver)
	if ffi.os ~= "Windows" then -- util must be delay-loaded
		return true
	end
	ver = winVersionNumber(ver)
	local winVer = windowsVersion()
	return winVer >= ver
end

--[[
local ffi = require "mffi"
local libhw = util.loadDll("libhw")

ffi.cdef[=[
void cpu_init(void); -- calls gcc build-in function: __builtin_cpu_init
int cpu_supports(const char *feature); -- __builtin_cpu_supports
int cpu_is(const char *cpuname); -- __builtin_cpu_is
]=]

local function init() -- hardware.cpuSupports("sse4.2")
	if initDone then
		return
	end
	libhw.cpu_init()
	initDone = true
end

function hardware.cpuSupports(feature) -- hardware.cpuSupports("sse4.2")
	init()
	local ret = libhw.cpu_supports(feature)
	return ret == 1
end


function hardware.cpuIs(cpuname) -- hardware.cpuSupports("sse4.2")
	init()
	local ret = libhw.cpu_is(cpuname)
	return ret == 1
end
]]

return hardware

--[[
/*
 *  hardware.c
 * test hardware in runtime
 * this MUST be compiled with real GCC compiler, not clang
 * https://gcc.gnu.org/onlinedocs/gcc-4.9.2/gcc/X86-Built-in-Functions.html
 *
 *  Created by pasi on 2017-11-22.
 *  Copyright (c) 2014 Manage Applications. All rights reserved.
 *
 */

-- #include <stdlib.h>     /* for realloc() and free() */
-- #include <string.h>     /* for memset() */
-- #include <errno.h>      /* for errno */

int main(){
    return 0;
}

void cpu_init(void) {
    __builtin_cpu_init();
}

int cpu_supports(const char *feature) {
    return __builtin_cpu_supports(feature);
}

int cpu_is(const char *cpuname) {
    return __builtin_cpu_is(const char *cpuname);
}

----

gcc -O2 -dynamiclib -m32 hadware.c -o hadwarex86.dylib
gcc -O2 -dynamiclib -m64 hadware.c -o hadware.dylib

gcc -O2 -shared -m64 hadware.c -o hadware.so

gcc -O2 -shared -m32 hadware.c -o hadwarex86.dll
gcc -O2 -shared -m64 hadware.c -o hadwarex.dll

file hadwarex86.dylib
file hadware.dylib
otool -L hadware.dylib
]]
