--- json.lua
-- https://github.com/xpol/lua-rapidjson/blob/master/API.md
-- @module serialize
local json = {}

local hardware = require "hardware"

local peg, json_sort -- load later
local util, l, lib, rapidjson, requireJson, jsonNull, isNan, isInf

local rapidJsonSse = ""
local defaultKeyNameArr = {
	"name",
	"return_type",
	"name_fi",
	"organization_id",
	"line_break",
	"main_data_tag",
	"namespace",
	"save_path",
	"save_path_mac",
	"hash",
	"component",
	"info",
	"comment",
	"table_name",
	"table",
	"table_prefix",
	"main_table",
	"record_type",
	"local_table",
	"local_record_type",
	"audit",
	"table_number",
	"table_name_language",
	"unique",
	"default_field",
	"field_name",
	"local_field",
	"field_type",
	"field_length",
	"field_number",
	"type",
	"font-class",
	"font",
	"length",
	"result", -- runner code start
	"function",
	"run",
	"parameter", -- runner code end, parameter also in query
	"default_order",
	"save_script",
	"calculated_field",
	"format",
	"field",
	"variable",
	"tag",
	"mandatory",
	"id",
	"text",
	"key",
	"icon",
	"attribute",
	"show",
	"value",
	"option",
	"options",
	"column",
	"hdr",
	"rec",
	"arr",
	"layout",
	"html",
	"css",
	"javascript",
	"query",
	"data_query",
	"run",
	"content",
	"url",
	"element",
	"left",
	"top",
	"width",
	"height",
	"model"
}

local function fixNull(tbl, name) -- warning name or "no-warning"
	if type(tbl) == "table" then
		for key, val in pairs(tbl) do
			if type(val) == "table" then
				fixNull(val, name)
			elseif val == jsonNull then
				tbl[key] = nil
				if name ~= "no-warning" then
					if name == nil then
						util.printWarning("json table key '%s' value was fixed from json null to nil", tostring(key))
					else
						util.printWarning("json table name '%s', key '%s' value was fixed from json null to nil", tostring(name), tostring(key))
					end
				end
			end
		end
	end
	return tbl
end
json.fixNull = fixNull

local function jsonFixError(fromTbl, toTbl, path, errTbl)
	for key, val in pairs(fromTbl) do
		toTbl[key] = fromTbl[key]
		if type(val) == "number" then
			if isNan(val) then
				errTbl[#errTbl + 1] = l("json key '%s' value is 'NaN (not a number)', fixed to 0", path .. "." .. key)
				util.printRed(errTbl[#errTbl])
				toTbl[key] = 0
			elseif isInf(val) then
				errTbl[#errTbl + 1] = l("json key '%s' value is 'Infinite', fixed to 0", path .. "." .. key)
				util.printRed(errTbl[#errTbl])
				toTbl[key] = 0
			end
		elseif val == jsonNull then
			errTbl[#errTbl + 1] = l("json key '%s' value was fixed from json null to nil", path .. "." .. key)
			util.printRed(errTbl[#errTbl])
			toTbl[key] = nil
		elseif type(val) == "cdata" then
			-- errTbl[#errTbl + 1] = l("json key '%s' value was fixed from cdata to nil", path .. "." .. key)
			-- util.printRed(errTbl[#errTbl])
			toTbl[key] = nil
		elseif type(val) == "userdata" then
			-- errTbl[#errTbl + 1] = l("json key '%s' value was fixed from userdata to nil", path .. "." .. key)
			-- util.printRed(errTbl[#errTbl])
			toTbl[key] = nil
		elseif type(val) == "function" then
			-- errTbl[#errTbl + 1] = l("json key '%s' value was fixed from function to nil", path .. "." .. key)
			-- util.printRed(errTbl[#errTbl])
			toTbl[key] = nil
		elseif type(val) == "thread" then
			-- errTbl[#errTbl + 1] = l("json key '%s' value was fixed from thread to nil", path .. "." .. key)
			-- util.printRed(errTbl[#errTbl])
			toTbl[key] = nil
		elseif type(val) == "table" then
			local prevPath = path
			path = path .. "." .. key
			jsonFixError(val, toTbl[key], path, errTbl)
			path = prevPath
		end
	end
	return fromTbl
end
json.fixError = jsonFixError

-- NOTE: -- we really need json.toJsonKeySorted(data, {"name", "table_prefix", "field"}, depth)
local function cleanJson(tbl)
	-- if getmetatable(variable) then
	local err = {}
	local tbl2 = {}
	jsonFixError(tbl, tbl2, tbl.name or "table", err) -- does also same fixNull()
	return tbl2
end
json.cleanJson = cleanJson

function json.defaultKeyNameArr()
	return defaultKeyNameArr
end

-- TODO, FIX: toJsonKeySorted and toJsonSorted can't handle always arrays, they make them in to a hash table

function json.toJsonKeySorted(variable, keyNameArr)
	if type(variable) ~= "table" then
		local err = l("json variable is not a table, type is: '%s'", type(variable))
		util.printError(err)
		return nil, err
	end
	if not json_sort then
		json_sort = require "json_sort"
	end
	return json_sort:encode_pretty(variable, nil, keyNameArr or defaultKeyNameArr) .. "\n" -- todo, use keyNameArr
end

--[[
-- toJsonKeySorted 4.8646 sec, toJsonSorted 30.0575 sec using bank data, toJsonKeySorted is 5 times faster
function json.toJsonSorted(variable, depth)
	if type(variable) ~= "table" then
		local err = l("json variable is not a table, type is: '%s'", type(variable))
		util.printError(err)
		return nil, err
	end
	if not depth then
		depth = 0
	end
	depth = depth + 1
	local sorted = {}
	for key, val in pairs(variable) do
		if type(val) == "table" then
			-- todo: isArray()
			val = json.toJsonSorted(val, depth) -- recursive call
			sorted[#sorted + 1] = string.rep("\t", depth) .. '"' .. key .. '": ' .. val .. ","
		elseif type(val) == "number" then
			sorted[#sorted + 1] = string.rep("\t", depth) .. '"' .. key .. '": ' .. val .. ","
		elseif type(val) == "nil" then
			sorted[#sorted + 1] = string.rep("\t", depth) .. '"' .. key .. '": null,"
		elseif type(val) == "boolean" then
			if val == true then
				sorted[#sorted + 1] = string.rep("\t", depth) .. '"' .. key .. '": true,'
			else
				sorted[#sorted + 1] = string.rep("\t", depth) .. '"' .. key .. '": false,'
			end
		else
			val = tostring(val)
			-- http://json.org
			if not peg then
				peg = require "peg"
			end
			val = peg.replace(val, "\\", "\\\\")
			val = peg.replace(val, '"', '\\"')
			val = peg.replace(val, "\b", "\\b")
			val = peg.replace(val, "\f", "\\f")
			val = peg.replace(val, "\n", "\\n")
			val = peg.replace(val, "\r", "\\r")
			val = peg.replace(val, "\t", "\\t")
			sorted[#sorted + 1] = string.rep("\t", depth) .. '"' .. key .. '": "' .. val .. '",'
		end
	end
	depth = depth - 1
	table.sort(sorted)
	sorted[#sorted] = sorted[#sorted]:sub(1, -2) -- remove "," form last element
	table.insert(sorted, 1, "{")
	sorted[#sorted + 1] = string.rep("\t", depth) .. "}"
	return table.concat(sorted, "\n")
end
--]]

local function jsonErrorText(jsonTxt, err)
	local pos = err:find("at character ", 1, true)
	if pos then
		pos = pos + ("at character "):len()
		pos = tonumber(err:sub(pos))
		if pos then
			err = err .. "\n" .. jsonTxt:sub(pos - 40, pos - 1) .. "ERR-->" .. jsonTxt:sub(pos, pos) .. "<--ERR" .. jsonTxt:sub(pos + 1, pos + 40)
		end
	end
	return err
end

--[=[
local function jsonError(ret, err, variable, pretty)
	if err == nil then
		err = "json encode failed, trying to fix errors"
	end
	-- ret = "{}"
	-- err = nil false and
	if type(variable) == "table" then
		print(err)
		local errTbl = {}
		jsonFixError(variable, "", errTbl)
		err = err .. "\n" .. table.concat(errTbl, "\n")
		local err2 -- ,ok
		if pretty then
			ret, err2 = json.toJson(variable)
		else
			ret, err2 = json.toJsonRaw(variable)
		end
		if err2 then --  ok == false or
			--[[ if err2 == nil then
				err2 = "json encode failed after jsonFixError()"
			end]]
			print(err2)
			err = err2 .. "\n" .. err
			-- else
			-- err = nil
		end
	end
	return ret, err
end
]=]

function json.cleanJsonText(txt)
	if not peg then
		peg = require "peg"
	end
	-- http://www.json.org/
	-- no: \/ or \ufour-hex-digits
	-- txt = peg.replace(txt, "\\", "") -- \
	-- txt = peg.replace(txt, '"', '') -- "
	txt = peg.replace(txt, "\r", "") -- return
	txt = peg.replace(txt, "\n", "") -- newline
	txt = peg.replace(txt, "\t", "") -- tab
	txt = peg.replace(txt, "\b", "") -- backspace
	txt = peg.replace(txt, "\f", "") -- form feed
	return txt
end

local function escapeJson(txt)
	if not peg then
		peg = require "peg"
	end
	-- http://www.json.org/
	-- no: \/ or \ufour-hex-digits
	-- txt = peg.replace(txt, string.char(9), "")
	-- txt = peg.replace(txt, string.char(10), "")
	-- txt = peg.replace(txt, string.char(13), "")
	txt = peg.replace(txt, "\\", "\\\\") -- \,
	txt = peg.replace(txt, '"', '\\"') -- "
	txt = peg.replace(txt, "\r", "\\r") -- return
	txt = peg.replace(txt, "\n", "\\n") -- newline
	txt = peg.replace(txt, "\t", "\\t") -- tab
	txt = peg.replace(txt, "\b", "\\b") -- backspace
	txt = peg.replace(txt, "\f", "\\f") -- form feed
	return txt
end
json.escapeJson = escapeJson

function json.validJson(txt)
	if type(txt) == "table" then
		return true
	end
	if #txt < 2 then
		return false
	end
	if txt == "{}" then
		return true
	end
	local _, err = json.fromJson(txt)
	if err then
		return false
	end
	return true
end

function json.printJson(label, variable)
	if not variable then
		variable = json.toJson(label)
		print(variable)
	elseif type(variable) == "table" then
		variable = json.toJson(variable)
		print(label .. " = " .. variable)
	elseif type(variable) == "string" then
		variable = json.toJson(label)
		print(variable .. " = " .. variable)
	end
end

function json.version()
	return lib._NAME .. " " .. lib._VERSION
end

function json.jsonLibraryName()
	return "rapidjson-" .. rapidJsonSse
end

-- rapidjson commands
local function jsonNullRapid()
	if rapidjson == nil then
		requireJson()
	end
	return rapidjson.null -- userdata: NULL
end

local function fromJsonRapid(txt, fileName, showError)
	if type(txt) == "table" then
		return txt
	end
	local ret, err
	if not txt then
		err = "json text is nil"
	elseif #txt < 2 then
		err = "json text size is too small: " .. #txt
	else
		if rapidjson == nil then
			requireJson()
		end
		ret, err = rapidjson.decode(txt)
	end
	if err then
		if fileName then
			err = string.format("invalid json in file '%s': ", fileName) .. jsonErrorText(txt, err)
		else
			err = "invalid json: " .. jsonErrorText(txt, err)
		end
		if showError ~= false then
			util.printError(err)
		end
	end
	return ret, err
end

local function toJsonRapid(variable, option) -- , clean
	if type(variable) ~= "table" then
		if option == "no-error" then
			return variable
		end
		local err = l("json variable is not a table, type is: '%s'", type(variable))
		util.printError(err)
		return nil, err
	end
	if rapidjson == nil then
		requireJson()
	end
	-- local tbl = setmetatable(variable, {__jsontype='array'})
	-- local txt, err = rapidjson.encode(tbl, {pretty = true}) -- , sort_keys = false
	-- if clean then
	local ok, txt = pcall(rapidjson.encode, variable, {pretty = true, empty_table_as_array = true})
	if not ok then
		ok, txt = pcall(rapidjson.encode, cleanJson(variable), {pretty = true, empty_table_as_array = true})
	end
	if not ok then
		local err = txt
		err = l("json encoding error '%s', data '%s'", tostring(err), tostring(variable):sub(1, 1500))
		util.printError(err)
		return "{}", err
	end
	-- if txt == "{}" then -- dows not help in nested situations: https://github.com/xpol/lua-rapidjson/issues/23
	-- txt = "[]" -- rapidjson.encode(setmetatable({}, {__jsontype='array'})) --> '[]'
	-- end
	return txt
end

local function toJsonRawRapid(variable, option) -- , clean
	--[[ -- not in rapidjson
	return json.minify(variable)
	--]]
	if type(variable) ~= "table" then
		if option == "no-error" then
			return variable
		end
		local err = l("json variable is not a table, type is: '%s'", type(variable))
		util.printError(err)
		return nil, err
	end
	if rapidjson == nil then
		requireJson()
	end
	local ok, txt = pcall(rapidjson.encode, variable, {empty_table_as_array = true})
	if not ok then
		ok, txt = pcall(rapidjson.encode, cleanJson(variable), {empty_table_as_array = true})
	end
	if not ok then
		local err = txt
		err = l("json encoding error '%s', data '%s'", tostring(err), tostring(variable):sub(1, 1500))
		util.printError(err)
		return "{}", err
	end
	return txt
end

local function toSortedString(obj, ignoreTbl)
	if type(obj) == 'table' then
		local ret = {}
		if util.isArray(obj) then
			for i in ipairs(obj) do
				ret[i] = toSortedString(obj[i])
			end
		else
			local keys = {}
			local i = 0
			for key in pairs(obj) do
				i = i + 1
				keys[i] = key
			end
			table.sort(keys)
			for j, key in ipairs(keys) do
				if not (ignoreTbl and ignoreTbl[key]) then
					ret[j] = tostring(key) .. ":" .. toSortedString(obj[key], ignoreTbl)
				end
			end
		end
		return table.concat(ret, "|")
	elseif type(obj) == 'string' then
		return escapeJson(obj)
	end
	return tostring(obj)
end
json.toSortedString = toSortedString

function json.getKey(data, key)
	if type(data) == "string" then
		local tbl = json.fromJson(data)
		if type(tbl) == "table" then
			data = tbl
		end
	end
	if type(data) == "table" and data[key] ~= nil then -- todo: use recData
		return data[key]
	end
end

requireJson = function()
	-- if true then -- not in use until {a={}} -> "{a=[]}" instead of "{a={}}", see: https://github.com/xpol/lua-rapidjson/issues/23
	util = require "util"
	l = require"lang".l
	isNan = util.isNan
	isInf = util.isInf
	local ok
	if util.isWin() then -- and not util.from4d() then
		local ffi = require "ffi"
		util.loadDll("graphics/libwinpthread-1.dll")
		if ffi.arch == "x64" then
			util.loadDll("graphics/libgcc_s_seh-1.dll")
		else
			util.loadDll("graphics/libgcc_s_sjlj-1.dll")
			util.loadDll("graphics/libgcc_s_dw2-1.dll") -- graphics/zlib1.dll needs this
		end
		util.loadDll("graphics/libstdc++-6.dll")
	end
	if hardware.cpuSupports("sse4.2") then
		ok, rapidjson = pcall(require, "rapidjson_sse42") -- require "rapidjson"
		rapidJsonSse = "sse4.2"
	end
	if not ok then
		ok, rapidjson = pcall(require, "rapidjson_sse2")
		rapidJsonSse = "sse2"
	end
	if not ok then -- arm
		ok, rapidjson = pcall(require, "rapidjson")
		rapidJsonSse = "arm"
	end
	if not ok then -- arm
		util.printError("failed to load rapidjson, error '%s'", tostring(rapidjson))
	end
end

local function init()
	lib = rapidjson
	json.jsonNull = jsonNullRapid
	json.fromJson = fromJsonRapid
	json.toJson = toJsonRapid
	json.toJsonRaw = toJsonRawRapid
	json.encode = json.toJsonRaw
	json.decode = json.fromJson
	jsonNull = json.jsonNull()
end

init()

return json
